﻿import utils.*;


class utils.Matrix4x4
{
	
	//|  m00  m01  m02  tx|
	//|  m10  m11  m12  ty|
	//|  m20  m21  m22  tz|
	//|  m30  m31  m32  tw|
	private var m00: Number, m01: Number, m02: Number, tx: Number;	
	private var m10: Number, m11: Number, m12: Number, ty: Number;	
	private var m20: Number, m21: Number, m22: Number, tz: Number;	
	private var m30: Number, m31: Number, m32: Number, tw: Number;	
	
	function toString () :String
	{
		return [
				[ "----------------------------" ],
				[ "[" + m00, m01, m02, tx + "]" ],
				[ "[" + m10, m11, m12, ty + "]" ],
				[ "[" + m20, m21, m22, tz + "]" ],
				[ "[" + m30, m31, m32, tw + "]" ]
				].join( "\n" );
	}
	
	public function Matrix4x4()
	{
		identity();
	};
	
	
	/**
	* 単位マトリックスに変換。
	*/
	public function identity()
	{
		//|  1  0  0  0|
		//|  0  1  0  0|
		//|  0  0  1  0|
		//|  0  0  0  1|
		
		m00 = 1;	m01 = 0; 	m02 = 0;	tx = 0;
		m10 = 0;	m11 = 1;	m12 = 0;	ty = 0;	
		m20 = 0;	m21 = 0;	m22 = 1;	tz = 0;
		m30 = 0;	m31 = 0; 	m32 = 0;	tw = 1;

		return this;
	};
	
	/**
	* 合成。
	*/
	public function concat( matrix: Matrix4x4 ): Void
	{
		var a: Matrix4x4 = Matrix4x4( clone() );
		//|  m00  m01  m02  tx|
		//|  m10  m11  m12  ty|
		//|  m20  m21  m22  tz|
		//|  m30  m31  m32  tw|
		
		var b: Matrix4x4 = Matrix4x4( matrix );
		//|  m00  m01  m02  tx|
		//|  m10  m11  m12  ty|
		//|  m20  m21  m22  tz|
		//|  m30  m31  m32  tw|
		
		m00 = a.m00 * b.m00 + a.m01 * b.m10 + a.m02 * b.m20 + a.tx * b.m30;
		m01 = a.m00 * b.m01 + a.m01 * b.m11 + a.m02 * b.m21 + a.tx * b.m31;
		m02 = a.m00 * b.m02 + a.m01 * b.m12 + a.m02 * b.m22 + a.tx * b.m32;
		tx  = a.m00 * b.tx  + a.m01 * b.ty  + a.m02 * b.tz	+ a.tx * b.tw;
		
		m10 = a.m10 * b.m00 + a.m11 * b.m10 + a.m12 * b.m20 + a.ty * b.m30;
		m11 = a.m10 * b.m01 + a.m11 * b.m11 + a.m12 * b.m21 + a.ty * b.m31;
		m12 = a.m10 * b.m02 + a.m11 * b.m12 + a.m12 * b.m22 + a.ty * b.m32;
		ty  = a.m10 * b.tx  + a.m11 * b.ty  + a.m12 * b.tz  + a.ty * b.tw;
		
		m20 = a.m20 * b.m00 + a.m21 * b.m10 + a.m22 * b.m20 + a.tz * b.m30;
		m21 = a.m20 * b.m01 + a.m21 * b.m11 + a.m22 * b.m21 + a.tz * b.m31;
		m22 = a.m20 * b.m02 + a.m21 * b.m12 + a.m22 * b.m22 + a.tz * b.m32;
		tz  = a.m20 * b.tx  + a.m21 * b.ty  + a.m22 * b.tz	+ a.tz * b.tw;
		
		m30 = a.m30 * b.m00 + a.m31 * b.m10 + a.m32 * b.m20 + a.tw * b.m30;
		m31 = a.m30 * b.m01 + a.m31 * b.m11 + a.m32 * b.m21 + a.tw * b.m31;
		m32 = a.m30 * b.m02 + a.m31 * b.m12 + a.m32 * b.m22 + a.tw * b.m32;
		tw  = a.m30 * b.tx  + a.m31 * b.ty  + a.m32 * b.tz	+ a.tw * b.tw;
	}
	
	public function clone(): Matrix4x4
	{
		var m: Matrix4x4 = new Matrix4x4();
		
		m.m00 = m00;	m.m01 = m01;	m.m02 = m02;	m.tx  = tx;	
		m.m10 = m10;	m.m11 = m11;	m.m12 = m12;	m.ty  = ty;	
		m.m20 = m20;	m.m21 = m21;	m.m22 = m22;	m.tz  = tz;
		m.m30 = m30;	m.m31 = m31;	m.m32 = m32;	m.tw  = tw;
		
		return m;
	}
	
	/**
	* マトリックスを転置。
	*/
	public function transpose(): Void
	{
		var m: Matrix4x4 = new Matrix4x4();
		
		m.m00 = m00;	m.m01 = m10;	m.m02 = m20;	m.tx = m30;
		m.m10 = m01;	m.m11 = m11;	m.m12 = m21;	m.ty = m31;
		m.m20 = m02;	m.m21 = m12;	m.m22 = m22;	m.tz = m32;
		m.m30 = tx;		m.m31 = ty;		m.m32 = tz;		m.tw = tw;

		identity();
		concat( m );
	}
	
	
	/**
	* 平行移動。
	*/
	public function translate( x: Number, y: Number, z: Number ): Void
	{
		//|  1  0  0  dx|
		//|  0  1  0  dy|
		//|  0  0  1  dz|
		//|  0  0  0   1|
		
		var m: Matrix4x4 = new Matrix4x4();
		
		m.tx = x;
		m.ty = y;
		m.tz = z;
		
		concat( m );
	}
	
	/**
	* 拡大縮小。
	*/
	public function scale( sx: Number, sy: Number, sz: Number ): Void
	{
		//| sx  0  0  0|
		//|  0 sy  0  0|
		//|  0  0 sz  0|
		//|  0  0  0  1|
		
		var m: Matrix4x4 = new Matrix4x4();
		
		m.m00 = sx;
		m.m11 = sy;
		m.m22 = sz;
		
		concat( m );
	}
	
	/**
	* 回転。
	*/
	//public function rotate( axis: Number, angle: Angle ): Void
	public function rotate( axis: Number, radian :Number ): Void
	{
		//var s: Number = Math.sin( angle.toRadian() );
		//var c: Number = Math.cos( angle.toRadian() );
		var s: Number = Math.sin( radian );
		var c: Number = Math.cos( radian );
		
		var m: Matrix4x4 = new Matrix4x4();
		
		switch( axis )
		{
			case Axis.X:
				//X軸回転
				//| 1     0     0 0|
				//| 0  cosX -sinX 0|
				//| 0  sinX  cosX 0|
				//| 0     0     0 1|	
				
				//-- rotate about x-axis
				m.m10 = 0;	m.m11 = c;	m.m12 = -s;
				m.m20 = 0;	m.m21 = s;	m.m22 =  c;
				break;
				
			case Axis.Y:		
				//Y軸回転
				//|  cosY  0  sinY  0|
				//|     0  1     0  0|
				//| -sinY  0  cosY  0|
				//|     0  0     0  1|
			
				//-- rotate about y-axis
				m.m00 =  c;	m.m01 = 0;	m.m02 = s;
				m.m20 = -s;	m.m21 = 0;	m.m22 = c;
				break;
				
			case Axis.Z:
				//Z軸回転
				//|  cosZ -sinZ 0 0|
				//|  sinZ  cosZ 0 0|
				//|     0     0 1 0|
				//|     0     0 0 1|	

				//-- rotate about z-axis
				m.m00 = c;	m.m01 = -s;	m.m02 = 0;
				m.m10 = s;	m.m11 =  c;	m.m12 = 0;
				break;
		}
		
		concat( m );
	}
	
	/**
	* ３軸回転。
	*/
	public function rotate3Axis( xa , ya , za  ): Void
	{
		var x :Matrix4x4 = new Matrix4x4();
		var y :Matrix4x4 = new Matrix4x4();
		var z :Matrix4x4 = new Matrix4x4();
		var r :Matrix4x4 = new Matrix4x4();
		
		x.rotate( Axis.X, xa );
		y.rotate( Axis.Y, ya );
		z.rotate( Axis.Z, za );
		
		r.concat( x );
		r.concat( y );
		r.concat( z );
		
		concat( r );
	};
	
	/*
	public function transformVector( v :Vector ): Vector
	{
		
		var x: Number = v.x;
		var y: Number = v.y;
		var z: Number = v.z;
		
		v.x = m00 * x + m01 * y + m02 * z + tx;
		v.y = m10 * x + m11 * y + m12 * z + ty;
		v.z = m20 * x + m21 * y + m22 * z + tz;
		
		return v;
	}
	*/
	public function transformNode( v )
	{
		
		var x: Number = v.x;
		var y: Number = v.y;
		var z: Number = v.z;
		
		v.x = m00 * x + m01 * y + m02 * z + tx;
		v.y = m10 * x + m11 * y + m12 * z + ty;
		v.z = m20 * x + m21 * y + m22 * z + tz;
		
		return v;
	}
}