/*
 * ProfilerComponent.java
 * Copyright (c) 2002 Boxed-Economy Project.  All rights reserved.
 */
package org.boxed_economy.components.profiler;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Date;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;

import org.boxed_economy.besp.container.BESP;
import org.boxed_economy.besp.model.fmfw.Clock;
import org.boxed_economy.besp.model.fmfw.World;
import org.boxed_economy.besp.model.fmfw.update.UpdateClockEvent;
import org.boxed_economy.besp.model.fmfw.update.UpdateClockListener;
import org.boxed_economy.besp.presentation.PresentationException;
import org.boxed_economy.besp.presentation.guifw.AbstractInternalFrameComponent;

/**
 * @author macchan
 * @version $Id: ProfilerComponent.java,v 1.1 2004/03/21 12:07:50 macchan Exp $
 */
public class ProfilerComponent
	extends AbstractInternalFrameComponent
	implements UpdateClockListener {

	/****************************************
	 * Constants.
	 ****************************************/

	public static final String SEPARATOR = ",";

	private static final String PROFILE_HISTORY_FILENAME =
		"profiling_history.log.csv";
	private static final String INSTANCE_HISTORY_FILENAME =
		"instance_history.log.csv";
	private static final String SNAPSHOT_FILENAME = "profiling_snapshot.csv";

	private static final String RUNNING_BUTTON_TEXT = "STOP";
	private static final String STOPPED_BUTTON_TEXT = "START";
	private static final String RUNNNING_STATE_TEXT = "state:RUNNING";
	private static final String STOPPED_STATE_TEXT = "state:STOPPED";

	private static final int RUNNING = 1;
	private static final int STOPPED = 2;

	/****************************************
	 * InstanceVariables.
	 ****************************************/

	private InstanceProfilingTableModel instanceModel =
		new InstanceProfilingTableModel();
	private MemoryProfilingTableModel memoryModel =
		new MemoryProfilingTableModel();

	private int profilingState = STOPPED;
	private Thread profilingThread = null;
	private long profilingInterval = 500;

	private int instanceHistoryWritingState = STOPPED;
	private PrintWriter instanceHistoryWriter = new PrintWriter(System.out);
	private long instanceHistoryStepInterval = 10;
	private long stepCounter = 0;

	private int memoryHistoryWritingState = STOPPED;
	private Thread memoryHistoryWritingThread = null;
	private PrintWriter memoryHistoryWriter = new PrintWriter(System.out);
	private long memoryHistoryInterval = 30000;

	private JButton profilingButton = new JButton(STOPPED_BUTTON_TEXT);
	private JTextField profilingIntervalTextField = new JTextField();
	private JLabel profilingStateLabel = new JLabel(STOPPED_STATE_TEXT);

	private JTextField memoryHistoryIntervalTextField = new JTextField();
	private JLabel memoryHistoryStateLabel = new JLabel(STOPPED_STATE_TEXT);
	private JButton memoryHistoryButton = new JButton(STOPPED_BUTTON_TEXT);

	private JTextField instanceHistoryIntervalTextField = new JTextField();
	private JLabel instanceHistoryStateLabel = new JLabel(STOPPED_STATE_TEXT);
	private JButton instanceHistoryButton = new JButton(STOPPED_BUTTON_TEXT);

	/****************************************
	 * Constructors.
	 ****************************************/

	/**
	 * Constructor for ProfilerComponent.
	 */
	public ProfilerComponent() {
		super();
	}

	/****************************************
	 * Initializers.
	 ****************************************/

	/**
	 * @see org.boxed_economy.besp.presentation.PresentationComponent#initialize()
	 */
	public void initialize() {
		try {
			this.addUpdateClockListener(this);
			initializeComponents();
		}
		catch (Exception ex) {
			throw new PresentationException(ex);
		}
	}

	/**
	 * @see org.boxed_economy.besp.presentation.guifw.AbstractInternalFrameComponent#initializeDefaultCloseOperation()
	 */
	protected void initializeDefaultCloseOperation() {
		this.setDefaultCloseOperation(HIDE_ON_CLOSE);
	}

	/**
	 * @see org.boxed_economy.besp.presentation.guifw.AbstractInternalFrameComponent#initializeFrame()
	 */
	protected void initializeFrame() {
		this.setTitle("Profiler");
		this.setLocation(100, 100);
		this.setSize(750, 400);
	}

	/**
	 * Method initializeComponents.
	 */
	private void initializeComponents() {
		Container container = this.getContentPane();
		container.setLayout(new BorderLayout());

		//Profiler Setting Panel
		JPanel profilerSettingPanel = new JPanel();
		profilerSettingPanel.setLayout(new BorderLayout());
		JPanel pNorthPanel = new JPanel();
		pNorthPanel.add(new JLabel("Profiler"));
		pNorthPanel.add(profilingStateLabel);
		profilerSettingPanel.add(pNorthPanel, BorderLayout.NORTH);
		JPanel pCenterPanel = new JPanel();
		pCenterPanel.add(new JLabel("Update Interval(ms)"));
		profilingIntervalTextField.setText(Long.toString(profilingInterval));
		profilingIntervalTextField.setPreferredSize(new Dimension(50, 20));
		pCenterPanel.add(profilingIntervalTextField);
		profilerSettingPanel.add(pCenterPanel, BorderLayout.CENTER);
		profilingButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				doTransition();
			}
		});
		profilerSettingPanel.add(profilingButton, BorderLayout.SOUTH);

		//Memory History Setting Panel
		JPanel memoryHistoryPanel = new JPanel();
		memoryHistoryPanel.setLayout(new BorderLayout());
		JPanel msNorthPanel = new JPanel();
		msNorthPanel.add(new JLabel("Memory History"));
		msNorthPanel.add(memoryHistoryStateLabel);
		memoryHistoryPanel.add(msNorthPanel, BorderLayout.NORTH);
		JPanel msCenterPanel = new JPanel();
		msCenterPanel.add(new JLabel("Interval(ms)"));
		memoryHistoryIntervalTextField.setText(
			Long.toString(memoryHistoryInterval));
		memoryHistoryIntervalTextField.setPreferredSize(new Dimension(50, 20));
		msCenterPanel.add(memoryHistoryIntervalTextField);
		memoryHistoryPanel.add(msCenterPanel, BorderLayout.CENTER);
		memoryHistoryButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				doMemoryStateTransition();
			}
		});
		memoryHistoryPanel.add(memoryHistoryButton, BorderLayout.SOUTH);

		//Instance History Setting Panel
		JPanel instanceHistoryPanel = new JPanel();
		instanceHistoryPanel.setLayout(new BorderLayout());
		JPanel insNorthPanel = new JPanel();
		insNorthPanel.add(new JLabel("Instance History"));
		insNorthPanel.add(instanceHistoryStateLabel);
		instanceHistoryPanel.add(insNorthPanel, BorderLayout.NORTH);
		JPanel insCenterPanel = new JPanel();
		insCenterPanel.add(new JLabel("Interval(step)"));
		instanceHistoryIntervalTextField.setText(
			Long.toString(instanceHistoryStepInterval));
		instanceHistoryIntervalTextField.setPreferredSize(new Dimension(50, 20));
		insCenterPanel.add(instanceHistoryIntervalTextField);
		instanceHistoryPanel.add(insCenterPanel, BorderLayout.CENTER);
		instanceHistoryButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				doInstanceStateTransition();
			}
		});
		instanceHistoryPanel.add(instanceHistoryButton, BorderLayout.SOUTH);

		//GC & SnapShot Panel
		JPanel gcPanel = new JPanel();
		JButton gcButton = new JButton("GC");
		gcButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				Runtime.getRuntime().gc();
			}
		});
		JButton snapshotButton = new JButton("SnapShot");
		snapshotButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				snapshot();
			}
		});
		gcPanel.add(gcButton);
		gcPanel.add(snapshotButton);

		//Setting Panel
		JPanel settingPanel = new JPanel();
		settingPanel.add(profilerSettingPanel);
		settingPanel.add(memoryHistoryPanel);
		settingPanel.add(instanceHistoryPanel);
		settingPanel.add(gcPanel);

		//Instance Panel
		JTable instanceTable = new JTable(instanceModel);
		JScrollPane instancePanel = new JScrollPane(instanceTable);

		//Memory Panel
		JPanel memoryPanel = new JPanel();
		JTable memoryTable = new JTable(memoryModel);
		JScrollPane memoryPane = new JScrollPane(memoryTable);
		memoryPanel.add(new JLabel("Memory State :"));
		memoryPanel.add(memoryPane);
		memoryPanel.setPreferredSize(new Dimension(300, 75));

		//ALL
		container.add(settingPanel, BorderLayout.NORTH);
		container.add(instancePanel, BorderLayout.CENTER);
		container.add(memoryPanel, BorderLayout.SOUTH);
	}

	/****************************************
	 * Terminaters.
	 ****************************************/

	/**
	 * @see org.boxed_economy.besp.presentation.PresentationComponent#terminate()
	 */
	public void terminate() {
		if (memoryHistoryWriter != null) {
			memoryHistoryWriter.close();
		}
		if (instanceHistoryWriter != null) {
			instanceHistoryWriter.close();
		}
		this.removeUpdateClockListener(this);
	}

	/****************************************
	 * UpdateClockListener
	 ****************************************/

	public void clockGained(UpdateClockEvent e) {
		if (instanceHistoryWritingState == RUNNING) {
			stepCounter++;
			if (stepCounter % instanceHistoryStepInterval == 0) {
				instanceHistoryWrite();
			}
		}
	}

	/****************************************
	 * Logics
	 ****************************************/

	private void doTransition() {
		if (profilingState == STOPPED) {
			startThread();
		}
		else {
			stopThread();
		}
	}

	private void setProfilingState(int state) {
		this.profilingState = state;
		if (state == RUNNING) {
			profilingButton.setText(RUNNING_BUTTON_TEXT);
			profilingStateLabel.setText(RUNNNING_STATE_TEXT);
		}
		else if (state == STOPPED) {
			profilingButton.setText(STOPPED_BUTTON_TEXT);
			profilingStateLabel.setText(STOPPED_STATE_TEXT);
		}
	}

	private void startThread() {
		try {
			profilingThread = new Thread() {
				public void run() {
					try {
						synchronized (this) {
							setProfilingState(RUNNING);
							while (profilingState == RUNNING) {
								updateViews();
								wait(profilingInterval);
							}
						}
					}
					catch (Exception ex) {
						ex.printStackTrace();
					}
				}
			};
			profilingInterval =
				Integer.parseInt(profilingIntervalTextField.getText());
			profilingThread.start();
		}
		catch (Exception ex) {
			throw new PresentationException(ex);
		}
	}

	private void stopThread() {
		synchronized (profilingThread) {
			setProfilingState(STOPPED);
			profilingThread.notifyAll();
		}
	}

	private void doMemoryStateTransition() {
		if (memoryHistoryWritingState == STOPPED) {
			startMemoryHistoryThread();
		}
		else {
			stopMemoryHistoryThread();
		}
	}

	private void setMemoryHistoryState(int state) {
		this.memoryHistoryWritingState = state;
		if (state == RUNNING) {
			memoryHistoryButton.setText(RUNNING_BUTTON_TEXT);
			memoryHistoryStateLabel.setText(RUNNNING_STATE_TEXT);
		}
		else if (state == STOPPED) {
			memoryHistoryButton.setText(STOPPED_BUTTON_TEXT);
			memoryHistoryStateLabel.setText(STOPPED_STATE_TEXT);
		}
	}

	private void startMemoryHistoryThread() {
		try {
			memoryHistoryWriter =
				new PrintWriter(
					new FileWriter(
						new File(BESP.boxHome, PROFILE_HISTORY_FILENAME),
						true));
			memoryHistoryWritingThread = new Thread() {
				public void run() {
					try {
						synchronized (this) {
							setMemoryHistoryState(RUNNING);
							while (memoryHistoryWritingState == RUNNING) {
								memoryHistoryWrite();
								wait(memoryHistoryInterval);
							}
						}
					}
					catch (Exception ex) {
						ex.printStackTrace();
					}
				}
			};
			memoryHistoryInterval =
				Integer.parseInt(memoryHistoryIntervalTextField.getText());
			memoryHistoryWritingThread.start();
		}
		catch (Exception ex) {
			throw new PresentationException(ex);
		}
	}

	private void stopMemoryHistoryThread() {
		synchronized (memoryHistoryWritingThread) {
			setMemoryHistoryState(STOPPED);
			memoryHistoryWritingThread.notifyAll();
		}
		if (memoryHistoryWriter != null) {
			memoryHistoryWriter.close();
			memoryHistoryWriter = null;
		}
	}

	private void doInstanceStateTransition() {
		if (instanceHistoryWritingState == STOPPED) {
			setInstanceHistoryState(RUNNING);
		}
		else {
			setInstanceHistoryState(STOPPED);
		}
	}

	private void setInstanceHistoryState(int state) {
		try {
			instanceHistoryStepInterval =
				Integer.parseInt(instanceHistoryIntervalTextField.getText());
			this.instanceHistoryWritingState = state;
			if (state == RUNNING) {
				instanceHistoryWriter =
					new PrintWriter(
						new FileWriter(
							new File(BESP.boxHome, INSTANCE_HISTORY_FILENAME),
							true));
				instanceHistoryButton.setText(RUNNING_BUTTON_TEXT);
				instanceHistoryStateLabel.setText(RUNNNING_STATE_TEXT);
			}
			else if (state == STOPPED) {
				instanceHistoryButton.setText(STOPPED_BUTTON_TEXT);
				instanceHistoryStateLabel.setText(STOPPED_STATE_TEXT);
				if (instanceHistoryWriter != null) {
					instanceHistoryWriter.close();
					instanceHistoryWriter = null;
				}
			}
		}
		catch (IOException ex) {
			throw new PresentationException(ex);
		}
	}

	/********************************************
	 * Update And Write
	 ********************************************/

	private void updateViews() {
		instanceModel.fireTableDataChanged();
		memoryModel.fireTableDataChanged();
	}

	private void memoryHistoryWrite() {
		//preparing
		memoryModel.fireTableDataChanged();

		//main
		StringBuffer buf = new StringBuffer();
		buf.append(getTimeStamp());
		buf.append(memoryModel.getModelString());

		//write
		memoryHistoryWriter.println(buf);
		memoryHistoryWriter.flush();
	}

	private void instanceHistoryWrite() {
		System.gc();

		Thread th = new Thread() {
			public void run() {
				instanceHistoryWriteImpl();
			}
		};
		th.setPriority(4);
		th.start();
	}

	private void instanceHistoryWriteImpl() {
		//preparing
		instanceModel.fireTableDataChanged();

		//main
		StringBuffer buf = new StringBuffer();
		buf.append(getTimeStamp());
		buf.append(instanceModel.getModelString());

		//write
		instanceHistoryWriter.println(buf);
		instanceHistoryWriter.flush();
	}

	private void snapshot() {
		System.gc();

		Thread th = new Thread() {
			public void run() {
				snapshotImpl();
			}
		};
		th.setPriority(4);
		th.start();
	}

	private void snapshotImpl() {
		try {
			//preparing
			updateViews();
			PrintWriter writer =
				new PrintWriter(
					new FileWriter(new File(BESP.boxHome, SNAPSHOT_FILENAME), true));

			//main
			StringBuffer buf = new StringBuffer();
			buf.append(getTimeStamp());
			buf.append(memoryModel.getModelString());
			buf.append(instanceModel.getModelString());

			//write
			writer.println(buf);
			writer.flush();
			writer.close();
		}
		catch (IOException ex) {
			throw new PresentationException(ex);
		}
	}

	public String getTimeStamp() {
		StringBuffer buf = new StringBuffer();

		Clock clock = getClock();
		Date date = new Date();

		buf.append(date);
		buf.append(SEPARATOR);
		buf.append(clock);

		return buf.toString();
	}

	public Clock getClock() {
		return getWorld().getClock();
	}

	public World getWorld() {
		return getPresentationContainer().getModelContainer().getModel();
	}

}