/*
 * VerticalFlowLayout.java
 * Copyright (c) 2002 Boxed-Economy Project.  All rights reserved.
 */
package jp.ac.keio.sfc.crew.swing.layout;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.io.Serializable;

/**
 * @author macchan
 * @version $Id: VerticalFlowLayout.java,v 1.1 2003/12/11 10:27:05 macchan Exp $
 */
public class VerticalFlowLayout implements LayoutManager, Serializable {

	/***********************************************
	 * Constants.
	 ***********************************************/

	public static final int CENTER = 1;
	public static final int LEFT = 2;
	public static final int RIGHT = 3;
	public static final int FILL = 4;

	/***********************************************
	 * Instance Variables.
	 ***********************************************/

	private int align;

	private int vGap;
	private int hGap;

	/***********************************************
	 * Constructors.
	 ***********************************************/

	/**
	 * Constructor for VerticalFlowLayout.
	 */
	public VerticalFlowLayout() {
		this(LEFT);
	}

	/**
	 * Constructor for VerticalFlowLayout.
	 */
	public VerticalFlowLayout(int align) {
		this(align, 5, 5);
	}

	/**
	 * Constructor for VerticalFlowLayout.
	 */
	public VerticalFlowLayout(int align, int vGap, int hGap) {
		this.align = align;
		this.vGap = vGap;
		this.hGap = hGap;
	}

	/***********************************************
	 * Implementor Methods for LayoutManager.
	 ***********************************************/

	/**
	 * @see java.awt.LayoutManager#addLayoutComponent(String, Component)
	 */
	public void addLayoutComponent(String name, Component comp) {
	}

	/**
	 * @see java.awt.LayoutManager#removeLayoutComponent(Component)
	 */
	public void removeLayoutComponent(Component comp) {
	}

	/**
	 * @see java.awt.LayoutManager#preferredLayoutSize(Container)
	 */
	public Dimension preferredLayoutSize(Container parent) {
		synchronized (parent.getTreeLock()) {

			int width = 0;
			int height = 0;

			int componentCount = parent.getComponentCount();
			for (int i = 0; i < componentCount; i++) {
				Component component = parent.getComponent(i);
				if (component.isVisible()) {
					Dimension d = component.getPreferredSize();
					width = Math.max(width, d.width);
					height = height + vGap + d.height;
				}
			}

			Insets insets = parent.getInsets();
			return new Dimension(
				insets.left + insets.right + width + hGap * 2,
				insets.top + insets.bottom + height + vGap * 2);
		}
	}

	/**
	 * @see java.awt.LayoutManager#minimumLayoutSize(Container)
	 */
	public Dimension minimumLayoutSize(Container parent) {
		return this.preferredLayoutSize(parent);
	}

	/**
	 * @see java.awt.LayoutManager#layoutContainer(Container)
	 */
	public void layoutContainer(Container parent) {
		synchronized (parent.getTreeLock()) {
			Insets insets = parent.getInsets();

			int x = insets.left + hGap;
			int y = insets.top + vGap;

			int componentCount = parent.getComponentCount();
			for (int i = 0; i < componentCount; i++) {
				Component component = parent.getComponent(i);
				if (component.isVisible()) {
					Dimension d = component.getPreferredSize();
					int w = d.width;
					int h = d.height;
					int parentWidth = parent.getWidth();
					switch (align) {
						case CENTER :
							x = parentWidth / 2 - d.width / 2;
							break;
						case RIGHT :
							x = parentWidth - d.width - insets.right - hGap;
							break;
						case FILL :
							w =
								parentWidth
									- insets.left
									- insets.right
									- hGap * 2;
							break;
					}
					component.setBounds(x, y, w, h);
					y = y + vGap + h;
				}
			}
		}
	}

	/**
	 * Method getHGap.
	 * @return
	 */
	public int getHGap() {
		return hGap;
	}

	/**
	 * Method getVGap.
	 * @return
	 */
	public int getVGap() {
		return vGap;
	}

	/**
	 * Method setHGap.
	 * @param i
	 */
	public void setHGap(int hGap) {
		this.hGap = hGap;
	}

	/**
	 * Method setVGap.
	 * @param i
	 */
	public void setVGap(int vGap) {
		this.vGap = vGap;
	}

	/**
	 * Method getAlign.
	 * @return
	 */
	public int getAlign() {
		return align;
	}

	/**
	 * Method setAlign.
	 * @param i
	 */
	public void setAlign(int i) {
		align = i;
	}

}
