/*
 * VisualComponent.java
 * Created on 2003/10/12
 * 
 * Copyright (c) 2003 CreW Project. All rights reserved.
 */
package jp.ac.keio.sfc.crew.swing.visuals;

import java.awt.Container;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;

import javax.swing.JComponent;

import jp.ac.keio.sfc.crew.swing.SwingUtil;

/**
 * Class VisualComponent.
 * 
 * @author macchan
 * @version $Id: VisualComponent.java,v 1.1 2003/12/11 10:27:01 macchan Exp $
 */
public class VisualComponent extends JComponent {

	/***********************************
	 * Instance Variables.
	 ***********************************/

	private VisualComponent parentVisual;

	/***********************************
	 * Constructors.
	 ***********************************/

	/**
	 * Constructor for VisualComponent.
	 */
	public VisualComponent() {
		this.setLayout(null);
		this.setOpaque(false);
		this.setForeground(SwingUtil.COLOR_OPAQUE);
		this.setBackground(SwingUtil.COLOR_OPAQUE);
	}

	/***********************************
	 * For Location.
	 ***********************************/

	/**
	 * Translates location to the absolute location.
	 * @return
	 */
	public void translateToAbsolute(Point point) {
		if (this instanceof LayerVisualComponent) {
			return;
		}

		Container current = this.getParent();
		while (current != null && !(current instanceof LayerVisualComponent)) {
			point.translate(current.getX(), current.getY());
			current = current.getParent();
		}
	}

	/**
	 * Gets the center location.
	 */
	public Point getCenterLocation() {
		Rectangle bounds = getBounds();
		return new Point((int) bounds.getCenterX(), (int) bounds.getCenterY());
	}

	/**
	 * Gets the (abstract) location.
	 */
	public Point getAbsoluteLocation() {
		Point location = getLocation();
		translateToAbsolute(location);
		return location;
	}

	/**
	 * Gets the (abstract) center location.
	 */
	public Point getAbsoluteCenterLocation() {
		Point location = getCenterLocation();
		translateToAbsolute(location);
		return location;
	}

	/**
	 * Moves the Location. 
	 * @param p
	 */
	public void moveLocation(int mx, int my) {
		Point location = getLocation();
		location.x += mx;
		location.y += my;
		setLocation(location);
	}

	/**
	 * Sets the Location by center Location. 
	 * @param p
	 */
	public void setLocationByCenter(Point location) {
		setLocationByCenter(location.x, location.y);
	}

	/**
	 * Sets the Location by center Location.
	 * @param p
	 */
	public void setLocationByCenter(int cx, int cy) {
		int x = cx - this.getWidth() / 2;
		int y = cy - this.getHeight() / 2;
		setLocation(x, y);
	}

	/***********************************
	 * Size Utility
	 ***********************************/

	public void setSizeToPreffered() {
		this.setSize(this.getPreferredSize());
	}

	/***********************************
	 * For Paint Strategy.
	 ***********************************/

	public void repaint() {
		if (getParentVisual() != null
			&& getParentVisual() instanceof LightWeightLayerVisualComponent) {
			getParentVisual().repaint();
		} else {
			super.repaint();
		}
	}

	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if (getParentVisual() != null
			&& getParentVisual() instanceof LightWeightLayerVisualComponent) {
			getParentVisual().repaint();
		}
	}

	public void paint(Graphics g) {
		Graphics2D g2 = (Graphics2D) g;
		g2.setRenderingHint(
			RenderingHints.KEY_ANTIALIASING,
			RenderingHints.VALUE_ANTIALIAS_ON);
		super.paint(g);
	}

	/* (non-Javadoc)
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		if (isOpaque()) {
			g.setColor(getBackground());
			g.fillRect(0, 0, getWidth(), getHeight());
		}
		paintVisual((Graphics2D) g);
	}

	/* (non-Javadoc)
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	protected void paintVisual(Graphics2D g) {
	}

	/**
	 * Method getParentVisual.
	 * @return
	 */
	public VisualComponent getParentVisual() {
		return parentVisual;
	}

	/**
	 * Method setParentVisual.
	 * @param component
	 */
	public void setParentVisual(VisualComponent component) {
		parentVisual = component;
	}

}
