/*
 * CThread.java
 * Created on 2003/10/17
 * 
 * Copyright (c) 2003 CreW Project. All rights reserved.
 */
package jp.ac.keio.sfc.crew.thread;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Class CThread.
 * 
 * @author macchan
 * @version $Id: CThread.java,v 1.1 2003/12/11 10:26:50 macchan Exp $
 */
public abstract class CThread implements Runnable {

	/**************************************
	 * Constants.
	 **************************************/

	public static final int STARTING = 1;
	public static final int RUNNING = 2;
	public static final int STOPPING = 3;
	public static final int STOPPED = 4;

	/**************************************
	 * Instance Variables.
	 **************************************/

	private Thread thread;

	private Object waitLock = new Object();
	private Object startLock = new Object();
	private Object threadLock = new Object();

	private int interval = 1000;

	private int state = STOPPED;

	private List listeners = new ArrayList();

	/**
	 * Constructor for CThread.
	 */
	public CThread() {
		super();
	}

	/**************************************
	 * Run Strategy.
	 **************************************/

	public boolean start() {
		synchronized (startLock) {
			if (getState() == STOPPED) {
				startInternal();
				return true;
			} else {
				return false;
			}
		}
	}

	private void startInternal() {
		setState(STARTING);
		thread = new Thread(this);
		thread.start();
	}

	public boolean startAndWait() {
		try {
			synchronized (waitLock) {
				if (start()) {
					waitLock.wait();
					return true;
				} else {
					return false;
				}
			}
		} catch (InterruptedException ex) {
			throw new RuntimeException("Interrupted while thread waiting", ex);
		}
	}

	public boolean stop() {
		try {
			synchronized (startLock) {
				if (getState() == STARTING) {
					startLock.wait();
				}
			}
			synchronized (threadLock) {
				if (getState() == RUNNING) {
					setState(STOPPING);
					threadLock.notify();
					return true;
				} else {
					return false;
				}
			}
		} catch (InterruptedException ex) {
			throw new RuntimeException("Interrupted while thread waiting", ex);
		}
	}

	public boolean stopAndWait() {
		try {
			synchronized (waitLock) {
				if (stop()) {
					waitLock.wait();
					return true;
				} else {
					return false;
				}
			}
		} catch (InterruptedException ex) {
			throw new RuntimeException("Interrupted while thread waiting", ex);
		}
	}

	/**************************************
	 * Run Strategy.
	 **************************************/

	public final void run() {
		synchronized (threadLock) {
			initializeThread();
			prepareStart();

			try {
				while (getState() == RUNNING) {
					threadLock.wait(interval);
					step();
				}
			} catch (Throwable ex) {
				this.fireExceptionOccured(ex);
			}

			prepareStop();
			terminateThread();
		}
	}

	private void initializeThread() {
		synchronized (startLock) {
			setState(RUNNING);
			startLock.notifyAll();
		}
	}

	private void terminateThread() {
		synchronized (waitLock) {
			thread = null;
			setState(STOPPED);
			waitLock.notifyAll();
		}
	}

	protected abstract void step();
	protected abstract void prepareStart();
	protected abstract void prepareStop();

	/**************************************
	 * For State
	 **************************************/

	/**
	 * Method getState.
	 * @return
	 */
	public int getState() {
		return state;
	}

	/**
	 * Method setState.
	 * @param i
	 */
	protected void setState(int state) {
		if (this.state != state) {
			this.state = state;
			this.fireStateChanged();
		}
	}

	/**************************************
	 * For Interval
	 **************************************/

	/**
	 * Method getInterval.
	 * @return
	 */
	public int getInterval() {
		return interval;
	}

	/**
	 * Method setInterval.
	 * @param i
	 */
	public void setInterval(int interval) {
		if (interval <= 0) {
			throw new IllegalArgumentException(
				"Interval must set upper zero. interval = " + interval);
		}
		this.interval = interval;
	}

	/**************************************
	 * For CThreadListener
	 **************************************/

	public synchronized void addThreadListener(CThreadListener listener) {
		this.listeners.add(listener);
	}

	public synchronized void removeThreadListener(CThreadListener listener) {
		this.listeners.remove(listener);
	}

	protected synchronized void fireStateChanged() {
		CThreadEvent event = new CThreadEvent(this, this.getState());
		for (Iterator i = listeners.iterator(); i.hasNext();) {
			CThreadListener listener = (CThreadListener) i.next();
			listener.stateChanged(event);
		}
	}

	protected synchronized void fireExceptionOccured(Throwable ex) {
		CThreadEvent event = new CThreadEvent(this, this.getState(), ex);
		for (Iterator i = listeners.iterator(); i.hasNext();) {
			CThreadListener listener = (CThreadListener) i.next();
			listener.exceptionOccured(event);
		}
	}

}
