/*
 * ELineConnectionVisualComponent.java
 * Created on 2003/10/09
 * 
 * Copyright (c) 2003 CreW Project. All rights reserved.
 */
package jp.ac.keio.sfc.crew.view.sgef.ext.visuals;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;

import jp.ac.keio.sfc.crew.swing.visuals.Decoration;
import jp.ac.keio.sfc.crew.swing.visuals.VisualComponent;

/**
 * Class ELineConnectionVisualComponent.
 * 
 * @author macchan
 * @version $Id: ELineConnectionVisualComponent.java,v 1.1 2003/12/11 10:26:57 macchan Exp $
 */
public class ELineConnectionVisualComponent
	extends EConnectionVisualComponent {

	/***************************************
	 * Instance Variables.
	 ***************************************/

	private Line2D line = new Line2D.Double();
	private BasicStroke stroke = new BasicStroke(1);

	private int multipleLineInterval = 0;
	private AffineTransform transform = new AffineTransform();

	private Decoration sourceDecoration = new Decoration();
	private Decoration targetDecoration = new Decoration();

	/***************************************
	 * Constructors.
	 ***************************************/

	/**
	 * Constructor for ELineConnectionVisualComponent.
	 */
	public ELineConnectionVisualComponent() {
		setForeground(Color.black);
	}

	/***************************************
	 * Update Strategy.
	 ***************************************/

	public void refreshVisual() {
		refreshLine();
		refreshDecorations();
		refreshConstraints();

		repaint();
	}

	protected void refreshLine() {
		//Pre process
		VisualComponent source = getSource();
		VisualComponent target = getTarget();

		//Creates ChopBox Anchor
		Point sCenterLoc = source.getAbsoluteCenterLocation();
		Point tCenterLoc = target.getAbsoluteCenterLocation();

		Rectangle sBounds =
			new Rectangle(source.getAbsoluteLocation(), source.getSize());
		Rectangle tBounds =
			new Rectangle(target.getAbsoluteLocation(), target.getSize());

		Point sAnchor = computeChopBoxAnchor(sBounds, tCenterLoc);
		Point tAnchor = computeChopBoxAnchor(tBounds, sCenterLoc);

		//Sets the Line.
		line.setLine(sAnchor, tAnchor);
	}

	protected void refreshDecorations() {
		double direction = computeDirection();
		this.sourceDecoration.setDirection(direction + Math.PI);
		this.sourceDecoration.setLocation(line.getX1(), line.getY1());
		this.targetDecoration.setDirection(direction);
		this.targetDecoration.setLocation(line.getX2(), line.getY2());
	}

	protected void refreshConstraints() {
		int count = getCount();
		int index = getIndex();

		AffineTransform transform = new AffineTransform();
		if (count <= 1) {
			//do nothing
		} else {
			//Calc Interval
			double interval = getMultipleLineInterval();
			//in case of the connection count is even			
			if (count % 2 == 0) {
				if (index < 2) {
					interval = interval / 2d;
				} else {
					interval = interval / 2d + interval * (index / 2);
				}
			}
			//in case of the connection count is odd
			else {
				if (index == 0) {
					interval = 0;
				} else {
					interval = interval * ((index + 1) / 2);
				}
			}

			if (index % 2 == 0) {
			} else {
				interval = interval * -1;
			}

			//Calc Direction
			double direction = computeDirection();
			//once set all direction to right
			if (direction > Math.PI / 2d) {
				direction = direction + Math.PI;
			} else if (direction < -Math.PI / 2d) {
				direction = direction - Math.PI;
			}
			//make direction perpendicularly
			direction = direction + Math.PI / 2d;

			//Calc distance to translate
			double tx = Math.cos(direction) * interval;
			double ty = Math.sin(direction) * interval;

			transform.translate(tx, ty);
		}
		setTransform(transform);

	}

	protected Point computeChopBoxAnchor(Rectangle r, Point reference) {
		float centerX = (float) r.x + 0.5f * (float) r.width;
		float centerY = (float) r.y + 0.5f * (float) r.height;

		//This avoids divide-by-zero
		if (r.isEmpty()
			|| (reference.x == (int) centerX && reference.y == (int) centerY)) {
			return new Point((int) centerX, (int) centerY);
		}

		float dx = (float) reference.x - centerX;
		float dy = (float) reference.y - centerY;

		//r.width, r.height, dx, and dy are guaranteed to be non-zero. 
		float scale =
			0.5f / Math.max(Math.abs(dx) / r.width, Math.abs(dy) / r.height);

		dx *= scale;
		dy *= scale;
		centerX += dx;
		centerY += dy;

		return new Point(Math.round(centerX), Math.round(centerY));
	}

	protected double computeDirection() {
		double dx = line.getX2() - line.getX1();
		double dy = line.getY2() - line.getY1();
		return Math.atan2(dy, dx);
	}

	/***************************************
	 * Paint Strategy
	 ***************************************/

	/* (non-Javadoc)
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	protected void paintVisual(Graphics2D g) {
		//pre process
		g.setColor(getForeground());
		Stroke defaultStroke = g.getStroke();
		g.setStroke(getStroke());
		AffineTransform defaultTransform = g.getTransform();
		AffineTransform transform = new AffineTransform(getTransform());
		transform.concatenate(g.getTransform());
		g.setTransform(transform);

		//draw strategy
		g.draw(line);
		g.fill(sourceDecoration.getShape());
		g.fill(targetDecoration.getShape());
		GeneralPath path = new GeneralPath(line);

		//post process
		g.setStroke(defaultStroke);
		g.setTransform(defaultTransform);
	}

	/***************************************
	 * Line
	 ***************************************/

	/**
	 * Method getLine.
	 * @return
	 */
	public Line2D getLine() {
		return line;
	}

	/**
	 * Method getLine.
	 * @return
	 */
	public Rectangle getLineBounds() {
		return transform.createTransformedShape(line).getBounds();
	}

	/***************************************
	 * Getters and Setters.
	 ***************************************/

	/**
	 * Method getTransform.
	 * @return
	 */
	public AffineTransform getTransform() {
		return transform;
	}

	/**
	 * Method setTransform.
	 * @param transform
	 */
	public void setTransform(AffineTransform transform) {
		this.transform = transform;
	}

	/**
	 * @return
	 */
	public BasicStroke getStroke() {
		return stroke;
	}

	/**
	 * @param stroke
	 */
	public void setStroke(BasicStroke stroke) {
		this.stroke = stroke;
	}

	/**
	 * @return
	 */
	public Decoration getSourceDecoration() {
		return sourceDecoration;
	}

	/**
	 * @return
	 */
	public Decoration getTargetDecoration() {
		return targetDecoration;
	}

	/**
	 * @param decoration
	 */
	public void setSourceDecoration(Decoration decoration) {
		sourceDecoration = decoration;
	}

	/**
	 * @param decoration
	 */
	public void setTargetDecoration(Decoration decoration) {
		targetDecoration = decoration;
	}

	/**
	 * Method getMultipleLineInterval.
	 * @return
	 */
	public int getMultipleLineInterval() {
		return multipleLineInterval;
	}

	/**
	 * Method setMultipleLineInterval.
	 * @param i
	 */
	public void setMultipleLineInterval(int interval) {
		multipleLineInterval = interval;
	}

}
