/*
 * CommunicationViewerFrame.java
 * Copyright (c) 2002 Boxed-Economy Project.  All rights reserved.
 */
package org.boxed_economy.components.commviewer;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Image;
import java.util.Iterator;
import java.util.ResourceBundle;

import javax.swing.ImageIcon;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;

import jp.ac.keio.sfc.crew.swing.ImageProvider;
import jp.ac.keio.sfc.crew.swing.SwingUtil;
import jp.ac.keio.sfc.crew.swing.visuals.BlowVisualComponent;
import jp.ac.keio.sfc.crew.swing.visuals.LabelVisualComponent;
import jp.ac.keio.sfc.crew.swing.visuals.VisualComponent;
import jp.ac.keio.sfc.crew.view.sgef.animation.LinePath;
import jp.ac.keio.sfc.crew.view.sgef.animation.MovingAnimationThreadWithLine;
import jp.ac.keio.sfc.crew.view.sgef.editparts.SEditPart;
import jp.ac.keio.sfc.crew.view.sgef.ext.editparts.EGraphicalEditPart;
import jp.ac.keio.sfc.crew.view.sgef.ext.visuals.ELineConnectionVisualComponent;

import org.apache.log4j.Logger;
import org.boxed_economy.besp.model.ModelContainerEvent;
import org.boxed_economy.besp.model.ModelContainerListener;
import org.boxed_economy.besp.model.fmfw.Agent;
import org.boxed_economy.besp.model.fmfw.Behavior;
import org.boxed_economy.besp.model.fmfw.Channel;
import org.boxed_economy.besp.model.fmfw.Goods;
import org.boxed_economy.besp.model.fmfw.Relation;
import org.boxed_economy.besp.model.fmfw.World;
import org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent;
import org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener;
import org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorEvent;
import org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorListener;
import org.boxed_economy.besp.model.fmfw.update.UpdateChannelEvent;
import org.boxed_economy.besp.model.fmfw.update.UpdateChannelListener;
import org.boxed_economy.besp.model.fmfw.update.UpdateRelationEvent;
import org.boxed_economy.besp.model.fmfw.update.UpdateRelationListener;
import org.boxed_economy.besp.model.fmfw.update.UpdateWorldEvent;
import org.boxed_economy.besp.model.fmfw.update.UpdateWorldListener;
import org.boxed_economy.besp.presentation.guifw.AbstractInternalFrameComponent;
import org.boxed_economy.components.commviewer.controller.MainSettingPanel;
import org.boxed_economy.components.commviewer.editparts.ChannelEditPart;
import org
	.boxed_economy
	.components
	.commviewer
	.editparts
	.CommunicationViewerRootEditPart;
import org.boxed_economy.components.commviewer.editparts.RelationEditPart;

/**
 * Class Communication Viewer.
 * @author aoyama
 * @version $Id: CommunicationViewerFrame.java,v 1.4 2004/03/21 12:08:03 macchan Exp $
 */
public class CommunicationViewerFrame
	extends AbstractInternalFrameComponent
	implements
		ModelContainerListener,
		UpdateWorldListener,
		UpdateAgentListener,
		UpdateBehaviorListener,
		UpdateChannelListener,
		UpdateRelationListener {

	/******************************************
	 * Constants.
	 ******************************************/

	private static final Logger logger =
		Logger.getLogger(CommunicationViewerFrame.class.getName());

	public static ResourceBundle resource = CommunicationViewerPlugin.resource;

	/******************************************
	 * Instance Variables.
	 ******************************************/

	private CommunicationViewer commViewer;
	private MainSettingPanel controller;

	private boolean dirty = false;

	/******************************************
	 * Initialize & Terminate.
	 ******************************************/

	/**
	 * @see org.boxed_economy.besp.presentation.fw.PresentationComponent#button_initialize()
	 */
	public void initialize() {
		initializeComponents();
		hookUpdateEvents();
	}

	private void initializeComponents() {
		Container container = this.getContentPane();

		//		commViewer = new CommunicationViewer();		
		//		container.add(commViewer.getEditor(), BorderLayout.CENTER);
		//
		//		controller = new MainSettingPanel(commViewer);
		//		container.add(controller, BorderLayout.EAST);

		JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		split.setDividerLocation(600);
		container.add(split);

		commViewer = new CommunicationViewer();
		split.add(commViewer.getEditor());

		JScrollPane scroll = new JScrollPane();
		split.add(scroll);
		controller = new MainSettingPanel(commViewer);
		scroll.getViewport().add(controller);

		container.validate();
		container.repaint();
	}

	/**
	 * EChE̐ݒ܂
	 * ftHg̎łBCɓȂ΁AI[o[Ch邩Aݒ肵Ȃ܂
	 */
	protected void initializeFrame() {
		this.setTitle(resource.getString("Name_Component"));
		this.setLocation(10, 10);
		this.setSize(800, 600);
	}

	/**
	 * @see org.boxed_economy.besp.presentation.fw.PresentationComponent#terminate()
	 */
	public void terminate() {
		unhookUpdateEvents();
		commViewer.stopAllThread();
	}

	public void hookUpdateEvents() {
		this.addUpdateWorldListener(this);
		this.addUpdateAgentListener(this);
		this.addUpdateBehaviorListener(this);
		this.addUpdateChannelListener(this);
		this.addUpdateRelationListener(this);
	}

	public void unhookUpdateEvents() {
		this.removeUpdateWorldListener(this);
		this.removeUpdateAgentListener(this);
		this.removeUpdateBehaviorListener(this);
		this.removeUpdateChannelListener(this);
		this.removeUpdateRelationListener(this);
	}

	/******************************************
	 * Implements for ModelContainerListener
	 ******************************************/

	/**
	 * @see org.boxed_economy.besp.model.ModelContainerListener#prepareWorldOpen(ModelContainerEvent)
	 */
	public void prepareWorldOpen(ModelContainerEvent ev) {
		World world = ev.getWorld();
		commViewer.setWorld(world);
		if (!isVisible()) {
			dirty = true;
		}
	}

	/**
	 * @see org.boxed_economy.besp.model.ModelContainerListener#worldOpened(ModelContainerEvent)
	 */
	public void worldOpened(ModelContainerEvent ev) {
	}

	/**
	 * @see org.boxed_economy.besp.model.ModelContainerListener#prepareWorldClose(ModelContainerEvent)
	 */
	public void prepareWorldClose(ModelContainerEvent ev) {
		commViewer.setWorld(null);
	}

	/**
	 * @see org.boxed_economy.besp.model.ModelContainerListener#worldClosed(ModelContainerEvent)
	 */
	public void worldClosed(ModelContainerEvent ev) {
	}

	public void show() {
		super.show();
		if (dirty) {
			refreshAgentLocations();
			dirty = false;
		}
	}

	private void refreshAgentLocations() {
		synchronized (getTreeLock()) {
			World world = getWorld();
			for (Iterator i = world.getAllAgents().iterator(); i.hasNext();) {
				Agent agent = (Agent) i.next();
				adjustAgentLocation(world, agent);
			}
		}
	}

	/******************************************
	 * Update Strategy.
	 ******************************************/

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateWorldListener#agentCreated(org.boxed_economy.besp.model.fmfw.update.UpdateWorldEvent)
	 */
	public void agentCreated(UpdateWorldEvent e) {
		synchronized (getTreeLock()) {
			World world = (World) e.getSource();
			Agent agent = (Agent) e.getObject();
			refresh(world);
			adjustAgentLocation(world, agent);
		}
	}

	private void adjustAgentLocation(World world, Agent agent) {
		VisualComponent agentVisual = getVisual(agent);
		VisualComponent worldVisual = getVisual(world);

		int hGap = 50;
		int vGap = 50;
		int wWidth = Math.max((int) worldVisual.getWidth(), 500);
		int wHeight = Math.max((int) worldVisual.getHeight(), 500);
		int aWidth = agentVisual.getWidth();
		int aHeight = agentVisual.getHeight();
		int vMargin = aWidth;
		int hMargin = aHeight;

		for (int y = vGap; y < wHeight; y += aHeight + vMargin) {
			for (int x = hGap; x < wWidth; x += aWidth + hMargin) {
				Component c = worldVisual.getComponentAt(x, y);
				if (c == worldVisual) {
					agentVisual.setLocation(x, y);
					commViewer
						.getEditor()
						.getEditPart(agent)
						.refreshVisualRecursively();
					return;
				}
			}
		}
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateWorldListener#agentDestroyed(org.boxed_economy.besp.model.fmfw.update.UpdateWorldEvent)
	 */
	public void agentDestroyed(UpdateWorldEvent e) {
		World world = (World) e.getSource();
		refresh(world);
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateWorldListener#goodsConsumed(org.boxed_economy.besp.model.fmfw.update.UpdateWorldEvent)
	 */
	public void goodsConsumed(UpdateWorldEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateWorldListener#goodsCreated(org.boxed_economy.besp.model.fmfw.update.UpdateWorldEvent)
	 */
	public void goodsCreated(UpdateWorldEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#behaviorAdded(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void behaviorAdded(UpdateAgentEvent e) {
		Agent agent = (Agent) e.getSource();
		refresh(agent);
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#behaviorRemoved(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void behaviorRemoved(UpdateAgentEvent e) {
		Agent agent = (Agent) e.getSource();
		refresh(agent);
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#goodsAdded(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void goodsAdded(UpdateAgentEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#goodsRemoved(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void goodsRemoved(UpdateAgentEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#informationAdded(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void informationAdded(UpdateAgentEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#informationRemoved(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void informationRemoved(UpdateAgentEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#relationAdded(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void relationAdded(UpdateAgentEvent e) {
		Relation relation = (Relation) e.getObject();
		Agent agent = (Agent) e.getSource();

		//refresh
		synchronized (getTreeLock()) {
			refresh(agent);
			getVisual(relation).setVisible(false);
		}

		if (commViewer.isAnimationEnable()) {
			//MovingLine Animation
			prepareMoveLineAnimation(
				commViewer.getRelationThread(),
				RelationEditPart.createRelationVisual(),
				agent,
				relation,
				false);
			commViewer.getRelationThread().startAndWait();
		}

		//post process
		terminateAnimation();
		getVisual(relation).setVisible(true);
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateAgentListener#relationRemoved(org.boxed_economy.besp.model.fmfw.update.UpdateAgentEvent)
	 */
	public void relationRemoved(UpdateAgentEvent e) {
		Relation relation = (Relation) e.getObject();
		Agent agent = (Agent) e.getSource();
		Agent targetAgent = relation.getTarget();

		getVisual(relation).setVisible(false);

		if (commViewer.isAnimationEnable()) {
			//MovingLine Animation
			prepareMoveLineAnimation(
				commViewer.getRelationThread(),
				RelationEditPart.createRelationVisual(),
				agent,
				relation,
				true);
			commViewer.getRelationThread().startAndWait();
		}

		//refresh
		refresh(agent);
		refresh(targetAgent);

		//post process
		terminateAnimation();
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorListener#behaviorUpdated(org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorEvent)
	 */
	public void stateChanged(UpdateBehaviorEvent e) {
		Behavior behavior = (Behavior) e.getSource();
		refresh(behavior);

		if (commViewer.isAnimationEnable()
			&& isStateVisible()
			&& isBehaviorVisible()) {
			Color color = SwingUtil.createAlphaedColor(Color.RED, 100);
			commViewer.getStateThread().setColor(color);
			commViewer.getStateThread().setParent(
				(EGraphicalEditPart) commViewer.getEditor().getEditPart(
					behavior));
			commViewer.getStateThread().startAndWait();
		}

	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorListener#prepareStateChange(org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorEvent)
	 */
	public void transitionStarted(UpdateBehaviorEvent e) {
		Behavior behavior = (Behavior) e.getSource();
		refresh(behavior);

		if (commViewer.isAnimationEnable()
			&& isStateVisible()
			&& isBehaviorVisible()) {
			Color color = SwingUtil.createAlphaedColor(Color.RED, 100);
			commViewer.getStateThread().setColor(color);
			commViewer.getStateThread().setParent(
				(EGraphicalEditPart) commViewer.getEditor().getEditPart(
					behavior));
			commViewer.getStateThread().startAndWait();
		}
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorListener#prepareTimeEventReceive(org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorEvent)
	 */
	public void prepareTimeEventReceive(UpdateBehaviorEvent e) {
		Behavior behavior = (Behavior) e.getSource();
		refresh(behavior);

		if (commViewer.isAnimationEnable() && isBehaviorVisible()) {
			Color color = SwingUtil.createAlphaedColor(Color.BLUE, 100);
			commViewer.getClockThread().setColor(color);
			commViewer.getClockThread().setParent(
				(EGraphicalEditPart) commViewer.getEditor().getEditPart(
					behavior));
			commViewer.getClockThread().startAndWait();
		}
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorListener#timeEventReceived(org.boxed_economy.besp.model.fmfw.update.UpdateBehaviorEvent)
	 */
	public void timeEventReceived(UpdateBehaviorEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateChannelListener#goodsReceive(org.boxed_economy.besp.model.fmfw.update.UpdateChannelEvent)
	 */
	public void goodsReceive(UpdateChannelEvent e) {
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateChannelListener#goodsSent(org.boxed_economy.besp.model.fmfw.update.UpdateChannelEvent)
	 */
	public void goodsSent(UpdateChannelEvent e) {
		//pre process
		Channel channel = (Channel) e.getSource();
		Relation relation = channel.getParentRelation();
		Goods goods = e.getGoods();

		if (commViewer.isAnimationEnable()) {
			//Blinking Animation.
			prepareBlinkingAnimation(channel);
			commViewer.getBlinkingThread().start();

			//GoodsVisual Moving Animation.
			VisualComponent goodsVisual = createGoodsVisual(goods);
			boolean reverse =
				e.getSourceBehavior().getAgent() != relation.getSource();
			prepareMoveAnimation(goodsVisual, relation, reverse);
			commViewer.getGoodsSendingThread().startAndWait();
		}

		//post process
		terminateAnimation();
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateRelationListener#channelClosed(org.boxed_economy.besp.model.fmfw.update.UpdateRelationEvent)
	 */
	public void channelClosed(UpdateRelationEvent e) {
		//pre process
		Relation relation = (Relation) e.getSource();
		Behavior behaviorA = e.getChannel().getBehaviorA();
		Behavior behaviorB = e.getChannel().getBehaviorB();
		Channel channel = e.getChannel();

		getVisual(channel).setVisible(false);

		if (commViewer.isAnimationEnable() && isBehaviorVisible()) {
			//Blinking Animation.
			prepareBlinkingAnimation(relation);
			commViewer.getBlinkingThread().start();

			//MovingLine Animation
			prepareMoveLineAnimation(
				commViewer.getChannelThread(),
				ChannelEditPart.createChannelVisual(),
				behaviorA,
				channel,
				true);
			commViewer.getChannelThread().startAndWait();
		}

		//refresh
		refresh(behaviorA);
		refresh(behaviorB);

		//post process
		terminateAnimation();
	}

	/**
	 * @see org.boxed_economy.besp.model.fmfw.update.UpdateRelationListener#channelOpened(org.boxed_economy.besp.model.fmfw.update.UpdateRelationEvent)
	 */
	public void channelOpened(UpdateRelationEvent e) {
		//pre process
		Relation relation = (Relation) e.getSource();
		Behavior behaviorA = e.getChannel().getBehaviorA();
		Behavior behaviorB = e.getChannel().getBehaviorB();
		Channel channel = e.getChannel();

		//refresh
		synchronized (getTreeLock()) {
			refresh(behaviorA);
			refresh(behaviorB);
			getVisual(channel).setVisible(false);
		}

		if (commViewer.isAnimationEnable() && isBehaviorVisible()) {
			//Blinking Animation.
			prepareBlinkingAnimation(relation);
			commViewer.getBlinkingThread().start();

			//MovingLine Animation
			prepareMoveLineAnimation(
				commViewer.getChannelThread(),
				ChannelEditPart.createChannelVisual(),
				behaviorA,
				channel,
				false);
			commViewer.getChannelThread().startAndWait();
		}

		//post process
		terminateAnimation();
		getVisual(channel).setVisible(true);
	}

	/********************************************
	 * Sub Utility Methods.
	 ********************************************/

	private boolean isStateVisible() {
		SEditPart root = commViewer.getEditor().getRootEditPart();
		if (root == null) {
			return false;
		}
		CommunicationViewerRootEditPart rootEditPart =
			(CommunicationViewerRootEditPart) root;
		return rootEditPart.isBehaviorStateVisible();
	}

	private boolean isBehaviorVisible() {
		return isVisible(
			CommunicationViewerRootEditPart.LAYER_BEHAVIOR_CONTAINERS);
	}

	private boolean isVisible(Object layerKey) {
		SEditPart root = commViewer.getEditor().getRootEditPart();
		if (root == null) {
			return false;
		}
		CommunicationViewerRootEditPart rootEditPart =
			(CommunicationViewerRootEditPart) root;
		VisualComponent layer = rootEditPart.getLayer(layerKey);
		if (layer == null) {
			return false;
		}
		return layer.isVisible();
	}

	private VisualComponent createGoodsVisual(Goods goods) {
		if (goods.hasAttachment()) {
			return createBlowVisual(goods);
		} else {
			return createLabelVisual(goods);
		}
	}

	private LabelVisualComponent createLabelVisual(Goods goods) {
		LabelVisualComponent label = new LabelVisualComponent();
		label.setText(goods.toString());
		Image image =
			ImageProvider.getInstance().getImage(
				"org/boxed_economy/components/commviewer/visuals/goods.gif");
		label.setIcon(new ImageIcon(image));
		label.setLabelOpaque(true);
		label.setSizeToPreffered();
		return label;
	}

	private BlowVisualComponent createBlowVisual(Goods goods) {
		BlowVisualComponent blow = new BlowVisualComponent();
		blow.setText(goods.getAttachment().toString());
		blow.setSizeToPreffered();
		return blow;
	}

	private void prepareMoveAnimation(
		VisualComponent visual,
		Object pathModel,
		boolean reverse) {
		ELineConnectionVisualComponent pathConnection =
			(ELineConnectionVisualComponent) getVisual(pathModel);
		LinePath path = new LinePath(pathConnection.getLine());
		path.setReverse(reverse);

		visual.setLocationByCenter(path.getPoint(0));
		addAnimationVisual(visual);

		commViewer.getGoodsSendingThread().setVisual(visual);
		commViewer.getGoodsSendingThread().setPath(path);
	}

	private void prepareBlinkingAnimation(Object blinkingModel) {
		commViewer.getBlinkingThread().setVisual(getVisual(blinkingModel));
	}

	private void prepareMoveLineAnimation(
		MovingAnimationThreadWithLine thread,
		ELineConnectionVisualComponent dummyLine,
		Object sourceModel,
		Object pathModel,
		boolean reverse) {
		VisualComponent dummy = new VisualComponent();
		dummyLine.setSource(getVisual(sourceModel));
		dummyLine.setTarget(dummy);

		ELineConnectionVisualComponent channelLine =
			(ELineConnectionVisualComponent) getVisual(pathModel);
		LinePath path = new LinePath(channelLine.getLine());
		dummyLine.setConstraints(
			channelLine.getIndex(),
			channelLine.getCount());
		path.setReverse(reverse);

		addAnimationVisual(dummy);
		addAnimationVisual(dummyLine);

		thread.setVisual(dummy);
		thread.setPath(path);
		thread.setConnection(dummyLine);
	}

	public VisualComponent getVisual(Object model) {
		return commViewer.getEditor().getEditPart(model).getVisual();
	}

	public void refresh(Object model) {
		synchronized (commViewer.getEditor().getTreeLock()) {
			commViewer.getEditor().getEditPart(model).refresh();
			commViewer.getEditor().getContentsLayer().validate();
			commViewer.getEditor().getContentsLayer().repaint();
		}
	}

	public void addAnimationVisual(VisualComponent visual) {
		synchronized (commViewer.getEditor().getTreeLock()) {
			commViewer.getEditor().getAnimationLayer().add(visual);
			commViewer.getEditor().getAnimationLayer().validate();
		}
	}

	public void terminateAnimation() {
		commViewer.stopAllThread();
		synchronized (commViewer.getEditor().getTreeLock()) {
			commViewer.getEditor().getAnimationLayer().removeAll();
			commViewer.getEditor().repaint();
		}
	}

}
