/*
 * RelationViewCanvas.java
 * Copyright (c) 2002 Boxed-Economy Project.  All rights reserved.
 */
package org.boxed_economy.components.relationview.canvas;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JPanel;

import org.boxed_economy.besp.model.fmfw.Agent;
import org.boxed_economy.besp.model.fmfw.AgentType;
import org.boxed_economy.besp.model.fmfw.Relation;
import org.boxed_economy.besp.model.fmfw.RelationType;
import org.boxed_economy.besp.model.fmfw.World;
import org.boxed_economy.components.relationview.AgentNode;
import org.boxed_economy.components.relationview.TypeSelection;

/**
 * @author macchan
 * @version $Id: AbstractRelationViewCanvas.java,v 1.2 2003/07/11 09:56:49 box Exp $
 */
public abstract class AbstractRelationViewCanvas extends JPanel {

	/*******************************************
	 * Constant
	 *******************************************/

	private static final Point DEFAULT_LOCATION = new Point(10, 10);

	/*******************************************
	 * Instance Variable
	 *******************************************/

	private World world = null;

	private TypeSelection agentTypeSelection = null;
	private TypeSelection relationTypeSelection = null;

	protected Map nodes = new HashMap();
	private Map nodesByType = new HashMap();

	/*******************************************
	 * Constructor
	 *******************************************/

	/**
	 * Constructor for RelationViewCanvas.
	 */
	public AbstractRelationViewCanvas(
		TypeSelection agentSelection,
		TypeSelection relationSelection) {

		this.agentTypeSelection = agentSelection;
		this.relationTypeSelection = relationSelection;
		initialize();
	}

	/*******************************************
	 * Initializer
	 *******************************************/

	private void initialize() {
		this.setLayout(null);
		this.setBackground(Color.white);
	}

	/*******************************************
	 * override for revalidate timing
	 *******************************************/

	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if (visible) {
			updateAll();
		}
	}

	public void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);
		this.layoutNodes();
	}

	/*******************************************
	 * update
	 *******************************************/

	/**
	 * update for public
	 */
	public void updateAll() {
		if (world != null && isEnabled()) {
			synchronized (getTreeLock()) {
				updateNodes();
				layoutNodes();
				repaint();
			}
		}
	}

	/**
	 * update nodes map
	 */
	private void updateNodes() {
		if (this.world != null) {
			List suitableAgents = getAllSuitableAgents();
			Set availableAgents = nodes.keySet();

			{ //suitable - available = add
				List addAgents = new ArrayList(suitableAgents);
				addAgents.removeAll(availableAgents);

				Iterator i = addAgents.iterator();
				while (i.hasNext()) {
					Agent agent = (Agent) i.next();
					addAgentNode(agent);
				}
			}

			{ //available - suitable  = remove
				List removeAgents = new ArrayList(availableAgents);
				removeAgents.removeAll(suitableAgents);

				Iterator i = removeAgents.iterator();
				while (i.hasNext()) {
					Agent agent = (Agent) i.next();
					removeAgentNode(agent);
				}
			}
		}
	}

	/**
	 * re-layout all nodes
	 * for override
	 */
	protected void layoutNodes() {
	}

	/*******************************************
	 * management nodes
	 *******************************************/

	/**
	 * get all suitable agents
	 */
	private List getAllSuitableAgents() {
		List agentTypes = agentTypeSelection.getSelectedTypes();
		List agents = new ArrayList();
		Iterator i = agentTypes.iterator();
		while (i.hasNext()) {
			AgentType agentType = (AgentType) i.next();
			agents.addAll(getWorld().getAgents(agentType));
		}
		return agents;
	}

	/**
	 * add agent node
	 */
	protected AgentNode addAgentNode(Agent agent) {
		AgentNode node = new AgentNode(agent);
		node.setLocation(DEFAULT_LOCATION);
		add(node);
		nodes.put(agent, node);
		getNodes(agent.getType()).add(node);
		return node;
	}

	/**
	 * remove agent node
	 */
	protected AgentNode removeAgentNode(Agent agent) {
		AgentNode node = (AgentNode) nodes.get(agent);
		remove(node);
		nodes.remove(agent);
		getNodes(agent.getType()).remove(node);
		return node;
	}

	/**
	 * get all agents (which have already node)
	 */
	public Set getAllAgents() {
		return nodes.keySet();
	}

	/**
	 * get all nodes
	 */
	public Collection getAllNodes() {
		return nodes.values();
	}

	/**
	 * get all nodes by order
	 */
	public List getOrderedAllNodes() {
		List allNodes = new ArrayList();

		Iterator i = agentTypeSelection.getSelectedTypes().iterator();
		while (i.hasNext()) {
			AgentType type = (AgentType) i.next();
			allNodes.addAll(getNodes(type));
		}

		return allNodes;
	}

	/**
	 * get nodes by agent type
	 */
	public List getNodes(AgentType type) {
		if (!nodesByType.containsKey(type)) {
			nodesByType.put(type, new ArrayList());
		}
		return (List) nodesByType.get(type);
	}

	/****************************************
	 * paint
	 ****************************************/

	/**
	 * @see javax.swing.JComponent#paintComponent(Graphics)
	 */
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		this.paintAgents(g);
	}

	/**
	 * paint all labels
	 */
	public void paintAgents(Graphics g) {
		Set agents = nodes.keySet();
		Iterator i = agents.iterator();
		while (i.hasNext()) {
			Agent agent = (Agent) i.next();
			paintLabel(g, agent);
			paintRelations(g, agents, agent);
		}
	}

	/**
	 * paint all relations
	 */
	public void paintRelations(Graphics g, Set agents, Agent source) {

		//check whether the agent has suitable relationType's relations
		List relationTypes = relationTypeSelection.getSelectedTypes();
		Iterator j = relationTypes.iterator();
		while (j.hasNext()) {
			RelationType relationType = (RelationType) j.next();
			Collection relations = source.getRelations(relationType);

			//paint all relation in relations
			Iterator k = relations.iterator();
			while (k.hasNext()) {
				Relation relation = (Relation) k.next();
				Agent target = relation.getTarget();
				if (agents.contains(target)) {
					paintRelation(g, source, target);
				}
			}
		}
	}

	/**
	 * paint a relation
	 */
	private void paintRelation(Graphics g, Agent source, Agent target) {
		//prepare
		AgentNode sourceNode = (AgentNode) nodes.get(source);
		AgentNode targetNode = (AgentNode) nodes.get(target);
		Point sourcePoint = sourceNode.getCenter();
		Point targetPoint = targetNode.getCenter();

		//paint
		g.drawLine(sourcePoint.x, sourcePoint.y, targetPoint.x, targetPoint.y);
		paintArrow(g, sourcePoint.x, sourcePoint.y, targetPoint.x, targetPoint.y);
	}

	/**
	 * paint arrow
	 * This code was written by Iba
	 */
	private void paintArrow(
		Graphics g,
		int node1x,
		int node1y,
		int node2x,
		int node2y) {

		int x1 = node1x;
		int y1 = node1y;
		int x2 = (int) (node1x + (double) (node2x - node1x) / 2.5);
		int y2 = (int) (node1y + (double) (node2y - node1y) / 2.5);

		if (!((x2 - x1 == 0) && (y2 - y2 == 0))) {
			int arrow = 6;
			double dd = 0.6d;

			double d = Math.atan2((double) (y2 - y1), (double) (x2 - x1));
			double xa = x2 - arrow * Math.cos(d + dd);
			double ya = y2 - arrow * Math.sin(d + dd);
			g.drawLine(x2, y2, (int) xa, (int) ya);
			double xb = x2 - arrow * Math.cos(d - dd);
			double yb = y2 - arrow * Math.sin(d - dd);
			g.drawLine(x2, y2, (int) xb, (int) yb);
		}
	}

	/**
	 * paint a label
	 */
	private void paintLabel(Graphics g, Agent agent) {
		//prepare
		AgentNode node = (AgentNode) nodes.get(agent);
		String text = agent.getType().getName();

		//calculate the label location
		Point point = node.getLocation();
		Point labelPoint = node.getLabelLocation();
		point.translate(labelPoint.x, labelPoint.y);

		//draw
		g.drawString(text, point.x, point.y);
	}

	/*******************************************
	 * getter & setter
	 *******************************************/

	/**
	 * Returns the world.
	 * @return World
	 */
	public World getWorld() {
		return world;
	}

	/**
	 * Sets the world.
	 * @param world The world to set
	 */
	public void setWorld(World world) {
		this.world = world;
		updateAll();
	}

	/**
	 * Returns the agentTypes.
	 * @return TypeSelection
	 */
	public TypeSelection getAgentTypeSelection() {
		return agentTypeSelection;
	}

	/**
	 * Returns the relationTypes.
	 * @return TypeSelection
	 */
	public TypeSelection getRelationTypeSelection() {
		return relationTypeSelection;
	}

	/**
	 * Returns the nodes.
	 * @return Map
	 */
	public Map getNodesMap() {
		return nodes;
	}

}
