(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    662857,      12392]*)
(*NotebookOutlinePosition[    663506,      12415]*)
(*  CellTagsIndexPosition[    663462,      12411]*)
(*WindowFrame->Normal*)



Notebook[{
Cell[BoxData[
    \(<< Utilities`MemoryConserve`\n$MemoryIncrement; 
    \n<< \ LinearAlgebra`MatrixManipulation`; \n
    Off[General::spell1, MemoryConserve::start, MemoryConserve::end]; \)], 
  "Input"],

Cell[BoxData[
    \(\[FilledSquare]\ \ Correlation\)], "Input"],

Cell["\<\
corRelation=
\tCompile[{{vectorA,_Real,1},{vectorB,_Real,1}},
\t\tModule[{meanA=0.,meanB=0.,normA=0.,normB=0.,
\t\t\t\tvectorAprime={0.},vectorBprime={0.}},
\t\t\tmeanA=Apply[Plus,vectorA]/Length[vectorA];
\t\t\tmeanB=Apply[Plus,vectorB]/Length[vectorB];
\t\t\tvectorAprime=vectorA-meanA;
\t\t\tnormA=Sqrt[vectorAprime.vectorAprime];
\t\t\tvectorBprime=vectorB-meanB;
\t\t\tnormB=Sqrt[vectorBprime.vectorBprime];
\t\t\tvectorAprime.vectorBprime/(normA normB)]];\
\>", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    RowBox[{"\[FilledSquare]", "  ", 
      StyleBox["\:57fa\:5e95\:95a2\:6570",
        FontFamily->"\.82l\.82r \.83S\.83V\.83b\.83N",
        FontWeight->"Bold"]}]], "Input"],

Cell[TextData[{
  StyleBox["     ",
    FontFamily->"\.82l\.82r \.83S\.83V\.83b\.83N",
    FontWeight->"Bold"],
  StyleBox["Coifman:",
    FontFamily->"\.82l\.82r \.83S\.83V\.83b\.83N",
    FontWeight->"Bold",
    FontVariations->{"Underline"->True}]
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
coif6={{-.051429728471, .238929728471, .602859456942, 
 \t\t .272140543058,-.051429972847,-.011070271529},
 \t   {-.011070271529, .051429972847, .272140543058,
\t\t-.602859456942, .238929728471, .051429728471}}*
\t\tSqrt[2.];
coif12={{ .011587596739,-.029320137980,-.047639590310, 
 \t\t  .273021046535, .574682393857, .294867193696,
\t\t -.054085607092,-.042026480461, .016744410163, 
 \t\t  .003967883613,-.001289203356,-.000509505399},
 \t\t{-.000509505399, .001289203356, .003967883613,
 \t\t -.016744410163,-.042026480461, .054085607092,
 \t\t  .294867193696,-.574682393857, .273021046535,
\t\t  .047639590310,-.029320137980,-.011587596739}}*
\t\t Sqrt[2.];
coif18={{-.002682418671, .005503126709, .016583560479, 
\t\t -.046507764479,-.043220763560, .286503335274,
 \t\t  .561285256870, .302983571773,-.050770140755, 
\t\t -.058196250762, .024434094321, .011229240962, 
\t\t -.006369601011,-.001820458916, .000790205101,
 \t\t  .000329665174,-.000050192775,-.000024465734},
 \t\t{-.000024465734, .000050192775, .000329665174,
 \t\t -.000790205101,-.001820458916, .006369601011,
 \t\t  .011229240962,-.024434094321,-.058196250762,
 \t\t  .050770140755, .302983571773,-.561285256870,
 \t\t  .286503335274, .043220763560,-.046507764479,
 \t\t -.016583560479, .005503126709, .002682418671}}*
 \t\tSqrt[2.];
coif24={{ .000630961046,-.001152224852,-.005194524026, 
 \t\t  .011362459244, .018867235378,-.057464234429, 
\t\t -.039652648517, .293667390895, .553126452562, 
 \t\t  .307157326198,-.047112738865,-.068038127051,
 \t\t  .027813640153, .017735837438,-.010756318517,
\t\t -.004001012886, .002652665946, .000895594529,
\t\t -.000416500571,-.000183829769, .000044080354, 
 \t\t  .000022082857,-.000002304942,-.000001262175},
 \t\t{-.000001262175, .000002304942, .000022082857,
 \t\t -.000044080354,-.000183829769, .000416500571,
 \t\t  .000895594529,-.002652665946,-.004001012886,
 \t\t  .010756318517, .017735837438,-.027813640153,
 \t\t -.068038127051, .047112738865, .307157326198,
 \t\t -.553126452562, .293667390895, .039652648517,
 \t\t -.057464234429,-.018867235378, .011362459244,
 \t\t  .005194524026,-.001152224852,-.000630961046}}*
 \t\tSqrt[2.];
coif30={{-.000149963800, .000253561200, .001540245700, 
\t\t -.002941110800,-.007163781900, .016552066400, 
 \t\t  .019917804300,-.064997262800,-.036800073600,
 \t\t  .298092323500, .547505429400, .309706849000, 
\t\t -.043866050800,-.074652238900, .029195879500, 
 \t\t  .023110777000,-.013973687900,-.006480090000, 
 \t\t  .004783001400, .001720654700,-.001175822200,
\t\t -.000451227000, .000213729800, .000099377600,
\t\t -.000029232100,-.000015072000, .000002640800, 
 \t\t  .000001459300,-.000000118400,-.000000067300},
 \t\t{-.000000067300, .000000118400, .000001459300,
 \t\t -.000002640800,-.000015072000, .000029232100,
 \t\t  .000099377600,-.000213729800,-.000451227000,
 \t\t  .001175822200, .001720654700,-.004783001400,
 \t\t -.006480090000, .013973687900, .023110777000,
 \t\t -.029195879500,-.074652238900, .043866050800,
 \t\t  .309706849000,-.547505429400, .298092323500,
 \t\t  .036800073600,-.064997262800,-.019917804300,
 \t\t  .016552066400, .007163781900,-.002941110800,
 \t\t -.001540245700, .000253561200, .000149963800}}*
 \t\tSqrt[2.]; \
\>", "Input",
  AspectRatioFixed->True,
  FontSize->9],

Cell[BoxData[
    \(\[FilledSquare]\ Wavelet\ Matrix\)], "Input"],

Cell["\<\
c[n_,baseType_]:=Module[{mat,i,result},
\tresult=mat=AppendRows[baseType,
\t\tZeroMatrix[2,n-Length[baseType[[1]]]]];
       \trotateRightofList[a_List]:=RotateRight/@ a;
        rotateRight[b_List,k_]:=
        Nest[rotateRightofList,b,k];
        Do[result=
        \tAppendColumns[result,rotateRight[mat,2 i]],
            {i,n/2-1}];
\tresult]\
\>", "Input",
  AspectRatioFixed->True,
  FontSize->9],

Cell["\<\
cPrime[i_,n_,baseType_]:=BlockMatrix[{{c[n/2^i,baseType],
\tZeroMatrix[n/2^i,n-n/2^i]},{ZeroMatrix[n-n/2^i,n/2^i],
\tIdentityMatrix[n-n/2^i]}}]\
\>", "Input",
  AspectRatioFixed->True,
  FontSize->9],

Cell["\<\
p[n_]:=Module[{mat,i},mat=IdentityMatrix[n];
\tDo[mat[[i]]=RotateRight[mat[[i]],i-1],{i,2,n}];
    Do[mat[[i]]=RotateRight[mat[[i]]],{i,n/2+1,n}];
    mat];
pPrime[i_,n_,baseType_]:=BlockMatrix[{{p[n/2^i],
\tZeroMatrix[n/2^i,n-n/2^i]},{ZeroMatrix[n-n/2^i,n/2^i],
  \tIdentityMatrix[n-n/2^i]}}];\
\>", "Input",
  AspectRatioFixed->True,
  FontSize->9],

Cell["\<\
waveletMatrix[n_,baseType_]:=
\tApply[Dot,Flatten[Table[{pPrime[j,n,baseType],
\t\tcPrime[j,n,baseType]},{j,Log[2,n]-
\t\tCeiling[Log[2.,Length[baseType[[2]]]]],1,-1}],
\t\t1]].p[n].c[n,baseType];\
\>", "Input",
  AspectRatioFixed->True,
  FontSize->9],

Cell[BoxData[
    \(\[FilledSquare]\ \ Sample\)], "Input"],

Cell[BoxData[
    \(\(a = 
      \(a = {{1, 0, 1, 1, 0, 0, 0, 1}, {0, 0, 1, 1, 1, 1, 0, 1}, {1, 1, 0, 1, 
            0, 0, 0, 1}, {1, 1, 1, 1, 0, 1\ , 0, 0}, {1, 0, 1, 1, 0, 0, 0, 
            1}, {0, 0, 1, 1, 1, 1, 0, 1}, {1, 1, 0, 1, 0, 1, 0, 1}, {1, 1, 1, 
            1, 0, 0\ , 0, 1}}\); \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[a, Mesh -> False]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.120192 0.0192308 0.120192 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.25962 -0.0125 -3 -9 ]
[.25962 -0.0125 3 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.74038 -0.0125 -3 -9 ]
[.74038 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -6 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -6 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.25962 0 m
.25962 .00625 L
s
[(2)] .25962 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(4)] .5 -0.0125 0 1 Mshowa
.74038 0 m
.74038 .00625 L
s
[(6)] .74038 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(8)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.07933 0 m
.07933 .00375 L
s
.13942 0 m
.13942 .00375 L
s
.19952 0 m
.19952 .00375 L
s
.31971 0 m
.31971 .00375 L
s
.37981 0 m
.37981 .00375 L
s
.4399 0 m
.4399 .00375 L
s
.5601 0 m
.5601 .00375 L
s
.62019 0 m
.62019 .00375 L
s
.68029 0 m
.68029 .00375 L
s
.80048 0 m
.80048 .00375 L
s
.86058 0 m
.86058 .00375 L
s
.92067 0 m
.92067 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(2)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(4)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(6)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(8)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .07933 m
.00375 .07933 L
s
0 .13942 m
.00375 .13942 L
s
0 .19952 m
.00375 .19952 L
s
0 .31971 m
.00375 .31971 L
s
0 .37981 m
.00375 .37981 L
s
0 .4399 m
.00375 .4399 L
s
0 .5601 m
.00375 .5601 L
s
0 .62019 m
.00375 .62019 L
s
0 .68029 m
.00375 .68029 L
s
0 .80048 m
.00375 .80048 L
s
0 .86058 m
.00375 .86058 L
s
0 .92067 m
.00375 .92067 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.25962 .99375 m
.25962 1 L
s
.5 .99375 m
.5 1 L
s
.74038 .99375 m
.74038 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.07933 .99625 m
.07933 1 L
s
.13942 .99625 m
.13942 1 L
s
.19952 .99625 m
.19952 1 L
s
.31971 .99625 m
.31971 1 L
s
.37981 .99625 m
.37981 1 L
s
.4399 .99625 m
.4399 1 L
s
.5601 .99625 m
.5601 1 L
s
.62019 .99625 m
.62019 1 L
s
.68029 .99625 m
.68029 1 L
s
.80048 .99625 m
.80048 1 L
s
.86058 .99625 m
.86058 1 L
s
.92067 .99625 m
.92067 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .07933 m
1 .07933 L
s
.99625 .13942 m
1 .13942 L
s
.99625 .19952 m
1 .19952 L
s
.99625 .31971 m
1 .31971 L
s
.99625 .37981 m
1 .37981 L
s
.99625 .4399 m
1 .4399 L
s
.99625 .5601 m
1 .5601 L
s
.99625 .62019 m
1 .62019 L
s
.99625 .68029 m
1 .68029 L
s
.99625 .80048 m
1 .80048 L
s
.99625 .86058 m
1 .86058 L
s
.99625 .92067 m
1 .92067 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
8 string
8 8 8 [8 0 0 8 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
FF00FFFF000000FF
0000FFFFFFFF00FF
FFFF00FF000000FF
FFFFFFFF00FF0000
FF00FFFF000000FF
0000FFFFFFFF00FF
FFFF00FF00FF00FF
FFFFFFFF000000FF
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
00ooool00`000?ooooooo`0noooo0`00043oool30000?_ooo`800011oooo00<0003oooooool01_oo
o`003_ooo`030000oooo000003ooool00`000?ooooooo`11oooo00<0003oooooool0?Oooo`030000
oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool00000@?ooo`030000oooooooo
03koool40000?_ooo`030000oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool0
0000@?ooo`030000oooooooo03koool00`000?ooo`00000ooooo0P00047oool00`000?ooooooo`06
oooo000>oooo00<0003oool00000?oooo`030000oooo0000043oool20000?oooo`030000oooooooo
03ooool00`000?ooo`000007oooo000?oooo00<0003oooooool0?oooo`030000oooooooo043oool0
0`000?ooooooo`0noooo0P00043oool00`000?ooooooo`06oooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo000:ooooo`0001D00002oooo000:
oooo00<0003oooooool00oooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`02oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooo`030000oooooooo03ooool00`000?ooooooo`0ooooo00<0
003oooooool0?oooo`030000oooooooo03ooool00`000?ooooooo`02oooo0@00007oool1oooo0004
oooo00<0003oooooool00oooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<0003ooooo3oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`030000
oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo0004oooo00<0003oooooool00ooo
o`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@00
04;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000V
oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@00
04;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000V
oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool200009Oooob400012ooooH`00
02Goool200000_ooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob400012
ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob400012
ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`80000Uoooo8@00
04;ooomS00009Oooo`800002oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0
003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?oo
ob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0
003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?oo
ob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0
003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool200001?oood800024
oooo8@0002Goool200000_ooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00
007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
od800024oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00
007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
od800024oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00
007oool1oooo0003oooo0`0000Coool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001
oooo0Oooo`000oooo`030000oooooooo00Coool00`000?ooooooo`03oooo@P0008CooolQ00009_oo
o`400001oooo0Oooo`001?ooo`030000oooooooo00?oool300000ooood800024oooo8@0002Coool3
00000_ooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`15oooo8@00027ooomS00009_oo
o`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`15oooo8@00027ooomS
00009_ooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`15oooo8@00
027ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooof<0000V
oooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000QooooH`0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`15oooo8@00027ooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo04GooolQ00008Oooof<0000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool0AOooob40000QooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00
027ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooof<0000V
oooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000QooooH`0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`15oooo8@00027ooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo04GooolQ00008Oooof<0000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool0AOooob40000QooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00
027ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooof<0000V
oooo0@00007oool1oooo000:oooo0P0004KooolQ00008Oooof<0000Uoooo0P0000;oool000[oool0
0`000?ooooooo`15oooo8@00027ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
04GooolQ00008Oooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Q
ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00027ooomS00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooof<0000Voooo0@00007oool1
oooo000:oooo00<0003oooooool0AOooob40000QooooH`0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`15oooo8@00027ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
04GooolQ00008Oooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Q
ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00027ooomS00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooof<0000Voooo0@00007oool1
oooo000:oooo00<0003oooooool0AOooob40000QooooH`0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`15oooo8@00027ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
04GooolQ00008Oooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Q
ooooH`0002Koool100000Oooo`7oool000[oool20000A_ooob40000QooooH`0002Goool200000_oo
o`002_ooo`030000oooooooo08OooolQ00008Ooood800005oooo0@00007oool1oooo000:oooo00<0
003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool000[oool00`000?ooooooo`27
oooo8@00027ooom200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooolQ00008Ooo
od800005oooo0@00007oool1oooo000:oooo00<0003oooooool0Qoooob40000Qoooo@P0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`27oooo8@00027ooom200001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo08OooolQ00008Ooood800005oooo0@00007oool1oooo000:oooo00<0
003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool000[oool00`000?ooooooo`27
oooo8@00027ooom200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooolQ00008Ooo
od800005oooo0@00007oool1oooo000:oooo00<0003oooooool0Qoooob40000Qoooo@P0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`27oooo8@00027ooom200001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo08OooolQ00008Ooood800005oooo0@00007oool1oooo000:oooo00<0
003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool000[oool00`000?ooooooo`27
oooo8@00027ooom200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooolQ00008Ooo
od800005oooo0@00007oool1oooo000:oooo0P0008SooolQ00008Ooood800004oooo0P0000;oool0
00[oool00`000?ooooooo`27oooo8@00027ooom200001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo08OooolQ00008Ooood800005oooo0@00007oool1oooo000:oooo00<0003oooooool0Qooo
ob40000Qoooo@P0000Goool100000Oooo`7oool000[oool00`000?ooooooo`27oooo8@00027ooom2
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooolQ00008Ooood800005oooo0@00
007oool1oooo000:oooo00<0003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool0
00[oool00`000?ooooooo`27oooo8@00027ooom200001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo08OooolQ00008Ooood800005oooo0@00007oool1oooo000:oooo00<0003oooooool0Qooo
ob40000Qoooo@P0000Goool100000Oooo`7oool000[oool00`000?ooooooo`27oooo8@00027ooom2
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooolQ00008Ooood800005oooo0@00
007oool1oooo000:oooo00<0003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool0
00[oool00`000?ooooooo`27oooo8@00027ooom200001Oooo`400001oooo0Oooo`001?ooo`<00003
oooo00<0003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool000Goool00`000?oo
ooooo`02oooo00<0003oooooool0Qoooob40000Qoooo@P0000Goool100000Oooo`7oool000?oool4
00000oooo`<00027oooo8@00027ooom200000oooo`<00002oooo0003oooo00<0003oool000001?oo
o`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo0004oooo0P0000Coool0
0`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`001Oooo`030000oooooooo
00;oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob400012ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool0
00[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob400012ooooH`0002Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool0
00[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:oooo0P0002GooolQ0000@_oo
of<0000Uoooo0P0000;oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool00`000?oo
ooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob400012ooooH`00
02Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool00`000?oo
ooooo`0Toooo8@0004;ooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
0000@_ooof<0000Voooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob400012ooooH`00
02Koool100000Oooo`7oool000[oool00`000?ooooooo`0Toooo8@0004;ooomS00009_ooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQ0000@_ooof<0000Voooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob400012ooooH`0002Koool100000Oooo`7oool000[oool200009Ooo
ob400012ooooH`0002Goool200000_ooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000V
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P00
08CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000V
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P00
08CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000V
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo0P0000Cooom2
0000Q?ooob40000Uoooo0P0000;oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1
oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024
oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000Q?ooob40000Voooo0@00007oool1
oooo000:oooo00<0003oooooool00ooood800024oooo8@0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo@P0008CooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooom20000Q?ooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800024
oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0008CooolQ00009_oo
o`400001oooo0Oooo`000oooo`800005oooo00<0003oooooool00ooood800024oooo8@0002Koool1
00000Oooo`7oool000?oool00`000?ooo`000004oooo00<0003oooooool00ooood800024oooo8@00
02Koool100000Oooo`7oool000?oool00`000?ooo`000004oooo0`0000?ooom20000Q?ooob40000T
oooo0`0000;oool000?oool200001Oooo`030000oooooooo04GooolQ00008Oooob40000Qoooo8@00
02Koool100000Oooo`7oool000?oool00`000?ooooooo`04oooo00<0003oooooool0AOooob40000Q
oooo8@00027ooolQ00009_ooo`400001oooo0Oooo`001?ooo`800004oooo00<0003oooooool0AOoo
ob40000Qoooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ
00008Oooob40000Qoooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00
027ooolQ00008Oooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Q
oooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Ooo
ob40000Qoooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00027ooolQ
00008Oooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Qoooo8@00
027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooob40000Q
oooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00027ooolQ00008Ooo
ob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Qoooo8@00027ooolQ
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooob40000Qoooo8@00
02Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00027ooolQ00008Oooob40000V
oooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Qoooo8@00027ooolQ00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Oooob40000Qoooo8@0002Koool1
00000Oooo`7oool000[oool20000A_ooob40000Qoooo8@00027ooolQ00009Oooo`800002oooo000:
oooo00<0003oooooool0AOooob40000Qoooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo04GooolQ00008Oooob40000Qoooo8@0002Koool100000Oooo`7oool000[oool0
0`000?ooooooo`15oooo8@00027ooolQ00008Oooob40000Voooo0@00007oool1oooo000:oooo00<0
003oooooool0AOooob40000Qoooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo04GooolQ00008Oooob40000Qoooo8@0002Koool100000Oooo`7oool000[oool00`000?oo
ooooo`15oooo8@00027ooolQ00008Oooob40000Voooo0@00007oool1oooo000:oooo00<0003ooooo
ool0AOooob40000Qoooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
04GooolQ00008Oooob40000Qoooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15
oooo8@00027ooolQ00008Oooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOoo
ob40000Qoooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ
00008Oooob40000Qoooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00
027ooolQ00008Oooob40000Voooo0@00007oool1oooo000:oooo00<0003oooooool0AOooob40000Q
oooo8@00027ooolQ00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo04GooolQ00008Ooo
ob40000Qoooo8@0002Koool100000Oooo`7oool000[oool00`000?ooooooo`15oooo8@00027ooolQ
00008Oooob40000Voooo0@00007oool1oooo000:oooo0P0004KooolQ00008Oooob40000Qoooo8@00
02Goool200000_ooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Ooo
o`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001
oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Ooo
o`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
08OooomS00009_ooo`400001oooo0Oooo`002_ooo`800028ooooH`0002Goool200000_ooo`002_oo
o`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Ooo
o`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo
08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001
oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000
oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_oo
o`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_oo
o`030000oooooooo08OooomS00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS
00009_ooo`400001oooo0Oooo`002_ooo`030000oooooooo08OooomS00009_ooo`400001oooo0Ooo
o`001?ooo`030000oooooooo00?oool00`000?ooooooo`27ooooH`0002Koool100000Oooo`7oool0
00?oool00`000?ooo`000004oooo00<0003oooooool0Qoooof<0000Voooo0@00007oool1oooo0003
oooo00<0003oool000001?ooo`<00027ooooH`0002Coool300000_ooo`001?ooo`030000oooooooo
00?oool00`000?ooooooo`3ooooo4Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool0
0`000?ooooooo`3ooooo4Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?oo
ooooo`3ooooo4Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?oool00`000?ooooooo`0=
oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool00_ooo`400001oooo0Oooo`002_ooool0000E00000_ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.489418, -0.536766, 0.0303025, 
  0.0303025}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    \(\[FilledSquare]\ Fourier\ Transform\)], "Input"],

Cell[BoxData[
    \(\(fa = Fourier[a]; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[Abs[fa], Mesh -> False]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.120192 0.0192308 0.120192 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.25962 -0.0125 -3 -9 ]
[.25962 -0.0125 3 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.74038 -0.0125 -3 -9 ]
[.74038 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -6 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -6 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.25962 0 m
.25962 .00625 L
s
[(2)] .25962 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(4)] .5 -0.0125 0 1 Mshowa
.74038 0 m
.74038 .00625 L
s
[(6)] .74038 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(8)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.07933 0 m
.07933 .00375 L
s
.13942 0 m
.13942 .00375 L
s
.19952 0 m
.19952 .00375 L
s
.31971 0 m
.31971 .00375 L
s
.37981 0 m
.37981 .00375 L
s
.4399 0 m
.4399 .00375 L
s
.5601 0 m
.5601 .00375 L
s
.62019 0 m
.62019 .00375 L
s
.68029 0 m
.68029 .00375 L
s
.80048 0 m
.80048 .00375 L
s
.86058 0 m
.86058 .00375 L
s
.92067 0 m
.92067 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(2)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(4)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(6)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(8)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .07933 m
.00375 .07933 L
s
0 .13942 m
.00375 .13942 L
s
0 .19952 m
.00375 .19952 L
s
0 .31971 m
.00375 .31971 L
s
0 .37981 m
.00375 .37981 L
s
0 .4399 m
.00375 .4399 L
s
0 .5601 m
.00375 .5601 L
s
0 .62019 m
.00375 .62019 L
s
0 .68029 m
.00375 .68029 L
s
0 .80048 m
.00375 .80048 L
s
0 .86058 m
.00375 .86058 L
s
0 .92067 m
.00375 .92067 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.25962 .99375 m
.25962 1 L
s
.5 .99375 m
.5 1 L
s
.74038 .99375 m
.74038 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.07933 .99625 m
.07933 1 L
s
.13942 .99625 m
.13942 1 L
s
.19952 .99625 m
.19952 1 L
s
.31971 .99625 m
.31971 1 L
s
.37981 .99625 m
.37981 1 L
s
.4399 .99625 m
.4399 1 L
s
.5601 .99625 m
.5601 1 L
s
.62019 .99625 m
.62019 1 L
s
.68029 .99625 m
.68029 1 L
s
.80048 .99625 m
.80048 1 L
s
.86058 .99625 m
.86058 1 L
s
.92067 .99625 m
.92067 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .07933 m
1 .07933 L
s
.99625 .13942 m
1 .13942 L
s
.99625 .19952 m
1 .19952 L
s
.99625 .31971 m
1 .31971 L
s
.99625 .37981 m
1 .37981 L
s
.99625 .4399 m
1 .4399 L
s
.99625 .5601 m
1 .5601 L
s
.99625 .62019 m
1 .62019 L
s
.99625 .68029 m
1 .68029 L
s
.99625 .80048 m
1 .80048 L
s
.99625 .86058 m
1 .86058 L
s
.99625 .92067 m
1 .92067 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
8 string
8 8 8 [8 0 0 8 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
FF84817D9F7D8184
11071A1E11071A1E
1152357158255886
111E312A111E312A
356F403A113A406F
112A311E112A311E
1186582558713552
111E1A07111E1A07
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
00ooool00`000?ooooooo`0noooo0`00043oool30000?_ooo`800011oooo00<0003oooooool01_oo
o`003_ooo`030000oooo000003ooool00`000?ooooooo`11oooo00<0003oooooool0?Oooo`030000
oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool00000@?ooo`030000oooooooo
03koool40000?_ooo`030000oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool0
0000@?ooo`030000oooooooo03koool00`000?ooo`00000ooooo0P00047oool00`000?ooooooo`06
oooo000>oooo00<0003oool00000?oooo`030000oooo0000043oool20000?oooo`030000oooooooo
03ooool00`000?ooo`000007oooo000?oooo00<0003oooooool0?oooo`030000oooooooo043oool0
0`000?ooooooo`0noooo0P00043oool00`000?ooooooo`06oooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo000:ooooo`0001D00002oooo000:
oooo00<0003oooooool00oooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`02oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooo`030000oooooooo03ooool00`000?ooooooo`0ooooo00<0
003oooooool0?oooo`030000oooooooo03ooool00`000?ooooooo`02oooo0@00007oool1oooo0004
oooo00<0003oooooool00oooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<0003ooooo3oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
02CooolQQ8B48H61PB5mOGdQWinO8GemOB61PH4QQ8B41Oooo`400001oooo0Oooo`000oooo`030000
oooo000000Coool00`000?ooooooo`0Toooo8HB4Q261PH4QOGem8InOWb5mOGdQPH618HB4Q0Goool1
00000Oooo`7oool000Coool00`000?ooooooo`03oooo00<0003oooooool09?ooob64Q8@QPH618Gem
OB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@Q
PH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0
003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5
oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61
PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQ
OGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618Gem
OB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@Q
PH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo0P00
02GooolQQ8B48H61PB5mOGdQWinO8GemOB61PH4QQ8B41?ooo`800002oooo000:oooo00<0003ooooo
ool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0
003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5
oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61
PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQ
OGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618Gem
OB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@Q
PH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:oooo00<0
003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5oooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob64Q8@QPH618GemOB6OWilQOGem8H61PB64Q8@5
oooo0@00007oool1oooo000:oooo0P0002GooolQQ8B48H61PB5mOGdQWinO8GemOB61PH4QQ8B41?oo
o`800002oooo000:oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ
7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A
4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B471`LQ
6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ
7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A
4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B471`LQ
6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ
7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A
4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B471`LQ
6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001oooo0Oooo`002_ooo`800004
oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh4oooo0P0000;oool000[oool00`000?oo
ooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN
7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q
1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ
6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN
7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q
1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q1`L78AXJ
6R4N7QhQ4A4A8@L71b4J6QXQ7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5oooo0@00007oool1oooo0003
oooo0`0000Coool00`000?ooooooo`03oooo8A4A4B471`LQ6QXJ8AhN7R4A4A4Q1`L78AXJ6R4N7Qh5
oooo0@00007oool1oooo0003oooo00<0003oooooool01?ooo`030000oooooooo00?ooolQ4A4A8@L7
1b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN7PGoool100000Oooo`7oool000Coool00`000?ooooooo`03
oooo0`0000?ooolQ4A4A8@L71b4J6QXQ7QhN8A4A4B471`LQ6QXJ8AhN7P?oool300000_ooo`001?oo
o`030000oooooooo00?oool00`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQH
F266QXH5oooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`030000oooooooo00?ooolQ
4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool100000Oooo`7oool000Coool00`000?oo
ooooo`03oooo00<0003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQ
F5QH8HJ6QPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5a
LB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q
DU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQ
F5QH8HJ6QPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5a
LB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q
DU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQ
F5QH8HJ6QPGoool100000Oooo`7oool000[oool200001?ooob4A4A4QDU9B8CDe=B5aLG4QF5QH8BDU
9B5HF5PQQXJ61?ooo`800002oooo000:oooo00<0003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4Q
F5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8E9B
DR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQH
F266QXH5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4Q
F5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8E9B
DR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4QF5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQH
F266QXH5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4QDU9B8CDe=B5aLG4Q
F5QH8BDU9B5HF5PQQXJ61Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8E9B
DR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B5BDU8Q=CDe8G5aLB5HF5PQ9BDU8EQHF266QXH5oooo0@00007oool1oooo000:oooo0P00
00CooolQ4A4A8E9BDR4e=CDQLG5a8EQHF24U9BDQF5QH8HJ6QPCoool200000_ooo`002_ooo`030000
oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q
:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A
4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ
<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q
:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A
4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ
<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q
:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A
4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool000[oool200001?ooob4A4A4Q7QhN8C4a<B4Z:RXQ
4A4A8AhN7R4a<C4Q:RXZ1?ooo`800002oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a
<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ
:PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q
7QhN8C4a<B4Z:RX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a
<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ:PGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q7QhN8C4a<B4Z:RX5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4a<C4Q:RXZ8A4A4B4N7QhQ<C4a8BXZ
:PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q
7QhN8C4a<B4Z:RX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8C4a
<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001oooo0Oooo`001?ooo`<00003oooo00<0003ooooo
ool00oooob4A4A4Q7QhN8C4a<B4Z:RXQ4A4A8AhN7R4a<C4Q:RXZ1Oooo`400001oooo0Oooo`001Ooo
o`030000oooooooo00;oool00`000?ooooooo`03oooo8A4A4B4N7QhQ<C4a8BXZ:R4A4A4Q7QhN8C4a
<B4Z:RX5oooo0@00007oool1oooo0003oooo100000?oool300000oooob4A4A4Q7QhN8C4a<B4Z:RXQ
4A4A8AhN7R4a<C4Q:RXZ0oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Oooo`7oool000Coool2
00001?ooo`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool1
00000Oooo`7oool000Goool00`000?ooooooo`02oooo00<0003oooooool00oooob4e=CDQKfm_8D10
@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_
K`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q
>SXj8D10@25_Kfl5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4e=CDQKfm_8D10
@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_
K`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q
>SXj8D10@25_Kfl5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4e=CDQKfm_8D10
@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Oooo`7oool000[oool200001?oo
ob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj>R50@40QKfm_1?ooo`800002oooo000:oooo00<0003ooooo
ool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj
>R50@40QKfm_1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q
>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8CDe
=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj
>R50@40QKfm_1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q
>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8CDe
=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob4e=CDQKfm_8D10@24j>SXQ4A4A8CXj>R50@40QKfm_1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_K`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8CDe=B5_KflQ@4108CXj>R4A4A4Q>SXj8D10@25_Kfl5
oooo0@00007oool1oooo000:oooo0P0000CooolQ=CDe8Fm_Kb50@40Q>SXj8A4A4B4j>SXQ@4108Fm_
K`Coool200000_ooo`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ
<C4a8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN
7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q
:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ
<C4a8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN
7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q
:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ
<C4a8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN
7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q
:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool100000Oooo`7oool000[oool2
00001?ooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1?ooo`800002oooo000:oooo00<0
003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a
<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ
4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ
:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a
<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ
4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8BXZ
:R4a<C4Q7QhN8A4A4B4Z:RXQ<C4a8AhN7PGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q7QhN1Oooo`400001oooo0Ooo
o`000oooo`800005oooo00<0003oooooool00oooob4A4A4Q:RXZ8C4a<B4N7QhQ4A4A8BXZ:R4a<C4Q
7QhN1Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`03oooo8A4A
4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a<B4N7Qh5oooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<00003oooo8A4A4B4Z:RXQ<C4a8AhN7R4A4A4Q:RXZ8C4a<B4N7Qh3oooo0`0000;oool0
00?oool200001Oooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9B
DPGoool100000Oooo`7oool000?oool00`000?ooooooo`04oooo00<0003oooooool00oooob4A4A4Q
QXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Oooo`001?ooo`800004oooo00<0
003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B66QXHQF5QH8BDU9B5HF5PQLG5a8CDe
=B5BDU85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQ
F5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6
QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B66QXHQF5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B66QXHQF5QH8BDU9B5HF5PQLG5a8CDe
=B5BDU85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQ
F5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6
QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8A4A4B66QXHQF5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool1
00000Oooo`7oool000[oool200001?ooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1?oo
o`800002oooo000:oooo00<0003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQ
DU9B1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQH
F25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B66QXHQ
F5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B66QXHQF5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQ
DU9B1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQH
F25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B66QXHQ
F5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQDU9B1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8A4A4B66QXHQF5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4A4A4QQXJ68EQHF24U9BDQF5QH8G5aLB4e=CDQ
DU9B1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8HJ6QR5HF5PQ9BDU8EQH
F25aLG4Q=CDe8E9BDPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B66QXHQ
F5QH8BDU9B5HF5PQLG5a8CDe=B5BDU85oooo0@00007oool1oooo000:oooo0P0000CooolQ4A4A8HJ6
QR5HF5PQ9BDU8EQHF25aLG4Q=CDe8E9BDPCoool200000_ooo`002_ooo`030000oooooooo00?ooolQ
4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L7
1`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q
7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ
6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L7
1`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q
7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ
6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L7
1`Goool100000Oooo`7oool000[oool200001?ooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN7R4J6QXQ
1`L71?ooo`800002oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN
7R4J6QXQ1`L71Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ
1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A
4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN
7R4J6QXQ1`L71Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ
1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8A4A
4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN7R4J6QXQ1`L71Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L71`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4A4A4Q7QhN8AXJ6R471`LQ4A4A8AhN
7R4J6QXQ1`L71Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03
oooo8A4A4B4N7QhQ6QXJ8@L71b4A4A4Q7QhN8AXJ6R471`L5oooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo00?ooolQ4A4A8AhN7R4J6QXQ1`L78A4A4B4N7QhQ6QXJ8@L7
1`Goool100000Oooo`7oool000?oool00`000?ooo`000004oooo0`0000?ooolQ4A4A8AhN7R4J6QXQ
1`L78A4A4B4N7QhQ6QXJ8@L71`?oool300000_ooo`001?ooo`030000oooooooo00?oool00`000?oo
ooooo`3ooooo4Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`3o
oooo4Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`3ooooo4Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?oool00`000?ooooooo`0=oooo00<0003ooooo
ool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000
oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool0
0`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`0=
oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool00_ooo`400001oooo0Oooo`002_ooool0000E00000_ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.489418, -0.536766, 0.0303025, 
  0.0303025}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    RowBox[{"\[FilledCircle]", " ", 
      StyleBox["Compression",
        FontVariations->{"Underline"->True}]}]], "Input"],

Cell[BoxData[
    \(\(n = 8; \)\)], "Input"],

Cell[TextData[StyleBox[
"nPrime=4;\ncutfa=TakeMatrix[fa,{1,1},{4,4}];\n\
compressedfSpec=BlockMatrix[{{cutfa,ZeroMatrix[nPrime,n-nPrime]},{ZeroMatrix[\
n-nPrime,n]}}];",
  FontSize->14]], "Input",
  PageBreakWithin->Automatic,
  FontSize->16],

Cell[CellGroupData[{

Cell[BoxData[
    \(MatrixForm[compressedfSpec]\)], "Input"],

Cell[BoxData[
    TagBox[
      RowBox[{"(", GridBox[{
            {\(\(4.62499999999999911`\[InvisibleSpace]\) + 0.`\ I\), 
              \(\(0.411611652351681467`\[InvisibleSpace]\) + 
                0.838388347648318266`\ I\), 
              \(\(0.249999999999999822`\[InvisibleSpace]\) - 
                0.874999999999999822`\ I\), 
              \(\(0.588388347648318266`\[InvisibleSpace]\) - 
                0.661611652351681378`\ I\), "0", "0", "0", "0"},
            {\(\(-1.38777878078144523`*^-17\) - 0.124999999999999977`\ I\), 
              \(\(0.0366116523516815339`\[InvisibleSpace]\) - 
                0.036611652351681565`\ I\), 
              \(\(-0.051776695296636932`\) - 0.176776695296636798`\ I\), 
              \(\(-0.21338834764831831`\) + 0.036611652351681645`\ I\), "0", 
              "0", "0", "0"},
            {\(\(-0.125000000000000044`\) + 0.`\ I\), 
              \(\(0.265165042944955064`\[InvisibleSpace]\) - 
                0.515165042944955331`\ I\), 
              \(\(-1.17756934401283119`*^-16\) - 0.375`\ I\), 
              \(\(0.441941738241592041`\[InvisibleSpace]\) - 
                0.661611652351681645`\ I\), "0", "0", "0", "0"},
            {\(1.38777878078144523`*^-17 + 0.124999999999999977`\ I\), 
              \(\(-0.0366116523516816183`\) - 0.21338834764831831`\ I\), 
              \(\(-0.301776695296636709`\) + 0.176776695296636976`\ I\), 
              \(\(0.213388347648318488`\[InvisibleSpace]\) + 
                0.213388347648318266`\ I\), "0", "0", "0", "0"},
            {"0", "0", "0", "0", "0", "0", "0", "0"},
            {"0", "0", "0", "0", "0", "0", "0", "0"},
            {"0", "0", "0", "0", "0", "0", "0", "0"},
            {"0", "0", "0", "0", "0", "0", "0", "0"}
            }], ")"}],
      (MatrixForm[ #]&)]], "Output"]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
    \(MatrixForm[fa]\)], "Input"],

Cell[BoxData[
    TagBox[
      RowBox[{"(", GridBox[{
            {\(\(4.62499999999999911`\[InvisibleSpace]\) + 0.`\ I\), 
              \(\(0.411611652351681467`\[InvisibleSpace]\) + 
                0.838388347648318266`\ I\), 
              \(\(0.249999999999999822`\[InvisibleSpace]\) - 
                0.874999999999999822`\ I\), 
              \(\(0.588388347648318266`\[InvisibleSpace]\) - 
                0.661611652351681378`\ I\), 
              \(\(-1.12499999999999977`\) + 0.`\ I\), 
              \(\(0.588388347648318354`\[InvisibleSpace]\) + 
                0.661611652351681378`\ I\), 
              \(\(0.250000000000000088`\[InvisibleSpace]\) + 
                0.874999999999999822`\ I\), 
              \(\(0.411611652351681378`\[InvisibleSpace]\) - 
                0.838388347648318266`\ I\)},
            {\(\(-1.38777878078144523`*^-17\) - 0.124999999999999977`\ I\), 
              \(\(0.0366116523516815339`\[InvisibleSpace]\) - 
                0.036611652351681565`\ I\), 
              \(\(-0.051776695296636932`\) - 0.176776695296636798`\ I\), 
              \(\(-0.21338834764831831`\) + 0.036611652351681645`\ I\), 
              \(1.38777878078144523`*^-17 + 0.124999999999999977`\ I\), 
              \(\(-0.0366116523516815339`\) + 0.0366116523516816005`\ I\), 
              \(\(0.051776695296636932`\[InvisibleSpace]\) + 
                0.176776695296636798`\ I\), 
              \(\(0.213388347648318399`\[InvisibleSpace]\) - 
                0.0366116523516816183`\ I\)},
            {\(\(-0.125000000000000044`\) + 0.`\ I\), 
              \(\(0.265165042944955064`\[InvisibleSpace]\) - 
                0.515165042944955331`\ I\), 
              \(\(-1.17756934401283119`*^-16\) - 0.375`\ I\), 
              \(\(0.441941738241592041`\[InvisibleSpace]\) - 
                0.661611652351681645`\ I\), 
              \(\(0.624999999999999911`\[InvisibleSpace]\) + 0.`\ I\), 
              \(\(-0.265165042944955331`\) + 0.0151650429449554202`\ I\), 
              \(\(-0.499999999999999822`\) + 0.375`\ I\), 
              \(\(-0.441941738241592307`\) - 0.838388347648318088`\ I\)},
            {\(1.38777878078144523`*^-17 + 0.124999999999999977`\ I\), 
              \(\(-0.0366116523516816183`\) - 0.21338834764831831`\ I\), 
              \(\(-0.301776695296636709`\) + 0.176776695296636976`\ I\), 
              \(\(0.213388347648318488`\[InvisibleSpace]\) + 
                0.213388347648318266`\ I\), 
              \(\(-1.38777878078144523`*^-17\) - 0.124999999999999977`\ I\), 
              \(\(0.0366116523516816494`\[InvisibleSpace]\) + 
                0.21338834764831831`\ I\), 
              \(\(0.301776695296636709`\[InvisibleSpace]\) - 
                0.176776695296636976`\ I\), 
              \(\(-0.213388347648318532`\) - 0.21338834764831831`\ I\)},
            {\(\(-0.374999999999999822`\) + 0.`\ I\), 
              \(\(0.765165042944955242`\[InvisibleSpace]\) - 
                0.161611652351681511`\ I\), 
              \(\(0.249999999999999911`\[InvisibleSpace]\) - 
                0.374999999999999955`\ I\), 
              \(\(0.234834957055044712`\[InvisibleSpace]\) + 
                0.338388347648318354`\ I\), 
              \(\(-0.124999999999999977`\) + 0.`\ I\), 
              \(\(0.234834957055044624`\[InvisibleSpace]\) - 
                0.338388347648318354`\ I\), 
              \(\(0.250000000000000044`\[InvisibleSpace]\) + 
                0.374999999999999955`\ I\), 
              \(\(0.765165042944955242`\[InvisibleSpace]\) + 
                0.161611652351681511`\ I\)},
            {\(\(-1.38777878078144523`*^-17\) - 0.124999999999999977`\ I\), 
              \(\(-0.21338834764831831`\) + 0.213388347648318399`\ I\), 
              \(\(0.301776695296636887`\[InvisibleSpace]\) + 
                0.176776695296636727`\ I\), 
              \(\(0.0366116523516814496`\[InvisibleSpace]\) - 
                0.213388347648318399`\ I\), 
              \(1.38777878078144523`*^-17 + 0.124999999999999977`\ I\), 
              \(\(0.21338834764831831`\[InvisibleSpace]\) - 
                0.213388347648318488`\ I\), 
              \(\(-0.301776695296636887`\) - 0.176776695296636727`\ I\), 
              \(\(-0.0366116523516814629`\) + 0.213388347648318443`\ I\)},
            {\(\(-0.125000000000000044`\) + 0.`\ I\), 
              \(\(-0.441941738241592041`\) + 0.838388347648318443`\ I\), 
              \(\(-0.500000000000000088`\) - 0.374999999999999822`\ I\), 
              \(\(-0.265165042944955242`\) - 0.0151650429449551849`\ I\), 
              \(\(0.624999999999999911`\[InvisibleSpace]\) + 0.`\ I\), 
              \(\(0.441941738241592219`\[InvisibleSpace]\) + 
                0.661611652351681378`\ I\), 
              \(1.17756934401283119`*^-16 + 0.374999999999999822`\ I\), 
              \(\(0.26516504294495542`\[InvisibleSpace]\) + 
                0.515165042944955153`\ I\)},
            {\(1.38777878078144523`*^-17 + 0.124999999999999977`\ I\), 
              \(\(0.213388347648318399`\[InvisibleSpace]\) + 
                0.0366116523516814851`\ I\), 
              \(\(0.0517766952966367632`\[InvisibleSpace]\) - 
                0.176776695296636887`\ I\), 
              \(\(-0.0366116523516816005`\) - 0.0366116523516814851`\ I\), 
              \(\(-1.38777878078144523`*^-17\) - 0.124999999999999977`\ I\), 
              \(\(-0.213388347648318443`\) - 0.0366116523516814851`\ I\), 
              \(\(-0.0517766952966367632`\) + 0.176776695296636887`\ I\), 
              \(\(0.0366116523516815783`\[InvisibleSpace]\) + 
                0.0366116523516815028`\ I\)}
            }], ")"}],
      (MatrixForm[ #]&)]], "Output"]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[InverseFourier[fa], Mesh -> False]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.120192 0.0192308 0.120192 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.25962 -0.0125 -3 -9 ]
[.25962 -0.0125 3 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.74038 -0.0125 -3 -9 ]
[.74038 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -6 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -6 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.25962 0 m
.25962 .00625 L
s
[(2)] .25962 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(4)] .5 -0.0125 0 1 Mshowa
.74038 0 m
.74038 .00625 L
s
[(6)] .74038 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(8)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.07933 0 m
.07933 .00375 L
s
.13942 0 m
.13942 .00375 L
s
.19952 0 m
.19952 .00375 L
s
.31971 0 m
.31971 .00375 L
s
.37981 0 m
.37981 .00375 L
s
.4399 0 m
.4399 .00375 L
s
.5601 0 m
.5601 .00375 L
s
.62019 0 m
.62019 .00375 L
s
.68029 0 m
.68029 .00375 L
s
.80048 0 m
.80048 .00375 L
s
.86058 0 m
.86058 .00375 L
s
.92067 0 m
.92067 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(2)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(4)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(6)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(8)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .07933 m
.00375 .07933 L
s
0 .13942 m
.00375 .13942 L
s
0 .19952 m
.00375 .19952 L
s
0 .31971 m
.00375 .31971 L
s
0 .37981 m
.00375 .37981 L
s
0 .4399 m
.00375 .4399 L
s
0 .5601 m
.00375 .5601 L
s
0 .62019 m
.00375 .62019 L
s
0 .68029 m
.00375 .68029 L
s
0 .80048 m
.00375 .80048 L
s
0 .86058 m
.00375 .86058 L
s
0 .92067 m
.00375 .92067 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.25962 .99375 m
.25962 1 L
s
.5 .99375 m
.5 1 L
s
.74038 .99375 m
.74038 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.07933 .99625 m
.07933 1 L
s
.13942 .99625 m
.13942 1 L
s
.19952 .99625 m
.19952 1 L
s
.31971 .99625 m
.31971 1 L
s
.37981 .99625 m
.37981 1 L
s
.4399 .99625 m
.4399 1 L
s
.5601 .99625 m
.5601 1 L
s
.62019 .99625 m
.62019 1 L
s
.68029 .99625 m
.68029 1 L
s
.80048 .99625 m
.80048 1 L
s
.86058 .99625 m
.86058 1 L
s
.92067 .99625 m
.92067 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .07933 m
1 .07933 L
s
.99625 .13942 m
1 .13942 L
s
.99625 .19952 m
1 .19952 L
s
.99625 .31971 m
1 .31971 L
s
.99625 .37981 m
1 .37981 L
s
.99625 .4399 m
1 .4399 L
s
.99625 .5601 m
1 .5601 L
s
.99625 .62019 m
1 .62019 L
s
.99625 .68029 m
1 .68029 L
s
.99625 .80048 m
1 .80048 L
s
.99625 .86058 m
1 .86058 L
s
.99625 .92067 m
1 .92067 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
8 string
8 8 8 [8 0 0 8 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
83007C830000007C
00008383897C0083
838300830000007C
83838383007C0000
83007C830000007C
00008383897C0083
838300830089007C
838383830000007C
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
00ooool00`000?ooooooo`0noooo0`00043oool30000?_ooo`800011oooo00<0003oooooool01_oo
o`003_ooo`030000oooo000003ooool00`000?ooooooo`11oooo00<0003oooooool0?Oooo`030000
oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool00000@?ooo`030000oooooooo
03koool40000?_ooo`030000oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool0
0000@?ooo`030000oooooooo03koool00`000?ooo`00000ooooo0P00047oool00`000?ooooooo`06
oooo000>oooo00<0003oool00000?oooo`030000oooo0000043oool20000?oooo`030000oooooooo
03ooool00`000?ooo`000007oooo000?oooo00<0003oooooool0?oooo`030000oooooooo043oool0
0`000?ooooooo`0noooo0P00043oool00`000?ooooooo`06oooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo000:ooooo`0001D00002oooo000:
oooo00<0003oooooool00oooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`02oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooo`030000oooooooo03ooool00`000?ooooooo`0ooooo00<0
003oooooool0?oooo`030000oooooooo03ooool00`000?ooooooo`02oooo0@00007oool1oooo0004
oooo00<0003oooooool00oooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<0003ooooo3oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1
oooo0004oooo00<0003oooooool00oooo`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00
025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q00008GalO263Ph=S
00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3
Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00025lO7`Q
Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q00008Gal
O263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3Pb40000Q
O7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00
025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q
00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3
Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
Ph>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool200001?oo
ob63Ph<Q00008GalO263Ph=S00008GalO0Coool200000_ooo`002_ooo`030000oooooooo00?ooolQ
Ph>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001
oooo0Oooo`002_ooo`800004oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1?ooo`800002oooo000:
oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1
oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00
007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3
P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000Q
Ph>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@00
0263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q
00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98Gal
O240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQ
O7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9
RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool200001?oood800012Ph>38HV9RB5lO7`Q
00008H>3P`Coool200000_ooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@00
0263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q
00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98Gal
O240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQ
O7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9
RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<Q
RHV98GalO240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3
Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012
Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P00
04:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2
0000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
od800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1oooo0003oooo0`0000Coool0
0`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Oooo`000ooo
o`030000oooooooo00Coool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Ooo
o`400001oooo0Oooo`001?ooo`030000oooooooo00?oool300000ooood800012Ph>38HV9RB5lO7`Q
00008H>3P`?oool300000_ooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03oooo@X>3
Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`030000
oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool000Coool00`000?oo
ooooo`03oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0
003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
od:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2
Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool000[oool200001?oood:3Ph<Q00008H>3
Pf<0000QO7al1?ooo`800002oooo000:oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000Q
O7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph=S00008Gal
O0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00
007oool1oooo000:oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1
oooo000:oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo00<0003oooooool00ooood:3Ph<Q00008H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom2Ph>38@000263Ph=S00008GalO0Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo@X>3Pb40000QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo0P00
00Cooom2Ph>38@000263Ph=S00008GalO0Coool200000_ooo`002_ooo`030000oooooooo00?ooon4
Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00
025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a2
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4
Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00
025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a2
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Ooo
o`002_ooo`800004ooooQ8>3Pb40000QO7al@P0000Coool200000_ooo`002_ooo`030000oooooooo
00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4
Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00
025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a2
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4
Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>38@00
025lO7a200001Oooo`400001oooo0Oooo`001?ooo`<00003oooo00<0003oooooool00oooohB3Ph<Q
00008GalO4800005oooo0@00007oool1oooo0005oooo00<0003oooooool00_ooo`030000oooooooo
00?ooon4Ph>38@00025lO7a200001Oooo`400001oooo0Oooo`000oooo`@00003oooo0`0000?ooon4
Ph>38@00025lO7a200000oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo0004oooo0P0000Coool0
0`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`001Ooo
o`030000oooooooo00;oool00`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q00008GalO263Ph=S00008Gal
O0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000Q
O7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00
025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q00008GalO263Ph=S
00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3
Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00025lO7`Q
Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q00008Gal
O263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3Pb40000Q
O7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00
025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q
00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3
Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
Ph>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo0P0000CooolQPh>38@00
025lO7`QPh>3H`00025lO7`4oooo0P0000;oool000[oool00`000?ooooooo`03oooo8H>3Pb40000Q
O7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQPh>38@00
025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob63Ph<Q
00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8H>3
Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
Ph>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8H>3Pb40000QO7al8H>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQPh>38@00025lO7`QPh>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob63Ph<Q00008GalO263Ph=S00008GalO0Goool100000Oooo`7oool0
00[oool200001?ooob63Ph<Q00008GalO263Ph=S00008GalO0Coool200000_ooo`002_ooo`030000
oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0
003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1oooo000:
oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1
oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5oooo0@00
007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3
P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000Q
Ph>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@00
0263Ph<5oooo0@00007oool1oooo000:oooo0P0000Cooom20000@X>3Pb69RHTQO7al8@000263Ph<4
oooo0P0000;oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000QPh>31Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@000263Ph<5
oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3
P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98GalO240000Q
Ph>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQO7al8@00
0263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q
00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<QRHV98Gal
O240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3Pb69RHTQ
O7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012Ph>38HV9
RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P0004:3Ph<Q
RHV98GalO240000QPh>31Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom20000@X>3
Pb69RHTQO7al8@000263Ph<5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooood800012
Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo@P00
04:3Ph<QRHV98GalO240000QPh>31Oooo`400001oooo0Oooo`000oooo`800005oooo00<0003ooooo
ool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Oooo`7oool000?oool00`000?oo
o`000004oooo00<0003oooooool00ooood800012Ph>38HV9RB5lO7`Q00008H>3P`Goool100000Ooo
o`7oool000?oool00`000?ooo`000004oooo0`0000?ooom20000@X>3Pb69RHTQO7al8@000263Ph<3
oooo0`0000;oool000?oool200001Oooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9
RB40000QO7al1Oooo`400001oooo0Oooo`000oooo`030000oooooooo00Coool00`000?ooooooo`03
oooo@X>3Pb40000QPh>38@000269RHTQ00008GalO0Goool100000Oooo`7oool000Coool200001?oo
o`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000Q
O7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9
RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q
00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@00
0263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2
Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000Q
O7al1Oooo`400001oooo0Oooo`002_ooo`800004oooo@X>3Pb40000QPh>38@000269RHTQ00008Gal
O0Coool200000_ooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000Q
O7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9
RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q
00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@00
0263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2
Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000QO7al1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9RB40000Q
O7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q00008HV9
RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@000263Ph<Q
00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2Ph>38@00
0263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooom2
Ph>38@000263Ph<Q00008HV9RB40000QO7al1Oooo`400001oooo0Oooo`002_ooo`800004oooo@X>3
Pb40000QPh>38@000269RHTQ00008GalO0Coool200000_ooo`002_ooo`030000oooooooo00?ooon4
Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooohB3Ph=S00008Gal
O0Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooohB3Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4
Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooohB3Ph=S00008Gal
O0Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooohB3Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4
Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooohB3Ph=S00008Gal
O0Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>3H`00025lO7`5oooo0@00007oool1oooo000:
oooo0P0000Cooon4Ph>3H`00025lO7`4oooo0P0000;oool000[oool00`000?ooooooo`03ooooQ8>3
Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>3H`00025lO7`5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooohB3Ph=S00008GalO0Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooon4Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooohB3Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooQ8>3
Pf<0000QO7al1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooon4Ph>3H`00025lO7`5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooohB3Ph=S00008GalO0Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooon4Ph>3H`00025lO7`5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooohB3Ph=S00008GalO0Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooQ8>3
Pf<0000QO7al1Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03
ooooQ8>3Pf<0000QO7al1Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?oo
ooooo`03ooooQ8>3Pf<0000QO7al1Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool3
00000oooohB3Ph=S00008GalO0?oool300000_ooo`001?ooo`030000oooooooo00?oool00`000?oo
ooooo`3ooooo4Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`3o
oooo4Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`3ooooo4Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?oool00`000?ooooooo`0=oooo00<0003ooooo
ool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000
oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool0
0`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`0=
oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool00_ooo`400001oooo0Oooo`002_ooool0000E00000_ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.489418, -0.536766, 0.0303025, 
  0.0303025}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[InverseFourier[compressedfSpec], Mesh -> False]\)], 
  "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.120192 0.0192308 0.120192 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.25962 -0.0125 -3 -9 ]
[.25962 -0.0125 3 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.74038 -0.0125 -3 -9 ]
[.74038 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -6 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -6 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.25962 0 m
.25962 .00625 L
s
[(2)] .25962 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(4)] .5 -0.0125 0 1 Mshowa
.74038 0 m
.74038 .00625 L
s
[(6)] .74038 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(8)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.07933 0 m
.07933 .00375 L
s
.13942 0 m
.13942 .00375 L
s
.19952 0 m
.19952 .00375 L
s
.31971 0 m
.31971 .00375 L
s
.37981 0 m
.37981 .00375 L
s
.4399 0 m
.4399 .00375 L
s
.5601 0 m
.5601 .00375 L
s
.62019 0 m
.62019 .00375 L
s
.68029 0 m
.68029 .00375 L
s
.80048 0 m
.80048 .00375 L
s
.86058 0 m
.86058 .00375 L
s
.92067 0 m
.92067 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(2)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(4)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(6)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(8)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .07933 m
.00375 .07933 L
s
0 .13942 m
.00375 .13942 L
s
0 .19952 m
.00375 .19952 L
s
0 .31971 m
.00375 .31971 L
s
0 .37981 m
.00375 .37981 L
s
0 .4399 m
.00375 .4399 L
s
0 .5601 m
.00375 .5601 L
s
0 .62019 m
.00375 .62019 L
s
0 .68029 m
.00375 .68029 L
s
0 .80048 m
.00375 .80048 L
s
0 .86058 m
.00375 .86058 L
s
0 .92067 m
.00375 .92067 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.25962 .99375 m
.25962 1 L
s
.5 .99375 m
.5 1 L
s
.74038 .99375 m
.74038 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.07933 .99625 m
.07933 1 L
s
.13942 .99625 m
.13942 1 L
s
.19952 .99625 m
.19952 1 L
s
.31971 .99625 m
.31971 1 L
s
.37981 .99625 m
.37981 1 L
s
.4399 .99625 m
.4399 1 L
s
.5601 .99625 m
.5601 1 L
s
.62019 .99625 m
.62019 1 L
s
.68029 .99625 m
.68029 1 L
s
.80048 .99625 m
.80048 1 L
s
.86058 .99625 m
.86058 1 L
s
.92067 .99625 m
.92067 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .07933 m
1 .07933 L
s
.99625 .13942 m
1 .13942 L
s
.99625 .19952 m
1 .19952 L
s
.99625 .31971 m
1 .31971 L
s
.99625 .37981 m
1 .37981 L
s
.99625 .4399 m
1 .4399 L
s
.99625 .5601 m
1 .5601 L
s
.99625 .62019 m
1 .62019 L
s
.99625 .68029 m
1 .68029 L
s
.99625 .80048 m
1 .80048 L
s
.99625 .86058 m
1 .86058 L
s
.99625 .92067 m
1 .92067 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
8 string
8 8 8 [8 0 0 8 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
0000000000000000
0000000000000000
0000000000000000
0000000000000000
0000000000000000
0000000000000000
0000000000000000
0000000000000000
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
00ooool00`000?ooooooo`0noooo0`00043oool30000?_ooo`800011oooo00<0003oooooool01_oo
o`003_ooo`030000oooo000003ooool00`000?ooooooo`11oooo00<0003oooooool0?Oooo`030000
oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool00000@?ooo`030000oooooooo
03koool40000?_ooo`030000oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool0
0000@?ooo`030000oooooooo03koool00`000?ooo`00000ooooo0P00047oool00`000?ooooooo`06
oooo000>oooo00<0003oool00000?oooo`030000oooo0000043oool20000?oooo`030000oooooooo
03ooool00`000?ooo`000007oooo000?oooo00<0003oooooool0?oooo`030000oooooooo043oool0
0`000?ooooooo`0noooo0P00043oool00`000?ooooooo`06oooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo000:ooooo`0001D00002oooo000:
oooo00<0003oooooool00oooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`02oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooo`030000oooooooo03ooool00`000?ooooooo`0ooooo00<0
003oooooool0?oooo`030000oooooooo03ooool00`000?ooooooo`02oooo0@00007oool1oooo0004
oooo00<0003oooooool00oooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<0003ooooo3oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000?oool00`000?ooo`000004oooo00<0003ooooo
ool00oooool0000900001Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03
ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`800004ooooo`0000T00004oooo0P00
00;oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03
ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@00
00Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?ooool0000900001?oo
o`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03
ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@00
00Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`00
00T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`800004ooooo`0000T00004oooo0P0000;oool0
00[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@00
00Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`00
00T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo0003
oooo0`0000Coool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo0003oooo00<0
003oooooool01?ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000Coool0
0`000?ooooooo`03oooo0`0000?ooooo00002@0000?oool300000_ooo`001?ooo`030000oooooooo
00?oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000Coool00`000?oo
ooooo`03oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`00
00T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?oo
ool0000900001?ooo`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo0P00
00Cooooo00002@0000Coool200000_ooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?ooool00009
00001?ooo`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03
ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Ooo
o`400001oooo0Oooo`001?ooo`<00003oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`001Oooo`030000oooooooo00;oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo0003oooo100000?oool300000oooool0000900000oooo`<00002oooo0003oooo00<0
003oool000001?ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000Coool2
00001?ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000Goool00`000?oo
ooooo`02oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`00
00T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?oo
ool0000900001?ooo`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo0P00
00Cooooo00002@0000Coool200000_ooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?ooool00009
00001?ooo`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03
ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Ooo
o`400001oooo0Oooo`000oooo`800005oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo0003oooo00<0003oool000001?ooo`<00003ooooo`0000T00003oooo0`0000;oool0
00?oool200001Oooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000?oool0
0`000?ooooooo`04oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`001?oo
o`800004oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`00
00T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?oo
ool0000900001?ooo`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo0P00
00Cooooo00002@0000Coool200000_ooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool200001?ooool00009
00001?ooo`800002oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool00009
00001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo00002@0000Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooooo
00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03ooooo`0000T00005oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooool0000900001Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooooo00002@0000Goool100000Oooo`7oool000[oool00`000?ooooooo`03
ooooo`0000T00005oooo0@00007oool1oooo000:oooo00<0003oooooool00oooool0000900001Ooo
o`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03ooooo`0000T00005
oooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`030000oooooooo00?ooooo00002@00
00Goool100000Oooo`7oool000?oool00`000?ooo`000004oooo0`0000?ooooo00002@0000?oool3
00000_ooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`3ooooo4Oooo`400001oooo0Ooo
o`000oooo`030000oooo000000Coool00`000?ooooooo`3ooooo4Oooo`400001oooo0Oooo`001?oo
o`030000oooooooo00?oool00`000?ooooooo`3ooooo4Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?oool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool0
0`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`0=
oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool00_ooo`400001oooo0Oooo`002_oo
ool0000E00000_ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.489418, -0.536766, 0.0303025, 
  0.0303025}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    \(\[FilledSquare]\ \ Wavelet\ Transform\)], "Input"],

Cell[BoxData[
    \(\(w = waveletMatrix[8, coif6]; \)\)], "Input"],

Cell[BoxData[
    \(\(wa = w . a . Transpose[w]; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[wa, Mesh -> False]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.120192 0.0192308 0.120192 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.25962 -0.0125 -3 -9 ]
[.25962 -0.0125 3 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.74038 -0.0125 -3 -9 ]
[.74038 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -6 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -6 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.25962 0 m
.25962 .00625 L
s
[(2)] .25962 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(4)] .5 -0.0125 0 1 Mshowa
.74038 0 m
.74038 .00625 L
s
[(6)] .74038 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(8)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.07933 0 m
.07933 .00375 L
s
.13942 0 m
.13942 .00375 L
s
.19952 0 m
.19952 .00375 L
s
.31971 0 m
.31971 .00375 L
s
.37981 0 m
.37981 .00375 L
s
.4399 0 m
.4399 .00375 L
s
.5601 0 m
.5601 .00375 L
s
.62019 0 m
.62019 .00375 L
s
.68029 0 m
.68029 .00375 L
s
.80048 0 m
.80048 .00375 L
s
.86058 0 m
.86058 .00375 L
s
.92067 0 m
.92067 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(2)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(4)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(6)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(8)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .07933 m
.00375 .07933 L
s
0 .13942 m
.00375 .13942 L
s
0 .19952 m
.00375 .19952 L
s
0 .31971 m
.00375 .31971 L
s
0 .37981 m
.00375 .37981 L
s
0 .4399 m
.00375 .4399 L
s
0 .5601 m
.00375 .5601 L
s
0 .62019 m
.00375 .62019 L
s
0 .68029 m
.00375 .68029 L
s
0 .80048 m
.00375 .80048 L
s
0 .86058 m
.00375 .86058 L
s
0 .92067 m
.00375 .92067 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.25962 .99375 m
.25962 1 L
s
.5 .99375 m
.5 1 L
s
.74038 .99375 m
.74038 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.07933 .99625 m
.07933 1 L
s
.13942 .99625 m
.13942 1 L
s
.19952 .99625 m
.19952 1 L
s
.31971 .99625 m
.31971 1 L
s
.37981 .99625 m
.37981 1 L
s
.4399 .99625 m
.4399 1 L
s
.5601 .99625 m
.5601 1 L
s
.62019 .99625 m
.62019 1 L
s
.68029 .99625 m
.68029 1 L
s
.80048 .99625 m
.80048 1 L
s
.86058 .99625 m
.86058 1 L
s
.92067 .99625 m
.92067 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .07933 m
1 .07933 L
s
.99625 .13942 m
1 .13942 L
s
.99625 .19952 m
1 .19952 L
s
.99625 .31971 m
1 .31971 L
s
.99625 .37981 m
1 .37981 L
s
.99625 .4399 m
1 .4399 L
s
.99625 .5601 m
1 .5601 L
s
.99625 .62019 m
1 .62019 L
s
.99625 .68029 m
1 .68029 L
s
.99625 .80048 m
1 .80048 L
s
.99625 .86058 m
1 .86058 L
s
.99625 .92067 m
1 .92067 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
8 string
8 8 8 [8 0 0 8 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
D2B289F3163F3942
FFAE8AC534442CB1
D2C2A5FF191D1E45
FFA18CCF31620EB3
304F635C4A9C0D73
510C4AAC345F7C4A
2E7B6B425237566E
51003FAC327A794A
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
00ooool00`000?ooooooo`0noooo0`00043oool30000?_ooo`800011oooo00<0003oooooool01_oo
o`003_ooo`030000oooo000003ooool00`000?ooooooo`11oooo00<0003oooooool0?Oooo`030000
oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool00000@?ooo`030000oooooooo
03koool40000?_ooo`030000oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool0
0000@?ooo`030000oooooooo03koool00`000?ooo`00000ooooo0P00047oool00`000?ooooooo`06
oooo000>oooo00<0003oool00000?oooo`030000oooo0000043oool20000?oooo`030000oooooooo
03ooool00`000?ooo`000007oooo000?oooo00<0003oooooool0?oooo`030000oooooooo043oool0
0`000?ooooooo`0noooo0P00043oool00`000?ooooooo`06oooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo000:ooooo`0001D00002oooo000:
oooo00<0003oooooool00oooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`02oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooo`030000oooooooo03ooool00`000?ooooooo`0ooooo00<0
003oooooool0?oooo`030000oooooooo03ooool00`000?ooooooo`02oooo0@00007oool1oooo0004
oooo00<0003oooooool00oooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<0003ooooo3oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Oooo`7oool000?oool0
0`000?ooo`000004oooo00<0003oooooool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ
@T921Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03oooo8M;B
dR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo
?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8K:b/R69RHTQ
lo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;B
dR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo
?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8K:b/R69RHTQ
lo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;B
dR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_oo
o`800004oooo8M;BdR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T84oooo0P0000;oool000[oool0
0`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ
>CTi8D92@PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?c
lb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q
/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ
>CTi8D92@PGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?c
lb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q
/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ>CTi8D92@PGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8M;BdR6b/[8QRHV98O?clb4F5QHQ?clo8CTi>B52@T85oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo?b4i>CTQ@T921Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8K:b/R69RHTQlo?c8AHF5R4o?clQ
>CTi8D92@PGoool100000Oooo`7oool000[oool200001?ooob7Bd]8Q/[:b8HV9RB7clo<Q5QHF8Clo
?b4i>CTQ@T921?ooo`800002oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@Q
A4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5
aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQ
RXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0
003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45
oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/
;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@Q
A4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5
aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQ
RXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0
003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:
oooo0P0002GooolQ[Zj^8HZ:RR75aLDQ=3@d8DA4A24/;2`Q/K6a1?ooo`800002oooo000:oooo00<0
003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00
007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45
oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/
;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@Q
A4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5
aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?ooob6^[ZhQ
RXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003oooooool09?oo
ob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0003ooooo
ool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:oooo00<0
003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo000:
oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1
oooo000:oooo00<0003oooooool09?ooob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00
007oool1oooo0003oooo0`0000Coool00`000?ooooooo`0Toooo8Jj^[R6:RXXQaLG58C@d=254A4@Q
;2`/8K6a/@Goool100000Oooo`7oool000?oool00`000?ooooooo`04oooo00<0003oooooool09?oo
ob6^[ZhQRXZ:8LG5aB4d=3@QA4A48B`/;26a/K45oooo0@00007oool1oooo0004oooo00<0003ooooo
ool00oooo`<0000Toooo8Jj^[R6:RXXQaLG58C@d=254A4@Q;2`/8K6a/@?oool300000_ooo`001?oo
o`030000oooooooo00?oool00`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN
7R55ADD5oooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`030000oooooooo00?ooolQ
d];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool100000Oooo`7oool000Coool00`000?oo
ooooo`03oooo00<0003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ
7QhN8DE5A@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Ooo
ob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q
`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ
7QhN8DE5A@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Ooo
ob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q
`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ
7QhN8DE5A@Goool100000Oooo`7oool000[oool200001?ooob7Bd]8Q`/;28JFUYB7ooolQ6ATI8AdM
7B4N7QhQADE51?ooo`800002oooo000:oooo00<0003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ
6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8L;2
`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN
7R55ADD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ
6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8L;2
`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN
7R55ADD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7Bd]8Q`/;28JFUYB7ooolQ
6ATI8AdM7B4N7QhQADE51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQd];B8L;2
`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8M;BdR72`/8QYJFU8Oooob4I6ATQ7AdM8AhN7R55ADD5oooo0@00007oool1oooo000:oooo0P00
00CooolQd];B8L;2`R6UYJDQoooo8ATI6B4M7AdQ7QhN8DE5A@Coool200000_ooo`002_ooo`030000
oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ
/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9R
HR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ
<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<
S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
XJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ
/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9R
HR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`80000Uoooo8J6QXB6<S8`Qclo?8C4a<B5RHV8Q
3Ph>8K>c/`Coool200000_ooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9R
HR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ
<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<
S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
XJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ
/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9R
HR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ
<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQXJ6Q8Hb<
S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Oooo`400001oooo0Oooo`001?ooo`<00003oooo00<0003ooooo
ool09?ooob6QXJ4QS8b<8Lo?cb4a<C4QHV9R8@h>3R6c/k<5oooo0@00007oool1oooo0005oooo00<0
003oooooool00_ooo`030000oooooooo02CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ/k>c1Ooo
o`400001oooo0Oooo`000oooo`@00003oooo0`0002CooolQXJ6Q8Hb<S27?cllQ<C4a8F9RHR4>3PhQ
/k>c0oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo00?ooolQ<30`8Dm?
Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000Coool200001?ooo`030000
oooooooo00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool0
00Goool00`000?ooooooo`02oooo00<0003oooooool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbL
W24=3@dQLg=c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<Q
G5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8C0`
<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbL
W24=3@dQLg=c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<Q
G5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8C0`
<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbL
W24=3@dQLg=c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<Q
G5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000[oool200001?ooob4`<30QCdm?8F=S
Hb5LG5`QBTY:8IbLW24=3@dQLg=c1?ooo`800002oooo000:oooo00<0003oooooool00oooob4`<30Q
Cdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q
3@d=8G=cL`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaL
G25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4`<30Q
Cdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob4`<30QCdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q
3@d=8G=cL`Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaL
G25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4`<30Q
Cdm?8F=SHb5LG5`QBTY:8IbLW24=3@dQLg=c1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8C0`<25?CdlQHf=S8EaLG25:BTXQW9bL8@d=3B5cLg<5oooo0@00007oool1
oooo000:oooo0P0000CooolQ<30`8Dm?Cb5SHf<QG5aL8DY:BR6LW9`Q3@d=8G=cL`Coool200000_oo
o`002_ooo`030000oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8Gal
O25:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q
=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@`<
325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8GalO25:BTX5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8Gal
O25:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q
=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@`<
325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8GalO25:BTX5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8Gal
O25:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q
=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@`<
325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool100000Oooo`7oool000[oool200001?ooob5ADE4Q
30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1?ooo`800002oooo000:oooo00<0003oooooool00ooo
ob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8GalO25:BTX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`Q
BTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d
=25OGelQO7al8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB4<30`Q
BTY:8Jb/[24d=3@QGemO8GalO25:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d=25OGelQO7al8DY:BPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8GalO25:BTX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`Q
BTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@`<325:BTXQ[:b/8C@d
=25OGelQO7al8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB4<30`Q
BTY:8Jb/[24d=3@QGemO8GalO25:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1Oooo`400001oooo0Oooo`000oooo`800005
oooo00<0003oooooool00oooob5ADE4Q30`<8DY:BR6/[:`Q=3@d8EmOGb5lO7`QBTY:1Oooo`400001
oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`03oooo8E5ADB4<30`QBTY:8Jb/
[24d=3@QGemO8GalO25:BTX5oooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`<00003
oooo8E5ADB4<30`QBTY:8Jb/[24d=3@QGemO8GalO25:BTX3oooo0`0000;oool000?oool200001Ooo
o`030000oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^KPGoool100000Ooo
o`7oool000?oool00`000?ooooooo`04oooo00<0003oooooool00oooob4^;RhQNg]k8F][Jb52@T8Q
DU9B8CLg=b5FEUHQKVi^1Oooo`400001oooo0Oooo`001?ooo`800004oooo00<0003oooooool00ooo
ob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q=cLg8EIFER5^KVh5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQ
KVi^1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9B
DR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Bh^;R5kNg/Q
Jf][8D92@R5BDU8Q=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q=cLg8EIFER5^KVh5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQ
KVi^1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9B
DR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Bh^;R5kNg/Q
Jf][8D92@R5BDU8Q=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool0
00[oool200001?ooob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1?ooo`800002oooo000:
oooo00<0003oooooool00oooob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^
KPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q
=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4^;RhQNg]k8F][
Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^
KPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q
=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob4^;RhQNg]k8F][
Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^KPGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob4^;RhQNg]k8F][Jb52@T8QDU9B8CLg=b5FEUHQKVi^1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQ;Rh^8G]kNb5[Jf/Q@T928E9BDR4g=cLQEUIF8Fi^
KPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Bh^;R5kNg/QJf][8D92@R5BDU8Q
=cLg8EIFER5^KVh5oooo0@00007oool1oooo000:oooo0P0000CooolQ;Rh^8G]kNb5[Jf/Q@T928E9B
DR4g=cLQEUIF8Fi^KPCoool200000_ooo`002_ooo`030000oooooooo00?ooolQDE5A8@00024o?clQ
[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5A
DB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8E5ADB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYj
NR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@00024o?clQ
[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5A
DB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8E5ADB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYj
NR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@00024o?clQ
[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5A
DB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Ooo
o`7oool000[oool200001?ooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1?ooo`800002
oooo000:oooo00<0003oooooool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQ
NGUi8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB40000Q?clo8Jb/
[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q
00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8E5ADB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQ
NGUi8DY:BPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8E5ADB40000Q?clo8Jb/
[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5ADE4Q
00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8E5ADB40000Q?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob5ADE4Q00008Clo?b6/[:`Q<S8b8GYjNR5iNGTQBTY:1Ooo
o`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03oooo8E5ADB40000Q
?clo8Jb/[24b<S8QNWYj8GUiNB5:BTX5oooo0@00007oool1oooo0003oooo00<0003oool000001?oo
o`030000oooooooo00?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQNGUi8DY:BPGoool100000Ooo
o`7oool000?oool00`000?ooo`000004oooo0`0000?ooolQDE5A8@00024o?clQ[:b/8C8b<R5jNWXQ
NGUi8DY:BP?oool300000_ooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`3ooooo4Ooo
o`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`3ooooo4Oooo`400001
oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`3ooooo4Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?oool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003ooooo
ool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool00_ooo`400001
oooo0Oooo`002_ooool0000E00000_ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.489418, -0.536766, 0.0303025, 
  0.0303025}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    RowBox[{"\[FilledCircle]", " ", 
      StyleBox["Compression",
        FontVariations->{"Underline"->True}]}]], "Input"],

Cell[BoxData[
    \(\(n = 8; \)\)], "Input"],

Cell[TextData[StyleBox[
"nPrime=4;\ncutwa=TakeMatrix[wa,{1,1},{4,4}];\n\
compressedwSpec=BlockMatrix[{{cutwa,ZeroMatrix[nPrime,n-nPrime]},\n\t\
{ZeroMatrix[n-nPrime,n]}}];",
  FontSize->14]], "Input",
  PageBreakWithin->Automatic,
  FontSize->16],

Cell[BoxData[
    \(\(b = Transpose[w] . compressedwSpec . w; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[Transpose[w] . compressedwSpec . w, Mesh -> False]\)], 
  "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.120192 0.0192308 0.120192 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.25962 -0.0125 -3 -9 ]
[.25962 -0.0125 3 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.74038 -0.0125 -3 -9 ]
[.74038 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -6 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -6 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.25962 0 m
.25962 .00625 L
s
[(2)] .25962 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(4)] .5 -0.0125 0 1 Mshowa
.74038 0 m
.74038 .00625 L
s
[(6)] .74038 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(8)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.07933 0 m
.07933 .00375 L
s
.13942 0 m
.13942 .00375 L
s
.19952 0 m
.19952 .00375 L
s
.31971 0 m
.31971 .00375 L
s
.37981 0 m
.37981 .00375 L
s
.4399 0 m
.4399 .00375 L
s
.5601 0 m
.5601 .00375 L
s
.62019 0 m
.62019 .00375 L
s
.68029 0 m
.68029 .00375 L
s
.80048 0 m
.80048 .00375 L
s
.86058 0 m
.86058 .00375 L
s
.92067 0 m
.92067 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(2)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(4)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(6)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(8)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .07933 m
.00375 .07933 L
s
0 .13942 m
.00375 .13942 L
s
0 .19952 m
.00375 .19952 L
s
0 .31971 m
.00375 .31971 L
s
0 .37981 m
.00375 .37981 L
s
0 .4399 m
.00375 .4399 L
s
0 .5601 m
.00375 .5601 L
s
0 .62019 m
.00375 .62019 L
s
0 .68029 m
.00375 .68029 L
s
0 .80048 m
.00375 .80048 L
s
0 .86058 m
.00375 .86058 L
s
0 .92067 m
.00375 .92067 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.25962 .99375 m
.25962 1 L
s
.5 .99375 m
.5 1 L
s
.74038 .99375 m
.74038 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.07933 .99625 m
.07933 1 L
s
.13942 .99625 m
.13942 1 L
s
.19952 .99625 m
.19952 1 L
s
.31971 .99625 m
.31971 1 L
s
.37981 .99625 m
.37981 1 L
s
.4399 .99625 m
.4399 1 L
s
.5601 .99625 m
.5601 1 L
s
.62019 .99625 m
.62019 1 L
s
.68029 .99625 m
.68029 1 L
s
.80048 .99625 m
.80048 1 L
s
.86058 .99625 m
.86058 1 L
s
.92067 .99625 m
.92067 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .07933 m
1 .07933 L
s
.99625 .13942 m
1 .13942 L
s
.99625 .19952 m
1 .19952 L
s
.99625 .31971 m
1 .31971 L
s
.99625 .37981 m
1 .37981 L
s
.99625 .4399 m
1 .4399 L
s
.99625 .5601 m
1 .5601 L
s
.99625 .62019 m
1 .62019 L
s
.99625 .68029 m
1 .68029 L
s
.99625 .80048 m
1 .80048 L
s
.99625 .86058 m
1 .86058 L
s
.99625 .92067 m
1 .92067 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
8 string
8 8 8 [8 0 0 8 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
8FBBFFA54B483364
BBBACA9663522F74
EDBA8F847A5A2785
B9B6C39870582D72
7AB2FEAE6452305A
B9BAC79F7965417D
FFC2898F947D58A5
C9BEC19971634585
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
00ooool00`000?ooooooo`0noooo0`00043oool30000?_ooo`800011oooo00<0003oooooool01_oo
o`003_ooo`030000oooo000003ooool00`000?ooooooo`11oooo00<0003oooooool0?Oooo`030000
oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool00000@?ooo`030000oooooooo
03koool40000?_ooo`030000oooo000003ooool00`000?ooo`000007oooo000>oooo00<0003oool0
0000@?ooo`030000oooooooo03koool00`000?ooo`00000ooooo0P00047oool00`000?ooooooo`06
oooo000>oooo00<0003oool00000?oooo`030000oooo0000043oool20000?oooo`030000oooooooo
03ooool00`000?ooo`000007oooo000?oooo00<0003oooooool0?oooo`030000oooooooo043oool0
0`000?ooooooo`0noooo0P00043oool00`000?ooooooo`06oooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo000:ooooo`0001D00002oooo000:
oooo00<0003oooooool00oooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool03Oooo`030000oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000
oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool0
0`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>
oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`02oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooo`030000oooooooo03ooool00`000?ooooooo`0ooooo00<0
003oooooool0?oooo`030000oooooooo03ooool00`000?ooooooo`02oooo0@00007oool1oooo0004
oooo00<0003oooooool00oooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo0?oooolAoooo0@00007oool1oooo0003oooo00<0003oool0
00001?ooo`<0003ooooo3oooo`<00002oooo0003oooo00<0003oool000001?ooo`030000oooooooo
00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Oooo`7oool000?oool0
0`000?ooo`000004oooo00<0003oooooool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<Q
I6AT1Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03oooo8Hn?
Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8
B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQShn?8K^k^b7ooolQ
YJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Hn?
Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8
B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQShn?8K^k^b7ooolQ
YJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Hn?
Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_oo
o`800004oooo8Hn?Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@4oooo0P0000;oool000[oool0
0`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ
<c<c8FATI0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFU
YB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6?ShlQ
^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ
<c<c8FATI0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFU
YB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6?ShlQ
^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ<c<c8FATI0Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8Hn?Sb6k^k/Qoooo8JFUYB5;Bd/QB4Q88C<c<b5TI6@5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8B24c<c<QI6AT1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQShn?8K^k^b7ooolQYJFU8D];Bb58B4PQ
<c<c8FATI0Goool100000Oooo`7oool000[oool200001?ooob6?ShlQ^k^k8Oooob6UYJDQBd];8DQ8
B24c<c<QI6AT1?ooo`800002oooo000:oooo00<0003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQ
Hf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^k^k8KZj
^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_
;b5dM7@5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQ
Hf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^k^k8KZj
^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_
;b5dM7@5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQ
Hf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^k^k8KZj
^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Ooo
o`002_ooo`800004oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@4oooo0P0000;oool0
00[oool00`000?ooooooo`03oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQ
M7Ad1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=S
Hb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8K^k^b6j^[XQ
b/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQ
M7Ad1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=S
Hb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8K^k^b6j^[XQ
b/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob6k^k/Q^[Zj8L[:bR6FUYHQHf=S8E9BDR4_;blQM7Ad1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<QDU9B8Bl_;b5dM7@5oooo0@00
007oool1oooo0003oooo0`0000Coool00`000?ooooooo`03oooo8K^k^b6j^[XQb/[:8IJFUR5SHf<Q
DU9B8Bl_;b5dM7@5oooo0@00007oool1oooo0003oooo00<0003oooooool01?ooo`030000oooooooo
00?ooolQ^k^k8KZj^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0Goool100000Oooo`7oool000Coool0
0`000?ooooooo`03oooo0`0000?ooolQ^k^k8KZj^R7:b/XQUYJF8F=SHb5BDU8Q;bl_8GAdM0?oool3
00000_ooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4
Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1oooo0003oooo00<0003oool000001?ooo`030000
oooooooo00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool0
00Coool00`000?ooooooo`03oooo00<0003oooooool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJ
FR4W9bLQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQ
Q8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Ng]
kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJ
FR4W9bLQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQ
Q8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Ng]
kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJ
FR4W9bLQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQ
Q8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool000[oool200001?ooob7]kNdQ^[Zj8Hn?
Sb64Q8@QNWYj8EYJFR4W9bLQQHF51?ooo`800002oooo000:oooo00<0003oooooool00oooob7]kNdQ
^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ
9bLW8HF5Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4
Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7]kNdQ
^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob7]kNdQ^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ
9bLW8HF5Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4
Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob7]kNdQ
^[Zj8Hn?Sb64Q8@QNWYj8EYJFR4W9bLQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8Ng]kB6j^[XQShn?8HB4Q25jNWXQFUYJ8BLW9b65QHD5oooo0@00007oool1
oooo000:oooo0P0000CooolQkNg]8KZj^R6?ShlQQ8B48GYjNR5JFUXQ9bLW8HF5Q@Coool200000_oo
o`002_ooo`030000oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd]
;B5bLW85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQ
L71`8EQHF24];BdQLW9b1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KJf
]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd];B5bLW85oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd]
;B5bLW85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQ
L71`8EQHF24];BdQLW9b1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KJf
]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool000[oool00`000?ooooooo`03
oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd];B5bLW85oooo0@00007oool1oooo000:oooo00<0
003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool1
00000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd]
;B5bLW85oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQ
L71`8EQHF24];BdQLW9b1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KJf
]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool000[oool200001?ooob6i^KTQ
][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1?ooo`800002oooo000:oooo00<0003oooooool00ooo
ob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd];B5bLW85oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQ
LW9b1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`
L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6f][HQ
`l?38IRHV25`L70QF5QH8Bd];B5bLW85oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8KVi^B6f][HQ`l?38IRHV25`L70QF5QH8Bd];B5bLW85oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQ
LW9b1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KJf]R73`l<QV9RH8G1`
L25HF5PQ;Bd]8G9bLPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6f][HQ
`l?38IRHV25`L70QF5QH8Bd];B5bLW85oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1Oooo`400001oooo0Oooo`001?ooo`<00003
oooo00<0003oooooool00oooob6i^KTQ][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b1Oooo`400001
oooo0Oooo`001Oooo`030000oooooooo00;oool00`000?ooooooo`03oooo8KVi^B6f][HQ`l?38IRH
V25`L70QF5QH8Bd];B5bLW85oooo0@00007oool1oooo0003oooo100000?oool300000oooob6i^KTQ
][Jf8L?3`b6HV9PQL71`8EQHF24];BdQLW9b0oooo`<00002oooo0003oooo00<0003oool000001?oo
o`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJFPGoool100000Ooo
o`7oool000Coool200001?ooo`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q
<30`8EYJFPGoool100000Oooo`7oool000Goool00`000?ooooooo`02oooo00<0003oooooool00ooo
ob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJFPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@QDU9B8C0`<25JFUX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30Q
FUYJ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FAT
I25BDU8Q<30`8EYJFPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8GYjNR6b/[8Q
o_kn8Jj^[R5TI6@QDU9B8C0`<25JFUX5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJFPGoool100000Oooo`7oool0
00[oool00`000?ooooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@QDU9B8C0`<25JFUX5oooo0@00
007oool1oooo000:oooo00<0003oooooool00oooob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30Q
FUYJ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FAT
I25BDU8Q<30`8EYJFPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8GYjNR6b/[8Q
o_kn8Jj^[R5TI6@QDU9B8C0`<25JFUX5oooo0@00007oool1oooo000:oooo00<0003oooooool00ooo
ob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJFPGoool100000Oooo`7oool0
00[oool200001?ooob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1?ooo`800002oooo000:
oooo00<0003oooooool00oooob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJ
FPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@Q
DU9B8C0`<25JFUX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5jNWXQ/[:b8Okn
oR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
NWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJFPGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@QDU9B8C0`<25JFUX5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJ
FPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@Q
DU9B8C0`<25JFUX5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob5jNWXQ/[:b8Okn
oR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
NWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJFPGoool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@QDU9B8C0`<25JFUX5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob5jNWXQ/[:b8OknoR6^[ZhQI6AT8E9BDR4`<30QFUYJ1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQNWYj8K:b/R7no_hQ[Zj^8FATI25BDU8Q<30`8EYJ
FPGoool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8GYjNR6b/[8Qo_kn8Jj^[R5TI6@Q
DU9B8C0`<25JFUX5oooo0@00007oool1oooo000:oooo0P0000CooolQNWYj8K:b/R7no_hQ[Zj^8FAT
I25BDU8Q<30`8EYJFPCoool200000_ooo`002_ooo`030000oooooooo00?ooolQ^KVi8KZj^R77alLQ
WinO8GUiNB5UIFDQ@D518GemO@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi
^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ@D518GemO@Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8KVi^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEU
IB51@D4QOGem1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KZj^R77alLQ
WinO8GUiNB5UIFDQ@D518GemO@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi
^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ@D518GemO@Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8KVi^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEU
IB51@D4QOGem1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KZj^R77alLQ
WinO8GUiNB5UIFDQ@D518GemO@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi
^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ@D518GemO@Goool100000Ooo
o`7oool000[oool200001?ooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1?ooo`800002
oooo000:oooo00<0003oooooool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ
@D518GemO@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6j^[XQalO78InO
Wb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ
^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ@D518GemO@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8KVi^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ
@D518GemO@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8KVi^B6j^[XQalO78InO
Wb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob6i^KTQ
^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
00?ooolQ^KVi8KZj^R77alLQWinO8GUiNB5UIFDQ@D518GemO@Goool100000Oooo`7oool000[oool0
0`000?ooooooo`03oooo8KVi^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1
oooo000:oooo00<0003oooooool00oooob6i^KTQ^[Zj8LO7ab6OWilQNGUi8FEUIB51@D4QOGem1Ooo
o`400001oooo0Oooo`000oooo`800005oooo00<0003oooooool00oooob6i^KTQ^[Zj8LO7ab6OWilQ
NGUi8FEUIB51@D4QOGem1Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?oo
ooooo`03oooo8KVi^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd5oooo0@00007oool1oooo0003
oooo00<0003oool000001?ooo`<00003oooo8KVi^B6j^[XQalO78InOWb5iNGTQIFEU8D51@B5mOGd3
oooo0`0000;oool000?oool200001Oooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8Gem
OB5HF5PQYJFU1Oooo`400001oooo0Oooo`000oooo`030000oooooooo00Coool00`000?ooooooo`0T
oooo8L;2`R69RHTQShn?8IBDU25mOGdQF5QH8JFUY@Goool100000Oooo`7oool000Coool200001?oo
o`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQ
YJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8Gem
OB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQ
U9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9
RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQ
YJFU1Oooo`400001oooo0Oooo`002_ooo`80000Uoooo8L;2`R69RHTQShn?8IBDU25mOGdQF5QH8JFU
Y@Coool200000_ooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQ
YJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8Gem
OB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQ
U9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9
RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo
02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000
oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Ooo
o`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQ
YJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQU9BD8Gem
OB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9RB6?ShlQ
U9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ`/;28HV9
RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`030000oooooooo02CooolQ
`/;28HV9RB6?ShlQU9BD8GemOB5HF5PQYJFU1Oooo`400001oooo0Oooo`002_ooo`80000Uoooo8L;2
`R69RHTQShn?8IBDU25mOGdQF5QH8JFUY@Coool200000_ooo`002_ooo`030000oooooooo00?ooolQ
bLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQbLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5
Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4Q
Hf=S8DE5AB65QHD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob79bLTQ_[jn8L71
`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
bLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQbLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5
Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4Q
Hf=S8DE5AB65QHD5oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob79bLTQ_[jn8L71
`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQ
bLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Oooo`7oool000[oool00`000?oo
ooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5oooo0@00007oool1oooo000:
oooo00<0003oooooool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001
oooo0Oooo`002_ooo`030000oooooooo00?ooolQbLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5
Q@Goool100000Oooo`7oool000[oool200001?ooob79bLTQ_[jn8L71`B6IVITQLG5a8F=SHb55ADDQ
QHF51?ooo`800002oooo000:oooo00<0003oooooool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=S
Hb55ADDQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQbLW98Kjn_R71`L4Q
VIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8LW9
bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQbLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=S
Hb55ADDQQHF51Oooo`400001oooo0Oooo`002_ooo`030000oooooooo00?ooolQbLW98Kjn_R71`L4Q
VIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Oooo`7oool000[oool00`000?ooooooo`03oooo8LW9
bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5oooo0@00007oool1oooo000:oooo00<0003ooooo
ool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=SHb55ADDQQHF51Oooo`400001oooo0Oooo`002_oo
o`030000oooooooo00?ooolQbLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5Q@Goool100000Ooo
o`7oool000[oool00`000?ooooooo`03oooo8LW9bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5
oooo0@00007oool1oooo000:oooo00<0003oooooool00oooob79bLTQ_[jn8L71`B6IVITQLG5a8F=S
Hb55ADDQQHF51Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`03
oooo8LW9bB6n_[hQ`L718IVIVB5aLG4QHf=S8DE5AB65QHD5oooo0@00007oool1oooo0003oooo00<0
003oool000001?ooo`030000oooooooo00?ooolQbLW98Kjn_R71`L4QVIVI8G5aLB5SHf<QADE58HF5
Q@Goool100000Oooo`7oool000?oool00`000?ooo`000004oooo0`0000?ooolQbLW98Kjn_R71`L4Q
VIVI8G5aLB5SHf<QADE58HF5Q@?oool300000_ooo`001?ooo`030000oooooooo00?oool00`000?oo
ooooo`3ooooo4Oooo`400001oooo0Oooo`000oooo`030000oooo000000Coool00`000?ooooooo`3o
oooo4Oooo`400001oooo0Oooo`001?ooo`030000oooooooo00?oool00`000?ooooooo`3ooooo4Ooo
o`400001oooo0Oooo`002_ooo`030000oooooooo00?oool00`000?ooooooo`0=oooo00<0003ooooo
ool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000
oooooooo00koool00`000?ooooooo`0=oooo00<0003oooooool03_ooo`030000oooooooo00goool0
0`000?ooooooo`0>oooo00<0003oooooool03Oooo`030000oooooooo00koool00`000?ooooooo`0=
oooo00<0003oooooool03_ooo`030000oooooooo00goool00`000?ooooooo`0>oooo00<0003ooooo
ool00_ooo`400001oooo0Oooo`002_ooool0000E00000_ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.489418, -0.536766, 0.0303025, 
  0.0303025}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[CellGroupData[{

Cell["corRelation[Flatten[a],Flatten[b]]", "Input",
  FormatType->InputForm],

Cell[BoxData[
    \(0.454120293858607659`\)], "Output"]
}, Open  ]],

Cell[BoxData[
    \(\[FilledSquare]\ \ Photo\ Sample\)], "Input"],

Cell[BoxData[
    \(\(a = Table[fukuzawa[\([i, j]\)], {i, 30, 157}, {j, 5, 132}]; \)\)], 
  "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[a, Mesh -> False]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.00751202 0.0192308 0.00751202 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.16947 -0.0125 -6 -9 ]
[.16947 -0.0125 6 0 ]
[.31971 -0.0125 -6 -9 ]
[.31971 -0.0125 6 0 ]
[.46995 -0.0125 -6 -9 ]
[.46995 -0.0125 6 0 ]
[.62019 -0.0125 -6 -9 ]
[.62019 -0.0125 6 0 ]
[.77043 -0.0125 -9 -9 ]
[.77043 -0.0125 9 0 ]
[.92067 -0.0125 -9 -9 ]
[.92067 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .16947 -12 -4.5 ]
[-0.0125 .16947 0 4.5 ]
[-0.0125 .31971 -12 -4.5 ]
[-0.0125 .31971 0 4.5 ]
[-0.0125 .46995 -12 -4.5 ]
[-0.0125 .46995 0 4.5 ]
[-0.0125 .62019 -12 -4.5 ]
[-0.0125 .62019 0 4.5 ]
[-0.0125 .77043 -18 -4.5 ]
[-0.0125 .77043 0 4.5 ]
[-0.0125 .92067 -18 -4.5 ]
[-0.0125 .92067 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.16947 0 m
.16947 .00625 L
s
[(20)] .16947 -0.0125 0 1 Mshowa
.31971 0 m
.31971 .00625 L
s
[(40)] .31971 -0.0125 0 1 Mshowa
.46995 0 m
.46995 .00625 L
s
[(60)] .46995 -0.0125 0 1 Mshowa
.62019 0 m
.62019 .00625 L
s
[(80)] .62019 -0.0125 0 1 Mshowa
.77043 0 m
.77043 .00625 L
s
[(100)] .77043 -0.0125 0 1 Mshowa
.92067 0 m
.92067 .00625 L
s
[(120)] .92067 -0.0125 0 1 Mshowa
.125 Mabswid
.05679 0 m
.05679 .00375 L
s
.09435 0 m
.09435 .00375 L
s
.13191 0 m
.13191 .00375 L
s
.20703 0 m
.20703 .00375 L
s
.24459 0 m
.24459 .00375 L
s
.28215 0 m
.28215 .00375 L
s
.35727 0 m
.35727 .00375 L
s
.39483 0 m
.39483 .00375 L
s
.43239 0 m
.43239 .00375 L
s
.50751 0 m
.50751 .00375 L
s
.54507 0 m
.54507 .00375 L
s
.58263 0 m
.58263 .00375 L
s
.65775 0 m
.65775 .00375 L
s
.69531 0 m
.69531 .00375 L
s
.73287 0 m
.73287 .00375 L
s
.80799 0 m
.80799 .00375 L
s
.84555 0 m
.84555 .00375 L
s
.88311 0 m
.88311 .00375 L
s
.95823 0 m
.95823 .00375 L
s
.99579 0 m
.99579 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .16947 m
.00625 .16947 L
s
[(20)] -0.0125 .16947 1 0 Mshowa
0 .31971 m
.00625 .31971 L
s
[(40)] -0.0125 .31971 1 0 Mshowa
0 .46995 m
.00625 .46995 L
s
[(60)] -0.0125 .46995 1 0 Mshowa
0 .62019 m
.00625 .62019 L
s
[(80)] -0.0125 .62019 1 0 Mshowa
0 .77043 m
.00625 .77043 L
s
[(100)] -0.0125 .77043 1 0 Mshowa
0 .92067 m
.00625 .92067 L
s
[(120)] -0.0125 .92067 1 0 Mshowa
.125 Mabswid
0 .05679 m
.00375 .05679 L
s
0 .09435 m
.00375 .09435 L
s
0 .13191 m
.00375 .13191 L
s
0 .20703 m
.00375 .20703 L
s
0 .24459 m
.00375 .24459 L
s
0 .28215 m
.00375 .28215 L
s
0 .35727 m
.00375 .35727 L
s
0 .39483 m
.00375 .39483 L
s
0 .43239 m
.00375 .43239 L
s
0 .50751 m
.00375 .50751 L
s
0 .54507 m
.00375 .54507 L
s
0 .58263 m
.00375 .58263 L
s
0 .65775 m
.00375 .65775 L
s
0 .69531 m
.00375 .69531 L
s
0 .73287 m
.00375 .73287 L
s
0 .80799 m
.00375 .80799 L
s
0 .84555 m
.00375 .84555 L
s
0 .88311 m
.00375 .88311 L
s
0 .95823 m
.00375 .95823 L
s
0 .99579 m
.00375 .99579 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.16947 .99375 m
.16947 1 L
s
.31971 .99375 m
.31971 1 L
s
.46995 .99375 m
.46995 1 L
s
.62019 .99375 m
.62019 1 L
s
.77043 .99375 m
.77043 1 L
s
.92067 .99375 m
.92067 1 L
s
.125 Mabswid
.05679 .99625 m
.05679 1 L
s
.09435 .99625 m
.09435 1 L
s
.13191 .99625 m
.13191 1 L
s
.20703 .99625 m
.20703 1 L
s
.24459 .99625 m
.24459 1 L
s
.28215 .99625 m
.28215 1 L
s
.35727 .99625 m
.35727 1 L
s
.39483 .99625 m
.39483 1 L
s
.43239 .99625 m
.43239 1 L
s
.50751 .99625 m
.50751 1 L
s
.54507 .99625 m
.54507 1 L
s
.58263 .99625 m
.58263 1 L
s
.65775 .99625 m
.65775 1 L
s
.69531 .99625 m
.69531 1 L
s
.73287 .99625 m
.73287 1 L
s
.80799 .99625 m
.80799 1 L
s
.84555 .99625 m
.84555 1 L
s
.88311 .99625 m
.88311 1 L
s
.95823 .99625 m
.95823 1 L
s
.99579 .99625 m
.99579 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .16947 m
1 .16947 L
s
.99375 .31971 m
1 .31971 L
s
.99375 .46995 m
1 .46995 L
s
.99375 .62019 m
1 .62019 L
s
.99375 .77043 m
1 .77043 L
s
.99375 .92067 m
1 .92067 L
s
.125 Mabswid
.99625 .05679 m
1 .05679 L
s
.99625 .09435 m
1 .09435 L
s
.99625 .13191 m
1 .13191 L
s
.99625 .20703 m
1 .20703 L
s
.99625 .24459 m
1 .24459 L
s
.99625 .28215 m
1 .28215 L
s
.99625 .35727 m
1 .35727 L
s
.99625 .39483 m
1 .39483 L
s
.99625 .43239 m
1 .43239 L
s
.99625 .50751 m
1 .50751 L
s
.99625 .54507 m
1 .54507 L
s
.99625 .58263 m
1 .58263 L
s
.99625 .65775 m
1 .65775 L
s
.99625 .69531 m
1 .69531 L
s
.99625 .73287 m
1 .73287 L
s
.99625 .80799 m
1 .80799 L
s
.99625 .84555 m
1 .84555 L
s
.99625 .88311 m
1 .88311 L
s
.99625 .95823 m
1 .95823 L
s
.99625 .99579 m
1 .99579 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
128 string
128 128 8 [128 0 0 128 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
000000000000000000FFFFFFFF0000000000000000000000
0000000000000000FF00000000000000000000000000FF00
0000000000000000000000000000000000000000000000FF
FF000000000000000000000000000000FF00000000000000
000000000000000000000000000000000000000000000000
0000FFFF00FFFFFF
000000000000000000000000000000000000000000000000
000000000000000000FF000000FFFFFFFF0000FFFF00FFFF
FFFFFFFF000000FF00FFFFFF000000FF0000FFFFFF00FFFF
0000000000FFFFFF00000000FF0000FF000000FFFF0000FF
000000000000000000000000000000000000000000000000
0000000000000000
000000000000000000000000000000000000000000000000
0000FF000000000000FF0000FFFFFFFFFFFFFFFFFFFFFFFF
FFFF00000000FF00FF00FFFFFFFF0000000000000000FF00
00FF00000000FFFFFFFF00FFFFFF00FFFF00000000FFFF00
000000000000000000000000000000000000000000000000
0000000000000000
000000000000000000000000000000000000000000000000
00000000FF000000FFFF000000000000FF000000FFFFFFFF
FFFFFFFFFF00000000FFFF00000000FF0000FF000000FFFF
FF00FFFF00FFFF00000000000000FF0000000000FF000000
00000000000000000000FF00FF00FFFFFF00FFFFFFFFFF00
0000000000000000
000000000000000000FFFFFF0000FFFF0000000000000000
00000000000000000000FF00000000000000FF000000FFFF
00FFFFFFFFFF00FF00FFFF00FF0000FFFFFFFF0000000000
FFFF00FF000000FFFFFFFF00000000000000000000000000
00000000000000000000FF00FFFFFFFFFFFFFFFFFFFFFFFF
00FFFFFFFF00FFFF
FFFF00000000000000000000000000000000000000000000
0000000000FFFFFFFFFFFF0000000000000000FF0000FFFF
0000000000FFFFFFFFFFFFFFFFFF0000FF0000FF00000000
000000FFFF0000FF00FF00000000000000000000000000FF
00000000000000000000FF00FFFFFFFFFFFFFFFF0000FFFF
00FFFFFFFF00FFFF
000000000000000000000000000000000000000000000000
000000000000FF0000FFFFFF000000000000000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF00FFFFFFFFFFFF0000000000000000000000000000FF
FF0000000000000000000000FFFFFF000000000000FFFFFF
0000FF00000000FF
FFFFFFFFFFFFFFFFFF0000000000000000000000000000FF
00FF0000FFFFFFFF00FFFFFFFF0000FFFFFFFFFFFF00FF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF00FFFF00FF00FFFFFFFF00FF
FFFFFF000000000000000000FFFFFFFFFF00FF0000FFFFFF
00FFFF000000FFFF
0000000000000000FFFFFFFF000000000000000000000000
0000FF00FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF00FF
00FFFFFFFF00000000000000FFFFFFFFFF00FF0000FFFFFF
00FFFFFFFF00FFFF
0000000000000000FFFFFFFF00000000000000000000FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFF00FFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFF00FF00FFFFFFFFFFFFFFFFFFFFFFFF00FF
FFFFFFFFFFFF00000000000000FFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
0000000000000000FFFFFFFF00000000FFFFFFFFFF00FFFF
FFFFFF00FFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF00
FFFFFFFF00FFFF00FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF00FF0000FFFFFFFFFFFFFF00FFFF00
FFFFFFFF00FFFF00000000000000FFFF0000FF0000FFFFFF
0000FFFFFF0000FF
0000000000000000FFFF0000000000FFFFFF000000FFFFFF
FF00FF00FF00FFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFF00FFFF00FF00FFFFFF00FFFFFFFFFF00FF00FFFFFF
FFFFFFFFFFFFFFFFFF00FF0000FFFFFFFFFFFF00FFFFFF00
0000FFFFFFFFFFFFFFFFFF00000000000000000000FFFFFF
0000FF00FF0000FF
0000000000000000FF0000000000FFFFFFFFFF00FFFFFFFF
000000FFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFF00FFFFFFFF00FFFFFFFF00FFFFFFFFFFFF00FFFFFF
FFFFFFFFFFFFFF00FFFF000000FFFFFFFFFF00000000FFFF
0000FFFF00FF00FFFFFFFFFFFF0000FF0000FFFF0000FFFF
0000FFFFFF00FFFF
000000000000000000000000FFFFFFFFFFFF00FF00FFFFFF
FFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFF00FFFFFFFF00FFFFFFFF00FFFF00FFFF
FFFFFFFFFF0000FFFF00000000FFFFFFFF00FFFFFF00FFFF
FF000000FFFFFFFFFFFFFFFFFFFFFF00FFFF000000FFFF00
0000FFFFFF0000FF
00000000000000000000FFFFFFFFFFFFFFFFFFFF00FF0000
FFFFFF00FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFF00FFFFFFFF00FFFFFFFFFF0000FFFF
FFFFFFFFFFFFFFFFFF00000000FFFFFF000000FFFF00FFFF
FF000000FFFFFF0000FFFFFFFFFFFFFFFFFFFFFF0000FFFF
0000FFFF00FFFFFF
00000000FF00000000000000FFFFFFFFFFFFFFFFFF000000
FFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFF00FFFF
FFFFFFFFFFFF0000FFFF00000000FFFFFFFFFFFFFF00FFFF
FF0000FFFFFFFF00FFFFFF00000000FFFFFFFFFFFF000000
0000FFFFFFFFFFFF
00000000FF00FF0000FF00FFFF000000FFFFFF00000000FF
00FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFF00FFFFFFFF
FFFF0000FFFF00FFFF00FFFF000000FFFFFFFFFFFF000000
0000FFFF00FFFFFF
000000000000FF00FFFFFFFFFF00FF00FFFF0000000000FF
0000FFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFF0000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF0000000000FFFFFFFFFF00FF00FFFF
FFFF00FF00FFFFFFFFFFFFFFFF00FFFFFF0000FFFFFF00FF
000000000000FFFF
0000000000FF00FFFFFFFFFFFF00FF00FF0000FF00FFFFFF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF00FFFFFF0000FFFF
FFFFFFFFFFFFFFFF00FF0000FFFFFF00FF00FF0000FFFFFF
FFFF0000FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFF00
0000000000FFFF00
000000FFFFFFFFFF00FF00FF000000FF0000FFFF00FFFFFF
FFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFF00FF
FFFFFFFFFFFFFFFF00FFFFFF00FFFF00FFFFFFFF00FFFFFF
FF0000FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFF
00FFFF0000000000
000000FF00FFFFFF0000FFFF00FFFF000000FFFFFFFFFFFF
FF00FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF0000FFFFFFFF00FF
FFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFF00FFFFFF
FF0000FF00FFFF00FFFFFFFFFFFFFFFF0000FFFFFFFFFFFF
00000000000000FF
000000FFFF00FFFFFF00FFFFFF000000FFFFFFFFFFFFFFFF
00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF
FFFFFFFFFFFFFFFFFFFFFF00FF00FFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFF0000FFFFFFFFFFFFFFFF00FFFFFF
FF00FFFFFFFF00FF0000FF00FFFFFFFF0000FFFFFFFFFF00
FF00000000000000
00000000FFFFFFFF000000FF000000FFFFFFFFFF00FFFFFF
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFF00FF
FFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFF00
FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFF0000FFFF00FFFFFF
FFFFFF00FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
0000000000000000
00000000FFFFFF00FF00FF0000FFFFFFFFFFFF00FFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF
FFFFFF00FFFFFFFF00FFFFFFFFFFFFFFFFFF0000FFFFFFFF
FFFFFFFFFFFFFF00FFFFFF00FFFF00FFFFFF00FF00FFFFFF
FFFF0000FF00FF0000FFFFFFFFFFFFFFFF00FFFFFFFFFF00
00000000000000FF
00000000FFFF0000FFFFFF00FFFFFFFFFF00FFFF00FFFFFF
FFFFFFFFFFFFFF00FF00FFFFFFFFFFFFFFFFFF00FF0000FF
FFFFFFFF00FF00FF00FFFFFFFFFFFFFFFFFF0000FFFFFFFF
00FFFFFFFFFFFF00000000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF0000FF00FFFF00FFFFFFFFFFFFFF0000FF00FFFF0000
00000000FF0000FF
000000000000000000FFFFFFFFFFFF000000FFFFFF00FFFF
0000FFFFFFFFFF00FFFFFFFFFFFFFFFFFF00FF00FF0000FF
FFFFFFFFFF00FF00FFFFFFFFFF00FFFFFFFFFFFF00FFFFFF
00FF00FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFF0000FF00FF00FFFFFF0000FF0000FF00FF0000
000000000000FFFF
0000000000FF000000FFFFFFFFFFFF0000FFFFFF00FF0000
00FFFFFFFFFFFFFFFF00FFFFFFFFFFFFFF000000000000FF
FFFF0000FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFF00FFFFFF
00FFFFFFFFFF00FFFFFFFF00FFFF00FFFFFFFF00FFFFFFFF
FF00FFFFFF00000000FFFFFFFFFFFFFFFFFFFFFFFF000000
000000000000FFFF
000000FF00FFFF00FFFFFFFFFFFFFFFF00FFFFFF00FFFFFF
FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFF00FF00000000FF
FFFFFF00FFFF0000FF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FFFFFFFF0000000000FF0000FFFF00000000FFFF000000
0000000000000000
00FFFFFFFFFF00FFFFFFFFFFFFFFFF000000FF00FF00FFFF
00FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF00FF000000FF
FFFF0000FFFF00FFFF0000FFFFFFFFFFFFFFFFFFFF00FFFF
FFFFFFFFFFFFFF00FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF
00FFFFFF000000000000FF00FFFFFFFF0000FFFFFF000000
000000FF0000FF00
FF00FFFFFFFFFFFFFFFF00FF0000FFFF00FFFFFF00FFFFFF
00FFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF00000000
FFFFFFFF00FFFFFF00FFFFFFFFFFFFFF00FFFFFFFFFF00FF
FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FFFFFFFF000000FFFFFFFFFFFFFF000000FFFF00000000
000000000000FF00
FFFFFFFFFFFFFFFFFFFF00FFFFFF0000000000FFFF0000FF
00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000
FFFF00FFFFFFFFFFFFFFFFFF00FFFF0000FFFFFFFF0000FF
FF0000FFFFFF00FFFFFFFFFF00FFFFFFFFFF0000FFFFFF00
FFFFFFFF00FF00FFFF000000FFFFFF0000FFFFFF00000000
0000000000000000
FFFF00FFFFFFFF00FFFFFF00FFFFFF0000FFFFFFFF00FF00
00FFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFF0000FFFF00
FFFF00FFFFFF00FF0000FFFFFFFFFF00FFFFFFFFFFFF0000
FF0000FFFFFF00FFFFFFFFFFFFFFFFFFFFFF0000FFFFFF00
FFFF00FF00FF00FFFFFF000000FFFF000000FF0000000000
0000000000000000
FFFFFFFF00FFFF00FFFF0000FFFF00FFFF00FFFF00000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF00
FF00FFFFFF0000FFFFFFFFFFFFFFFFFF0000FFFFFF0000FF
00FF00FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFF00FFFF00FF
00FFFF00FFFF00FFFFFF000000FFFF000000FFFF00000000
0000FF0000000000
FFFFFFFFFFFFFFFF00FF0000FFFF00000000FFFF00FF0000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF00
FFFFFF00FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF
FFFFFFFF00FF0000FF00FF0000FF00FF00FF000000000000
00FFFFFFFF000000
FFFFFFFFFFFFFFFFFFFF0000FFFF0000FF00FFFF00000000
FFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFF00FFFF
FF00FFFF0000FFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFF
FFFFFF00FFFFFFFFFFFFFFFF00FFFFFFFF0000FFFF00FFFF
FFFFFF00FF000000FF00FF00FFFFFF000000000000000000
0000FFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFF00FFFF0000FF00FF0000000000
FFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFF00FF00
FFFF0000FF00FFFFFFFFFFFFFFFF00FFFFFF0000FFFF00FF
FFFF0000FF00FFFFFFFF00FF00FFFFFFFFFF00FFFF00FFFF
FFFF0000FFFFFF00FFFF0000FF000000FF00FF000000FFFF
00FFFFFF00000000
00000000000000FFFFFFFF00FF000000FF000000FFFF0000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFF
FF000000FFFFFF00FF00FFFF00FFFFFFFFFFFFFFFFFFFFFF
FFFFFF00FFFFFFFFFFFFFFFF00FFFFFFFFFF0000FF00FFFF
FFFFFF00FFFF0000FF0000000000000000000000FFFF0000
FFFFFFFFFFFFFFFF
00000000000000000000FF00FFFF00FFFF0000FFFFFF0000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF0000FF
FFFF00FF000000FF00FFFFFFFFFFFFFFFFFF000000FFFFFF
FFFFFF00FFFF00FFFFFFFFFFFFFFFFFFFFFF0000FF0000FF
FF0000FFFFFF0000FFFFFF000000FF000000FF0000FF0000
0000FFFFFFFFFF00
00000000000000000000FFFFFFFFFF00FF000000FFFF0000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF0000FF00
FF00FFFFFF00FFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF00FFFFFFFFFFFF00FFFFFFFFFF00FF00FF00FFFF
FF00FFFFFF000000FFFFFF00000000FF000000FFFFFFFFFF
00FFFF00000000FF
FFFFFFFFFF00FFFFFFFFFFFF00FFFF00FF000000FFFF0000
00FFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFF00FF0000FFFF
FF00FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFF00FF00FF
FFFFFFFF0000FF00FF00FFFF00FFFFFFFF00FFFFFFFFFFFF
0000FFFFFFFFFFFFFFFFFF00FF00FFFF0000FF00FF0000FF
FFFFFFFF0000FF00
00000000000000000000FFFF00000000FF00000000FF0000
00FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFF00000000FFFF
0000FFFF00FF00FFFFFF0000FFFFFFFF00FFFFFFFFFFFFFF
FFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FFFFFFFFFF0000FF0000FFFF0000FF0000FF000000FFFF
00FFFF00FF00FFFF
00000000000000000000FFFF0000FF00FFFF000000FF0000
00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF00
0000000000000000FFFFFF00FF00FFFFFFFFFF00FFFFFFFF
FFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFF000000FFFFFF00
00FFFFFF0000FF00FF00000000FFFF00FF0000FFFFFFFF00
FF00FFFF00FFFFFF
00000000000000000000FFFF000000000000000000FFFF00
00FF00FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFF0000FFFF
FFFFFFFF00FFFFFFFFFFFFFF00FFFFFFFFFFFF0000FF0000
FFFF00FFFF00FFFFFFFFFFFF00FFFFFFFF00FF00FFFF0000
FFFFFFFFFFFF0000FF00FFFFFF00FFFFFF0000000000FFFF
FF00FFFFFFFFFFFF
00000000000000000000FFFF00000000FF00FF0000FFFF00
00FFFF00FFFFFF00FFFF00FFFFFFFFFFFFFFFFFF0000FFFF
FFFFFF0000FFFFFFFFFFFFFF00FFFFFFFFFFFFFF0000FF00
0000FF00FFFFFFFFFF00FFFF00FFFFFFFFFFFF00FFFF00FF
FFFFFFFFFFFF000000FFFFFFFFFF0000FFFFFFFFFFFFFFFF
00FFFFFFFFFFFFFF
00000000000000000000FFFFFF000000000000FF0000FF00
0000FFFFFF00FF00FFFFFFFFFFFFFFFFFFFFFF00FFFFFFFF
FFFF00FFFFFFFFFFFFFFFFFFFFFFFF00FFFFFF00FFFF0000
FF00FFFF00FFFFFF00FFFF00FFFFFFFFFFFF0000FF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFF
FF00FFFFFFFFFFFF
00000000000000000000FFFF00000000000000FFFFFFFF00
0000FFFFFF000000FFFFFF00FFFF00FFFF00FFFF00FFFF00
FFFFFFFFFFFFFFFFFFFF000000FF00FF0000FFFF000000FF
FFFFFF0000FFFFFF00FFFF00FFFFFFFFFFFFFFFF00FF0000
00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FF00FFFFFFFFFFFF
00000000000000000000FFFF000000000000FFFFFFFFFFFF
0000FFFFFFFFFFFFFF00FFFFFF00FFFFFFFFFFFF00FFFF00
FFFFFFFFFFFFFFFFFF000000FF0000FFFF00FFFFFFFFFF00
00000000FF0000FF00FFFF00FFFFFFFFFFFFFF00FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFF
FF00FF00FFFFFFFF
00000000000000000000FFFF000000000000FFFFFFFFFFFF
FFFFFFFFFFFF00FFFFFFFFFFFF0000FFFFFFFFFF00FFFF00
FFFFFFFFFFFFFFFF0000FFFFFF000000FFFFFF000000FF00
FFFF0000000000FF00FFFF00FFFFFFFFFF000000FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFF
FFFFFFFFFF00FFFF
FFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF0000FFFF00FFFF00FF00FFFF00FFFFFF
FF0000FFFFFF00000000FF000000000000000000FFFFFF00
FF000000000000FFFFFF00FFFFFF0000FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FF00FFFFFF00FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFF00FF0000FF00FFFFFFFFFF00
FF00FFFF0000000000FFFF0000000000000000000000FF00
000000FF000000FFFFFF000000FFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF000000000000
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF00FFFFFFFF00
FFFFFFFF00FF0000FF00000000000000000000000000FF00
000000FF000000FFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF0000FFFFFF
00FFFFFF00FF00FF00000000000000000000000000000000
FF0000FF000000FFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000
FFFFFFFFFF00FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFF
FF0000000000FF0000000000000000000000000000000000
FF00FF00000000FFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF00FF00FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFF00
FF0000000000000000000000000000000000000000000000
00000000000000FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF00FFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
00FFFF000000000000000000000000000000000000000000
000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FF00FF000000000000000000000000000000000000000000
0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FF00FF0000000000000000000000000000000000000000
000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FF00000000000000000000000000000000000000000000
0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
00FFFF000000000000000000000000000000000000000000
0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFF00000000000000000000000000000000000000000000
00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FF0000000000000000000000000000000000000000000000
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF000000000000000000000000000000000000000000
00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFF000000000000000000000000000000000000000000
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FF00000000000000FFFFFF00000000000000000000000000
000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FF00000000000000FFFFFFFF000000000000000000000000
00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
0000000000000000000000FFFF00FF000000000000000000
00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00000000000000000000000000FFFFFF000000000000FF00
0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000
0000000000FF0000FFFF0000000000000000000000000000
0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000
00000000FF000000FF000000000000000000000000FF0000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000
0000000000FFFF00000000000000FFFFFF00000000000000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000
000000FF0000FF00FF000000FFFF00000000000000000000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
000000FF0000FFFF00FFFF0000000000000000000000FF00
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
0000000000FF00FFFFFFFFFFFFFFFFFF00FFFF0000000000
0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF0000
00000000FFFFFF00FFFFFFFF0000FF0000000000FF000000
0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
000000FF00FFFFFFFFFFFFFF000000000000FFFF00FF0000
0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000
0000FF00000000FF00FF0000000000000000000000FF0000
0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000
0000FFFFFF0000FFFF00FF00000000000000000000FF0000
0000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000
FFFFFFFFFFFF00FFFF00FFFF00FF000000FF00FFFFFF0000
00FF000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
00FFFFFFFFFFFFFF00FFFF00FFFFFF000000FFFFFFFFFFFF
FFFFFF0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
00FFFFFF0000000000FFFF00FF00FF000000FF0000000000
00FFFF00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FF
FFFFFFFF00FF00000000FFFFFFFFFF0000FF000000000000
00FFFF00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000
00000000000000000000FFFFFFFFFFFF000000FF00000000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000
FF0000000000000000000000FFFFFFFF0000000000000000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000000000
000000FF00FF00000000000000FFFF000000000000000000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFFFF
0000000000FFFFFF0000000000FFFF00000000FF000000FF
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFF
00FF00FF0000FFFFFF000000000000000000000000000000
00000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFF
FF0000000000000000000000FFFFFFFF0000000000000000
0000000000FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFF00
FF00FFFFFF00FF0000000000FF00FFFF000000000000FFFF
FFFF00FFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FFFF
FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF0000FF00FFFF
FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFF00FF00FFFF
00FFFF00FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000FF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0000
FF00FF0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF0000FFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FF00FF
00FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00
0000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FF0000FFFFFFFFFFFFFF00FFFFFFFF000000FF00FF00FF00
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFF00FF00FFFFFFFFFFFF00000000FFFFFF0000FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFF00FFFFFFFF00FFFFFFFFFF00FFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFF00FFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FFFF00FFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
00FF0000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFF00
00FF00FFFFFFFFFF00FFFFFFFFFF00FFFFFFFFFFFF00FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
000000FFFFFFFF0000FFFFFFFFFFFFFFFFFF00FF00FFFFFF
FFFFFF00FF00FF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
000000FF00FFFFFFFF00FF000000FFFFFF00FF00FFFFFFFF
FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
FF0000FF000000000000000000FFFF00FFFFFFFF00FFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFF
FFFFFFFFFF0000FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFF
FFFFFFFFFFFF00FFFFFFFFFFFF00FFFFFF00FFFFFFFFFFFF
FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFFFFFF
FFFFFFFFFFFFFFFF
00000000FF000000FF00FF000000FFFFFF00FFFFFFFFFFFF
FFFF00FFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFF00FFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
000000000000FFFF00FF000000FF0000FFFFFF00FF0000FF
FFFFFF0000FFFFFFFFFFFFFFFFFFFFFF00FFFF00FFFFFF00
FFFF00FFFFFFFFFFFFFFFF00FF00FFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
00000000000000FF000000000000FF0000000000FFFF00FF
00FFFF00FF00FFFF00FFFFFF00FF00FFFFFFFFFFFFFF00FF
FFFFFF00FFFFFFFFFFFFFFFF00FFFFFF00FFFFFFFFFF00FF
FF00FF00FFFFFF00FFFFFFFF00FFFF00FFFFFFFFFFFF00FF
FFFFFFFFFFFFFF00FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
FFFFFFFFFFFFFFFF
0000000000000000FF0000000000000000000000FF00FF00
000000FF00FFFFFF0000FF00FFFFFF000000FFFFFFFF00FF
FFFF00FF00FFFFFFFF00FFFFFF00FF00FFFFFFFFFFFF00FF
00FFFFFF00FFFFFFFFFFFF00FFFFFFFFFFFFFF00FFFFFFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00FFFFFF0000FFFF
FFFF00FFFFFFFFFF
00000000000000000000000000000000000000000000FF00
00000000FF0000000000FFFFFFFF00FF0000FFFFFFFFFFFF
FFFF00FF0000FFFF00FFFF0000FFFF00FFFFFFFFFF0000FF
FFFF00FFFFFFFFFFFF00FFFFFFFF00FFFFFFFFFFFFFFFF00
FFFFFFFFFFFF0000FFFFFFFFFFFFFFFFFFFFFFFFFF0000FF
00FF00FFFFFFFFFF
00000000000000000000000000000000000000FF00000000
00000000FF00FF00000000FF00000000FF0000FFFF000000
FFFFFFFF00FF00FF00FFFF0000FFFF00FFFFFF00FFFFFFFF
FF00FFFF00FFFF00FFFFFF00FFFFFFFFFF00FFFFFF00FFFF
FFFFFFFFFFFFFFFFFF00FFFF00FFFFFFFFFFFFFF00FFFFFF
FF00FFFFFF00FFFF
00FF000000000000000000000000000000FF000000000000
000000000000FFFF00FF0000FFFF00FF00000000FF00FFFF
FF00000000FF0000FF00000000FFFFFFFF00FFFFFF00FF00
FF00FFFF0000FF0000FFFF00FF00FF000000FFFFFF00FFFF
00FF00FF0000FFFFFF00FF00FFFFFF0000FFFFFF00FFFF00
00FFFF00FF0000FF
000000000000000000000000000000000000000000000000
00000000000000000000FF0000FF000000000000FF0000FF
000000FFFF00000000FFFF000000FF00FF00FFFF00000000
FF0000FF0000FFFF00FF00FF00FFFFFFFFFFFFFF00FF0000
FFFFFFFF00FFFF00000000FF00FFFFFFFFFF00FFFFFFFF00
FF00FF00FFFF0000
0000000000000000FF000000000000000000000000000000
000000FFFF000000000000000000000000000000000000FF
000000FF00000000000000FFFFFF00000000FF000000FF00
FFFF0000FF000000FF00FF0000FF00FF000000FF0000FFFF
0000000000000000FF00FFFF00FF0000FFFFFF000000FF00
00000000FF000000
000000000000000000000000000000000000000000000000
000000000000000000000000000000000000FF0000FF0000
000000000000000000000000000000FFFF0000000000FFFF
FF00FF0000FF00000000FF00FFFF00FFFF0000FFFF00FF00
0000000000FFFF00FF000000000000FF0000FF0000000000
FF0000FF00000000
000000000000000000000000000000000000000000000000
000000000000000000000000000000FF0000000000000000
0000FF00000000000000FFFF000000000000FF000000FF00
0000FF0000FF000000000000000000000000FFFF0000FFFF
00000000FF00FF00000000000000FF000000000000000000
0000000000000000
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo003ooooo8Oooo`006Oooo`030000
oooooooo02;oool300000oooo`030000oooooooo023oool300000_ooo`030000oooooooo01ooool2
00001?ooo`030000oooooooo023oool00`000?ooooooo`02oooo00<0003oooooool06oooo`D00002
oooo00<0003oooooool00_ooo`030000oooooooo01Woool5000000Coool0000000000003oooo00<0
003oooooool03oooo`006?ooo`030000oooo000002?oool00`000?ooooooo`02oooo00<0003oool0
00008_ooo`060000oooooooo0000oooo00008?ooo`030000oooo000000;oool00`000?ooo`00000P
oooo00<0003oool000000_ooo`030000oooo000001koool01`000?ooooooooooo`000?ooo`000002
oooo00<0003oool000007?ooo`050000oooooooooooo000000Coool00`000?ooo`00000@oooo000H
oooo00<0003oool000009?ooo`070000oooooooooooo0000oooo0000023oool4000000Coool0003o
ool0000Poooo00<0003oool000000_ooo`030000oooo0000023oool00`000?ooo`000002oooo00<0
003oool000007_ooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`00000Loooo00<0
003oooooool00_ooo`070000oooooooooooo0000oooo0000013oool001Soool00`000?ooo`00000T
oooo00L0003oooooooooool0003oool000008?ooo`030000oooo000000;oool00`000?ooo`00000P
oooo0P0000?oool00`000?ooo`00000Qoooo00L0003oooooooooool0003oool000007_ooo`070000
oooooooooooo0000oooo000000;oool00`000?ooo`00000Loooo00<0003oooooool00_ooo`070000
oooooooooooo0000oooo0000013oool001Soool00`000?ooo`00000Soooo00<0003oool000000_oo
o`030000oooo0000027oool200000_ooo`030000oooo0000023oool00`000?ooooooo`02oooo00<0
003oool000008?ooo`030000oooo000000;oool00`000?ooo`00000Noooo00L0003oooooooooool0
003oool000000_ooo`030000oooo000001coool01`000?ooooooooooo`000?ooo`000002oooo00<0
003oool000004?ooo`006Oooo`030000oooooooo02?oool00`000?ooooooo`02oooo00<0003ooooo
ool08Oooo`050000oooooooooooo000002;oool200000oooo`030000oooooooo023oool00`000?oo
ooooo`02oooo00<0003oooooool07?ooo`800004oooo00<0003oooooool00_ooo`030000oooooooo
01[oool200001?ooo`030000oooooooo00;oool00`000?ooooooo`0?oooo003ooooo8Oooo`00oooo
ob7oool00?oooolQoooo003ooooo8Oooo`00ooooob7oool001Gooooo00002`00007oool001Goool0
0`000?ooooooo`02oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Koool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool200000Oooo`005Oooo`030000oooooooo00;oool00`000?ooooooo`0Uoooo00<0
003oooooool09Oooo`030000oooooooo02Goool00`000?ooooooo`0Uoooo00<0003oooooool09?oo
o`030000oooooooo02Goool00`000?ooooooo`0Boooo0@00007oool000ooool00`000?ooooooo`03
oooo00<0003oooooool0ooooo`Ooool100000Oooo`003_ooo`030000oooo000000Coool00`000?oo
ooooo`3ooooo1oooo`400001oooo000>oooo00<0003oool000001?ooo`80003ooooo1oooo`800001
oooo000>oooo00<0003oool000001?ooo`030000oooooooo00;ooolB00002?ooobH00002oooo6P00
00;oool_00001?oooah00002oooo@P0000Coool200002oooo`400001oooo000>oooo00<0003oool0
00001?ooo`030000oooooooo00;ooolB00002?ooobH00002oooo6P0000;oool_00001?oooah00002
oooo@P0000Coool200002oooo`400001oooo000?oooo00<0003oooooool00oooo`030000oooooooo
00;ooom200000_ooo`H00008oooo100000Coool200003?ooo`H00002oooo0P0000Koool600000_oo
o`<00006oooo0P0000Coool:00001_ooo`P00002oooo100000;oool600001?ooo`@00002oooo@000
00Goool100000Oooo`005Oooo`030000oooooooo00;ooom200000_ooo`H00008oooo100000Coool2
00003?ooo`H00002oooo0P0000Koool600000_ooo`<00006oooo0P0000Coool:00001_ooo`P00002
oooo100000;oool600001?ooo`@00002oooo@00000Goool100000Oooo`005Oooo`030000oooooooo
00;ooold00000_ooo``00002oooo100001coool800000_ooo`800002oooo0P0000Soool?00000_oo
o`@00002oooo200000Soool200001_ooo`800004oooo200000Cooom200001Oooo`400001oooo000E
oooo00<0003oooooool00_oooc@00002oooo300000;oool400007?ooo`P00002oooo0P0000;oool2
00002?ooo`l00002oooo100000;oool800002?ooo`800006oooo0P0000Coool800001?oood800005
oooo0@00007oool001Goool00`000?ooooooo`02oooo>00000;oool600001?ooo``00002oooo1P00
01;oool800001?ooo`P00002oooo0`0000;oool600001_ooo`800004oooo0P0000Coool>00000_oo
o`X00002oooo6P0000;oool200000_ooo`800006oooo0P0000[ooolB00001Oooo`400001oooo000E
oooo00<0003oooooool00_ooocP00002oooo1P0000Coool<00000_ooo`H0000Boooo200000Coool8
00000_ooo`<00002oooo1P0000Koool200001?ooo`800004oooo3P0000;oool:00000_oooaX00002
oooo0P0000;oool200001_ooo`80000:oooo4P0000Goool100000Oooo`005Oooo`030000oooooooo
00;ooolB00001_ooo`@00004oooo900000;oool>00000_ooo`H00004oooo0P0000[oool200000_oo
o`800004oooo0P0000;oool400001oooo`X00004oooo0P0000;oool600002?ooobh00002oooo0P00
01Soool200002?ooo`800009oooo0@00007oool001Goool200000ooooa800006oooo100000CooolT
00000_ooo`h00002oooo1P0000Coool200002_ooo`800002oooo0P0000Coool200000_ooo`@00007
oooo2P0000Coool200000_ooo`H00008oooo;P0000;oool200006?ooo`800008oooo0P0000Soool2
00000Oooo`005Oooo`030000oooooooo00Kooolf00003?oooa000002oooo100000Coool:00004_oo
o`@000000oooo`000000000200000_ooo`h00004oooo100000;oool200000_oooaX00002oooo5000
00;oool200004?ooo`@00004oooo0P0000Soool200002Oooo`400001oooo000Eoooo00<0003ooooo
ool01_ooocH0000<oooo400000;oool400001?ooo`X0000Boooo10000003oooo0000000000800002
oooo3P0000Coool400000_ooo`800002oooo6P0000;ooolD00000_ooo`80000@oooo100000Coool2
00002?ooo`800009oooo0@00007oool001Goool00`000?ooooooo`02oooo?00000;oool400001_oo
oaP0000coooo0P0000cooolL00001?oooaH00006oooo300000Koool400000_ooo`P00007oooo0@00
007oool001Goool00`000?ooooooo`02oooo?00000;oool400001_oooaP0000coooo0P0000cooolL
00001?oooaH00006oooo300000Koool400000_ooo`P00007oooo0@00007oool001Goool00`000?oo
ooooo`0Doooo700000;oool200000_ooo`@00008oooo0P0000Soool400003?ooo`800002oooo0P00
04Ooool200001?ooo`800002oooo0P0000Soool200002?oooa80000:oooo0P0000;oool400001_oo
o`800004oooo1P0000Woool100000Oooo`005Oooo`030000oooooooo01CooolL00000_ooo`800002
oooo100000Soool200002?ooo`@0000<oooo0P0000;oool20000Aoooo`800004oooo0P0000;oool2
00002?ooo`800008oooo4P0000[oool200000_ooo`@00006oooo0P0000Coool600002Oooo`400001
oooo000Eoooo00<0003oooooool00_oooa000008oooo700000;oool200002_ooo`80000Joooo0P00
01Koool20000;Oooo`80000Doooo0P0000;oool200002?ooo`h0000:oooo0P0000;oool400001_oo
o`800008oooo0P0000Woool100000Oooo`005Oooo`030000oooooooo00;oool@00002?oooa`00002
oooo0P0000[oool200006_ooo`80000Foooo0P0002goool200005?ooo`800002oooo0P0000Soool>
00002_ooo`800002oooo100000Koool200002?ooo`800009oooo0@00007oool001Goool00`000?oo
ooooo`02oooo400000SooolD0000<_ooo`80000<oooo0P0000Soool400008oooo`800002oooo0P00
01Soool200003_ooo`h0000[oooo0@00007oool001Goool200000ooooa000008oooo500003;oool2
00003?ooo`800008oooo100002?oool200000_ooo`80000Hoooo0P0000koool>0000:_ooo`800001
oooo000Eoooo00<0003oooooool00_oooa000008oooo200000[oool200002_ooo`80000@oooo0P00
01Coool200002?ooo`800004oooo0P0000[oool200009Oooo`800002oooo100000koool200001?oo
o`800008oooo0P0000Coool>00001?ooo`@00002oooo100000Koool400001_ooo`@00007oooo0@00
007oool001Goool00`000?ooooooo`02oooo400000Soool800002_ooo`80000:oooo0P00013oool2
00005?ooo`800008oooo0P0000Coool200002_ooo`80000Uoooo0P0000;oool400003_ooo`800004
oooo0P0000Soool200001?ooo`h00004oooo100000;oool400001_ooo`@00006oooo100000Ooool1
00000Oooo`005Oooo`030000oooooooo00;oool@00001?ooo`X00006oooo1P0000Soool200000_oo
o`800002oooo0P0000Koool400006?ooo`800006oooo0P0000Coool200000_ooo`800006oooo0P00
00Woool200000_ooo`80000Hoooo0P0000;oool400003?ooo`800006oooo1P0001;ooolD00001_oo
o`@00002oooo0P0000;oool400001oooo`400001oooo000Eoooo00<0003oooooool00_oooa000004
oooo2P0000Koool600002?ooo`800002oooo0P0000;oool200001_ooo`@0000Hoooo0P0000Koool2
00001?ooo`800002oooo0P0000Koool200002Oooo`800002oooo0P0001Soool200000_ooo`@0000<
oooo0P0000Koool600004_oooa@00006oooo100000;oool200000_ooo`@00007oooo0@00007oool0
01Goool00`000?ooooooo`02oooo400000;oool:00002_ooo`800008oooo1P0000Coool200008_oo
o`800006oooo0P0000Soool200002?ooo`80000;oooo0P0001Coool200001?ooo`H0000:oooo2000
00Coool400001?ooo`800002oooo0P0000coool400000_ooo`@00004oooo100000Coool400001_oo
o`800009oooo0@00007oool001Goool00`000?ooooooo`02oooo400000;oool:00002_ooo`800008
oooo1P0000Coool200008_ooo`800006oooo0P0000Soool200002?ooo`80000;oooo0P0001Coool2
00001?ooo`H0000:oooo200000Coool400001?ooo`800002oooo0P0000coool400000_ooo`@00004
oooo100000Coool400001_ooo`800009oooo0@00007oool001Goool00`000?ooooooo`02oooo6000
00coool200000_ooo`80000<oooo100002Coool200004?ooo`800008oooo0P0000Ooool200001?oo
o`80000>oooo100000Coool800002?ooo`800006oooo0P0000Koool600005_ooo`800004oooo1P00
00Coool600001_ooo`@00007oooo0@00007oool001Goool00`000?ooooooo`02oooo600000coool2
00000_ooo`80000<oooo100002Coool200004?ooo`800008oooo0P0000Ooool200001?ooo`80000>
oooo100000Coool800002?ooo`800006oooo0P0000Koool600005_ooo`800004oooo1P0000Coool6
00001_ooo`@00007oooo0@00007oool001Goool00`000?ooooooo`02oooo500001Coool200000_oo
o`@00006oooo0P0000coool200006?ooo`80000Boooo0P0000Soool200002Oooo`@0000Foooo2000
00Koool600001?ooo`800006oooo1P0000Koool400005_ooo`@00004oooo100000Coool200002ooo
o`400001oooo000Eoooo0P0000?ooolD00005?ooo`800002oooo100000Koool200003?ooo`80000H
oooo0P0001;oool200002?ooo`800009oooo100001Koool800001_ooo`H00004oooo0P0000Koool6
00001_ooo`@0000Foooo100000Coool400001?ooo`80000:oooo0P00007oool001Goool00`000?oo
ooooo`02oooo200000;oool>00004_ooo`H0000Doooo0P0001Soool200007?ooo`@00009oooo0P00
013oool400001?ooo`P0000>oooo0P0000Koool400002?ooo`800006oooo200000coool:00004Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`P00002oooo3P0001;oool600005?ooo`80000H
oooo0P0001coool400002Oooo`80000@oooo100000Coool800003_ooo`800006oooo100000Soool2
00001_ooo`P0000<oooo2P00017oool100000Oooo`005Oooo`030000oooooooo00;oool800000_oo
o`800002oooo100000;oool200001?ooo`H00006oooo200000;oool200000_ooo`80000Xoooo0P00
01coool400008oooo`P0000:oooo0P0000coool400001?ooo`800004oooo0P0000Coool600003?oo
o`X00004oooo0P0000_oool100000Oooo`005Oooo`030000oooooooo00;oool800000_ooo`800002
oooo100000;oool200001?ooo`H00006oooo200000;oool200000_ooo`80000Xoooo0P0001coool4
00008oooo`P0000:oooo0P0000coool400001?ooo`800004oooo0P0000Coool600003?ooo`X00004
oooo0P0000_oool100000Oooo`005Oooo`030000oooooooo00;oool<00000_ooo`80000:oooo0P00
00;oool200001?ooo`X00002oooo1000013oool200005_ooo`@0000Noooo0P00027oool:00002_oo
o`800002oooo0P0000Soool200000_ooo`80000@oooo0P0000Koool400001_ooo`800002oooo3000
00Woool100000Oooo`005Oooo`030000oooooooo00;oool<00000_ooo`80000:oooo0P0000;oool2
00001?ooo`X00002oooo1000013oool200005_ooo`@0000Noooo0P00027oool:00002_ooo`800002
oooo0P0000Soool200000_ooo`80000@oooo0P0000Koool400001_ooo`800002oooo300000Woool1
00000Oooo`005Oooo`030000oooooooo00;oool:00000_ooo`80000<oooo0P0000;oool200000_oo
o`@00002oooo0P0000Koool40000:_ooo`80000Loooo0P0000;oool00`000?ooooooo`04oooo1000
01Coool200000_ooo`@00006oooo0P0000;oool200000_ooo`@0000:oooo100000Soool200007?oo
o``00004oooo0P0000Goool100000Oooo`002Oooo`<00003oooo00<0003oooooool00oooo`030000
oooooooo00;oool:00000_ooo`80000<oooo0P0000;oool200000_ooo`@00002oooo0P0000Koool4
0000:_ooo`80000Loooo0P0000;oool00`000?ooooooo`04oooo100001Coool200000_ooo`@00006
oooo0P0000;oool200000_ooo`@0000:oooo100000Soool200007?ooo``00004oooo0P0000Goool1
00000Oooo`002Oooo`030000oooooooo00;oool00`000?ooo`000004oooo00<0003oooooool00_oo
o`H0000:oooo0P0000;oool200000_ooo`H00002oooo100000Coool200002_ooo`80000Xoooo0P00
01koool300002_ooo`80000Boooo0P0000Koool200001?ooo`800008oooo0P0000Soool400000_oo
o`80000Loooo0P0000Soool200001?ooo`X00005oooo0@00007oool000[oool01`000?oooooooooo
o`000?ooo`000004oooo0P0000?oool600002_ooo`800002oooo0P0000;oool600000_ooo`@00004
oooo0P0000[oool20000:?ooo`80000Noooo0`0000[oool200004_ooo`800006oooo0P0000Coool2
00002?ooo`800008oooo100000;oool200007?ooo`800008oooo0P0000Coool:00001?ooo`800001
oooo000:oooo00L0003oooooooooool0003oool000001?ooo`030000oooooooo00;oool600000_oo
o`800006oooo100000Coool200001?ooo`H0000>oooo0P0000koool200005_ooo`@0000Noooo0P00
00;oool300002?ooo`80000Hoooo0P00013oool200002?ooo`@00002oooo0P0000Coool200004?oo
o`@0000<oooo3P0000Ooool100000Oooo`002Oooo`030000oooo000000;oool00`000?ooo`000004
oooo00<0003oooooool00_ooo`H00002oooo0P0000Koool400001?ooo`800004oooo1P0000koool2
00003_ooo`80000Foooo100001koool200000_ooo`<00008oooo0P0001Soool200004?ooo`800008
oooo100000;oool200001?ooo`80000@oooo100000coool>00001oooo`400001oooo000:oooo00<0
003oooooool00_ooo`030000oooooooo00?oool00`000?ooooooo`02oooo1P0000Coool200001_oo
o`800006oooo1P00013oool20000:?ooo`@0000Hoooo0P0000;oool200004Oooo`80000Doooo1000
013oool200002?ooo`800008oooo0P0000;oool400000_ooo`800008oooo100000[oool200000_oo
o`h00005oooo0@00007oool001Goool00`000?ooooooo`02oooo1P0000Coool200001_ooo`800006
oooo1P00013oool20000:?ooo`@0000Hoooo0P0000;oool200004Oooo`80000Doooo1000013oool2
00002?ooo`800008oooo0P0000;oool400000_ooo`800008oooo100000[oool200000_ooo`h00005
oooo0@00007oool001Goool00`000?ooooooo`02oooo200000Soool600000_ooo`H0000:oooo0P00
00Koool400008?ooo`800006oooo0P0000Soool400006Oooo`800008oooo0P0000coool200004_oo
o`@00004oooo0P0000coool200000_ooo`80000Roooo4P0000Goool100000Oooo`005Oooo`030000
oooooooo00;oool800002?ooo`H00002oooo1P0000[oool200001_ooo`@0000Poooo0P0000Koool2
00002?ooo`@0000Ioooo0P0000Soool200003?ooo`80000Boooo100000Coool200003?ooo`800002
oooo0P0002;ooolB00001Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`P00006oooo0P00
00;oool200000_ooo`@0000<oooo0P0000Koool20000;?ooo`800008oooo0P0000Soool200004Ooo
o`@0000Foooo0P0000Koool200001?ooo`800006oooo0P0000;oool200002_ooo`@00002oooo0P00
00;oool400004?ooo`80000:oooo400000Ooool100000Oooo`005Oooo`030000oooooooo00;oool8
00001_ooo`800002oooo0P0000;oool400003?ooo`800006oooo0P0002coool200002?ooo`800008
oooo0P00017oool400005_ooo`800006oooo0P0000Coool200001_ooo`800002oooo0P0000[oool4
00000_ooo`800002oooo1000013oool200002_oooa000007oooo0@00007oool001Goool00`000?oo
ooooo`02oooo200000Coool400001_ooo`80000:oooo0P0000Coool200005?ooo`800002oooo0P00
01;oool200000_ooo`@0000:oooo0P0000;oool200000_ooo`80000Aoooo100000Soool200003?oo
o`P0000Noooo100000;oool200001?ooo`80000>oooo100000;oool200001?ooo``00002oooo1000
00Ooool100000Oooo`005Oooo`800003oooo200000Coool400001_ooo`80000:oooo0P0000Coool2
00005?ooo`800002oooo0P0001;oool200000_ooo`@0000:oooo0P0000;oool200000_ooo`80000A
oooo100000Soool200003?ooo`P0000Noooo100000;oool200001?ooo`80000>oooo100000;oool2
00001?ooo``00002oooo100000Koool200000Oooo`005Oooo`030000oooooooo00;ooolB00003?oo
o`H00006oooo0P0000Coool400002_ooo`80000Boooo0P0000;oool200000_ooo`@0000<oooo0P00
00;oool200002_ooo`80000;oooo0P0000Koool200000_ooo`800008oooo0P0002[oool400000_oo
o`800002oooo0P0000Koool400000_ooo`@00002oooo0P0000;oool@00002Oooo`400001oooo000E
oooo00<0003oooooool00_oooa80000<oooo1P0000Koool200001?ooo`@0000:oooo0P0001;oool2
00000_ooo`800002oooo100000coool200000_ooo`80000:oooo0P0000_oool200001_ooo`800002
oooo0P0000Soool20000:_ooo`@00002oooo0P0000;oool200001_ooo`@00002oooo100000;oool2
00000_oooa000009oooo0@00007oool001Goool00`000?ooooooo`02oooo2P0000;oool600003?oo
o`@00006oooo0P0000;oool600004?ooo`80000>oooo300000Koool400003_ooo`80000?oooo0P00
00Koool200002_ooo`800008oooo0P0000Coool200002?ooo`80000:oooo0P0000Koool800006?oo
oa800009oooo0@00007oool001Goool00`000?ooooooo`02oooo2P0000;oool600003?ooo`@00006
oooo0P0000;oool600004?ooo`80000>oooo300000Koool400003_ooo`80000?oooo0P0000Koool2
00002_ooo`800008oooo0P0000Coool200002?ooo`80000:oooo0P0000Koool800006?oooa800009
oooo0@00007oool001Goool00`000?ooooooo`02oooo1P0000;oool200001?ooo`80000@oooo0P00
00Koool200005?ooo`80000Boooo0P0000;oool800002?ooo`800004oooo100000;oool400008ooo
o`80000Toooo0P0000Soool:00000_ooo`@00004oooo200000CooolF00001Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`H00002oooo0P0000Coool200004?ooo`800006oooo0P0001Coool2
00004_ooo`800002oooo200000Soool200001?ooo`@00002oooo100002?oool200009?ooo`800008
oooo2P0000;oool400001?ooo`P00004oooo5P0000Goool100000Oooo`005Oooo`030000oooooooo
00;oool200002_ooo`80000@oooo1P0000;oool200000_ooo`800004oooo0P0000coool200005?oo
o`800002oooo1P0000Koool400001?ooo`800004oooo100001?oool200004_ooo`80000:oooo0P00
01Coool200001_ooo``00002oooo0P0000Soool400001_ooo``00002oooo100000;oool200001Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`80000:oooo0P00013oool600000_ooo`800002
oooo0P0000Coool200003?ooo`80000Doooo0P0000;oool600001_ooo`@00004oooo0P0000Coool4
00004oooo`80000Boooo0P0000[oool200005?ooo`800006oooo300000;oool200002?ooo`@00006
oooo300000;oool400000_ooo`800005oooo0@00007oool001Goool00`000?ooooooo`04oooo0P00
013oool200000_ooo`@00004oooo0P0000Koool200001_ooo`80000@oooo0P0001Coool800002?oo
o`800006oooo0P0000koool00`000?ooooooo`08oooo0P0000koool200008_ooo`800008oooo1P00
00koool600001?oooa@00002oooo0P0000Goool100000Oooo`005Oooo`800005oooo0P00013oool2
00000_ooo`@00004oooo0P0000Koool200001_ooo`80000@oooo0P0001Coool800002?ooo`800006
oooo0P0000koool00`000?ooooooo`08oooo0P0000koool200008_ooo`800008oooo1P0000koool6
00001?oooa@00002oooo0P0000Coool200000Oooo`005Oooo`030000oooooooo01Koool200001_oo
o`X00004oooo100000;oool20000:?ooo`H00004oooo0P0001;oool200001?ooo`<00008oooo1000
00Coool400001_ooo`80000:oooo0P0000[oool400001_ooo`800008oooo0P0000;oool200001?oo
o`H00006oooo100000KooolH00001Oooo`400001oooo000Eoooo00<0003oooooool05_ooo`800006
oooo2P0000Coool400000_ooo`80000Xoooo1P0000Coool200004_ooo`800004oooo0`0000Soool4
00001?ooo`@00006oooo0P0000[oool200002_ooo`@00006oooo0P0000Soool200000_ooo`800004
oooo1P0000Koool400001_oooaP00005oooo0@00007oool001Goool00`000?ooooooo`06oooo0P00
00Soool200001_ooo`800006oooo100000Soool200000_ooo`@0000@oooo0P0001;oool400001?oo
o`800004oooo0P0000Koool200000_ooo`@0000:oooo0P0000_oool400000_ooo`@00006oooo0P00
01Koool400001_ooo`800004oooo0P0000;oool200000_ooo`800006oooo1P0000Coool600000_oo
oaX00005oooo0@00007oool001Goool00`000?ooooooo`06oooo0P0000Soool200001_ooo`800006
oooo100000Soool200000_ooo`@0000@oooo0P0001;oool400001?ooo`800004oooo0P0000Koool2
00000_ooo`@0000:oooo0P0000_oool400000_ooo`@00006oooo0P0001Koool400001_ooo`800004
oooo0P0000;oool200000_ooo`800006oooo1P0000Coool600000_oooaX00005oooo0@00007oool0
01Goool00`000?ooooooo`0:oooo0P0000Coool200001?ooo`@00004oooo0P0000Coool200001?oo
o`P0000Xoooo100000;oool200000_ooo`800006oooo100001;oool300001_ooo`@00002oooo0P00
00;oool200003?ooo`80000Boooo0P0000Coool200000_ooo`800004oooo0P0000Coool200001_oo
o`H00004oooo1P0000Coool<00000_ooo`X00005oooo0@00007oool001Goool00`000?ooooooo`0:
oooo0P0000Coool200001?ooo`@00004oooo0P0000Coool200001?ooo`P0000Xoooo100000;oool2
00000_ooo`800006oooo100001;oool300001_ooo`@00002oooo0P0000;oool200003?ooo`80000B
oooo0P0000Coool200000_ooo`800004oooo0P0000Coool200001_ooo`H00004oooo1P0000Coool<
00000_ooo`X00005oooo0@00007oool001Goool00`000?ooooooo`0Boooo0P0000;oool400001?oo
o`P00004oooo0P0000;oool40000:_ooo`800002oooo0P0000Koool200000_ooo`@0000Ooooo1P00
02Soool200002_ooo`800002oooo100000;oool200000_ooo`@00002oooo0P0000;oool200000_oo
o`h00008oooo1P0000Goool100000Oooo`005Oooo`030000oooooooo01;oool200000_ooo`@00004
oooo200000Coool200000_ooo`@0000Zoooo0P0000;oool200001_ooo`800002oooo100001ooool6
0000:?ooo`80000:oooo0P0000;oool400000_ooo`800002oooo100000;oool200000_ooo`800002
oooo3P0000Soool600001Oooo`400001oooo000Eoooo00<0003oooooool05_ooo`@00004oooo1000
00;oool200001?ooo`P0000Boooo0P0001Koool200001_ooo`800004oooo100001Ooool200004?oo
o`80000@oooo0P0000Soool400001?ooo`80000:oooo0P0000;oool600000_ooo`800002oooo0P00
00KooolF00004Oooo`400001oooo000Eoooo0P0001Ooool400001?ooo`@00002oooo0P0000Coool8
00004_ooo`80000Foooo0P0000Koool200001?ooo`@0000Goooo0P00013oool200004?ooo`800008
oooo100000Coool200002_ooo`800002oooo1P0000;oool200000_ooo`800006oooo5P00013oool2
00000Oooo`005Oooo`030000oooooooo01Soool200001?ooo`@00002oooo0P0000;oool:00004_oo
o`80000Foooo0P0000;oool200001?ooo`@00002oooo0P00013oool200001Oooo`@00004oooo0P00
00Koool400000_ooo`800008oooo0P0000;oool200002_ooo`800004oooo0P0000Soool400001_oo
o`800004oooo100000;oool600000_ooo`800002oooo1P0000Coool200001_ooo`P00005oooo0@00
007oool001Goool00`000?ooooooo`0Hoooo0P0000Coool400000_ooo`800002oooo2P0001;oool2
00005_ooo`800002oooo0P0000Coool400000_ooo`80000@oooo0P0000Goool400001?ooo`800006
oooo100000;oool200002?ooo`800002oooo0P0000[oool200001?ooo`800008oooo100000Koool2
00001?ooo`@00002oooo1P0000;oool200000_ooo`H00004oooo0P0000Koool800001Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`h00008oooo0P0000;oool600000_ooo`H00004oooo1000
02[oool200001_ooo`H00006oooo0P0000;oool200001?ooo`80000Koooo0P00013oool200002_oo
o`@00002oooo0P0000[oool200001?ooo`@00002oooo5P0000Coool400005Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`h00008oooo0P0000;oool600000_ooo`H00004oooo100002[oool2
00001_ooo`H00006oooo0P0000;oool200001?ooo`80000Koooo0P00013oool200002_ooo`@00002
oooo0P0000[oool200001?ooo`@00002oooo5P0000Coool400005Oooo`400001oooo000Eoooo00<0
003oooooool00_oooa@00002oooo0P0000Coool200001?ooo`@00006oooo100002;oool400001?oo
o`@00006oooo0P0000;oool600000_ooo`80000Aoooo1P0000coool200001?ooo`80000Foooo1000
00;oool400001?ooo`@00006oooo100000Koool600000_ooo`H00002oooo100000;oool800002_oo
o`800005oooo0@00007oool001Goool00`000?ooooooo`02oooo500000;oool200001?ooo`800004
oooo100000Koool400008_ooo`@00004oooo100000Koool200000_ooo`H00002oooo0P00017oool6
00003?ooo`800004oooo0P0001Koool400000_ooo`@00004oooo100000Koool400001_ooo`H00002
oooo1P0000;oool400000_ooo`P0000:oooo0P0000Goool100000Oooo`005Oooo`030000oooooooo
00;ooolD00002_ooo`800002oooo1P0000Coool400009?ooo`800002oooo100000;oool200000_oo
o`800006oooo0P0000Koool200008oooo`80000<oooo0P0000[oool200000_ooo`800002oooo0P00
00Koool200001_ooo`H00006oooo200000;oool600002_ooo`800004oooo200000Ooool100000Ooo
o`002_ooo`<00002oooo00<0003oooooool00oooo`030000oooooooo00;ooolD00002_ooo`800002
oooo1P0000Coool400009?ooo`800002oooo100000;oool200000_ooo`800006oooo0P0000Koool2
00008oooo`80000<oooo0P0000[oool200000_ooo`800002oooo0P0000Koool200001_ooo`H00006
oooo200000;oool600002_ooo`800004oooo200000Ooool100000Oooo`002oooo`060000oooooooo
0000oooo00001?ooo`030000oooooooo00coool200003?ooo`800004oooo0P0000;oool600001?oo
o`H00006oooo0P0001[oool200000_ooo`@00006oooo0P0000[oool200005oooo`800002oooo0P00
00[oool400000_ooo`800002oooo0P0000Coool200002?ooo`80000<oooo100001;oool200000_oo
o`800004oooo100000;oool200000_ooo`@0000:oooo100000;oool200001Oooo`400001oooo0009
oooo10000004oooo0000oooo00001?ooo`80000=oooo0P0000coool200001?ooo`800002oooo1P00
00Coool600001_ooo`80000Joooo0P0000;oool400001_ooo`80000:oooo0P0001Ooool200000_oo
o`80000:oooo100000;oool200000_ooo`800004oooo0P0000Soool200003?ooo`@0000Boooo0P00
00;oool200001?ooo`@00002oooo0P0000;oool400002_ooo`@00002oooo0P0000Coool200000Ooo
o`002Oooo`030000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool00_oooa@00004
oooo200000;oool800000_ooo`H00008oooo0P0001Soool800001?ooo`@00004oooo0P0000;oool2
00001_ooo`@00008oooo00<0003oooooool05?ooo`@0000Toooo0P0000[oool400000_ooo`@00004
oooo100000;oool400000_ooo`H00004oooo0P0000Coool200000_ooo`800009oooo0@00007oool0
00[oool200000_ooo`030000oooo000000Coool00`000?ooooooo`02oooo500000Coool800000_oo
o`P00002oooo1P0000Soool200006?ooo`P00004oooo100000Coool200000_ooo`800006oooo1000
00Soool00`000?ooooooo`0Doooo100002Coool200002_ooo`@00002oooo100000Coool400000_oo
o`@00002oooo1P0000Coool200001?ooo`800002oooo0P0000Woool100000Oooo`002oooo`050000
oooooooooooo000000Goool00`000?ooooooo`02oooo500000Coool400000_ooo`800004oooo1P00
00;oool600009_ooo`@00002oooo4P0000Koool200000_ooo`800009oooo0P00013oool400005_oo
o`H00006oooo100000Koool400000_ooo`800002oooo200000Coool200000_ooo`@00008oooo0P00
00;oool200001?ooo`80000;oooo0@00007oool001Goool00`000?ooooooo`02oooo500000Coool4
00000_ooo`800004oooo1P0000;oool600009_ooo`@00002oooo4P0000Koool200000_ooo`800009
oooo0P00013oool400005_ooo`H00006oooo100000Koool400000_ooo`800002oooo200000Coool2
00000_ooo`@00008oooo0P0000;oool200001?ooo`80000;oooo0@00007oool001Goool00`000?oo
ooooo`02oooo500000CooolB00001?ooo`@00002oooo0P0000koool200004_ooo`@0000<oooo0P00
00koool200002oooo`@00002oooo100000Coool200001?ooo`80000<oooo0P0000Soool200000_oo
o`800004oooo100000coool400000_ooo`800006oooo0P0000Koool:00001_ooo`80000Aoooo0@00
007oool001Goool00`000?ooooooo`02oooo500000CooolB00001?ooo`@00002oooo0P0000koool2
00004_ooo`@0000<oooo0P0000koool200002oooo`@00002oooo100000Coool200001?ooo`80000<
oooo0P0000Soool200000_ooo`800004oooo100000coool400000_ooo`800006oooo0P0000Koool:
00001_ooo`80000Aoooo0@00007oool001Goool00`000?ooooooo`02oooo500000Coool800000_oo
o`800002oooo100000Coool400001?ooo`800006oooo0P0000Coool200004_ooo`@0000:oooo1000
00koool200003Oooo`@00002oooo1P0000;oool200002_ooo`800004oooo0P0000coool200001?oo
o`80000>oooo1P0000[oool400004?ooo`80000Coooo0@00007oool001Goool00`000?ooooooo`02
oooo500000Coool800000_ooo`800002oooo100000Coool400001?ooo`800006oooo0P0000Coool2
00004_ooo`@0000:oooo100000koool200003Oooo`@00002oooo1P0000;oool200002_ooo`800004
oooo0P0000coool200001?ooo`80000>oooo1P0000[oool400004?ooo`80000Coooo0@00007oool0
01Goool00`000?ooooooo`02oooo500000Koool<00000_ooo`@00002oooo1P0000Koool200000_oo
o`80000Foooo0P0000coool200006?ooo`800005oooo0P0000Coool400000_ooo`800004oooo0P00
00Koool200001?ooo`80000<oooo100000;oool400009_ooo`80000<oooo0P00017oool100000Ooo
o`005Oooo`800003oooo500000Koool<00000_ooo`@00002oooo1P0000Koool200000_ooo`80000F
oooo0P0000coool200006?ooo`800005oooo0P0000Coool400000_ooo`800004oooo0P0000Koool2
00001?ooo`80000<oooo100000;oool400009_ooo`80000<oooo0P00013oool200000Oooo`005Ooo
o`030000oooooooo00;ooolD00001?ooo`h00008oooo1P0000Koool600001_ooo`800004oooo0P00
00Coool200001?ooo`800004oooo0P0001Coool600000_ooo`800002oooo0`0000Coool600002?oo
o`@00006oooo0P0000Coool200004?ooo`800002oooo1P00033oool200004Oooo`400001oooo000E
oooo00<0003oooooool00_oooa@00004oooo3P0000Soool600001_ooo`H00006oooo0P0000Coool2
00001?ooo`800004oooo0P0000Coool200005?ooo`H00002oooo0P0000;oool300001?ooo`H00008
oooo100000Koool200001?ooo`80000@oooo0P0000;oool60000<?ooo`80000Aoooo0@00007oool0
01Goool00`000?ooooooo`02oooo500000Coool<00003?ooo`@0000>oooo0P0000Koool200003?oo
o`800004oooo0P0001;oool600000_ooo`@00003oooo0P0000[oool:00000_ooo`@00002oooo0P00
00Coool200003_ooo`80000boooo0P0000Koool200000_ooo`80000=oooo0@00007oool001Goool0
0`000?ooooooo`02oooo500000Coool<00003?ooo`@0000>oooo0P0000Koool200003?ooo`800004
oooo0P0001;oool600000_ooo`@00003oooo0P0000[oool:00000_ooo`@00002oooo0P0000Coool2
00003_ooo`80000boooo0P0000Koool200000_ooo`80000=oooo0@00007oool001Goool00`000?oo
ooooo`02oooo500000Coool<00006?ooo`80000<oooo100000[oool200001?ooo`80000@oooo1000
00Koool600001Oooo`H00002oooo0P0000Coool:00000_ooo`800004oooo0P0000[oool60000<_oo
o`80000>oooo0P0000Woool100000Oooo`005Oooo`030000oooooooo00;ooolD00001?ooo``0000H
oooo0P0000coool400002_ooo`800004oooo0P00013oool400001_ooo`H00005oooo1P0000;oool2
00001?ooo`X00002oooo0P0000Coool200002_ooo`H0000boooo0P0000koool200002Oooo`400001
oooo000Eoooo00<0003oooooool01_ooo`80000joooo100000Coool200001?ooo`800002oooo0P00
00Coool200002?ooo`@00006oooo200000;ooolA00001_ooo`800002oooo300000Koool200001_oo
o`@00010oooo0P0000;oool200001_ooo`800007oooo0@00007oool001Goool00`000?ooooooo`06
oooo0P0003[oool400001?ooo`800004oooo0P0000;oool200001?ooo`800008oooo100000Koool8
00000_oooa400006oooo0P0000;oool<00001_ooo`800006oooo1000043oool200000_ooo`800006
oooo0P0000Ooool100000Oooo`005Oooo`030000oooooooo04[oool200000_ooo`@00002oooo0P00
00[oool200000_ooo`800004oooo2P0000CooolE00000_ooo`P00002oooo1P0000Koool60000B_oo
o``00005oooo0@00007oool001Goool20000Boooo`800002oooo100000;oool200002_ooo`800002
oooo0P0000Coool:00001?oooaD00002oooo200000;oool600001_ooo`H0001:oooo300000Coool2
00000Oooo`005Oooo`030000oooooooo05;oool200000_ooo`800008oooo0P0000Soool200000_oo
o`@00002oooo6@0000;oool800000_ooo`H00004oooo100005Soool200001Oooo`400001oooo000E
oooo00<0003oooooool0D_ooo`800002oooo0P0000Soool200002?ooo`800002oooo100000;ooolI
00000_ooo`P00002oooo1P0000Coool40000F?ooo`800005oooo0@00007oool001Goool00`000?oo
ooooo`1Doooo0P0000;oool400001_ooo`800006oooo0P0000;oool200000_oooal00002oooo1000
00;oool600001?ooo`@00014oooo1P0000[oool200002Oooo`400001oooo000Eoooo00<0003ooooo
ool0E?ooo`800002oooo100000Koool200001_ooo`800002oooo0P0000;ooolO00000_ooo`@00002
oooo1P0000Coool40000A?ooo`H0000:oooo0P0000Woool100000Oooo`005Oooo`030000oooooooo
05Koool200003?ooo`X00002oooo8@0000;oool200000_ooo`P00004oooo0P0005;oool200000_oo
o`800009oooo0@00007oool001Goool00`000?ooooooo`1Foooo0P0000coool:00000_ooob400002
oooo0P0000;oool800001?ooo`80001Boooo0P0000;oool200002Oooo`400001oooo000Eoooo00<0
003oooooool0F_ooo`800004oooo0P0000;ooolk00000_ooo`80001Foooo0P0000_oool100000Ooo
o`005Oooo`030000oooooooo05[oool200001?ooo`800002oooo>`0000;oool20000E_ooo`80000;
oooo0@00007oool001Goool00`000?ooooooo`1Poooo100000Cooolk0000Hoooo`400001oooo000E
oooo0P00067oool400001?oooc/0001Roooo0P00007oool001Goool00`000?ooooooo`1Toooo0P00
00;oooli0000IOooo`400001oooo000Eoooo00<0003oooooool0I?ooo`800002oooo>@0006Goool1
00000Oooo`005Oooo`030000oooooooo06;oool200000_ooo`800002oooo<`0006Woool100000Ooo
o`005Oooo`030000oooooooo06;oool200000_ooo`800002oooo<`0006Woool100000Oooo`005Ooo
o`030000oooooooo06;oool200000_ooocD0001[oooo0@00007oool001Goool00`000?ooooooo`1R
oooo0P0000;ooole0000Joooo`400001oooo000Eoooo00<0003oooooool0H?ooo`@00004oooo<`00
06_oool100000Oooo`002Oooo`800004oooo00<0003oooooool00oooo`030000oooooooo063oool4
00001?oooc<0001[oooo0@00007oool000Woool00`000?ooo`000002oooo00<0003oool000001?oo
o`030000oooooooo06Kooolc0000KOooo`400001oooo0009oooo00<0003oool000000_ooo`030000
oooo000000Coool20000Iooooc<0001/oooo0P00007oool000Woool200000oooo`030000oooo0000
00Coool00`000?ooooooo`1Toooo<@00077oool100000Oooo`002Oooo`030000oooooooo00;oool0
0`000?ooo`000004oooo00<0003oooooool0I?oooc40001aoooo0@00007oool000[oool200000ooo
o`030000oooooooo00?oool00`000?ooooooo`1Xoooo:`0007?oool100000Oooo`005Oooo`030000
oooooooo06Soool[0000Loooo`400001oooo000Eoooo00<0003oooooool0J?ooobd0001aoooo0@00
007oool001Goool00`000?ooooooo`1Xoooo;@00077oool100000Oooo`005Oooo`030000oooooooo
06Coool>00001_oooal0001_oooo0@00007oool001Goool00`000?ooooooo`1Toooo3P0000KooolO
0000Koooo`400001oooo000Eoooo00<0003oooooool0I?ooo`h00008oooo7`0006goool100000Ooo
o`005Oooo`80001Uoooo3P0000SooolO0000K?ooo`800001oooo000Eoooo00<0003oooooool0H_oo
oaH00004oooo0P0000;ooolI0000KOooo`400001oooo000Eoooo00<0003oooooool0H_oooaH00004
oooo0P0000;ooolI0000KOooo`400001oooo000Eoooo00<0003oooooool0H_oooaX00006oooo2`00
00;oool<0000Joooo`400001oooo000Eoooo00<0003oooooool0H_oooaX00006oooo2`0000;oool<
0000Joooo`400001oooo000Eoooo00<0003oooooool0G?oooa000002oooo100000CooolU0000Jooo
o`400001oooo000Eoooo00<0003oooooool0G?oooa000002oooo100000CooolU0000Joooo`400001
oooo000Eoooo00<0003oooooool0G_ooo``00002oooo1P0000;ooolG00000_oooa80001Woooo0@00
007oool001Goool00`000?ooooooo`1Noooo300000;oool600000_oooaL00002oooo4P0006Ooool1
00000Oooo`005Oooo`030000oooooooo05koool>00001?ooo`h00005oooo700006Ooool100000Ooo
o`005Oooo`80001Ooooo3P0000Coool>00001Ooooa`0001Voooo0P00007oool001Goool00`000?oo
ooooo`1Loooo300000;oool400000_ooo`800002oooo1P0000CooolQ0000Ioooo`400001oooo000E
oooo00<0003oooooool0G?ooo``00002oooo100000;oool200000_ooo`H00004oooo8@0006Ooool1
00000Oooo`005Oooo`030000oooooooo05[oool>00000_ooo`@00004oooo0P0000CooolE00000_oo
oa00001Woooo0@00007oool001Goool00`000?ooooooo`1Joooo3P0000;oool400001?ooo`800004
oooo5@0000;oool@0000Ioooo`400001oooo000Eoooo00<0003oooooool0F_oooa800002oooo0P00
01;oool00`000?ooooooo`02oooo6P0006Goool100000Oooo`005Oooo`030000oooooooo05[ooolB
00000_ooo`80000Boooo00<0003oooooool00_oooaX0001Uoooo0@00007oool001Goool00`000?oo
ooooo`1Joooo0P0000;oool<00001_ooo`800008oooo100000;oool900000_oooaH0001Uoooo0@00
007oool001Goool00`000?ooooooo`1Joooo0P0000;oool<00001_ooo`800008oooo100000;oool9
00000_oooaH0001Uoooo0@00007oool001Goool00`000?ooooooo`1Joooo3P0000;oool200003_oo
o`/00004oooo0P0000;ooolD0000IOooo`400001oooo000Eoooo0P0005_oool>00000_ooo`80000>
oooo2`0000Coool200000_oooa@0001Toooo0P00007oool001Goool00`000?ooooooo`1Hoooo3P00
00;oool800000_ooo`800002oooo5@0000;ooolD0000IOooo`400001oooo000Eoooo00<0003ooooo
ool0F?ooo`h00002oooo200000;oool200000_oooaD00002oooo500006Goool100000Oooo`005Ooo
o`030000oooooooo05Soool>00001_ooo`@00004oooo0P0000;ooolC00000_oooa@0001Uoooo0@00
007oool001Goool00`000?ooooooo`1Hoooo3P0000Koool400001?ooo`800002oooo4`0000;ooolD
0000IOooo`400001oooo000Eoooo00<0003oooooool0F?ooo`X0000<oooo0P0000Coool200001?oo
o`800002oooo1@0000;oool200001_ooo`H00002oooo300006Goool100000Oooo`005Oooo`030000
oooooooo05Soool:00003?ooo`800004oooo0P0000Coool200000_ooo`D00002oooo0P0000Koool6
00000_ooo``0001Uoooo0@00007oool001Goool00`000?ooooooo`1Joooo2P0000koool200001?oo
o`800006oooo1@0001;oool:0000IOooo`400001oooo000:oooo00<0003oooooool00_ooo`030000
oooooooo00?oool00`000?ooooooo`1Joooo2P0000koool200001?ooo`800006oooo1@0001;oool:
0000IOooo`400001oooo0009oooo00<0003oool000000_ooo`030000oooo000000Coool00`000?oo
ooooo`1Joooo2P0000Koool:00001?ooo`800002oooo0P0000;oool500000_ooo``00004oooo2000
06Ooool100000Oooo`002Oooo`030000oooo000000;oool00`000?ooo`000004oooo0P0005_oool:
00001_ooo`X00004oooo0P0000;oool200000_ooo`D00002oooo300000Coool80000I_ooo`800001
oooo000:oooo00L0003oooooooooool0003oool000001?ooo`030000oooooooo05Soool800002_oo
o`800002oooo200000[oool300000_ooo`h00004oooo200006Ooool100000Oooo`002Oooo`030000
oooo000000;oool00`000?ooo`000004oooo00<0003oooooool0F?ooo`P0000:oooo0P0000;oool8
00002_ooo`<00002oooo3P0000Coool80000Ioooo`400001oooo000:oooo00<0003oooooool00_oo
o`030000oooooooo00?oool00`000?ooooooo`1Joooo700000coool500000_oooaH0001Woooo0@00
007oool001Goool00`000?ooooooo`1Joooo700000coool500000_oooaH0001Woooo0@00007oool0
01Goool00`000?ooooooo`1Hoooo2P0000;ooolF00002?oooad0001Woooo0@00007oool001Goool0
0`000?ooooooo`1Hoooo2P0000;ooolF00002?oooad0001Woooo0@00007oool001Goool00`000?oo
ooooo`1Hoooo400000;oool200000_ooo`h00004oooo7`0006Ooool100000Oooo`005Oooo`030000
oooooooo05Soool@00000_ooo`800002oooo3P0000CooolO0000Ioooo`400001oooo000Eoooo00<0
003oooooool0F?ooo`@00006oooo2P0000Koool:00001?ooo`L00002oooo1P0000;oool>0000Iooo
o`400001oooo000Eoooo0P0005Woool400001_ooo`X00006oooo2P0000Coool700000_ooo`H00002
oooo3P0006Koool200000Oooo`005Oooo`030000oooooooo05Soool200002?ooo`800002oooo0P00
00;oool400001_ooob/0001Woooo0@00007oool001Goool00`000?ooooooo`1Hoooo0P0000Soool2
00000_ooo`800002oooo100000Koool[0000Ioooo`400001oooo000Eoooo00<0003oooooool0F?oo
o`80000:oooo5P0000SooolI00000_ooo`80001Woooo0@00007oool001Goool00`000?ooooooo`1H
oooo0P0000[ooolF00002?oooaT00002oooo0P0006Ooool100000Oooo`005Oooo`030000oooooooo
05[oool200001?ooo`800002oooo0P0000Koool200000_ooo`X00002oooo0P0000Coool;00002?oo
o`800004oooo100006Ooool100000Oooo`005Oooo`030000oooooooo05[oool200001?ooo`800002
oooo0P0000Koool200000_ooo`X00002oooo0P0000Coool;00002?ooo`800004oooo100006Ooool1
00000Oooo`005Oooo`030000oooooooo05[oool400004_ooo`80000Coooo100000;oool200003_oo
o`80001Yoooo0@00007oool001Goool00`000?ooooooo`1Joooo100001;oool200004oooo`@00002
oooo0P0000koool20000JOooo`400001oooo000Eoooo00<0003oooooool0H_ooo`80000Boooo0P00
00ooool200000_ooo`800004oooo0P0000Coool200000_ooo`80001Yoooo0@00007oool001Goool2
0000Hoooo`80000Boooo0P0000ooool200000_ooo`800004oooo0P0000Coool200000_ooo`80001X
oooo0P00007oool001Goool00`000?ooooooo`1Loooo100002goool400000_ooo`800002oooo1000
06_oool100000Oooo`005Oooo`030000oooooooo05coool40000;Oooo`@00002oooo0P0000;oool4
0000Joooo`400001oooo000Eoooo00<0003oooooool0K_ooo`80000Coooo100000Soool20000MOoo
o`400001oooo000Eoooo00<0003oooooool0K_ooo`80000Coooo100000Soool20000MOooo`400001
oooo000Eoooo00<0003oooooool0H?ooo`80000Woooo0P0000;oool200000_ooo`800002oooo0P00
06ooool100000Oooo`005Oooo`030000oooooooo063oool200009oooo`800002oooo0P0000;oool2
00000_ooo`80001_oooo0@00007oool001Goool00`000?ooooooo`1Roooo200002Goool60000LOoo
o`400001oooo000Eoooo00<0003oooooool0H_ooo`P0000Uoooo1P00077oool100000Oooo`005Ooo
o`030000oooooooo06Coool400003_ooo`800008oooo1@0000;oool200000_ooo`800002oooo0P00
07Goool100000Oooo`005Oooo`80001Uoooo100000koool200002?ooo`D00002oooo0P0000;oool2
00000_ooo`80001doooo0P00007oool001Goool00`000?ooooooo`1Zoooo0P0000;oool200003?oo
o`L00006oooo100007Woool100000Oooo`005Oooo`030000oooooooo06[oool200000_ooo`80000<
oooo1`0000Koool40000NOooo`400001oooo000Eoooo00<0003oooooool0K?ooo`800008oooo0P00
00[oool00`000?ooooooo`21oooo0@00007oool001Goool00`000?ooooooo`1/oooo0P0000Soool2
00002_ooo`030000oooooooo087oool100000Oooo`005Oooo`030000oooooooo0?ooool7oooo0@00
007oool001Goool00`000?ooooooo`3ooooo1oooo`400001oooo000Eoooo00<0003oooooool0oooo
o`Ooool100000Oooo`000oooo`D00002oooo00<0003oooooool00_ooo`030000oooooooo00?oool0
0`000?ooooooo`3ooooo1oooo`400001oooo0005oooo00L0003oooooooooool0003oool000000_oo
o`030000oooo000000Coool00`000?ooooooo`3ooooo1oooo`400001oooo0005oooo00L0003ooooo
ooooool0003oool000000_ooo`030000oooo000000Coool20000ooooo`Ooool200000Oooo`001Ooo
o`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool0oooo
o`Ooool100000Oooo`001Oooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004
oooo00<0003oooooool0ooooo`Ooool100000Oooo`001?ooo`800004oooo00<0003oooooool00_oo
o`030000oooooooo00?oool00`000?ooooooo`3ooooo1oooo`400001oooo000Eoooo00<0003ooooo
ool0ooooo`Ooool100000Oooo`005Oooo`030000oooooooo0?ooool7oooo0@00007oool001Goool0
0`000?ooooooo`3ooooo1oooo`400001oooo000Eoooo00<0003oooooool0ooooo`Ooool100000Ooo
o`005Oooo`030000oooooooo0?ooool7oooo0@00007oool001Goool00`000?ooooooo`3ooooo1ooo
o`400001oooo000Eoooo0P000?ooool7oooo0P00007oool001Goool00`000?ooooooo`06oooo0P00
0?koool100000Oooo`005Oooo`030000oooooooo00Koool20000o_ooo`400001oooo000Eoooo00<0
003oooooool0ooooo`Ooool100000Oooo`005Oooo`030000oooooooo0?ooool7oooo0@00007oool0
01Goool00`000?ooooooo`3ooooo1oooo`400001oooo000Eoooo00<0003oooooool0ooooo`Ooool1
00000Oooo`005Oooo`030000oooooooo00[oool200002_ooo`80003^oooo0@00007oool001Goool0
0`000?ooooooo`0:oooo0P0000[oool20000k_ooo`400001oooo000Eoooo00<0003oooooool06?oo
o`80003/oooo0@00007oool001Goool200006Oooo`80003[oooo0P00007oool001Goool00`000?oo
ooooo`0:oooo0P000?[oool100000Oooo`005Oooo`030000oooooooo00[oool20000n_ooo`400001
oooo000Eoooo00<0003oooooool01_ooo`80000<oooo0P000?3oool100000Oooo`005Oooo`030000
oooooooo00Koool200003?ooo`80003`oooo0@00007oool001Goool00`000?ooooooo`02oooo0P00
00;oool:0000koooo`800005oooo0@00007oool001Goool00`000?ooooooo`02oooo0P0000;oool:
0000koooo`800005oooo0@00007oool001Goool00`000?ooooooo`02oooo0P0000;oool200002_oo
o`80000:oooo0P0000coool20000f?ooo`400001oooo000Eoooo00<0003oooooool00_ooo`800002
oooo0P0000[oool200002_ooo`80000<oooo0P000=Soool100000Oooo`005Oooo`030000oooooooo
00;oool600002?ooo`@0000Boooo0P0000;oool200003?ooo`800002oooo0P0000;oool20000a?oo
o`400001oooo000Eoooo0P0000?oool600002?ooo`@0000Boooo0P0000;oool200003?ooo`800002
oooo0P0000;oool20000`oooo`800001oooo000Eoooo00<0003oooooool00_ooo`H00002oooo0P00
00Soool200000_ooo`H00006oooo0P0000;oool200004?ooo`80003:oooo0@00007oool001Goool0
0`000?ooooooo`02oooo1P0000;oool200002?ooo`800002oooo1P0000Koool200000_ooo`80000@
oooo0P000<[oool100000Oooo`005Oooo`030000oooooooo00Coool400000_oooa800004oooo0P00
00Soool200009_ooo`80000Hoooo100000koool200007Oooo`80000<oooo0P0000Koool200005?oo
o`80000Joooo0P0001ooool100000Oooo`005Oooo`030000oooooooo00Coool400000_oooa800004
oooo0P0000Soool200009_ooo`80000Hoooo100000koool200007Oooo`80000<oooo0P0000Koool2
00005?ooo`80000Joooo0P0001ooool100000Oooo`005Oooo`030000oooooooo00;oool800000_oo
o`H00002oooo0P0000;oool600001_ooo`80000@oooo0P0000Soool200008?ooo`80000ooooo0P00
01Coool20000Boooo`400001oooo000Eoooo00<0003oooooool00_ooo`P00002oooo1P0000;oool2
00000_ooo`H00006oooo0P00013oool200002?ooo`80000Poooo0P0003ooool200005?ooo`80001;
oooo0@00007oool001Goool00`000?ooooooo`02oooo300000Coool200000_ooo`H00002oooo1000
00Koool200000_ooo`@00008oooo100001Koool200001?ooo`800006oooo0P0000Coool200004?oo
o`800002oooo0P0008Soool100000Oooo`000oooo`D000001?ooo`000000000000?oool00`000?oo
ooooo`03oooo00<0003oooooool00_ooo``00004oooo0P0000;oool600000_ooo`@00006oooo0P00
00;oool400002?ooo`@0000Foooo0P0000Coool200001_ooo`800004oooo0P00013oool200000_oo
o`800028oooo0@00007oool000Goool01@000?ooooooooooo`000004oooo00<0003oool000001?oo
o`030000oooooooo00;oool>00000_ooo``00002oooo2P0000Coool200000_ooo`800004oooo0P00
00;oool200001?ooo`800006oooo0P0000;oool200003_ooo`800008oooo0P00013oool200001_oo
o`030000oooooooo00Soool200001?ooo`800002oooo0P0000Koool200002?ooo`800004oooo0P00
00coool200004?ooo`80000eoooo0@00007oool000Goool00`000?ooooooo`02oooo00L0003ooooo
ooooool0003oool000001?ooo`800003oooo3P0000;oool<00000_ooo`X00004oooo0P0000;oool2
00001?ooo`800002oooo0P0000Coool200001_ooo`800002oooo0P0000koool200002?ooo`80000@
oooo0P0000Koool00`000?ooooooo`08oooo0P0000Coool200000_ooo`800006oooo0P0000Soool2
00001?ooo`80000<oooo0P00013oool20000=?ooo`800001oooo0005oooo00<0003oooooool00_oo
o`070000oooooooooooo0000oooo000000Coool00`000?ooooooo`02oooo400000;ooolF00000_oo
o`800002oooo200000;oool200001_ooo`@00002oooo0P0000Koool600002?ooo`800006oooo0P00
00;oool200002?ooo`800006oooo0P0000;oool200002oooo`800002oooo0P0000Koool200003?oo
o`80000>oooo0P0002Soool200001_ooo`@00008oooo0P0000ooool100000Oooo`001Oooo`070000
oooooooooooo0000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool00_oooa000002
oooo5P0000;oool200000_ooo`P00002oooo0P0000Koool400000_ooo`800006oooo1P0000Soool2
00001_ooo`800002oooo0P0000Soool200001_ooo`800002oooo0P0000_oool200000_ooo`800006
oooo0P0000coool200003_ooo`80000Xoooo0P0000Koool400002?ooo`80000?oooo0@00007oool0
00Coool200001?ooo`030000oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool00_oo
ob`00002oooo2P0000;oool:00002?ooo`800002oooo1000013oool200000_ooo`@00004oooo0P00
00Coool400001?ooo`800009oooo100000Koool200003?ooo`800008oooo0P00013oool200003?oo
o`@0000Joooo100000;oool200000_ooo`80000?oooo0@00007oool001Goool00`000?ooooooo`02
oooo;00000;oool:00000_ooo`X00008oooo0P0000;oool400004?ooo`800002oooo100000Coool2
00001?ooo`@00004oooo0P0000Woool400001_ooo`80000<oooo0P0000Soool200004?ooo`80000<
oooo100001[oool400000_ooo`800002oooo0P0000ooool100000Oooo`005Oooo`030000oooooooo
00;ooolV00000_oooa000002oooo0P0000;oool800000_ooo`P00002oooo100000Coool600002?oo
o`800002oooo0P0000;oool200001?ooo`@00004oooo0P0000Goool200002_ooo`800004oooo0P00
00Coool200001_ooo`80000:oooo0P0000Koool200005_ooo`800004oooo0P0000koool200002?oo
o`800006oooo0P0000Woool100000Oooo`005Oooo`030000oooooooo00;ooolV00000_oooa000002
oooo0P0000;oool800000_ooo`P00002oooo100000Coool600002?ooo`800002oooo0P0000;oool2
00001?ooo`@00004oooo0P0000Goool200002_ooo`800004oooo0P0000Coool200001_ooo`80000:
oooo0P0000Koool200005_ooo`800004oooo0P0000koool200002?ooo`800006oooo0P0000Woool1
00000Oooo`005Oooo`030000oooooooo00;oool200000_oooah00002oooo600000Coool200000_oo
o`@00004oooo0P0000;oool800000_ooo`800006oooo200000;oool400000_ooo`P00007oooo0P00
00Koool200000_ooo`800002oooo0P0000Coool400000_ooo`@00004oooo0P0000;oool200000_oo
o`H00006oooo0P0000Coool200000_ooo`800002oooo100000Koool200000_ooo`800006oooo1000
00Koool200001?ooo`@00004oooo0P0000;oool400001oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`800002oooo7P0000;ooolH00001?ooo`800002oooo100000Coool200000_ooo`P00002
oooo0P0000Koool800000_ooo`@00002oooo200000Ooool200001_ooo`800002oooo0P0000;oool2
00001?ooo`@00002oooo100000Coool200000_ooo`800002oooo1P0000Koool200001?ooo`800002
oooo0P0000;oool400001_ooo`800002oooo0P0000Koool400001_ooo`800004oooo100000Coool2
00000_ooo`@00007oooo0@00007oool001Goool00`000?ooooooo`02ooooA00000;oool400000_oo
o``00002oooo100000;oool600001?ooo`P00004oooo1P0000;oool2000000?oool0000000001?oo
o`P00002oooo100000;oool400001?ooo`800002oooo0P0000;oool200003_ooo`800002oooo1000
00Soool200001?ooo`P00002oooo0P0000[oool200002?ooo`800002oooo0P0000;oool200001?oo
o`@00005oooo0@00007oool001Goool200000ooood@00002oooo100000;oool<00000_ooo`@00002
oooo1P0000Coool800001?ooo`H00002oooo0P000003oooo0000000000Coool800000_ooo`@00002
oooo100000Coool200000_ooo`800002oooo0P0000koool200000_ooo`@00008oooo0P0000Coool8
00000_ooo`80000:oooo0P0000Soool200000_ooo`800002oooo0P0000Coool400001?ooo`800001
oooo000Eoooo00<0003oooooool00_oooa000002oooo900000CooolT00000_ooo`H00002oooo3P00
00Koool700000_ooo`H00002oooo0P0000Coool400000_ooo`H00002oooo0P0000;oool400000_oo
o`800002oooo1P0000;oool400001?oooa000002oooo0P0000Coool200000_ooo`@00006oooo1P00
00;oool:00000_ooo`H00005oooo0@00007oool001Goool00`000?ooooooo`02oooo400000;ooolT
00001?ooob@00002oooo1P0000;oool>00001_ooo`L00002oooo1P0000;oool200001?ooo`@00002
oooo1P0000;oool200000_ooo`@00002oooo0P0000;oool600000_ooo`@00004oooo400000;oool2
00001?ooo`800002oooo100000Koool600000_ooo`X00002oooo1P0000Goool100000Oooo`005Ooo
o`030000oooooooo00;ooomD00000_ooo`@00002oooo8P0000?oool:00001_ooo`800002oooo1000
00;oool800000_ooo`800004oooo0P0000Coool400001?ooo`800002oooo300000Coool200000_oo
o``00002oooo100000;oool:00000_ooo`@00002oooo200000Goool100000Oooo`005Oooo`030000
oooooooo00;ooomD00000_ooo`@00002oooo8P0000?oool:00001_ooo`800002oooo100000;oool8
00000_ooo`800004oooo0P0000Coool400001?ooo`800002oooo300000Coool200000_ooo``00002
oooo100000;oool:00000_ooo`@00002oooo200000Goool100000Oooo`005Oooo`030000oooooooo
00;ooom>00000_oooa@00002oooo3P0000Coool;00000_ooo`H00002oooo1P0000;oool400000_oo
oaP00004oooo100000Coool800000_ooo`800002oooo3P0000;ooolR00001Oooo`400001oooo000E
oooo00<0003oooooool00_ooodh00002oooo500000;oool>00001?ooo`/00002oooo1P0000;oool6
00000_ooo`@00002oooo600000Coool400001?ooo`P00002oooo0P0000;oool>00000_ooob800005
oooo0@00007oool001Goool00`000?ooooooo`3ooooo1oooo`400001oooo000Eoooo00<0003ooooo
ool0ooooo`Ooool100000Oooo`005Oooo`030000oooooooo00;oool00`000?ooooooo`0Uoooo00<0
003oooooool09Oooo`030000oooooooo02Goool00`000?ooooooo`0Uoooo00<0003oooooool09?oo
o`030000oooooooo02Goool00`000?ooooooo`0Boooo0@00007oool001Goool200000oooo`030000
oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool00`000?ooooooo`06oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo0P00007oool0
01Gooooo00002`00007oool00?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo0000
\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-13.248, -11.0235, 0.501294, 
  0.501294}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    RowBox[{"\[FilledCircle]", " ", 
      StyleBox["Wavelet",
        FontVariations->{"Underline"->True}], 
      StyleBox[" ",
        FontVariations->{"Underline"->True}], 
      StyleBox["Transform",
        FontVariations->{"Underline"->True}]}]], "Input"],

Cell[BoxData[
    \(\(w = waveletMatrix[128, coif30]; \)\)], "Input"],

Cell[BoxData[
    \(\(wa = w . a . Transpose[w]; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[wa, Mesh -> False]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.00751202 0.0192308 0.00751202 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.16947 -0.0125 -6 -9 ]
[.16947 -0.0125 6 0 ]
[.31971 -0.0125 -6 -9 ]
[.31971 -0.0125 6 0 ]
[.46995 -0.0125 -6 -9 ]
[.46995 -0.0125 6 0 ]
[.62019 -0.0125 -6 -9 ]
[.62019 -0.0125 6 0 ]
[.77043 -0.0125 -9 -9 ]
[.77043 -0.0125 9 0 ]
[.92067 -0.0125 -9 -9 ]
[.92067 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .16947 -12 -4.5 ]
[-0.0125 .16947 0 4.5 ]
[-0.0125 .31971 -12 -4.5 ]
[-0.0125 .31971 0 4.5 ]
[-0.0125 .46995 -12 -4.5 ]
[-0.0125 .46995 0 4.5 ]
[-0.0125 .62019 -12 -4.5 ]
[-0.0125 .62019 0 4.5 ]
[-0.0125 .77043 -18 -4.5 ]
[-0.0125 .77043 0 4.5 ]
[-0.0125 .92067 -18 -4.5 ]
[-0.0125 .92067 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.16947 0 m
.16947 .00625 L
s
[(20)] .16947 -0.0125 0 1 Mshowa
.31971 0 m
.31971 .00625 L
s
[(40)] .31971 -0.0125 0 1 Mshowa
.46995 0 m
.46995 .00625 L
s
[(60)] .46995 -0.0125 0 1 Mshowa
.62019 0 m
.62019 .00625 L
s
[(80)] .62019 -0.0125 0 1 Mshowa
.77043 0 m
.77043 .00625 L
s
[(100)] .77043 -0.0125 0 1 Mshowa
.92067 0 m
.92067 .00625 L
s
[(120)] .92067 -0.0125 0 1 Mshowa
.125 Mabswid
.05679 0 m
.05679 .00375 L
s
.09435 0 m
.09435 .00375 L
s
.13191 0 m
.13191 .00375 L
s
.20703 0 m
.20703 .00375 L
s
.24459 0 m
.24459 .00375 L
s
.28215 0 m
.28215 .00375 L
s
.35727 0 m
.35727 .00375 L
s
.39483 0 m
.39483 .00375 L
s
.43239 0 m
.43239 .00375 L
s
.50751 0 m
.50751 .00375 L
s
.54507 0 m
.54507 .00375 L
s
.58263 0 m
.58263 .00375 L
s
.65775 0 m
.65775 .00375 L
s
.69531 0 m
.69531 .00375 L
s
.73287 0 m
.73287 .00375 L
s
.80799 0 m
.80799 .00375 L
s
.84555 0 m
.84555 .00375 L
s
.88311 0 m
.88311 .00375 L
s
.95823 0 m
.95823 .00375 L
s
.99579 0 m
.99579 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .16947 m
.00625 .16947 L
s
[(20)] -0.0125 .16947 1 0 Mshowa
0 .31971 m
.00625 .31971 L
s
[(40)] -0.0125 .31971 1 0 Mshowa
0 .46995 m
.00625 .46995 L
s
[(60)] -0.0125 .46995 1 0 Mshowa
0 .62019 m
.00625 .62019 L
s
[(80)] -0.0125 .62019 1 0 Mshowa
0 .77043 m
.00625 .77043 L
s
[(100)] -0.0125 .77043 1 0 Mshowa
0 .92067 m
.00625 .92067 L
s
[(120)] -0.0125 .92067 1 0 Mshowa
.125 Mabswid
0 .05679 m
.00375 .05679 L
s
0 .09435 m
.00375 .09435 L
s
0 .13191 m
.00375 .13191 L
s
0 .20703 m
.00375 .20703 L
s
0 .24459 m
.00375 .24459 L
s
0 .28215 m
.00375 .28215 L
s
0 .35727 m
.00375 .35727 L
s
0 .39483 m
.00375 .39483 L
s
0 .43239 m
.00375 .43239 L
s
0 .50751 m
.00375 .50751 L
s
0 .54507 m
.00375 .54507 L
s
0 .58263 m
.00375 .58263 L
s
0 .65775 m
.00375 .65775 L
s
0 .69531 m
.00375 .69531 L
s
0 .73287 m
.00375 .73287 L
s
0 .80799 m
.00375 .80799 L
s
0 .84555 m
.00375 .84555 L
s
0 .88311 m
.00375 .88311 L
s
0 .95823 m
.00375 .95823 L
s
0 .99579 m
.00375 .99579 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.16947 .99375 m
.16947 1 L
s
.31971 .99375 m
.31971 1 L
s
.46995 .99375 m
.46995 1 L
s
.62019 .99375 m
.62019 1 L
s
.77043 .99375 m
.77043 1 L
s
.92067 .99375 m
.92067 1 L
s
.125 Mabswid
.05679 .99625 m
.05679 1 L
s
.09435 .99625 m
.09435 1 L
s
.13191 .99625 m
.13191 1 L
s
.20703 .99625 m
.20703 1 L
s
.24459 .99625 m
.24459 1 L
s
.28215 .99625 m
.28215 1 L
s
.35727 .99625 m
.35727 1 L
s
.39483 .99625 m
.39483 1 L
s
.43239 .99625 m
.43239 1 L
s
.50751 .99625 m
.50751 1 L
s
.54507 .99625 m
.54507 1 L
s
.58263 .99625 m
.58263 1 L
s
.65775 .99625 m
.65775 1 L
s
.69531 .99625 m
.69531 1 L
s
.73287 .99625 m
.73287 1 L
s
.80799 .99625 m
.80799 1 L
s
.84555 .99625 m
.84555 1 L
s
.88311 .99625 m
.88311 1 L
s
.95823 .99625 m
.95823 1 L
s
.99579 .99625 m
.99579 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .16947 m
1 .16947 L
s
.99375 .31971 m
1 .31971 L
s
.99375 .46995 m
1 .46995 L
s
.99375 .62019 m
1 .62019 L
s
.99375 .77043 m
1 .77043 L
s
.99375 .92067 m
1 .92067 L
s
.125 Mabswid
.99625 .05679 m
1 .05679 L
s
.99625 .09435 m
1 .09435 L
s
.99625 .13191 m
1 .13191 L
s
.99625 .20703 m
1 .20703 L
s
.99625 .24459 m
1 .24459 L
s
.99625 .28215 m
1 .28215 L
s
.99625 .35727 m
1 .35727 L
s
.99625 .39483 m
1 .39483 L
s
.99625 .43239 m
1 .43239 L
s
.99625 .50751 m
1 .50751 L
s
.99625 .54507 m
1 .54507 L
s
.99625 .58263 m
1 .58263 L
s
.99625 .65775 m
1 .65775 L
s
.99625 .69531 m
1 .69531 L
s
.99625 .73287 m
1 .73287 L
s
.99625 .80799 m
1 .80799 L
s
.99625 .84555 m
1 .84555 L
s
.99625 .88311 m
1 .88311 L
s
.99625 .95823 m
1 .95823 L
s
.99625 .99579 m
1 .99579 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
128 string
128 128 8 [128 0 0 128 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF86787B76818063C6
00FF8995FF00BA61984C54DB81879F6800A76B827A7C7B7C
7D7A7980787D787C7B7B787E7E73B1627D7B7B7C7B7B7A7C
7B7B788849545A9C2E63E97C6A9172819F55B2707C506F9B
707E7B7C7A7D7C7C7C7C7C7C7C7C7B7C7D7B7D7A7C7E7F7C
7C7C7C7A7D7B7C7B
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF86797D7D7B7F69DF
000004868500C75B6ACCA6A3FF1EAED300CD60847A7C7C7C
7C7C7C7B7D7C7D7C7C7C7C7D76847CFF7B7C7C7C7C7C7C7C
7C7C769C99318889491F00B8D96EBC3572007224D933A89C
767C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B7B7B
7C7B7C7C7C7C7B7C
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF827A7D7D7C7D76AD
9F5DFFFFB3009A698C837A65E46354B6009B6B817A7C7C7C
7C7C7B7C7A7A7A7D7C7B7C7D79874CFF7D7C7B7B7C7C7B7C
7C7A7B88D46181821C85C473A3A9705D61E54C6C919900A5
6C817B7C7C7B7C7C7C7C7C7C7C7C7C7C7B7C7B7B7C7C7B7D
7C7D7C7B7B7C7D7C
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7C7C79787A7F7483
44C1234F669B7681883B2A2BCC83B2858D787D7C7D7D7C7B
7A7A7D7A817D827A797F7B7C787B8973787C7D7D7C7B7F7C
7B7E7C7D62977C8E776F5F4836B26576989CCC6C8E8D9476
7D7C7A7C797E7B7C7C7D7B7C7B7C7C7C7D7B7D7E7C7B7E77
7B7A7A7E7F7B797C
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7C7C868C8175877B
8783847C7E757A70728F79818E836D7B767E7D7B7A797B7C
8283787F647B677A8473807A8A777C7B867D79797B7E717D
7E76797E86737C796C817D7B807F7F6C7B7B6B81737A7B7F
7F78827983777D7B7C7A7E7E7D7B7A7B797C78767C7F7A7F
757D817373808278
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF748560F397A35671
7D6D6193767E849D9B6C7665698988767977738080847F7B
67638182908B9A68689B658B549075915F8081838477A876
778B8171757F7D8092887A7C6F79578974868F718D74677B
6E8E6E866A83797F7B7C76737B817E7C817D828978728A0E
00516E7FA86F6E90
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF75AF52FF949DB914
91B98600806A6D0E269059D89E524A788BA69C7F6D9A798D
ADBE63A2FD1D63E16E8C5DB4FF54F10DAEB35C479ECD927B
972E1C6DCA636D3925664392BD76E363A4552FC15660BCC7
B178ABC0944F8474514ABFA57461764737866D4F73DB0052
28B35ABC8E54E5A0
FFFFFFFFFFFFFFFF66B01DCDFFFFFFFFBC1958D61B666A93
FF6000B07FA28FFF9163C1C6B1FF1ED4A0839B828A79C79C
B6C1AC4D8F767856856AAB37429487516388888177CD7D3D
B0203ED393D816A1067B4795763A7FFFA7FFD6FF403EA952
F2451CFF00FF6251518AFEFF54558F6D6D89E9B882983D58
8E768B6D7FA66D4F
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF9DF300A3D127
483B3F76A0CE8B0086A680F01DCB5B8337AE7F327E7E7088
E3FFFFFF334E53FF2EC3549EC7E6439C86A1547BFF96A467
9B727AB357E9FF49AD3D4E57F47B2CA84786954514936CBB
9E49ABA3842D00B19D846A80FFB17EFF86470939C4FF0075
7168B1A2317C6880
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF900093FF456A86FF
0043323601FF59FFC7A2ED3165FB138600FF93FF0D00DFE2
43FFFFD2C501515B00FF31B23809AF00364C438ECAB66C5C
00896F678948FF65C35739758762A06300CD779D90719EBE
7047C8A0B4FF9DAF3600AE93829E3EAA49AE04FF80BE0070
86BBD65D62D8BA6F
FFFFFFFFFFFFFFBAFFFFFFFFFFFFFFFF00B4FFFF794B93E2
A0005E9E00F3E3AB00E983655A5DCC54506A8EFF4C23FF8A
FF00E9DDC300FF8DE0F7FF09BA6261005038377F6972E8C2
79828396B6800098B43E8B49D09E582E794785287761B80B
FE6F75BC2500998865FF2539BA8F36FF9F78AA687CB60063
647F66BF39A3B0D1
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF4A0020006D0400FF
7E006D40838DC900005C6B51ED3C3F0036BE0012D8570085
FF5CAB7E33570000FF8EA794C3578AAD45A06A2DA26DD3FF
758176D072B2FF65938C00C03C7B953E759E2261755A45C2
475C811CCEB05913A7185EA3E618DD8B6087628D64C1FF82
4704FF00F34C53BA
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF63C500FF009200FF
AC6958E5908BFF0E3B9E1ACE2F9131FF53FF210B6F0000E7
8E67FFFFB07EC84E0000CFEA4B6655FF9F00FF267C8E988B
68887C4F00FFB3FF42BE93AD7D536C2EA1006983A7AFACC1
3310A9B9E1FF24A6C417FFEDD9DF4BEAD17C42B29C975974
77BD48D8E719FFFF
FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF8A8BBAFF0061926E
874DB513CBA5C600A28982946CDA7FF300A2381FC577915E
609D7080C414FFA8967AFF107A7256FF417F935FB6A3ADA2
667168FF000BFF993200769AB76C6CD7C37FFFBA289A0048
F731AA93FF2366557562878A717C958342FF59E587D60A5E
5E8F53FF00988AB9
71FFFFFFFFFFFFFFFFFFFFFFFFFFE2348464611FB3767B6D
B0A55595D700B984FFDA8F7D618972BB00597E8A70787683
896750B8409454767D71518A908C7B6A877B76877172608A
6F648984435CCF5A6951798591797B747994BABE3A6C2F9D
40807B7C68907E847F817C79807E757C8B7E885985A1AE7A
777C7F689174797C
77FFFFFFFFFFFFFFFFFFFFFFFFFFB5FF7C8184926A7D7A7C
A2FF0069E5E8657643883CBE8E72B559D0757E777E7D7E79
788185728A75897D7D7F8A7677787296787D7E7A7F7F837A
7F81797A8E935A76C157BB846A92867C73817471B75C876A
8F797D7B82767B797B7A7B7C7B7B7E7B777E79867A746F7C
7C7C7B82767E7B7D
7B6F7E72706C756184BB8F8C7C81747C7E77726D5C7A6E89
8273919880777D8581808E807482777D76797A7C7D767E7A
7B797F6F81A18667787B79766C816B8C75877B817D79777C
7A7F847F7D797B81807E8A807E7A778176817A7B7E7E7276
7975787377817A7C81837A7A7B7D7C82857A81817D759753
8470669C8F756874
7D9792A8B7BD98EB5622709E7964A97F7479A7A56A46BA8A
50C19930978D633E3E9F97A1995867818D7A757C6B4D7A82
9CA250834BFF0068B224857BCF6DA157EB994962776C2786
855B697EA5428D4D6E71828D8AAF9C468B4B32859473AE98
7F46AF55AB487883967B9A69848664837F8B605A597F83C0
CF70707E6B84A666
5C0012001F00002190DCFFA0B5014D0068954FAA9CA8488E
356F1DDF6F9887FFFF66951B2EC2843E8A583DB7795C8E7E
272E4D97A85C78A8746A3CA909E78BF0DC3A40745100B888
61B0692EB2049559F4D698AA28D0007E4F767957FF1A2CCB
2E6E6588166551A2B28BB61C61A25F95C79C93B81D15BCC3
6DAB67809A71BED3
FFB5ABFF83ECF5F8FF082AD200007DFFF8FF4F5E73569B5C
035CFF6E2B3559208539424E853978008CBAE8AA24B95143
A9F69F89389B4062888A587EFF4B8595758166454A9741E1
718C2F62925F65BD68489D737173F4A86C4ACE9CA1916CBE
AA179899245F307D7E8AC7954D1E9A91AF44A479CBA16E6E
816A9FBA64637470
0043000000FF2CFF14CD00530014FF821C087A33FF6A592E
62995323834DBAFC8C6E90006399385000AF747300189B39
3F646DFF7E82EA00FC043D9E9C896ED07A2673253E2E9704
A6A0778E59984D6653C5709CA12F7AAC80DA7D6538881B00
0BE67C869EFF248C9BAF76A0DC77B1602BE14F763848CE5B
9ED03455CCA87E7B
7126A976B0532BBB00FFFF4F948F00A740665D6DFF6A7060
CE7AD7F92C8CAC8E49C5F3E5006FD7421985B1339138946D
89C8CE1E2A38537F4A556ED760746465BD50FE7EC4819C78
6E7A7F639A751197B5623B4C81D895673DBDC3589959F4D7
7D3D006500F27DC4FFFF87467C5A70905374E4CB80B80087
DCAC9DFFE1104357
00A67900F8BAB92CFF00D78DE4F6EB44D4A76200FF00F167
00A7161BA594C9F0CDA4B584FF0007356D7B9E8C35E70A78
FFFC911600D5672D3B5E89977562986359CC6F7E6D7D0089
8A7A7E02445EE5902671006E86598458CE84C2F33E36037F
747B985C3000D56F570091A785836975B35CCB64BA16ED8E
4AF26AAB2F5B33A4
F557C07E6F876D0022E37D8E6B006D586453F9363CFFBD68
D000FF510001577A8639B251FFC8458A458F7484FF5E58C1
FF681BDFC35AB4FF581AFF966BA63999B7AA00117A719337
82788A470013516B2D00787C6AAD916E895E47C9003D91C0
BCB96AAED96F8892007E427960DB3EA271945C8561AFB481
7A9A82E600BABA00
E5004E71FF7AC110B5C7FF9E00C5B302682E66627A808712
475693DBD364C2FF8100F1002C684F653C636FFC1C345BE8
A56A546D408E374FA5001F79BCA54D009EB344FF475300F7
AD8086AA4046CEC261FF4BC7A98949A63CA49048E91F3A00
FF778E6236FF61D28FB36A45AE9C2B69D7675127A47F8574
7198627AE74F61B6
7649B807A7BAD4B000000A00E4ADCEFF8484D5FF006B8F8E
157EDF068128992B47AC7827A5642A8EFFF9556E828F8F6B
6DAAA57FEC3EF77BB987FF294F634FC948B2789D9C94B88A
B7CB469985409C16CFDB796560715EC6844B3462FF708800
FF748262B04270667979626A78658E6356A45EFF7C62007C
827673B947905992
86488191796F6A86D7B895A48D8C73637B777150A77D7981
D1A60B9A006C858D5B7974997968973E7475858474767583
817073B6688E5A7771715C8C7785679B85757C7E74787677
80807C66687695FF8E8C91837982846C717C6F9C74B14B93
6387757C6B8B7C827E7D807D7F80767E886F837981868F7F
807C7E6B8F758179
B1BC6B767B7E827B57636D7175A0FF617B7F828C6A7B7E7C
754B711D007574772BEE9D6E5B89B7583B8F75797F7D7E7A
798080748575897E7E7E89767A798889787D7C7B7F7E807C
7D7F7786535FB39143B48475CB5A7485989F60926AB05683
837B7D7C82767B797B7B7B7C7B7B7E7B788079837A79727B
7B7C7B83757E7B7D
BFB07B7F7C7C797A8784808276C426B07B7B7A76817C7B6C
4800E6F33D6676806F8F9521A59469FF3178797F7A7C7B7C
7D7A7A7F787E787B7B7B787D7F7599047D7B7B7C7B7B7A7C
7B7B7D7572A56C464D8BB284FF82947377C38C57A868008B
6F807A7C7A7E7C7D7C7C7C7C7C7C7B7C7D7B7D797C7D7F7C
7C7C7C7A7E7B7C7B
FF95717D7C7C7D7D797B778368FFFF057B7C7D7E7A7B816D
971E48583F61827BDC9B3B86198370568B897B7B7C7C7C7C
7C7C7C7B7C7B7D7C7C7C7D7B7C7C7E7E7C7C7C7C7C7C7C7C
7C7C7B7D6C8A9677BCEE6585AC9B7671CB79863C918EE274
82797D7C7C7B7C7C7C7C7C7C7C7C7C7C7B7C7C7C7C7B7B7C
7C7C7C7C7B7C7C7C
0049817A7D7C7C7A7A7F7C7E814900007C7C7B7B7C7C7D81
DFFF84B97E6C7D7BA9896E3AFFF464244F7D7A7C7C7C7C7C
7B7B7C7C7D7E7C7B7B7D7B7C798441C27B7C7C7C7C7C7C7C
7C7B8074D5429D9614A7867577A1626366B277722B4B2A8C
767E7B7C7B7C7C7C7C7C7B7C7C7C7C7C7C7C7C7C7C7C7C7A
7B7A7B7D7D7B7B7C
2D96767F7D7F7C7F7C7276728C458B767B7C7E80787A7E7F
59D3493B739E757A794282669A70AF6A94787F7B7B7B7C7C
7D7E7A7D75707B807E797D7B82788567817C7A7A7B7C797C
7C7A7A7B768086817F575F7A738296968670A76F75729579
7F7A7E7C7E7A7C7C7C7B7E7C7C7C7B7B7A7C7B7A7B7D7A7D
81857A81797C807B
907CD04C9D9376B600133979C08FFF678C4F41A92561BC63
AD42FF352FFF7972B4FF0021BC827C71B88EA39588857888
8D664AAFF591A27E9B4BDC471B7B582B659F5FB87B8BBE88
D9E2002E936CAE1A9B399D6E42831DD9258421786E7A57B3
BDC35266825D7E7081837374807E73748596A3FFC2BEB196
B171789670796C80
4A206C97706FA680C4B887984CAB736E76966B68A97F7D5A
230020849D7C6782E68E24C472A7FF10489C877D7E7D7B7A
76848E168F89677D6D87678C8975CF2584707F6D797C7B74
614D7F70A4A4B2A0AA118C898C76836F6FB47009158BCE28
4E68807D767D807B7B7A80817A7E7D817B647C543B12CE78
8079806E817B8372
70438671817E8B7F676A766F9B3B51857C838F826F7B7B85
4F6E8E796B75887672BE7E6D82776545E08F7C7B7B7C7C7D
7984849E8979827B8079857675787B6878807B817D7D817C
848774766D5DFF86FF7F8774767C77827A747072B3494B43
A1797C797E7A7A7C7C7C7A7A7D7B7C7A7A7B778E6B61757B
7F7B7B807B7C7A7E
5168817D7B7B7B7A82817F7D747BB7817C7B787A7F7C7B79
6045506BDB7D7C7CC6B65EA66A8F5B917F747D7C7C7C7C7C
7C7B7B71797C7A7C7B7D797D7F7B86617D7B7C7A7B7B797C
78777E7E7E847D24A3668F809F697C7A7D828277415A877C
757B7C7D7B7D7C7C7C7C7C7C7C7C7C7C7C7C7C757B7B807B
7B7C7C7B7C7C7C7B
D6A2757D7C7C7C7C7B7B7A806E992B1D7B7C7D7C7B7C7C79
780FFF933D7A7A7D8ECF5B6BAE77D0D286807B7C7C7C7C7C
7C7C7C807C7C7C7C7C7C7D7B7B7B816C7B7C7C7C7C7C7D7C
7D7D7B7B7F946C60C14D61569D4C5E777594898C55489174
807C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7E7C7C7B7C
7C7C7C7C7C7C7C7C
58B7777E7B7C7C7C7C7C7A8071975BED7B7C7B7C7C7C7D7A
A3647AF89DA6747FB6DCB939D2BC7183A0727F7B7C7C7C7C
7C7C7C7A7C7C7C7C7C7C7C7C7C7F82D47C7C7C7C7C7C7C7C
7B7C7A845BD8487A00B94F50A96A716C6DFFAD6B94708076
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B7C7C7C7C
7C7C7C7C7C7C7C7C
776D827A7C7C7C7C7C7C7C7C7C8C25007C7C7C7C7C7C7C78
7084DC59AA6B7E7BA7D1794149439579627A7B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7A71417C7C7C7C7C7C7C7C
7C7B7F72BD9B70A52DC2C45D83668CEA22186C8F6E894583
787D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
FFA2767D7C7C7C7C7C7B7E749EADF4FF7B7C7C7C7C7C7A72
34C035743C737B7D6A81FFA5FF995AB477807B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7B7F75B35C7C7C7C7C7C7C7C7C
7B7E768600647219524DEA0095DCA9D4AFCE728CAA6E8979
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
FF3486787D7C7C7C7C7B7F75861BAB4F7D7C7C7C7C7C7986
4496FF81E95A8A786D503277AF2C6B35947F7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7A7E587C7C7C7C7C7C7C7C
7C7C7C7A907CD778949C55649B2371AE7773828C5880B578
7F7A7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
1AAA727F7B7C7C7C7C7C7C7D7D6100F27B7C7C7C7C7C7B76
680246FF43A1737F193470705E6E7C417D787D7B7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7D78924A7C7C7C7C7C7C7C7C
7C7C7A7D577BF0FF9FD5AE7AB3F10084968BDE6D585A6180
7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
00228C787C7C7C7C7C7C7A7E7CB21C407D7C7C7C7C7B7E74
A4D85D15F77B837856AA3677BB00738CA8767E7B7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7B511A7C7C7C7C7C7C7C7C
7C7A8365FF794269577904AD90686ABF3900C0C38B008475
7F7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
7F777D7B7C7C7C7C7C7C7C7B7D6D961B7C7C7C7C7C7C7B7D
746F45765A647D7B3E755B2FAF9A4DF77882797D7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7B878A7C7C7C7C7C7C7C7C
7C7C7B7E6FA2009340507D4D47968C6CFFFFD5ECE17B9176
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
22767C7C7C7C7C7C7C7B7E758F347D947C7C7C7C7C7C7A84
6EFFAF6E7D8B797C731DCD7B7556B6677F7A7D7C7C7C7C7C
7C7C7C7C7C7C7B7C7C7C7C7C7C7C758E7C7C7C7C7C7C7C7C
7C7C7C7B837976877957858B9863C4C0AD9AAF837671787D
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7B7C7B7B7C7C7C
9E7A7D7C7C7C7B7E7F7D7C7D788A919B7C7B7B767A7B7C7B
7C618C8580777D7B79847D866C7076837B7C7C7C7C7C7C7C
7C7C7C7B80807C7D7C7C7C7C7C7C7E7A7C7C7C7C7C7C7B7C
7C7C7C7C797E78798385827D7E767B7B7E78747F817F7D7B
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C80
837B7B807E7B7C7C
727C7B7C7B7C7A868E7A7A797E77787A7C7C7A677B7E7B7A
7B857A797B7D7C7C7D767F7A7D7C80797C7C7C7C7C7D7C7C
7B7B7D7C8D7F847F797E7C7D7B7C7B7C7A7A7D7C7D7D7F7C
7C7C7C7B7C7C7D7D7B787A7D7D7B7F807C7C7F7A7A7A7B7C
7C7D7B7D7B7C7C7C7B7B7C7C7C7C7C7B7B7C7C7C7C7D7491
8D7C7F79747C7C7D
7D7C7E7C7D7C7F779FA27C807B7B7F7A7C7B7F91567F7E80
797D807B7E7D7A79797F7E7F7F7B797E7E7B7C7C7B787C7C
7E7F787D4C44B18C7C737F7A7E7C7B7B8476787B7C7A767C
7C7A7C7D7E797D7A7C7C7F7C7C817C787C7A787D7E7D807C
7C767F777D7A7B7C7E7E7D7B7C7C7B7E7E7C7C7B7B778C5B
709E6AD8877A8177
7E838381798072C6FB37643984837F8281768F66F7B38156
8A7281827B777A80777A7B7D7C7E7F7C7A7D7F747D7D7B7B
7C7D809E956FCDB667868BA17182707D634789829C92AB77
7B7E8080758580877A787C777D7581817C82857D74817C6E
7F7D787B7D887C7A7A7E76817B7A7F7D7D787D807E852DFF
2E50D565B6928D7A
708FB2929782BD39FFA9213A944DCB70895852CF569A596B
52B3E674A88E5061338ABBC3B270718AA5748E72612A8777
9F9F798BB287F032353A36E478A02D7AA9B81F86FDAE8565
6C548B85AF50B570985AB97683CE7F55845F68A4AE9DD150
6F09BB15976E5C7CAFB096737E826FB0B46E858574804C23
BBA5390080554433
EB7F7F778684F27A76DFB5A7A152838DB06B8D37AE137076
874ADCC57254959EBD3C1EDC5E9AC363A583DF926CC67877
6457FFB33D9BA28C7FA400A23986007C51284A84C097E71E
14945F498FFCED80C4307332CE4157A395C4FFE75CA6A261
86D780C31DCF585D48A745CD8E67BF8C73199BD0E3AC5E65
97B25E3A9C409180
DE9064A0C631008B68586D345C7C59539E62597E717F8591
14778BC4707183EB7AFF9E7E12B6C047A9205E6D890956B2
B4CC28C0766D9E6B9281854DB8447265349B9E72ACA79691
0BEA48D64308FF80943A4A8080319724D2982498E300C4E6
FF6D52D480FF997CDB998D9797C034D5AB64007C00D76E8B
777A766D7F7A6994
7FF5FF768FD79A004309C871BDFF739965529233FF512688
5C7762F9417DE5DD5E4C794358C6B1A06E80784AD4788B8C
508F3D00579E228E8E71A04067686E74877573969799652A
9588B186D700395B61C2CA8F906834A2839AF63FB172B5AC
862A2E6558A1D7757D3ABAB9747341C8B084DC6C819DA698
7B7995BA6A538C84
FFE6E3FF4D000003B3009180FFD0E0FF5BFFA528FF456797
8C7734D9FFD6D5A100F37D4DE63EA899B600624E448F373D
838F57AA769F50008F7C7CAAA56752007777566B998D8500
56856D584A94E09E42C3E4D1005A925366C7442AC18AAD67
2E7E52C36561659D897C7C30227A947B943F7CB29D058D89
7F8075A6AD559C80
08400000B500332F0780E022000052923E99B43DD241F10D
9693338A92C792FF5991CF5D4D6F6D59749F5DFF7EC1BDAF
AFB6AC3A1A995A773FA052A385E08EB17D90969751AA9C4F
4F518B7257703987853EA5690B9296852A4E8A5C53A426BE
5A537260787DFFC03C4CA4878F917B1D5A8655CEC165387D
6C75745793728477
A286ACFF008AE4A48BD9B3FC83C6826CFF17598E003F00D8
666C60417A5A572DB1790649877B6A7FB857D60E0341E7CE
91604D72B732F446CD4400C8538F84683EA9E475F2089443
4262897580748D860052EFF68F0F6A93BADC738288557AE2
82AF3A813557CB6686C3F94479439F5FB27B889042027F72
9D9558807B37859F
00703E00B0FFA93F6DC1E56171827B6F7FFF6A539FFFDB44
927DB49E0F56FBC4A178AA83B7C8429AAA2AFF77636EE12E
A1FFD55E7FAF7BC7E435A3AA8E8878920003EEADBF52886B
8B7D7B847488A87076926DA268B4005937A85B866DB11572
7B08BED100966FCA7CB09D00AE5CF13A8564AD37A864B581
3D0000C07F7742A0
80750071A1E7674B7B9F00B191A24ED09928BCE19656597D
7BD585DB6468EA64A5B7A1882EB8A7668A56725526E38986
054E7C7E756430A300CCD67B538F5731516C567D91756D1A
7377854A6100BA75B1536E606857D4155FC0D87D875D6C7E
679167A7BF769800FFF13C505D9CB8815B69B6657767D561
FF56DE7689C8A25C
ACDF6A6D5792C81875776BA65B31859CB13F6A5F8800B16F
DDE9E40EA50282001E72286FA0716DA700B9A7AEFF405EA5
86865B98A8D7FD0159A5B0A25F956279692662648E7A3756
7E7B8190035F505DF085EDBEB0966AB3206DFFA19B710089
C66A3BA4807059FF9F518BE5BE67A2332BF6AD9E4655CB76
209C40950C8F6000
003A5A46E88D3B4600940A855E64E350B93ADC8B183A86AA
91904812D396DBA7B849B9D9A7CC87064381234D3863141E
22DF39B91167C13BBE10889B7A6878521D955586798135B2
728277C4FF74C15612DBA16CD36ADB765B085C5C373B088F
92008946FA70FFD6453D1C080C72D158682C7B73D1624E7D
6E83BF29B060B338
BB7528A36ABF76E6FF1982549E9AD65239BC8AB78E6F8DAA
22E9849C9948B75C19FFD78A1E67007B2EFF756F70859800
5BC47D669B64545566AE8C8396459E2C87A31B5F818064DB
7281795F90A5484F2483CB45713897008A568CD0CE7B5928
1634957400FFEF7E74C6DABECBE2A383729B7A64317E90A4
B6BB07970A8EA662
8C6C78BD97FF2DABFFA170756484EAD540BC5700EF84727A
8D0956675F7A4B81FFCC1949896DE9BD996DFF70D8907FD9
85958D9828A29F46FF806EA3827D61956357808C625F7690
807C81513188A7FF81C76668A3BC6D912D7E7393166CC399
D08B61FF555F43DE56D650AEE563033D46D61B609E91868A
7BA792322033B7A1
00C82D1C727DA75A00AFC38F74D6BDD7075C9FAD63E0C8B8
794A77736EE6930C00FBFFC324BB6D047BAB6362000199FF
4541B73BA55F9B4F9C5E433A864FC03A94A2CA5B982FB22C
438A779C3D798F6853CD4ED1082AC892615FC661D93A54AA
77618496FF739EFF52835073BD5DCB58423B6E9161305D7D
81677895B07C4386
9F427AA8193F17790D22D6CF7B552462AFA958A16C65CBC6
0036A56AAF00EE93B1D475780CA26F2A2DAE7144EB9A6484
8CB946B28474A4D2849F477A94488AD5D81B4927531D00AD
49EA5A770073C18BA16778ACA5E22EBA8680C10086FF477C
577F75EDBE53ED557775B8D05D9C9ACE74EBB0B090895D7A
7C7478933663926B
61806F7562773B6A62EF5E6E6C618A567880FFC000CC71A9
6B4E8B8E7FDBC84C4275748552C5788E112F7D65675CC87B
F24B5B44A5BA707D9D32FF8C8E618E8868AA44FFBF6E80B0
7480788587A0248B68857980885D00D44AB390757C7FA26E
F7B549B6A581834AB68A8C468F8B619A1872AAA43B836283
67117CFFD08EFF61
4DAF9142A9F4748D4A43636A6F727E88154F9291FFA39E93
ABBC1FD861009DC87AA6A434A7295F519DC6997A56BA5790
2DB9BE9D4EA0FAD3E78E635E8B6A60568059A8EF638966B0
797D825D524A797AA5618A7B2506547D6D5D2C917E604DC5
8A6E55846B70824A63ADFF561974956A749D0E4884754481
871954271B243E81
6441B3A3D7685A757766774D6351589D856A4B813A788889
AD4F51496397929992B7358D7664B47EA85EFF5C89CA3463
837669358D6E5C345DE4A84D9576577200740097817AB694
7C7A835692B170931223859A6F9283734593AA5585672186
00E081C0CF8E7285B283149D7586652FB7447D7E9497B776
98771C2657CB4ADC
9451304B5BFF937AB6A5AF81664D8173846F5193E57C2DBE
93B6B4D840CC7DC94A858F647E3EC1AF633F408F544FE6FF
CA73653E4230649D4E7731D0825061AD5100E42F7A7F46F1
71817E4A68648C5D80000057BC069450BD8E50B2B3691D7A
A14B6770AE7A8E8E7CA1C49E9EAB3D4FFFCF6974AD7A3E80
9122B88FC3656A69
4889308369FF93F65043007B7744BE91583E8F702D4184B2
93FE8D4F5481656B408A6CA48BC390427B804D9F7F7A764F
49B27BCAAD6D7E59A04BC69A7D644F5DCFB04C4C887C47AC
767B881F00877C4F47B94469D5856C94478F6D4996009C79
C398588FB27478DD6068834082B01FAF90787E882EC19355
ED5D00686985E094
5F64336C1852AE6CC9682B93878E5558FF6B592C10DF574C
95D1B667A488A28D200025495DAE4B5BAFC53B5B5FFE8A5E
8C50963371776000950054576EA95454D2FF45867976C214
8873865283944E2FAD827FE21B5DDB8879FFA0AE57AACC6F
4F9690567984DB872E1D4AB0D0865147A89C747A45624B00
8D64659FDE2AC7AD
538A6D9298A9B284782849ADB146E74E8F2C6A5A969C664F
008A31886FB240455EA81EFFEAAF66989881988C998E782E
95827D9D9AF8522E183923775A9CAF3C252425A9707E7014
83787A8539EC55D883BECF49626D147D7DDB003B6B877BB5
67567B89B54F10A2FE71749D7A92754812717994A97877C9
965860370D5E14EA
7F6A3C3191983F8688DC54696865A03E7394FF839C6035A7
958B77EE6C98B6C0C97C7E68895300A28A916F868A594569
FF7E89A99B82349738B0498E89697AD66EED8B83787D7EBB
777F7980585E56F60091847A838C5D826394211D7E9286B4
617081A17BD17340FFEA48237BCDFFFF679266E58448746F
5E4BFFAF856B86CA
FFEA688C305B21FF005AAB8883AE4AA86495A7FF3974AA5A
9489B16C8446B000947AA7CFDDABD780CCBC6C66B2946C9F
BB005453E25751675B228D6E718E7B963D636D7E7E7F7C4E
807A7C84889900FB3D24B49E5694B67E2AB2B6AFE009739E
A7668298928576B0B46746FF6E537F008F1ECAC498A13476
877F4D615CB5725E
DE7C3CD36D26FF32EB645A79AF95BB8F6DB8AA5537A57794
31BFE000C5AD9080A92FC362D54664C896B3A2B4B4D05B49
12092BF02C9043915F93827B80707DFFFE7D8879796B8074
7B7B7A3C6D83FFCEFFB7FFFF445C75BA8CA162831BA1BF79
8F597DC69CB0B278ED977E736957205F43DE6D3C826B0B88
717E6A784BC6FF67
DEE4AE7F003B84000067849C70202E55337DFFB0619C387F
8086889EF161A400CFD1044DA75BB98C4639639C55514B6A
4FB1F5878430FF266292563E668961654C56979AAAF1678A
84788076AA7790665A8100909BA2614FACF3A28176322604
FFBA7450A5A78E6F6D83A5A22129437000738726FF820081
7C217AA39CAA5687
3A6698EEC57D26A4A000715449E0FF7742B90D953699C58C
DC3AE24B778C00447CCD8263ED6F513D829EB9A82800C13D
41FF3A61678275707B898A7A536352A3868A954561A67861
5489805A848A1CB05418515C120B5964323E9E4C84C07A9F
355878848A3F537DC50F36A25DFF00B3CEB20E1CE3EC7C80
7A837B79774629DD
95A16505C7872E8380672F9D859AAA61F1FF7D8D5DB77D65
7B845DBA4054958F931880584B1A25955393723C6AC235EB
4459574581738150B84B8F486E9070450EC1839D00C43F92
97618384CE62865810B66F7C9486BD60FF00848500956B00
D78B7F985CAEB2836F999EF05BE46E703D83B8FF4C6E717C
7A777E769CAB4700
784262D59E775E72617E65D3D4A487FF6C86777764728ABB
8A866C3EC285FF865B6BB6091534FF0071892F7B1F678071
6F76539C8E7D834F9FE359B9DD7A4E91759748917D8A90AE
12FDA100DE8D53596A8470C18EF6F874D07BD79D61AD4992
8F818442F1FF48BC6C827068806F795963BC6F7DBA8E827F
837D7C7B949A6CB7
A36CA4BA6393467DFFFF76605B6E1A1D7F76F47AD56CA7FB
B1B6A488FF500086711F0FD3AE00DE145C1C9FD578887682
7A83861C6269527D6780B0A5F587E780807D569469B57EC9
5F5BF4B966765AA66572AF79AA6920AB5C684AAFA99E8D70
37638763875D90737D7D818479817D8480536E6DA8007B63
5D8081608A77835B
B37386727F788FFF69938B5BA785B6597C4D41FF6B934FBC
EC68DA8C9E189C764C2F8E35774D557DAE00647776787E78
816BDB337F50807479816C89AD89B0A47B7B85757C815B85
84CC6A195E62501D822084E750869062A863A6928D7585A2
409F8086758476807B7B7A797D7A7C797981585C596F665C
5B847A78817E7C7D
738B96768373FF53867A778192A3328972DDE25E88768666
5ACF56878777757F13666288809FD1A191A280807E7D7A82
66BD9A5CB2847C7D7C7A8278716D75007B7C797E7B7D8679
817C680025B6E2D56FC7FF82797B728667A56D70CA746FA4
987A78797C787E7A7C7C7C7D7C7C7C7D7451627DBD42C779
89787D7C7B7B7C7A
AB6E727C7B7D6D82777A7C7C77B3A46F7D727783787D7A7B
988D898A3C6D847BA2BCA6737B6D55816B6D787A7C7B7C7B
7D76684D63787D7C7C7C7A7C7E7C58327C7C7C7B7C7B787D
7C7D8B6896BE125FAD968E727F7C7F79846C96C1A168C9C8
777F7C7D7C7D7B7C7C7C7C7C7C7C7C7C7C80806CE4F45981
767E7B7C7C7C7C7D
738D787E7C7B86807D7D7D7983587D7C7B807D7D7D7C7B7C
47567F75969A787DBD7A6C807C818BA2BA7C807B7C7C7C7C
7B7F8178837C7C7C7C7C7C7C7C798C617C7C7C7C7C7C7D7C
7B7A797B7D7D8C0041757B7E7B7C7B7D7981746C6BA74946
86787D7B7C7B7C7C7C7C7C7C7C7C7C7C7B797A7D6C64847B
7D7B7C7B7C7C7C7B
7866827A7C7C787C7B7B7C7C7885A3757D797A7C7B7C7C7B
6D3F887B8B677F7AB28E757D7B7D72915D7B7A7C7C7C7C7C
7C7A7A7B797C7C7C7C7C7C7C7C7B7D657C7C7C7C7C7C7C7C
7C7C7D7C7F7D9693607A7B7C7C7C7C7C7D7A7F817AAE808E
777E7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7D7D7B8286797C
7B7C7C7C7C7C7C7C
8F8A787D7C7C7D7B7C7B7E77883D75797B7D7D7C7C7C7A80
4C7B807A51797A7DDA8980787E78A6B27D807B7C7C7C7C7C
7C7C7D7D7D7C7C7C7C7C7C7C7C7A836A7C7C7C7C7C7C7C7C
7C7C7C7B7A8095A5AD737E7B787E7C7C7B7E7297504F7F74
7E7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B7B7C79787D7C
7C7C7C7C7C7C7C7C
60757D7C7C7C7B7C7C7C7C7A81695C807C7C7C7C7C7C7B7E
81B38F7098857C7C5D4A7F7B8086A54E7E797D7C7C7C7C7C
7C7C7C7B7C7C7C7C7C7C7C7C7C7C798A7C7C7C7C7C7C7C7C
7C7C7C7C7C7D789D4F87767B87767A7C7A8275E76E8B797F
7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7D7D7C7C
7C7C7C7C7C7C7C7C
837A7C7C7C7C7C7C7C7C7A806FAF73667C7C7C7C7C7C7D76
84208C806D757C7C91BCA93F79727B9E7A7E7B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7B7E6E7C7C7C7C7C7C7C7C
7C7C7C7B7D777D5FAA7A677C2A957D7F7D73728DCE757D7A
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
8769817A7C7C7C7C7C7C7C7B7E7AAE817C7C7C7C7C7C7C7D
87A85676AA747F7B476C41FF65A3879C897B7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7D7A877C7C7C7C7C7C7C7C
7C7C7C7C7B8395A35F7199809D00427E73A79B77615D8978
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
B64582797D7C7C7C7C7D79836EFB00BF7D7C7C7C7C7B7F6F
B982DFE07A528379C76DA27CA372BB6F5483797D7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7E6B877C7C7C7C7C7C7C7C
7C7C7D7A7B2A6864ABFA3EC9687F69777C649D73828D6184
797D7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
81797F7B7C7C7C7C7C7B7D79804DB7007C7C7C7C7C7C7B80
5E460066A6747D7B7F430000A7779178767B7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7B85777C7C7C7C7C7C7C7C
7C7C7B7D7DB47D8457E595B59E33745185357E7A7B7F6C7E
7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
446D777D7C7C7C7C7C7C7B7D7C9D53FF7C7C7C7C7C7C7C77
817FBBA323827B7C67513D00C25D477183827B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7A78557C7C7C7C7C7C7C7C
7C7B7D7698957675BB1CDBC458845A58B15C46877A7AAD78
7E7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
A8757D7C7C7C7C7C7C7B7D7A7D3983E67C7C7C7C7C7C7A82
6E68563572797D7B62DE454BA4FF82817C7D7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C8BA37C7C7C7C7C7C7C7C
7C7C7A8264DD708B1926005E965A1A9C54069577807A827B
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
7E707E7B7C7C7C7C7C7C7D788792FF987C7C7C7C7C7C7B7B
5990672F8C7A7D7B7B1DAA7C8D7175767D7B7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7D7A8A6B7C7C7C7C7C7C7C7C
7C7C7B7D5B77806B128FC8FF56737A9559717776767E7B7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
9C8A797C7C7C7C7C7C7B7E74923F4E367C7C7C7C7C7C797E
3FAAA4E67B787C7CEB985FB8A3D591757C7D7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7D7893787C7C7C7C7C7C7C7C
7C7C7A804B93447C8DDA6910FFE184FFAE7285E28D7A807C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
FF627E7A7C7C7C7C7C7C7A806EB1FFCD7C7C7C7C7C7B7E7D
A792B9B39561847A3564E423994B6D6488817B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7F6A967C7C7C7C7C7C7C7C
7C7B7D7A9C72647F884B6AAA431393203F93758E7D79AF78
7F7A7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
53A2747E7B7C7C7C7C7D78894EBF8D2A7B7C7C7C7C7B807C
DC01AF8C3284767E388646624BAB96B1707E7B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C79824F977C7C7C7C7C7C7C7C
7C7A8075E96F85B456428E5C9A89AE3AFA688E5D9F6F717C
7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
AE6D7F7B7C7C7C7C7C7C7B7D7FAC82BB7C7C7C7C7C7C7E79
B1FF768897727F7B4FC579618475696E7D7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7F71A27C7C7C7C7C7C7C7C
7C7C7C7D6F638B7FD1D266788A903C90358B3F8A708E817C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
8A8F797D7C7C7C7C7C7C7B7E755F00EA7C7C7C7C7C7C7C7F
9D85BFDB818D7A7D5A3AD880F26F69648E787D7B7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7F74B07C7C7C7C7C7C7C7C
7C7C7B806D5296F4CF2E583BA26C91754474487F7B38797A
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
258A7C7D7B7C7C7C7C7C7A80729776FF7C7C7C7C7C7B7E7B
BAA22A54909E777D217156734D4EA9C1AB757E7B7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7F638D7C7C7C7C7C7C7C7C
7C7B7F74C7CD25317E2090846D169158AC8F67A0B049A66F
807B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
AC9B777D7C7C7C7C7C7C7C7E6D43B77F7B7C7C7C7C7C7B84
7A00398669827A7D006AB6B9651B4485847C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7B7D8FB17C7C7C7C7C7C7C7C
7C7D78874A9C90C76B7CA4ABB370905D3E44D2FD224D8379
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
2A258C787D7C7C7C7C7B7E77843898E77D7C7C7C7C7C7982
4A78C600CA6E84785DC7A66696E747938D7A7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C79834B7C7C7C7C7C7C7C7C
7C7C7C7891917F757AB5A87180746D55A202008981538578
7D7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
5BA5727E7C7C7C7C7C7C7A816CC3C3007B7C7C7C7C7B7E75
830B6ED42C7E777E6F8A5864ABA9007D707F7B7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7B7D5E7C7C7C7C7C7C7C7C
7C7C7C7B7B66F5A52A677C69678A959A8E735E0E4C8D777D
7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
87747E7B7C7C7C7C7C7B7D79835F80C57C7C7C7C7C7C7B80
74CDFFD19B7B7D7B673D5B62D600B77D817A7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7A897C7C7C7C7C7C7C7C
7C7C7C7C7C806A8D81FFAE383561898BFE8FB78D8B767E7B
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
877E7C7C7C7C7C7C7C7C7B7D788A5A5A7C7C7C7C7C7C7C79
825D69827F7A7C7C859E4B6E9F8973797C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C777C7C7C7C7C7C7C7C
7C7C7C7C7C7A82797CA67969618C53576C6D6C75777D7B7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C
7A7B7C7C7C7C7C7B7B7C7C7B7D77858A7C7C7C7D7C7C7C7D
7A86867B7D7C7C7C797189807474817C7C7C7C7C7C7C7C7C
7C7C7C7C7B7B7C7C7C7C7C7C7C7C7C7D7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7A7C7D767E7F82768786827F817E7D7B7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B
7B7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7D7D7C7C7C7B7D78767C7C7C7A7B7C7C7C
7D79777B7B7C7C7C7D7F777A7F807A7C7C7C7C7C7C7C7C7C
7C7C7C7C7E7D7C7C7C7C7C7C7C7C7C7B7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7B7D7B7B7A7E78787A7B7A7B7B7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B7E
7E7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7A787B7C7C7C7B7D7E7C7C7C807D7C7C7C
7C7D7D7C7C7C7C7C7C7B7D7C7B7B7C7C7C7C7C7C7C7C7C7C
7C7C7C7C797B7A7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7B7D7D7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7D78
777B7C7A7B7C7C7C
7C7C7C7C7B7E7599A07A7D7C7C7C7C7B7C79793E77797D7B
7C7C7B7C7C7C7C7C7C7C7B7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7B7E77ADA57C837A7C7C7C7C7C7C7C7C7B7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B7F70B2
B9747E837F7B7C7C
7C7C7C7C7C7C7D7552707E7C7C7C7C7C7C7C7F8B8E7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7B7D7AA15B657F7B7C7B7C7C7C7C7C7D7C7C7B7C7B7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7D7A
636286586482797C
7C7C7C7C7B7D7B80659475797C7C7C7C7C7A89826092787B
7C7C7C7C7C7C7C7C7B7C7C7C7C7C7C7C7C7C7C7C7C7B7C7C
7C7D7A837CD0924C757A7E7D7C7C7B7C79777C7C7E7E7F7C
7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7B
7C7B7C7B7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7199
3C3D9765008E767E
7C7A797A7C78844339A17B8E797C797B7A7F6779026F7785
7A7B787C7B7B7D7D7E7B797A7A7C7C7B7B7D7B7C7C7E7C7C
7B798068BAE52DB1748279767D7A7F7C83867B7C7174717D
7B7D7A7A7C7C7B7B7C7D7A7C7C7A7A7C7C7C7C7B7C7A7A7E
7B7F7B7F7B7B7C7C7A7A7C7C7C7C7C7A7A7C7C7B7C7F9373
A84D6924906F7D7F
7B818680817A8D3124B572348678877E7D7867A25DEF875C
827E8B7D80807879757E8583837B7B8080787E7C7A727C7B
82807D6F90833CBC7A639C877A856F7C5A048576B5A5AF78
7D7881807E7B817C7B77847A7D8481797D7B7C7E7C828476
7C6E7F6F7F7C7B7B82837B7C7C7C7B83837A7D7C7D7BAA99
9B6DABA263B9B056
7A64506C687754BA24C4706763824B6E6F98930056944A6C
8A6339736B618B8C8F72525F5E7F7E666691717482A87980
62647B746700AF1FA2C56C1C7C7FAC7B004E9866DCE9AA81
80846666767D6B7E7D94588776586285797D7A747D5D588D
78B76EB6757D7D825F5C80797C7B805D5C8678797A768C9C
8EB29D006D36AC0F
81826A955E78779A555E49CD4867433F6D9F9083A8555663
A06B535B80186AA72E55369050867B4852BE721879A87E80
55578C825DFF7F008FCE60FF5985A575B8594BBCC94E1D75
63843D34C368F8986E9C557AAE814B827C7E888B905E6B1E
73B282AEA2BB6188595F887D7D778A615F7B7B88877899AF
006D37006500E11E
B3CAC896B46E3C6BADAC55857A456B6C7D7B75A05A29988C
F1AC9F5463C5C8ACB0598E24228BB0E53F4946B8B4886770
B1B24262896E6B4A35D03E837049FF7B99FFE1B00000ACC1
E1F09F66C780CE35008388A3E2FFD8BB5C9C750007741784
CEA3096C9D97C36E89801B757588586C709B80515A777674
9095EF50920A5494
A04E7369EA25BA79808087B9193CB17BB8C382617344C6F7
17317AA3A86FE1CE3C8B7B3D4984FFC844FFADA250D29E8B
B6FF97925C804D7A982B7CDF927D7029627E548E1F7D8149
72A2FF7B597369ADB20081DA42BD6972B43C65D69717DE08
FDB0C8A7B9CC078462C64EB86A735B6A4236EEE47BBA717C
746B94A86F907D6D
69A9FBBF9BAFDB776F6A8A293163E255EA908FCF688F2D48
00001463B2F17700BD7FB6C25ECC2E5F407930685163D2C5
00435B44D34D907EAD4E7900CD9C612C347F7B75AD8AB9FF
1C66FC8B4A6E555540DE0005D87EF842AC0F58594D6C688F
00004E895101CEE1CA288A5A0034759865A364FF4679BED4
7C847B747E7B70E9
7974FFCAC438D38B43686135D48D6077FFA56A4CA792BC3B
70B9C52A7DB228557B4E6FFF135FA6E2DB475F2971D5AE98
B8004C8E7D8BC27E79835EF7988AA74E8E7A7984A032787F
B93A7183D04B9B74DA988E87B1B5A2AD118128FF223E4000
51ADCA85BB2FDA795853BD107CE7176B554F9BD053238973
7A85464C86748170
B95FD0006769B2F8A3CF64953A8F233D483445B60A928451
5DE3FFEE7071AD7B7C63246D1159004D747400007B9D4662
50AD7DA78F619C76717C77665C57A2BC757D7F75757C9BA5
6387DB9C8D53B43C40766C30A078625C4C6E25AB5CA86FB7
D94D7E787CABA48A5F0090A3814403B6A07881B80C7E647D
7C7C7D6384967384
01628B3C7D5471C7A2AE7A9A651F0A10887D82A51C89625F
4EF2A4FF0A4F9C576C508602827D698A219C19DBC1877F7B
8364A8AD9D669E5D6E7D9298551E65BE7D7D7498A172A0B5
8EB16445B82EBB54B59CFFD3A262F03E8322BA6B7CBDB461
1331FFB3AE0DFF648588716C768C8F75727A79A1754A6B75
7D7C76618E907A7C
00C2FF757E2C6FEB84CC8B27FF6C9DF4897414A74AAB2BA1
5CD07E67A400A7864FC394004A3058A7005B08D1B5766759
75B85090727290D68075A5EDE176FFEA7A7F75AC6242447D
C680467366875A406E4785B48776BA50616BC10089C1246D
912C0AAF2E39A2668172836E530F7C1A8480345D72CA6C7A
7B7C7A705C916679
FFBE4C598139A100A5D784001A91BD9F67DDC28676A31A2B
8D00F15F925E6979985360E256CC0FC8799A896E46906C60
8E1CC58FDE51A6C57E738250D17A636E7B7D776A73D78283
4D7F7495FF51AA67968F2B5DA4D62B4AFF55A6654C37CDFF
817C99798978D39F777A84646F919B68B63777B19A4C26AC
807F6F66659B6479
7E40322098306E0E00AA045A56299078749956D6007B7077
E68A9B7FAFDF837773846279916E6C5D522B46BA930E7255
9C7092378E4C999C7860DBA95C99BE528074D0FD6C82CBBE
5B54BB5E77126F837B838183A5638E7B6C827B566A8B8674
D3BC0628739B43774FA5667C7471A03B5F877AAA628A2E7B
7A788E5A786F807B
78BB7E757137004099D5622F809B869E76D466517900AAB3
6B865D88A5A73D948AB5C234648176B43CD59595DBD48C85
82C4FD6D4A7F7C756E545D92927D7C7F7A7C7191FF6E72BF
58677D7F85838181538DEE668DDC6280827D7C887D781761
CF539185656EC24E6DB5A86E99D051DDB34152628AC25C8A
797C7D787770917E
AF544E830DDC9FAC5CBFBBD08F689F958F54448F4C1AA118
6886572A837BCE0C88B49E507F78795F607D6BDD28CB8E68
CD8E8571807CB36FD86D7452A5FF7085BCF3511721537A4E
A6438F757A7774892688FF6FA0FF67B2CA8978797D7DBB36
5D7E7A7400767A793C167E8235CB1CAA78A67D9414A2857A
7A8178C55D99C59E
6DE2462D547CA6B088A4475E727C678F4E3FAAA36B017FBA
829E8A5F87A3B72263876D41AAFF7678CB8B618AFA75959D
CAD729CAC17078C4885AAF094B46807EFF5887826AFF696B
5A90757E7C7F8375C57C7EC4792E49BE700594737E43F723
8C6D79005F814B905A4F7F51949C00E938EB0C8B5D6FC769
836883CC4A9465A6
7ED5468149DB814AB673327C867C6C8D71FF847700959A64
7A68AD8AD2BEE78F7B827E76658185DF8E59A3334D93BD39
9AC08FCD7BB5EA623BF661809E947B7A71154F38A8637D8D
93737F7B7D7A7180647D84BB7981558B6B40837D6DBF44CA
6F5F5271965B8E41E56624A0298CA0775977668C373D7482
58929E5F47819380
4C9E808D3A8999B7803FB68E9DA87E4EA0001BA3A457B62B
A6AD526A592DBA86B873845D73BA6F698E6B84868831955F
81C0AA6A9F316D45838B835AADB36BAD666D98174E8F6CAC
B86E807B6EA8C96A8C777B6B81490E69618F827B7D756D00
A263BCA97C009151FF9829AB67125B52C2B6FB84800FCA65
B9571E31617947D8
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo003ooooo8Oooo`006Oooo`030000
oooooooo02;oool300000oooo`030000oooooooo023oool300000_ooo`030000oooooooo01ooool2
00001?ooo`030000oooooooo023oool00`000?ooooooo`02oooo00<0003oooooool06oooo`D00002
oooo00<0003oooooool00_ooo`030000oooooooo01Woool5000000Coool0000000000003oooo00<0
003oooooool03oooo`006?ooo`030000oooo000002?oool00`000?ooooooo`02oooo00<0003oool0
00008_ooo`060000oooooooo0000oooo00008?ooo`030000oooo000000;oool00`000?ooo`00000P
oooo00<0003oool000000_ooo`030000oooo000001koool01`000?ooooooooooo`000?ooo`000002
oooo00<0003oool000007?ooo`050000oooooooooooo000000Coool00`000?ooo`00000@oooo000H
oooo00<0003oool000009?ooo`070000oooooooooooo0000oooo0000023oool4000000Coool0003o
ool0000Poooo00<0003oool000000_ooo`030000oooo0000023oool00`000?ooo`000002oooo00<0
003oool000007_ooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`00000Loooo00<0
003oooooool00_ooo`070000oooooooooooo0000oooo0000013oool001Soool00`000?ooo`00000T
oooo00L0003oooooooooool0003oool000008?ooo`030000oooo000000;oool00`000?ooo`00000P
oooo0P0000?oool00`000?ooo`00000Qoooo00L0003oooooooooool0003oool000007_ooo`070000
oooooooooooo0000oooo000000;oool00`000?ooo`00000Loooo00<0003oooooool00_ooo`070000
oooooooooooo0000oooo0000013oool001Soool00`000?ooo`00000Soooo00<0003oool000000_oo
o`030000oooo0000027oool200000_ooo`030000oooo0000023oool00`000?ooooooo`02oooo00<0
003oool000008?ooo`030000oooo000000;oool00`000?ooo`00000Noooo00L0003oooooooooool0
003oool000000_ooo`030000oooo000001coool01`000?ooooooooooo`000?ooo`000002oooo00<0
003oool000004?ooo`006Oooo`030000oooooooo02?oool00`000?ooooooo`02oooo00<0003ooooo
ool08Oooo`050000oooooooooooo000002;oool200000oooo`030000oooooooo023oool00`000?oo
ooooo`02oooo00<0003oooooool07?ooo`800004oooo00<0003oooooool00_ooo`030000oooooooo
01[oool200001?ooo`030000oooooooo00;oool00`000?ooooooo`0?oooo003ooooo8Oooo`00oooo
ob7oool00?oooolQoooo003ooooo8Oooo`00ooooob7oool001Gooooo00002`00007oool001Goool0
0`000?ooooooo`02oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Koool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool200000Oooo`005Oooo`030000oooooooo00;oool00`000?ooooooo`0Uoooo00<0
003oooooool09Oooo`030000oooooooo02Goool00`000?ooooooo`0Uoooo00<0003oooooool09?oo
o`030000oooooooo02Goool00`000?ooooooo`0Boooo0@00007oool000ooool00`000?ooooooo`03
oooo00<0003oooooool0ooooo`Ooool100000Oooo`003_ooo`030000oooo000000Coool00`000?oo
ooooo`3ooooo1oooo`400001oooo000>oooo00<0003oool000001?ooo`80003ooooo1oooo`800001
oooo000>oooo00<0003oool000001?ooo`030000oooooooo00;oool2Ogmo7_ooo`:6QXH2N7Qh0W]k
N`9fMWH2PH610X20P09SHf<2a/K60P0000;oool2RHV90YFEU@;oool200000[Zj^P9QHF42V9RH0Ta<
C09DE5@2fm_K0X61P@:7QhL2WinO0VQXJ0800002YjNW0V][J`:2PX82NWYj0WalO09kNg/2O7al0Wem
O@9jNWX2NGUi0X20P09hN7P2OGem0WQhN09lO7`4Ng]k0WQhN0AnOWh2Lg=c0[6a/@9RHV800gemOG]k
Ng]kN`02Ng]k0WalO0AkNg/2NWYj0WalO0AkNg/2N7Qh0XR8R099BDT2E5AD0UYJFP:LW9`2;Rh^0V=S
H`;YjNT2O7al0VYZJP:ATI42LW9b0X61P@:OWil2EEEE0[:b/P9`L702O7al0U1@D09_Kfl2Vi^K0W1`
L09nOWh2Ng]k0WalO09jNWX2OGem47alO09kNg/2O7al0WemO@9kNg/2OGem0WYjNP9lO7`2OWin0Wmo
O`QlO7`2NWYj0WemO@9kNg/2O7al0W]kN`Goool100000Oooo`003_ooo`030000oooo000000Coool0
0`000?ooooooo`02oooo0WmoOakoool2QXJ60WQhN09kNg/2MWIf0X61P@:0P802Hf=S0/K6aP800002
oooo0XV9R@:EUID2oooo0P0000:j^[X2HF5Q0YRHV09<C4`2E5AD0]_Kf`:1PH42QhN70YnOW`9XJ6P2
00000ZNWY`9[Jf/2PX:20WYjNP9lO7`2Ng]k0WalO09mOGd2NWYj0WUiN@:0P802N7Qh0WemO@9hN7P2
O7al17]kN`9hN7P4OWin0W=cL`:a/K42HV9R00=mOGekNg]kNg/00W]kN`9lO7`4Ng]k0WYjNP9lO7`4
Ng]k0WQhN0:8R8P2BDU90UADE09JFUX2W9bL0Rh^;P9SHf<2jNWY0WalO09ZJVX2TI6A0W9bLP:1PH42
WinO0UEEE@:b/[82L71`0WalO09@D502Kfm_0Y^KV`9`L702OWin0W]kN`9lO7`2NWYj0WemOA1lO7`2
Ng]k0WalO09mOGd2Ng]k0WemO@9jNWX2O7al0WinOP9oOgl8O7al0WYjNP9mOGd2Ng]k0WalO09kNg/5
oooo0@00007oool000ooool00`000?ooooooo`03oooo00<0003oooooool08_ooo`:6QXH2NGUi17em
O@9kNg/2Ogmo0VUYJ@;Ogml400000P@410:6QXH2QHF50P0000;7alL2Fe]K0VYZJP;<c<`2YZJV0Z>S
X`;oool27QhN0Zj^[P;Cdm<200000/g=c@9PH602Q8B40WYjNPalO7`2Ng]k0WemO@9lO7`2OGem27al
O09mOGd2MWIf0XB4Q09lO7`2oooo00=kNg]lO7alO7`047alO09fMWH2W9bL0YVIV@8a<C42R8R80XV9
R@99BDT27alO0P0000:h^;P2fMWI0Vi^KP:l_;`2=CDe0W9bLP800002LW9b0R@T90;IfMT2<c<c0ZRX
Z0:LW9`2MWIf:7alO0IkNg/2O7al0W]kN`QlO7`2Ng]k0WalO0Goool100000Oooo`005Oooo`030000
oooooooo02;oool2QXJ60WUiN@AmOGd2Ng]k0WmoO`9YJFT2gmoO1000008410@2QXJ60XF5Q@800002
alO70U]KF`9ZJVX2c<c<0ZJVYP:SXj<2oooo0QhN7P:^[Zh2dm?C0P0000;=cLd2H61P0XB4Q09jNWX<
O7al0W]kN`9mOGd2O7al0WemO@QlO7`2OGem0WIfMP:4Q8@2O7al0_ooo`03Ng]kO7alO7al011lO7`2
MWIf0YbLW0:IVIT2<C4a0XR8R0:9RHT2BDU90QlO7`800002^;Rh0]WIf@9^KVh2_;bl0SDe=@9bLW82
00000W9bLP8T92@2fMWI0S<c<`:XZ:P2W9bL0WIfMRQlO7`6Ng]k0WalO09kNg/8O7al0W]kN`9lO7`5
oooo0@00007oool001Goool00`000?ooooooo`0Roooo0X:2PP9jNWX4OGem0WalO09mOGd2MWIf0Zf]
[@:OWil2GEeM1?ooo`:c/k<200000YZJVP9YJFT2S8b<0X>3P`9jNWX2IFEU0^CTi09SHf<2E5AD0[Jf
]P800002Vi^K0V][J`:1PH42NWYj2WalO09kNg/2O7al1WYjNP9mOGd2O7al0W]kN`9lO7`2OGem0WUi
N@:7QhL2C4a<0_ooo`03OGemO7alO7al00AkNg/4O7al0W]kN`AlO7`2NWYj0W]kN`:8R8P2e=CD0V5Q
H@:1PH42PX:20Q`L70:5QHD2a<C40W=cL`:SXj<2ZJVY0W1`L09MGEd2HF5Q0^GUi@9<C4`2K6a/0Y6A
T@:IVIT200000ZFUY@9/K6`2PH610W]kN`AlO7`2Ng]k57alO09kNg/2O7al17]kN`AlO7`2Ng]k0Wem
O@9lO7`2OGem0WalO0AkNg/2O7al0WemO@9lO7`5oooo0@00007oool001Goool00`000?ooooooo`0R
oooo0X:2PP9jNWX4OGem0WalO09mOGd2MWIf0Zf][@:OWil2GEeM1?ooo`:c/k<200000YZJVP9YJFT2
S8b<0X>3P`9jNWX2IFEU0^CTi09SHf<2E5AD0[Jf]P800002Vi^K0V][J`:1PH42NWYj2WalO09kNg/2
O7al1WYjNP9mOGd2O7al0W]kN`9lO7`2OGem0WUiN@:7QhL2C4a<0_ooo`03OGemO7alO7al00AkNg/4
O7al0W]kN`AlO7`2NWYj0W]kN`:8R8P2e=CD0V5QH@:1PH42PX:20Q`L70:5QHD2a<C40W=cL`:SXj<2
ZJVY0W1`L09MGEd2HF5Q0^GUi@9<C4`2K6a/0Y6AT@:IVIT200000ZFUY@9/K6`2PH610W]kN`AlO7`2
Ng]k57alO09kNg/2O7al17]kN`AlO7`2Ng]k0WemO@9lO7`2OGem0WalO0AkNg/2O7al0WemO@9lO7`5
oooo0@00007oool001Goool00`000?ooooooo`0Roooo17alO09iNGT2N7Qh0WYjNP9oOgl2M7Ad0X>3
P`94A4@2`L710R<S8`9?Cdl2IVIV0Y^KV`9fMWH2PH610XR8R08k>c/2:RXZ0R/[:`;<c<`2Ph>30[:b
/P:5QHD2SHf=0WQhN09mOGd2O7al17emO@9lO7`2Ng]k17YjNP9mOGd2NWYj0X61P@9mOGd2PX:20WYj
NP9iNGT2Ogmo0W]kN`9lO7`2N7Qh0W]kN`:9RHT2Lg=c00=hN7QlO7alO7`017emO@9lO7`2Ng]k0Wmo
O`9lO7`2Ng]k0WinOP9lO7`2OGem0V9RHP:GUiL2O7al0Xj>SP9gMgL2Kfm_0UmOG`98B4P2=SHf0[:b
/P9UIFD2MWIf0YRHV0:LW9`2c<c<0Va/K0:>SXh2SHf=0YBDU09fMWH2OGem0WalO09jNWX2O7al0WUi
N@9nOWh2Ng]k17alO09mOGd2Ng]k0WalO09kNg/6O7al0WemO@9kNg/2OGem0WinOP9lO7`2Ng]k0Win
OP9gMgL2Ng]k17YjNP9nOWh2Ogmo0W]kN`9iNGT2O7al1Oooo`400001oooo000Eoooo00<0003ooooo
ool08_ooo`AlO7`2NGUi0WQhN09jNWX2Ogmo0WAdM0:3Ph<2A4A40/71`@8S8b<2Cdm?0VIVIP:KVi/2
MWIf0X61P@:8R8P2>c/k0RXZ:P8[:b/2c<c<0X>3P`:b/[82QHF50Xf=S@9hN7P2OGem0WalO0AmOGd2
O7al0W]kN`AjNWX2OGem0WYjNP:1PH42OGem0X:2PP9jNWX2NGUi0WmoO`9kNg/2O7al0WQhN09kNg/2
RHV90W=cL`03N7QhO7alO7al00AmOGd2O7al0W]kN`9oOgl2O7al0W]kN`9nOWh2O7al0WemO@9RHV82
UiNG0WalO0:>SXh2MgMg0Vm_K`9OGel2B4Q80SHf=P:b/[82IFEU0WIfMP:HV9P2W9bL0/c<c09/K6`2
SXj>0Xf=S@:DU9@2MWIf0WemO@9lO7`2NWYj0WalO09iNGT2OWin0W]kN`AlO7`2OGem0W]kN`9lO7`2
Ng]k1WalO09mOGd2Ng]k0WemO@9nOWh2O7al0W]kN`9nOWh2MgMg0W]kN`AjNWX2OWin0WmoO`9kNg/2
NGUi0WalO0Goool100000Oooo`005Oooo`030000oooooooo02;oool4O7al0XJ6QP:<S8`2PH610WEe
M@:7QhL2Ng]k0XN7Q`:3Ph<2Q8B40WalO09nOWh2MGEe0WYjNP9`L702LW9b0Xn?S`9iNGT2PH610Xj>
SP:3Ph<2KFe]0W]kN`9fMWH2OWin0WemO@9kNg/2NWYj0WUiN@9kNg/2O7al0X:2PP:3Ph<2N7Qh0Wmo
O`9TI6@2Ng]k0VMWI`9jNWX2Q8B40W=cL`:0P802NWYj0XZ:RP9gMgL2O7al0W]kN`03QXJ6OGemOGem
00AiNGT2Ng]k0WinOP9aLG42OGem0WinOP9fMWH2NGUi0WinOP:6QXH2Lg=c0WalO09iNGT2K6a/0X61
P@9mOGd2Ng]k0X20P0AoOgl2K6a/17]kN`9[Jf/2PH610W=cL`9jNWX2Ng]k17moO`9hN7P2PX:20WUi
N@:3Ph<2MgMg0WemO@9kNg/2O7al0WYjNPAnOWh2OGem0W]kN`9jNWX2Ng]k0WUiN@9lO7`2N7Qh0WIf
MP9lO7`2Ogmo0WYjNP9oOgl2MGEe0WemO@:1PH44Lg=c0X20P0:2PX82N7Qh1Oooo`400001oooo000E
oooo0P0002?oool4O7al0XJ6QP:<S8`2PH610WEeM@:7QhL2Ng]k0XN7Q`:3Ph<2Q8B40WalO09nOWh2
MGEe0WYjNP9`L702LW9b0Xn?S`9iNGT2PH610Xj>SP:3Ph<2KFe]0W]kN`9fMWH2OWin0WemO@9kNg/2
NWYj0WUiN@9kNg/2O7al0X:2PP:3Ph<2N7Qh0WmoO`9TI6@2Ng]k0VMWI`9jNWX2Q8B40W=cL`:0P802
NWYj0XZ:RP9gMgL2O7al0W]kN`03QXJ6OGemOGem00AiNGT2Ng]k0WinOP9aLG42OGem0WinOP9fMWH2
NGUi0WinOP:6QXH2Lg=c0WalO09iNGT2K6a/0X61P@9mOGd2Ng]k0X20P0AoOgl2K6a/17]kN`9[Jf/2
PH610W=cL`9jNWX2Ng]k17moO`9hN7P2PX:20WUiN@:3Ph<2MgMg0WemO@9kNg/2O7al0WYjNPAnOWh2
OGem0W]kN`9jNWX2Ng]k0WUiN@9lO7`2N7Qh0WIfMP9lO7`2Ogmo0WYjNP9oOgl2MGEe0WemO@:1PH44
Lg=c0X20P0:2PX82N7Qh1?ooo`800001oooo000Eoooo00<0003oooooool08_ooo`9dM7@2QHF50V1P
H0;clo<2UiNG0Z>SX`9FEUH2LG5a0WemO@9]KFd2HF5Q0Y>CT`9fMWH2OWin0XB4Q0:MWId2Vi^K0Va/
K09fMWH2IFEU0VUYJ@:9RHT2R8R80WIfMP9iNGT2MgMg0W=cL`B0P802Q8B40WmoO`9kNg/2IfMW0V=S
H`:1PH42PX:20Y2@T0:;Rh/2VYZJ16QXJ0:KVi/2IFEU0X^;R`9DE5@2T92@0WEeM@:ATI400emOGh20
P820P002PH610X>3P`:4Q8@2MgMg0ZRXZ09fMWH2MgMg0X^;R`:1PH42LG5a0WEeM@9oOgl2OGem0X20
P0:BTY82R8R80WYjNP9lO7`2Kfm_0WUiN@9GEeL2RHV90WAdM0:6QXH2Shn?0W5aL@:=SHd2M7Ad0VMW
I`9kNg/2KVi^0Xj>SP9^KVh2QXJ60VYZJP:3Ph<2NGUi0WmoO`9kNg/2O7al0WIfMP9cLg<2Ng]k0X61
P@9nOWh2O7al0X61P@9mOGd2PX:20XV9R@9hN7P2LW9b0XZ:RP8>3Ph200000U5AD@9^KVh2Ogmo0ZRX
Z09_Kfl2KVi^0Y2@T0Goool100000Oooo`005Oooo`030000oooooooo02;oool2M7Ad0XF5Q@9PH602
lo?c0YNGU`:SXj<2EUIF0W5aL@9mOGd2KFe]0V5QH@:CTi<2MWIf0WinOP:4Q8@2WIfM0Y^KV`9/K6`2
MWIf0VEUI@9YJFT2RHV90XR8R09fMWH2NGUi0WMgM`9cLg<4P8200XB4Q09oOgl2Ng]k0VMWI`9SHf<2
PH610X:2PP:@T902Rh^;0YZJVPAXJ6P2Vi^K0VEUI@:;Rh/2E5AD0Y2@T09eMGD2TI6A00=OGen0P820
P8000X61P@:3Ph<2Q8B40WMgM`:XZ:P2MWIf0WMgM`:;Rh/2PH610W5aL@9eMGD2Ogmo0WemO@:0P802
TY:B0XR8R09jNWX2O7al0Vm_K`9iNGT2EeMG0XV9R@9dM7@2QXJ60Xn?S`9aLG42SHf=0WAdM09WIfL2
Ng]k0Vi^KP:>SXh2KVi^0XJ6QP9ZJVX2Ph>30WUiN@9oOgl2Ng]k0WalO09fMWH2Lg=c0W]kN`:1PH42
OWin0WalO0:1PH42OGem0X:2PP:9RHT2N7Qh0W9bLP::RXX23Ph>0P00009ADE42KVi^0WmoO`:XZ:P2
Kfm_0Vi^KP:@T905oooo0@00007oool001Goool00`000?ooooooo`0Roooo0WEeM@:_[jl2DU9B0_oo
o`:DU9@2WIfM0[Vi^@8D51@2TI6A0[Vi^@:6QXH200000X20P09ZJVX2KFe]0Ph>3P8V9RH2T92@0UUI
F@;Hf=P2WYjN0U9BDP9:BTX2N7Qh0X^;R`:VYZH2W9bL0WmoO`9]KFd2VYZJ0WUiN@:=SHd2[Jf]0[jn
_P9SHf<2XZ:R0_gmo@8M7Ad2Hf=S0^7Qh@9^KVh2S8b<0UeMG@:d];@2oooo0UADE0;alO423@d=00>^
[Zjc/k>c/k<00UaLG097AdL2WYjN0/g=c@:BTY82Ng]k0YNGU`8^;Rh271`L0Ve]K@;:b/X2Hf=S0Ve]
K@8i>CT29BDU0VIVIP93@d<2TY:B0[fm_@9fMWH2hn?S0V=SH`:TY:@2EEEE0Rl_;`;1`L42EUIF0V1P
H0:l_;`2alO70[6a/@9hN7P2Zj^[0/30`0:DU9@2Cdm?0XB4Q09dM7@2DE5A0TY:BP:o_kl2YJFU0WAd
M09QHF42MWIf0TM7A`8g=cL2QXJ60Ve]K@9?Cdl2Lg=c0]_Kf`800002DU9B0RPX:0:c/k<2FUYJ0[bl
_0:>SXh2E5AD0^GUi@:PX:05oooo0@00007oool001Goool00`000?ooooooo`0Roooo0WEeM@:_[jl2
DU9B0_ooo`:DU9@2WIfM0[Vi^@8D51@2TI6A0[Vi^@:6QXH200000X20P09ZJVX2KFe]0Ph>3P8V9RH2
T92@0UUIF@;Hf=P2WYjN0U9BDP9:BTX2N7Qh0X^;R`:VYZH2W9bL0WmoO`9]KFd2VYZJ0WUiN@:=SHd2
[Jf]0[jn_P9SHf<2XZ:R0_gmo@8M7Ad2Hf=S0^7Qh@9^KVh2S8b<0UeMG@:d];@2oooo0UADE0;alO42
3@d=00>^[Zjc/k>c/k<00UaLG097AdL2WYjN0/g=c@:BTY82Ng]k0YNGU`8^;Rh271`L0Ve]K@;:b/X2
Hf=S0Ve]K@8i>CT29BDU0VIVIP93@d<2TY:B0[fm_@9fMWH2hn?S0V=SH`:TY:@2EEEE0Rl_;`;1`L42
EUIF0V1PH0:l_;`2alO70[6a/@9hN7P2Zj^[0/30`0:DU9@2Cdm?0XB4Q09dM7@2DE5A0TY:BP:o_kl2
YJFU0WAdM09QHF42MWIf0TM7A`8g=cL2QXJ60Ve]K@9?Cdl2Lg=c0]_Kf`800002DU9B0RPX:0:c/k<2
FUYJ0[bl_0:>SXh2E5AD0^GUi@:PX:05oooo0@00007oool001Goool00`000?ooooooo`0Boooo0VIV
IP:`/;027AdM0/g=c@Soool2_;bl0QTI6@9HF5P2e]KF0Q/K6`9VIVH2JVYZ0Y>CT`;oool2H61P0P00
00:`/;02Ogmo0Z:RXP:?Shl2oooo0Y6AT@9SHf<2`L710/K6aP:a/K42oooo0QhN7P;De=@2X:2P0X>3
P`:KVi/2PX:20XZ:RP9iNGT2alO70YbLW0:f][H2`L710Zb/[09=CDd2Shn?0WIfMP9hN7P2EUIF0XF5
Q@9ZJVX2Zj^[0SLg=`92@T82U9BD0XN7Q`9ADE400f=SHhR8R8R8R002R8R80X61P@9gMgL2cLg=0Wem
O@8m?Cd2/;2`0R0P808n?Sh2dm?C0Y>CT`;Hf=P25QHF0Z6QX@861PH2Ng]k0TM7A`:EUID2MWIf0SXj
>P9oOgl2oooo0ZNWY`;oool2e]KF0_ooo`90@402?Shn0ZVYZ@9BDU82l_;b0TE5A@8L71`2oooo0P00
00;oool2HV9R155AD@::RXX2o_kn0_ooo`9DE5@2EEEE0Xn?S`A]KFd2RHV90^WYj@:h^;P2PX:20YRH
V08m?Cd2F5QH0Xj>SP9fMWH2Rh^;0Ve]K@9oOgl2YZJV0Ve]K@9?Cdl5oooo0@00007oool001Goool0
0`000?ooooooo`0Boooo0VIVIP:`/;027AdM0/g=c@Soool2_;bl0QTI6@9HF5P2e]KF0Q/K6`9VIVH2
JVYZ0Y>CT`;oool2H61P0P0000:`/;02Ogmo0Z:RXP:?Shl2oooo0Y6AT@9SHf<2`L710/K6aP:a/K42
oooo0QhN7P;De=@2X:2P0X>3P`:KVi/2PX:20XZ:RP9iNGT2alO70YbLW0:f][H2`L710Zb/[09=CDd2
Shn?0WIfMP9hN7P2EUIF0XF5Q@9ZJVX2Zj^[0SLg=`92@T82U9BD0XN7Q`9ADE400f=SHhR8R8R8R002
R8R80X61P@9gMgL2cLg=0WemO@8m?Cd2/;2`0R0P808n?Sh2dm?C0Y>CT`;Hf=P25QHF0Z6QX@861PH2
Ng]k0TM7A`:EUID2MWIf0SXj>P9oOgl2oooo0ZNWY`;oool2e]KF0_ooo`90@402?Shn0ZVYZ@9BDU82
l_;b0TE5A@8L71`2oooo0P0000;oool2HV9R155AD@::RXX2o_kn0_ooo`9DE5@2EEEE0Xn?S`A]KFd2
RHV90^WYj@:h^;P2PX:20YRHV08m?Cd2F5QH0Xj>SP9fMWH2Rh^;0Ve]K@9oOgl2YZJV0Ve]K@9?Cdl5
oooo0@00007oool001Goool00`000?ooooooo`0Voooo0YfMW@;clo<200000Z>SX`;AdM429bLW0TQ8
B08k>c/2?clo0WIfMP:PX:02c/k>0X^;R`800002QXJ60ZJVYP:0P802l?3`0QdM7@;;bl/2Fe]K0X>3
P`8g=cL2[Zj^0WmoO`8b<S84OWin0W1`L0:8R8P2hn?S1_ooo`8c<c<2CTi>0U=CD`;oool2;Rh^0/?3
``9DE5@2WYjN0/O7a`;Vi^H2@d=30YbLW003QXJ6XJ6QXJ6Q009DE5@2Ng]k0_ooo`:FUYH2Y:BT0VMW
I`:KVi/2LW9b0WYjNP:c/k<2EeMG0^WYj@;oool2BDU90Zf][@8m?Cd2CTi>0UMGE`;dm?@2Ng]k0R`/
;0:XZ:P2AdM70XJ6QP:EUID2ADE50Q@D50:CTi<2K6a/0[^k^`:NWYh2BDU90Z^[Z`:SXj<2Q8B40Rd]
;@800002/K6a0YfMW@:4Q8@2JVYZ0X20P0;oool2/K6a0WinOP;oool2QXJ60TM7A`892@T2>CTi0/C4
a0;oool200000WEeM@9aLG42J6QX0[6a/@:RXZ82<C4a0WalO09XJ6P2P8201Oooo`400001oooo000E
oooo00<0003oooooool09_ooo`:MWId2lo?c0P0000:SXj<2dM7A0RLW9`98B4P2>c/k0Slo?`9fMWH2
X:2P0/k>cP:;Rh/200000XJ6QP:VYZH2P8200_3`l08M7Ad2bl_;0U]KF`:3Ph<2=cLg0Zj^[P9oOgl2
<S8b17inOP9`L702R8R80^?Sh`Koool2<c<c0Ti>CP9CDe<2oooo0Rh^;P;3`l<2E5AD0YjNWP;7alL2
i^KV0T=3@`:LW9`00hJ6QZ6QXJ6QX@02E5AD0W]kN`;oool2UYJF0ZBTY09WIfL2Vi^K0W9bLP9jNWX2
/k>c0UMGE`;YjNT2oooo0TU9B@:][Jd2?Cdm0Ti>CP9GEeL2m?Cd0W]kN`8/;2`2Z:RX0TM7A`:6QXH2
UIFE0TE5A@8D51@2Ti>C0Va/K0:k^k/2WYjN0TU9B@:[Zj/2Xj>S0XB4Q08];Bd200000[6a/@:MWId2
Q8B40VYZJP:0P802oooo0[6a/@9nOWh2oooo0XJ6QP97AdL22@T90STi>@;4a<@2oooo0P00009eMGD2
LG5a0VQXJ0:a/K42XZ:R0S4a<@9lO7`2J6QX0X20P0Goool100000Oooo`005Oooo`030000oooooooo
02;oool2T92@0P0000:CTi<2oooo0TE5A@9ZJVX2QXJ60_ooo`800002@d=30S8b<P8f=SH20@410_oo
o`9IFET2oooo0/O7a`:RXZ82kNg]0S4a<@9UIFD2no_k0Q<C4`:6QXH200000_ooo`:CTi<2oooo0Pd=
3@800002gmoO0^;RhP93@d<4oooo0];BdP;5aLD20@410U5AD@9KFe/200000_ooo`8a<C42/[:b0SPh
>0892@T2[jn_0P000003=SHfC4a<C4a<0093@d<2SXj>0/[:bP:f][H2K6a/0UaLG0800002RHV90Vm_
K`9WIfL2RHV90TQ8B0;oool2IFEU0/?3``9GEeL2>CTi0WEeM@:7QhL2HV9R0Z2PX09SHf<200000/g=
c@9gMgL2WIfM0Y2@T09aLG42WYjN0[jn_P9`L702AdM70/S8b0:PX:02];Bd0_ooo`:MWId2[jn_0SHf
=P800002[Zj^0Y>CT`:2PX82WYjN0Shn?P:ZZZX2BDU90Zj^[P8410@2oooo0X20P0:n_[h200000W1`
L0:6QXH2^k^k0]KFeP9MGEd2HV9R0]SHf0:j^[X2Kfm_1Oooo`400001oooo000Eoooo0P0002?oool2
T92@0P0000:CTi<2oooo0TE5A@9ZJVX2QXJ60_ooo`800002@d=30S8b<P8f=SH20@410_ooo`9IFET2
oooo0/O7a`:RXZ82kNg]0S4a<@9UIFD2no_k0Q<C4`:6QXH200000_ooo`:CTi<2oooo0Pd=3@800002
gmoO0^;RhP93@d<4oooo0];BdP;5aLD20@410U5AD@9KFe/200000_ooo`8a<C42/[:b0SPh>0892@T2
[jn_0P000003=SHfC4a<C4a<0093@d<2SXj>0/[:bP:f][H2K6a/0UaLG0800002RHV90Vm_K`9WIfL2
RHV90TQ8B0;oool2IFEU0/?3``9GEeL2>CTi0WEeM@:7QhL2HV9R0Z2PX09SHf<200000/g=c@9gMgL2
WIfM0Y2@T09aLG42WYjN0[jn_P9`L702AdM70/S8b0:PX:02];Bd0_ooo`:MWId2[jn_0SHf=P800002
[Zj^0Y>CT`:2PX82WYjN0Shn?P:ZZZX2BDU90Zj^[P8410@2oooo0X20P0:n_[h200000W1`L0:6QXH2
^k^k0]KFeP9MGEd2HV9R0]SHf0:j^[X2Kfm_1?ooo`800001oooo000Eoooo00<0003oooooool04?oo
o`:j^[X@oooo0P0000:d];@4oooo0WUiN@9;Bd/2Ti>C0^;RhP:PX:0200000UiNGP:NWYh200000_?c
l`;Shn<2Zj^[0P0000;YjNT2Ph>30VEUI@9JFUX2GEeM0/c<c09DE5@2D51@0VYZJP:>SXh2oooo0Ta<
C08S8b<2oooo0XZ:RP;oool200000^WYj@;MgMd2`l?30P0000;oool2SHf=0^3Ph0;gmoL2oooo0PT9
2@:j^[X2HV9R0V5QH@8000000e1@D3Ph>3Ph>002=cLg0WmoO`9YJFT2LW9b0^SXj0;2`/82NGUi0X:2
PP:3Ph<2UYJF0[Jf]P:0P80200000YRHV0:d];@2?Shn0X^;R`99BDT2d=3@0YjNWP9HF5P2;Rh^0WUi
N@97AdL2QHF50RPX:09gMgL2HF5Q0[Rh^08;2`/2o_kn0Vm_K`9eMGD2_;bl0RDU9@800002VIVI0XR8
R09UIFD2oooo0RDU9@8i>CT2^[Zj0Xn?S`8f=SH2oooo0YnOW`9hN7P2ZZZZ0VQXJ09lO7`2][Jf0P00
009SHf<2I6AT0WmoO`9VIVH2_kno0STi>@:SXj<2/;2`0]7Ad@Goool100000Oooo`005Oooo`030000
oooooooo013oool2^[Zj4?ooo`800002];Bd1?ooo`9iNGT2Bd];0Y>CT`;Rh^82X:2P0P00009NGUh2
WYjN0P0000;clo<2hn?S0Z^[Z`800002jNWY0X>3P`9UIFD2FUYJ0UeMG@;<c<`2E5AD0U1@D09ZJVX2
SXj>0_ooo`9<C4`28b<S0_ooo`::RXX2oooo0P0000;YjNT2gMgM0/?3``800002oooo0Xf=S@;Ph>02
moOg0_ooo`892@T2^[Zj0V9RHP9QHF42000000=@D50h>3Ph>3P00SLg=`9oOgl2JFUY0W9bLP;Xj>P2
`/;20WUiN@:2PX82Ph>30YJFUP:f][H2P8200P0000:HV9P2];Bd0Shn?P:;Rh/2BDU90]3@d0:NWYh2
F5QH0Rh^;P9iNGT2AdM70XF5Q@8X:2P2MgMg0V5QH@:h^;P22`/;0_knoP9_Kfl2MGEe0[bl_08U9BD2
00000YVIV@:8R8P2IFEU0_ooo`8U9BD2>CTi0[Zj^P:?Shl2=SHf0_ooo`:OWil2N7Qh0ZZZZP9XJ6P2
O7al0[Jf]P800002Hf=S0VATI09oOgl2IVIV0[no_`8i>CT2Xj>S0[2`/0;AdM45oooo0@00007oool0
01Goool00`000?ooooooo`0Roooo0TY:BP800002820P0P00009]KFd210@40P0000;oool2OWin0P00
009]KFd2@4100X>3P`:=SHd2bLW91000009LG5`2Jf][0U5AD@;]kNd2?3`l0Slo?`800002=SHf0[jn
_P8000024Q8B0]SHf09GEeL200000XF5Q@;oool2G5aL0Z^[Z`9nOWh2<c<c0UMGE`@00002oooo0Xj>
SP:WYjL2U9BD0/?3``9GEeL2RXZ:0Zf][@03ADE5X:2PX:2P009ZJVX2;Bd]0Z:RXP9]KFd2dm?C0_oo
o`9eMGD2PH610WIfMP;@d=02LW9b0[:b/P;oool2IFEU0Y>CT`:<S8`200000/30`08l?3`2Ng]k0YFE
U@8n?Sh2MGEe0YjNWP8R8R82HF5Q0WEeM@9JFUX2ADE50/;2`P97AdL2G5aL0X61P@8L71`2c/k>0[2`
/09IFET24a<C0ZNWY`8H61P2GUiN0Z>SX`;Vi^H261PH0]gMg@:;Rh/2H61P0XN7Q`9RHV82SHf=0VAT
I0;1`L42oooo0X:2PP97AdL210@40_ooo`800002lo?c0Ta<C09CDe<2^[Zj1Oooo`400001oooo000E
oooo00<0003oooooool08_ooo`9:BTX200000R0P80800002KFe]0P@410800002oooo0WinOP800002
KFe]0T10@0:3Ph<2SHf=0/W9b@@00002G5aL0V][J`9ADE42kNg]0S`l?08o?cl200000SHf=P:n_[h2
00000Q8B4P;Hf=P2EeMG0P0000:5QHD2oooo0UaLG0:[Zj/2OWin0S<c<`9GEeL400000_ooo`:>SXh2
YjNW0YBDU0;3`l<2EeMG0XZ:RP:][Jd00dE5AJ2PX:2PX002JVYZ0Rd];@:RXZ82KFe]0]?Cd`;oool2
MGEe0X61P@9fMWH2d=3@0W9bLP:b/[82oooo0VEUI@:CTi<2S8b<0P0000;0`<02?3`l0W]kN`:EUID2
?Shn0WEeM@:NWYh28R8R0V5QH@9eMGD2FUYJ0TE5A@;2`/82AdM70UaLG0:1PH4271`L0/k>cP:`/;02
FEUI0Q<C4`:WYjL261PH0UiNGP:SXj<2i^KV0QPH60;MgMd2Rh^;0V1PH0:7QhL2HV9R0Xf=S@9TI6@2
`L710_ooo`:2PX82AdM70P@410;oool200000_?cl`9<C4`2De=C0[Zj^PGoool100000Oooo`005Ooo
o`030000oooooooo02;oool2Hf=S0/G5a@800002oooo0P0000:BTY8200000_ooo`:/[:`2JFUY0UQH
F0;UiND2T92@0X^;R`;oool23Ph>0S/k>`:NWYh26QXJ0/k>cP8_;bl2TI6A0S4a<@;oool2De=C0_oo
o`8Q8B422`/;0Vm_K`@00002inOW0Xj>SP9WIfL4oooo0[2`/09nOWh2b<S80Ti>CP@00002clo?0^[Z
jP9;Bd/2IVIV0UEEE@;oool00inOW`0000000002oooo0RHV9P9lO7`2SXj>0YRHV0:;Rh/2J6QX0XR8
R09lO7`2Cdm?0P0000;oool2/k>c0_ooo`92@T82_[jn0Y>CT`:][Jd2OGem0U=CD`9/K6`2;Rh^0Z6Q
X@800002JFUY0X>3P`:WYjL2[jn_0Zb/[0;1`L42<c<c0Q0@40:YZJT2^KVi0^7Qh@;oool292@T0ZJV
YP;4a<@25aLG0_ooo`;]kNd2fMWI0]oOg`9;Bd/2j^[Z0]7Ad@9lO7`2@T920[:b/P:LW9`2UiNG0UUI
F@9dM7@2MgMg0[fm_@98B4P2f=SH0^OWi`8I6AT9oooo0@00007oool001Goool00`000?ooooooo`0R
oooo0V=SH`;5aLD200000_ooo`800002TY:B0P0000;oool2[:b/0VUYJ@9HF5P2iNGU0Y2@T0:;Rh/2
oooo0Ph>3P8k>c/2WYjN0QXJ6P;>c/h2;bl_0Y6AT@8a<C42oooo0U=CD`;oool28B4Q0P/;2`9_Kfl4
00000^OWi`:>SXh2IfMW1?ooo`:`/;02OWin0/S8b09>CTh400000/o?c`;Zj^X2Bd];0VIVIP9EEED2
oooo00>OWil0000000000_ooo`8V9RH2O7al0Xj>SP:HV9P2Rh^;0VQXJ0:8R8P2O7al0Tm?C`800002
oooo0[>c/`;oool2@T920[jn_P:CTi<2[Jf]0WemO@9CDe<2K6a/0Rh^;P:QXJ4200000VUYJ@:3Ph<2
YjNW0Zn_[`:/[:`2`L710S<c<`8@4102ZJVY0[Vi^@;QhN42oooo0R@T90:VYZH2a<C40QLG5`;oool2
kNg]0]WIf@;Ogml2Bd];0^[ZjP;AdM42O7al0T92@P:b/[82W9bL0YNGU`9IFET2M7Ad0WMgM`:m_Kd2
B4Q80]SHf0;WinL26ATI2Oooo`400001oooo000Eoooo00<0003oooooool08_ooo`::RXX2Rh^;0[Zj
^P;oool200000V5QH@:BTY82KVi^0XN7Q`9=CDd2]KFe0Q<C4`;;bl/2YJFU0/K6aP800002XZ:R0XV9
R@:2PX82U9BD0Va/K0;Jf]X2Ogmo0_?cl`800002XZ:R0SPh>08O7al2aLG50WMgM`:ATI42GUiN0V1P
H0:MWId2L71`0X20P0;4a<@251@D0_ooo`:XZ:P2UYJF0WYjNP;oool2410@0WYjNP9bLW82EUIF0_oo
o`03@D51OgmoOgmo00:CTi<2GemO0[Jf]P:SXj<2[Jf]0Z:RXP9VIVH2LG5a0VQXJ0;oool200000P/;
2`;oool2VIVI0S8b<P800002MWIf0YZJVP:g]kL4K6a/0]OGe`;3`l<2Ogmo0_ooo`:j^[X2:2PX0YZJ
VP800002B4Q80_Ogm`8a<C42ZZZZ0Y>CT`;oool28b<S0VIVIP9EEED2MGEe0V9RHP:7QhL2RXZ:0W5a
L@9lO7`2UIFE0X>3P`92@T82oooo0UUIF@;UiND2QhN70]KFeP8:2PX4GUiN0Xn?S`9CDe<2oooo0P00
00:HV9P2RXZ:0[Vi^@Goool100000Oooo`005Oooo`030000oooooooo02;oool2RXZ:0X^;R`:j^[X2
oooo0P00009QHF42TY:B0Vi^KP:7QhL2CDe=0[Fe]@8C4a<2bl_;0ZFUY@;6a/H200000Z:RXP:9RHT2
PX:20YBDU09/K6`2f][J0WmoO`;clo<200000Z:RXP8h>3P27alO0/G5a@9gMgL2TI6A0UiNGP9PH602
WIfM0W1`L0:0P802a<C40Q@D50;oool2Z:RX0YJFUP9jNWX2oooo0Q0@409jNWX2LW9b0UIFEP;oool0
0d51@GmoOgmoO`02Ti>C0UmOG`:f][H2Xj>S0Zf][@:RXZ82IVIV0W5aL@9XJ6P2oooo0P00008;2`/2
oooo0YVIV@8b<S8200000WIfMP:JVYX2]kNg16a/K0;GemL2`l?30WmoO`;oool2^[Zj0RPX:0:JVYX2
00000TQ8B0;gmoL2<C4a0ZZZZP:CTi<2oooo0R<S8`9VIVH2EEEE0WEeM@9RHV82QhN70XZ:RP9aLG42
O7al0YFEU@:3Ph<2@T920_ooo`9IFET2iNGU0XN7Q`;Fe]H22PX:15iNGP:?Shl2De=C0_ooo`800002
V9RH0XZ:RP:i^KT5oooo0@00007oool001Goool00`000?ooooooo`02oooo0W5aLA[oool2h^;R0S@d
=0:4Q8@2I6AT0V5QH@8O7al2/k>c0WIfMP9kNg/2KFe]0[2`/0:UYJD2EEEE0YFEU@;GemL200000[Vi
^@:4Q8@2oooo0][JfP:?Shl2OGem0V5QH@:9RHT2LW9b0[^k^`800002FEUI0WinOP::RXX2L71`0WQh
N09fMWH2Ph>30XV9R@9WIfL2D51@0[Rh^090@402U9BD0UADE09fMWH2OGem0W5aL@9ADE42RXZ:0Y2@
T0:<S8`2Ng]k0VYZJP03QhN7Ng]kNg]k009fMWH2QhN70W5aL@9bLW82H61P0XZ:RP9_Kfl2I6AT0XV9
R@:4Q8@2@d=30UaLG0;?cll2FUYJ0VUYJ@9ADE42NGUi0XF5Q@:ATI42NGUi0W]kN`9dM7@2NGUi0YBD
U0:j^[X2_[jn0SXj>P9/K6`2;bl_0YfMW@90@402P8200W]kN`9lO7`2J6QX0Y2@T09nOWh2Q8B40Wmo
O`:1PH42O7al0WUiN@:0P802OWin0WEeM@9lO7`2Rh^;0WinOP:8R8P2FEUI0XF5Q@:QXJ42[Zj^0WYj
NP9gMgL2O7al0WmoO`9XJ6P2TI6A0WAdM09iNGT2O7al1Oooo`400001oooo000Eoooo0P0000?oool2
LG5a6_ooo`;Rh^82=3@d0XB4Q09TI6@2HF5Q0QlO7`:c/k<2MWIf0W]kN`9]KFd2/;2`0ZFUY@9EEED2
UIFE0]OGe`800002^KVi0XB4Q0;oool2f][J0Xn?S`9mOGd2HF5Q0XV9R@9bLW82^k^k0P00009IFET2
OWin0XZ:RP9`L702N7Qh0WIfMP:3Ph<2RHV90VMWI`9@D502^;Rh0T10@0:DU9@2E5AD0WIfMP9mOGd2
LG5a0U5AD@::RXX2T92@0Xb<S09kNg/2JVYZ00>7QhMkNg]kNg/00WIfMP:7QhL2LG5a0W9bLP9PH602
RXZ:0Vm_K`9TI6@2RHV90XB4Q093@d<2G5aL0/o?c`9JFUX2JFUY0U5AD@9iNGT2QHF50Y6AT@9iNGT2
Ng]k0WAdM09iNGT2U9BD0[Zj^P:n_[h2>SXj0Va/K08_;bl2WIfM0T10@0:0P802Ng]k0WalO09XJ6P2
T92@0WinOP:4Q8@2Ogmo0X61P@9lO7`2NGUi0X20P09nOWh2MGEe0WalO0:;Rh/2OWin0XR8R09IFET2
QHF50Z6QX@:^[Zh2NWYj0WMgM`9lO7`2Ogmo0VQXJ0:ATI42M7Ad0WUiN@9lO7`4oooo0P00007oool0
01Goool00`000?ooooooo`02oooo0WMgMa[oool2]KFe0_ooo`9lO7`2PH610XB4Q0:BTY82JVYZ0Wem
O@9jNWX2O7al0Z:RXP;oool200000VUYJ@;UiND2j>SX0VEUI@9fMWH2@d=30XR8R08l?3`2_[jn0Xj>
SP9bLW82]KFe0UUIF@;@d=02MGEe0WinOP9gMgL2OWin0WemO@9nOWh2NGUi0WQhN0:1PH42QHF50W9b
LP::RXX2MGEe0XV9R@AmOGd2Ogmo0XZ:RP9fMWH2MgMg0WQhN09bLW82UYJF00=hN7QmOGemOGd00Win
OP9jNWX4Ogmo0X>3P`9jNWX2Ogmo0X61P@9iNGT2NWYj0Xj>SP:CTi<2FUYJ0WIfMP;1`L42EeMG0[^k
^`:4Q8@2JVYZ0Y:BTP:6QXH2O7al0W=cL`:1PH42M7Ad0W5aL@:g]kL2G5aL0XN7Q`9ZJVX2Shn?0WUi
N@9mOGd2Ng]k0X:2PP9fMWH2Ng]k0WUiN@9kNg/2NWYj0W]kN`9lO7`4Ng]k0WinOP9kNg/2MgMg0Win
OP9iNGT2QXJ60WYjNP9dM7@2Kfm_1WalO09kNg/2PX:20WIfMP9nOWh2Ng]k0WemO@Goool100000Ooo
o`005Oooo`030000oooooooo00;oool2MgMg6_ooo`:e]KD2oooo0WalO0:1PH42Q8B40Y:BTP9ZJVX2
OGem0WYjNP9lO7`2XZ:R0_ooo`800002JFUY0^GUi@;Xj>P2IFEU0WIfMP93@d<2R8R80S`l?0:n_[h2
SXj>0W9bLP:e]KD2FEUI0]3@d09eMGD2OWin0WMgM`9nOWh2OGem0WinOP9iNGT2N7Qh0X61P@:5QHD2
LW9b0XZ:RP9eMGD2RHV917emO@9oOgl2RXZ:0WIfMP9gMgL2N7Qh0W9bLP:FUYH00gQhN7emOGemO@02
OWin0WYjNPAoOgl2Ph>30WYjNP9oOgl2PH610WUiN@9jNWX2SXj>0Y>CT`9JFUX2MWIf0/71`@9GEeL2
^k^k0XB4Q09ZJVX2TY:B0XJ6QP9lO7`2Lg=c0X61P@9dM7@2LG5a0[Ng]`9LG5`2QhN70VYZJP:?Shl2
NGUi0WemO@9kNg/2PX:20WIfMP9kNg/2NGUi0W]kN`9jNWX2Ng]k0WalO0AkNg/2OWin0W]kN`9gMgL2
OWin0WUiN@:6QXH2NWYj0WAdM09_Kfl6O7al0W]kN`:2PX82MWIf0WinOP9kNg/2OGem1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9kNg/2Kfm_0WinOP9bLW82L71`0Va/K09eMGD2HF5Q0XB4
Q0:k^k/2Shn?0Xb<S09lO7`2PH610WAdM09lO7`2OWin0WMgM`9bLW82KFe]0UaLG09jNWX2KVi^0XV9
R@:2PX82Lg=c0Y6AT@:HV9P2P8200WMgM`9mOGd2QHF50X61P@:0P802SXj>0X20P09dM7@2PX:20WMg
M`9mOGd2MWIf0WUiN@9jNWX2O7al0WemO@9fMWH2OWin0WYjNP9kNg/2NGUi0WmoO`9_Kfl2PH610Z6Q
X@:6QXH2IfMW0WQhN09kNg/2NGUi0WIfMP9/K6`2PH610V][J`:<S8`00gEeMHN7QhN7Q`02Ng]k0X61
P@9mOGd2NGUi0WMgM`9lO7`2NWYj0WmoO`:4Q8@2Ogmo0WemO@9iNGT2Ng]k0X61P@:0P802OWin0XZ:
RP:0P802OWin0WYjNP9gMgL2PH610WIfMP:1PH42NWYj0W]kN`AnOWh2LW9b0WIfMP9iNGT2MGEe0WQh
N09cLg<2MgMg0X61P@9jNWX2O7al0X61P@:3Ph<4NWYj0W]kN`9mOGd2O7al0X:2PP:5QHD2NWYj1861
P@9mOGd2MGEe0YNGU`9CDe<2Q8B40W1`L09VIVH2W9bL0Xn?S`9eMGD2J6QX0WAdM0Goool100000Ooo
o`005Oooo`030000oooooooo00;oool2Ng]k0Vm_K`9nOWh2LW9b0W1`L09/K6`2MGEe0V5QH@:4Q8@2
^k^k0Xn?S`:<S8`2O7al0X61P@9dM7@2O7al0WinOP9gMgL2LW9b0Ve]K@9LG5`2NWYj0Vi^KP:9RHT2
PX:20W=cL`:ATI42V9RH0X20P09gMgL2OGem0XF5Q@:1PH42P8200Xj>SP:0P802M7Ad0X:2PP9gMgL2
OGem0WIfMP9iNGT2NWYj0WalO09mOGd2MWIf0WinOP9jNWX2Ng]k0WUiN@9oOgl2Kfm_0X61P@:QXJ42
QXJ60VMWI`9hN7P2Ng]k0WUiN@9fMWH2K6a/0X61P@9[Jf/2S8b<00=eMGF7QhN7QhL00W]kN`:1PH42
OGem0WUiN@9gMgL2O7al0WYjNP9oOgl2Q8B40WmoO`9mOGd2NGUi0W]kN`:1PH42P8200WinOP::RXX2
P8200WinOP9jNWX2MgMg0X61P@9fMWH2PH610WYjNP9kNg/4OWin0W9bLP9fMWH2NGUi0WEeM@9hN7P2
Lg=c0WMgM`:1PH42NWYj0WalO0:1PH42Ph>317YjNP9kNg/2OGem0WalO0:2PX82QHF50WYjNPB1PH42
OGem0WEeM@:GUiL2De=C0XB4Q09`L702IVIV0YbLW0:?Shl2MGEe0VQXJ09dM7@5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0WemO@:GUiL2TY:B0ZRXZ0:g]kL2_Kfm0YRHV0;[jn/2EUIF0R8R
8P9`L702WYjN0WUiN@9TI6@2ZJVY0WmoO`9dM7@2NGUi0ZNWY`:UYJD2JVYZ0TI6AP:j^[X2RXZ:0U1@
D0;1`L42VIVI0S0`<0:GUiL2SHf=0V=SH`@n?Sh2WinO0YNGU`:QXJ42VIVI0UQHF09WIfL2PH610Xf=
S@9jNWX2MGEe0WalO09[Jf/2CDe=0WYjNP:2PX82W9bL0Z:RXP9@D502Ph>30T];B`;oool200000VQX
J0:b/[8292@T0XF5Q@9kNg/2clo?0Ve]K@:QXJ42EeMG00?[jn^IVIVIVIT00TU9B@9RHV82MgMg0Va/
K08W9bL2QXJ60XF5Q@9KFe/2JFUY0WinOP:UYJD2@T920Xf=S@9=CDd2KVi^0W5aL@:2PX82SHf=0XZ:
RP:_[jl2W9bL0TI6AP:;Rh/2Bd];0S8b<P:5QHD2U9BD0W=cL`:^[Zh2V9RH0WmoO`96ATH2[jn_0UEE
E@:[Zj/2B4Q80WQhN0:3Ph<2UYJF0W]kN`:JVYX2JFUY0XB4Q0:6QXH2I6AT0X>3P`9oOgl2Rh^;0V1P
H09JFUX2FEUI0WmoO`:3Ph<2`<300/o?c`A`L702OWin0V][J`:4Q8@2YZJV0VIVIPGoool100000Ooo
o`005Oooo`030000oooooooo00;oool2OGem0YNGU`:BTY82Z:RX0[Ng]`:m_Kd2V9RH0^_[j`9FEUH2
8R8R0W1`L0:NWYh2NGUi0VATI0:YZJT2Ogmo0WAdM09iNGT2YjNW0ZFUY@9ZJVX2ATI60[Zj^P::RXX2
D51@0/71`@:IVIT2<30`0YNGU`:=SHd2Hf=S13hn?P:OWil2UiNG0Z6QX@:IVIT2F5QH0VMWI`:1PH42
SHf=0WYjNP9eMGD2O7al0V][J`9=CDd2NWYj0X:2PP:LW9`2XZ:R0U1@D0:3Ph<2Bd];0_ooo`800002
J6QX0[:b/P8T92@2QHF50W]kN`;?cll2KFe]0Z6QX@9GEeL00n_[jiVIVIVIV@02BDU90V9RHP9gMgL2
K6a/0RLW9`:6QXH2QHF50U]KF`9YJFT2OWin0ZFUY@92@T82SHf=0Te=C@9^KVh2LG5a0X:2PP:=SHd2
RXZ:0Zn_[`:LW9`2ATI60X^;R`9;Bd/2<S8b0XF5Q@:DU9@2Lg=c0Zj^[P:HV9P2Ogmo0TI6AP:_[jl2
EEEE0Z^[Z`98B4P2N7Qh0X>3P`:FUYH2Ng]k0YZJVP9YJFT2Q8B40XJ6QP9TI6@2Ph>30WmoO`:;Rh/2
H61P0UYJFP9IFET2Ogmo0X>3P`;0`<02clo?171`L09nOWh2Jf][0XB4Q0:VYZH2IVIV1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9LG5`200000Q8B4P8000027alO1000008Q8B42T92@0]cL
g0;oool2X:2P0[Fe]@810@42CDe=0P00009XJ6P2UIFE0Tm?C`:ZZZX2W9bL0ZRXZ098B4P2SXj>0SDe
=@9_Kfl27AdM0]oOg`9_Kfl2V9RH0XN7Q`Coool2IVIV0YFEU@8K6a/2;Rh^0/;2`P:4Q8@2?Shn0XZ:
RP9HF5P2?Cdm0[Ng]`9iNGT2G5aL0Xj>SP9nOWh29bLW0Rh^;P9=CDd2UiNG0ZRXZ09LG5`2N7Qh0ZRX
Z09dM7@2JVYZ0S`l?0:YZJT22@T90^OWi`:;Rh/2l?3`00?Lg=`j>SXj>SX00T10@09dM7@2DE5A0P00
00:h^;P2R8R80V5QH@:`/;02JFUY0Rh^;P:b/[8210@40YFEU@9IFET2m?Cd0]KFeP:HV9P2ZZZZ0RPX
:0;@d=0200000WinOP9?Cdl2MWIf0WUiN@9GEeL2oooo0QXJ6P8/;2`2bl_;0Rh^;P9^KVh2IFEU0XR8
R08F5QH2IFEU0U5AD@:RXZ82/[:b0X^;R`:f][H271`L0V5QH@:RXZ82GemO0YFEU@;7alL2W9bL0Y>C
T`:h^;P27AdM0QDE5@:l_;`2`l?30Ve]K@:[Zj/2IfMW0X20P0:JVYX2LG5a0[jn_P;Cdm<5oooo0@00
007oool000Woool300000oooo`030000oooooooo00?oool00`000?ooooooo`02oooo0UaLG0800002
4Q8B0P00008O7al400000R4Q8@:@T902g=cL0_ooo`:PX:02]KFe0P410@9=CDd200000VQXJ0:EUID2
Cdm?0ZZZZP:LW9`2Z:RX0TQ8B0:>SXh2=CDe0Vm_K`8M7Ad2gmoO0Vm_K`:HV9P2QhN71?ooo`9VIVH2
UIFE0Q/K6`8^;Rh2`/;20XB4Q08n?Sh2RXZ:0UQHF08m?Cd2]kNg0WUiN@9LG5`2SXj>0WinOP8W9bL2
;Rh^0Te=C@:GUiL2Z:RX0UaLG09hN7P2Z:RX0WAdM09ZJVX2?3`l0ZVYZ@892@T2inOW0X^;R`;`l?00
0mcLg3Xj>SXj>P02@4100WAdM09ADE4200000[Rh^0:8R8P2HF5Q0[2`/09YJFT2;Rh^0[:b/P8410@2
UIFE0UUIF@;dm?@2e]KF0YRHV0:ZZZX2:2PX0]3@d0800002OWin0Tm?C`9fMWH2NGUi0UMGE`;oool2
6QXJ0R`/;0;;bl/2;Rh^0Vi^KP9UIFD2R8R80QHF5P9UIFD2DE5A0Z:RXP:b/[82Rh^;0[Jf]P8L71`2
HF5Q0Z:RXP9OGel2UIFE0/O7a`:LW9`2Ti>C0[Rh^08M7Ad25ADE0[bl_0;3`l<2KFe]0Z^[Z`9WIfL2
P8200YZJVP9aLG42_[jn0]?Cd`Goool100000Oooo`002Oooo`030000oooooooo00;oool00`000?oo
o`000004oooo00<0003oooooool01?ooo`:e]KD2Zj^[0_ooo`:3Ph<2k>c/0_Gem@;hn?P2oooo0PP8
208Z:RX2d];B1000009mOGd2oooo0_Shn0;oool2Cdm?0UiNGP9cLg<2EUIF0Y^KV`9LG5`20`<30UaL
G0;oool2KVi^0R/[:`8e=CD2FEUI0R0P80:5QHD2>CTi0T92@P9>CTh2QHF50STi>@9hN7P200000Xb<
S0:j^[X2j>SX0ZZZZP8T92@2^KVi0U5AD@93@d<2ZJVY0_KfmP:OWil2RHV90SPh>0:KVi/2@4100V9R
HP:8R8P2RXZ:0UQHF09nOWh2oooo0T];B`:5QHD2UIFE00=eMGF1PH61PH400VIVIP95ADD2BTY:0YNG
U`91@D42hN7Q0W5aL@:<S8`2;bl_0V9RHP:BTY82GemO0VEUI@:m_Kd2J6QX0TQ8B0:MWId2Lg=c0W5a
L@9cLg<2m?Cd0ZRXZ09/K6`2BTY:0/k>cP:LW9`2XJ6Q0Y6AT@9/K6`2_[jn0ZZZZP8G5aL2V9RH0YVI
V@8T92@2GemO0S0`<09mOGd2OWin0XZ:RP;7alL2UIFE0Te=C@8N7Qh2VYZJ0Y6AT@:_[jl2A4A40ZBT
Y09iNGT2bl_;0Z6QX@A^KVh2PH610VYZJP:OWil2^[Zj0VATI09SHf<2M7Ad0W1`L0Goool100000Ooo
o`002_ooo`070000oooooooooooo0000oooo000000Coool200001Oooo`:e]KD2Zj^[0_ooo`:3Ph<2
k>c/0_Gem@;hn?P2oooo0PP8208Z:RX2d];B1000009mOGd2oooo0_Shn0;oool2Cdm?0UiNGP9cLg<2
EUIF0Y^KV`9LG5`20`<30UaLG0;oool2KVi^0R/[:`8e=CD2FEUI0R0P80:5QHD2>CTi0T92@P9>CTh2
QHF50STi>@9hN7P200000Xb<S0:j^[X2j>SX0ZZZZP8T92@2^KVi0U5AD@93@d<2ZJVY0_KfmP:OWil2
RHV90SPh>0:KVi/2@4100V9RHP:8R8P2RXZ:0UQHF09nOWh2oooo0T];B`:5QHD2UIFE00=eMGF1PH61
PH400VIVIP95ADD2BTY:0YNGU`91@D42hN7Q0W5aL@:<S8`2;bl_0V9RHP:BTY82GemO0VEUI@:m_Kd2
J6QX0TQ8B0:MWId2Lg=c0W5aL@9cLg<2m?Cd0ZRXZ09/K6`2BTY:0/k>cP:LW9`2XJ6Q0Y6AT@9/K6`2
_[jn0ZZZZP8G5aL2V9RH0YVIV@8T92@2GemO0S0`<09mOGd2OWin0XZ:RP;7alL2UIFE0Te=C@8N7Qh2
VYZJ0Y6AT@:_[jl2A4A40ZBTY09iNGT2bl_;0Z6QX@A^KVh2PH610VYZJP:OWil2^[Zj0VATI09SHf<2
M7Ad0W1`L0Coool200000Oooo`002_ooo`070000oooooooooooo0000oooo000000Coool00`000?oo
ooooo`02oooo0P000093@d<600000_ooo`8/;2`2oooo0Q@D50;=cLd200000U=CD`80000251@D0_oo
o`:2PX8271`L0PP8209jNWX2<c<c0_ooo`9ZJVX2FEUI0Rh^;P9RHV82VIVI0U=CD`8S8b<2Ph>30Te=
C@:j^[X2o?cl0Xb<S09^KVh2T92@0P00009SHf<2VIVI0SPh>09@D50200000Zn_[`9dM7@2Lg=c0P00
008H61P2Vi^K0STi>@8o?cl2I6AT0Ve]K@;oool2OWin0X:2PP;Zj^X200000_clo08410@2?Cdm0YjN
WP:LW9`2RHV90Vi^KP;@d=000gYjNRHV9RHV9P02Lg=c0RDU9@8n?Sh2;Rh^0YNGU`8410@2YZJV0Z2P
X09gMgL2SXj>0UUIF@:HV9P2CDe=0VIVIP9CDe<2aLG50W1`L0:LW9`2XJ6Q0Rl_;`9jNWX2[:b/0X20
P0;Jf]X2OGem0VEUI@8h>3P2R8R80Q/K6`8000022`/;0^KViP9lO7`2QXJ60YjNWP;oool292@T0Xb<
S0:KVi/2[jn_0WIfMP:PX:02g=cL0WMgM`:a/K42H61P0R/[:`;QhN42Cdm?0WIfMP8h>3P2B4Q80/k>
cP9KFe/2WYjN0]3@d08d=3@2EEEE0/c<c0:XZ:P2OWin0W]kN`Goool100000Oooo`002Oooo`030000
oooo000000;oool00`000?ooo`000004oooo00<0003oooooool00_ooo`800002@d=31P0000;oool2
;2`/0_ooo`8D51@2cLg=0P00009CDe<200000Q@D50;oool2PX:20Q`L708820P2NWYj0S<c<`;oool2
JVYZ0UUIF@8^;Rh2HV9R0YVIV@9CDe<28b<S0X>3P`9=CDd2^[Zj0_clo0:<S8`2KVi^0Y2@T0800002
Hf=S0YVIV@8h>3P2D51@0P0000:_[jl2M7Ad0W=cL`80000261PH0Y^KV`8i>CT2?clo0VATI09]KFd2
oooo0WinOP:2PX82j^[Z0P0000;lo?`210@40Sdm?@:NWYh2W9bL0XV9R@9^KVh2d=3@00=jNWXV9RHV
9RH00W=cL`8U9BD2?Shn0Rh^;P:GUiL210@40ZJVYP:PX:02MgMg0Xj>SP9IFET2V9RH0Te=C@9VIVH2
De=C0/G5a@9`L702W9bL0Z6QX@8_;bl2NWYj0Zb/[0:0P802f][J0WemO@9UIFD2>3Ph0XR8R08K6a/2
00000P/;2`;Vi^H2O7al0XJ6QP:NWYh2oooo0R@T90:<S8`2Vi^K0Zn_[`9fMWH2X:2P0]cLg09gMgL2
/K6a0V1PH08[:b/2hN7Q0Tm?C`9fMWH2>3Ph0TQ8B0;>c/h2Fe]K0YjNWP;@d=02=3@d0UEEE@;<c<`2
Z:RX0WinOP9kNg/5oooo0@00007oool000[oool00`000?ooooooo`02oooo00<0003oooooool00ooo
o`030000oooooooo00;oool2LG5a0RHV9P:YZJT2MWIf0[2`/09CDe<2:b/[0[^k^`800004oooo0Tm?
C`:DU9@2Shn?0P0000:WYjL2@4100VIVIP9MGEd2KFe]0_ooo`9ZJVX2L71`0V1PH0;>c/h2NWYj0]OG
e`;inOT2;2`/0Xb<S0:/[:`2SXj>0TU9B@;5aLD2lo?c0^GUi@800002Kfm_0]OGe`92@T826ATI0XF5
Q@:a/K42<c<c0Y6AT@8h>3P2U9BD0Ve]K@:9RHT2b<S80/k>cP8N7Qh2:RXZ0SPh>09CDe<2Ogmo0TY:
BP9EEED2KVi^0]OGe`9PH602M7Ad0VATI09UIFD00kfm_E1@D51@D002o_kn0WinOP;4a<@2PH610YbL
W09hN7P2KVi^0WYjNP9oOgl2Hf=S0YZJVP9eMGD24A4A0YNGU`:e]KD2HV9R0S/k>`9<C4`2PH610]SH
f0:EUID2IfMW0Sdm?@:m_Kd2`l?30UQHF0:IVIT2FEUI0_Cdm0;GemL2OGem0Sdm?@800002IFEU0P00
00;bl_82OGem0/C4a0Coool2QhN70TI6AP9lO7`2FUYJ0W1`L0:@T902De=C0WAdM0;Ti>@2bl_;0X20
P0:h^;P200000XN7Q`;Lg=`2[:b/0YfMW@;oool2hN7Q0Q0@4093@d<2EeMG1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`9aLG429RHV0ZVYZ@9fMWH2/;2`0U=CD`8[:b/2^k^k0P0000Coool2
Cdm?0YBDU0:?Shl200000ZNWY`90@402IVIV0UeMG@9]KFd2oooo0VYZJP9`L702H61P0/k>cP9jNWX2
emOG0_Win@8/;2`2S8b<0Zb/[0:>SXh2BDU90/G5a@;clo<2iNGU0P00009_Kfl2emOG0T92@P8I6AT2
QHF50[6a/@8c<c<2TI6A0SPh>0:DU9@2KFe]0XV9R@;8b<P2c/k>0QhN7P8Z:RX2>3Ph0U=CD`9oOgl2
BTY:0UEEE@9^KVh2emOG0V1PH09dM7@2I6AT0VEUI@03_KfmD51@D51@00;no_h2OWin0/C4a0:1PH42
W9bL0WQhN09^KVh2NWYj0WmoO`9SHf<2VYZJ0WEeM@8A4A42UiNG0[Fe]@9RHV82>c/k0Ta<C0:1PH42
f=SH0YFEU@9WIfL2?Cdm0[fm_@;3`l<2F5QH0YVIV@9IFET2m?Cd0]OGe`9mOGd2?Cdm0P00009UIFD2
00000_;blP9mOGd2a<C41?ooo`:7QhL2ATI60WalO09JFUX2L71`0Y2@T09CDe<2M7Ad0^CTi0;;bl/2
P8200[Rh^0800002QhN70]cLg0:/[:`2WIfM0_ooo`;QhN42410@0T=3@`9GEeL5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0P0000:VYZH2NGUi0P0000;hn?P2^[Zj0[Vi^@8/;2`2oooo0P00
00;GemL2SHf=0^CTi0;fm_H2jn_[0TA4A0;De=@2YjNW0V9RHP800002oooo0P0000;alO42IfMW0P00
00:WYjL25QHF0Q/K6`:UYJD2U9BD0/W9b@;`l?02cLg=0ZBTY0:e]KD2Q8B40_ooo`8000021`L70SDe
=@9]KFd2Ng]k0YjNWP:<S8`2=CDe0^OWi`8:2PX2N7Qh0_ooo`;lo?`2TI6A0QHF5P800002eMGE0VMW
I`8];Bd2>c/k0UiNGP:9RHT2UiNG0WEeM@9RHV82V9RH0V=SH`03FEUIc<c<c<c<009_Kfl2OWin0Ve]
K@9mOGd200000XV9R@::RXX2NWYj0WinOP820P82A4A40UiNGP;UiND2T92@0RHV9P9aLG4200000Vi^
KP:6QXH2FEUI0XB4Q09HF5P2c/k>0XB4Q0;2`/82lo?c0Shn?P8f=SH20`<30WmoO`9dM7@2Ng]k0YRH
V09LG5`2<30`0P0000;EeMD2Kfm_0UMGE`800002TI6A0ZNWY`:5QHD2Ph>30VUYJ@9eMGD2/k>c0UaL
G0;;bl/2I6AT0[Zj^P8F5QH2kNg]0Xj>SP9:BTX2l_;b0VYZJP:[Zj/2;bl_0U]KF`8c<c<2Y:BT1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`800002YZJV0WUiN@800002n?Sh0[Zj^P:i^KT2
;2`/0_ooo`800002emOG0Xf=S@;Ti>@2m_Kf0^_[j`94A4@2e=CD0ZNWY`9RHV8200000_ooo`800002
lO7a0VMWI`800002YjNW0QHF5P8K6a/2YJFU0YBDU0;9bLT2l?3`0/g=c@:TY:@2]KFe0XB4Q0;oool2
00000PL71`8e=CD2KFe]0W]kN`:NWYh2S8b<0SDe=@;WinL22PX:0WQhN0;oool2o?cl0Y6AT@8F5QH2
00000]GEe@9WIfL2;Bd]0S/k>`9NGUh2RHV90YNGU`9eMGD2HV9R0YRHV09SHf<00eUIFLc<c<c<c002
Kfm_0WinOP9]KFd2OGem0P0000:9RHT2RXZ:0WYjNP9nOWh20P820TA4A09NGUh2iNGU0Y2@T08V9RH2
LG5a0P00009^KVh2QXJ60UUIF@:4Q8@2F5QH0/k>cP:4Q8@2`/;20_?cl`8n?Sh2=SHf0P<30`9oOgl2
M7Ad0W]kN`:HV9P2G5aL0S0`<0800002eMGE0Vm_K`9GEeL200000Y6AT@:WYjL2QHF50X>3P`9YJFT2
MGEe0[>c/`9LG5`2bl_;0VATI0:j^[X25QHF0^g]k@:>SXh2BTY:0_;blP9ZJVX2Zj^[0Rl_;`9KFe/2
<c<c0ZBTY0Goool100000Oooo`005Oooo`030000oooooooo00;oool2mOGe0UMGE`;0`<02OWin0Vm_
K`:7QhL2KFe]0P00008R8R82hn?S0WemO@:>SXh2Jf][0P00009]KFd2F5QH0VATI09CDe<2nOWi0SHf
=P8l?3`2oooo0[fm_@9XJ6P2d=3@0P0000;oool2DE5A0P0000810@42EeMG0WYjNP:6QXH2>CTi0[:b
/P9ADE42oooo0/S8b095ADD2RXZ:0TE5A@:?Shl2M7Ad0XB4Q0;oool2GUiN0UQHF0;1`L42oooo0VQX
J08K6a/2gmoO0/?3``9JFUX2];Bd0_ooo`9HF5P26QXJ0_ooo`:FUYH2Jf][0ZJVYP8i>CT2VIVI00>g
]kNZZZZZZZX00P00008A4A42NWYj0W5aL@:CTi<2=cLg0X:2PP9hN7P2RXZ:0TM7A`8000024a<C0U5A
D@9[Jf/2;Bd]0P00009hN7P2O7al0VYZJP:][Jd2TI6A0Vi^KP:9RHT2GUiN0TM7A`;9bLT200000Sdm
?@:ATI42`<300[bl_0:i^KT2JVYZ0Zj^[P;IfMT2Kfm_0XR8R0:BTY8200000WinOP92@T82NGUi0V1P
H0;Kfm/2?Shn0Z:RXP9aLG42U9BD0UaLG0:5QHD2HF5Q0Zn_[`:d];@2PH610WYjNP:JVYX2PX:20^KV
iP800004^[Zj0P0000Goool100000Oooo`005Oooo`030000oooooooo00;oool2mOGe0UMGE`;0`<02
OWin0Vm_K`:7QhL2KFe]0P00008R8R82hn?S0WemO@:>SXh2Jf][0P00009]KFd2F5QH0VATI09CDe<2
nOWi0SHf=P8l?3`2oooo0[fm_@9XJ6P2d=3@0P0000;oool2DE5A0P0000810@42EeMG0WYjNP:6QXH2
>CTi0[:b/P9ADE42oooo0/S8b095ADD2RXZ:0TE5A@:?Shl2M7Ad0XB4Q0;oool2GUiN0UQHF0;1`L42
oooo0VQXJ08K6a/2gmoO0/?3``9JFUX2];Bd0_ooo`9HF5P26QXJ0_ooo`:FUYH2Jf][0ZJVYP8i>CT2
VIVI00>g]kNZZZZZZZX00P00008A4A42NWYj0W5aL@:CTi<2=cLg0X:2PP9hN7P2RXZ:0TM7A`800002
4a<C0U5AD@9[Jf/2;Bd]0P00009hN7P2O7al0VYZJP:][Jd2TI6A0Vi^KP:9RHT2GUiN0TM7A`;9bLT2
00000Sdm?@:ATI42`<300[bl_0:i^KT2JVYZ0Zj^[P;IfMT2Kfm_0XR8R0:BTY8200000WinOP92@T82
NGUi0V1PH0;Kfm/2?Shn0Z:RXP9aLG42U9BD0UaLG0:5QHD2HF5Q0Zn_[`:d];@2PH610WYjNP:JVYX2
PX:20^KViP800004^[Zj0P0000Goool100000Oooo`005Oooo`030000oooooooo00;oool2iNGU0P00
009>CTh2LG5a0_ooo`9jNWX2`L710Q0@40:e]KD2alO70_ooo`:NWYh200000/G5a@:c/k<20P820VQX
J08^;Rh2IVIV0V9RHP9jNWX2P8200XN7Q`8B4Q82AdM70UIFEP:CTi<2fm_K0]?Cd`9TI6@2`/;20_oo
o`:1PH4200000_7al@800002;2`/0VQXJ09?Cdl2IFEU0S`l?09SHf<2Kfm_0_clo08L71`2=3@d0U]K
F`;Xj>P2YJFU0VYZJP9DE5@2KFe]0T10@0:>SXh2=cLg0Tm?C`:UYJD200000QlO7`9iNGT2_;bl0ZFU
Y@9=CDd2000000>NWYjc/k>c/k<00TA4A0;oool2AdM70U=CD`800002moOg0Zf][@:0P802QXJ60ZZZ
ZP90@402ATI60/k>cP;2`/82HF5Q0_ooo`9;Bd/2alO70ZVYZ@:9RHT2BDU90ZJVYP8l?3`2Y:BT0Y2@
T098B4P2jNWY0QlO7`8j>SX200000_ooo`9gMgL2SXj>0V9RHP8f=SH2oooo0V5QH@;Bd]82Shn?0[>c
/`9ZJVX2ADE50Zj^[P:LW9`2:b/[0VUYJ@;GemL2IfMW0U5AD@8W9bL2Y:BT0WmoO`:5QHD2M7Ad0W5a
L@:HV9P2HV9R0WYjNP;WinL2Cdm?0V5QH@:f][H5oooo0@00007oool001Goool200000oooo`;UiND2
00000Ti>CP9aLG42oooo0WYjNP;1`L42410@0[Fe]@;7alL2oooo0YjNWP800002aLG50[>c/`820P82
J6QX0Rh^;P9VIVH2HV9R0WYjNP:0P802QhN70Q8B4P97AdL2EUIF0Y>CT`;Kfm/2dm?C0VATI0;2`/82
oooo0X61P@800002lO7a0P00008/;2`2J6QX0Tm?C`9UIFD2?3`l0V=SH`9_Kfl2o?cl0Q`L708d=3@2
Fe]K0^SXj0:UYJD2JVYZ0UADE09]KFd2@4100Xj>SP8g=cL2Cdm?0ZFUY@8000027alO0WUiN@:l_;`2
YJFU0Te=C@8000000ijNW[>c/k>c/`02A4A40_ooo`97AdL2De=C0P0000;gmoL2[Jf]0X20P0:6QXH2
ZZZZ0T10@096ATH2c/k>0/;2`P9QHF42oooo0T];B`;7alL2ZJVY0XV9R@99BDT2YZJV0S`l?0:TY:@2
T92@0TQ8B0;YjNT27alO0SXj>P800002oooo0WMgM`:>SXh2HV9R0SHf=P;oool2HF5Q0];BdP:?Shl2
/k>c0VYZJP95ADD2[Zj^0YbLW08[:b/2JFUY0]OGe`9WIfL2DE5A0RLW9`:TY:@2Ogmo0XF5Q@9dM7@2
LG5a0YRHV09RHV82NWYj0^OWi`9?Cdl2HF5Q0[Jf]PCoool200000Oooo`005Oooo`030000oooooooo
00;oool2MWIf0TU9B@:h^;P21`L70ZNWY`:j^[X2e=CD0[2`/0@000022PX:0P0000;Ti>@2[Jf]0/k>
cP;oool4Q8B40]GEe@;oool200000V][J`:?Shl2SXj>0QDE5@9nOWh2gmoO0PH61P:1PH42:2PX0YVI
V@8[:b/2AdM70Zb/[09hN7P29bLW0ZFUY@9TI6@2:RXZ0Xj>SP;oool2nOWi0UEEE@9^KVh2PX:218n?
S`9[Jf/2KFe]0ZZZZP:UYJD2Ogmo0^c/k08n?Sh2moOg0W]kN`:i^KT2QhN70_ooo`8Y:BT2Cdm?0V=S
H`9?Cdl2bLW900=8B4Rb/[:b/[800WQhN0:MWId2W9bL0YBDU0:h^;P2RXZ:0[Ng]`;;bl/2ATI60YVI
V@:5QHD2@4100YbLW08F5QH2clo?0]_Kf`9iNGT2IFEU0V1PH09aLG42GUiN0/K6aP:4Q8@2Bd];0S@d
=09RHV82oooo0W1`L0:8R8P200000_ooo`9dM7@2PX:20V9RHP:`/;02@T920W1`L09VIVH4NGUi0V9R
HP9ZJVX2N7Qh0VEUI@:>SXh2Hf=S0UIFEP:TY:@2GUiN0_ooo`9lO7`2HV9R0P00009lO7`2PX:20WIf
MP9cLg<2^KVi0TM7A`:@T902FEUI0Y:BTPGoool100000Oooo`005Oooo`030000oooooooo00;oool2
MWIf0TU9B@:h^;P21`L70ZNWY`:j^[X2e=CD0[2`/0@000022PX:0P0000;Ti>@2[Jf]0/k>cP;oool4
Q8B40]GEe@;oool200000V][J`:?Shl2SXj>0QDE5@9nOWh2gmoO0PH61P:1PH42:2PX0YVIV@8[:b/2
AdM70Zb/[09hN7P29bLW0ZFUY@9TI6@2:RXZ0Xj>SP;oool2nOWi0UEEE@9^KVh2PX:218n?S`9[Jf/2
KFe]0ZZZZP:UYJD2Ogmo0^c/k08n?Sh2moOg0W]kN`:i^KT2QhN70_ooo`8Y:BT2Cdm?0V=SH`9?Cdl2
bLW900=8B4Rb/[:b/[800WQhN0:MWId2W9bL0YBDU0:h^;P2RXZ:0[Ng]`;;bl/2ATI60YVIV@:5QHD2
@4100YbLW08F5QH2clo?0]_Kf`9iNGT2IFEU0V1PH09aLG42GUiN0/K6aP:4Q8@2Bd];0S@d=09RHV82
oooo0W1`L0:8R8P200000_ooo`9dM7@2PX:20V9RHP:`/;02@T920W1`L09VIVH4NGUi0V9RHP9ZJVX2
N7Qh0VEUI@:>SXh2Hf=S0UIFEP:TY:@2GUiN0_ooo`9lO7`2HV9R0P00009lO7`2PX:20WIfMP9cLg<2
^KVi0TM7A`:@T902FEUI0Y:BTPGoool100000Oooo`005Oooo`030000oooooooo00;oool2QXJ60TQ8
B0:1PH42TI6A0WUiN@9_Kfl2JVYZ0XJ6QP;GemL2^;Rh0YFEU@:TY:@2SHf=0Xb<S09cLg<2Hf=S0W]k
N`9gMgL2LG5a0U1@D0:WYjL2OGem0WUiN@:1PH42dM7A0ZJVYP8;2`/2VYZJ0P00009/K6`2QHF50Xf=
S@9KFe/2NGUi0WAdM0:IVIT2NGUi0VQXJ0:GUiL2?Shn0WAdM09eMGD2QHF50XB4Q09dM7@2MWIf0WEe
M@:3Ph<2PH610W1`L09cLg<2][Jf0VQXJ0:>SXh2FUYJ0WMgM`AaLG42G5aL0Xb<S09gMgL2QHF50VMW
I`:KVi/00hF5QGEeMGEeM@02O7al0WinOP9dM7@2N7Qh0WIfMP9gMgL4P8200WalO09VIVH2J6QX0WIf
MP:EUID2oooo0Xj>SP:<S8`2TI6A0X>3P`9iNGT2PX:20XB4Q09/K6`2LG5a0WalO09_Kfl2W9bL0WAd
M0:a/K42Bd];0Y>CT`9SHf<2QhN70WEeM@9lO7`2Jf][0X^;R`9lO7`2PX:20WinOP9mOGd2P8200Wem
O@9oOgl2P8200WIfMP9nOWh2R8R80Vm_K`:3Ph<2NGUi0X61P@:6QXH2Shn?0WmoO`:0P802O7al0Win
OP9[Jf/2Shn?0WEeM@:1PH42NGUi1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:6QXH2
B4Q80X61P@:ATI42NGUi0Vm_K`9ZJVX2QXJ60]OGe`:h^;P2UIFE0ZBTY0:=SHd2S8b<0W=cL`9SHf<2
Ng]k0WMgM`9aLG42D51@0ZNWY`9mOGd2NGUi0X61P@;AdM42YZJV0P/;2`:JVYX200000Va/K0:5QHD2
SHf=0U]KF`9iNGT2M7Ad0YVIV@9iNGT2J6QX0YNGU`8n?Sh2M7Ad0WEeM@:5QHD2Q8B40WAdM09fMWH2
MGEe0X>3P`:1PH42L71`0W=cL`:f][H2J6QX0Xj>SP9JFUX2MgMg175aL@9LG5`2S8b<0WMgM`:5QHD2
IfMW0Y^KV`03QHF5MGEeMGEe009lO7`2OWin0WAdM09hN7P2MWIf0WMgM`B0P802O7al0VIVIP9XJ6P2
MWIf0YFEU@;oool2SXj>0Xb<S0:ATI42Ph>30WUiN@:2PX82Q8B40Va/K09aLG42O7al0Vm_K`:LW9`2
M7Ad0[6a/@9;Bd/2Ti>C0V=SH`:7QhL2MGEe0WalO09[Jf/2Rh^;0WalO0:2PX82OWin0WemO@:0P802
OGem0WmoO`:0P802MWIf0WinOP:8R8P2Kfm_0X>3P`9iNGT2PH610XJ6QP:?Shl2Ogmo0X20P09lO7`2
OWin0V][J`:?Shl2MGEe0X61P@9iNGT5oooo0@00007oool001Goool00`000?ooooooo`02oooo0[6a
/@:l_;`2Jf][0WIfMP9kNg/2OWin0X:2PP9kNg/2EeMG0V=SH`9]KFd2LG5a0WEeM@:PX:02oooo0V5Q
H@9kNg/2Ogmo0X:2PP:<S8`2JVYZ0W]kN`9nOWh2O7al0WEeM@9;Bd/2LG5a0QdM7@800002MGEe0WAd
M09gMgL2:b/[0^k^kP:MWId2KVi^0U]KF`:9RHT2]kNg0UQHF08k>c/2Shn?0WEeM@9iNGT2Ogmo0Wem
O@9nOWh2NWYj0WUiN@B0P802M7Ad0XF5Q@9eMGD2RHV91WinOP:9RHT2MWIf0WYjNP9iNGT2R8R80XV9
R@03N7QhOGemOGem009lO7`2Ng]k0WmoO`9nOWh2P8200WalO09mOGd2Ogmo0WMgM`:6QXH2De=C0UmO
G`:c/k<2TI6A0T=3@`:d];@2Q8B40WEeM@;;bl/2FUYJ0WAdM0:5QHD2V9RH0YnOW`9PH602TY:B0VYZ
JP:`/;02EUIF18>3P`9kNg/2OGem0WalO0:2PX82MWIf0W]kN`9iNGT6Ng]k0WalO0AkNg/2OWin0W]k
N`9hN7P2P8200WUiN@:3Ph<2NWYj0WUiN@9bLW84Ng]k0WalO09kNg/2Ph>30WEeM@9nOWh2Ng]k0Wem
O@Goool100000Oooo`005Oooo`030000oooooooo00;oool2/K6a0[bl_09[Jf/2MWIf0W]kN`9nOWh2
PX:20W]kN`9GEeL2Hf=S0Ve]K@9aLG42MGEe0Z2PX0;oool2HF5Q0W]kN`9oOgl2PX:20Xb<S09ZJVX2
Ng]k0WinOP9lO7`2MGEe0T];B`9aLG427AdM0P00009eMGD2M7Ad0WMgM`8[:b/2k^k^0YfMW@9^KVh2
Fe]K0XV9R@:g]kL2F5QH0S/k>`:?Shl2MGEe0WUiN@9oOgl2OGem0WinOP9jNWX2NGUi1820P09dM7@2
QHF50WEeM@:9RHT6OWin0XV9R@9fMWH2NWYj0WUiN@:8R8P2RHV900=hN7QmOGemOGd00WalO09kNg/2
Ogmo0WinOP:0P802O7al0WemO@9oOgl2MgMg0XJ6QP9CDe<2GemO0[>c/`:ATI42@d=30[Bd]0:4Q8@2
MGEe0/_;b`9JFUX2M7Ad0XF5Q@:HV9P2WinO0V1PH0:BTY82JVYZ0[2`/09FEUH4Ph>30W]kN`9mOGd2
O7al0X:2PP9fMWH2Ng]k0WUiN@IkNg/2O7al17]kN`9nOWh2Ng]k0WQhN0:0P802NGUi0X>3P`9jNWX2
NGUi0W9bLPAkNg/2O7al0W]kN`:3Ph<2MGEe0WinOP9kNg/2OGem1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`:o_kl2/;2`0W]kN`9oOgl4O7al0WUiN@9jNWX2QhN70XB4Q0:0P802PX:20WIf
MP;4a<@29RHV0[2`/0AkNg/2NWYj0WIfMP:1PH42O7al0W]kN`9/K6`2B4Q80P0000;Vi^H2lo?c0Sdm
?@9VIVH2MWIf0X20P09_Kfl2Shn?0YFEU@8Q8B42YJFU0YBDU09YJFT2oooo0S4a<@9hN7P2NGUi0Wmo
O`9jNWX2O7al0W]kN`9lO7`2OGem17YjNP9oOgl2N7Qh0WinOP9hN7P6Ng]k0WQhN09mOGd2Ogmo0WEe
M@:IVIT210@400=mOGekNg]kNg/00W]kN`9lO7`4Ng]k0WYjNP9lO7`4Ng]k0WemO@9eMGD2LW9b0ZFU
Y@9/K6`2ATI60Te=C@:;Rh/2/[:b0XB4Q0;oool2PX:20YBDU09cLg<2MgMg0/?3``:<S8`2EeMG0ZRX
Z09XJ6P200000X^;R`9_Kfl2P8200WYjNP9lO7`2NWYj0WinOP9lO7`2OGem37alO09kNg/2O7al0Wem
O@9kNg/2OGem0WUiN@9lO7`2OGem0WmoO`QlO7`2NWYj0WinOP9kNg/2O7al0W]kN`Goool100000Ooo
o`005Oooo`030000oooooooo00;oool2_kno0[2`/09kNg/2Ogmo17alO09iNGT2NWYj0XN7Q`:4Q8@2
P8200X:2PP9fMWH2a<C40RHV9P:`/;04Ng]k0WYjNP9fMWH2PH610WalO09kNg/2K6a/0TQ8B0800002
i^KV0_?cl`8m?Cd2IVIV0WIfMP:0P802Kfm_0Xn?S`:EUID28B4Q0ZFUY@:DU9@2JFUY0_ooo`8a<C42
N7Qh0WUiN@9oOgl2NWYj0WalO09kNg/2O7al0WemO@AjNWX2Ogmo0WQhN09nOWh2N7Qh1W]kN`9hN7P2
OGem0WmoO`9eMGD2VIVI0P@41003OGemNg]kNg]k009kNg/2O7al17]kN`9jNWX2O7al17]kN`9mOGd2
MGEe0W9bLP:UYJD2K6a/0TI6AP9=CDd2Rh^;0[:b/P:4Q8@2oooo0X:2PP:DU9@2Lg=c0WMgM`;3`l<2
S8b<0UMGE`:XZ:P2J6QX0P0000:;Rh/2Kfm_0X20P09jNWX2O7al0WYjNP9nOWh2O7al0WemO@alO7`2
Ng]k0WalO09mOGd2Ng]k0WemO@9iNGT2O7al0WemO@9oOgl8O7al0WYjNP9nOWh2Ng]k0WalO09kNg/5
oooo0@00007oool001Goool00`000?ooooooo`04oooo0YFEU@9aLG42OGem17alO0AmOGd2NGUi0W]k
N`9gMgL2Ph>30VQXJ0Coool21@D50W]kN`9lO7`2OGem0WinOP9jNWX2Ng]k0X61P@9]KFd2UiNG0QhN
7P98B4P2F5QH0Slo?`9QHF42PX:20W]kN`;Lg=`2Vi^K0S/k>`:6QXH26ATI0X>3P`9`L702EUIF0X^;
R`:9RHT4Ng]k3WalO09kNg/2O7al0W]kN`9mOGd6O7al0WemO@9kNg/4O7al17inOQ=lO7`2Ng]k0Wem
O@9/K6`2RXZ:0YJFUP9gMgL2_;bl0^k^kP9UIFD2QHF50Zb/[0:KVi/2MWIf0W5aL@;;bl/2NGUi0XJ6
QP8l?3`2TI6A0Xj>SP;Rh^82M7Ad0X:2PP9iNGT2OGem17alO09kNg/DO7al0W]kN`QlO7`4Ng]k2Wal
O09kNg/6O7al1Oooo`400001oooo000Eoooo0P0000Goool2UIFE0W5aL@9mOGd4O7al17emO@9iNGT2
Ng]k0WMgM`:3Ph<2J6QX1?ooo`851@D2Ng]k0WalO09mOGd2OWin0WYjNP9kNg/2PH610Ve]K@:GUiL2
7QhN0TQ8B09HF5P2?clo0V5QH@:2PX82Ng]k0]cLg0:KVi/2>c/k0XJ6QP8I6AT2Ph>30W1`L09FEUH2
Rh^;0XV9R@AkNg/>O7al0W]kN`9lO7`2Ng]k0WemO@IlO7`2OGem0W]kN`AlO7`4OWin4galO09kNg/2
OGem0Va/K0::RXX2UYJF0WMgM`:l_;`2k^k^0VEUI@:5QHD2[:b/0Y^KV`9fMWH2LG5a0/_;b`9iNGT2
QXJ60S`l?0:ATI42SXj>0^;RhP9dM7@2PX:20WUiN@9mOGd4O7al0W]kNaAlO7`2Ng]k27alO0AkNg/:
O7al0W]kN`IlO7`4oooo0P00007oool001Goool00`000?ooooooo`02oooo0P000099BDT2PH610WYj
NP9mOGd4O7al17YjNP9oOgl2O7al0WinOP:1PH42BDU9100000AlO7`4Ng]k17alO09mOGd2PH610]oO
g`;oool2Q8B40[Vi^@9nOWh2K6a/0WemO@9kNg/2ZJVY0XV9R@9^KVh2>SXj0_ooo`;dm?@2I6AT0R@T
909?Cdl2OGem0WYjNPYlO7`4Ng]k17alO09mOGd2OWin0WalO0AkNg/2OGem0W]kN`9lO7`2NGUi0XB4
Q091@D42`/;200=kNg]lO7alO7`03WalO09kNg/2P8200WAdM0;EeMD2@T920YfMW@:FUYH251@D0ZNW
Y`:6QXH2MGEe0WMgM`:QXJ42HV9R0V=SH`9VIVH2/[:b0WMgM`9bLW82:b/[0T];B`8Z:RX2S8b<0WIf
MP9nOWh2Ng]k0WalO09kNg/:O7al0W]kNaQlO7`2NWYj0W]kN`9jNWX2Ng]k17emO@AkNg/2O7al1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`800002BDU90X61P@9jNWX2OGem17alO0AjNWX2
Ogmo0WalO09nOWh2PH610TU9B@@00004O7al17]kN`AlO7`2OGem0X61P@;Ogml2oooo0XB4Q0:i^KT2
OWin0Va/K09mOGd2Ng]k0ZVYZ@:9RHT2KVi^0SXj>P;oool2m?Cd0VATI08T92@2Cdm?0WemO@9jNWX:
O7al17]kN`AlO7`2OGem0WinOP9lO7`4Ng]k0WemO@9kNg/2O7al0WUiN@:4Q8@2@D510/;2`P03Ng]k
O7alO7al00ilO7`2Ng]k0X20P09dM7@2eMGE0T92@P:MWId2UYJF0Q@D50:WYjL2QXJ60WEeM@9gMgL2
XJ6Q0V9RHP9SHf<2IVIV0[:b/P9gMgL2LW9b0R/[:`9;Bd/2:RXZ0Xb<S09fMWH2OWin0W]kN`9lO7`2
Ng]k2WalO09kNg/HO7al0WYjNP9kNg/2NWYj0W]kN`AmOGd4Ng]k0WalO0Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2;Bd]0YJFUP9fMWH2Ogmo0WemO@9oOgl2O7al0WmoO`9lO7`2LW9b0WIf
MP9bLW82S8b<0TE5A@:;Rh/2MWIf0W]kN`9lO7`2OWin0X20P09hN7P2NWYj0WinOP9oOgl2FEUI0]?C
d`99BDT2>c/k0W=cL`:NWYh2MGEe0WYjNP9iNGT2@T920X:2PP9VIVH2VYZJ0W1`L0:_[jl2JVYZ0YBD
U09hN7P2Ogmo1W]kN`AlO7`2OGem0WinOP9jNWX2OGem0WEeM@9`L702Ng]k0X20P09nOWh2NGUi0Wem
O@9kNg/2PX:20WQhN0:5QHD2IfMW00>1PH5lO7alO7`017YjNP9kNg/2O7al0WUiN@AlO7`4NWYj0W]k
N`9fMWH2P8200XJ6QP:1PH42Ogmo0UMGE`9OGel2NWYj0W=cL`:2PX84UYJF0XJ6QP9`L702YjNW0Vm_
K`9eMGD2LW9b0YFEU@9iNGT2Ogmo0WYjNP9nOWh2O7al0WinOP9jNWX6O7al0W]kN`9nOWh6O7al17]k
N`9jNWX2O7al0W]kN`9jNWX2Ng]k0WemO@9jNWX2OGem0X61P@:5QHD2NWYj0X61P@9iNGT2O7al0X20
P09kNg/5oooo0@00007oool001Goool00`000?ooooooo`02oooo0Rd];@:FUYH2MWIf0WmoO`9mOGd2
Ogmo0WalO09oOgl2O7al0W9bLP9fMWH2LW9b0Xb<S095ADD2Rh^;0WIfMP9kNg/2O7al0WinOP:0P802
N7Qh0WYjNP9nOWh2Ogmo0UUIF@;Cdm<2BDU90S/k>`9cLg<2WYjN0WEeM@9jNWX2NGUi0T92@P:2PX82
IVIV0YZJVP9`L702[jn_0VYZJP:DU9@2N7Qh0WmoO`IkNg/4O7al0WemO@9nOWh2NWYj0WemO@9eMGD2
L71`0W]kN`:0P802OWin0WUiN@9mOGd2Ng]k0X:2PP9hN7P2QHF50VMWI`03PH61O7alO7al00AjNWX2
Ng]k0WalO09iNGT4O7al17YjNP9kNg/2MWIf0X20P0:6QXH2PH610WmoO`9GEeL2GemO0WYjNP9cLg<2
PX:219JFUP:6QXH2L71`0ZNWY`9_Kfl2MGEe0W9bLP:EUID2NGUi0WmoO`9jNWX2OWin0WalO09nOWh2
NWYj1WalO09kNg/2OWin1WalO0AkNg/2NWYj0WalO09kNg/2NWYj0W]kN`9mOGd2NWYj0WemO@:1PH42
QHF50WYjNP:1PH42NGUi0WalO0:0P802Ng]k1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`:@T902O7al0]3@d09<C4`2WIfM0Y>CT`9fMWH2][Jf0P00008C4a<2>CTi0WUiN@;0`<02Shn?0_oo
o`9WIfL2S8b<0Tm?C`91@D42ZJVY0RDU9@9QHF42_;bl0V=SH`:][Jd2@T920_ooo`8e=CD2;bl_0_oo
o`9iNGT2LW9b0[Bd]0;oool200000R4Q8@:l_;`2PX:20WalO09aLG42^;Rh0Xj>SP:SXj<2UIFE0XR8
R0:5QHD2N7Qh0XR8R0:=SHd2IVIV0TY:BP:_[jl2mOGe0Y6AT@:RXZ82OWin0Y^KV`9;Bd/2g=cL0TM7
A`8K6a/2Ng]k0UQHF08[:b/00fEUIInOWinOW`02GemO0[Rh^09kNg/2Rh^;0[jn_P:8R8P2fMWI0^;R
hP800002;Rh^0Y>CT`9/K6`2[Zj^0QXJ6P:KVi/2>CTi0YfMW@9^KVh2@T920X>3P`8M7Ad2fMWI0RDU
9@:4Q8@28B4Q0WQhN09^KVh2NWYj0UMGE`:c/k<2_Kfm0/?3``9BDU82IVIV0X:2PP9MGEd2OWin0W1`
L0:1PH42Ph>30W=cL`9dM7@2P8200WinOP9cLg<2M7Ad0XF5Q@:FUYH2Xj>S0_ooo`;2`/82_[jn0[6a
/@:FUYH2/K6a0W5aL@9hN7P2UYJF0W1`L09iNGT2K6a/0X20P0Goool100000Oooo`005Oooo`030000
oooooooo00;oool2T92@0WalO0;@d=02C4a<0YfMW@:CTi<2MWIf0[Jf]P8000024a<C0STi>@9iNGT2
`<300Xn?S`;oool2IfMW0Xb<S09?Cdl2@D510ZVYZ@8U9BD2HF5Q0[bl_09SHf<2[Jf]0T92@P;oool2
=CDe0Rl_;`;oool2NGUi0W9bLP:d];@2oooo0P00008Q8B42_;bl0X:2PP9lO7`2LG5a0[Rh^0:>SXh2
Xj>S0YFEU@:8R8P2QHF50WQhN0:8R8P2SHf=0VIVIP9:BTX2[jn_0_Gem@:ATI42XZ:R0WinOP:KVi/2
Bd];0]cLg097AdL26a/K0W]kN`9HF5P2:b/[00=UIFFOWinOWil00UmOG`:h^;P2Ng]k0X^;R`:n_[h2
R8R80]WIf@;Rh^8200000Rh^;P:CTi<2K6a/0Zj^[P8J6QX2Vi^K0STi>@:MWId2KVi^0T92@P:3Ph<2
7AdM0]WIf@8U9BD2Q8B40R4Q8@9hN7P2KVi^0WYjNP9GEeL2/k>c0[fm_@;3`l<2DU9B0VIVIP:2PX82
GEeM0WinOP9`L702PH610X>3P`9cLg<2M7Ad0X20P09nOWh2Lg=c0WAdM0:5QHD2UYJF0Z>SX`;oool2
`/;20[jn_P:a/K42UYJF0[6a/@9aLG42N7Qh0YJFUP9`L702NGUi0Va/K0:0P805oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0TY:BP8P8202K6a/0YNGU`9`L702Kfm_0ZJVYP:0P802a<C40[Rh
^0:7QhL2V9RH0Ta<C0:[Zj/2Lg=c0Vi^KP9fMWH2UYJF0V][J`9XJ6P2ZJVY0WmoO`9mOGd2FUYJ0R<S
8`800002820P0XB4Q0:MWId2O7al0VMWI`:2PX82i^KV0Xj>SP8T92@2a<C40W9bLP:WYjL2oooo0Q0@
4098B4P2W9bL0XN7Q`9mOGd2OWin0WemO@9kNg/2NWYj0WIfMP:4Q8@2SXj>0QHF5P:?Shl2RHV90VMW
I`9mOGd2KFe]0XN7Q`9WIfL2S8b<0XV9R@9eMGD2clo?0RDU9@03Q8B4L71`L71`009oOgl2KFe]0WUi
N@9lO7`2Ng]k0WAdM09QHF42CDe=0WmoO`9`L704Y:BT0[:b/P:PX:02ZZZZ0Q4A4@:<S8`2RHV90Xb<
S09fMWH2Ph>316m_K`:d];@2L71`0PT92@8E5AD2Rh^;0/k>cP8X:2P2CTi>0VQXJ0:0P802OGem0WIf
MP9mOGd2P82017]kN`9jNWX2P8200X61P@9jNWX2OWin0WemO@:1PH42Ng]k0VATI09lO7`2E5AD0S/k
>`8B4Q82c/k>0WQhN0:0P802NGUi0X20P09^KVh2PH610W]kN`:3Ph<2LW9b1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`9:BTX2820P0Va/K0:GUiL2L71`0Vm_K`:VYZH2P8200/C4a0:h^;P2
QhN70YRHV09<C4`2Zj^[0W=cL`9^KVh2MWIf0YJFUP9[Jf/2J6QX0ZVYZ@9oOgl2OGem0UYJFP8S8b<2
00000R0P80:4Q8@2WIfM0WalO09WIfL2PX:20^KViP:>SXh292@T0/C4a09bLW82YjNW0_ooo`8@4102
B4Q80YbLW0:7QhL2OGem0WinOP9mOGd2Ng]k0WYjNP9fMWH2Q8B40Xj>SP8F5QH2Shn?0XV9R@9WIfL2
OGem0Ve]K@:7QhL2IfMW0Xb<S0:9RHT2MGEe0/o?c`8U9BD00hB4Q71`L71`L002Ogmo0Ve]K@9iNGT2
O7al0W]kN`9dM7@2HF5Q0Te=C@9oOgl2L71`1:BTY0:b/[82X:2P0ZZZZP8A4A42S8b<0XV9R@:<S8`2
MWIf0X>3P`A_Kfl2];Bd0W1`L0892@T25ADE0X^;R`;>c/h2:2PX0Ti>CP9XJ6P2P8200WemO@9fMWH2
OGem0X20P0AkNg/2NWYj0X20P0:1PH42NWYj0WinOP9mOGd2PH610W]kN`9TI6@2O7al0UADE08k>c/2
4Q8B0/k>cP9hN7P2P8200WUiN@:0P802KVi^0X61P@9kNg/2Ph>30W9bLPGoool100000Oooo`005Ooo
o`030000oooooooo00;oool2L71`0T=3@`:6QXH2LG5a0X61P@9nOWh2Rh^;0WmoO`9WIfL2JVYZ0WIf
MP9_Kfl2Vi^K0S/k>`9ADE42QHF50WalO0:3Ph<2Shn?0X:2PP9_Kfl4Ng]k0XF5Q@9?Cdl2KVi^0Xj>
SP9iNGT2Jf][0WEeM@:8R8P2MWIf0W9bLP:n_[h2OWin0Ve]K@:2PX82MgMg0VEUI@95ADD2h>3P0Xn?
S`9lO7`4Ng]k17alO09mOGd2NGUi18B4Q0:NWYh2RHV90WUiN@:2PX82Ng]k0X20P09iNGT2QHF50WIf
MP9eMGD2N7Qh0W]kN`9XJ6P00gQhN820P820P002Ng]k0X61P@AmOGd2PH610WalO0:4Q8@2QhN70WAd
M09fMWH2KFe]0UeMG@;oool2QXJ60_ooo`9oOgl2QhN70WAdM09fMWH2O7al0WMgM`:2PX82NWYj0WAd
M09`L702LW9b0[>c/`99BDT2Bd];0T=3@`:QXJ42NGUi0WalO09iNGT2OWin17YjNPIlO7`4NWYj0Wem
O@9kNg/2O7al17YjNP9kNg/2MgMg0Xj>SP9[Jf/2HF5Q0WEeM@9kNg/2Ogmo17]kN`:0P802Ng]k0Wal
O09jNWX2OWin1Oooo`400001oooo000Eoooo0P0000?oool2L71`0T=3@`:6QXH2LG5a0X61P@9nOWh2
Rh^;0WmoO`9WIfL2JVYZ0WIfMP9_Kfl2Vi^K0S/k>`9ADE42QHF50WalO0:3Ph<2Shn?0X:2PP9_Kfl4
Ng]k0XF5Q@9?Cdl2KVi^0Xj>SP9iNGT2Jf][0WEeM@:8R8P2MWIf0W9bLP:n_[h2OWin0Ve]K@:2PX82
MgMg0VEUI@95ADD2h>3P0Xn?S`9lO7`4Ng]k17alO09mOGd2NGUi18B4Q0:NWYh2RHV90WUiN@:2PX82
Ng]k0X20P09iNGT2QHF50WIfMP9eMGD2N7Qh0W]kN`9XJ6P00gQhN820P820P002Ng]k0X61P@AmOGd2
PH610WalO0:4Q8@2QhN70WAdM09fMWH2KFe]0UeMG@;oool2QXJ60_ooo`9oOgl2QhN70WAdM09fMWH2
O7al0WMgM`:2PX82NWYj0WAdM09`L702LW9b0[>c/`99BDT2Bd];0T=3@`:QXJ42NGUi0WalO09iNGT2
OWin17YjNPIlO7`4NWYj0WemO@9kNg/2O7al17YjNP9kNg/2MgMg0Xj>SP9[Jf/2HF5Q0WEeM@9kNg/2
Ogmo17]kN`:0P802Ng]k0WalO09jNWX2OWin1?ooo`800001oooo000Eoooo00<0003oooooool00_oo
o`9ADE42J6QX0X61P@9mOGd6Ng]k0WYjNP:2PX82PH610WmoO`9mOGd2M7Ad0W]kN`:g]kL2PH610Wal
O09kNg/2N7Qh0WYjNP9oOgl2O7al0W]kN`9iNGT2H61P0TE5A@9@D502Jf][0]_Kf`9mOGd4O7al0/K6
aP:f][H2GUiN0ZJVYP9ZJVX2Shn?0U]KF`:ATI42Ogmo0WAdM09mOGd<O7al17]kN`9aLG42NGUi0Wal
O09jNWX2O7al0W]kN`9mOGd2NGUi0WemO@9oOgl2Ng]k0XJ6QP9QHF400gemOG]kNg]kN`02O7al0WYj
NPAkNg/2NGUi0WalO09hN7P2MgMg1WinOP:4Q8@2OGem0R@T90:SXj<2IVIV0Xn?S`:0P802WinO0VUY
J@9lO7`2NWYj0WemO@B2PX82MgMg0T51@@9JFUX2QhN70WalO09eMGD2Ng]k0WalO09mOGd2Ng]k0Wem
OAYlO7`2MGEe17]kN`:0P804Ng]k17alO09kNg/6O7al0W]kN`Goool100000Oooo`005Oooo`030000
oooooooo00;oool2DE5A0VQXJ0:1PH42OGem1W]kN`9jNWX2PX:20X61P@9oOgl2OGem0WAdM09kNg/2
]kNg0X61P@9lO7`2Ng]k0WQhN09jNWX2Ogmo0WalO09kNg/2NGUi0V1PH095ADD2D51@0V][J`;Kfm/2
OGem17alO0;6a/H2][Jf0UiNGP:VYZH2JVYZ0Xn?S`9KFe/2TI6A0WmoO`9dM7@2OGem37alO0AkNg/2
LG5a0WUiN@9lO7`2NWYj0WalO09kNg/2OGem0WUiN@9mOGd2Ogmo0W]kN`:6QXH2HF5Q00=mOGekNg]k
Ng/00WalO09jNWX4Ng]k0WUiN@9lO7`2N7Qh0WMgM`InOWh2Q8B40WemO@8T92@2Xj>S0VIVIP:?Shl2
P8200YnOW`9YJFT2O7al0WYjNP9mOGd4PX:20WMgM`91@D42FUYJ0XN7Q`9lO7`2MGEe0W]kN`9lO7`2
OGem0W]kN`9mOGdJO7al0WEeM@AkNg/2P82017]kN`AlO7`2Ng]k1WalO09kNg/5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0]KFeP:RXZ82MGEe0WemO@QlO7`4Ng]k0WYjNP:0P802KVi^0YVI
V@8[:b/27AdM0W]kN`9lO7`2OGem0WalO09kNg/4O7al0WUiN@9hN7P23`l?0_ooo`:CTi<2?Cdm17Yj
NP9mOGd2SXj>0/o?c`9KFe/2Jf][0Zj^[P9gMgL2d=3@0];BdP:6QXH2P8200W]kNa1lO7`2P82037al
O09mOGd6Ng]k0X61P@9/K6`00g]kNgalO7alO008O7al0WemO@9lO7`4OGem17]kN`9oOgl2U9BD0Va/
K09PH602`L710Te=C@9QHF42EUIF0YfMW@9<C4`2GUiN0WMgM`9eMGD2U9BD0XV9R@:<S8`2EEEE0TQ8
B0:ATI42M7Ad0X20P2AlO7`2OWin17alO09kNg/BO7al1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`;Fe]H2XZ:R0WEeM@9mOGd8O7al17]kN`9jNWX2P8200Vi^KP:IVIT2:b/[0QdM7@9kNg/2
O7al0WemO@9lO7`2Ng]k17alO09iNGT2N7Qh0Pl?3`;oool2Ti>C0Sdm?@AjNWX2OGem0Xj>SP;?cll2
Fe]K0V][J`:^[Zh2MgMg0]3@d0;Bd]82QXJ60X20P09kNg/@O7al0X20P0alO7`2OGem1W]kN`:1PH42
K6a/00=kNg]lO7alO7`027alO09mOGd2O7al17emO@AkNg/2Ogmo0YBDU09/K6`2H61P0/71`@9=CDd2
HF5Q0UIFEP:MWId2C4a<0UiNGP9gMgL2MGEe0YBDU0:9RHT2S8b<0UEEE@98B4P2TI6A0WAdM0:0P80T
O7al0WinOPAlO7`2Ng]k4WalO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2F5QH0[Ng
]`9gMgL2OWin0W]kN`YlO7`2NWYj0X20P09aLG42UiNG0U]KF`;]kNd2Ng]k0WalO09kNg/6O7al0Wem
O@9jNWX2Xj>S0VATI09jNWX2n?Sh0YfMW@:VYZH2M7Ad0WmoO`:f][H2g=cL0[Vi^@8i>CT2d];B0[bl
_09aLG42Ph>30Z2PX09bLW82Ogmo0W]kN`ilO7`2NWYj4WalO09oOgl2PX:20]CDe0mlO7`2Ng]k0Wal
O09jNWX2Q8B40U]KF`;Hf=P2B4Q80WYjNP800002^KVi0Tm?C`9@D502ZJVY0VYZJP9aLG42K6a/0Ve]
K@;oool2[Jf]0V][J`:DU9@2L71`0X20P09fMWH2OGem0W]kNb9lO7`2Ng]k67alO0Goool100000Ooo
o`005Oooo`030000oooooooo00;oool2F5QH0[Ng]`9gMgL2OWin0W]kN`YlO7`2NWYj0X20P09aLG42
UiNG0U]KF`;]kNd2Ng]k0WalO09kNg/6O7al0WemO@9jNWX2Xj>S0VATI09jNWX2n?Sh0YfMW@:VYZH2
M7Ad0WmoO`:f][H2g=cL0[Vi^@8i>CT2d];B0[bl_09aLG42Ph>30Z2PX09bLW82Ogmo0W]kN`ilO7`2
NWYj4WalO09oOgl2PX:20]CDe0mlO7`2Ng]k0WalO09jNWX2Q8B40U]KF`;Hf=P2B4Q80WYjNP800002
^KVi0Tm?C`9@D502ZJVY0VYZJP9aLG42K6a/0Ve]K@;oool2[Jf]0V][J`:DU9@2L71`0X20P09fMWH2
OGem0W]kNb9lO7`2Ng]k67alO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2MgMg0Ve]
K@:2PX82NWYj4WalO0:<S8`29BDU0P0000ilO7`2N7Qh0W1`L0:4Q8@2g=cL0UUIF@:ZZZX2Jf][0Win
OP9kNg/2YjNW0]7Ad@9iNGT2@D510TU9B@93@d<2UIFE0WUiN@9RHV82NWYj0W]kNbAlO7`2NWYj0W5a
L@91@D4AO7al0W]kN`9oOgl2LW9b0[fm_@:KVi/2L71`0ZFUY@8];Bd2`/;20/C4a09MGEd2Ph>30VIV
IP:<S8`2j^[Z0R8R8P8H61P2K6a/0Xn?S`9^KVh2RHV90TE5A@:3Ph<2N7Qh0WemO@9kNg/jO7al1Ooo
o`400001oooo000:oooo0`0000;oool00`000?ooooooo`03oooo00<0003oooooool00_ooo`9gMgL2
KFe]0X:2PP9jNWXBO7al0Xb<S08U9BD200003WalO09hN7P2L71`0XB4Q0;Lg=`2FEUI0ZZZZP9[Jf/2
OWin0W]kN`:WYjL2dM7A0WUiN@91@D42BDU90T=3@`:EUID2NGUi0V9RHP9jNWX2Ng]k97alO09jNWX2
LG5a0T51@A5lO7`2Ng]k0WmoO`9bLW82_Kfm0Y^KV`9`L702YJFU0Rd];@;2`/82a<C40UeMG@:3Ph<2
IVIV0Xb<S0;Zj^X28R8R0QPH609/K6`2Shn?0Vi^KP:9RHT2ADE50X>3P`9hN7P2OGem0W]kNcYlO7`5
oooo0@00007oool000_oool01P000?ooooooo`000?ooo`0000Coool00`000?ooooooo`04oooo0Z:R
XP9fMWH2OGem2WalO09kNg/2OWin0WAdM0:NWYh2[Jf]0_Cdm0;oool2Ng]k2WalO09jNWX2LW9b0S@d
=0;0`<02=CDe0WAdM08l?3`2Lg=c0W]kN`9mOGd2JVYZ0X61P@;oool2YJFU0_ooo`:IVIT2FUYJ0[Bd
]09gMgL2P8200W]kNb1lO7`2Ng]k0WmoO`9eMGD2/k>c0UaLG0mlO7`2Ng]k0WinOP9fMWH2QXJ60P00
009TI6@2LW9b0QTI6@9BDU82CDe=0^[ZjP800002UIFE0]cLg0:YZJT2e=CD0Zn_[`;>c/h2LW9b0Xb<
S0:ZZZX2KVi^0XV9R@9iNGT2OGem0W]kNcalO7`5oooo0@00007oool000Woool4000000Coool0003o
ool00004oooo0P0000Goool2XZ:R0WIfMP9mOGd:O7al0W]kN`9nOWh2M7Ad0YjNWP:][Jd2m?Cd0_oo
o`9kNg/:O7al0WYjNP9bLW82=3@d0/30`08e=CD2M7Ad0S`l?09cLg<2Ng]k0WemO@9ZJVX2PH610_oo
o`:UYJD2oooo0YVIV@9JFUX2];Bd0WMgM`:0P802Ng]k87alO09kNg/2Ogmo0WEeM@:c/k<2G5aL3gal
O09kNg/2OWin0WIfMP:6QXH200000VATI09bLW826ATI0U9BDP9=CDd2j^[Z0P0000:EUID2g=cL0ZVY
Z@;De=@2[jn_0/k>cP9bLW82S8b<0ZZZZP9^KVh2RHV90WUiN@9mOGd2Ng]k?7alO0Coool200000Ooo
o`002Oooo`030000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool01?ooo`8d=3@2
QXJ60WQhN09mOGd8O7al0W]kN`9oOgl2MGEe0XJ6QP8K6a/2Zj^[0Tm?C`9mOGd:O7al0WUiN@:6QXH2
A4A40YJFUP;oool2PH610^WYj@9JFUX2RXZ:0WQhN09]KFd2D51@0S8b<P9gMgL2[jn_0R`/;09[Jf/2
=CDe0YBDU09oOglVO7al0WYjNP9nOWh2F5QH5GalO09jNWX2T92@0WalO0;GemL2N7Qh0YBDU0:LW9`2
EEEE0VATI0:KVi/28b<S0W5aL@:^[Zh2MgMg0W=cL`:2PX82S8b<0UQHF0:0P802]KFe0WQhN09oOgl2
NWYj?7alO0Goool100000Oooo`002_ooo`800002oooo00<0003oool000001?ooo`030000oooooooo
00Coool2=3@d0XJ6QP9hN7P2OGem27alO09kNg/2Ogmo0WEeM@:6QXH26a/K0Z^[Z`9?Cdl2OGem2Wal
O09iNGT2QXJ60TA4A0:FUYH2oooo0X61P@;YjNT2FUYJ0XZ:RP9hN7P2KFe]0U1@D08b<S82MgMg0Zn_
[`8/;2`2Jf][0SDe=@:DU9@2Ogmo9WalO09jNWX2OWin0UQHF1ElO7`2NWYj0Y2@T09lO7`2emOG0WQh
N0:DU9@2W9bL0UEEE@9TI6@2Vi^K0R<S8`9aLG42[Zj^0WMgM`9cLg<2PX:20Xb<S09HF5P2P8200[Fe
]@9hN7P2Ogmo0WYjNSalO7`5oooo0@00007oool000_oool01@000?ooooooooooo`000005oooo00<0
003oooooool00_ooo`8J6QX2ZZZZ0W9bLP9oOgl2Ng]k37alO0AmOGd2HF5Q0P0000;bl_82Ng]k2Wal
O09kNg/2MWIf0VQXJ0820P82ATI60_ooo`93@d<2XJ6Q0W=cL`9oOgl26ATI0S@d=0A`L702GUiN0Vi^
KP9lO7`2@D510WemO@9hN7P2OGem0W]kNb1lO7`2OGem0WQhN0:BTY82BTY:4galO09jNWX2OGem0UMG
E`9kNg/2l?3`0_ooo`:OWil2eMGE0Zj^[P9jNWX2/k>c0_7al@800002Q8B40YJFUP:;Rh/2g]kN0Ve]
K@9HF5P2FUYJ0V5QH@:0P802Ng]k?WalO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2
6QXJ0ZZZZP9bLW82Ogmo0W]kN`alO7`4OGem0V5QH@800002l_;b0W]kN`YlO7`2Ng]k0WIfMP9XJ6P2
0P820TI6AP;oool2@d=30Z6QX@9cLg<2Ogmo0QTI6@8d=3@4L71`0UiNGP9^KVh2O7al0T51@@9mOGd2
N7Qh0WemO@9kNg/PO7al0WemO@9hN7P2TY:B0TY:BQ=lO7`2NWYj0WemO@9GEeL2Ng]k0_3`l0;oool2
WinO0]GEe@:^[Zh2NWYj0[>c/`;alO4200000XB4Q0:FUYH2Rh^;0]kNgP9]KFd2F5QH0UYJFP9QHF42
P8200W]kNcilO7`5oooo0@00007oool001Goool00`000?ooooooo`02oooo0P00008R8R82S8b<0WQh
N0alO7`2NWYj0WinOP9lO7`2/[:b0Q`L7090@402OGem27alO09kNg/2OWin0WAdM0:TY:@2f=SH0UeM
G@8E5AD2moOg0W]kN`:3Ph<2N7Qh0UIFEP:ZZZX2=SHf0WMgM`:k^k/200000W=cL`:<S8`2Z:RX0WIf
MP9nOWh2Ng]k87alO0AkNg/2DE5A0QXJ6Q5lO7`2NWYj0X>3P`9UIFD2oooo0WUiN@92@T82JFUY0UMG
E`9iNGT210@40Zf][@:@T902J6QX0VYZJP:o_kl2>CTi0P0000;0`<02`l?30X^;R`800002Q8B40WEe
M@9oOgl2Ng]k?7alO0Goool100000Oooo`005Oooo`030000oooooooo00;oool200000R8R8P:<S8`2
N7Qh37alO09jNWX2OWin0WalO0:b/[8271`L0T10@09mOGd8O7al0W]kN`9nOWh2M7Ad0ZBTY0;Hf=P2
GEeM0QDE5@;gmoL2Ng]k0X>3P`9hN7P2EUIF0ZZZZP8f=SH2MgMg0[^k^`800002Lg=c0Xb<S0:XZ:P2
MWIf0WinOP9kNg/PO7al17]kN`9ADE426QXJ4GalO09jNWX2Ph>30VEUI@;oool2NGUi0T92@P9YJFT2
EeMG0WUiN@8410@2[Jf]0Y2@T09XJ6P2JVYZ0[no_`8i>CT200000/30`0;3`l<2Rh^;0P0000:4Q8@2
MGEe0WmoO`9kNg/lO7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9oOgl2MgMg0Wem
O@9kNg/>O7al0W]kN`9mOGd2KFe]0YJFUP8K6a/<O7al0W]kN`9mOGd2M7Ad0Vm_K`95ADD2MWIf0UYJ
FP9TI6@2OGem0W]kN`8n?Sh2MGEe0U]KF`8_;bl2[jn_0YZJVP9=CDd2moOg0WQhN0:2PX82NGUi0Wem
OB9lO7`2Ng]k0XN7Q`::RXXCO7al0W]kN`9nOWh2Kfm_0Z:RXP800002Ti>C0T10@09@D502OGem0Te=
C@97AdL2UYJF0Xb<S09/K6`4oooo0]GEe@;/k>`2hN7Q0W]kN`:ATI42MWIf0WemO@9kNg/lO7al1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`9oOgl2MgMg0WemO@9kNg/>O7al0W]kN`9mOGd2
KFe]0YJFUP8K6a/<O7al0W]kN`9mOGd2M7Ad0Vm_K`95ADD2MWIf0UYJFP9TI6@2OGem0W]kN`8n?Sh2
MGEe0U]KF`8_;bl2[jn_0YZJVP9=CDd2moOg0WQhN0:2PX82NGUi0WemOB9lO7`2Ng]k0XN7Q`::RXXC
O7al0W]kN`9nOWh2Kfm_0Z:RXP800002Ti>C0T10@09@D502OGem0Te=C@97AdL2UYJF0Xb<S09/K6`4
oooo0]GEe@;/k>`2hN7Q0W]kN`:ATI42MWIf0WemO@9kNg/lO7al1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`8R8R82MWIf3WalO09kNg/2OWin0WEeM@:?Shl2=3@d0WemO@:DU9@<O7al0WYj
NP:4Q8@2KVi^0_ooo`:_[jl2KVi^0WemO@:;Rh/2NGUi0WalO09cLg<27AdM0/g=c@9kNg/2MGEe0UIF
EP:f][H2IfMW0WmoO`9jNWX2OGem5WalO09kNg/>O7al0WEeM@:>SXhEO7al0W]kN`:3Ph<2NGUi0WIf
MP:7QhL2NGUi0UMGE`:5QHD2Rh^;0YRHV09SHf<2a<C40/30`0:][Jd2VYZJ0Zn_[`:3Ph<2MWIf0W5a
L@9hN7P2OGem<WalO09kNg/2O7al17]kN`IlO7`5oooo0@00007oool001Goool200000oooo`8R8R82
MWIf3WalO09kNg/2OWin0WEeM@:?Shl2=3@d0WemO@:DU9@<O7al0WYjNP:4Q8@2KVi^0_ooo`:_[jl2
KVi^0WemO@:;Rh/2NGUi0WalO09cLg<27AdM0/g=c@9kNg/2MGEe0UIFEP:f][H2IfMW0WmoO`9jNWX2
OGem5WalO09kNg/>O7al0WEeM@:>SXhEO7al0W]kN`:3Ph<2NGUi0WIfMP:7QhL2NGUi0UMGE`:5QHD2
Rh^;0YRHV09SHf<2a<C40/30`0:][Jd2VYZJ0Zn_[`:3Ph<2MWIf0W5aL@9hN7P2OGem<WalO09kNg/2
O7al17]kN`IlO7`4oooo0P00007oool001Goool00`000?ooooooo`02oooo0YjNWP9jNWX2OGem1Wal
O09kNg/2OWin0WmoO`9mOGd2O7al0WemO@9hN7P2RXZ:0Y6AT@:KVi/2O7al17]kN`9fMWH2NWYj0W]k
N`9lO7`2Ng]k0WalO09QHF42S8b<0XF5Q@:0P802MgMg0WemO@9kNg/2NGUi0XB4Q09mOGd2QXJ60Va/
K09`L702MWIf0X>3P`9kNg/DO7al0W]kN`B0P802O7al0WemO@alO7`2OWin0WYjNP]lO7`2Ng]k2Wal
O09iNGT2OWin0WQhN09iNGT2Ph>30XF5Q@:2PX82OGem0WinOP9fMWH4Ng]k0WinOP9hN7P2M7Ad0Wmo
O`:1PH42Ogmo0WemO@9kNg/^O7al0X20P0:3Ph<4Ng]k0X20P09nOWh2Ng]k17alO0Goool100000Ooo
o`005Oooo`030000oooooooo00;oool2WYjN0WYjNP9mOGd6O7al0W]kN`9nOWh2Ogmo0WemO@9lO7`2
OGem0WQhN0::RXX2TI6A0Y^KV`9lO7`4Ng]k0WIfMP9jNWX2Ng]k0WalO09kNg/2O7al0V5QH@:<S8`2
QHF50X20P09gMgL2OGem0W]kN`9iNGT2Q8B40WemO@:6QXH2K6a/0W1`L09fMWH2Ph>30W]kNaAlO7`2
Ng]k1820P09lO7`2OGem37alO09nOWh2NWYj2galO09kNg/:O7al0WUiN@9nOWh2N7Qh0WUiN@:3Ph<2
QHF50X:2PP9mOGd2OWin0WIfMPAkNg/2OWin0WQhN09dM7@2Ogmo0X61P@9oOgl2OGem0W]kNbilO7`2
P8200X>3P`AkNg/2P8200WinOP9kNg/4O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`9bLW82O7al0W]kN`9lO7`2Ng]k0WalO09jNWX2QXJ60Xj>SPAjNWX2NGUi0WinOP9gMgL2N7Qh0WYj
NPAlO7`2NWYj0VMWI`9kNg/2OWin0W]kN`9jNWX2Ng]k0XF5Q@9jNWX2NGUi0W]kN`9mOGd4O7al0Wem
O@9fMWH2Ogmo0WYjNP9mOGd2O7al0X20P09iNGT:O7al0WemO@AlO7`4Ng]k0WemO@9lO7`2SHf=0Wmo
O`:4Q8@2Ogmo0WUiN@9nOWh2O7al0WemO@9kNg/2O7al0W]kN`9lO7`3NWYj0WemO@9lO7`4OGem0Wmo
O`QlO7`2Ng]k17alO0AmOGd2Ng]k0WQhN09jNWX4OGem0W]kN`9oOgl2P82017alO09oOgl6NWYj0W]k
N`AlO7`2OGem0W]kN`9mOGd2Ng]k1WalO0AkNg/:O7al17]kN`QlO7`2OGem0WAdM0:ATI42SHf=0Wal
O09oOgl2NGUi0WAdM0AlO7`2OGem1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9bLW82
O7al0W]kN`9lO7`2Ng]k0WalO09jNWX2QXJ60Xj>SPAjNWX2NGUi0WinOP9gMgL2N7Qh0WYjNPAlO7`2
NWYj0VMWI`9kNg/2OWin0W]kN`9jNWX2Ng]k0XF5Q@9jNWX2NGUi0W]kN`9mOGd4O7al0WemO@9fMWH2
Ogmo0WYjNP9mOGd2O7al0X20P09iNGT:O7al0WemO@AlO7`4Ng]k0WemO@9lO7`2SHf=0WmoO`:4Q8@2
Ogmo0WUiN@9nOWh2O7al0WemO@9kNg/2O7al0W]kN`9lO7`3NWYj0WemO@9lO7`4OGem0WmoO`QlO7`2
Ng]k17alO0AmOGd2Ng]k0WQhN09jNWX4OGem0W]kN`9oOgl2P82017alO09oOgl6NWYj0W]kN`AlO7`2
OGem0W]kN`9mOGd2Ng]k1WalO0AkNg/:O7al17]kN`QlO7`2OGem0WAdM0:ATI42SHf=0WalO09oOgl2
NGUi0WAdM0AlO7`2OGem1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9mOGd2O7al0Win
OP9lO7`2OGem0WalO09oOgl2MgMg0YnOW`:RXZ82O7al0X20P0AkNg/2Ogmo0WYjNP9lO7`2Ng]k0Wmo
O`:ATI42EUIF0WmoO`9nOWh2P8200WUiN@9mOGd2P8200W]kN`9nOWh2OGem0WYjNPAiNGT2Ogmo0Win
OPAoOgl2Ng]k0WUiN@AnOWh2Ng]k17alO09kNg/2N7Qh17alO09nOWh2Ogmo0WQhN09mOGd2C4a<0TA4
A0:a/K42S8b<0WalO09cLg<2Ogmo0WYjNP9nOWh2O7al17]kN`03Q8B4MWIfMWIf009hN7P2Ng]k0Wal
O09jNWX2MWIf17alO09jNWX2O7al0WemO@9nOWh2NGUi0WemO@9jNWX4O7al0WmoO`AlO7`2PH610Wal
O09hN7P2O7al0WYjNP9hN7P2OGem0WinOP9mOGd2P82017alO09fMWH2Ogmo0WMgM`9mOGd2NWYj0W]k
N`9lO7`4OWin0WemO@9kNg/4O7al0W]kN`AnOWh4O7al17]kN`9gMgL2S8b<0U]KF`9`L702WYjN0VYZ
JP;Hf=P2QhN70WYjNP:1PH42MgMg1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9mOGd2
O7al0WinOP9lO7`2OGem0WalO09oOgl2MgMg0YnOW`:RXZ82O7al0X20P0AkNg/2Ogmo0WYjNP9lO7`2
Ng]k0WmoO`:ATI42EUIF0WmoO`9nOWh2P8200WUiN@9mOGd2P8200W]kN`9nOWh2OGem0WYjNPAiNGT2
Ogmo0WinOPAoOgl2Ng]k0WUiN@AnOWh2Ng]k17alO09kNg/2N7Qh17alO09nOWh2Ogmo0WQhN09mOGd2
C4a<0TA4A0:a/K42S8b<0WalO09cLg<2Ogmo0WYjNP9nOWh2O7al17]kN`03Q8B4MWIfMWIf009hN7P2
Ng]k0WalO09jNWX2MWIf17alO09jNWX2O7al0WemO@9nOWh2NGUi0WemO@9jNWX4O7al0WmoO`AlO7`2
PH610WalO09hN7P2O7al0WYjNP9hN7P2OGem0WinOP9mOGd2P82017alO09fMWH2Ogmo0WMgM`9mOGd2
NWYj0W]kN`9lO7`4OWin0WemO@9kNg/4O7al0W]kN`AnOWh4O7al17]kN`9gMgL2S8b<0U]KF`9`L702
WYjN0VYZJP;Hf=P2QhN70WYjNP:1PH42MgMg1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`9nOWh4Ph>30X61P@9iNGT2P8200W9bLP;6a/H2no_k0SLg=`9TI6@2>CTi0XB4Q0:3Ph<2Ogmo0X:2
PP:1PH42MWIf0Xn?S`9VIVH2moOg0[>c/`:1PH42EUIF0XZ:RP9bLW82PH610X:2PP9kNg/2MgMg0WYj
NP:0P802MgMg0WYjNP9kNg/2OGem0WalO09nOWh2Ogmo0WalO09jNWX2OGem0WmoO`9dM7@4OGem17]k
N`9lO7`2OGem0X20P0:NWYh2UIFE0Vm_K`;=cLd2][Jf0VMWI`:6QXH2Rh^;0Z6QX@9aLG42PX:20W1`
L09mOGd00f=SHdM7AdM7A`02RHV90X:2PP:LW9`2TY:B0Z^[Z`9gMgL2Ng]k0WinOPB0P802MGEe0XF5
Q@:0P802QhN70WYjNP9hN7P2O7al0WMgM`9mOGd2MGEe1861P@9lO7`2PX:20XF5Q@9mOGd2M7Ad0X61
P@9lO7`2KVi^0WmoO`9mOGd2N7Qh0W]kN`9mOGd2R8R80WalO0AjNWX2OWin0WIfMP:1PH42Ng]k0WYj
NP9oOgl4OGem0WQhN09mOGd2P8200WinOP:5QHD2;Bd]0_ooo`8^;Rh2D51@0]GEe@9UIFD2][Jf0Y:B
TP:=SHd2NWYj1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9nOWh4Ph>30X61P@9iNGT2
P8200W9bLP;6a/H2no_k0SLg=`9TI6@2>CTi0XB4Q0:3Ph<2Ogmo0X:2PP:1PH42MWIf0Xn?S`9VIVH2
moOg0[>c/`:1PH42EUIF0XZ:RP9bLW82PH610X:2PP9kNg/2MgMg0WYjNP:0P802MgMg0WYjNP9kNg/2
OGem0WalO09nOWh2Ogmo0WalO09jNWX2OGem0WmoO`9dM7@4OGem17]kN`9lO7`2OGem0X20P0:NWYh2
UIFE0Vm_K`;=cLd2][Jf0VMWI`:6QXH2Rh^;0Z6QX@9aLG42PX:20W1`L09mOGd00f=SHdM7AdM7A`02
RHV90X:2PP:LW9`2TY:B0Z^[Z`9gMgL2Ng]k0WinOPB0P802MGEe0XF5Q@:0P802QhN70WYjNP9hN7P2
O7al0WMgM`9mOGd2MGEe1861P@9lO7`2PX:20XF5Q@9mOGd2M7Ad0X61P@9lO7`2KVi^0WmoO`9mOGd2
N7Qh0W]kN`9mOGd2R8R80WalO0AjNWX2OWin0WIfMP:1PH42Ng]k0WYjNP9oOgl4OGem0WQhN09mOGd2
P8200WinOP:5QHD2;Bd]0_ooo`8^;Rh2D51@0]GEe@9UIFD2][Jf0Y:BTP:=SHd2NWYj1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9`L702Shn?0[:b/P:BTY82UiNG0X:2PP:m_Kd2>CTi0_oo
o`:YZJT28B4Q0SXj>P:DU9@2CDe=0/_;b`9`L702RHV90UQHF09BDU82clo?0UIFEP:JVYX2FEUI0V][
J`9BDU82/k>c0^KViP9dM7@2Z:RX0Xj>SP9@D502HF5Q0S<c<`::RXX2^k^k0/?3``:b/[82L71`0W5a
L@::RXX2YJFU0WAdM0:>SXh2LW9b0V5QH@8Z:RX2QhN70WMgM`BOWil2NGUi0X^;R`:b/[82QhN70_3`
l08b<S82=CDe0SXj>P8f=SH2i>CT0WQhN0:PX:02;Bd]0WYjNP03ZJVY^;Rh^;Rh008O7al2QXJ60_gm
o@:^[Zh2QHF50VEUI@9/K6`2E5AD0X^;R`:5QHD2[jn_0U1@D0:e]KD2L71`0YRHV09JFUX2^KVi0WIf
MP:3Ph<2c/k>0WmoO`9EEED2Q8B40UmOG`9XJ6P2Y:BT0Zj^[P:MWId2dM7A0U1@D09_Kfl22@T90[^k
^`8E5AD2UiNG0Vi^KP9LG5`2O7al0Zn_[`:`/;02UYJF0W=cL`9nOWh2PX:20Vm_K`:`/;02];Bd0Vi^
KPB5QHD2M7Ad0X20P09<C4`28b<S0[^k^`:UYJD2>CTi0P0000:0P802EEEE0TA4A08c<c<5oooo0@00
007oool001Goool200000oooo`9`L702Shn?0[:b/P:BTY82UiNG0X:2PP:m_Kd2>CTi0_ooo`:YZJT2
8B4Q0SXj>P:DU9@2CDe=0/_;b`9`L702RHV90UQHF09BDU82clo?0UIFEP:JVYX2FEUI0V][J`9BDU82
/k>c0^KViP9dM7@2Z:RX0Xj>SP9@D502HF5Q0S<c<`::RXX2^k^k0/?3``:b/[82L71`0W5aL@::RXX2
YJFU0WAdM0:>SXh2LW9b0V5QH@8Z:RX2QhN70WMgM`BOWil2NGUi0X^;R`:b/[82QhN70_3`l08b<S82
=CDe0SXj>P8f=SH2i>CT0WQhN0:PX:02;Bd]0WYjNP03ZJVY^;Rh^;Rh008O7al2QXJ60_gmo@:^[Zh2
QHF50VEUI@9/K6`2E5AD0X^;R`:5QHD2[jn_0U1@D0:e]KD2L71`0YRHV09JFUX2^KVi0WIfMP:3Ph<2
c/k>0WmoO`9EEED2Q8B40UmOG`9XJ6P2Y:BT0Zj^[P:MWId2dM7A0U1@D09_Kfl22@T90[^k^`8E5AD2
UiNG0Vi^KP9LG5`2O7al0Zn_[`:`/;02UYJF0W=cL`9nOWh2PX:20Vm_K`:`/;02];Bd0Vi^KPB5QHD2
M7Ad0X20P09<C4`28b<S0[^k^`:UYJD2>CTi0P0000:0P802EEEE0TA4A08c<c<4oooo0P00007oool0
01Goool00`000?ooooooo`02oooo0^_[j`AoOgl2MgMg0XJ6QP:4Q8@2l_;b0WYjNP9fMWH2gmoO0[Fe
]@:WYjL2XJ6Q0U9BDP:3Ph<2SHf=0[2`/09[Jf/2SHf=0SLg=`:^[Zh24a<C0W1`L09fMWH2QhN70TY:
BP;Lg=`2aLG50W9bLP9DE5@2UIFE0YjNWP:m_Kd2?3`l0QhN7P;Lg=`2GUiN0YZJVP;3`l<2Hf=S0ZFU
Y@:3Ph<2gmoO0Y:BTP9/K6`2a/K60WQhN09gMgL2I6AT0UMGE`;oool2/k>c0Sdm?@:KVi/2XZ:R0Xb<
S09oOgl2Y:BT0P0000:RXZ82>CTi0XJ6QP800002O7al00=ADE4X:2PX:2P00TY:BP:4Q8@2`<300YNG
U`;WinL27QhN0Q@D50:DU9@2GemO0TU9B@:?Shl2o?cl0^g]k@:0P802a<C40S0`<09cLg<2<S8b0/k>
cP91@D42EeMG0Z>SX`:EUID2a<C40_ooo`;WinL2G5aL0ZJVYP:RXZ82HF5Q0XJ6QP;GemL2P8200/?3
``8M7Ad2clo?0UQHF09MGEd2B4Q80ZNWY`95ADD2cLg=0Xj>SP9WIfL2_kno0Xb<S09cLg<26ATI0Y^K
V`;@d=02hn?S0Zb/[09NGUh2IFEU0YNGU`:b/[82GUiN0SXj>P:LW9`2@4100Y6AT@:0P805oooo0@00
007oool001Goool00`000?ooooooo`02oooo0^_[j`AoOgl2MgMg0XJ6QP:4Q8@2l_;b0WYjNP9fMWH2
gmoO0[Fe]@:WYjL2XJ6Q0U9BDP:3Ph<2SHf=0[2`/09[Jf/2SHf=0SLg=`:^[Zh24a<C0W1`L09fMWH2
QhN70TY:BP;Lg=`2aLG50W9bLP9DE5@2UIFE0YjNWP:m_Kd2?3`l0QhN7P;Lg=`2GUiN0YZJVP;3`l<2
Hf=S0ZFUY@:3Ph<2gmoO0Y:BTP9/K6`2a/K60WQhN09gMgL2I6AT0UMGE`;oool2/k>c0Sdm?@:KVi/2
XZ:R0Xb<S09oOgl2Y:BT0P0000:RXZ82>CTi0XJ6QP800002O7al00=ADE4X:2PX:2P00TY:BP:4Q8@2
`<300YNGU`;WinL27QhN0Q@D50:DU9@2GemO0TU9B@:?Shl2o?cl0^g]k@:0P802a<C40S0`<09cLg<2
<S8b0/k>cP91@D42EeMG0Z>SX`:EUID2a<C40_ooo`;WinL2G5aL0ZJVYP:RXZ82HF5Q0XJ6QP;GemL2
P8200/?3``8M7Ad2clo?0UQHF09MGEd2B4Q80ZNWY`95ADD2cLg=0Xj>SP9WIfL2_kno0Xb<S09cLg<2
6ATI0Y^KV`;@d=02hn?S0Zb/[09NGUh2IFEU0YNGU`:b/[82GUiN0SXj>P:LW9`2@4100Y6AT@:0P805
oooo0@00007oool001Goool00`000?ooooooo`02oooo0]kNgP:@T902I6AT0Z2PX0;6a/H2<C4a0P00
00:;Rh/2J6QX0UQHF09]KFd2=3@d0UaLG09lO7`2FEUI0U=CD`:NWYh2HV9R0UUIF@9nOWh2LG5a0Wmo
O`:5QHD2TI6A0Q@D509gMgL2Rh^;0/C4a09`L702LG5a0X>3P`;[jn/2NWYj0_ooo`:NWYh2OWin0Q8B
4P:f][H2`<300TM7A`:YZJT2820P0UiNGP9]KFd2RHV90PT92@9FEUH2/[:b0[Bd]0;<c<`2:2PX0/30
`09fMWH2KFe]0YjNWP9[Jf/2TY:B0X61P@:5QHD2CDe=0[Rh^094A4@2LW9b0VEUI@03=3@dVi^KVi^K
00:NWYh2LW9b0Zb/[0:WYjL2UYJF0Y6AT@8;2`/2j^[Z0TQ8B0;Fe]H2@d=30PP820;oool2P8200YBD
U08j>SX2BTY:1820P08a<C42UiNG0R@T90;Bd]82V9RH0R@T90:HV9P2hn?S0P0000;4a<@2i^KV0_oo
o`9]KFd2DU9B0]CDe0:0P802oooo0YVIV@9lO7`2fm_K0YVIV@:=SHd4UiNG0/30`08d=3@2eMGE0Z^[
Z`9TI6@200000WalO0800002emOG0Vi^KP:;Rh/2MgMg0WYjNP9fMWH2KFe]0WmoO`9jNWX2JFUY0YBD
U0Goool100000Oooo`005Oooo`030000oooooooo00;oool2g]kN0Y2@T09TI6@2X:2P0/K6aP8a<C42
00000X^;R`9XJ6P2F5QH0Ve]K@8d=3@2G5aL0WalO09IFET2De=C0YjNWP9RHV82FEUI0WinOP9aLG42
Ogmo0XF5Q@:ATI4251@D0WMgM`:;Rh/2a<C40W1`L09aLG42Ph>30^_[j`9jNWX2oooo0YjNWP9nOWh2
4Q8B0[Jf]P;0`<02AdM70ZVYZ@8P8202GUiN0Ve]K@:9RHT22@T90UIFEP:b/[82];Bd0/c<c08X:2P2
`<300WIfMP9]KFd2WYjN0V][J`:BTY82PH610XF5Q@9=CDd2^;Rh0TA4A09bLW82IFEU00<d=3BKVi^K
Vi/00YjNWP9bLW82[:b/0ZNWY`:FUYH2TI6A0P/;2`;Zj^X2B4Q80]KFeP93@d<220P80_ooo`:0P802
U9BD0SXj>P9:BTX4P8200S4a<@:GUiL292@T0];BdP:HV9P292@T0YRHV0;Shn<200000/C4a0;Vi^H2
oooo0Ve]K@9BDU82e=CD0X20P0;oool2VIVI0WalO0;Kfm/2VIVI0Xf=S@BGUiL2`<300S@d=0;EeMD2
Zj^[0VATI0800002O7al0P0000;GemL2KVi^0X^;R`9gMgL2NWYj0WIfMP9]KFd2Ogmo0WYjNP9YJFT2
U9BD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9oOgl2mOGe0_ooo`9fMWH2Shn?0]OG
e`:JVYX200000T=3@`892@T2b<S80W5aL@:m_Kd2oooo0W=cL`:IVIT2IFEU0U9BDP:BTY82<c<c0_oo
o`9ADE429RHV0XR8R09LG5`2MgMg0V9RHP;inOT2@D510WemO@;UiND2gMgM0UiNGP9<C4`2NGUi0T=3
@`9HF5P2a/K60[6a/@:PX:02KVi^0X20P09hN7P2BTY:0]CDe09hN7P2Rh^;0Xb<S09@D502Shn?0Sdm
?@800002EeMG0YjNWP8R8R84SXj>0W5aL@:PX:02@4100VMWI`9XJ6P2KVi^0WAdM003QhN7MGEeMGEe
009cLg<2UYJF0YNGU`:IVIT2IFEU0RXZ:P:EUID2R8R80[6a/@:6QXH2emOG0P00008i>CT2Fe]K0V5Q
H@;2`/82b/[:0Xn?S`:@T902J6QX0S@d=0:RXZ82Ph>30YZJVP;fm_H2?clo0[6a/@9bLW82]KFe0Zb/
[0:6QXH2:RXZ0Rh^;P9UIFD2F5QH0Z6QX@;GemL2MGEe0WemO@8j>SX2^[Zj0[Vi^@9dM7@2Lg=c0T51
@@;8b<P2/;2`0XB4Q0;Lg=`2K6a/0X61P@:MWId2YZJV0YRHV09kNg/2NGUi0YFEU@:j^[X2JVYZ0U=C
D`:<S8`2Q8B41Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9oOgl2mOGe0_ooo`9fMWH2
Shn?0]OGe`:JVYX200000T=3@`892@T2b<S80W5aL@:m_Kd2oooo0W=cL`:IVIT2IFEU0U9BDP:BTY82
<c<c0_ooo`9ADE429RHV0XR8R09LG5`2MgMg0V9RHP;inOT2@D510WemO@;UiND2gMgM0UiNGP9<C4`2
NGUi0T=3@`9HF5P2a/K60[6a/@:PX:02KVi^0X20P09hN7P2BTY:0]CDe09hN7P2Rh^;0Xb<S09@D502
Shn?0Sdm?@800002EeMG0YjNWP8R8R84SXj>0W5aL@:PX:02@4100VMWI`9XJ6P2KVi^0WAdM003QhN7
MGEeMGEe009cLg<2UYJF0YNGU`:IVIT2IFEU0RXZ:P:EUID2R8R80[6a/@:6QXH2emOG0P00008i>CT2
Fe]K0V5QH@;2`/82b/[:0Xn?S`:@T902J6QX0S@d=0:RXZ82Ph>30YZJVP;fm_H2?clo0[6a/@9bLW82
]KFe0Zb/[0:6QXH2:RXZ0Rh^;P9UIFD2F5QH0Z6QX@;GemL2MGEe0WemO@8j>SX2^[Zj0[Vi^@9dM7@2
Lg=c0T51@@;8b<P2/;2`0XB4Q0;Lg=`2K6a/0X61P@:MWId2YZJV0YRHV09kNg/2NGUi0YFEU@:j^[X2
JVYZ0U=CD`:<S8`2Q8B41Oooo`400001oooo000Eoooo00<0003oooooool01?ooo`;Vi^H2hn?S0_oo
o`9=CDd400000P<30`:c/k<200000Y6AT@:0P802oooo0]3@d0;Ph>02oooo0U]KF`;oool2YJFU0RPX
:0;oool2ADE50VMWI`:GUiL2S8b<0WMgM`8d=3@2fMWI0_ooo`;Fe]H2eMGE0Z6QX@800002lo?c0Wem
O@9=CDd2i^KV0Shn?P:XZ:P2VIVI0[Jf]P800002HV9R0Ti>CP94A4@2Shn?0SLg=`8m?Cd2Ph>30Xn?
S`9GEeL2ZZZZ0WIfMP:OWil2D51@0P0000:?Shl4O7al0ZZZZP:UYJD2IfMW0U9BDP800003MgMg0UIF
EP9[Jf/2VIVI0Xf=S@:5QHD200000UIFEP:5QHD2KFe]0UQHF09:BTX2U9BD0^3Ph0:NWYh2@T920/?3
``;Ti>@2dM7A0P00009JFUX2TY:B0U=CD`9VIVH2alO70TA4A08Z:RX2`L710XZ:RP:][Jd2IfMW0Rh^
;P9nOWh2DU9B0/?3``9UIFD2HF5Q0VEUI@:MWId2RHV917alO08`<3028R8R0WYjNP:DU9@2Ng]k0YBD
U08o?cl2O7al0[:b/P:MWId21@D50Xf=S@:9RHT2Ogmo0X20P09eMGD2YZJV0Zf][@9EEED2W9bL0X20
P0Goool100000Oooo`005Oooo`030000oooooooo00Coool2i^KV0^?Sh`;oool2CDe=100000830`<2
/k>c0P0000:ATI42P8200_ooo`;@d=02h>3P0_ooo`9KFe/2oooo0ZFUY@8X:2P2oooo0TE5A@9WIfL2
UiNG0Xb<S09gMgL2=3@d0]WIf@;oool2e]KF0]GEe@:QXJ4200000_?cl`9mOGd2CDe=0^KViP8n?Sh2
Z:RX0YVIV@:f][H200000V9RHP9>CTh2A4A40Xn?S`8g=cL2?Cdm0X>3P`:?Shl2EeMG0ZZZZP9fMWH2
WinO0U1@D0800002Shn?17alO0:ZZZX2YJFU0VMWI`9BDU8200000gMgM`9FEUH2Jf][0YVIV@:=SHd2
QHF50P00009FEUH2QHF50Ve]K@9HF5P2BTY:0YBDU0;Ph>02WYjN0T92@P;3`l<2i>CT0]7Ad@800002
FUYJ0Y:BTP9CDe<2IVIV0/O7a`94A4@2:RXZ0/71`@::RXX2[Jf]0VMWI`8^;Rh2OWin0U9BDP;3`l<2
IFEU0V5QH@9UIFD2WIfM0XV9R@AlO7`2<30`0R8R8P9jNWX2U9BD0W]kN`:DU9@2?clo0WalO0:b/[82
WIfM0PD51@:=SHd2RHV90WmoO`:0P802MGEe0ZJVYP:][Jd2EEEE0YbLW0:0P805oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0PP82090@40400000[Fe]@800002<c<c0Rl_;`871`L2P8200^3P
h08R8R8400000U9BDP:BTY82?Shn0YVIV@:d];@2?Cdm0];BdP91@D42lO7a0Pd=3@:FUYH2Ti>C0S<c
<`::RXX2TY:B0/O7a`:BTY82oooo0UUIF@:ATI42clo?0UeMG@9=CDd2Kfm_0Ve]K@9IFET2M7Ad0YnO
W`9MGEd2oooo0WinOP;1`L42_Kfm1:n_[`:f][H2[:b/0SXj>P8J6QX2VIVI0UYJFP9gMgL2?clo0Z2P
X09BDU82Xj>S0XF5Q@;Ph>02SXj>0[6a/@03OGemT92@T92@00:FUYH2UiNG0U5AD@:ZZZX2W9bL14m?
C`9ADE42Rh^;0W9bLP9GEeL2L71`0STi>@:7QhL2QHF50Shn?P:UYJD2JFUY0P/;2`:BTY82UYJF0XF5
Q@8Z:RX2CTi>0XZ:RP9LG5`2De=C0ZBTY08V9RH2_[jn0UYJFP9CDe<2LW9b0V1PH09hN7P2OGem0_oo
o`;0`<02?3`l0Ta<C0:TY:@2QhN70Xn?S`:ATI42Ng]k0QdM7@9JFUX2QXJ60UEEE@;>c/h2`L710VEU
I@8h>3P2OGem0Va/K09eMGD2M7Ad0UMGE`:CTi<2LW9b0XB4Q09gMgL5oooo0@00007oool001Goool2
00000oooo`8820P2@410100000:e]KD200000S<c<`8_;bl21`L70X20P0;Ph>028R8R1000009BDU82
TY:B0Shn?P:IVIT2];Bd0Sdm?@;Bd]82@D510_7al@8=3@d2UYJF0Y>CT`8c<c<2RXZ:0Y:BTP;7alL2
TY:B0_ooo`9IFET2TI6A0/o?c`9MGEd2CDe=0Vm_K`9]KFd2FEUI0WAdM0:OWil2GEeM0_ooo`9nOWh2
`L710[fm_@B_[jl2][Jf0Zb/[08j>SX26QXJ0YVIV@9JFUX2MgMg0Slo?`:PX:02DU9B0Z>SX`:5QHD2
h>3P0Xj>SP:a/K400gemOI2@T92@T002UYJF0YNGU`9ADE42ZZZZ0YbLW0A?Cdl2DE5A0X^;R`9bLW82
EeMG0W1`L08i>CT2QhN70XF5Q@8n?Sh2YJFU0VUYJ@8;2`/2TY:B0YJFUP:5QHD2:RXZ0Ti>CP::RXX2
G5aL0U=CD`:TY:@29RHV0[jn_P9JFUX2De=C0W9bLP9PH602N7Qh0WemO@;oool2`<300S`l?09<C4`2
Y:BT0XN7Q`:?Shl2TI6A0W]kN`8M7Ad2FUYJ0XJ6QP9EEED2c/k>0/71`@9UIFD2>3Ph0WemO@9/K6`2
MGEe0WAdM09GEeL2Ti>C0W9bLP:4Q8@2MgMg1?ooo`800001oooo000Eoooo00<0003oooooool00_oo
o`:RXZ82QXJ60Zb/[0;oool200000XZ:RP;Ti>@2Y:BT0X^;R`;IfMT2/k>c0_clo0:3Ph<2a/K60X:2
PP9/K6`2oooo0QLG5`9IFET2SXj>0P00008o?cl200000]SHf09VIVH2K6a/0V1PH091@D42NWYj0UYJ
FP9GEeL2;Bd]0[6a/@9iNGT21PH60TU9B@:7QhL2Ng]k0VYZJP9oOgl2^;Rh0UMGE`;Fe]H23Ph>0P<3
0`91@D42inOW0/k>cP:ATI42H61P0Te=C@9bLW82]kNg0S8b<P;dm?@2ATI60/g=c@94A4@200000/S8
b09CDe<2Shn?0XB4Q09XJ6P00chn?ZVYZJVYZ@02i>CT0WEeM@;bl_8220P80YBDU093@d<2@T920V9R
HP:9RHT2MGEe0X20P09dM7@2SHf=0XJ6QP800002DU9B0^o_k`;fm_H2Shn?0Pl?3`9ZJVX2Ti>C0[Zj
^P;Lg=`2Lg=c0X:2PP:8R8P2EEEE0WYjNP;Rh^82PX:20Zn_[`8j>SX2PH610SDe=@9GEeL2bl_;0VIV
IP:6QXH2`l?30_Win@94A4@2NGUi0T=3@`:OWil2GemO0[:b/P9kNg/2R8R80Y2@T092@T820P820Wmo
O`9bLW82WIfM0YFEU@9HF5P2P8200W]kN`8g=cL2QHF50YnOW`Goool100000Oooo`005Oooo`030000
oooooooo00;oool2XZ:R0XJ6QP:/[:`2oooo0P0000::RXX2i>CT0ZBTY0:;Rh/2fMWI0[>c/`;lo?`2
Ph>30/K6aP:2PX82K6a/0_ooo`8G5aL2FEUI0Xj>SP800002?clo0P0000;Hf=P2IVIV0Va/K09PH602
@D510WYjNP9JFUX2EeMG0Rd];@:a/K42NGUi0PH61P99BDT2QhN70W]kN`9ZJVX2Ogmo0[Rh^09GEeL2
e]KF0Ph>3P830`<2@D510^OWi`;>c/h2TI6A0V1PH09=CDd2LW9b0[Ng]`8b<S82m?Cd0TI6AP;=cLd2
A4A40P0000;8b<P2De=C0Xn?S`:4Q8@2J6QX00<n?SjYZJVYZJT00^CTi09eMGD2l_;b0PP820:DU9@2
@d=30T92@P9RHV82RHV90WEeM@:0P802M7Ad0Xf=S@:6QXH200000U9BDP;_knl2m_Kf0Xn?S`8?3`l2
JVYZ0Y>CT`:j^[X2g=cL0W=cL`:2PX82R8R80UEEE@9jNWX2h^;R0X:2PP:_[jl2>SXj0X61P@8e=CD2
EeMG0/_;b`9VIVH2QXJ60/?3``;inOT2A4A40WUiN@93@d<2WinO0UmOG`:b/[82Ng]k0XR8R0:@T902
@T920P820P9oOgl2LW9b0YfMW@:EUID2F5QH0X20P09kNg/2=cLg0XF5Q@:OWil5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0P00009`L702?Shn0P0000:`/;02oooo0ZVYZ@8o?cl2KFe]0/71
`@;UiND2HF5Q0W5aL@:2PX82Ng]k0Vm_K`9oOgl2oooo0VYZJP9CDe<2WinO0_ooo`;Kfm/2A4A40Y:B
TP9mOGd2];Bd0YjNWP8?3`l2EUIF0__kn`;4a<@2XJ6Q0WQhN0:ZZZX2Ph>30[Ng]`;8b<P2@T920YZJ
VP:ZZZX2:RXZ0_ooo`9gMgL2Hf=S0Vi^KP;QhN42;Rh^0Z6QX@;oool2eMGE0UiNGP9oOgl2[jn_0W]k
N`;7alL2i>CT0SDe=@:SXj<2ZZZZ0Xj>SP:8R8P2N7Qh0Y:BTP0300000`<30`<300;^k^h2[Jf]0[no
_`9BDU82R8R80V][J`:;Rh/2OGem0W]kN`:4Q8@2M7Ad0XR8R0:XZ:P2L71`0WIfMP:BTY82KFe]0Z:R
XP9XJ6P2];Bd0P00009IFET2=cLg0ZRXZ09KFe/2QXJ60Ve]K@:a/K425ADE0W9bLP9kNg/220P80[jn
_P;AdM4200000YJFUP9_Kfl2b/[:0WalO0:`/;02WIfM0P0000:^[Zh2G5aL0_7al@8j>SX2QHF50VAT
I0:][Jd2=cLg0ZRXZ09TI6@2]KFe0X61P@8m?Cd400000/30`09oOgl2MgMg0T92@P:PX:05oooo0@00
007oool001Goool00`000?ooooooo`02oooo0P00009`L702?Shn0P0000:`/;02oooo0ZVYZ@8o?cl2
KFe]0/71`@;UiND2HF5Q0W5aL@:2PX82Ng]k0Vm_K`9oOgl2oooo0VYZJP9CDe<2WinO0_ooo`;Kfm/2
A4A40Y:BTP9mOGd2];Bd0YjNWP8?3`l2EUIF0__kn`;4a<@2XJ6Q0WQhN0:ZZZX2Ph>30[Ng]`;8b<P2
@T920YZJVP:ZZZX2:RXZ0_ooo`9gMgL2Hf=S0Vi^KP;QhN42;Rh^0Z6QX@;oool2eMGE0UiNGP9oOgl2
[jn_0W]kN`;7alL2i>CT0SDe=@:SXj<2ZZZZ0Xj>SP:8R8P2N7Qh0Y:BTP0300000`<30`<300;^k^h2
[Jf]0[no_`9BDU82R8R80V][J`:;Rh/2OGem0W]kN`:4Q8@2M7Ad0XR8R0:XZ:P2L71`0WIfMP:BTY82
KFe]0Z:RXP9XJ6P2];Bd0P00009IFET2=cLg0ZRXZ09KFe/2QXJ60Ve]K@:a/K425ADE0W9bLP9kNg/2
20P80[jn_P;AdM4200000YJFUP9_Kfl2b/[:0WalO0:`/;02WIfM0P0000:^[Zh2G5aL0_7al@8j>SX2
QHF50VATI0:][Jd2=cLg0ZRXZ09TI6@2]KFe0X61P@8m?Cd400000/30`09oOgl2MgMg0T92@P:PX:05
oooo0@00007oool001Goool00`000?ooooooo`02oooo0X20P09eMGD200000W5aL@:QXJ42inOW0VMW
I`9;Bd/2Ng]k0YnOW`800002/K6a0Y6AT@:RXZ82CTi>0]3@d0:IVIT2:2PX0[bl_0;QhN42UYJF0UIF
EP9IFET2OGem0W]kN`;EeMD2QHF50]_Kf`9TI6@2J6QX0^[ZjP9TI6@2YJFU0[Ng]`:QXJ42R8R80Rh^
;P:h^;P2YjNW0VIVIP::RXX2EUIF0W9bLP9EEED29RHV0^?Sh`:9RHT2QXJ60PD51@9>CTh2O7al0Win
OP9eMGD2I6AT0S0`<0:SXj<200000/c<c0;Fe]H2Ng]k0U=CD`:?Shl2EeMG0S4a<@03DE5AK6a/K6a/
009FEUH2OGem0Y6AT@9eMGD2KFe]0QXJ6P9cLg<2MgMg0XF5Q@9:BTX2HF5Q0P0000:j^[X2MGEe0[6a
/@9CDe<2KVi^0V1PH09XJ6P2EeMG0]CDe08E5AD2GemO0/30`0;Hf=P2OGem0XN7Q`9MGEd2K6a/0Win
OP9WIfL2TI6A0VMWI`:WYjL2_kno0WIfMP:HV9P200000_ooo`;alO42?3`l0U1@D09MGEd2W9bL0[Rh
^0:1PH42Fe]K0VUYJ@:f][H2IFEU0WMgM`9WIfL2eMGE0V5QH@;oool2EUIF0]kNgP9fMWH2RHV90/S8
b0:RXZ82G5aL1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:0P802MGEe0P00009aLG42
XJ6Q0^OWi`9WIfL2Bd];0W]kN`:OWil200000[6a/@:ATI42XZ:R0Ti>CP;@d=02VIVI0RPX:0:l_;`2
hN7Q0YJFUP9FEUH2FEUI0WemO@9kNg/2eMGE0XF5Q@;Kfm/2I6AT0VQXJ0;Zj^X2I6AT0ZFUY@:g]kL2
XJ6Q0XR8R08^;Rh2^;Rh0ZNWY`9VIVH2RXZ:0UIFEP9bLW82EEEE0RHV9P;Shn<2RHV90XJ6QP851@D2
CTi>0WalO09nOWh2MGEe0VATI08`<302Xj>S0P0000;<c<`2e]KF0W]kN`9CDe<2Shn?0UMGE`8a<C40
0e5ADFa/K6a/K002EUIF0WemO@:ATI42MGEe0Ve]K@8J6QX2Lg=c0WMgM`:5QHD2BTY:0V5QH@800002
^[Zj0WEeM@:a/K42De=C0Vi^KP9PH602J6QX0UMGE`;De=@25ADE0UmOG`;0`<02f=SH0WemO@:7QhL2
GEeM0Va/K09nOWh2IfMW0Y6AT@9WIfL2YjNW0[no_`9fMWH2V9RH0P0000;oool2lO7a0S`l?09@D502
GEeM0YbLW0:h^;P2PH610U]KF`9YJFT2][Jf0VEUI@9gMgL2IfMW0]GEe@9QHF42oooo0UIFEP;Ng]h2
MWIf0XV9R@;8b<P2XZ:R0UaLG0Goool100000Oooo`005Oooo`030000oooooooo00;oool2[:b/0]oO
g`9ZJVX2KFe]0UMGE`:BTY82b<S80QPH609eMGD2MgMg0V][J`:VYZH2Fe]K0S4a<@:5QHD2W9bL0[6a
/@8o?cl2JVYZ0UmOG`:8R8P200000[6a/@9_Kfl2gMgM0^WYj@;Ti>@23Ph>0ZFUY@820P82PX:20P00
008N7Qh2LW9b0RPX:09_Kfl2X:2P0W5aL@9]KFd2YjNW0P0000:i^KT2YjNW0Zj^[P;oool2@4100UiN
GP:UYJD4QXJ60U]KF`:HV9P2Z:RX0]OGe`;moOd20@410UUIF@:UYJD2/;2`0Z:RXP9OGel2UIFE0V9R
HP9iNGT00fUYJBHV9RHV9P02HV9R0VATI0:>SXh2NWYj0SLg=`9FEUH2OWin0W]kN`:1PH42T92@0P<3
0`9OGel2D51@0UeMG@;`l?02QHF50^g]k@:n_[h2/;2`0YJFUP9ZJVX2/k>c0R0P809]KFd2oooo0Z6Q
X@:KVi/2LG5a0P0000:9RHT2a/K60VYZJP8k>c/2Y:BT0X20P09`L702FEUI0_ooo`:OWil2DE5A0X^;
R`;UiND2_[jn0VMWI`:RXZ82<c<c0R/[:`;fm_H2[Jf]0YjNWP96ATH2EEEE0/_;b`9fMWH2820P0YbL
W090@402UIFE0P`<30:?Shl2H61P0P0000Goool100000Oooo`002Oooo`800004oooo00<0003ooooo
ool00oooo`030000oooooooo00;oool2[:b/0]oOg`9ZJVX2KFe]0UMGE`:BTY82b<S80QPH609eMGD2
MgMg0V][J`:VYZH2Fe]K0S4a<@:5QHD2W9bL0[6a/@8o?cl2JVYZ0UmOG`:8R8P200000[6a/@9_Kfl2
gMgM0^WYj@;Ti>@23Ph>0ZFUY@820P82PX:20P00008N7Qh2LW9b0RPX:09_Kfl2X:2P0W5aL@9]KFd2
YjNW0P0000:i^KT2YjNW0Zj^[P;oool2@4100UiNGP:UYJD4QXJ60U]KF`:HV9P2Z:RX0]OGe`;moOd2
0@410UUIF@:UYJD2/;2`0Z:RXP9OGel2UIFE0V9RHP9iNGT00fUYJBHV9RHV9P02HV9R0VATI0:>SXh2
NWYj0SLg=`9FEUH2OWin0W]kN`:1PH42T92@0P<30`9OGel2D51@0UeMG@;`l?02QHF50^g]k@:n_[h2
/;2`0YJFUP9ZJVX2/k>c0R0P809]KFd2oooo0Z6QX@:KVi/2LG5a0P0000:9RHT2a/K60VYZJP8k>c/2
Y:BT0X20P09`L702FEUI0_ooo`:OWil2DE5A0X^;R`;UiND2_[jn0VMWI`:RXZ82<c<c0R/[:`;fm_H2
[Jf]0YjNWP96ATH2EEEE0/_;b`9fMWH2820P0YbLW090@402UIFE0P`<30:?Shl2H61P0P0000Goool1
00000Oooo`002Oooo`030000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool00_oo
o`800002>SXj0UYJFP96ATH2j>SX0Xf=S@8k>c/2ATI60P0000:DU9@22PX:0XF5Q@9NGUh2I6AT0^?S
h`9@D502^KVi0SXj>P;Lg=`2Rh^;0QPH608j>SX2QXJ60ZZZZP:ATI42T92@0TQ8B08B4Q82dm?C0YJF
UP;Kfm/2YjNW0[Rh^099BDT2^KVi0]WIf@:WYjL2c<c<0XN7Q`861PH2@d=30X61P@8S8b<2CDe=0SPh
>09SHf<251@D0QhN7P8R8R82gmoO0STi>@:i^KT24A4A0VMWI`;1`L42>c/k0[jn_P8@4102R8R80Y^K
V`9jNWX2J6QX0WQhN09BDU800adM7IFEUIFEU@02EEEE0XJ6QP9iNGT2PH610SDe=@:b/[82LW9b0X:2
PP9gMgL2a<C40_ooo`9dM7@2`L710UIFEP8B4Q82fm_K0Z6QX@9/K6`2dm?C0VYZJP;Kfm/2MWIf0U]K
F`8820P4G5aL0SLg=`8k>c/220P80Xn?S`:BTY8200000XV9R@96ATH2n_[j0W1`L0;oool2e]KF0TE5
A@8m?Cd271`L0PP8208<30`2LW9b0]7Ad@9HF5P2J6QX0R`/;09kNg/2Lg=c0]7Ad@9RHV82CTi>0Wem
O@9^KVh2Ph>30[no_`8Y:BT2/;2`0V1PH0:c/k<2>3Ph1Oooo`400001oooo0009oooo00<0003oool0
00000_ooo`030000oooo000000Coool200000oooo`800002>SXj0UYJFP96ATH2j>SX0Xf=S@8k>c/2
ATI60P0000:DU9@22PX:0XF5Q@9NGUh2I6AT0^?Sh`9@D502^KVi0SXj>P;Lg=`2Rh^;0QPH608j>SX2
QXJ60ZZZZP:ATI42T92@0TQ8B08B4Q82dm?C0YJFUP;Kfm/2YjNW0[Rh^099BDT2^KVi0]WIf@:WYjL2
c<c<0XN7Q`861PH2@d=30X61P@8S8b<2CDe=0SPh>09SHf<251@D0QhN7P8R8R82gmoO0STi>@:i^KT2
4A4A0VMWI`;1`L42>c/k0[jn_P8@4102R8R80Y^KV`9jNWX2J6QX0WQhN09BDU800adM7IFEUIFEU@02
EEEE0XJ6QP9iNGT2PH610SDe=@:b/[82LW9b0X:2PP9gMgL2a<C40_ooo`9dM7@2`L710UIFEP8B4Q82
fm_K0Z6QX@9/K6`2dm?C0VYZJP;Kfm/2MWIf0U]KF`8820P4G5aL0SLg=`8k>c/220P80Xn?S`:BTY82
00000XV9R@96ATH2n_[j0W1`L0;oool2e]KF0TE5A@8m?Cd271`L0PP8208<30`2LW9b0]7Ad@9HF5P2
J6QX0R`/;09kNg/2Lg=c0]7Ad@9RHV82CTi>0WemO@9^KVh2Ph>30[no_`8Y:BT2/;2`0V1PH0:c/k<2
>3Ph1?ooo`800001oooo0009oooo0P0000?oool00`000?ooo`000004oooo00<0003oooooool00_oo
o`:k^k/2MGEe0RPX:0:SXj<2JVYZ0[no_`9fMWH2i^KV0_ooo`8I6AT2PX:20UADE0:NWYh2VYZJ0]KF
eP9BDU82>CTi0[bl_0::RXX2]kNg0Xj>SP9_Kfl2SHf=0ZZZZP8R8R82jNWY0XB4Q0:LW9`2VIVI0TQ8
B0:g]kL2G5aL0QTI6@;oool2emOG0XZ:RP8N7Qh2IfMW0P00009kNg/2;Rh^0_ooo`9eMGD2Kfm_0W1`
L0:5QHD2V9RH0P00009KFe/2a<C40WemO@9VIVH2Vi^K0VATI09DE5@2EEEE0VIVIP:^[Zh2S8b<0X>3
P`:FUYH2ADE50YjNWP8/;2`00hN7Qj>SXj>SX`026a/K0UmOG`:1PH42P8200VATI0;Kfm/2LW9b0X61
P@9iNGT2GemO0Y2@T0:UYJD2B4Q80Tm?C`8T92@2Ph>30/_;b`95ADD2LG5a0SPh>0:GUiL200000XZ:
RP9FEUH2S8b<0]3@d0;>c/h2Ng]k0UUIF@8X:2P25QHF0S@d=0:EUID2M7Ad0P0000;oool2kno_0Win
OP9dM7@2a/K60][JfP:n_[h2bl_;0^;RhP:SXj<2Ph>30W9bLP:KVi/2NWYj0VATI08a<C42OWin0Y2@
T0:TY:@2][Jf0[^k^`871`L2UiNG0PX:2P:>SXh2YZJV0V9RHPGoool100000Oooo`002Oooo`030000
oooooooo00;oool00`000?ooo`000004oooo00<0003oooooool00_ooo`:k^k/2MGEe0RPX:0:SXj<2
JVYZ0[no_`9fMWH2i^KV0_ooo`8I6AT2PX:20UADE0:NWYh2VYZJ0]KFeP9BDU82>CTi0[bl_0::RXX2
]kNg0Xj>SP9_Kfl2SHf=0ZZZZP8R8R82jNWY0XB4Q0:LW9`2VIVI0TQ8B0:g]kL2G5aL0QTI6@;oool2
emOG0XZ:RP8N7Qh2IfMW0P00009kNg/2;Rh^0_ooo`9eMGD2Kfm_0W1`L0:5QHD2V9RH0P00009KFe/2
a<C40WemO@9VIVH2Vi^K0VATI09DE5@2EEEE0VIVIP:^[Zh2S8b<0X>3P`:FUYH2ADE50YjNWP8/;2`0
0hN7Qj>SXj>SX`026a/K0UmOG`:1PH42P8200VATI0;Kfm/2LW9b0X61P@9iNGT2GemO0Y2@T0:UYJD2
B4Q80Tm?C`8T92@2Ph>30/_;b`95ADD2LG5a0SPh>0:GUiL200000XZ:RP9FEUH2S8b<0]3@d0;>c/h2
Ng]k0UUIF@8X:2P25QHF0S@d=0:EUID2M7Ad0P0000;oool2kno_0WinOP9dM7@2a/K60][JfP:n_[h2
bl_;0^;RhP:SXj<2Ph>30W9bLP:KVi/2NWYj0VATI08a<C42OWin0Y2@T0:TY:@2][Jf0[^k^`871`L2
UiNG0PX:2P:>SXh2YZJV0V9RHPGoool100000Oooo`002_ooo`800003oooo00<0003oooooool00ooo
o`030000oooooooo00;oool2S8b<0Va/K09hN7P2_Kfm0YNGU`;oool2;Bd]0Z^[Z`;oool2XJ6Q0W1`
L09eMGD2I6AT0XB4Q0;Zj^X2eMGE0T10@0:l_;`2EeMG0P0000;_knl2Q8B40W9bLP9jNWX2SHf=0PT9
2@9FEUH2IfMW0UmOG`9jNWX2Bd];0X61P@;oool2c<c<0QTI6@99BDT2RHV90Ve]K@;YjNT2_Kfm0YVI
V@9]KFd2oooo0W1`L0;Hf=P2T92@0WmoO`;IfMT2QHF50YFEU@:=SHd2V9RH0RPX:0:RXZ82WinO0TI6
AP;oool2P8200Vi^KP:SXj<2PX:20WemO@9QHF42UIFE00=SHf=GEeMGEeL00X20P0:<S8`2HV9R0UmO
G`9fMWH2T92@0X20P09lO7`2PH610U5AD@8a<C42R8R80ZNWY`;oool2PH610/O7a`9VIVH2J6QX0Z>S
X`:l_;`2KFe]0Y6AT@8];Bd2OWin0W=cL`:CTi<25QHF0Va/K0;3`l<2VIVI0]3@d0:;Rh/2HF5Q0_oo
o`9EEED2GemO0T=3@`;Ng]h2EUIF0]KFeP9@D502[Zj^0^GUi@9SHf<20`<30Sdm?@96ATH2e]KF0Q/K
6`9PH602WYjN0Y6AT@:6QXH2RXZ:0W]kN`:WYjL2TY:B0S8b<P8P8202<c<c0[Ng]`:QXJ45oooo0@00
007oool001Goool00`000?ooooooo`02oooo0Xb<S09/K6`2N7Qh0[fm_@:GUiL2oooo0Rd];@:[Zj/2
oooo0Z6QX@9`L702MGEe0VATI0:4Q8@2j^[Z0]GEe@90@402_;bl0UMGE`800002kno_0XB4Q09bLW82
NWYj0Xf=S@892@T2EUIF0VMWI`9OGel2NWYj0T];B`:1PH42oooo0/c<c08I6AT2BDU90XV9R@9]KFd2
jNWY0[fm_@:IVIT2KFe]0_ooo`9`L702f=SH0Y2@T09oOgl2fMWI0XF5Q@:EUID2SHf=0YRHV08X:2P2
XZ:R0YnOW`96ATH2oooo0X20P09^KVh2Xj>S0X:2PP9mOGd2HF5Q0YFEU@03Hf=SEeMGEeMG00:0P802
S8b<0V9RHP9OGel2MWIf0Y2@T0:0P802O7al0X61P@9ADE42<C4a0XR8R0:WYjL2oooo0X61P@;7alL2
IVIV0VQXJ0:SXj<2_;bl0Ve]K@:ATI42;Bd]0WinOP9cLg<2Ti>C0QHF5P9/K6`2`l?30YVIV@;@d=02
Rh^;0V5QH@;oool2EEEE0UmOG`93@d<2g]kN0UIFEP;Fe]H2D51@0Zj^[P;UiND2Hf=S0P<30`8m?Cd2
ATI60]KFeP8K6a/2H61P0YjNWP:ATI42QXJ60XZ:RP9kNg/2YjNW0Y:BTP8b<S82820P0S<c<`:g]kL2
XJ6Q1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`800002b<S80Rd];@8L71`2LW9b0Wem
O@:WYjL2FUYJ0P0000:_[jl2`l?30Xn?S`9dM7@2e]KF0[fm_@;GemL21`L70UaLG0:OWil2[Jf]0V=S
H`;Ph>02b<S80[Rh^09iNGT2BTY:0WMgM`9cLg<2KVi^0^KViP:CTi<230`<0P0000;kno/2oooo0/?3
``8T92@2^k^k0Ve]K@8410@2Ng]k0Z^[Z`9SHf<2HV9R0P0000810@42VIVI0_ooo`95ADD2@D510[Ng
]`8k>c/2YJFU0UmOG`:KVi/2Cdm?0YbLW09NGUh2@d=30SXj>P:6QXH2Cdm?0/30`08j>SX00iBDU::R
XZ:RXP02b/[:0U]KF`:HV9P2;bl_0[:b/P8/;2`2@d=30XZ:RP9gMgL2W9bL0Sdm?@9iNGT2Shn?0VQX
J09CDe<2cLg=0Ti>CP;AdM4220P80RXZ:P;8b<P2TY:B0V5QH@9OGel2a/K60V5QH@;IfMT2>SXj0UAD
E0:ZZZX2MgMg0V5QH@:4Q8@2UYJF0_ooo`9cLg<2WYjN0_ooo`9BDU82Ph>30U1@D09cLg<2_Kfm0UeM
G@;;bl/2F5QH0T92@P8k>c/2KVi^0Y6AT@9QHF42<30`0UeMG@9mOGd2PH610VMWI`9hN7P2UIFE0[2`
/09lO7`2@d=30XJ6QPGoool100000Oooo`005Oooo`030000oooooooo00;oool200000/S8b08];Bd2
71`L0W9bLP9mOGd2YjNW0UYJFP800002[jn_0/?3``:?Shl2M7Ad0]KFeP:m_Kd2emOG0PL71`9LG5`2
WinO0Zf][@9SHf<2h>3P0/S8b0:h^;P2NGUi0TY:BP9gMgL2Lg=c0Vi^KP;Vi^H2Ti>C0P`<30800002
no_k0_ooo`;3`l<292@T0[^k^`9]KFd210@40W]kN`:[Zj/2Hf=S0V9RHP8000020@410YVIV@;oool2
ADE50T51@@:g]kL2>c/k0ZFUY@9OGel2Vi^K0Tm?C`:LW9`2GUiN0T=3@`8j>SX2QXJ60Tm?C`;0`<02
>SXj00>DU9BRXZ:RXZ800/[:bP9KFe/2V9RH0Rl_;`:b/[82;2`/0T=3@`::RXX2MgMg0YbLW08m?Cd2
NGUi0Xn?S`9XJ6P2De=C0/g=c@9>CTh2dM7A0PP8208Z:RX2b<S80Y:BTP9QHF42GemO0/K6aP9QHF42
fMWI0SXj>P9DE5@2ZZZZ0WMgM`9QHF42Q8B40YJFUP;oool2Lg=c0YjNWP;oool2DU9B0X>3P`9@D502
Lg=c0[fm_@9MGEd2bl_;0UQHF092@T82>c/k0Vi^KP:ATI42HF5Q0S0`<09MGEd2OGem0X61P@9WIfL2
N7Qh0YFEU@:`/;02O7al0T=3@`:6QXH5oooo0@00007oool001Goool00`000?ooooooo`02oooo0YnO
W`92@T82NWYj0ZRXZ08I6AT2?clo0QLG5`9iNGT23@d=0R8R8P;Fe]H2clo?0W]kN`9EEED292@T0V9R
HP:_[jl2ZJVY0UQHF0:QXJ42K6a/0VEUI@;;bl/2a/K60P00008f=SH2YJFU0VYZJP:_[jl200000^k^
kP:CTi<2/K6a0]CDe09eMGD2N7Qh0P`<30:RXZ82Kfm_0RXZ:P8];Bd2[Zj^0W5aL@94A4@2jn_[0YZJ
VP9TI6@2Q8B40Xb<S0:i^KT2ATI60[:b/P:4Q8@2M7Ad0ZBTY0;Bd]82Q8B40YnOW`97AdL2NWYj0YBD
U098B4P2RXZ:0]GEe@03f=SH6a/K6a/K0099BDT29bLW0U=CD`8M7Ad200000Zf][@99BDT2j^[Z0UYJ
FP9gMgL200000W=cL`;1`L42Rh^;0Z6QX@9WIfL2N7Qh0Zb/[0:UYJD2h^;R0Rh^;P:j^[X2QXJ60X20
P0;1`L4200000XJ6QP;oool2AdM70WalO09GEeL2Ogmo0WEeM@;]kNd2_[jn0U=CD`;]kNd2EEEE0WMg
M`9eMGD2^;Rh0]3@d09MGEd2W9bL0YZJVP;>c/h2M7Ad0^_[j`B`/;02T92@0XV9R@9MGEd2NWYj0Wal
O09dM7@2N7Qh0Y>CT`8f=SH2Hf=S0Y:BTP9[Jf/5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0YnOW`92@T82NWYj0ZRXZ08I6AT2?clo0QLG5`9iNGT23@d=0R8R8P;Fe]H2clo?0W]kN`9EEED2
92@T0V9RHP:_[jl2ZJVY0UQHF0:QXJ42K6a/0VEUI@;;bl/2a/K60P00008f=SH2YJFU0VYZJP:_[jl2
00000^k^kP:CTi<2/K6a0]CDe09eMGD2N7Qh0P`<30:RXZ82Kfm_0RXZ:P8];Bd2[Zj^0W5aL@94A4@2
jn_[0YZJVP9TI6@2Q8B40Xb<S0:i^KT2ATI60[:b/P:4Q8@2M7Ad0ZBTY0;Bd]82Q8B40YnOW`97AdL2
NWYj0YBDU098B4P2RXZ:0]GEe@03f=SH6a/K6a/K0099BDT29bLW0U=CD`8M7Ad200000Zf][@99BDT2
j^[Z0UYJFP9gMgL200000W=cL`;1`L42Rh^;0Z6QX@9WIfL2N7Qh0Zb/[0:UYJD2h^;R0Rh^;P:j^[X2
QXJ60X20P0;1`L4200000XJ6QP;oool2AdM70WalO09GEeL2Ogmo0WEeM@;]kNd2_[jn0U=CD`;]kNd2
EEEE0WMgM`9eMGD2^;Rh0]3@d09MGEd2W9bL0YZJVP;>c/h2M7Ad0^_[j`B`/;02T92@0XV9R@9MGEd2
NWYj0WalO09dM7@2N7Qh0Y>CT`8f=SH2Hf=S0Y:BTP9[Jf/5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0V5QH@:0P802Kfm_0WEeM@9RHV82MgMg0S/k>`9ZJVX2HV9R0^o_k`9NGUh2KVi^0Va/
K09QHF42RXZ:0UIFEP9hN7P2P8200_ooo`;0`<0200000/c<c09aLG42ZJVY0V][J`9>CTh2Rh^;0Xj>
SP9oOgl2fm_K0/S8b09<C4`2@T920WEeM@9dM7@2QHF50U9BDP;5aLD2N7Qh0Xj>SP8A4A42;bl_0Wem
O@9UIFD2IfMW0UaLG0;8b<P2Ng]k0_;blP9;Bd/2Fe]K0TA4A0:UYJD2^[Zj0W1`L09mOGd2WIfM0S8b
<P;oool2S8b<0Xj>SP9QHF42SXj>0XR8R003J6QXZZZZZZZZ0094A4@2oooo0[no_`9^KVh2P8200[2`
/09dM7@2P8200WQhN0:5QHD2QhN70Z2PX08T92@2Rh^;0VQXJ0:5QHD2NGUi0X20P0:8R8P2GEeM0P00
00;De=@2BTY:0[>c/`:@T902MGEe0WalO09oOgl2XZ:R0Vi^KP;gmoL2]KFe0TU9B@:f][H2YJFU0X61
P@:3Ph<2BTY:0[Jf]P::RXX2S8b<0TI6AP:?Shl2Rh^;0V5QH@:JVYX261PH0W9bLP:ZZZX2Y:BT0S/k
>`:3Ph<2HV9R0X>3P`9WIfL24A4A0WalO0;oool2d=3@0Xj>SP;oool2HF5Q1Oooo`400001oooo000E
oooo0P0000?oool2HF5Q0X20P09_Kfl2MGEe0V9RHP9gMgL2>c/k0VYZJP9RHV82kno_0UiNGP9^KVh2
K6a/0V5QH@::RXX2EUIF0WQhN0:0P802oooo0/30`0800002c<c<0W5aL@:YZJT2Jf][0Ti>CP:;Rh/2
SXj>0WmoO`;Kfm/2b<S80Ta<C092@T82MGEe0WAdM0:5QHD2DU9B0/G5a@9hN7P2SXj>0Q4A4@8_;bl2
OGem0VEUI@9WIfL2G5aL0/S8b09kNg/2l_;b0T];B`9KFe/2A4A40ZFUY@:j^[X2L71`0WemO@:MWId2
<S8b0_ooo`:<S8`2SXj>0V5QH@:>SXh2R8R800=XJ6RZZZZZZZX00TA4A0;oool2_kno0Vi^KP:0P802
/;2`0WAdM0:0P802N7Qh0XF5Q@:7QhL2X:2P0R@T90:;Rh/2J6QX0XF5Q@9iNGT2P8200XR8R09MGEd2
00000]CDe09:BTX2/k>c0Y2@T09eMGD2O7al0WmoO`:RXZ82KVi^0_Ogm`:e]KD2BDU90[Jf]P:UYJD2
PH610X>3P`9:BTX2][Jf0XZ:RP:<S8`2ATI60Xn?S`:;Rh/2HF5Q0YZJVP8H61P2LW9b0ZZZZP:TY:@2
>c/k0X>3P`9RHV82Ph>30VMWI`8A4A42O7al0_ooo`;@d=02SXj>0_ooo`9QHF44oooo0P00007oool0
01Goool00`000?ooooooo`02oooo0Te=C@:_[jl2TI6A0T92@P:YZJT2m?Cd0WAdM0:=SHd2BTY:0T=3
@`9SHf<2JVYZ0Vm_K`9bLW82OWin0XR8R08E5AD2Cdm?0Y:BTP:ATI42oooo0Z>SX`:NWYh2Ti>C0Z^[
Z`:l_;`27alO0]SHf09QHF4200000YfMW@;8b<P2NWYj0ZJVYP:TY:@2=3@d0ZNWY`8Y:BT2GemO0U5A
D@:MWId2a/K60YVIV@9jNWX2EUIF0[Zj^P9GEeL2T92@0Rd];@:i^KT2_[jn0YfMW@9>CTh2X:2P0_[j
nP;Cdm<2inOW0Xj>SP9SHf<2GUiN0X^;R`9ZJVX2H61P0UIFEP03P820FEUIFEUI00:XZ:P2kno_0V=S
H`:9RHT2IVIV0[2`/09iNGT2OGem0X:2PP9MGEd2DU9B0TY:BP9iNGT2NWYj0ZFUY@9QHF42RXZ:0W]k
N`8U9BD21PH60UADE09mOGd2KFe]0UeMG@8/;2`2TI6A0WinOP9PH602CDe=0/G5a@::RXX2KVi^0UEE
E@:4Q8@2Jf][0W1`L0:2PX82BTY:0V=SH`:][Jd2oooo0UIFEP8I6AT2M7Ad0YFEU@9ZJVX2M7Ad0YfM
W@8>3Ph2B4Q80XB4Q09eMGD2A4A40X61P@:7QhL26ATI0UADE08W9bL26a/K0R@T908n?Sh2PH611Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`9=CDd2[jn_0Y6AT@92@T82ZJVY0_Cdm09dM7@2
SHf=0TY:BP93@d<2Hf=S0VYZJP9_Kfl2LW9b0WinOP:8R8P25ADE0Tm?C`:BTY82TI6A0_ooo`:SXj<2
WYjN0Y>CT`:[Zj/2_;bl0QlO7`;Hf=P2HF5Q0P0000:MWId2b<S80WYjNP:VYZH2Y:BT0S@d=0:WYjL2
:BTY0UmOG`9ADE42WIfM0/K6aP:IVIT2NWYj0UIFEP:j^[X2EeMG0Y2@T08];Bd2^KVi0[jn_P:MWId2
CTi>0Z2PX0;jn_X2dm?C0^OWi`:>SXh2Hf=S0UiNGP:;Rh/2JVYZ0V1PH09FEUH00h20P5UIFEUIF@02
Z:RX0^o_k`9SHf<2RHV90VIVIP:`/;02NGUi0WemO@:2PX82GEeM0U9BDP9:BTX2NGUi0WYjNP:UYJD2
HF5Q0XZ:RP9kNg/29BDU0PH61P9DE5@2OGem0Ve]K@9MGEd2;2`/0Y6AT@9nOWh2H61P0Te=C@;5aLD2
RXZ:0Vi^KP9EEED2Q8B40V][J`9`L702PX:20TY:BP9SHf<2[Jf]0_ooo`9FEUH26ATI0WAdM0:EUID2
JVYZ0WAdM0:MWId23Ph>0TQ8B0:4Q8@2MGEe0TA4A0:1PH42QhN70QTI6@9DE5@29bLW0Q/K6`8T92@2
?Shn0X61P@Goool100000Oooo`005Oooo`030000oooooooo00;oool2I6AT0T51@@:c/k<2Xj>S0]OG
e`9XJ6P2FUYJ0WEeM@9gMgL2IVIV0WMgM`9=CDd2Hf=S0U5AD@9HF5P2WIfM0XF5Q@9ZJVX2Bd];0X61
P@8j>SX2N7Qh0XR8R0:9RHT2[Jf]0Tm?C`9ADE42BDU90V=SH`:GUiL2TY:B0YVIV@:BTY82]kNg0SDe
=@:=SHd2MWIf0VATI0:d];@2OWin0ZRXZ09NGUh2oooo0UaLG0:9RHT2b/[:0S@d=09SHf<2Ph>30WIf
MP9YJFT2=CDe0Xf=S@9^KVh2G5aL0S@d=09MGEd2i>CT0ZRXZ09=CDd2UIFE0WIfMP9GEeL2LW9b00<0
001dM7AdM7@00P0000:GUiL2PH610WYjNP:f][H2U9BD0WalO09jNWX2Ph>30UIFEP:BTY82/K6a0W1`
L0:CTi<24Q8B0R<S8`:5QHD2VYZJ0Vm_K`:BTY82Ph>30W=cL`95ADD2Ti>C0ZZZZP9EEED2QHF50VMW
I`8Q8B42QXJ60P0000;Ph>02PH610/30`0;?cll2SXj>0W9bLP:5QHD2/[:b0X>3P`8D51@2WIfM0WEe
M@:6QXH2IFEU0Rl_;`:g]kL2A4A40WemO@9nOWh2U9BD0YNGU`:g]kL2MWIf0YRHV09gMgL271`L0RHV
9P9GEeL2bl_;0TY:BP;Lg=`5oooo0@00007oool001Goool00`000?ooooooo`02oooo0VATI091@D42
/k>c0Z>SX`;GemL2J6QX0UYJFP9eMGD2MgMg0VIVIP9gMgL2CDe=0V=SH`9ADE42F5QH0YfMW@:5QHD2
JVYZ0T];B`:1PH42>SXj0WQhN0:8R8P2RHV90Zf][@9?Cdl2DE5A0TU9B@9SHf<2UiNG0Y:BTP:IVIT2
TY:B0[Ng]`8e=CD2SHf=0WIfMP9TI6@2];Bd0WinOP:XZ:P2GUiN0_ooo`9LG5`2RHV90/[:bP8d=3@2
Hf=S0X>3P`9fMWH2JFUY0SDe=@:=SHd2KVi^0UaLG08d=3@2GEeM0^CTi0:XZ:P2CDe=0YFEU@9fMWH2
EeMG0W9bLP030000M7AdM7Ad00800002UiNG0X61P@9jNWX2][Jf0YBDU09lO7`2NWYj0X>3P`9FEUH2
TY:B0[6a/@9`L702Ti>C0Q8B4P8S8b<2QHF50YZJVP9_Kfl2TY:B0X>3P`9cLg<2ADE50Y>CT`:ZZZX2
EEEE0XF5Q@9WIfL28B4Q0XJ6QP800002h>3P0X61P@;0`<02clo?0Xj>SP9bLW82QHF50[:b/P:3Ph<2
51@D0YfMW@9eMGD2QXJ60VEUI@8_;bl2]kNg0TA4A09mOGd2OWin0YBDU0:GUiL2]kNg0WIfMP:HV9P2
MgMg0Q`L708V9RH2EeMG0/_;b`9:BTX2g=cL1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`:DU9@2DE5A0S0`<09;Bd/2Fe]K0_ooo`:CTi<2NWYj0[Jf]P:UYJD2[jn_0X61P@9VIVH2CDe=0X61
P@9cLg<2Q8B40Vm_K`9ADE42Ti>C0^GUi@9lO7`2;Bd]0[jn_P:CTi<2][Jf0[Bd]0;Hf=P2@4100/c<
c09mOGd2bLW90TY:BP:5QHD2Shn?0VATI09nOWh2?Shn0/71`@:_[jl2Hf=S0Slo?`90@402Shn?0UAD
E09?Cdl2i^KV0_ooo`;:b/X2Lg=c0VEUI@8n?Sh2@T920S0`<09TI6@2WIfM0Ti>CP9gMgL2<C4a0]3@
d0:2PX82D51@0V5QH@:][Jd00e5AD@0000000002i>CT0Rl_;`9jNWX2Ogmo0TI6AP;alO42LG5a0X61
P@9nOWh2BTY:0VQXJ09TI6@2S8b<0UeMG@:0P80400000UMGE`:l_;`21PH60YBDU09@D502_Kfm0Xj>
SP9@D502/[:b0[>c/`9YJFT27AdM0WYjNP:QXJ42Bd];0VMWI`9`L702[Zj^0WYjNPB>SXh2O7al0Z6Q
X@;4a<@4WYjN0Z^[Z`8m?Cd2Cdm?0_ooo`;?cll2JFUY0WAdM0:][Jd2NWYj0Shn?P:0P802TI6A0R8R
8P:h^;P2Shn?0/?3``9UIFD2JVYZ0VUYJ@Goool100000Oooo`005Oooo`030000oooooooo00;oool2
U9BD0U5AD@8`<302Bd];0U]KF`;oool2Ti>C0WYjNP:f][H2YJFU0Zn_[`:1PH42IVIV0Te=C@:1PH42
Lg=c0XB4Q09_Kfl2DE5A0Y>CT`;UiND2O7al0Rd];@:n_[h2Ti>C0[Jf]P:d];@2f=SH0T10@0;<c<`2
OGem0/W9b@9:BTX2QHF50Xn?S`9TI6@2OWin0Shn?P;1`L42[jn_0V=SH`8o?cl2@4100Xn?S`9DE5@2
Cdm?0^KViP;oool2b/[:0W=cL`9UIFD2?Shn0T92@P8`<302I6AT0YfMW@9>CTh2MgMg0S4a<@;@d=02
PX:20U1@D09QHF42[Jf]00=ADE40000000000^CTi08_;bl2NWYj0WmoO`96ATH2lO7a0W5aL@:1PH42
OWin0TY:BP9XJ6P2I6AT0Xb<S09MGEd2P8201000009GEeL2_;bl0PH61P:DU9@2D51@0[fm_@:>SXh2
D51@0[:b/P:c/k<2JFUY0QdM7@9jNWX2XJ6Q0T];B`9WIfL2L71`0Zj^[P9jNWX4SXj>0WalO0:QXJ42
a<C419jNWP:[Zj/2?Cdm0Tm?C`;oool2clo?0VUYJ@9dM7@2[Jf]0WYjNP8n?Sh2P8200Y6AT@8R8R82
^;Rh0Xn?S`;3`l<2IFEU0VYZJP9YJFT5oooo0@00007oool001Goool00`000?ooooooo`02oooo0TQ8
B0:9RHT2<30`0X>3P`9YJFT2oooo0Y>CT`;fm_H2D51@0T=3@`800002Ng]k0WMgM`94A4@2_[jn0Y6A
T@9HF5P2?Shn0Xn?S`9`L702;Bd]0T51@@:4Q8@2/[:b0Y>CT`;no_h2SHf=0Tm?C`9DE5@2PH610VEU
I@9[Jf/2@4100XZ:RP9/K6`2Y:BT0X^;R`;3`l<2T92@0T92@P9kNg/2P8200Te=C@:OWil2Ogmo0WYj
NP9fMWH2Cdm?0TU9B@:b/[82Ng]k0/[:bP:][Jd2KFe]0WinOP9IFET2X:2P0T];B`;6a/H2VYZJ0Wem
O@9TI6@2Cdm?0UeMG@03clo?/;2`/;2`00A<C4`2R8R80WalO097AdL2[:b/0WIfMP9kNg/2R8R80QlO
7`800002QhN70WalO09?Cdl2AdM70[Vi^@94A4@2JFUY0]GEe@:5QHD2K6a/0YBDU097AdL2Shn?0Ve]
K@99BDT2UYJF0P0000:LW9`2NGUi0/?3``:HV9P2F5QH0Xn?S`:b/[82M7Ad0WQhN0;MgMd2H61P0VQX
J0:3Ph<2@4100X:2PP:`/;027alO0Zn_[`:@T902N7Qh0WinOP:8R8P2;Rh^0/71`@:CTi<2EEEE0^g]
k@9MGEd200000VQXJ09YJFT2QHF50^3Ph0:DU9@5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0TQ8B0:9RHT2<30`0X>3P`9YJFT2oooo0Y>CT`;fm_H2D51@0T=3@`800002Ng]k0WMgM`94A4@2
_[jn0Y6AT@9HF5P2?Shn0Xn?S`9`L702;Bd]0T51@@:4Q8@2/[:b0Y>CT`;no_h2SHf=0Tm?C`9DE5@2
PH610VEUI@9[Jf/2@4100XZ:RP9/K6`2Y:BT0X^;R`;3`l<2T92@0T92@P9kNg/2P8200Te=C@:OWil2
Ogmo0WYjNP9fMWH2Cdm?0TU9B@:b/[82Ng]k0/[:bP:][Jd2KFe]0WinOP9IFET2X:2P0T];B`;6a/H2
VYZJ0WemO@9TI6@2Cdm?0UeMG@03clo?/;2`/;2`00A<C4`2R8R80WalO097AdL2[:b/0WIfMP9kNg/2
R8R80QlO7`800002QhN70WalO09?Cdl2AdM70[Vi^@94A4@2JFUY0]GEe@:5QHD2K6a/0YBDU097AdL2
Shn?0Ve]K@99BDT2UYJF0P0000:LW9`2NGUi0/?3``:HV9P2F5QH0Xn?S`:b/[82M7Ad0WQhN0;MgMd2
H61P0VQXJ0:3Ph<2@4100X:2PP:`/;027alO0Zn_[`:@T902N7Qh0WinOP:8R8P2;Rh^0/71`@:CTi<2
EEEE0^g]k@9MGEd200000VQXJ09YJFT2QHF50^3Ph0:DU9@5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0UmOG`9TI6@2<c<c0Va/K08H61P2DU9B0Zj^[P9/K6`2bLW90VQXJ08[:b/2Ti>C0XN7
Q`:>SXh2EEEE0UQHF0;oool2Jf][0UUIF@8/;2`2410@0]oOg`9GEeL2C4a<0YFEU@;AdM42][Jf0VMW
I`:TY:@2R8R80Z:RXP:=SHd2820P0P00008U9BD2BDU90UeMG@:^[Zh2Bd];0U]KF`:_[jl2aLG50S/k
>`9KFe/2GemO0_knoP::RXX2GUiN0Xb<S09@D502UYJF0S<c<`9aLG42MgMg0V1PH0800002UIFE0P00
009DE5@2EeMG0Vi^KP:YZJT4E5AD00?Bd];oooooool00TE5A@:6QXH2NGUi0WIfMP;2`/8251@D0XR8
R09cLg<2QXJ60U9BDP:3Ph<2U9BD0Ti>CP8_;bl2[Jf]0X:2PP9oOgl2h^;R0Q/K6`9MGEd2fm_K0XR8
R09iNGT2oooo0Z2PX0:^[Zh2EeMG0ZZZZP;<c<`2Kfm_0Tm?C`:FUYH2T92@0UIFEP9iNGT2Q8B40]_K
f`:7QhL2;Rh^0QdM7@9:BTX2/;2`0]3@d0:6QXH2DE5A0TM7A`:XZ:P2W9bL0WAdM09jNWX2ADE50V9R
HP9;Bd/200000Xf=S@9TI6@2IFEU0YnOW`;Ng]h2:RXZ0/O7a`:][Jd5oooo0@00007oool001Goool2
00000oooo`9OGel2I6AT0S<c<`9/K6`261PH0U9BDP:^[Zh2K6a/0/W9b@9XJ6P2:b/[0Y>CT`:7QhL2
SXj>0UEEE@9HF5P2oooo0V][J`9IFET2;2`/0Q0@40;Ogml2EeMG0Ta<C0:EUID2dM7A0[Jf]P9WIfL2
Y:BT0XR8R0:RXZ82SHf=0R0P808000029BDU0TU9B@9MGEd2[Zj^0T];B`9KFe/2[jn_0/G5a@8k>c/2
Fe]K0UmOG`;no_h2RXZ:0UiNGP:<S8`2D51@0YJFUP8c<c<2LG5a0WMgM`9PH60200000YFEU@800002
E5AD0UMGE`9^KVh2ZJVY15ADE003d];Boooooooo0095ADD2QXJ60WUiN@9fMWH2`/;20Q@D50:8R8P2
Lg=c0XJ6QP9BDU82Ph>30YBDU09>CTh2;bl_0Zf][@:2PX82Ogmo0^;RhP8K6a/2GEeM0]_Kf`:8R8P2
NGUi0_ooo`:PX:02[Zj^0UMGE`:ZZZX2c<c<0Vm_K`9?Cdl2UYJF0Y2@T09FEUH2NGUi0XB4Q0;Kfm/2
QhN70Rh^;P8M7Ad2BTY:0[2`/0;@d=02QXJ60U5AD@97AdL2Z:RX0YbLW09dM7@2NWYj0TE5A@9RHV82
Bd];0P0000:=SHd2I6AT0VEUI@:OWil2g]kN0RXZ:P;7alL2[Jf]1?ooo`800001oooo000Eoooo00<0
003oooooool00_ooo`9CDe<2RXZ:0Ve]K@:BTY82V9RH0ZVYZ@:b/[82Q8B40WQhN08X:2P2BDU90Zf]
[@:a/K42ATI60^OWi`9>CTh2Shn?0R`/;09ZJVX2FUYJ0YJFUP:LW9`2IVIV0Tm?C`800002RXZ:0S4a
<@:8R8P2Kfm_0[:b/P90@402ADE50UiNGP:XZ:P27QhN0_ooo`;Zj^X2[jn_0VIVIPBHV9P2PH610YRH
V0:<S8`2VIVI0Xj>SP9hN7P2;Rh^0YFEU@:2PX82OGem0YfMW@:JVYX2n?Sh0U9BDP8^;Rh261PH0STi
>@8S8b<2MgMg0UYJFP:LW9`2[jn_0S`l?0039BDU92@T92@T008U9BD2ZJVY0W1`L09nOWh2L71`0Q@D
50:3Ph<2N7Qh0WYjNP:5QHD2>CTi0^c/k09EEED2f=SH0X>3P`:n_[h2clo?0TU9B@9RHV82KFe]0Q@D
50AmOGd2fm_K0P00008k>c/2Jf][0XN7Q`9kNg/2]KFe0VMWI`9FEUH2Ng]k0XV9R@:e]KD2Cdm?0Q0@
40:RXZ82o_kn0W5aL@9dM7@2WIfM0WYjNP:BTY82MGEe0TQ8B08B4Q82LG5a0WUiN@:DU9@2ZJVY0WQh
N09gMgL2bLW90YJFUP9HF5P2H61P0SLg=`8=3@d2GUiN0Q@D50;Zj^X5oooo0@00007oool001Goool0
0`000?ooooooo`02oooo0U=CD`::RXX2KFe]0Y:BTP:HV9P2ZJVY0[:b/P:4Q8@2N7Qh0RPX:099BDT2
[Jf]0[6a/@96ATH2inOW0Ti>CP:?Shl2;2`/0VYZJP9JFUX2UYJF0YbLW09VIVH2Cdm?0P0000::RXX2
<C4a0XR8R09_Kfl2/[:b0T10@095ADD2GUiN0ZRXZ08N7Qh2oooo0^[ZjP:_[jl2IVIV19RHV0:1PH42
V9RH0Xb<S0:IVIT2SXj>0WQhN08^;Rh2UIFE0X:2PP9mOGd2WIfM0YZJVP;hn?P2DU9B0Rh^;P8H61P2
>CTi0R<S8`9gMgL2FUYJ0YbLW0:_[jl2?3`l00<U9BDT92@T92@00RDU9@:YZJT2L71`0WinOP9`L702
51@D0X>3P`9hN7P2NWYj0XF5Q@8i>CT2k>c/0UEEE@;Hf=P2Ph>30[jn_P;?cll2BDU90V9RHP9]KFd2
51@D17emO@;Kfm/200000S/k>`9[Jf/2QhN70W]kN`:e]KD2IfMW0UIFEP9kNg/2RHV90[Fe]@9?Cdl2
410@0Z:RXP;no_h2LG5a0WAdM0:MWId2NWYj0Y:BTP9eMGD2B4Q80Q8B4P9aLG42NGUi0YBDU0:YZJT2
N7Qh0WMgM`;9bLT2UYJF0UQHF09PH602=cLg0Pd=3@9NGUh251@D0^[ZjPGoool100000Oooo`005Ooo
o`030000oooooooo00;oool2Ogmo0VYZJP8l?3`2<C4a0Y6AT@:HV9P2?clo0XJ6QP:8R8P2g=cL0UAD
E09YJFT2J6QX0VEUI@:PX:02?Shn0W=cL`:DU9@2oooo0X>3P`:LW9`2H61P0SDe=@:WYjL2UIFE0X^;
R`9gMgL2k^k^0Va/K0:HV9P2][Jf0/30`0;9bLT2O7al0WinOP9XJ6P2RHV90U=CD`800002XZ:R0XZ:
RP:ATI42Kfm_0XJ6QP::RXX2FEUI0TE5A@9YJFT2oooo0WinOP:9RHT2ZJVY0Y^KV`:2PX82=3@d0YNG
U`8h>3P2/;2`0TU9B@:>SXh2RHV90VUYJ@9jNWX2e]KF00=^KVk]kNg]kNd00X^;R`:3Ph<2N7Qh0Wem
O@9nOWh2^k^k0WMgM`9oOgl2NGUi0X20P09HF5P2GUiN0UIFEP;fm_H200000Y6AT@:4Q8@2NWYj0X>3
P`:<S8`2GEeM0X:2PP9SHf<2U9BD0R4Q8@8M7Ad2OWin0Y:BTP:6QXH2];Bd0V5QH@9`L702PH610Z6Q
X@9kNg/2dM7A0W=cL`90@402oooo0^[ZjP98B4P28b<S0W]kN`;=cLd4oooo0VMWI`:BTY82IVIV0^GU
i@:4Q8@2B4Q80WAdM09_Kfl2GUiN0T];B`;oool2[jn_0XF5Q@9[Jf/2QXJ60/[:bPGoool100000Ooo
o`005Oooo`030000oooooooo00;oool2Ogmo0VYZJP8l?3`2<C4a0Y6AT@:HV9P2?clo0XJ6QP:8R8P2
g=cL0UADE09YJFT2J6QX0VEUI@:PX:02?Shn0W=cL`:DU9@2oooo0X>3P`:LW9`2H61P0SDe=@:WYjL2
UIFE0X^;R`9gMgL2k^k^0Va/K0:HV9P2][Jf0/30`0;9bLT2O7al0WinOP9XJ6P2RHV90U=CD`800002
XZ:R0XZ:RP:ATI42Kfm_0XJ6QP::RXX2FEUI0TE5A@9YJFT2oooo0WinOP:9RHT2ZJVY0Y^KV`:2PX82
=3@d0YNGU`8h>3P2/;2`0TU9B@:>SXh2RHV90VUYJ@9jNWX2e]KF00=^KVk]kNg]kNd00X^;R`:3Ph<2
N7Qh0WemO@9nOWh2^k^k0WMgM`9oOgl2NGUi0X20P09HF5P2GUiN0UIFEP;fm_H200000Y6AT@:4Q8@2
NWYj0X>3P`:<S8`2GEeM0X:2PP9SHf<2U9BD0R4Q8@8M7Ad2OWin0Y:BTP:6QXH2];Bd0V5QH@9`L702
PH610Z6QX@9kNg/2dM7A0W=cL`90@402oooo0^[ZjP98B4P28b<S0W]kN`;=cLd4oooo0VMWI`:BTY82
IVIV0^GUi@:4Q8@2B4Q80WAdM09_Kfl2GUiN0T];B`;oool2[jn_0XF5Q@9[Jf/2QXJ60/[:bPGoool1
00000Oooo`005Oooo`030000oooooooo00Coool2j^[Z0VQXJ0:<S8`2<30`0U]KF`8Q8B42oooo0P00
009JFUX2Zj^[0XR8R0:3Ph<2[Zj^0TY:BP:XZ:P2I6AT0YFEU@:WYjL2oooo0STi>@9dM7@2ZZZZ0UYJ
FP:DU9@2RHV90[6a/@9/K6`2Q8B40TI6AP:`/;0200000YBDU09jNWX2YjNW0/o?c`;MgMd2Zj^[0]OG
e`:0P802c<c<0[bl_09/K6`2IVIV0[:b/P:DU9@2K6a/0YnOW`:k^k/200000UADE09CDe<2h^;R0UMG
E`9ADE42IfMW0U]KF`8R8R82SHf=0Vi^KP9aLG42SXj>0W]kN`:FUYH00cdm?F=SHf=SH`02KFe]17in
OP9oOgl2O7al0Ti>CP:0P802NWYj0WalO0:4Q8@2R8R80YVIV@800002no_k0Sdm?@8T92@2];Bd0YjN
WP9FEUH2U9BD0[Jf]P9nOWh2:RXZ0[:b/P:f][H2[jn_0^3Ph0892@T2Lg=c0YjNWP:WYjL2IVIV0X:2
PP:HV9P2TY:B0XF5Q@9fMWH2/;2`0[Bd]09WIfL2ATI60_ooo`9^KVh2De=C0WmoO`800002Shn?0QhN
7P;:b/X2a<C40YRHV0:QXJ42=3@d0WIfMP:7QhL2Ogmo0Te=C@9QHF42G5aL0[Fe]@9bLW82GUiN1Ooo
o`400001oooo000Eoooo00<0003oooooool01?ooo`;Zj^X2J6QX0Xb<S08`<302Fe]K0R4Q8@;oool2
00000UYJFP:[Zj/2R8R80X>3P`:^[Zh2BTY:0ZRXZ09TI6@2UIFE0ZNWY`;oool2>CTi0WAdM0:ZZZX2
FUYJ0YBDU0:9RHT2/K6a0Va/K0:4Q8@2ATI60[2`/0800002U9BD0WYjNP:WYjL2clo?0]gMg@:[Zj/2
emOG0X20P0;<c<`2_;bl0Va/K09VIVH2/[:b0YBDU09/K6`2WinO0[^k^`800002E5AD0U=CD`;Rh^82
EeMG0U5AD@9WIfL2Fe]K0R8R8P:=SHd2KVi^0W5aL@:>SXh2Ng]k0YJFUP03?CdmHf=SHf=S009]KFd4
OWin0WmoO`9lO7`2CTi>0X20P09jNWX2O7al0XB4Q0:8R8P2VIVI0P0000;kno/2?Cdm0R@T90:d];@2
WYjN0UIFEP:DU9@2][Jf0WinOP8Z:RX2/[:b0[Jf]P:_[jl2h>3P0PT92@9cLg<2WYjN0ZNWY`9VIVH2
PX:20YRHV0:BTY82QHF50WIfMP:`/;02];Bd0VMWI`96ATH2oooo0Vi^KP9CDe<2Ogmo0P0000:?Shl2
7QhN0/[:bP;4a<@2V9RH0Z6QX@8d=3@2MWIf0XN7Q`9oOgl2CDe=0V5QH@9LG5`2]KFe0W9bLP9NGUh5
oooo0@00007oool001Goool00`000?ooooooo`02oooo0]kNgP9lO7`2?3`l0]?Cd`9]KFd29RHV0_oo
o`8b<S82jn_[0VATI09JFUX2NGUi0Zn_[`:EUID2^k^k0Xn?S`9]KFd2^;Rh0ZZZZP9EEED2=cLg0ZFU
Y@9gMgL2U9BD0S4a<@:o_kl2h>3P0P0000;5aLD2[Jf]0Y2@T0:0P802ZJVY0Rl_;`;3`l<2HV9R0]GE
e@96ATH2I6AT0/S8b0:FUYH2/k>c0Z:RXPBd];@2d=3@0U]KF`99BDT24Q8B0PT92@8[:b/2l?3`0R`/
;0:@T902@d=30Y6AT@9OGel2Ti>C0X:2PP9kNg/2P8200W1`L09mOGd2oooo00?no_imOGemOGd00XR8
R0AiNGT2Jf][0X20P09dM7@4Ng]k0WYjNP8l?3`2KFe]0X>3P`;oool2c/k>0_ooo`:g]kL4oooo0TA4
A09LG5`2MGEe0[Zj^P:<S8`2XJ6Q0V9RHP:3Ph<26a/K0Z6QX@:o_kl2NGUi0Xn?S`9IFET2OGem0/K6
aP:LW9`2/;2`0[:b/P9hN7P2kNg]0YNGU`9nOWh2Lg=c0VUYJ@9GEeL2820P0UmOG`93@d<2g]kN0Ve]
K@8l?3`2PX:20V][J`8;2`/2R8R80W5aL@9nOWh2JVYZ0WQhN09;Bd/2a/K60_ooo`9WIfL5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0]kNgP9lO7`2?3`l0]?Cd`9]KFd29RHV0_ooo`8b<S82
jn_[0VATI09JFUX2NGUi0Zn_[`:EUID2^k^k0Xn?S`9]KFd2^;Rh0ZZZZP9EEED2=cLg0ZFUY@9gMgL2
U9BD0S4a<@:o_kl2h>3P0P0000;5aLD2[Jf]0Y2@T0:0P802ZJVY0Rl_;`;3`l<2HV9R0]GEe@96ATH2
I6AT0/S8b0:FUYH2/k>c0Z:RXPBd];@2d=3@0U]KF`99BDT24Q8B0PT92@8[:b/2l?3`0R`/;0:@T902
@d=30Y6AT@9OGel2Ti>C0X:2PP9kNg/2P8200W1`L09mOGd2oooo00?no_imOGemOGd00XR8R0AiNGT2
Jf][0X20P09dM7@4Ng]k0WYjNP8l?3`2KFe]0X>3P`;oool2c/k>0_ooo`:g]kL4oooo0TA4A09LG5`2
MGEe0[Zj^P:<S8`2XJ6Q0V9RHP:3Ph<26a/K0Z6QX@:o_kl2NGUi0Xn?S`9IFET2OGem0/K6aP:LW9`2
/;2`0[:b/P9hN7P2kNg]0YNGU`9nOWh2Lg=c0VUYJ@9GEeL2820P0UmOG`93@d<2g]kN0Ve]K@8l?3`2
PX:20V][J`8;2`/2R8R80W5aL@9nOWh2JVYZ0WQhN09;Bd/2a/K60_ooo`9WIfL5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0]kNgP;Ti>@2[Zj^0WmoO`800002>c/k0XB4Q0@00002IfMW0XB4
Q0:LW9`2L71`0R0P808^;Rh2EEEE0S<c<`9mOGd2oooo0[2`/09QHF42W9bL0SPh>09oOgl2P8200XJ6
QP:8R8P2WYjN0_7al@9QHF42Y:BT0P0000;?cll2dM7A0P@4109=CDd2YjNW0U]KF`:i^KT2S8b<0TI6
AP8i>CT2Hf=S0YbLW09EEED2DE5A0T];B`9ZJVX2Cdm?0[6a/@;emOD2QhN70XB4Q08`<302oooo0RHV
9P9RHV82TY:B0UIFEP8n?Sh2IVIV0XV9R@9QHF42IFEU00=<C4aFEUIFEUH00YNGU`:JVYX2ZZZZ0_7a
l@9WIfL2RXZ:0XB4Q09hN7P2P8200WIfMP:ZZZX2MgMg0Y2@T09VIVH2FUYJ0X61P@800002T92@0Y^K
V`:RXZ82HF5Q0Tm?C`:/[:`2lo?c0Z:RXP:1PH42MWIf0S8b<P8V9RH210@40_ooo`:j^[X2M7Ad0U1@
D0:UYJD2YjNW0Xj>SP9_Kfl2KFe]0X>3P`:UYJD2XZ:R0R4Q8@8Y:BT2@d=30W1`L0800002Lg=c0XN7
Q`8V9RH2oooo0X:2PP800002PH610WalO08Q8B42NWYj0Z>SX`:LW9`2ZZZZ0UIFEP:7QhL5oooo0@00
007oool001Goool200000oooo`;Ng]h2i>CT0Zj^[P9oOgl200000S/k>`:4Q8@400000VMWI`:4Q8@2
W9bL0W1`L08P8202;Rh^0UEEE@8c<c<2OGem0_ooo`:`/;02HF5Q0YbLW08h>3P2Ogmo0X20P0:6QXH2
R8R80YjNWP;alO42HF5Q0ZBTY0800002clo?0]7Ad@8410@2CDe=0ZNWY`9KFe/2^KVi0Xb<S096ATH2
>CTi0V=SH`:LW9`2EEEE0U5AD@9;Bd/2JVYZ0Tm?C`:a/K42mOGe0XN7Q`:4Q8@2<30`0_ooo`8V9RH2
HV9R0Y:BTP9FEUH2?Shn0VIVIP:9RHT2HF5Q0VEUI@03C4a<EUIFEUIF00:GUiL2VYZJ0ZZZZP;alO42
IfMW0XZ:RP:4Q8@2N7Qh0X20P09fMWH2ZZZZ0WMgM`:@T902IVIV0UYJFP:1PH4200000Y2@T0:KVi/2
XZ:R0V5QH@9?Cdl2[:b/0_?cl`:RXZ82PH610WIfMP8b<S829RHV0P@410;oool2^[Zj0WAdM09@D502
YJFU0ZNWY`:>SXh2Kfm_0Ve]K@:3Ph<2YJFU0Z:RXP8Q8B42:BTY0T=3@`9`L70200000W=cL`:7QhL2
9RHV0_ooo`:2PX8200000X61P@9lO7`28B4Q0WYjNP:SXj<2W9bL0ZZZZP9FEUH2QhN71?ooo`800001
oooo000Eoooo00<0003oooooool00_ooo`8j>SX2IVIV0YRHV0;^k^h2aLG50WemO@8V9RH2Y:BT0Z2P
X0800002LG5a0UADE099BDT2h>3P0_ooo`9gMgL2@T920[Vi^@8=3@d2UIFE0SHf=P:IVIT2aLG50Xb<
S0;Lg=`2>SXj0^;RhP9;Bd/2MgMg0Xb<S0800002A4A40WalO0;=cLd2PX:20V=SH`;]kNd2Kfm_0U5A
D@8m?Cd2PX:20YjNWP:i^KT2Z:RX0RPX:0800002`L710Sdm?@91@D42oooo0SXj>P9QHF42IfMW0X:2
PP9eMGD2L71`0W]kN`:9RHT2RXZ:0WYjNP9CDe<2Hf=S0U9BDP:SXj<00hJ6QXZ:RXZ:RP02UIFE0TE5
A@9QHF42YZJV0WQhN09QHF42E5AD0XV9R@:0P802FUYJ0XB4Q0::RXX271`L0[2`/09DE5@261PH0U5A
D@9LG5`24Q8B0P/;2`9IFET2I6AT0S8b<P8n?Sh2WYjN0Ta<C0:4Q8@2`<300WYjNP:OWil2=CDe0UQH
F09hN7P2Q8B40XZ:RP8o?cl2De=C0WemO@;5aLD23`l?0SHf=P:RXZ82GEeM0_ooo`800002/k>c0/k>
cP:b/[823Ph>0Q`L70;Shn<2k>c/0WalO0:0P802NWYj0X>3P`9kNg/2NGUi0WMgM`96ATH2:BTY0]gM
g@Goool100000Oooo`005Oooo`030000oooooooo00;oool2>SXj0VIVIP:HV9P2k^k^0/G5a@9mOGd2
9RHV0ZBTY0:PX:0200000W5aL@9DE5@2BDU90^3Ph0;oool2MgMg0T92@P:i^KT23@d=0YFEU@8f=SH2
VIVI0/G5a@:<S8`2g=cL0SXj>P;Rh^82Bd];0WMgM`:<S8`200000TA4A09lO7`2cLg=0X:2PP9SHf<2
kNg]0Vm_K`9ADE42?Cdm0X:2PP:NWYh2^KVi0ZRXZ08X:2P200000/71`@8m?Cd2@D510_ooo`8j>SX2
HF5Q0VMWI`:2PX82MGEe0W1`L09kNg/2RHV90XZ:RP9jNWX2De=C0V=SH`9BDU82Xj>S00>6QXJ:RXZ:
RXX00YFEU@95ADD2HF5Q0ZJVYP9hN7P2HF5Q0UADE0:9RHT2P8200UYJFP:4Q8@2RXZ:0Q`L70:`/;02
E5AD0QPH609ADE42G5aL0Q8B4P8;2`/2FEUI0VATI08b<S82?Shn0YjNWP9<C4`2Q8B40/30`09jNWX2
WinO0SDe=@9HF5P2N7Qh0XB4Q0::RXX2?clo0U=CD`9mOGd2aLG50Pl?3`8f=SH2XZ:R0UeMG@;oool2
00000[>c/`;>c/h2/[:b0Ph>3P8L71`2hn?S0^c/k09lO7`2P8200WYjNP:3Ph<2Ng]k0WUiN@9gMgL2
ATI60RTY:@;MgMd5oooo0@00007oool001Goool00`000?ooooooo`02oooo0YFEU@:QXJ42IFEU0PD5
1@;7alL2QhN70Rh^;P:3Ph<2P8200VMWI`8_;bl2WIfM0XF5Q@:JVYX2ZZZZ0V5QH@;alO42oooo0Wem
O@:=SHd2GEeM0[Ng]`9mOGd2IFEU0W]kN`:4Q8@2GEeM0[Zj^P90@402E5AD0YFEU@:?Shl2Ti>C0QPH
60:0P802F5QH0T];B`8J6QX29BDU0YFEU@9CDe<2Ti>C0W9bLP8l?3`2JVYZ0/;2`P8e=CD2jn_[0TA4
A09IFET2EeMG0TE5A@:1PH42Lg=c0X61P@9@D502^;Rh0T];B`:?Shl2B4Q80Vi^KP:@T902L71`0TE5
A@033Ph>`L71`L7100:3Ph<2WIfM0P0000;4a<@2?clo0Y:BTP:GUiL2HF5Q0X>3P`:4Q8@2c/k>0V9R
HP:6QXH2F5QH0Q0@40:f][H2Kfm_0WalO0:DU9@2QXJ60[fm_@9PH602oooo0P0000:4Q8@2QHF50P00
00:EUID2Jf][0P0000;GemL2Rh^;0WmoO`:HV9P2G5aL0Zj^[P:b/[82Ph>30Vm_K`:IVIT2WYjN0_3`
l09KFe/2i>CT0Vi^KP9`L702?Cdm0X>3P`:h^;P2oooo0Ta<C09^KVh2LG5a0WalO09jNWX2MgMg0Win
OP9fMWH2W9bL0Z^[Z`97AdL200001Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:EUID2
XJ6Q0VEUI@851@D2alO70XN7Q`8^;Rh2Ph>30X20P09WIfL2;bl_0YfMW@:5QHD2VYZJ0ZZZZP9QHF42
lO7a0_ooo`9mOGd2SHf=0UeMG@:g]kL2OGem0VEUI@9kNg/2Q8B40UeMG@:j^[X2@4100UADE0:EUID2
Shn?0Y>CT`8H61P2P8200UQHF09;Bd/26QXJ0RDU9@:EUID2De=C0Y>CT`9bLW82?3`l0VYZJP;2`/82
=CDe0^_[j`94A4@2FEUI0UMGE`95ADD2PH610W=cL`:1PH42D51@0[Rh^09;Bd/2Shn?0TQ8B09^KVh2
T92@0W1`L095ADD00`h>3/71`L71`@02Ph>30YfMW@800002a<C40Slo?`:BTY82UiNG0V5QH@:3Ph<2
Q8B40/k>cP9RHV82QXJ60UQHF08@4102][Jf0Vm_K`9lO7`2U9BD0XJ6QP:m_Kd2H61P0_ooo`800002
Q8B40XF5Q@800002UIFE0V][J`800002emOG0X^;R`9oOgl2V9RH0UaLG0:^[Zh2/[:b0X>3P`9_Kfl2
VIVI0YjNWP;`l?02Fe]K0^CTi09^KVh2L71`0Sdm?@:3Ph<2^;Rh0_ooo`9<C4`2KVi^0W5aL@9lO7`2
NWYj0WMgM`9nOWh2MWIf0YbLW0:[Zj/2AdM70P0000Goool100000Oooo`005Oooo`030000oooooooo
00;oool2N7Qh0T92@P9RHV82eMGE0YjNWP9gMgL2GUiN0W9bLP9QHF42OWin0VEUI@;Cdm<2e=CD0ZBT
Y0:7QhL2oooo0Va/K0:6QXH4MgMg0VATI09bLW82RXZ:0[^k^`::RXX2QXJ60Va/K08n?Sh2`/;20XF5
Q@;oool2QXJ60U]KF`9[Jf/2][Jf0PT92@8E5AD2=3@d0_ooo`800002LG5a0XV9R@8_;bl2Ng]k0QlO
7`9WIfL2P8200W5aL@9_Kfl2MWIf0U=CD`:LW9`2SXj>0WemO@:3Ph<2Cdm?0YnOW`;Shn<2FEUI0[Vi
^@;MgMd2NWYj0Ti>CP:ATI400gEeMINGUiNGU`02B4Q80Y6AT@9mOGd2RXZ:0Y2@T0:^[Zh24Q8B0_gm
o@:QXJ4200000]kNgP:=SHd2De=C0UUIF@9ZJVX2Q8B40W1`L0;1`L42SXj>0_KfmP;hn?P2M7Ad0]3@
d09kNg/2emOG0YfMW@9QHF42[Jf]0TU9B@:BTY82Shn?0X61P@:4Q8@2@T920_7al@;oool2B4Q80[bl
_09/K6`2PX:20W1`L09XJ6P2P8200Vm_K`9iNGT2FEUI0V=SH`:l_;`2Kfm_0WemO@:j^[X2SXj>0X:2
PP9oOgl2Ph>30WemO@9lO7`2Ng]k0YBDU0:JVYX2K6a/0[Ng]`Goool100000Oooo`005Oooo`030000
oooooooo00;oool2N7Qh0T92@P9RHV82eMGE0YjNWP9gMgL2GUiN0W9bLP9QHF42OWin0VEUI@;Cdm<2
e=CD0ZBTY0:7QhL2oooo0Va/K0:6QXH4MgMg0VATI09bLW82RXZ:0[^k^`::RXX2QXJ60Va/K08n?Sh2
`/;20XF5Q@;oool2QXJ60U]KF`9[Jf/2][Jf0PT92@8E5AD2=3@d0_ooo`800002LG5a0XV9R@8_;bl2
Ng]k0QlO7`9WIfL2P8200W5aL@9_Kfl2MWIf0U=CD`:LW9`2SXj>0WemO@:3Ph<2Cdm?0YnOW`;Shn<2
FEUI0[Vi^@;MgMd2NWYj0Ti>CP:ATI400gEeMINGUiNGU`02B4Q80Y6AT@9mOGd2RXZ:0Y2@T0:^[Zh2
4Q8B0_gmo@:QXJ4200000]kNgP:=SHd2De=C0UUIF@9ZJVX2Q8B40W1`L0;1`L42SXj>0_KfmP;hn?P2
M7Ad0]3@d09kNg/2emOG0YfMW@9QHF42[Jf]0TU9B@:BTY82Shn?0X61P@:4Q8@2@T920_7al@;oool2
B4Q80[bl_09/K6`2PX:20W1`L09XJ6P2P8200Vm_K`9iNGT2FEUI0V=SH`:l_;`2Kfm_0WemO@:j^[X2
SXj>0X:2PP9oOgl2Ph>30WemO@9lO7`2Ng]k0YBDU0:JVYX2K6a/0[Ng]`Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2Xj>S0Va/K0:TY:@2^[Zj0V=SH`:CTi<2ATI60WemO@Coool2MWIf0V1P
H09KFe/2KVi^0QXJ6P8M7Ad2Ogmo0WIfMP;dm?@2NWYj0]GEe@9/K6`2YjNW0__kn`:a/K42][Jf0ZBT
Y0:8R8P2oooo0U1@D0800002QXJ60W5aL@8O7al23`l?0]?Cd`:^[Zh200000]kNgP8D51@2G5aL0Q`L
70:OWil2eMGE0WQhN0:8R8P2MWIf0X:2PP9jNWX2Ph>30XJ6QP8L71`2HV9R0VUYJ@9BDU82OGem0VMW
I`:0P802/;2`0ZFUY@;emOD2QhN70^OWi`>0P802OGem0UIFEP:DU9@2JFUY0[Fe]@9nOWh2bLW90UmO
G`9KFe/2m?Cd0[Vi^@9VIVH2MWIf0UYJFP:VYZH2IFEU0W9bLP:_[jl2NGUi0ZZZZP9YJFT2820P0Z^[
Z`9LG5`2J6QX0TY:BP:_[jl2ZJVY0YjNWP:=SHd2L71`0SLg=`9SHf<2QhN70V=SH`:7QhL2GEeM0Y2@
T09cLg<4OGem0X61P@:4Q8@2NGUi0X61P@9mOGd2Q8B40X20P09CDe<2KVi^0Ve]K@:XZ:P200000W]k
N`9SHf<2GEeM0X20P0:1PH42H61P0XZ:RP9gMgL2Ph>30U]KF`Goool100000Oooo`002_ooo`030000
oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool00_ooo`:SXj<2K6a/0ZBTY0:j^[X2
Hf=S0Y>CT`96ATH2OGem1?ooo`9fMWH2H61P0U]KF`9^KVh26QXJ0QdM7@9oOgl2MWIf0_Cdm09jNWX2
eMGE0Va/K0:WYjL2no_k0[6a/@:f][H2Y:BT0XR8R0;oool2D51@0P0000:6QXH2LG5a0QlO7`8?3`l2
dm?C0Zj^[P800002g]kN0Q@D509LG5`271`L0YnOW`;EeMD2N7Qh0XR8R09fMWH2PX:20WYjNP:3Ph<2
QXJ60Q`L709RHV82JFUY0U9BDP9mOGd2IfMW0X20P0:`/;02YJFU0_Gem@:7QhL2inOW0h20P09mOGd2
EUIF0YBDU09YJFT2]KFe0WinOP;9bLT2GemO0U]KF`;dm?@2^KVi0VIVIP9fMWH2FUYJ0ZJVYP9UIFD2
LW9b0Zn_[`9iNGT2ZZZZ0VUYJ@8P8202Zj^[0UaLG09XJ6P2BTY:0Zn_[`:YZJT2WYjN0Xf=S@9`L702
=cLg0V=SH`:7QhL2Hf=S0XN7Q`9MGEd2T92@0W=cL`AmOGd2PH610XB4Q09iNGT2PH610WemO@:4Q8@2
P8200U=CD`9^KVh2KFe]0ZRXZ0800002Ng]k0V=SH`9MGEd2P8200X61P@9PH602RXZ:0WMgM`:3Ph<2
Fe]K1Oooo`400001oooo0009oooo00<0003oool000000_ooo`030000oooo000000Coool00`000?oo
ooooo`02oooo0[>c/`9cLg<2QXJ60W9bLP9oOgl2N7Qh0Xn?S`;oool2JFUY0Y>CT`:;Rh/2Fe]K0ZNW
Y`:5QHD2][Jf0UUIF@9lO7`2CDe=0T51@@;oool2Jf][0Y>CT`9?Cdl2_;bl0^c/k09XJ6P2f][J0Xb<
S0:NWYh261PH0YbLW09fMWH2C4a<0Rl_;`:>SXh2=CDe0WMgM`9=CDd2EEEE0WemO@:^[Zh200000VAT
I09gMgL2MWIf0WQhN09nOWh2N7Qh0X61P@9[Jf/2fm_K0S<c<`9oOgl2D51@0X20P09dM7@2NGUi0X61
P@9/K6`2RHV90Zf][@:9RHT2/;2`0ZBTY0=kNg/2QHF50WEeM@9lO7`2PH610U]KF`:5QHD2Q8B40/c<
c09ZJVX26ATI0UiNGP9RHV82D51@0QdM7@:2PX82820P0XB4Q0;WinL2D51@0XJ6QP:@T902HV9R0ZRX
Z09SHf<2YZJV0Y:BTP:=SHd2MGEe0XF5Q@:RXZ82@4100YnOW`:0P802QXJ60WEeM@:4Q8@2MWIf0X20
P0AkNg/2NWYj0WUiN@9mOGd2NWYj0WalO0AiNGT2PH610UQHF09LG5`2FEUI0Vm_K`9VIVH2G5aL0U]K
F`:4Q8@2NWYj0WQhN0:1PH42OWin0WalO09mOGd5oooo0@00007oool000Woool00`000?ooo`000002
oooo00<0003oool000001?ooo`800003oooo0[>c/`9cLg<2QXJ60W9bLP9oOgl2N7Qh0Xn?S`;oool2
JFUY0Y>CT`:;Rh/2Fe]K0ZNWY`:5QHD2][Jf0UUIF@9lO7`2CDe=0T51@@;oool2Jf][0Y>CT`9?Cdl2
_;bl0^c/k09XJ6P2f][J0Xb<S0:NWYh261PH0YbLW09fMWH2C4a<0Rl_;`:>SXh2=CDe0WMgM`9=CDd2
EEEE0WemO@:^[Zh200000VATI09gMgL2MWIf0WQhN09nOWh2N7Qh0X61P@9[Jf/2fm_K0S<c<`9oOgl2
D51@0X20P09dM7@2NGUi0X61P@9/K6`2RHV90Zf][@:9RHT2/;2`0ZBTY0=kNg/2QHF50WEeM@9lO7`2
PH610U]KF`:5QHD2Q8B40/c<c09ZJVX26ATI0UiNGP9RHV82D51@0QdM7@:2PX82820P0XB4Q0;WinL2
D51@0XJ6QP:@T902HV9R0ZRXZ09SHf<2YZJV0Y:BTP:=SHd2MGEe0XF5Q@:RXZ82@4100YnOW`:0P802
QXJ60WEeM@:4Q8@2MWIf0X20P0AkNg/2NWYj0WUiN@9mOGd2NWYj0WalO0AiNGT2PH610UQHF09LG5`2
FEUI0Vm_K`9VIVH2G5aL0U]KF`:4Q8@2NWYj0WQhN0:1PH42OWin0WalO09mOGd4oooo0P00007oool0
00[oool01`000?ooooooooooo`000?ooo`000004oooo00<0003oooooool00_ooo`9cLg<2Rh^;0YJF
UP9fMWH2Ph>30W=cL`;oool2De=C0XJ6QP9jNWX2MgMg0X61P@:BTY82Xj>S0S8b<P:9RHT2LW9b0]gM
g@;Rh^82GUiN0XR8R09fMWH2QXJ60VIVIP9JFUX2clo?0UIFEPB7QhL2MgMg0WEeM@9oOgl24a<C0VIV
IP9RHV82R8R80X20P0:OWil2dM7A0Z6QX@:ATI42XZ:R1820P09nOWh2OGem0WYjNP:2PX82IVIV0[fm
_@:JVYX2G5aL0[:b/P:4Q8@2O7al0WemO@9lO7`2NWYj0X:2PP9hN7P2LG5a0Ve]K@9eMGD2000000=k
Ng]lO7alO7`00WUiN@9nOWh2Ng]k0WemO@:6QXH2NGUi0X61P@9lO7`2J6QX0P00008U9BD2][Jf0^;R
hP;EeMD2Kfm_0/O7a`;oool2PX:20WUiN@9kNg/2LW9b0XJ6QP9WIfL2YJFU0Ve]K@9`L702b/[:0WAd
M09_Kfl2Y:BT0YRHV09jNWX2N7Qh0WUiN@9lO7`2N7Qh0WinOP9jNWX6O7al0WemO@IlO7`2OGem0WAd
M09ADE42HV9R0WemO@:m_Kd2@T920/O7a`9iNGT2RHV90WQhN09mOGd2O7al17]kN`9lO7`2NWYj1Ooo
o`400001oooo0009oooo00<0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`02
oooo0W=cL`:;Rh/2UYJF0WIfMP:3Ph<2Lg=c0_ooo`9CDe<2QXJ60WYjNP9gMgL2PH610Y:BTP:SXj<2
<S8b0XV9R@9bLW82gMgM0^;RhP9NGUh2R8R80WIfMP:6QXH2IVIV0UYJFP;?cll2EUIF18N7Q`9gMgL2
MGEe0WmoO`8C4a<2IVIV0V9RHP:8R8P2P8200YnOW`;AdM42XJ6Q0Y6AT@:RXZ84P8200WinOP9mOGd2
NWYj0X:2PP9VIVH2_Kfm0YZJVP9LG5`2/[:b0XB4Q09lO7`2OGem0WalO09jNWX2PX:20WQhN09aLG42
KFe]0WEeM@8000000g]kNgalO7alO002NGUi0WinOP9kNg/2OGem0XJ6QP9iNGT2PH610WalO09XJ6P2
00000RDU9@:f][H2h^;R0]GEe@9_Kfl2alO70_ooo`:2PX82NGUi0W]kN`9bLW82QXJ60VMWI`:UYJD2
KFe]0W1`L0;:b/X2M7Ad0Vm_K`:TY:@2V9RH0WYjNP9hN7P2NGUi0WalO09hN7P2OWin0WYjNPIlO7`2
OGem1WalO09mOGd2M7Ad0U5AD@9RHV82OGem0[fm_@92@T82alO70WUiN@:9RHT2N7Qh0WemO@9lO7`4
Ng]k0WalO09jNWX5oooo0@00007oool000[oool00`000?ooooooo`02oooo00<0003oooooool00ooo
o`030000oooooooo00;oool2Zj^[0Vi^KP9bLW82O7al0W]kN`9mOGd2KFe]0X:2PP9gMgL2NWYj17al
O09gMgL2/k>c0ZBTY09_Kfl2OGem0W9bLP9gMgL2Ph>30WQhN09mOGd2NWYj0W]kN`:HV9P2SHf=0XV9
R@::RXX2?3`l0Ve]K@:4Q8@2Ng]k0Z:RXP:l_;`2YZJV0W=cL`9kNg/2KFe]0UEEE@:1PH42Jf][0Ve]
K@9hN7P2NWYj0WalO09kNg/2O7al0W]kN`9mOGd2MWIf0VQXJ09=CDd2Hf=S0WQhN09mOGd6O7al0WYj
NP9lO7`2OWin0WalO09HF5P2<S8b1GalO09kNg/2O7al0W]kN`9hN7P2OGem0WalO09mOGd2Rh^;0VQX
J0:FUYH2_[jn0Q8B4P9OGel2[Jf]0YJFUP:>SXh2LW9b0WmoO`9lO7`2Ogmo0WUiN@:4Q8@2K6a/0YJF
UP;1`L42XJ6Q0VQXJ0;9bLT2b<S80WMgM`9oOgl2O7al0WemO@9lO7`2OGem0W]kNaAlO7`4P8200Va/
K0;Ti>@2m?Cd0UUIF@:1PH42MWIf0WinOP9kNg/8O7al0WemO@Goool100000Oooo`005Oooo`030000
oooooooo00;oool2Zj^[0Vi^KP9bLW82O7al0W]kN`9mOGd2KFe]0X:2PP9gMgL2NWYj17alO09gMgL2
/k>c0ZBTY09_Kfl2OGem0W9bLP9gMgL2Ph>30WQhN09mOGd2NWYj0W]kN`:HV9P2SHf=0XV9R@::RXX2
?3`l0Ve]K@:4Q8@2Ng]k0Z:RXP:l_;`2YZJV0W=cL`9kNg/2KFe]0UEEE@:1PH42Jf][0Ve]K@9hN7P2
NWYj0WalO09kNg/2O7al0W]kN`9mOGd2MWIf0VQXJ09=CDd2Hf=S0WQhN09mOGd6O7al0WYjNP9lO7`2
OWin0WalO09HF5P2<S8b1GalO09kNg/2O7al0W]kN`9hN7P2OGem0WalO09mOGd2Rh^;0VQXJ0:FUYH2
_[jn0Q8B4P9OGel2[Jf]0YJFUP:>SXh2LW9b0WmoO`9lO7`2Ogmo0WUiN@:4Q8@2K6a/0YJFUP;1`L42
XJ6Q0VQXJ0;9bLT2b<S80WMgM`9oOgl2O7al0WemO@9lO7`2OGem0W]kNaAlO7`4P8200Va/K0;Ti>@2
m?Cd0UUIF@:1PH42MWIf0WinOP9kNg/8O7al0WemO@Goool100000Oooo`005Oooo`030000oooooooo
00;oool2Lg=c0Xf=S@9hN7P2OWin0WalO09kNg/2QXJ60X20P0ImOGd2NGUi0X>3P`9HF5P2OGem0Wal
O09kNg/2P8201WemO@9lO7`2Ng]k0WalO097AdL2EUIF0WmoO`9eMGD2UYJF0YZJVP9hN7P2OGem0[fm
_@9jNWX2K6a/0X20P09lO7`2PH610X^;R`:RXZ82^[Zj0WalO0:0P802Ng]k27alO09kNg/2Ogmo0X61
P@9hN7P2Ph>347alO09iNGT2S8b<0V5QH@]lO7`2OGem0WalO09kNg/2NWYj0WUiN@9kNg/4OGem0Xb<
S0800002@D510WEeM@9kNg/2OWin0W]kN`9lO7`2Ng]k0WemO@9iNGT2PH610WAdM09/K6`2Jf][0ZNW
Y`99BDT2ATI60XJ6QP9hN7P2OGem0W]kN`9lO7`2Ng]k57alO09kNg/2NGUi0WYjNP9mOGd2K6a/0VAT
I0:4Q8@2Ng]k0WemO@9kNg/2O7al0W]kN`IlO7`2Ng]k1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9cLg<2SHf=0WQhN09nOWh2O7al0W]kN`:6QXH2P8201WemO@9iNGT2Ph>30UQHF09mOGd2
O7al0W]kN`:0P806OGem0WalO09kNg/2O7al0TM7A`9FEUH2Ogmo0WEeM@:FUYH2VYZJ0WQhN09mOGd2
_Kfm0WYjNP9/K6`2P8200WalO0:1PH42Rh^;0Z:RXP:j^[X2O7al0X20P09kNg/8O7al0W]kN`9oOgl2
PH610WQhN0:3Ph<@O7al0WUiN@:<S8`2HF5Q2galO09mOGd2O7al0W]kN`9jNWX2NGUi0W]kN`AmOGd2
S8b<0P000091@D42MGEe0W]kN`9nOWh2Ng]k0WalO09kNg/2OGem0WUiN@:1PH42M7Ad0Va/K09[Jf/2
YjNW0TU9B@96ATH2QXJ60WQhN09mOGd2Ng]k0WalO09kNg/DO7al0W]kN`9iNGT2NWYj0WemO@9/K6`2
I6AT0XB4Q09kNg/2OGem0W]kN`9lO7`2Ng]k1WalO09kNg/5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0WQhN09VIVH2PX:20WYjNPAlO7`2N7Qh0WalO0AkNg/4O7al0WQhN0:5QHD2Xj>S0WEe
M@9mOGd2NGUi0WYjNP9lO7`2Ng]k17alO09kNg/2KFe]0Slo?`:8R8P2Ng]k0X^;R`9WIfL2Ogmo0WYj
NP:b/[82SXj>0WEeM@9mOGd2Ng]k0WemO@9bLW82TI6A0UeMG@9kNg/2NWYj37alO0AjNWX2Ng]k0WUi
NA1lO7`2Ng]k0WemO@9UIFDCO7al0WemO@9lO7`2Ogmo0WemO@:FUYH2Ti>C0V1PH09jNWX2Ng]k2Wal
O09mOGd2NWYj0WmoO`:1PH42NWYj0Zj^[P:0P802SXj>0WMgM`9nOWh2Ng]k77alO0AmOGd2Ng]k0X:2
PP:6QXH2NGUi0WalO09kNg/>O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9hN7P2
IVIV0X:2PP9jNWX4O7al0WQhN09lO7`4Ng]k17alO09hN7P2QHF50Z>SX`9eMGD2OGem0WUiN@9jNWX2
O7al0W]kN`AlO7`2Ng]k0Ve]K@8o?cl2R8R80W]kN`:;Rh/2IfMW0WmoO`9jNWX2/[:b0Xj>SP9eMGD2
OGem0W]kN`9mOGd2LW9b0Y6AT@9MGEd2Ng]k0WYjNPalO7`4NWYj0W]kN`9iNGT@O7al0W]kN`9mOGd2
IFEU4galO09mOGd2O7al0WmoO`9mOGd2UYJF0Y>CT`9PH602NWYj0W]kN`YlO7`2OGem0WYjNP9oOgl2
PH610WYjNP:^[Zh2P8200Xj>SP9gMgL2OWin0W]kNaalO7`4OGem0W]kN`:2PX82QXJ60WUiN@9lO7`2
Ng]k3WalO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2Shn?0XZ:RP9hN7P2OGem17al
O09mOGd2Ng]k0WalO09kNg/2OWin0WMgM`:8R8P2?Cdm0WEeM@9iNGT2Ng]k17emO@IlO7`2NWYj0X20
P09<C4`2Ng]k0X20P09jNWX2DE5A0WUiN@9jNWX2OGem0][JfP:9RHT2P8200WQhN09nOWh2N7Qh0ZJV
YP:b/[82OGem0X20P09kNg/>O7al1WemOA1lO7`2NWYj0X>3P`9ZJVXEO7al0W]kN`9jNWX2P8200YFE
U@:UYJD2[Jf]0W=cL`9nOWh2Ng]k0WQhN09nOWh4O7al0W]kN`9nOWh2LW9b0YNGU`9@D502Cdm?0Wmo
O`9dM7@2OWin0W]kNailO7`4Ng]k0WalO09iNGT2N7Qh0WemOA9lO7`5oooo0@00007oool001Goool2
00000oooo`:?Shl2RXZ:0WQhN09mOGd4O7al0WemO@9kNg/2O7al0W]kN`9nOWh2MgMg0XR8R08m?Cd2
MGEe0WUiN@9kNg/4OGem1WalO09jNWX2P8200Ta<C09kNg/2P8200WYjNP9ADE42NGUi0WYjNP9mOGd2
f][J0XV9R@:0P802N7Qh0WinOP9hN7P2YZJV0[:b/P9mOGd2P8200W]kN`ilO7`6OGem47alO09jNWX2
Ph>30VYZJQElO7`2Ng]k0WYjNP:0P802UIFE0ZFUY@:][Jd2Lg=c0WinOP9kNg/2N7Qh0WinOPAlO7`2
Ng]k0WinOP9bLW82UiNG0U1@D09?Cdl2Ogmo0WAdM09nOWh2Ng]k7WalO0AkNg/2O7al0WUiN@9hN7P2
OGem4WalO0Coool200000Oooo`005Oooo`030000oooooooo00;oool2H61P0WEeM@9mOGd6O7al0W]k
N`QlO7`2NWYj0X61P@9YJFT2G5aL0X20P0alO7`2Ng]k0WinOP:1PH42/k>c0Xn?S`9`L702V9RH0XF5
Q@AlO7`2GEeM0TY:BP9oOgl2Ng]k0X20P0:6QXH2YJFU0Ti>CP9nOWh2NGUi0WemOA1lO7`2Ng]k57al
O09iNGT2RXZ:6GalO09mOGd2N7Qh0YfMW@9?Cdl2QhN70WIfMP9kNg/2QhN70WIfMP9jNWX2O7al0WYj
NP:2PX82MGEe0^OWi`9^KVh2Rh^;0WUiN@9oOgl2Ng]k9WalO0AmOGdDO7al1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`9PH602MGEe0WemO@IlO7`2Ng]k27alO09jNWX2PH610VUYJ@9LG5`2
P82037alO09kNg/2OWin0X61P@:c/k<2Shn?0W1`L0:HV9P2QHF517alO09MGEd2BTY:0WmoO`9kNg/2
P8200XJ6QP:UYJD2CTi>0WinOP9iNGT2OGem47alO09kNg/DO7al0WUiN@::RXXIO7al0WemO@9hN7P2
WIfM0Tm?C`:7QhL2MWIf0W]kN`:7QhL2MWIf0WYjNP9lO7`2NWYj0X:2PP9eMGD2inOW0Vi^KP:;Rh/2
NGUi0WmoO`9kNg/VO7al17emOAAlO7`5oooo0@00007oool001Goool00`000?ooooooo`02oooo0X>3
P`9jNWX@O7al0WYjNP:0P802Kfm_0Zn_[`9cLg<2IVIV37alO09mOGd2MWIf0XB4Q08P8202S8b<0X20
P09]KFd2MGEe17alO0:ATI42_;bl0ZVYZ@8o?cl2NGUi0W9bLP9kNg/2WYjN0WYjNP9nOWh2Ng]k97al
O09kNg/2OWin0Vi^KQElO7`2Ng]k0WemO@9gMgL2OGem0UmOG`:ZZZX2NWYj0VMWI`9lO7`2:RXZ0YFE
U@9mOGd2Ogmo0WemO@9cLg<2LW9b0Xf=S@;>c/h2MGEe0WemO@9jNWY0O7al1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`:3Ph<2NWYj47alO09jNWX2P8200Vm_K`:_[jl2Lg=c0VIVIPalO7`2
OGem0WIfMP:4Q8@2820P0Xb<S0:0P802KFe]0WEeM@AlO7`2TI6A0[bl_0:YZJT2?clo0WUiN@9bLW82
Ng]k0YjNWP9jNWX2OWin0W]kNbAlO7`2Ng]k0WinOP9^KVhEO7al0W]kN`9mOGd2MgMg0WemO@9OGel2
ZZZZ0WYjNP9WIfL2O7al0RXZ:P:EUID2OGem0WmoO`9mOGd2Lg=c0W9bLP:=SHd2c/k>0WEeM@9mOGd2
NWYj@7alO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2QhN70VUYJ@:1PH42NWYj3Wal
O09kNg/2OWin0WYjNP:^[Zh2PH613WalO09mOGd2QhN70ZRXZ09FEUH2MWIf0ZZZZP9dM7@2Ogmo0W]k
N`97AdL2K6a/0T51@@;oool2IFEU0Z>SX`:7QhL2W9bL0XV9R@9kNg/VO7al0WemO@9jNWX2QhN75gal
O09kNg/2Ph>30YFEU@:SXj<2GemO0W5aL@:IVIT2P8200YfMW@800002@T920WinOP9cLg<2YjNW0Y^K
V`9gMgL2HF5Q0UeMG@:9RHT2N7Qh0WemO@9kNg/lO7al1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`:7QhL2JFUY0X61P@9jNWX>O7al0W]kN`9nOWh2NWYj0Zj^[P:1PH4>O7al0WemO@:7QhL2
Z:RX0UIFEP9fMWH2ZZZZ0WAdM09oOgl2Ng]k0TM7A`9/K6`2@D510_ooo`9UIFD2Xj>S0XN7Q`:LW9`2
RHV90W]kNbIlO7`2OGem0WYjNP:7QhLGO7al0W]kN`:3Ph<2UIFE0Z>SX`9OGel2LG5a0YVIV@:0P802
WIfM0P000092@T82OWin0W=cL`:WYjL2Vi^K0WMgM`9QHF42GEeM0XV9R@9hN7P2OGem0W]kNcalO7`5
oooo0@00007oool001Goool00`000?ooooooo`02oooo0[Jf]P95ADD2PX:20WUiN@9mOGd8O7al0Wem
O@9iNGT2Ph>30Vi^KP;kno/200000[no_`9mOGd8O7al0W]kN`9oOgl2Kfm_0[Vi^@:2PX82gmoO0^3P
h09jNWX2DU9B0X>3P`9iNGT2alO70Ve]K@:RXZ82O7al0Z>SX`9bLW82^k^k0Vm_K`9DE5@2Ph>30WUi
N@9mOGdPO7al0W]kN`9nOWh2Jf][0XN7Qa=lO7`2OGem0WYjNP9kNg/2:RXZ0VQXJ09TI6@2Zj^[0_[j
nP8n?Sh2bLW90VQXJ09oOgl2JFUY0WMgM`9lO7`2I6AT0YfMW@9cLg<2PX:20Xf=S@9QHF42Q8B40WUi
N@9mOGdlO7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:f][H2ADE50X:2PP9iNGT2
OGem27alO09mOGd2NGUi0X>3P`9^KVh2no_k0P0000:o_kl2OGem27alO09kNg/2Ogmo0Vm_K`:i^KT2
PX:20]oOg`;Ph>02NWYj0U9BDP:3Ph<2NGUi0/O7a`9]KFd2XZ:R0WalO0:SXj<2LW9b0[^k^`9_Kfl2
E5AD0X>3P`9iNGT2OGem87alO09kNg/2OWin0V][J`:7QhLCO7al0WemO@9jNWX2Ng]k0RXZ:P9XJ6P2
I6AT0Z^[Z`;jn_X2?Shn0/W9b@9XJ6P2Ogmo0VUYJ@9gMgL2O7al0VATI0:MWId2Lg=c0X:2PP:=SHd2
HF5Q0XB4Q09iNGT2OGem?7alO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2PH610WUi
N@9oOgl2Ng]k2WalO09kNg/2OGem0WUiN@:0P802CDe=0[Ng]`80000<O7al0W]kN`:0P802GUiN0TI6
AP800002IVIV0ZJVYP9dM7@2OGem0W]kN`9oOgl2@d=3100000:WYjL2MgMg0Y6AT@9hN7P2MWIf0W]k
NbIlO7`2Ng]k0XF5Q@9gMgLCO7al0W]kN`AmOGd2];Bd0WemO@:4Q8@2EeMG0^GUi@:EUID2]KFe0YjN
WP8c<c<2M7Ad0U5AD@:5QHD2=CDe0WinOP9jNWX2Ng]k0WmoO`9/K6`2OWin0W]kNcilO7`5oooo0@00
007oool001Goool200000oooo`:1PH42NGUi0WmoO`9kNg/:O7al0W]kN`9mOGd2NGUi0X20P09=CDd2
]kNg0P0000alO7`2Ng]k0X20P09NGUh2ATI60P00009VIVH2YZJV0WAdM09mOGd2Ng]k0WmoO`93@d<4
00000ZNWY`9gMgL2TI6A0WQhN09fMWH2Ng]k9WalO09kNg/2QHF50WMgMa=lO7`2Ng]k17emO@:d];@2
OGem0XB4Q09GEeL2iNGU0YFEU@:e]KD2WYjN0S<c<`9dM7@2DE5A0XF5Q@8e=CD2OWin0WYjNP9kNg/2
Ogmo0Va/K09nOWh2Ng]k?WalO0Coool200000Oooo`005Oooo`030000oooooooo00;oool2A4A40Ve]
K@9gMgL2OGem37alO09kNg/2OGem0WalO0:MWId2De=C0_ooo`ilO7`2MgMg0X61P@9oOgl2^k^k0Z>S
X`8S8b<2PX:20W]kN`9lO7`2IfMW0U5AD@8m?Cd200000/;2`P9MGEd2AdM70W5aL@:3Ph<2PX:20W]k
NbAlO7`2NWYj0WQhN09EEEDAO7al0W]kN`9mOGd2MWIf0YRHV0:EUID2MWIf0WEeM@:k^k/271`L0]_K
f`;4a<@2F5QH0XB4Q09JFUX2F5QH0[6a/@9LG5`2ATI60XN7Q`AjNWX2[Jf]0WQhN09nOWh2Ng]k?7al
O0Goool100000Oooo`005Oooo`030000oooooooo00;oool2A4A40Ve]K@9gMgL2OGem37alO09kNg/2
OGem0WalO0:MWId2De=C0_ooo`ilO7`2MgMg0X61P@9oOgl2^k^k0Z>SX`8S8b<2PX:20W]kN`9lO7`2
IfMW0U5AD@8m?Cd200000/;2`P9MGEd2AdM70W5aL@:3Ph<2PX:20W]kNbAlO7`2NWYj0WQhN09EEEDA
O7al0W]kN`9mOGd2MWIf0YRHV0:EUID2MWIf0WEeM@:k^k/271`L0]_Kf`;4a<@2F5QH0XB4Q09JFUX2
F5QH0[6a/@9LG5`2ATI60XN7Q`AjNWX2[Jf]0WQhN09nOWh2Ng]k?7alO0Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2Z:RX0WEeM@9mOGd<O7al0W]kN`9mOGd2NWYj0WemO@8i>CT2Ph>30^KV
iPalO7`2NWYj0X:2PP9^KVh2J6QX0UIFEP8e=CD2LW9b0WUiN@9mOGd2Ng]k0V9RHP;Ng]h2ADE50T];
B`:TY:@2oooo0X:2PP:1PH42O7al0WemOBQlO7`2Rh^;0Z>SXa=lO7`2NWYj0X:2PP9TI6@2gMgM0W1`
L0:;Rh/26ATI0RHV9P800002GUiN0YJFUP9JFUX26QXJ0YbLW09DE5@21PH60YFEU@9gMgL2P8200WYj
NP:2PX82Ng]k@7alO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2Z:RX0WEeM@9mOGd<
O7al0W]kN`9mOGd2NWYj0WemO@8i>CT2Ph>30^KViPalO7`2NWYj0X:2PP9^KVh2J6QX0UIFEP8e=CD2
LW9b0WUiN@9mOGd2Ng]k0V9RHP;Ng]h2ADE50T];B`:TY:@2oooo0X:2PP:1PH42O7al0WemOBQlO7`2
Rh^;0Z>SXa=lO7`2NWYj0X:2PP9TI6@2gMgM0W1`L0:;Rh/26ATI0RHV9P800002GUiN0YJFUP9JFUX2
6QXJ0YbLW09DE5@21PH60YFEU@9gMgL2P8200WYjNP:2PX82Ng]k@7alO0Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2OWin0W1`L09nOWh2Ng]k37alO09mOGd2N7Qh0XN7Q`:BTY82oooo0YRH
V0alO7`4Ng]k0UUIF@:@T902IfMW0Rl_;`:<S8`2NWYj0WemO@AkNg/27AdM0ZZZZP9lO7`2SHf=0W5a
L@9eMGD2MWIf0WemO@9kNg/TO7al0WemO@9jNWX2RXZ:0V][Ja=lO7`2Ng]k0WemO@9KFe/2MgMg0X20
P09[Jf/24Q8B0Xn?S`;8b<P2oooo0UIFEP9cLg<2NWYj0YFEU@9IFET2LG5a0WMgM`AfMWH2OWin0W]k
Nd9lO7`5oooo0@00007oool001Goool00`000?ooooooo`02oooo0WinOP9`L702OWin0W]kN`alO7`2
OGem0WQhN0:7QhL2TY:B0_ooo`:HV9P<O7al17]kN`9IFET2T92@0VMWI`8_;bl2S8b<0WYjNP9mOGd4
Ng]k0QdM7@:ZZZX2O7al0Xf=S@9aLG42MGEe0WIfMP9mOGd2Ng]k97alO09mOGd2NWYj0XZ:RP9[Jf/C
O7al0W]kN`9mOGd2Fe]K0WMgM`:0P802Jf][0Q8B4P:?Shl2b<S80_ooo`9FEUH2Lg=c0WYjNP:EUID2
FEUI0W5aL@9gMgL4MWIf0WinOP9kNg]2O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`:LW9`2RXZ:0WUiN@alO7`2Ng]k0WinOP9dM7@2TY:B0Slo?`9>CTh2=SHf37alO09iNGT2OWin0Slo
?`:ZZZX2Y:BT0^KViP9kNg/2N7Qh17alO0;[jn/2V9RH0UmOG`:h^;P2Xj>S0]GEe@:ATI42MGEe0Wal
O09mOGdTO7al0WemO@9hN7P2Ti>C0WQhN1=lO7`2NWYj0X20P09;Bd/2Ti>C0TA4A09lO7`2SHf=0][J
fP9YJFT2410@0_ooo`;QhN42Q8B40_ooo`:^[Zh2LW9b0XF5Q@;Rh^82SHf=0WYjNP:0P812O7al1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`:LW9`2RXZ:0WUiN@alO7`2Ng]k0WinOP9dM7@2
TY:B0Slo?`9>CTh2=SHf37alO09iNGT2OWin0Slo?`:ZZZX2Y:BT0^KViP9kNg/2N7Qh17alO0;[jn/2
V9RH0UmOG`:h^;P2Xj>S0]GEe@:ATI42MGEe0WalO09mOGdTO7al0WemO@9hN7P2Ti>C0WQhN1=lO7`2
NWYj0X20P09;Bd/2Ti>C0TA4A09lO7`2SHf=0][JfP9YJFT2410@0_ooo`;QhN42Q8B40_ooo`:^[Zh2
LW9b0XF5Q@;Rh^82SHf=0WYjNP:0P812O7al1Oooo`400001oooo000Eoooo00<0003oooooool01?oo
o`9RHV82OWin0WYjNPalO7`2NWYj0X20P09^KVh2/K6a0_ooo`;=cLd:O7al0W]kN`9nOWh2OGem0ZNW
Y`:BTY82^KVi0[>c/`:EUID2HF5Q0XB4Q09jNWX2=CDe0VATI0;Ti>@28b<S0YVIV@9;Bd/2KFe]0VAT
I0:8R8P2PH610W]kNb9lO7`2Ng]k0WmoO`9ZJVX2UYJF4GalO09kNg/2OGem0WYjNP:LW9`2LW9b0VAT
I09oOgl2R8R80T];B`9ZJVX2ZZZZ0T=3@`8C4a<2Ti>C0R0P808o?cl2Ti>C0WEeM@:>SXh2OGem0WUi
N@:_[jl2N7Qh0WmoO`9jNWXlO7al1Oooo`400001oooo000Eoooo0P0000Goool2HV9R0WinOP9jNWX<
O7al0WYjNP:0P802KVi^0[6a/@;oool2cLg=2WalO09kNg/2OWin0WemO@:WYjL2TY:B0[Vi^@:c/k<2
UIFE0V5QH@:4Q8@2NWYj0SDe=@9TI6@2i>CT0R<S8`:IVIT2Bd];0Ve]K@9TI6@2R8R80X61P@9kNg/R
O7al0W]kN`9oOgl2JVYZ0YJFUQ5lO7`2Ng]k0WemO@9jNWX2W9bL0W9bLP9TI6@2Ogmo0XR8R09;Bd/2
JVYZ0ZZZZP93@d<24a<C0Y>CT`8P8202?clo0Y>CT`9eMGD2SXj>0WemO@9iNGT2[jn_0WQhN09oOgl2
NWYj?7alO0Coool200000Oooo`005Oooo`030000oooooooo00;oool2De=C0Z:RXP9dM7@2OWin0W]k
N`QlO7`2OGem0WQhN0:9RHT2CTi>0[no_`:=SHd2:RXZ0W]kN`QlO7`2Ng]k0X20P09lO7`2g=cL0P41
0@:_[jl2S8b<0S8b<P:4Q8@2MWIf0WinOP8h>3P2QXJ60TI6AP9RHV82Bd];0Z^[Z`:FUYH2/K6a0W1`
L09nOWh2Ng]k8WalO09iNGT2PX:20Tm?C`:GUiLAO7al0WYjNP:0P802MGEe0^WYj@9_Kfl2QHF50[Bd
]09FEUH2@T920Xj>SP9LG5`2VYZJ0XV9R@:^[Zh2>SXj0_[jnP9XJ6P2SXj>0UeMG@:OWil2Kfm_0W5a
L@9lO7`2Ng]k?WalO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2De=C0Z:RXP9dM7@2
OWin0W]kN`QlO7`2OGem0WQhN0:9RHT2CTi>0[no_`:=SHd2:RXZ0W]kN`QlO7`2Ng]k0X20P09lO7`2
g=cL0P410@:_[jl2S8b<0S8b<P:4Q8@2MWIf0WinOP8h>3P2QXJ60TI6AP9RHV82Bd];0Z^[Z`:FUYH2
/K6a0W1`L09nOWh2Ng]k8WalO09iNGT2PX:20Tm?C`:GUiLAO7al0WYjNP:0P802MGEe0^WYj@9_Kfl2
QHF50[Bd]09FEUH2@T920Xj>SP9LG5`2VYZJ0XV9R@:^[Zh2>SXj0_[jnP9XJ6P2SXj>0UeMG@:OWil2
Kfm_0W5aL@9lO7`2Ng]k?WalO0Goool100000Oooo`005Oooo`030000oooooooo00;oool2[Zj^0Ve]
K@9oOgl2Ng]k37alO09kNg/2OGem0WmoO`:/[:`2PX:20[^k^`alO7`2OWin0WUiN@:a/K42oooo0WIf
MP:8R8P2UiNG0W9bLP9oOgl2Ng]k0Tm?C`;5aLD2NGUi0V5QH@:4Q8@2MGEe0VUYJ@9^KVh2OGem9Wal
O09kNg/2Ogmo0W5aL@:RXZ8EO7al0WemO@9_Kfl2Hf=S0X^;R`9oOgl2dM7A0];BdP9VIVH2N7Qh0XZ:
RP:@T902?3`l0Y2@T08e=CD2Rh^;0Slo?`::RXX2L71`0Xj>SP:1PH52O7al1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`:^[Zh2KFe]0WmoO`9kNg/<O7al0W]kN`9mOGd2Ogmo0Zb/[0:2PX82
^k^k37alO09nOWh2NGUi0[6a/@;oool2MWIf0XR8R0:GUiL2LW9b0WmoO`9kNg/2Cdm?0/G5a@9iNGT2
HF5Q0XB4Q09eMGD2JFUY0Vi^KP9mOGdVO7al0W]kN`9oOgl2LG5a0Z:RXQElO7`2OGem0Vm_K`9SHf<2
Rh^;0WmoO`;AdM42d];B0VIVIP9hN7P2RXZ:0Y2@T08l?3`2T92@0SDe=@:;Rh/2?clo0XZ:RP9`L702
SXj>0X61PD9lO7`5oooo0@00007oool001Goool00`000?ooooooo`02oooo0XZ:RP:?Shl2NGUi0Wem
O@alO7`2Ng]k0WinOP9eMGD2GemO0P0000;Zj^X>O7al0WmoO`:MWId2QHF50[no_`;Kfm/2PH610Xf=
S@9jNWX2OGem0UYJFP8j>SX2f=SH0X20P0;bl_82Kfm_0VUYJ@9TI6@2SXj>0WQhN09mOGd2Ng]k87al
O09kNg/2Ogmo0WAdM0:`/;0CO7al0W]kN`:0P802KFe]0U9BDP:FUYH2m?Cd0/o?c`8^;Rh2F5QH0S/k
>`:RXZ82K6a/0Y6AT@9eMGD2A4A40WAdM098B4P2Ogmo0W]kN`8h>3P2NGUi0WYjNP9mOGd2Ng]k?7al
O0Goool100000Oooo`005Oooo`030000oooooooo00;oool2RXZ:0Xn?S`9iNGT2OGem37alO09kNg/2
OWin0WEeM@9OGel200000^[ZjPilO7`2Ogmo0YfMW@:5QHD2_kno0]_Kf`:1PH42SHf=0WYjNP9mOGd2
FUYJ0SXj>P;Hf=P2P8200_;blP9_Kfl2JFUY0VATI0:>SXh2N7Qh0WemO@9kNg/PO7al0W]kN`9oOgl2
M7Ad0[2`/1=lO7`2Ng]k0X20P09]KFd2DU9B0YJFUP;dm?@2clo?0Rh^;P9HF5P2>c/k0Z:RXP9/K6`2
TI6A0WEeM@94A4@2M7Ad0TQ8B09oOgl2Ng]k0SPh>09iNGT2NWYj0WemO@9kNg/lO7al1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`8U9BD2RXZ:0WalO09mOGd2Ng]k2WalO09jNWX2P8200W9b
LP:GUiL2MWIf0_ooo`YlO7`2Ng]k0WinOP9kNg/2^[Zj0Z:RXP8Z:RX2E5AD0Y2@T0:NWYh2MgMg0Wem
O@8Q8B42LG5a0UIFEP9cLg<2CDe=0Ti>CP:YZJT2`L710Z^[Z`9eMGD2OWin0W]kNb1lO7`2Ng]k0Wmo
O`9SHf<2SHf=4GalO09kNg/2Ogmo0WAdM0;7alL2cLg=0RDU9@8a<C42OWin0R0P80:@T902Q8B40Ve]
K@8F5QH2TI6A0UQHF0:/[:`2Shn?0VMWI`:PX:02/;2`0TU9B@:VYZH2Kfm_0X20P09kNg/lO7al1Ooo
o`400001oooo0003oooo1@0000;oool00`000?ooooooo`02oooo00<0003oooooool00oooo`030000
oooooooo00;oool29BDU0XZ:RP9lO7`2OGem0W]kN`YlO7`2NWYj0X20P09bLW82UiNG0WIfMP;oool:
O7al0W]kN`9nOWh2Ng]k0[Zj^P:RXZ82:RXZ0UADE0:@T902WYjN0WMgM`9mOGd28B4Q0W5aL@9FEUH2
Lg=c0Te=C@9>CTh2ZJVY0/71`@:[Zj/2MGEe0WinOP9kNg/PO7al0W]kN`9oOgl2Hf=S0Xf=SA5lO7`2
Ng]k0WmoO`9dM7@2alO70/g=c@8U9BD2<C4a0WinOP8P8202T92@0XB4Q09]KFd25QHF0Y6AT@9HF5P2
[:b/0Xn?S`9WIfL2X:2P0[2`/099BDT2YZJV0Vm_K`:0P802Ng]k?7alO0Goool100000Oooo`001Ooo
o`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool00_oo
o`:/[:`2Vi^K0WMgM`9mOGd>O7al0WinOP9]KFd2@d=30[Ng]`9oOgl2Ng]k2WalO09kNg/2Q8B40WYj
NP800002>CTi0XJ6QP9YJFT2PX:20WYjNP9mOGd200000VYZJP:f][H2^KVi0VEUI@8K6a/2A4A40XF5
Q@:4Q8@VO7al0W]kN`9mOGd2Shn?0[6a/A5lO7`2OGem0WQhN0:7QhL2BTY:0YbLW0:@T902alO70V][
J`9lO7`2Y:BT0Z^[Z`:c/k<2L71`0Y2@T09MGEd2?Shn0TA4A0;Bd]82oOgm0R8R8P9=CDd2Ph>30WUi
N@9mOGd2Ng]k?7alO0Goool100000Oooo`001Oooo`070000oooooooooooo0000oooo000000;oool0
0`000?ooo`000004oooo0P0000?oool2[:b/0Y^KV`9gMgL2OGem3WalO09nOWh2KFe]0T=3@`:g]kL2
Ogmo0W]kN`YlO7`2Ng]k0XB4Q09jNWX200000STi>@:6QXH2JFUY0X:2PP9jNWX2OGem0P00009ZJVX2
][Jf0[Vi^@9UIFD26a/K0TA4A0:5QHD2Q8B49WalO09kNg/2OGem0Xn?S`:a/K4AO7al0WemO@9hN7P2
QhN70TY:BP:LW9`2T92@0/O7a`9[Jf/2O7al0ZBTY0:[Zj/2/k>c0W1`L0:@T902GEeM0Shn?P94A4@2
d];B0_gmo@8R8R82CDe=0X>3P`9iNGT2OGem0W]kNcalO7`4oooo0P00007oool000Goool01`000?oo
ooooooooo`000?ooo`000002oooo00<0003oool000001?ooo`030000oooooooo00;oool2:RXZ0RDU
9@:<S8`2N7Qh0WemO@QlO7`2Ng]k0WinOP9gMgL2Q8B40SPh>0:HV9P2inOW0WemO@YlO7`2NGUi0X:2
PP9:BTX2N7Qh0/K6aP800002b/[:0Vi^KP:4Q8@2N7Qh0UeMG@;7alL2YZJV0VIVIP:FUYH2inOW0TM7
A`:CTi<2SHf=0WYjNRIlO7`2NGUi0X>3P`9;Bd/EO7al0WQhN0BATI42Ogmo0WEeM@9jNWX2]KFe0ZRX
Z09aLG42P8200WAdM09]KFd2EEEE0Z:RXP820P8200000XV9R@:1PH42De=C0XF5Q@9hN7P2OGem0W]k
NcalO7`5oooo0@00007oool000Goool01`000?ooooooooooo`000?ooo`000002oooo00<0003oool0
00001?ooo`030000oooooooo00;oool2:RXZ0RDU9@:<S8`2N7Qh0WemO@QlO7`2Ng]k0WinOP9gMgL2
Q8B40SPh>0:HV9P2inOW0WemO@YlO7`2NGUi0X:2PP9:BTX2N7Qh0/K6aP800002b/[:0Vi^KP:4Q8@2
N7Qh0UeMG@;7alL2YZJV0VIVIP:FUYH2inOW0TM7A`:CTi<2SHf=0WYjNRIlO7`2NGUi0X>3P`9;Bd/E
O7al0WQhN0BATI42Ogmo0WEeM@9jNWX2]KFe0ZRXZ09aLG42P8200WAdM09]KFd2EEEE0Z:RXP820P82
00000XV9R@:1PH42De=C0XF5Q@9hN7P2OGem0W]kNcalO7`5oooo0@00007oool000Coool200001?oo
o`030000oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool00_ooo`9KFe/2YJFU0W9b
LP9nOWh<O7al0WYjNP:1PH42K6a/1<?3``800002Ng]k27alO09kNg/2OWin0WEeM@:3Ph<22`/;0Vi^
KP;De=@2;2`/0WinOP9gMgL2OWin0Vm_K`::RXX2F5QH0VATI0:[Zj/2ZJVY0P00009mOGd2L71`0Wmo
O`9kNg/TO7al0W]kN`9mOGd2GUiN5GalO0AkNg/2IVIV0_Gem@:UYJD2:RXZ0VMWI`9lO7`2JFUY0VMW
I`::RXX2UIFE0YZJVP:>SXh2Lg=c0UiNGP8>3Ph2C4a<0Xf=S@9gMgL2OGem0W]kNcilO7`5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0U]KF`:UYJD2LW9b0WinOPalO7`2NWYj0X61P@9/K6`4
`l?30P00009kNg/8O7al0W]kN`9nOWh2MGEe0X>3P`8;2`/2KVi^0]CDe08/;2`2OWin0WMgM`9nOWh2
Kfm_0XZ:RP9HF5P2I6AT0Z^[Z`:YZJT200000WemO@9`L702Ogmo0W]kNbAlO7`2Ng]k0WemO@9NGUhE
O7al17]kN`9VIVH2mOGe0ZFUY@8Z:RX2IfMW0WalO09YJFT2IfMW0XZ:RP:EUID2VYZJ0Xj>SP9cLg<2
GUiN0Ph>3P9<C4`2SHf=0WMgM`9mOGd2Ng]k?WalO0Goool100000Oooo`005Oooo`030000oooooooo
00;oool2QhN70WAdM09nOWh2Ng]k2WalO09kNg/2OGem0WUiN@:3Ph<2GemO0X20P0;5aLD<O7al0W]k
N`:0P802M7Ad0/g=c@;oool2dM7A0Y^KV`9kNg/2OGem0W]kN`9WIfL2?Cdm0U]KF`9RHV82e]KF0P00
00:g]kL2OGem0X61P@9jNWXXO7al0WYjNP:9RHTIO7al0X20P09ZJVX2SHf=0X61P@;oool2[Zj^0SPh
>08e=CD2HF5Q0XV9R@:;Rh/2o_kn0Xn?S`:g]kL2SHf=0X^;R`9fMWH2OWin0W]kNd1lO7`5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0XN7Q`9dM7@2OWin0W]kN`YlO7`2Ng]k0WemO@9iNGT2
Ph>30UmOG`:0P802aLG537alO09kNg/2P8200WAdM0;=cLd2oooo0]7Ad@:KVi/2Ng]k0WemO@9kNg/2
IfMW0Sdm?@9KFe/2HV9R0]KFeP800002]kNg0WemO@:1PH42NWYj:7alO09jNWX2RHV96GalO0:0P802
JVYZ0Xf=S@:1PH42oooo0Zj^[P8h>3P2=CDe0V5QH@:9RHT2Rh^;0_knoP:?Shl2]kNg0Xf=S@:;Rh/2
MWIf0WinOP9kNg]0O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:7QhL2OWin47al
O09kNg/2OGem0WQhN0::RXX4FUYJ3WalO09iNGT2PX:20UeMG@9YJFT2PX:20WmoO`9jNWX4O7al0XF5
Q@:NWYh2Bd];0Vi^KP:OWil2RHV90W=cL`9iNGT^O7al0WMgMaUlO7`2NWYj0X:2PP9iNGT2O7al0ZJV
YP9iNGT2JFUY0V5QH@:<S8`2De=C0UMGE`9/K6`2KFe]0Va/K09eMGD2MgMg0WemO@9kNg]2O7al1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`:7QhL2OWin47alO09kNg/2OGem0WQhN0::RXX4
FUYJ3WalO09iNGT2PX:20UeMG@9YJFT2PX:20WmoO`9jNWX4O7al0XF5Q@:NWYh2Bd];0Vi^KP:OWil2
RHV90W=cL`9iNGT^O7al0WMgMaUlO7`2NWYj0X:2PP9iNGT2O7al0ZJVYP9iNGT2JFUY0V5QH@:<S8`2
De=C0UMGE`9/K6`2KFe]0Va/K09eMGD2MgMg0WemO@9kNg]2O7al1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`9jNWX2Ng]k2WalO0AkNg/4O7al0W]kN`9mOGd2MgMg0XF5Q@::RXX6O7al0Wem
O@IlO7`2OGem0WYjNPB6QXH2Ng]k0WemO@IlO7`2NGUi0W5aL@:9RHT2P82017AdM0:1PH4JO7al17]k
Na9lO7`2OGem6galO09jNWX2O7al0WemO@9fMWH2OWin0WmoO`:2PX82MWIf0XN7Q`:6QXH2PX:20Wmo
O`:1PH42OWin0WemO@9kNg/bO7al17]kN`ilO7`5oooo0@00007oool001Goool200000oooo`9jNWX2
Ng]k2WalO0AkNg/4O7al0W]kN`9mOGd2MgMg0XF5Q@::RXX6O7al0WemO@IlO7`2OGem0WYjNPB6QXH2
Ng]k0WemO@IlO7`2NGUi0W5aL@:9RHT2P82017AdM0:1PH4JO7al17]kNa9lO7`2OGem6galO09jNWX2
O7al0WemO@9fMWH2OWin0WmoO`:2PX82MWIf0XN7Q`:6QXH2PX:20WmoO`:1PH42OWin0WemO@9kNg/b
O7al17]kN`ilO7`4oooo0P00007oool001Goool00`000?ooooooo`02oooo3WalO0AmOGd6O7al0W]k
N`9mOGd2N7Qh0WIfMPIlO7`2NWYj0W]kN`IlO7`2OGem0WUiN@9gMgL4Ng]k1WalO09mOGd2Ogmo0WMg
M`9jNWX2Ogmo0X20P09jNWXJO7al0WinOP9mOGdBO7al0W]kNamlO7`2Ng]k0WemO@AkNg/2NWYj0Win
OPAhN7P2NWYj0W]kN`9jNWX4Ng]k<WalO09kNg/4OWin3WalO0Goool100000Oooo`005Oooo`030000
oooooooo00;oool>O7al17emO@IlO7`2Ng]k0WemO@9hN7P2MWIf1WalO09jNWX2Ng]k1WalO09mOGd2
NGUi0WMgM`AkNg/6O7al0WemO@9oOgl2MgMg0WYjNP9oOgl2P8200WYjNQYlO7`2OWin0WemOA9lO7`2
Ng]k7galO09kNg/2OGem17]kN`9jNWX2OWin17QhN09jNWX2Ng]k0WYjNPAkNg/bO7al0W]kN`AnOWh>
O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`ilO7`2NWYj0WQhN09kNg/6O7al0W]k
N`9mOGd2OWin1WalO0:0P802OGem27alO0AmOGd<O7al0W]kN`9mOGd2O7al17]kNaalO7`2NGUi0W]k
N`9jNWX2Ng]k>GalO09kNg/4OGem?7alO09mOGd2N7Qh0WMgM`9kNg/2O7al0WYjNP9kNg/6O7al1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`ilO7`2NWYj0WQhN09kNg/6O7al0W]kN`9mOGd2
OWin1WalO0:0P802OGem27alO0AmOGd<O7al0W]kN`9mOGd2O7al17]kNaalO7`2NGUi0W]kN`9jNWX2
Ng]k>GalO09kNg/4OGem?7alO09mOGd2N7Qh0WMgM`9kNg/2O7al0WYjNP9kNg/6O7al1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`QlO7`2Ng]k0WinOP9eMGD2VIVI0Z2PX09jNWX2OGem27al
O09kNg/2O7al17UiN@8n?Sh2MgMg0WUiN@9mOGd2Ng]k17alO09kNg/>O7al0W]kNaalO7`2Ng]k0Win
OP9gMgL2[Jf]0ZFUY@9lO7`2Ph>30WYjNPmlO7`2Ng]kI7alO09kNg/2Ogmo0W1`L0:b/[82^KVi0WAd
M09nOWh2Ph>30WmoO`9kNg/4O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`QlO7`2
Ng]k0WinOP9eMGD2VIVI0Z2PX09jNWX2OGem27alO09kNg/2O7al17UiN@8n?Sh2MgMg0WUiN@9mOGd2
Ng]k17alO09kNg/>O7al0W]kNaalO7`2Ng]k0WinOP9gMgL2[Jf]0ZFUY@9lO7`2Ph>30WYjNPmlO7`2
Ng]kI7alO09kNg/2Ogmo0W1`L0:b/[82^KVi0WAdM09nOWh2Ph>30WmoO`9kNg/4O7al1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`alO7`2OGem0WEeM@9BDU82L71`0WinOPilO7`2Ogmo0X^;
R`:>SXhjO7al0W]kN`9mOGd2NWYj0Z6QX@9KFe/2IFEU0WmoO`9kNg/2O7al0W]kN`UlO7`2OGem17al
O09kNg/2O7al0W]kNeilO7`2OGem0WYjNP9SHf<2HV9R0XJ6QP9HF5P2I6AT0X:2PP9iNGT2O7al1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`alO7`2OGem0WEeM@9BDU82L71`0WinOPilO7`2
Ogmo0X^;R`:>SXhjO7al0W]kN`9mOGd2NWYj0Z6QX@9KFe/2IFEU0WmoO`9kNg/2O7al0W]kN`UlO7`2
OGem17alO09kNg/2O7al0W]kNeilO7`2OGem0WYjNP9SHf<2HV9R0XJ6QP9HF5P2I6AT0X:2PP9iNGT2
O7al1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`QlO7`2Ng]k0WemO@9kNg/2P8200VEU
I@:DU9@2MGEe0WUiN@YlO7`2NWYj0XV9R@:2PX82H61P0Y:BTP9hN7P2Ng]k47alO09kNg/HO7al0W]k
N`IlO7`2OGem0WYjNP:3Ph<2O7al0]3@d0:BTY82C4a<0WEeM@9jNWX2OWin0WemO@AlO7`2Ng]k0Wal
O003NGUiMgMgMgMg00AlO7`4OWin0WmoOc1lO7`2Ng]k0WalO09kNg/2O7al0W]kNbAlO7`2LG5a0YVI
V@8l?3`2?Cdm0YNGU`9UIFD200000Xj>SP9fMWH2OWin1Oooo`400001oooo000Eoooo0P0000?oool8
O7al0W]kN`9mOGd2Ng]k0X20P09UIFD2U9BD0WEeM@9iNGT:O7al0WYjNP:9RHT2PX:20V1PH0:BTY82
N7Qh0W]kNa1lO7`2Ng]k67alO09kNg/6O7al0WemO@9jNWX2Ph>30WalO0;@d=02TY:B0Ta<C09eMGD2
NWYj0WinOP9mOGd4O7al0W]kN`9lO7`00gUiNGMgMgMgM`04O7al17inOP9oOgl`O7al0W]kN`9lO7`2
Ng]k0WalO09kNg/TO7al0W5aL@:IVIT2?3`l0Sdm?@:GUiL2IFEU0P0000:>SXh2MWIf0WinOPCoool2
00000Oooo`005Oooo`030000oooooooo00;oool2O7al0WYjNP9iNGT2NWYj0WalO09hN7P2Q8B40T=3
@`8i>CT2XJ6Q0W]kN`:>SXh2NGUi0WalO09iNGT2Ng]k0WYjNP9oOgl2IfMW0WUiN@820P82Kfm_0WMg
M`:5QHD2NWYj0W]kN`9hN7P2O7al17]kN`AmOGd2OWin0W]kN`9iNGT4NWYj17alO0AkNg/2OGem0W]k
N`AlO7`2OWin17alO09kNg/2NGUi0X20P09XJ6P2^[Zj0^GUi@8];Bd2/K6a0WAdM0:2PX82NGUi0WIf
MP9mOGd2NWYj0WmoO`9lO7`00h>3PhJ6QXJ6QP02Ng]k0WalO09aLG42M7Ad0W5aL@9mOGd2Ng]k0Wem
O@AjNWX4O7al17]kN`9lO7`2OGem0WYjNPAlO7`4NWYj27alO09kNg/2O7al17YjNP9nOWh2Ng]k0Wmo
O`9kNg/2Ogmo17]kN`AlO7`4NWYj2WalO0AjNWX4O7al0W]kN`9lO7`2Ogmo0Y>CT`9cLg<2Z:RX0Te=
C@9YJFT292@T0Y2@T09_Kfl2OGem0WmoO`Goool100000Oooo`005Oooo`030000oooooooo00;oool2
O7al0WYjNP9iNGT2NWYj0WalO09hN7P2Q8B40T=3@`8i>CT2XJ6Q0W]kN`:>SXh2NGUi0WalO09iNGT2
Ng]k0WYjNP9oOgl2IfMW0WUiN@820P82Kfm_0WMgM`:5QHD2NWYj0W]kN`9hN7P2O7al17]kN`AmOGd2
OWin0W]kN`9iNGT4NWYj17alO0AkNg/2OGem0W]kN`AlO7`2OWin17alO09kNg/2NGUi0X20P09XJ6P2
^[Zj0^GUi@8];Bd2/K6a0WAdM0:2PX82NGUi0WIfMP9mOGd2NWYj0WmoO`9lO7`00h>3PhJ6QXJ6QP02
Ng]k0WalO09aLG42M7Ad0W5aL@9mOGd2Ng]k0WemO@AjNWX4O7al17]kN`9lO7`2OGem0WYjNPAlO7`4
NWYj27alO09kNg/2O7al17YjNP9nOWh2Ng]k0WmoO`9kNg/2Ogmo17]kN`AlO7`4NWYj2WalO0AjNWX4
O7al0W]kN`9lO7`2Ogmo0Y>CT`9cLg<2Z:RX0Te=C@9YJFT292@T0Y2@T09_Kfl2OGem0WmoO`Goool1
00000Oooo`005Oooo`030000oooooooo00;oool2Ng]k0X61P@:6QXH2P8200X61P@9jNWX2SHf=0S4a
<@8T92@2]KFe0W9bLP8d=3@2QXJ60WQhN0:7QhL2OWin0WemO@9hN7P2IfMW0Z:RXP9MGEd2kno_0XN7
Q`9LG5`2PX:20WinOP:;Rh/2OGem1820P09hN7P2NGUi0WEeM@9nOWh2QHF518>3P`AkNg/4P8200WQh
N09nOWh2O7al0WYjNP9bLW82O7al0W]kN`:2PX82P8200WemO@9_Kfl2T92@0X>3P`8l?3`2_;bl0WYj
NP9SHf<2W9bL0XN7Q`9jNWX2QHF50Vm_K`9lO7`00eYJFP@410@41002QHF50WIfMP:e]KD2YJFU0Zn_
[`9hN7P2OGem0WQhN0:1PH42P8200WinOP9kNg/2PH610WalO09kNg/2MgMg0XB4Q09jNWX2OGem0XB4
Q0:1PH42NGUi0WemO@9kNg/2O7al0WinOP9lO7`2PX:20XB4Q09fMWH2O7al0Vi^KP9oOgl2Kfm_0Wmo
O`9lO7`4Ng]k0X:2PP:3Ph<2Ng]k1WalO09kNg/4Ph>30WYjNP9mOGd2O7al0WemO@9kNg/2ZZZZ0YVI
V@:KVi/2KFe]0Z^[Z`:RXZ82Hf=S0[Vi^@:`/;02EUIF1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9kNg/2PH610XJ6QP:0P802PH610WYjNP:=SHd2<C4a0R@T90:e]KD2LW9b0S@d=0:6QXH2
N7Qh0XN7Q`9nOWh2OGem0WQhN09WIfL2XZ:R0UeMG@;_knl2QhN70UaLG0:2PX82OWin0X^;R`9mOGd4
P8200WQhN09iNGT2MGEe0WinOP:5QHD4Ph>317]kN`B0P802N7Qh0WinOP9lO7`2NWYj0W9bLP9lO7`2
Ng]k0X:2PP:0P802OGem0Vm_K`:@T902Ph>30S`l?0:l_;`2NWYj0V=SH`:LW9`2QhN70WYjNP:5QHD2
Kfm_0WalO003FUYJ10@410@400:5QHD2MWIf0[Fe]@:UYJD2[jn_0WQhN09mOGd2N7Qh0X61P@:0P802
OWin0W]kN`:1PH42O7al0W]kN`9gMgL2Q8B40WYjNP9mOGd2Q8B40X61P@9iNGT2OGem0W]kN`9lO7`2
OWin0WalO0:2PX82Q8B40WIfMP9lO7`2KVi^0WmoO`9_Kfl2Ogmo0WalO0AkNg/2PX:20X>3P`9kNg/6
O7al0W]kN`B3Ph<2NWYj0WemO@9lO7`2OGem0W]kN`:ZZZX2VIVI0Y^KV`9]KFd2Zj^[0Z:RXP9SHf<2
^KVi0[2`/09FEUH5oooo0@00007oool001Goool00`000?ooooooo`02oooo0WYjNP9TI6@2D51@0Va/
K09XJ6P2MgMg0UADE0:j^[X292@T0/C4a09`L702IfMW0V=SH`:2PX82Bd];0Vi^KP9_Kfl2V9RH0Y>C
T`800002EUIF0YBDU09:BTX2K6a/0XZ:RP9SHf<2>CTi0W=cL`9[Jf/2HF5Q0X^;R`:<S8`2Shn?0W9b
LP9BDU82GemO0UiNGP9oOgl2OWin16IVIP:ATI42LG5a0WAdM0:2PX82Z:RX0WUiN@:0P802HV9R0VAT
I09kNg/2M7Ad0VMWI`800002[jn_0QlO7`:RXZ82aLG50Va/K08L71`2O7al0WmoO`:/[:`2Ng]k00<0
001>CTi>CTh00YRHV09VIVH2g=cL0^WYj@:ZZZX2PH610X20P0:4Q8@4IVIV0WIfMP9mOGd2Jf][0Win
OP9mOGd2U9BD0UQHF0:7QhL2MWIf0UQHF09RHV82QHF50WUiN@9mOGd2NWYj0WAdM09mOGd2GEeM0UQH
F0:=SHd2N7Qh0[Ng]`9^KVh2][Jf0WEeM@AmOGd2PX:20UmOG`9LG5`2P8200WUiN@9lO7`2Ng]k0X20
P09MGEd2G5aL0XJ6QP9hN7P2NGUi0WYjNP9fMWH2S8b<0YbLW0:>SXh2/[:b0YfMW@800002KFe]0SHf
=P:/[:`23`l?1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9jNWX2I6AT0U1@D09/K6`2
J6QX0WMgM`9DE5@2^[Zj0R@T90;4a<@2L71`0VMWI`9SHf<2PX:20T];B`9^KVh2Kfm_0YRHV0:CTi<2
00000UIFEP:DU9@2BTY:0Va/K0::RXX2Hf=S0STi>@9cLg<2Jf][0V5QH@:;Rh/2S8b<0Xn?S`9bLW82
DU9B0UmOG`9NGUh2Ogmo0WinOPAVIVH2TI6A0W5aL@9dM7@2PX:20ZRXZ09iNGT2P8200V9RHP9TI6@2
Ng]k0WAdM09WIfL200000Zn_[`8O7al2XZ:R0/G5a@9/K6`271`L0WalO09oOgl2[:b/0W]kN`030000
CTi>CTi>00:HV9P2IVIV0]cLg0;YjNT2ZZZZ0X61P@:0P802Q8B416IVIP9fMWH2OGem0V][J`9nOWh2
OGem0YBDU09HF5P2QhN70WIfMP9HF5P2HV9R0XF5Q@9iNGT2OGem0WYjNP9dM7@2OGem0UeMG@9HF5P2
SHf=0WQhN0:g]kL2KVi^0[Jf]P9eMGD4OGem0X:2PP9OGel2G5aL0X20P09iNGT2O7al0W]kN`:0P802
GEeM0UaLG0:6QXH2N7Qh0WUiN@9jNWX2MWIf0Xb<S0:LW9`2SXj>0[:b/P:MWId200000Ve]K@8f=SH2
[:b/0Pl?3`Goool100000Oooo`005Oooo`030000oooooooo00;oool2PH610X:2PP9ZJVX2UIFE0UiN
GP9hN7P2MgMg0YZJVP9EEED2GUiN0TU9B@;=cLd2B4Q80VMWI`93@d<2?clo0Ve]K@:OWil2T92@0X>3
P`:XZ:P2EEEE0UIFEP9SHf<2X:2P0V][J`9CDe<2Fe]K0X20P08H61P2JVYZ0ZNWY`8^;Rh2EEEE0SHf
=P:@T902D51@0XJ6QP9kNg/2B4Q80U9BDP:n_[h2LW9b0QPH609iNGT2Z:RX0WinOP:0P802EEEE0UMG
E`:<S8`2PX:20UeMG@;oool2Ogmo0P0000:?Shl2c/k>0V1PH0;oool2FEUI0XF5Q@:UYJD2MGEe00>h
^;QIFEUIFET00T];B`:l_;`2bLW90Ti>CP8M7Ad2MGEe0V=SH`:4Q8@2?Cdm0S@d=0;3`l<2J6QX0_Sh
n0:HV9P2KVi^0YbLW09EEED2NWYj0Zj^[P:1PH42Bd];0X:2PP9lO7`2OWin0XR8R0:;Rh/2T92@0UiN
GP9[Jf/27QhN0W=cL`:b/[82PX:20Zj^[P:RXZ82^k^k0V5QH@:8R8P2FEUI0UmOG`:8R8P4OGem0WMg
M`::RXX2HF5Q0UmOG`AkNg/2R8R80XN7Q`9hN7P2VIVI0Zn_[`800002KFe]0SLg=`800002IFEU0P00
00;QhN427QhN1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:1PH42PX:20VYZJP:EUID2
GUiN0WQhN09gMgL2VYZJ0UEEE@9NGUh2BDU90/g=c@98B4P2IfMW0T=3@`8o?cl2KFe]0YnOW`:@T902
Ph>30ZRXZ09EEED2EUIF0V=SH`:PX:02Jf][0U=CD`9KFe/2P8200QPH609ZJVX2YjNW0Rh^;P9EEED2
=SHf0Y2@T09@D502QXJ60W]kN`98B4P2DU9B0[jn_P9bLW8261PH0WUiN@:XZ:P2OWin0X20P09EEED2
EeMG0Xb<S0:2PX82GEeM0_ooo`9oOgl200000Xn?S`;>c/h2H61P0_ooo`9IFET2QHF50ZFUY@9eMGD0
0kRh^5UIFEUIF@02Bd];0[bl_0;9bLT2CTi>0QdM7@9eMGD2Hf=S0XB4Q08m?Cd2=3@d0/?3``9XJ6P2
n?Sh0YRHV09^KVh2W9bL0UEEE@9jNWX2[Zj^0X61P@9;Bd/2PX:20WalO09nOWh2R8R80X^;R`:@T902
GUiN0V][J`8N7Qh2Lg=c0[:b/P:2PX82[Zj^0Z:RXP:k^k/2HF5Q0XR8R09IFET2GemO0XR8R0AmOGd2
MgMg0XZ:RP9QHF42GemO17]kN`:8R8P2QhN70WQhN0:IVIT2[jn_0P00009]KFd2=cLg0P00009UIFD2
00000^7Qh@8N7Qh5oooo0@00007oool001Goool00`000?ooooooo`02oooo0[>c/`;:b/X2b<S80YJF
UP:d];@2KVi^0S`l?09[Jf/2[Jf]0Zb/[09EEED2QHF50WYjNP95ADD2Jf][0Va/K09mOGd2Ng]k0WEe
M@:PX:02FUYJ0RTY:@:HV9P2S8b<0_7al@:/[:`2WinO0UADE09SHf<2aLG50/S8b0:/[:`2/;2`0UUI
F@:>SXh292@T0R8R8P:;Rh/2/;2`0^GUi@8o?cl2BDU90TI6AP:h^;P2];Bd0XR8R09WIfL2L71`0[6a
/@:b/[82@T920V9RHP:9RHT2KVi^0V][J`9:BTX2=CDe0]3@d08n?Sh2Ph>30W1`L099BDT2oooo0W]k
N`03VIVIoooooooo00;QhN42/;2`100000:/[:`2`L710^7Qh@;`l?02WinO0VIVIP;7alL2P8200/k>
cP8e=CD200000X>3P`:8R8P2Xj>S0^;RhP;oool2f=SH0[^k^`9LG5`2W9bL0WEeM@8000021`L70WAd
M08G5aL2Q8B40/k>cP:SXj<22@T90Va/K0:MWId2UiNG0/?3``9^KVh2RHV90X20P08K6a/4MGEe0XR8
R09HF5P2K6a/0W1`L0:KVi/2P8200U5AD@9JFUX2MgMg0WIfMP9dM7@2T92@0YFEU@;_knl2D51@0Y:B
TP8:2PX2E5AD0YBDU0Goool100000Oooo`005Oooo`800003oooo0[>c/`;:b/X2b<S80YJFUP:d];@2
KVi^0S`l?09[Jf/2[Jf]0Zb/[09EEED2QHF50WYjNP95ADD2Jf][0Va/K09mOGd2Ng]k0WEeM@:PX:02
FUYJ0RTY:@:HV9P2S8b<0_7al@:/[:`2WinO0UADE09SHf<2aLG50/S8b0:/[:`2/;2`0UUIF@:>SXh2
92@T0R8R8P:;Rh/2/;2`0^GUi@8o?cl2BDU90TI6AP:h^;P2];Bd0XR8R09WIfL2L71`0[6a/@:b/[82
@T920V9RHP:9RHT2KVi^0V][J`9:BTX2=CDe0]3@d08n?Sh2Ph>30W1`L099BDT2oooo0W]kN`03VIVI
oooooooo00;QhN42/;2`100000:/[:`2`L710^7Qh@;`l?02WinO0VIVIP;7alL2P8200/k>cP8e=CD2
00000X>3P`:8R8P2Xj>S0^;RhP;oool2f=SH0[^k^`9LG5`2W9bL0WEeM@8000021`L70WAdM08G5aL2
Q8B40/k>cP:SXj<22@T90Va/K0:MWId2UiNG0/?3``9^KVh2RHV90X20P08K6a/4MGEe0XR8R09HF5P2
K6a/0W1`L0:KVi/2P8200U5AD@9JFUX2MgMg0WIfMP9dM7@2T92@0YFEU@;_knl2D51@0Y:BTP8:2PX2
E5AD0YBDU0Coool200000Oooo`005Oooo`030000oooooooo00;oool2X:2P0Ti>CP9cLg<2JFUY0^[Z
jP8U9BD2^[Zj0WUiN@B0P802QhN70[Vi^@8I6AT2?3`l0[6a/@9kNg/2^;Rh0/?3``:2PX82HF5Q0W=c
L`94A4@2a/K60_Ogm`8G5aL2<C4a0WYjNP:SXj<2Z:RX0Vm_K`;QhN42c/k>0S`l?0:;Rh/2Ng]k0Sdm
?@99BDT2Q8B40_ooo`;8b<P2A4A40_ooo`:][Jd2XZ:R0U1@D0;Bd]82WYjN0X^;R`:f][H2oooo0YNG
U`:BTY82G5aL0X20P09=CDd2NWYj0YRHV08[:b/2O7al0]oOg`:BTY82OGem0W1`L08Y:BT00f9RHWin
OWinOP02E5AD0Xj>SP8O7al2OGem0X61P@99BDT2LW9b0Z:RXP;oool2Ng]k0UUIF@9cLg<2JFUY0Zf]
[@:b/[8200000X61P@;Jf]X2@T920[fm_@9YJFT2LW9b0[Bd]08l?3`2IFEU0]KFeP:GUiL25aLG0]kN
gP8820P2oOgm0[2`/0;8b<P2YjNW0[Vi^@;<c<`21`L70XB4Q09RHV82a/K60Ti>CP:h^;P2JVYZ0W=c
L`9KFe/2JVYZ0T92@P8f=SH2k^k^0^CTi09kNg/2^[Zj0W5aL@9lO7`2M7Ad0V][J`:DU9@2Z:RX0Vm_
K`:@T902OGem0Ve]K@Goool100000Oooo`005Oooo`030000oooooooo00;oool2X:2P0Ti>CP9cLg<2
JFUY0^[ZjP8U9BD2^[Zj0WUiN@B0P802QhN70[Vi^@8I6AT2?3`l0[6a/@9kNg/2^;Rh0/?3``:2PX82
HF5Q0W=cL`94A4@2a/K60_Ogm`8G5aL2<C4a0WYjNP:SXj<2Z:RX0Vm_K`;QhN42c/k>0S`l?0:;Rh/2
Ng]k0Sdm?@99BDT2Q8B40_ooo`;8b<P2A4A40_ooo`:][Jd2XZ:R0U1@D0;Bd]82WYjN0X^;R`:f][H2
oooo0YNGU`:BTY82G5aL0X20P09=CDd2NWYj0YRHV08[:b/2O7al0]oOg`:BTY82OGem0W1`L08Y:BT0
0f9RHWinOWinOP02E5AD0Xj>SP8O7al2OGem0X61P@99BDT2LW9b0Z:RXP;oool2Ng]k0UUIF@9cLg<2
JFUY0Zf][@:b/[8200000X61P@;Jf]X2@T920[fm_@9YJFT2LW9b0[Bd]08l?3`2IFEU0]KFeP:GUiL2
5aLG0]kNgP8820P2oOgm0[2`/0;8b<P2YjNW0[Vi^@;<c<`21`L70XB4Q09RHV82a/K60Ti>CP:h^;P2
JVYZ0W=cL`9KFe/2JVYZ0T92@P8f=SH2k^k^0^CTi09kNg/2^[Zj0W5aL@9lO7`2M7Ad0V][J`:DU9@2
Z:RX0Vm_K`:@T902OGem0Ve]K@Goool100000Oooo`005Oooo`030000oooooooo00;oool2JFUY0ZVY
Z@;kno/2_kno0Y^KV`:_[jl2fm_K0WMgM`9_Kfl2JVYZ0XZ:RP8Y:BT2<C4a0V=SH`;Rh^82EEEE0^[Z
jP:@T902Shn?0/o?c`9XJ6P2Shn?0Rd];@98B4P400000Q@D509SHf<2/[:b0_7al@9gMgL200000[fm
_@9oOgl2][Jf0/;2`P9NGUh2c<c<0Rh^;P9OGel2@4100WUiN@8`<302J6QX0U5AD@9SHf<2d];B0/G5
a@800002@d=30U]KF`94A4@2dm?C0Te=C@:@T902OWin0Zf][@9>CTh2NGUi0P0000;=cLd2W9bL0V5Q
H@8/;2`00c@d=7moOgmoO`02Ng]k0WEeM@:][Jd2RXZ:0[Vi^@;oool271`L0VIVIP;lo?`2Rh^;0TY:
BP9^KVh4EEEE0T10@0;Ng]h200000PD51@;Hf=P2OWin0_Shn092@T82[:b/0Pl?3`9HF5P2FEUI0Te=
C@9/K6`2J6QX0Xn?S`@00002CTi>0XV9R@9ADE420@410/k>cP;QhN42b/[:0RPX:0::RXX2FUYJ0P00
008d=3@2MGEe0YRHV09UIFD2Xj>S0VATI0;oool2ATI60WUiN@:n_[h2e=CD0WalO0:4Q8@2Ng]k0WAd
M09nOWh2Ng]k0W1`L0;YjNT5oooo0@00007oool001Goool00`000?ooooooo`02oooo0VUYJ@:YZJT2
no_k0[no_`:KVi/2[jn_0]_Kf`9gMgL2Kfm_0VYZJP::RXX2:BTY0S4a<@9SHf<2h^;R0UEEE@;Zj^X2
T92@0Xn?S`;?cll2J6QX0Xn?S`8];Bd2B4Q81000008D51@2Hf=S0[:b/P;alO42MgMg0P0000:m_Kd2
Ogmo0[Jf]P;2`/82GUiN0/c<c08^;Rh2GemO0T10@09iNGT2<30`0VQXJ09ADE42Hf=S0];BdP;5aLD2
00000T=3@`9KFe/2A4A40]?Cd`9=CDd2T92@0WinOP:][Jd2CTi>0WUiN@800002cLg=0YbLW09QHF42
;2`/00<d=3AoOgmoOgl00W]kN`9eMGD2[Jf]0XZ:RP:i^KT2oooo0Q`L709VIVH2o?cl0X^;R`9:BTX2
KVi^15EEE@90@402g]kN0P0000851@D2f=SH0WinOP;hn?P2@T920Zb/[08?3`l2F5QH0UUIF@9=CDd2
K6a/0VQXJ0:?Shl400000Ti>CP:9RHT2DE5A0P410@;>c/h2hN7Q0/[:bP8X:2P2RXZ:0UYJFP800002
=3@d0WEeM@:HV9P2IFEU0Z>SX`9TI6@2oooo0TI6AP9iNGT2_[jn0]CDe09lO7`2Q8B40W]kN`9dM7@2
OWin0W]kN`9`L702jNWY1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9iNGT2M7Ad0_oo
o`;:b/X2a<C40SPh>0;Cdm<2Rh^;0T=3@`9XJ6P2HF5Q0SDe=@;De=@2SHf=0V1PH09gMgL2oooo0ZFU
Y@9ZJVX2C4a<0ZNWY`:BTY82_;bl0S/k>`9`L702^KVi0/G5a@8Z:RX2OGem0[:b/P8X:2P2EEEE0W]k
N`9>CTh2Kfm_0_ooo`8C4a<2GemO0ZJVYP;Rh^82fm_K0TM7A`9OGel2:BTY0W5aL@;EeMD2[Zj^0YRH
V0:h^;P200000Ta<C0:>SXh2OGem0X^;R`;2`/82OWin0WUiN@:3Ph<2GUiN0_Ogm`:HV9P2RXZ:0ZNW
Y`9>CTh00hj>SWYjNWYjNP02NGUi0XB4Q0:PX:02<S8b0WQhN09oOgl2^KVi0SXj>P9aLG42Ph>30]3@
d09;Bd/2Vi^K0WAdM0;Jf]X2V9RH0Xj>SP:7QhL2/K6a0[Fe]@:RXZ82[Jf]0Q4A4@:1PH42:2PX0_oo
o`8R8R82?Shn0T10@0800002DE5A0Zf][@;:b/X2QHF50[^k^`8_;bl2f][J0WUiN@9HF5P2De=C0[fm
_@8@4102O7al0^OWi`8G5aL2Jf][0UEEE@9?Cdl2Vi^K0]3@d09CDe<28b<S0XV9R@9cLg<2NWYj0XF5
Q@96ATH2C4a<0XJ6QP9dM7@2PH610W1`L0Goool100000Oooo`005Oooo`030000oooooooo00;oool2
NGUi0WAdM0;oool2b/[:0/C4a08h>3P2dm?C0X^;R`93@d<2J6QX0V5QH@8e=CD2e=CD0Xf=S@9PH602
MgMg0_ooo`:UYJD2JVYZ0Ta<C0:WYjL2TY:B0[bl_08k>c/2L71`0[Vi^@;5aLD2:RXZ0WemO@:b/[82
:2PX0UEEE@9kNg/2CTi>0Vm_K`;oool24a<C0UmOG`:VYZH2h^;R0]_Kf`97AdL2GemO0RTY:@9aLG42
eMGE0Zj^[P:HV9P2^;Rh0P00009<C4`2SXj>0WemO@:;Rh/2`/;20WinOP9iNGT2Ph>30UiNGP;gmoL2
V9RH0XZ:RP:WYjL2CTi>00>>SXijNWYjNWX00WUiN@:4Q8@2X:2P0S8b<P9hN7P2Ogmo0[Vi^@8j>SX2
LG5a0X>3P`;@d=02Bd];0Y^KV`9dM7@2f][J0YRHV0:>SXh2QhN70[6a/@:e]KD2XZ:R0Zf][@8A4A42
PH610RPX:0;oool28R8R0Shn?P90@40200000U5AD@:][Jd2b/[:0XF5Q@:k^k/2;bl_0][JfP9iNGT2
F5QH0U=CD`:m_Kd2410@0WalO0;WinL25aLG0V][J`9EEED2Cdm?0Y^KV`;@d=02De=C0R<S8`:9RHT2
Lg=c0WYjNP:5QHD2ATI60Ta<C0:6QXH2M7Ad0X61P@9`L705oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0[Vi^@9OGel2d=3@0P00009WIfL2JFUY0[:b/P;hn?P2Xj>S0/o?c`9TI6@2UIFE0SXj
>P:?Shl28b<S0Sdm?@98B4P2=3@d0TE5A@:f][H22PX:0Y:BTP:4Q8@2DE5A0UeMG@;Shn<2oooo0^k^
kP9`L702LG5a0Zf][@9kNg/2O7al0V=SH`8T92@2KFe]0Q4A4@9IFET200000Te=C@AdM7@400000W]k
N`:MWId2ATI60V9RHP9@D502[Jf]0WemO@:WYjL2Shn?0V5QH@:LW9`2MWIf0W5aL@9lO7`2MgMg0VIV
IP9LG5`2EeMG0Z:RXP:l_;`00gEeMGemOGemO@02Ogmo17EeM@9lO7`2Vi^K0ZFUY@9SHf<2QhN70]_K
f`:LW9`2SHf=0U=CD`:d];@2?3`l0T10@09fMWH2K6a/0S0`<0:PX:02N7Qh0V9RHP9LG5`2C4a<0Vi^
KP8U9BD2Zj^[0UaLG0:XZ:P2Kfm_0[Ng]`;IfMT2CDe=0WinOP9hN7P2O7al0Z^[Z`:TY:@2RXZ:0UmO
G`800002T92@0Z>SX`:1PH42A4A40P<30`:f][H2X:2P0WQhN0:1PH42^;Rh0P`<309nOWh2I6AT0Wem
O@AlO7`2OGem0V=SH`:4Q8@2UYJF0W=cL`:4Q8@5oooo0@00007oool000?oool5000000Coool00000
00000003oooo00<0003oooooool00oooo`030000oooooooo00;oool2^KVi0UmOG`;@d=0200000VMW
I`9YJFT2/[:b0_Shn0:SXj<2clo?0VATI0:EUID2>SXj0Xn?S`8S8b<2?Cdm0TQ8B08d=3@2ADE50[Jf
]P8:2PX2TY:B0XB4Q09ADE42GEeM0^?Sh`;oool2k^k^0W1`L09aLG42[Jf]0W]kN`9lO7`2Hf=S0R@T
909]KFd24A4A0UUIF@800002CDe=17AdM0@00002Ng]k0YfMW@96ATH2HV9R0U1@D0:][Jd2OGem0ZNW
Y`:?Shl2HF5Q0YbLW09fMWH2LG5a0WalO09gMgL2IVIV0UaLG09GEeL2XZ:R0[bl_003MGEeOGemOGem
009oOgl4MGEe0WalO0:KVi/2YJFU0V=SH`:7QhL2fm_K0YbLW0:=SHd2De=C0[Bd]08l?3`2@4100WIf
MP9/K6`2<30`0Z2PX09hN7P2HV9R0UaLG09<C4`2KVi^0RDU9@:[Zj/2G5aL0ZRXZ09_Kfl2]kNg0]WI
f@9=CDd2OWin0WQhN09lO7`2Zj^[0ZBTY0::RXX2GemO0P0000:@T902Xj>S0X61P@94A4@20`<30[Jf
]P:PX:02N7Qh0X61P@:h^;P230`<0WinOP9TI6@2OGem17alO09mOGd2Hf=S0XB4Q0:FUYH2Lg=c0XB4
Q0Goool100000Oooo`001Oooo`050000oooooooooooo000000Coool00`000?ooo`000004oooo00<0
003oooooool00_ooo`810@42HV9R0X^;R`8l?3`2OGem0UADE09aLG42alO70Z:RXP:^[Zh2NWYj0YZJ
VP9UIFD27alO0PX:2P8@4102R8R80WemO@:2PX82YJFU0Q`L70:9RHT2HV9R0UmOG`9>CTh2l_;b0ZBT
Y0;oool22PX:0Tm?C`:LW9`2EeMG0Va/K09@D502QXJ60P820P:2PX82OGem0VUYJ@::RXX28B4Q0YbL
W08I6AT2fm_K0/71`@:7QhL2Ogmo0W]kN`:3Ph<2I6AT0ZRXZ0:][Jd2WIfM0VIVIP:NWYh2GEeM0Vi^
KP9mOGd2TY:B0YRHV09EEED27QhN0VEUI@:n_[h3OGem0WAdM0:HV9P2XJ6Q0W9bLP:PX:02]KFe0Xj>
SP:a/K42I6AT0TE5A@:h^;P2;Rh^0[^k^`9DE5@2]KFe0YbLW0;oool2dm?C0Z:RXP9RHV82l?3`0Shn
?P:3Ph<28R8R0[Zj^P9[Jf/2O7al0[fm_@:d];@2HF5Q0Q<C4`8a<C42oooo0[>c/`:^[Zh23@d=0_oo
o`9TI6@2QHF50XR8R09aLG42K6a/0WIfMP:<S8`2Shn?0WEeM@9bLW82NWYj0WUiN@:QXJ42MGEe0TY:
BP9[Jf/2MGEe0WemO@9lO7`2MWIf0V5QH@:>SXh2T92@0WYjNP9lO7`5oooo0@00007oool000Goool0
0`000?ooooooo`02oooo00L0003oooooooooool0003oool000001?ooo`800003oooo0P410@9RHV82
Rh^;0S`l?09mOGd2E5AD0W5aL@;7alL2XZ:R0Zj^[P9jNWX2VYZJ0VEUI@8O7al22PX:0Q0@40:8R8P2
OGem0X:2PP:UYJD271`L0XV9R@9RHV82GemO0Ti>CP;bl_82Y:BT0_ooo`8:2PX2Cdm?0YbLW09GEeL2
K6a/0U1@D0:6QXH20P820X:2PP9mOGd2JFUY0XZ:RP8Q8B42W9bL0QTI6@;Kfm/2`L710XN7Q`9oOgl2
Ng]k0X>3P`9TI6@2Z:RX0Zf][@:MWId2IVIV0YjNWP9MGEd2KVi^0WemO@:BTY82V9RH0UEEE@8N7Qh2
IFEU0[jn_P=mOGd2M7Ad0YRHV0:QXJ42LW9b0Z2PX0:e]KD2SXj>0[6a/@9TI6@2ADE50[Rh^08^;Rh2
^k^k0UADE0:e]KD2W9bL0_ooo`;Cdm<2XZ:R0V9RHP;`l?02?Shn0X>3P`8R8R82^[Zj0V][J`9lO7`2
_Kfm0[Bd]09QHF424a<C0S4a<@;oool2/k>c0Zj^[P8=3@d2oooo0VATI0:5QHD2R8R80W5aL@9/K6`2
MWIf0Xb<S0:?Shl2MGEe0W9bLP9jNWX2NGUi0Z6QX@9eMGD2BTY:0V][J`9eMGD2OGem0WalO09fMWH2
HF5Q0Xj>SP:@T902NWYj0WalO0Coool200000Oooo`001Oooo`030000oooooooo00;oool01`000?oo
ooooooooo`000?ooo`000004oooo00<0003oooooool00_ooo`800002`/;20_ooo`9eMGD2OWin0R`/
;09_Kfl2jn_[0XB4Q0;<c<`2Rh^;0RLW9`;oool2K6a/0YfMW@;dm?@2RHV90WAdM08D51@2YjNW0TY:
BP:[Zj/2:b/[0Z6QX@9LG5`2d=3@0WinOP9WIfL2Y:BT0P0000:WYjL2QXJ60Tm?C`;3`l<2U9BD0P00
009:BTX2<30`0UQHF0:WYjL200000U]KF`8820P2dM7A0[Fe]@9fMWH2IfMW0UUIF@9eMGD2^;Rh0U1@
D0:@T904LW9b0Y2@T0;Fe]H2P8200WEeM@:UYJD2kNg]0^7Qh@9fMWH2oooo0^[ZjP03NWYjOgmoOgmo
009eMGD2[:b/0V9RHP92@T82A4A40WemO@;6a/H2P8200TI6AP9cLg<2IVIV0XN7Q`9JFUX2@4100Vi^
KP97AdL2QHF50[Bd]0:7QhL2MWIf0[Zj^P9@D502HF5Q0V][J`;1`L4200000XV9R@;1`L4292@T0Ve]
K@:ATI42;2`/0PX:2P:_[jl2;Rh^0STi>@:RXZ82IVIV0X61P@9bLW82Ph>30Vi^KP9CDe<23`l?0Wal
O08J6QX2Q8B40X20P08d=3@2GEeM0W9bLP;:b/X2K6a/0WYjNP9kNg/2O7al0WYjNP9`L702G5aL0Y6A
T@9VIVH2NGUi1Oooo`400001oooo0005oooo00L0003oooooooooool0003oool000000_ooo`030000
oooo000000Coool00`000?ooooooo`02oooo0P0000;2`/82oooo0WEeM@9nOWh2;2`/0Vm_K`;[jn/2
Q8B40/c<c0:;Rh/29bLW0_ooo`9/K6`2WIfM0_Cdm0:9RHT2M7Ad0Q@D50:WYjL2BTY:0Z^[Z`8[:b/2
XJ6Q0UaLG0;@d=02OWin0VMWI`:TY:@200000ZNWY`:6QXH2Cdm?0/?3``:DU9@200000TY:BP8`<302
F5QH0ZNWY`800002Fe]K0PP820;AdM42]KFe0WIfMP9WIfL2FEUI0WEeM@:h^;P2D51@0Y2@T0AbLW82
T92@0]KFeP:0P802MGEe0ZFUY@;]kNd2hN7Q0WIfMP;oool2j^[Z00=jNWYoOgmoOgl00WEeM@:/[:`2
HV9R0T92@P94A4@2OGem0/K6aP:0P802ATI60W=cL`9VIVH2QhN70UYJFP90@402KVi^0TM7A`:5QHD2
];Bd0XN7Q`9fMWH2^[Zj0U1@D09QHF42Jf][0/71`@800002RHV90/71`@8T92@2KFe]0Y6AT@8/;2`2
2PX:0Zn_[`8^;Rh2>CTi0Z:RXP9VIVH2PH610W9bLP:3Ph<2KVi^0U=CD`8?3`l2O7al0QXJ6P:4Q8@2
P8200S@d=09MGEd2LW9b0/[:bP9/K6`2NWYj0W]kN`9lO7`2NWYj0W1`L09LG5`2TI6A0VIVIP9iNGT5
oooo0@00007oool000Coool200001?ooo`030000oooooooo00;oool00`000?ooooooo`03oooo00<0
003oooooool01?ooo`:n_[h2C4a<0UUIF@:1PH42>CTi0Z6QX@800002YJFU0]OGe`:4Q8@200000QXJ
6P:ATI42_Kfm0YnOW`9WIfL2gMgM0/;2`P:6QXH2MWIf0Z>SX`8J6QX2:b/[0Xf=S@800002lO7a0UmO
G`:BTY82GUiN0VUYJ@9iNGT2V9RH0U=CD`9PH602h^;R0UIFEP;<c<`23`l?0/S8b09iNGT2VYZJ0XV9
R@9^KVh2ATI60Y2@T09/K6`2H61P0Xj>SP8L71`2aLG50Xn?S`;Ng]h2DE5A0ZJVYP;5aLD2OWin0W=c
L`:2PX82D51@0]7Ad@9jNWX2Hf=S0Vi^KP03Ng]kOGemOGem009gMgL2JVYZ0W=cL`;GemL2PX:20X>3
P`9=CDd2Ogmo0WAdM0:EUID2oooo0U5AD@:ZZZX2IfMW0YJFUP:?Shl2:b/[0UeMG@:TY:@2e]KF0R/[
:`9:BTX2oooo0UEEE@:VYZH2IFEU0Ta<C08g=cL2cLg=0_ooo`:1PH42O7al0YVIV@9iNGT2RHV90WQh
N0;Cdm<2WinO0WMgM`9jNWX2Q8B40VATI09_Kfl2TI6A0Y^KV`9XJ6P2][Jf0SLg=`9gMgL2/K6a0YZJ
VP9<C4`29RHV0Zb/[0:0P802Ogmo0Vm_K`9VIVH2IFEU0Y^KV`9TI6@2NGUi1Oooo`400001oooo000E
oooo00<0003oooooool01?ooo`:n_[h2C4a<0UUIF@:1PH42>CTi0Z6QX@800002YJFU0]OGe`:4Q8@2
00000QXJ6P:ATI42_Kfm0YnOW`9WIfL2gMgM0/;2`P:6QXH2MWIf0Z>SX`8J6QX2:b/[0Xf=S@800002
lO7a0UmOG`:BTY82GUiN0VUYJ@9iNGT2V9RH0U=CD`9PH602h^;R0UIFEP;<c<`23`l?0/S8b09iNGT2
VYZJ0XV9R@9^KVh2ATI60Y2@T09/K6`2H61P0Xj>SP8L71`2aLG50Xn?S`;Ng]h2DE5A0ZJVYP;5aLD2
OWin0W=cL`:2PX82D51@0]7Ad@9jNWX2Hf=S0Vi^KP03Ng]kOGemOGem009gMgL2JVYZ0W=cL`;GemL2
PX:20X>3P`9=CDd2Ogmo0WAdM0:EUID2oooo0U5AD@:ZZZX2IfMW0YJFUP:?Shl2:b/[0UeMG@:TY:@2
e]KF0R/[:`9:BTX2oooo0UEEE@:VYZH2IFEU0Ta<C08g=cL2cLg=0_ooo`:1PH42O7al0YVIV@9iNGT2
RHV90WQhN0;Cdm<2WinO0WMgM`9jNWX2Q8B40VATI09_Kfl2TI6A0Y^KV`9XJ6P2][Jf0SLg=`9gMgL2
/K6a0YZJVP9<C4`29RHV0Zb/[0:0P802Ogmo0Vm_K`9VIVH2IFEU0Y^KV`9TI6@2NGUi1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9nOWh2@4100S8b<P8P8202V9RH0S0`<09^KVh23Ph>0P00
00:ZZZX210@40UYJFP9FEUH2:BTY0Y2@T09hN7P2M7Ad0YVIV@9FEUH2e]KF0P00009kNg/2L71`0WMg
M`;Vi^H2RXZ:0Y^KV`9oOgl2[jn_0]oOg`:3Ph<2MgMg0W=cL`:4Q8@2HV9R0WUiN@:ATI42KVi^0Va/
K09MGEd2DU9B0R/[:`96ATH2^[Zj0Y>CT`8>3Ph2LW9b0UEEE@:LW9`2L71`0Y:BTP8g=cL2SXj>0Ta<
C0:IVIT2W9bL0WQhN09PH602fm_K0ZVYZ@9LG5`2VIVI0[jn_P9BDU800h20P7AdM7AdM002d=3@0_gm
o@9/K6`2PX:20/_;b`:n_[h2Fe]K0UADE0:k^k/2GUiN0WMgM`8B4Q82Kfm_0X>3P`9kNg/2Ph>30X61
P@:3Ph<2YJFU0V=SH`:>SXh2Ng]k0Va/K0:2PX82Ng]k0UIFEP9ZJVX2Rh^;0XJ6QP9dM7@2dm?C0[bl
_0861PH2:2PX0W=cL`:KVi/2@d=30WMgM`9?Cdl2YJFU0VIVIP9lO7`2M7Ad0W5aL@:PX:02>c/k0UmO
G`:7QhL2NWYj0ZZZZP9RHV82RXZ:0Rh^;P9kNg/2NWYj0WQhN0:>SXh2FUYJ0WQhN09_Kfl2P8200W]k
N`Goool100000Oooo`005Oooo`030000oooooooo00;oool2OWin0T10@08b<S82820P0YRHV08`<302
KVi^0Ph>3P800002ZZZZ0P@4109JFUX2EUIF0RTY:@:@T902N7Qh0WAdM0:IVIT2EUIF0]KFeP800002
Ng]k0W1`L09gMgL2i^KV0XZ:RP:KVi/2Ogmo0Zn_[`;Ogml2Ph>30WMgM`9cLg<2Q8B40V9RHP9iNGT2
TI6A0Vi^KP9/K6`2GEeM0U9BDP8[:b/2ATI60[Zj^P:CTi<23Ph>0W9bLP9EEED2W9bL0W1`L0:BTY82
=cLg0Xj>SP9<C4`2VIVI0YbLW09hN7P2H61P0]_Kf`:YZJT2G5aL0YVIV@:n_[h2DU9B00>0P81dM7Ad
M7@00]3@d0;moOd2K6a/0X:2PP;;bl/2_[jn0U]KF`9DE5@2^k^k0UiNGP9gMgL24Q8B0Vm_K`:3Ph<2
Ng]k0X>3P`:1PH42Ph>30ZFUY@9SHf<2SXj>0W]kN`9/K6`2PX:20W]kN`9FEUH2JVYZ0X^;R`:6QXH2
M7Ad0]?Cd`:l_;`21PH60RPX:09cLg<2Vi^K0T=3@`9gMgL2Cdm?0ZFUY@9VIVH2O7al0WAdM09aLG42
X:2P0S/k>`9OGel2QhN70WYjNP:ZZZX2HV9R0XZ:RP8^;Rh2Ng]k0WYjNP9hN7P2SXj>0UYJFP9hN7P2
Kfm_0X20P09kNg/5oooo0@00007oool001Goool00`000?ooooooo`02oooo0WQhN0:k^k/2OWin0WEe
M@9aLG42=cLg0P000090@402VIVI0]GEe@9RHV82;bl_0X20P0:KVi/2QXJ60YjNWP9fMWH2e=CD0VIV
IP9ADE42NGUi0P0000:ZZZX2/k>c0V][J`:6QXH2GEeM0XR8R0:UYJD2YjNW0Sdm?@:DU9@2RXZ:0[Fe
]@;2`/82=3@d0VATI0:1PH42MWIf0[Bd]08l?3`2eMGE19FEU@;Kfm/2e=CD0Xb<S0:5QHD2PX:20/C4
a0;moOd2KFe]0TY:BP9oOgl2O7al0WEeM@9^KVh2E5AD0UeMG@BBTY82OGem0WalO09oOgl00gYjNWal
O7alO002LG5a0Y6AT@;oool2KVi^0W9bLP:o_kl2F5QH0VMWI`9mOGd2Ogmo0XF5Q@:3Ph<4PH610U=C
D`:=SHd2k^k^0VIVIP:=SHd2g=cL0V9RHP:0P802PX:20WemO@9lO7`2R8R80WemO@9hN7P25aLG0V5Q
H@;?cll2De=C0Y6AT@:5QHD2IFEU0Vi^KP;2`/82CTi>0Ve]K@:e]KD2Z:RX0Vi^KP:IVIT2d=3@0U5A
D@;MgMd2/k>c0T51@@9BDU82HV9R0XZ:RP;2`/82G5aL0XZ:RP9iNGT2O7al0WemO@9hN7P2MgMg0W1`
L0:ATI42OWin1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9hN7P2^k^k0WinOP9eMGD2
LG5a0SLg=`800002@4100YVIV@;EeMD2HV9R0Rl_;`:0P802Vi^K0XJ6QP:NWYh2MWIf0]CDe09VIVH2
DE5A0WUiN@800002ZZZZ0[>c/`9[Jf/2QXJ60UeMG@:8R8P2YJFU0ZNWY`8m?Cd2U9BD0XZ:RP:e]KD2
`/;20S@d=09TI6@2PH610WIfMP:d];@2?3`l0]GEe@BEUID2fm_K0]CDe0:<S8`2QHF50X:2PP;4a<@2
oOgm0Ve]K@9:BTX2Ogmo0WalO09eMGD2KVi^0UADE09MGEd4TY:B0WemO@9lO7`2Ogmo00=jNWYlO7al
O7`00W5aL@:ATI42oooo0Vi^KP9bLW82_kno0UQHF09WIfL2OGem0WmoO`:5QHD2Ph>31861P@9CDe<2
SHf=0^k^kP9VIVH2SHf=0]cLg09RHV82P8200X:2PP9mOGd2O7al0XR8R09mOGd2N7Qh0QLG5`9QHF42
clo?0U=CD`:ATI42QHF50VEUI@9^KVh2`/;20Ti>CP9]KFd2]KFe0ZRXZ09^KVh2VIVI0]3@d09ADE42
gMgM0[>c/`91@D42DU9B0V9RHP::RXX2`/;20UaLG0::RXX2NGUi0WalO09mOGd2N7Qh0WMgM`9`L702
TI6A0WinOPGoool100000Oooo`005Oooo`030000oooooooo00;oool2[jn_0UADE09>CTh2Ph>30Pd=
3@;Lg=`2WinO0Zb/[09LG5`2_kno0[^k^`;@d=02Shn?0VQXJ0:OWil2UIFE0Xn?S`9DE5@2A4A40Xn?
S`9<C4`26QXJ0Z6QX@8H61P2J6QX0XJ6QP9GEeL2:RXZ0X>3P`9kNg/2c/k>0P`<30:8R8P2];Bd0YjN
WP9@D502Ogmo0WQhN09iNGT2GemO0V1PH09mOGd2Jf][0]gMg@8X:2P2bl_;0Xj>SP9XJ6P2cLg=0Xj>
SP:5QHD2LG5a0X20P09lO7`2/k>c0Vm_K`;Hf=P2KFe]0WAdM09BDU82YJFU0_ooo`9`L702QHF500>l
_;cclo?clo<00U5AD@8G5aL28B4Q0U=CD`9jNWX2CTi>0ZJVYP93@d<2Shn?0WEeM@9jNWX2MgMg0WAd
M0:9RHT29RHV0XR8R0;oool2Kfm_0Z2PX0;oool2IfMW0[:b/P;:b/X2RHV90WQhN09iNGT4OGem0[^k
^`8f=SH2GEeM0WinOP9jNWX2M7Ad0P00009fMWH2NWYj0WUiN@8l?3`25QHF0WinOP:2PX82=CDe0/_;
b`8L71`2ZZZZ0WQhN0:VYZH2OGem0YBDU08D51@2XZ:R0XF5Q@AjNWX2PH610WQhN0;5aLD2GEeM0YVI
V@;5aLD2WYjN1Oooo`400001oooo000Eoooo0P0000?oool2[jn_0UADE09>CTh2Ph>30Pd=3@;Lg=`2
WinO0Zb/[09LG5`2_kno0[^k^`;@d=02Shn?0VQXJ0:OWil2UIFE0Xn?S`9DE5@2A4A40Xn?S`9<C4`2
6QXJ0Z6QX@8H61P2J6QX0XJ6QP9GEeL2:RXZ0X>3P`9kNg/2c/k>0P`<30:8R8P2];Bd0YjNWP9@D502
Ogmo0WQhN09iNGT2GemO0V1PH09mOGd2Jf][0]gMg@8X:2P2bl_;0Xj>SP9XJ6P2cLg=0Xj>SP:5QHD2
LG5a0X20P09lO7`2/k>c0Vm_K`;Hf=P2KFe]0WAdM09BDU82YJFU0_ooo`9`L702QHF500>l_;cclo?c
lo<00U5AD@8G5aL28B4Q0U=CD`9jNWX2CTi>0ZJVYP93@d<2Shn?0WEeM@9jNWX2MgMg0WAdM0:9RHT2
9RHV0XR8R0;oool2Kfm_0Z2PX0;oool2IfMW0[:b/P;:b/X2RHV90WQhN09iNGT4OGem0[^k^`8f=SH2
GEeM0WinOP9jNWX2M7Ad0P00009fMWH2NWYj0WUiN@8l?3`25QHF0WinOP:2PX82=CDe0/_;b`8L71`2
ZZZZ0WQhN0:VYZH2OGem0YBDU08D51@2XZ:R0XF5Q@AjNWX2PH610WQhN0;5aLD2GEeM0YVIV@;5aLD2
WYjN1?ooo`800001oooo000Eoooo00<0003oooooool00_ooo`9]KFd2h^;R0TI6AP8];Bd2E5AD0Wal
O0:VYZH2/;2`0XR8R0:TY:@2AdM70UiNGP9bLW82O7al0VMWI`:?Shl2CTi>0Slo?`:ZZZX2Xj>S0V][
J`810@42Ogmo0[Zj^P:2PX82WYjN0XZ:RP9OGel2QhN70Z>SX`:g]kL28R8R0V=SH`:7QhL2KFe]0T51
@@:ZZZX2oooo0WIfMP9hN7P2bl_;0X^;R`9QHF42RXZ:0_[jnP9eMGD2UIFE0YfMW@;:b/X2emOG0RTY
:@;:b/X2`L710W1`L09hN7P2a<C40XR8R09JFUX2[jn_0PT92@9;Bd/2ATI60X20P09nOWh00ooooeQH
F5QHF002QhN70X:2PP9ZJVX2oooo0VUYJ@9[Jf/2FUYJ0Y2@T09eMGD2OWin0WalO09oOgl2Ph>30WEe
M@;5aLD2O7al0WinOP;4a<@2NGUi0Rh^;P99BDT2_[jn0W1`L0851@D2U9BD0W=cL`9nOWh2@d=30_Og
m`8S8b<2S8b<0Ve]K@9iNGT200000UmOG`:1PH42Bd];0Y2@T09JFUX2Cdm?0WmoO`9ADE42U9BD0YbL
W0800002jNWY0SPh>0;[jn/230`<0X^;R`9MGEd2Kfm_0/O7a`9YJFT2Ph>30VQXJ0:3Ph<2c<c<0TY:
BP:DU9@2IFEU0ZJVYPGoool100000Oooo`005Oooo`030000oooooooo00;oool2KFe]0^;RhP96ATH2
;Bd]0UADE09lO7`2YZJV0[2`/0:8R8P2Y:BT0TM7A`9NGUh2LW9b0WalO09WIfL2Shn?0Ti>CP8o?cl2
ZZZZ0Z>SX`9[Jf/20@410WmoO`:j^[X2PX:20YjNWP::RXX2GemO0XN7Q`:SXj<2]kNg0R8R8P9SHf<2
QhN70Ve]K@91@D42ZZZZ0_ooo`9fMWH2N7Qh0/_;b`:;Rh/2HF5Q0XZ:RP;jn_X2MGEe0YFEU@:MWId2
b/[:0]OGe`8Y:BT2b/[:0/71`@9`L702N7Qh0/C4a0:8R8P2FUYJ0Zn_[`892@T2Bd];0TI6AP:0P802
OWin00?ooomHF5QHF5P00XN7Q`:2PX82JVYZ0_ooo`9YJFT2Jf][0UYJFP:@T902MGEe0WinOP9lO7`2
Ogmo0X>3P`9eMGD2aLG50WalO09nOWh2a<C40WUiN@8^;Rh2BDU90[jn_P9`L7021@D50YBDU09cLg<2
OWin0T=3@`;gmoL28b<S0Xb<S09]KFd2NGUi0P00009OGel2PH610T];B`:@T902FUYJ0Tm?C`9oOgl2
DE5A0YBDU0:LW9`200000^WYj@8h>3P2jn_[0P`<30:;Rh/2GEeM0Vm_K`;7alL2JFUY0X>3P`9XJ6P2
Ph>30/c<c09:BTX2U9BD0VEUI@:VYZH5oooo0@00007oool001Goool00`000?ooooooo`02oooo0Win
OP;EeMD2ATI60X61P@99BDT2fm_K0X61P@9:BTX2][Jf0W=cL`8b<S82O7al0XJ6QP9lO7`2K6a/0Xf=
S@9aLG42oooo0XB4Q09gMgL200000YFEU@:JVYX2I6AT0WYjNP9XJ6P2[Jf]0XZ:RP;Bd]82_[jn0^OW
i`:?Shl2Ng]k0X:2PP9nOWh2MWIf0VEUI@:1PH42QHF50]oOg`:>SXh2FEUI0Z>SX`8c<c<2CDe=0Y>C
T`:m_Kd2>CTi0YZJVP;0`<02Shn?0/g=c@9kNg/2]KFe0^[ZjP9RHV82>c/k0_KfmP9QHF42P8200YjN
WP:DU9@2Ng]k0WYjNP03LG5a5ADE5ADE009?Cdl2>3Ph0ZRXZ09SHf<2OGem0Xf=S@:CTi<2Lg=c0Wmo
O`9kNg/2OGem0WYjNP9aLG42P8200VATI09mOGd2Q8B40[^k^`9iNGT2PH610UEEE@:;Rh/2Jf][0T10
@0:3Ph<2OGem0Ve]K@:o_kl2A4A40/[:bP9_Kfl2GemO0U9BDP9aLG42UYJF0U]KF`:>SXh2@D510^GU
i@9VIVH292@T0Z2PX08Y:BT2S8b<0Z2PX09gMgL2FEUI0WMgM`9VIVH2S8b<0SLg=`8m?Cd2M7Ad0X:2
PP9HF5P2TY:B0YjNWP9OGel2AdM70X61P@:CTi<2P8201Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9nOWh2eMGE0TI6AP:1PH42BDU90]_Kf`:1PH42BTY:0[Jf]P9cLg<2<S8b0WalO0:6QXH2
O7al0Va/K0:=SHd2LG5a0_ooo`:4Q8@2MgMg0P0000:EUID2VYZJ0VATI09jNWX2J6QX0Zf][@::RXX2
d];B0[jn_P;WinL2Shn?0W]kN`:2PX82OWin0WIfMP9UIFD2PH610XF5Q@;Ogml2SXj>0UUIF@:SXj<2
<c<c0Te=C@:CTi<2_Kfm0STi>@:JVYX2`<300Xn?S`;=cLd2Ng]k0[Fe]@;Zj^X2HV9R0S/k>`;fm_H2
HF5Q0X20P0:NWYh2U9BD0W]kN`9jNWX00g5aLADE5ADE5@02Cdm?0SPh>0:XZ:P2Hf=S0WemO@:=SHd2
Ti>C0W=cL`9oOgl2Ng]k0WemO@9jNWX2LG5a0X20P09TI6@2OGem0XB4Q0:k^k/2NGUi0X61P@9EEED2
Rh^;0V][J`90@402Ph>30WemO@9]KFd2_kno0TA4A0;:b/X2Kfm_0UmOG`9BDU82LG5a0YJFUP9KFe/2
SXj>0T51@@;UiND2IVIV0R@T90:PX:02:BTY0Xb<S0:PX:02MgMg0UUIF@9gMgL2IVIV0Xb<S08g=cL2
?Cdm0WAdM0:2PX82F5QH0Y:BTP:NWYh2GemO0TM7A`:1PH42Ti>C0X20P0Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2C4a<0YjNWP:0P802SHf=0SXj>P:9RHT2VIVI0[Ng]`:0P802?clo0[Jf
]P:>SXh2WIfM0ZRXZ09nOWh2CTi>0Z2PX08000026a/K0Z>SX`:TY:@2EeMG0[Jf]P8[:b/2YZJV0Zf]
[@9BDU82JVYZ0UUIF@8];Bd2^[Zj0XJ6QP:h^;P2Lg=c0XB4Q09MGEd2Lg=c0[Zj^P9_Kfl2JFUY0Xj>
SP9[Jf/2Q8B40XJ6QP:8R8P2<C4a0YFEU@9OGel2PH610/30`0:ZZZX2JVYZ0YnOW`8a<C42KFe]0TE5
A@:3Ph<2Rh^;0X>3P`9JFUX2[Jf]0[>c/`9[Jf/2[Jf]00=VIVI]KFe]KFd00YRHV08G5aL2CTi>0Xn?
S`9/K6`2[:b/0[Rh^09^KVh2P8200W]kN`9^KVh2Z:RX0/W9b@9ZJVX2S8b<0WMgM`9kNg/2Jf][0X61
P@99BDT23Ph>0VUYJ@9QHF42Shn?0X:2PP9kNg/2OGem0WEeM@9]KFd200000Z:RXP9SHf<2_;bl0ZVY
Z@9lO7`200000Y6AT@9ADE42oooo0YRHV08Y:BT2Zj^[0VMWI`8B4Q82Fe]K0U9BDP;2`/82][Jf0__k
n`:4Q8@2P8200Pl?3`;:b/X2IFEU0[Vi^@9GEeL27QhN0S4a<@9QHF42NGUi0TM7A`;Hf=P5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0Ta<C0:NWYh2P8200Xf=S@8j>SX2RHV90YVIV@:g]kL2
P8200Slo?`:f][H2SXj>0YfMW@:XZ:P2OWin0Ti>CP:PX:0200000Q/K6`:SXj<2Y:BT0UMGE`:f][H2
:b/[0ZJVYP:][Jd2DU9B0VYZJP9IFET2;Bd]0[Zj^P:6QXH2^;Rh0W=cL`:4Q8@2GEeM0W=cL`:j^[X2
Kfm_0VUYJ@:>SXh2Jf][0XB4Q0:6QXH2R8R80S4a<@:EUID2GemO0X61P@;0`<02ZZZZ0VYZJP:OWil2
<C4a0Ve]K@95ADD2Ph>30X^;R`:3Ph<2FUYJ0Zf][@:c/k<2Jf][0Zf][@03IVIVKFe]KFe]00:HV9P2
5aLG0Ti>CP:?Shl2K6a/0Zb/[0:h^;P2KVi^0X20P09kNg/2KVi^0ZRXZ0;9bLT2JVYZ0Xb<S09gMgL2
Ng]k0V][J`:1PH42BDU90Ph>3P9YJFT2HF5Q0Xn?S`:2PX82Ng]k0WemO@9eMGD2KFe]0P0000:RXZ82
Hf=S0[bl_0:YZJT2O7al0P0000:ATI42DE5A0_ooo`:HV9P2:BTY0Z^[Z`9WIfL24Q8B0U]KF`9BDU82
`/;20[Jf]P;kno/2Q8B40X20P08?3`l2b/[:0VEUI@:i^KT2EeMG0QhN7P8a<C42HF5Q0WUiN@97AdL2
f=SH1Oooo`400001oooo000Eoooo00<0003oooooool0ooooo`Ooool100000Oooo`005Oooo`030000
oooooooo0?ooool7oooo0@00007oool001Goool00`000?ooooooo`02oooo00<0003oooooool09Ooo
o`030000oooooooo02Goool00`000?ooooooo`0Uoooo00<0003oooooool09Oooo`030000oooooooo
02Coool00`000?ooooooo`0Uoooo00<0003oooooool04_ooo`400001oooo000Eoooo0P0000?oool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01_ooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`800001
oooo000Eooooo`0000/00001oooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo003ooooo8Ooo
o`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-13.248, -11.0235, 0.501294, 
  0.501294}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    RowBox[{"\[FilledCircle]", " ", 
      StyleBox["Compression",
        FontVariations->{"Underline"->True}]}]], "Input"],

Cell[BoxData[
    \(\(n = 128; \)\)], "Input"],

Cell[TextData[StyleBox[
"nPrime=32;\ncutwa=TakeMatrix[wa,{1,1},{32,32}];\n\
compressedwSpec=BlockMatrix[{{cutwa,ZeroMatrix[nPrime,n-nPrime]},{ZeroMatrix[\
n-nPrime,n]}}];",
  FontSize->14]], "Input",
  PageBreakWithin->Automatic,
  FontSize->16],

Cell[BoxData[
    \(\(b = Transpose[w] . compressedwSpec . w; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ListDensityPlot[Transpose[w] . compressedwSpec . w, Mesh -> False]\)], 
  "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0192308 0.00751202 0.0192308 0.00751202 [
[.01923 -0.0125 -3 -9 ]
[.01923 -0.0125 3 0 ]
[.16947 -0.0125 -6 -9 ]
[.16947 -0.0125 6 0 ]
[.31971 -0.0125 -6 -9 ]
[.31971 -0.0125 6 0 ]
[.46995 -0.0125 -6 -9 ]
[.46995 -0.0125 6 0 ]
[.62019 -0.0125 -6 -9 ]
[.62019 -0.0125 6 0 ]
[.77043 -0.0125 -9 -9 ]
[.77043 -0.0125 9 0 ]
[.92067 -0.0125 -9 -9 ]
[.92067 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .16947 -12 -4.5 ]
[-0.0125 .16947 0 4.5 ]
[-0.0125 .31971 -12 -4.5 ]
[-0.0125 .31971 0 4.5 ]
[-0.0125 .46995 -12 -4.5 ]
[-0.0125 .46995 0 4.5 ]
[-0.0125 .62019 -12 -4.5 ]
[-0.0125 .62019 0 4.5 ]
[-0.0125 .77043 -18 -4.5 ]
[-0.0125 .77043 0 4.5 ]
[-0.0125 .92067 -18 -4.5 ]
[-0.0125 .92067 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.01923 0 m
.01923 .00625 L
s
[(0)] .01923 -0.0125 0 1 Mshowa
.16947 0 m
.16947 .00625 L
s
[(20)] .16947 -0.0125 0 1 Mshowa
.31971 0 m
.31971 .00625 L
s
[(40)] .31971 -0.0125 0 1 Mshowa
.46995 0 m
.46995 .00625 L
s
[(60)] .46995 -0.0125 0 1 Mshowa
.62019 0 m
.62019 .00625 L
s
[(80)] .62019 -0.0125 0 1 Mshowa
.77043 0 m
.77043 .00625 L
s
[(100)] .77043 -0.0125 0 1 Mshowa
.92067 0 m
.92067 .00625 L
s
[(120)] .92067 -0.0125 0 1 Mshowa
.125 Mabswid
.05679 0 m
.05679 .00375 L
s
.09435 0 m
.09435 .00375 L
s
.13191 0 m
.13191 .00375 L
s
.20703 0 m
.20703 .00375 L
s
.24459 0 m
.24459 .00375 L
s
.28215 0 m
.28215 .00375 L
s
.35727 0 m
.35727 .00375 L
s
.39483 0 m
.39483 .00375 L
s
.43239 0 m
.43239 .00375 L
s
.50751 0 m
.50751 .00375 L
s
.54507 0 m
.54507 .00375 L
s
.58263 0 m
.58263 .00375 L
s
.65775 0 m
.65775 .00375 L
s
.69531 0 m
.69531 .00375 L
s
.73287 0 m
.73287 .00375 L
s
.80799 0 m
.80799 .00375 L
s
.84555 0 m
.84555 .00375 L
s
.88311 0 m
.88311 .00375 L
s
.95823 0 m
.95823 .00375 L
s
.99579 0 m
.99579 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .16947 m
.00625 .16947 L
s
[(20)] -0.0125 .16947 1 0 Mshowa
0 .31971 m
.00625 .31971 L
s
[(40)] -0.0125 .31971 1 0 Mshowa
0 .46995 m
.00625 .46995 L
s
[(60)] -0.0125 .46995 1 0 Mshowa
0 .62019 m
.00625 .62019 L
s
[(80)] -0.0125 .62019 1 0 Mshowa
0 .77043 m
.00625 .77043 L
s
[(100)] -0.0125 .77043 1 0 Mshowa
0 .92067 m
.00625 .92067 L
s
[(120)] -0.0125 .92067 1 0 Mshowa
.125 Mabswid
0 .05679 m
.00375 .05679 L
s
0 .09435 m
.00375 .09435 L
s
0 .13191 m
.00375 .13191 L
s
0 .20703 m
.00375 .20703 L
s
0 .24459 m
.00375 .24459 L
s
0 .28215 m
.00375 .28215 L
s
0 .35727 m
.00375 .35727 L
s
0 .39483 m
.00375 .39483 L
s
0 .43239 m
.00375 .43239 L
s
0 .50751 m
.00375 .50751 L
s
0 .54507 m
.00375 .54507 L
s
0 .58263 m
.00375 .58263 L
s
0 .65775 m
.00375 .65775 L
s
0 .69531 m
.00375 .69531 L
s
0 .73287 m
.00375 .73287 L
s
0 .80799 m
.00375 .80799 L
s
0 .84555 m
.00375 .84555 L
s
0 .88311 m
.00375 .88311 L
s
0 .95823 m
.00375 .95823 L
s
0 .99579 m
.00375 .99579 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.16947 .99375 m
.16947 1 L
s
.31971 .99375 m
.31971 1 L
s
.46995 .99375 m
.46995 1 L
s
.62019 .99375 m
.62019 1 L
s
.77043 .99375 m
.77043 1 L
s
.92067 .99375 m
.92067 1 L
s
.125 Mabswid
.05679 .99625 m
.05679 1 L
s
.09435 .99625 m
.09435 1 L
s
.13191 .99625 m
.13191 1 L
s
.20703 .99625 m
.20703 1 L
s
.24459 .99625 m
.24459 1 L
s
.28215 .99625 m
.28215 1 L
s
.35727 .99625 m
.35727 1 L
s
.39483 .99625 m
.39483 1 L
s
.43239 .99625 m
.43239 1 L
s
.50751 .99625 m
.50751 1 L
s
.54507 .99625 m
.54507 1 L
s
.58263 .99625 m
.58263 1 L
s
.65775 .99625 m
.65775 1 L
s
.69531 .99625 m
.69531 1 L
s
.73287 .99625 m
.73287 1 L
s
.80799 .99625 m
.80799 1 L
s
.84555 .99625 m
.84555 1 L
s
.88311 .99625 m
.88311 1 L
s
.95823 .99625 m
.95823 1 L
s
.99579 .99625 m
.99579 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .16947 m
1 .16947 L
s
.99375 .31971 m
1 .31971 L
s
.99375 .46995 m
1 .46995 L
s
.99375 .62019 m
1 .62019 L
s
.99375 .77043 m
1 .77043 L
s
.99375 .92067 m
1 .92067 L
s
.125 Mabswid
.99625 .05679 m
1 .05679 L
s
.99625 .09435 m
1 .09435 L
s
.99625 .13191 m
1 .13191 L
s
.99625 .20703 m
1 .20703 L
s
.99625 .24459 m
1 .24459 L
s
.99625 .28215 m
1 .28215 L
s
.99625 .35727 m
1 .35727 L
s
.99625 .39483 m
1 .39483 L
s
.99625 .43239 m
1 .43239 L
s
.99625 .50751 m
1 .50751 L
s
.99625 .54507 m
1 .54507 L
s
.99625 .58263 m
1 .58263 L
s
.99625 .65775 m
1 .65775 L
s
.99625 .69531 m
1 .69531 L
s
.99625 .73287 m
1 .73287 L
s
.99625 .80799 m
1 .80799 L
s
.99625 .84555 m
1 .84555 L
s
.99625 .88311 m
1 .88311 L
s
.99625 .95823 m
1 .95823 L
s
.99625 .99579 m
1 .99579 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
128 string
128 128 8 [128 0 0 128 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
2E27211E1D1F25303D484F4D463C343131333432302E2D2F
3235373735343230303137414E5A61605A5452596470787B
7870634E3826212E465F6F6F614F3F353132394656656E6D
6559514F535758544C444349546066645C534F535C646151
3922161C2F435151473A302E32383E43474949453E383435
393E424443403B35
30251C1614161D2A3A4952514A4037323233343433323131
32323332333334343639425061717A78706868758A9EABAD
A4957E6347322C3A5674847F6A503A2F2F353F4C5A666C68
5D504A4F5B686F695C4F484B57636864584B44464F57574A
3622171A27374347443F3B3838393C434B5255534D453E3B
3A3B3D404244413A
3B2D211814151C28394852524B4138333131323131313030
302F303135393D41454A525F6E7C838179737788A1B9C9CB
C1AF977A5F4943526E8B9A92785941383B444F5A636B6D67
5C514E57697B847F6F5D51515962655E51443D3F48505249
3A2A1F1D232B333A3E424547494A4E545B6165645F58514A
454140444A4F4E47
4E4032282321252E3A474F4F4A4139332F2D2B2928282929
2A2C30363E464E565D62676C707373716F7078899EB3C1C6
C1B7A7927D6C68758CA3AEA38A6E5851565F686E7173716B
645D5E697B8C948F7F6C5D5757595851463D3A3F48515550
44372B232020232A333F4B58656F767978767371706E6A63
5A524F535A61625B
67594B41393535393F464B4B474039332D2723201E1E1F21
262C35404C5965717A7F7E766A5D55545B65727F8A959FA7
ADB0AEA79D94939BAAB8BDB39E8979757A818787837E7975
737478828F9A9E998C7B6A5E554E48413C3A3D4550595E5C
54483A2C2018161B26374D69859CA9A89B8B7E7B7F848680
756B656870777973
7D6F6258504A464547494A48454039322B241E1A18181A1E
25303D4D5E6E7E8D979B9480654A3A3B495D6D7475747984
95A5B2B8B9B9BABEC4C8C6BEB1A49C9B9FA4A59F958B8483
878D959DA3A7A7A19688776655483E393A3E47515C646A6A
665C4D3B281910111C2F4D76A1C6D9D3BA9C86828B99A19D
91847D7E858C8D88
877A6D645C5754535454534E463E352E2924211F1F202429
313C4B5D718495A5AFB1A68C694834384B6374766C62616E
859FB5C4CCD0D1D1D0CEC9C4BEB9B8B9BDBFBDB4A79B9495
9CA6AFB5B6B5B0A99F9282705E50464449515B646B707578
797467533B261813192B4B7AAFDDF4EAC8A1858293A8B7B5
A8998F8F949A9A94
8273665E5A595C61676B695E4D3C2D2726292E33383E4347
4C545F708497A9B6BDBDB29A7C6052576A818E8A7B6C6671
88A2B9C8CFD2D2D0CDC8C4C2C2C3C6C9CCCDCAC2B8AEA9AA
B1BAC1C6C6C3BDB3A6998B7E736B67696E767C7D7D7B7D83
8A8D87755C432F23222B4672A6D5EDE1BD94797C94B2C6C6
B7A59794999D9B91
715E514B4C525D6D7C878675593B2620273443515D676E70
7071778496A8B7C0C2C0B7A99A8B868D9DADB4AC9B898086
98ACBDC6C8C6C3C0BDBBBABCBFC3C8CBCECFCDCAC5C1BFC0
C4C8CCD0D2D0C9BCAB9B8F8A8C91979DA1A3A0998E84828A
98A4A69A836A523F343241638FB8CDC1A07C687291B5CCCD
BDA7979295989485
5E4737343A475B748D9E9F8A664127222E445C71828F9797
938E8D96A6B6C2C5C3BEB9B7B7B9BEC7D1D9D9CEBCAA9FA0
ABB8C1C2BCB4AEAAABACAFB3B8BEC3C6C8CACBCCCED0D1D2
D2D1D3D6DADBD3C2AB968C91A2B5C6CFD0CBC0B09D8D8790
A2B3BDB8A79178604C3F41557494A49A806459688BB0C8CA
BAA4938E90918A78
52362320293B53718DA2A69473513A37465D768C9DAAB1B0
A9A19EA4B0BEC7C7C1BBB9C0CCD9E6F0F6F8F3E6D4C2B6B4
B9C0C3BEB3A79E9C9EA2A8ADB3B8BCBFC1C2C5C9CFD5D9DA
D9D6D5D9DEDFD6BFA38A818EAACAE3EEEBE0CFBBA6958D94
A4B6C3C4BBAC977F68544B526477817B6A59556683A2B7B9
AD9C8F8D90908770
52311B161F3148617789928D7F6F64667180909DA6AEB1AF
AAA5A4AAB5C1C7C6BFB9B9C2D2E3F1FBFFFFF8EBDAC9BDBA
BDC2C3BDB2A69D9B9EA3A8ADB1B4B7B9BBBCBFC3C8CED3D6
D7D6D6DADDDCCFB492766C7DA0C7E5EFE8D9C6B5A79C9698
A0AAB3B9BAB5AB9A86726461666D72706A64656E7D8D989B
9892909399998F75
5C371D151C2B3D4A55606C7A87939CA1A4A5A5A3A2A09E9D
9D9EA3ABB6BFC5C4BFBABAC1CDDAE6EFF4F4EFE2D2C2B7B5
BAC0C3C0B7AEA8A6A8ABAEB0B1B2B3B5B7B8BABBBCBFC3C9
CED2D6D9DAD4C3A37E5E526489B3D1D9CEBCACA3A2A2A19C
9693959CA7B1B4AEA394877D767373767A7F817F79757377
7E87929DA6A89D81
684124191E2B3638373A48668DB3CDD7D1C4B3A4968C8685
8A939EA9B4BDC2C2BFBCBABDC3CBD3DBE1E4E0D5C6B6ADAD
B4BDC4C4C0BAB6B5B7B9B9B7B3B0AEAFB2B5B6B4B0AEB1B9
C3CDD4D8D6CDB8956C493B4B7099B6BBAE9A8D8F9BA8ABA0
8C7A737C90A6B7BEBBB4A99B8C807C83919FA1937A625356
667B91A4B0B4AA8E
71492B20263139362E2C3A6092C3E5EFE5CFB69D8A7A7273
7B899AA8B4BDC2C3C1BDBBBABCBFC4CBD2D7D5CABAABA2A4
AFBCC7CBC9C6C4C4C6C8C5BEB4AAA5A6ACB3B5B1AAA4A4AD
BBC9D3D8D5CBB38F64402F3C5E86A1A49582798398ADB3A3
866A5D667F9DB7C6CBC9C1B19F8E8892A6B8BBA5805B4343
556F8BA2B1B7AF96
704B3029313F4A4B4644506E97BDD7DDD3BFA89280726B6D
788799AAB8C3C8C8C4BFBBB9BABCC2C9D0D5D2C6B4A39B9F
ADBECCD2D1CDCCCED3D5D1C4B2A1989AA6B2BAB6ADA4A2AA
B9C8D4DBD9D0BA956B45323D5B809A9D907F79859DB2B8A7
886A5B637D9CB8C8CECEC7B9A898929EB2C4C7B18C664B45
50647B92A4AEAA93
694933313F53656F73777E8A99A6ADADA89E92867A726F73
7E8C9DAFBFCCD2D0CAC1BBB9BCC2C9D1D7DAD6C7B2A0969C
AEC3D3D9D6D0CED3DBE1DCC9AF97898E9FB3C0C0B7ADA9AF
BDCBD7DEE0DAC7A57C56424A65869FA49A8D8893A6B8BCAC
9076686F86A0B7C4C8C6C0B5A99E9CA6B6C4C7B79C7D6558
565B67798C9A9B88
6045363A4E698397A4ACACA598897D78787B7D7C79787A7F
8893A2B3C6D5DCD9CFC3BBBBC1C9D2DAE0E1DAC9B29D939A
AEC6D9DED9D1CFD5E1E8E4CEAE907F859BB4C6C9C1B8B3B7
C2CED9E2E6E4D5B7906C585D7592A9AFA99F9BA3B1BDBFB1
9B867C8191A5B5BDBDB9B4AEA9A6A6ADB7BFC1BAAC9A8571
5F53515D7082887A
56423A4661809EB6C5CBC5B091725B545B67747C8084888D
939BA6B7CADBE3DFD4C6BEBEC5CED9E1E6E7DECBB29C9199
AFC8DBE0D9D0CDD4E2EBE8D1AF8F7E8399B4C7CCC6BEB9BC
C5CFD9E2E8E9DDC3A1816E7287A1B5BAB4ABA7ACB7C0C1B7
A5938A8D9AA8B3B7B4B0ACABACAFB2B5B7B8BABAB9B3A389
6A4D3D425569736A
4D4141547496B2C4C9C6B8A0836755535E6F808B9196989B
9DA1A9B8CBDCE5E2D7CAC1C1C7D0DAE2E8E9E1CDB49D929A
AFC8DADED6CDC9D0DEE8E6D3B59786899BB0C0C5C0BAB7BB
C4CDD6DEE4E4DBC5A98E808598AFBFC0B6AAA5AAB6C2C5BB
A9978D8F9AA8B2B5B3AFADAFB4BABDBCB8B4B3B9C0C2B799
70482F2D3D515E5A
4641486285A7BEC2B7A6917F7268666D7A8B99A3A7A9A8A6
A5A6ABB8C9D9E2E0D7CDC5C4C9CFD7DFE4E6DFCDB59F949B
AEC5D5D8D1C7C3CAD6E0E0D3BCA697969FABB4B6B3B0B1B6
BFC8D1D7DAD9D1BFA9958C94A7BBC7C2B2A199A0B1C2C9BF
AB95888994A3B0B5B6B5B5B9BFC5C6C1B8B0AEB6C1C8BE9F
7144241E2A3E4C4D
41424E6B90B1C0B89F816A616571808F9DAAB5BABCBAB5B1
ACABAEB8C6D3DCDCD5CEC8C7CBCFD5DADFDFD9C8B3A0969C
ADC0CED0C9C0BBC0CAD3D7D1C4B6ABA5A5A7A8A7A6A6A9B1
BBC4CBCFCFCCC3B6A79A97A1B3C6CDC3AE988D96ACC2CCC3
AC9383828C9CAAB3B7BABEC3C8CCCAC2B7ACA9B1BDC5BC9D
6E3F1D141E303F43
41424E6A8DABB7AA8C6C5656677E97A9B6C0C5C7C5C1BBB6
B2B0B3BAC3CDD3D3CFCAC8C9CDD2D5D6D5D1C8BAAA9C969C
ABBCC6C7BFB6B1B4BBC4CACCCAC5BEB6AEA6A09D9DA1A7B0
BAC2C8C8C5BFB7ADA49E9FAABBCBD0C4AE978C96ACC2CDC6
B098857F848E9AA6B1BAC3C9CDCDC9BFB3A7A3A9B4BAB193
673A1A121C2E3D42
4742475B7893A19B8873676B7B90A3B2BBC0C3C2BEBAB5B3
B4B6B9BEC2C7C9C8C5C2C3C9D1D7D9D3C7B9AB9F9691929B
A9B6BEBDB5ACA6A6ACB4BDC6CCD0CEC5B8AAA09C9EA5ADB6
BEC5C7C5BFB7AFA8A3A2A5B0BDC9CDC4B3A39BA2B3C4CDC8
B9A693857D7A7F8C9EB1C1C9CAC7C0B6ACA29D9FA5A89D81
5A341B18263B4A4D
52443E485C73858D8F8F90949AA1A6ABAFB1B2AFABA7A7AB
B3BBC1C3C3C2BFBBB9B7BCC7D4DFDFD0B89E887E7D828C99
A6B1B7B5ACA29B999DA5B1BFCCD7D9D1C2B2A6A2A6AEB7BF
C5C8C8C4BCB3ABA6A5A6ABB2BBC3C7C4BDB6B3B7BFC6CBCA
C4B9A89077645E6A84A2BAC4C3BCB3ABA49C97959490846B
4B2E1F243950605E
5E4A3B3A45566C849BAFBCBFB9AFA59F9E9E9D99949295A0
AFBEC7C9C4BEB7B1AEADB4C4D7E5E4CEAB866A6065728394
A2ADB2AEA59A928F929AA7B8CAD9DFD9CBBBAFABAFB7BFC5
C9CBC9C3BAB1A9A6A7ABAFB3B7BBBFC3C6C9CBCBCAC7C7CA
CDCBBB9C7450404C6C92B0BCB8AFA59F9C98928B82776752
3B2824324B667570
6A57473F424D6282A6C7DBDBCCB6A194908F8D8883818897
ACBFCBCCC6BDB3ACA8A9B1C3D8E7E6CEA57B5B4F5563778B
9CAAB0ADA3978D898D95A2B3C4D4DCD9D0C3B9B4B5B9BEC3
C7CAC9C3B9AFA8A7AAB0B5B7B7B7BAC2CCD5DAD7CFC6C2C7
D0D4C7A37347313B5E87A7B3AEA298959796908372614E3C
2D232639536D7C7A
726C655F5C606F8AADCDDFDDCAB19A8E8A8A89847F7D8393
A8BDCACDC8C0B7B0ACACB3C4D7E6E5CFAA8363525058687D
94A8B4B2A7988D898E97A3B0BCC8CFD1CEC9C2BBB5B1B0B5
BDC4C7C1B7ABA5A6AEB7BEBFBDBABBC2CDD7DBD6CCC0BAC0
CAD0C6A5784E373F5E84A1AAA499909094958F7F69523D2D
231F25344A5F6F74
77868E8F8B878B9BB0C5CFCAB8A3928A8A8D8D8983818593
A6B9C7CCCAC5BFB9B5B4BAC6D6E1E1D2B7997B655752596F
8BA8BABBAF9E908E949EA8AFB4B8BDC3C9CCC8BEB0A29CA1
ADBBC2BFB4A8A2A6B2BEC7CAC7C3C1C4C9CFD0CBC1B7B3B7
BEC3BBA2805F4E536A869BA19B928C8D92958E7D644A3425
1F1E222A35435465
7A9EB8C1BDB3ACADB3B9B8AFA19288878B9294918A858792
A4B6C4CBCDCBC8C4C0BFC1C9D3DBDDD6C7B39A7E64525063
84A8C0C2B5A295949DA8B0B1ADAAABB5C2CCCCBFAA94898D
9DAFBBBAB1A7A3A9B6C4CFD2D1CCC8C6C5C3C0BBB4AEACAE
B2B3AE9E8974696C79899598938C888B90928C7A61453024
2022242323283855
80B1D6E5E2D5C7BCB5ADA4978A807C818B9599958C83828D
9FB4C5CFD2D1CFCBC8C5C6CAD0D6DADAD5CAB799775B5160
81A6BFC2B5A39799A6B4BCB8ADA3A0AABAC8CCBEA58C7E80
8FA2B0B4B0ACABB1BCC7D0D5D5D4CFC8C0B7B0ABA8A7A7A8
A9A8A49B90847D7D82898E8E8A8684868B8C85735A402D26
292F322B211E2B4F
88BAE1F1EFE2D1C2B5A798897B72717987959A928375727F
99B5CBD7D8D5D0CBC7C5C5C8CDD3D9DDDDD9C9AE8C6F616A
83A1B5B7AB9C959DAEC0C9C4B6A7A0A6B4C2C5B9A48E7F7D
8693A0AAB1B7BCC0C3C5C8CDD2D5D3CABDAFA49F9FA2A5A7
A7A6A29B938A838181848685827F7E7F818178664E36292B
394850483A303758
92BBDAE8E6DCCEC0B3A59584746967718291968A73605B6E
91B7D5E1DFD7CDC6C2C0C1C5CBD2DADFE1DDD1BCA188797B
8899A5A39A90909DB3C9D5D2C4B4A9AAB1B9BAB3A5968A84
838690A0B3C5D1D1C9BFBABFC9D2D5CBBAA99C979A9FA5A9
ABAAA69D92877D79797B7D7D7B787675757268563F2C2633
4C6777736355556C
99B4C9D2D1CAC1B7AEA395847265626C7D8D907F644B465E
8ABADDEBE5D8C9C0BCBBBDC2CAD3DBDFDFDBD1C2B19F928D
8E92938F88848A9CB6CEDEDED3C4B7B1B0AFAFACA8A39B90
857E8397B4D1E2DECDB9ADB1BFCED5CCB8A49591969EA7AD
B0B0AA9F9081756F6F72757674716D6B686359473426283E
62869D9E8F7D7580
97A6B1B6B5B1ADA9A6A197887869646B7A8889775A413E59
89BCE2EFE8D7C7BDBAB9BCC3CBD5DCDDD9D2C9BFB6ABA299
928D88817D7D8699B3CCDDE2DCD1C4B9B0A9A5A7ACB1AFA2
90807F92B2D3E5E0CBB3A4A9BACCD3CAB59F908D959FA9B0
B2B1AB9F8F7F726C6C6F71706D68635F5C585042332B324D
759DB7BAAC998B8C
8C929899979595989C9E9B9183756D717A8382725A474661
8FBEE1ECE5D6C8C0BEBEC2C8CFD6D9D6CEC4B9B1ACA6A099
928C86807C7D8596ABC1D2DADBD7CEC1B2A5A0A6B2BFC3B7
A38F8793ACC6D6D3C2B0A6ACBCCCD1C6B19A8C8B95A1ACB1
B1ADA69C90857B767473716D655D575453535049403D4660
83A6BCBEB09D8D88
7A7C7E7C79777B84909BA09C91847B787B7D7B6F60565B73
98BED9E3DFD5CBC6C6C7CBCFD4D7D6CEC1B3A69D98959290
8F8D8B8784848892A1B0BFCBD3D6D3C6B6A6A0A8B9CBD3CB
B9A49898A4B1BBBBB5AFAEB6C3CECFC2AC96898A95A4AEB0
ACA49D98938F8B87827D766B5F534C4A4E54585856576074
8DA5B2B0A290817A
6866645F5A595F6F8498A4A69E92867E7976726C67677085
A0BBCFD7D7D2CDCCCDCFD3D6D9D9D4C7B5A2928782808285
8A8F92918F8D8D9197A0ACBAC8D3D5CBBAAAA3ABBED2DDDA
CCBAAAA09C9C9EA2A8AFB8C2CBD0CCBDA894898A96A4AEAE
A69D97959799999690887C6C5C4E45464D59636A6F737B87
949EA29D9081746C
5E5750474140485D7893A5AAA497897D746E696769707B8C
A0B3C1CACECFCFCFD1D3D6DADDDDD6C6AF9884777272767E
868F9597969492919397A0AFC0CFD5CDBEAFA8AEBECFDADC
D4C7B8A89A908D93A0B1C0CBD0D1C9BBA8978C8C95A1A9A9
A39C97979B9E9E9C968D817160514949525E6B7781899096
99999690867B7067
61524438302F39506E8BA0A59E9181746B64605F62697584
95A5B3BEC6CCCECFCFCED1D8E0E5DFCDB2967F726F71777D
848A909395959596979AA2AFBECCD2CDC1B5ADAEB6C0C8CC
CBC7BEB0A0948F94A2B3C3CDD0CFC8BCAD9D938F93999FA3
A4A4A2A09D9995928F8B83786C6059575B626D7A88949DA0
9E9A95918D867D70
6C543F2F272631486683969A908070655F5C5957575B6472
8395A6B4C0C8CCCBC8C5C7D2E0EBEBD9BC9C8377767B8083
8485878B8F94999CA0A5ACB6C0C9CCCAC2B9B1ACAAA9ACB0
B6BABAB4ABA29EA1AAB6C0C8CCCDC8BFB3A79B938E8E929C
A7B1B4ADA0908480818485837E78726C68676B768696A2A6
A49F9B9C9E9D9584
7A5A3E2B22232E45617C8D8C7F6C5C54545657534F4E5360
73889BACBAC4C8C5BFBABCCADDEEF2E3C6A68D8283888C8B
868181848C949DA5ABB2B7BDC0C3C3C2C0BCB4A99C928E92
9CA7B1B4B4B2B0B0B3B6BBC1C6C9C9C3BAAFA3958A838796
ACBFC7BCA4897570757F899092928E84796F6C748496A4A9
A8A4A4AAB2B5AE98
835C3B251C1D2A415D7785816E56454249535A58514C4E5A
6C8094A5B2BCC0BDB6B0B2C0D5E8EFE4CCB09A9191949692
8B858488919BA5AEB4B9BCBDBBB8B7B8BBBCB6A79482787A
8593A2ACB3B8BAB8B6B3B4B9C0C7C9C6BEB3A695867D8093
B0CAD5C8AB8B736D7684939FA7AAA79C8D7D757A889AA8AD
ABA8A9B4C1C8C1A8
835630180F12213B5B77847A60432F3040556467625C5C64
7383929FA9B1B4B3AEAAACB7C7D6DDD8CAB8A9A09D9C9B98
9493959BA3ACB3B7B8B8B5B1ADA8A8ACB3B8B5A894807270
76818E9BA5AEB3B3B0ACACB3BDC5CAC6BDB1A292847C8095
B2CEDBD1B79A847F8896A6B1B9BDBAB0A1928A8C98A6B0B2
ADA8AAB6C6D0CAAE
7D4D240B030618375B7B887A593620243D5B747E7D79777B
848D969CA1A5A8A8A8A8AAAFB6BDC2C4C2BDB7AFA69F9B9B
9FA6AEB5BCC0C1BDB6AEA59D9895979EA8AFB0A898877871
70737A838E99A1A3A3A2A5AEBAC5C9C4B9AB9C8F8581889B
B4CCDAD6C5B1A29FA7B2BCC4C7C9C6BFB6ACA6A7AEB7BBB7
AEA6A6B3C4CFC9AB
77482008000418396183917F5A321B22416787989B999697
9A9C9D9D9C9C9EA1A5A9ABAAA7A5A6ADB7BFC1BAADA19B9F
AAB8C5CDD1D0CABEAF9F91878180848D98A1A6A49B8F8276
6D66656B74818B9194969DA9B8C4C9C2B5A5968D8A8B94A4
B7CAD6D8D1C8C2C2C6CDD2D3D2D1CECBC7C4C2C2C5C7C4BC
AFA5A3ADBEC8C2A5
774F2E19121628496F909D8C663F2932527A9CAFB5B4B2B1
B0ADA9A49F9C9CA0A7ADAEA89E94929BAABAC2BDB0A39DA3
B2C3D1D8D7D2C6B6A28E7D716B6B7078828C94989792887A
6B5D5658616F7B83878C95A3B4C2C7C1B3A39792949AA4AF
BCC8D2D8DAD9D9DBDDE0E0DDD9D5D2D2D4D5D6D5D3CFC8BD
AFA4A0A8B4BBB69C
7E67534640424F6988A2AD9E80614F577293B0C0C6C6C5C3
C1BDB8B1ABA6A5A9AFB4B3AB9D908B919FAEB7B5ADA4A1A8
B5C2CCCECAC0B3A28F7D6C605957595E666F7881888B8679
68574D4F5A69767E81858D9CAEBEC5C2B7AAA19FA4ADB5BD
C4CBD1D7DBDFE2E3E5E5E3DFDAD6D4D6D9DDDEDBD5CDC4B9
AFA6A1A3A7A9A494
8C8A88837F7E8392A6B7BDB5A28E84899AAFC1CCCFD0CFCE
CDCBC7C2BCB8B6B7BABCBAB1A4978F90969EA5A7A7A6A7AC
B3B9BAB6ADA19386786B5F534B454344484E5764717B7D73
63534A4D5A6B787F80818895A8B9C3C4BEB6B1B2B8C0C7CB
CDCFD1D5D8DCDEDFDFDFDEDBD8D5D5D6DADCDCD7CEC4BBB3
AEA9A49F9A96928E
9CAFBCC1BEB9B7BBC3CACDCAC3BDB9BBC1C9CFD3D4D4D4D5
D5D5D4D2CECBC9C8C7C5C1B9AFA49B9491919499A0A7ADAF
AEAAA2988B7E7268615A52493F37302D2D3039485969706B
5E50494E5C6D7B8181828793A4B5C2C6C5C3C2C6CCD2D6D7
D6D4D3D3D4D5D6D6D6D6D5D5D4D3D4D5D7D8D5CFC5BAB2AE
AEADA99E9186838C
ACCBE3ECEBE4DDD8D7D7D8DADCDEE0E0DEDBD8D6D6D6D7D8
DADCDDDDDBD9D7D4D1CDC8C2BBB3AA9F948C8B929EAAB1B0
A79A8B7B6C5F554F4D4B4740372D241E1B1C23324456605F
574D484D5968757D81858D99A8B7C2C9CCCED0D4D9DDDFDE
DBD7D3D1D0CFCFCECECECECFD0D1D2D3D4D3D0C9BFB5AEAE
B0B3AEA18F817F90
BAD9F0FAF9F2E9E3DFDCDCDFE4EAEDECE7E1DBD7D6D6D7D9
DBDDDEDFDFDEDCD9D6D1CDCAC6C2BAAEA1959197A4AFB5AF
9F8C7866584C444140403F393128201A16161C2736454F51
4E4946474D56626F7E8D9CA9B5BEC6CCD0D4D7DBDEE0E1DF
DCD7D4D1CFCECDCDCCCCCDCED0D1D2D3D3D2CEC8C0B8B3B3
B6B9B6A9998C8B9D
C4DAEAF1EFE9E3DEDCDADADDE0E4E6E5E2DDD9D7D6D6D7D7
D8D9DADBDCDBDAD9D6D3D1D0D0CFCAC0B3A8A2A6AFB7B7AC
98816B5A4D433D3A393938342E28221E1C1C20262F383E42
4443413E3D3F4A5F7B98B0BFC7CACCCFD2D6D8DADBDCDCDB
D8D6D3D2D1D1D0D0D0CFD0D0D1D2D2D3D3D3D1CCC7C1BEBD
BFC0BDB5AAA2A2AF
CCD6DCDEDBD8D4D3D4D5D6D6D6D7D7D6D6D5D5D5D5D5D5D4
D4D4D4D5D5D6D6D5D4D3D3D5D7D9D7D0C7BDB8B8BCBEB9A9
9279635247403B37353433302D2A27262627282A2B2D3034
393D3D37302E38547CA4C4D5D8D5D1D1D3D5D6D6D5D5D5D4
D4D4D4D3D4D4D4D4D4D4D4D4D3D3D3D4D4D4D3D2CFCCCAC8
C8C7C5C2BEBBBDC3
D2D1CFCDCBC9C8CACDD0D2D1CECBC9CACCCFD1D3D3D3D2D2
D1D0CFCFD0D1D1D2D2D2D4D7DBDEDFDCD6CFCAC8C7C4BAA7
8E745E4E433D3834302E2D2C2C2C2D2E2F302F2D2A27272A
2F3537322C2B395985B2D4E4E4DDD6D3D3D4D4D3D0CFCFCF
D1D2D4D5D6D6D6D7D7D7D7D6D5D5D4D4D5D5D5D5D5D4D3D1
CFCDCCCCCED0D2D2
D5D0CBC8C6C6C6C9CCCFD1CFCCC8C6C7CACDD0D2D2D2D2D1
D0CFCECECFCFD0D1D2D3D5D8DBDEE0DFDCD9D5D2CEC7BAA5
8C735C4B4039332D292625272A2D30313231302E2B282626
272A2D30353F537399BFDAE6E6DED7D3D3D4D4D2CFCECDCE
D0D2D4D5D6D6D7D7D7D7D7D6D6D5D5D5D5D5D6D6D7D8D7D5
D3D1D0D3D6D9DBD9
D5D2CFCDCCCCCCCECFD1D2D1CFCDCCCCCED0D1D2D3D2D2D2
D1D1D0D0D1D1D2D2D3D4D5D6D8D9DBDBDCDAD8D6D0C8BAA5
8D745D4B3D332C25201D1D21272D3031302D2B2B2D2E2C27
2220253348627E9AB4CAD9E0DFDBD6D3D3D4D4D3D1D0D0D0
D1D3D4D5D5D5D5D5D5D6D6D5D5D5D4D4D4D4D5D5D6D7D7D6
D4D3D3D5D7D9DAD9
D5D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D3D3D3D3D3
D3D4D4D4D3D3D3D4D4D5D5D4D3D3D3D5D7D8D8D7D2C9BBA8
9178604C3C2F251E19181A1F252B2F2F2C2826292E34332B
201A223B618AAEC4CFD3D5D5D6D5D4D3D3D4D4D4D4D3D3D3
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D5D5D5
D5D4D4D5D5D5D5D5
D3D5D7D9D9D9D9D8D7D6D5D6D7D8D9D9D8D6D5D5D4D4D5D5
D5D6D6D6D6D5D5D5D5D5D4D2D0CECECFD2D4D6D6D4CEC2B0
997F664F3B2B211B1A1B1F23272A2B2B282524282E35352E
25212D4F7EAED3E3E2D9D0CDCED1D3D3D3D4D4D5D6D6D6D5
D5D4D4D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3
D4D5D5D4D2D1D1D2
D3D5D7D9DADADAD9D7D6D5D6D7D9DAD9D8D7D5D5D5D5D5D5
D6D6D6D6D6D5D5D5D5D4D3D2D0CFCECECFD0D3D6D9D7CEBD
A5896D523B291F1D222A30312D2925242628292A2B2D2E2E
2F374C6F9CC6E4EEE8DACECACCD0D3D4D4D3D4D5D6D7D7D6
D5D4D4D3D3D3D3D3D3D3D3D3D3D3D3D3D4D3D3D3D3D2D2D3
D4D5D5D3D2D0D0D1
D3D4D5D6D7D7D7D6D6D5D5D5D6D6D7D7D6D5D5D4D4D4D5D5
D5D5D5D5D5D5D5D4D4D3D3D2D3D3D2D0CECDCFD6DEE2DCCB
B19272543B2820243140484438291F1F262E312D2620212B
3E567394B6D3E6EAE4D9CFCBCDD1D4D5D4D3D3D4D5D5D5D5
D5D4D4D3D3D3D3D3D3D3D3D3D3D4D4D4D4D4D4D3D3D3D3D4
D4D4D4D4D3D2D2D2
D4D4D3D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D3D2D2D3D6D8D7D4CECACCD6E2EAE8D6
B99774543A28232D40555F57432B1B1B2733393021151629
4A7195B3C8D7DFE0DDD6D0CED0D3D6D6D5D4D3D3D3D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D3D3D2D2D2D2D2D3D3D3D3D3D2D2D2D2D3D3D3D4D4D3D3
D3D3D3D3D3D4D4D3D3D2D2D4D8DBDBD6CFC9CAD5E3EDEBD7
B6916C4D362827354C636E654D311E1D28363B311F111229
507BA1BCCCD5D9D9D7D3D0D0D1D5D7D7D6D4D3D2D2D3D3D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D5D4
D4D3D3D4D4D5D5D5
D4D3D3D2D2D2D2D2D3D3D3D3D3D2D2D2D2D3D3D3D3D3D3D3
D3D3D3D3D3D4D4D4D3D2D2D4D8DBDBD7D0CACBD4E1E9E3CB
A67E593F2F282D3C52667068543B29252B34362F22191B2C
4A6B8CA8BECFD9DBD9D3CFCED1D5D8D8D6D4D2D2D2D3D3D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D4D5D5D5
D4D4D3D3D3D3D3D3D3D3D4D3D3D3D3D3D3D3D4D4D4D4D4D4
D3D3D3D3D4D4D4D4D3D3D3D4D6D8D8D6D1CDCED4DBDDD1B4
8C63412D272B35435361686458493B33302F2E2B292A2D32
3B49607FA3C4DBE3DED5CDCBCFD4D8D8D6D4D2D2D3D3D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D2D2D3D2CCB998
6E45281E23303F4B53595C5D5C574E42352A2528313C3F37
2B242E5081B4D9E7E2D6CCCACED4D8D8D6D4D2D2D3D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D3D3D4D5D5D5D3D1D0D1D4D7D9D7D2C8B79D7A
522F1A18263B4D55555251565E6360523D291F2637484C3C
200A0A2C639FCDE1E1D7CECCD0D4D6D6D5D3D3D3D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D3D3D3
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D3D3D4D5D7D6D3CFCDCED4DCE0DDD0BBA0805F
3F261B21344B5C5F594F4C53616D6F60462D1F25384A4F3E
2007031E5087B5CED7D7D4D3D4D4D3D3D3D3D4D5D5D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D3D3
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D3D2D2D4D6D8D8D4CECACBD4E0E7E2CEAD886448
34292835495E6C6B60534C54647377695035252736464B3E
28151022456E95B2C7D4DCDDDAD5D0CED0D3D6D6D5D4D3D3
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D3D2D2D4D6D9D9D4CEC8CAD5E3ECE5C89E714B35
2D303B4C60727C786B5B525867757A6E573E2E2C353F443E
3227252D3F577495B5D1E3E7E1D5CCCACED3D7D8D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D3D2D2D3D6D9D9D5CEC9CAD5E3EBE1C08F5E3726
2733465C70828B8779695E5F6771736A58463834373B3E3B
3733313238435A7DA6CDE7EEE5D6CAC7CCD3D9D9D7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D3D3D3D3D5D7D7D5D0CCCDD6E0E4D7B382502B1B
1F2E455E768A97978D7E71696766635C534A433F3E3D3A36
322F2C2A2C344A6F9DC9E6EEE6D6C9C7CCD3D9DAD7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D3D3D3D4D5D5D4D2D0D1D6DBD9C8A476472515
17263D58748D9FA5A196877666584E4A4A4B4D4C48423A31
28211C1B1F2A436A98C4E2EBE4D6CBC8CDD3D8D9D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D3D3D3D3D4D5D5D6D7D5CDB9966B412213
15233954718CA1AEB2AC9C84674C3B38414D5759544A3D2F
21160F0E1525416895BFDDE7E2D6CCCACED4D8D8D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D5D5D5D3D2D1D2D4D7D8D9D7D2C6AF8B623B2017
1F314860768A9DAEB9BBAE906A462F2D3C50616661564736
271A110F15243F6693BDDAE5E1D6CDCBCED4D8D8D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D5D5D5D3D2D1D2D4D7D8D9D7D2C5AD875B341E20
37567282878A91A1B3BEB79B71482E2D3F566A726E64574A
3E3329211F273C6290BCDCE7E2D6CCCACED4D8D8D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D4D5D5D3D2D2D3D5D6D7D7D7D4CAB288572D1B2C
5686A8AE9F8A808AA1B5B8A179523836475D717978726B65
5F584D3F312C395C8CBDE0ECE5D7CAC8CDD4D9D9D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D5D4D3D3D3D3D5D5D5D5D6D6CFB78B55281937
72B1D8D4B2886D718AA5B1A2826049465163727B7C7B7B7C
7C7A6F5A423338598BC0E6F1E8D7C8C5CCD4D9DAD7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D3D3D3D4D5D5D4D4D6D7D1BA8C5425183C
7FC4EDE3B57F5A5A7492A5A08A725F595D656D73777A7E83
88887E674D393B5B8EC4EAF5EAD6C7C4CBD3DADAD7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D4D4D4D5D4D3D2D2D3D5D6D5D5D6D6CEB68953261939
76B5D9CFA26E4A49617F969B9285786E68646263666A7075
787871604C3F456596C8EBF4E9D6C7C4CBD3DADAD7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D4D5D5D3D2D1D2D5D7D8D8D7D2C7AE85542B1C31
5F8FABA2815A4040546E869499989183725F524C4E525658
5755514943435273A0CCEAF1E7D6C8C5CBD3D9DAD7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D5D6D5D3D1D0D1D4D9DBDBD7CEBFA5815936242D
466375715E493B3D4B5E74899BA7A69578594035353A3D3A
34302E3139475F82AACFE8EDE4D6CAC7CCD3D8D9D7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D3D3D3D4D5D6D6D3D0CECFD4DBDFDED7C9B79E8163493834
3A434B4B46423F40444D5E7895ABB09C774E2D2022292D28
1F181721334B698DB1D1E6EAE3D6CBC8CDD3D8D9D7D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D3D3D3D4D5D7D6D4D0CDCED4DCE2E1D7C6B09A867566584C
423B383B41484B47403D466083A1AB966C3E1C111927302C
21181721354E6D90B3D1E5E9E2D5CBC9CDD3D8D9D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D3D3D3D4D5D7D6D4D0CDCDD4DDE3E2D7C3AC9A908B888070
5B473D404C595D53413131486D8F9C885D2E0F0B1B314041
372D292F3E536F8FB2D0E4E9E2D5CBC9CDD3D8D9D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D3D3D3D4D5D7D7D4D0CCCDD3DDE4E3D6C2AC9D9BA1A7A593
785E4F5160707464482E26395D808D7B52260B0E2542575C
534942434C5B718EB0CFE3E9E2D5CCC9CDD3D8D8D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D3D3D3D4D5D6D6D4D0CDCED3DCE2E1D6C3B0A5A7B0BABBAC
947B6C6E7B898A77583A2E3E5F7F8D7C5730191F39586F75
6E655D5A5E677892B1CEE2E7E1D5CCCACED3D7D8D6D4D2D2
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D3D3D3D4D5D6D6D4D1CFCFD4D9DEDDD5C8BBB2B1B6BBBCB4
A69890929BA29F8C71584F5D7A97A39473503B3F546E8287
8179726F727A889DB6CFE0E4DFD5CDCCCFD3D7D7D6D4D3D3
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D4D5D5D4D2D2D2D4D6D8D8D5CFC8C1BAB4B0AEAE
B0B3B6B8B9B7B0A190817F8EA6BDC6B99C7D696773849092
8C86828288909CADC0D1DDE0DCD5CFCED0D3D6D6D5D4D3D3
D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D3D3D5D6D6D0C2B0A19CA4
B6C9D6D8D1C7BCB3AEACB1BFD2E2E8DCC5AB978F90969A98
949190959DA6B2BECAD4DADBD9D4D1D0D1D4D5D5D5D4D3D3
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D3D3D3D4D5D6D6D4D2CFCFD4DBDFDAC9B19B939E
B6D0E3E5DCCFC4C1C5CBD5E1ECF5F6EDDDC9B7AAA3A09D9A
98999DA5AFBAC4CCD2D5D7D7D6D4D2D2D3D4D4D5D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D3D3D3D4D5D6D6D4D1CECFD4DCE0DDCEB9A49AA0
B2C6D4D7D2CBC7C9D1DAE2E7EAEAE7E1D7CCBFB2A79E9897
999EA7B1BDC7CFD4D6D6D5D4D4D4D4D3D3D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D3D3D4D5D5D5D4D2D0D1D4D9DCDAD2C6B9AFAB
ABAFB4B8BCC0C6CDD4D9DBD8D2CBC4C0BEBBB5ABA0969092
99A3AFBBC6CFD6D8D8D6D3D2D3D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D3D3D3D4D5D5D5D5D3CFC6B7
A6979196A3B3C2CDD1D2CDC3B6A89F9DA0A5A6A0968D898E
99A7B5C2CCD3D8D9D8D5D2D1D2D4D5D5D5D4D3D3D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D3D3D3D4D5D5D4D2D0D1D6DCDFD7C2
A68C7E8295ACC0CBCECBC1B3A29287878E979D9A938C8A91
9EADBCC7CFD5D8D9D7D4D1D1D2D4D5D5D5D4D3D3D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D3D3D3D3D5D6D6D4D1CED0D6DEE3DECB
B096888A9AAEBFC9CCC9C0B3A3938A8A919BA3A39E9A99A0
ABB8C4CCD2D6D7D7D6D4D2D2D2D4D5D5D4D4D3D3D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D3D3D3D3D3D4D4D4D4D4D3D3D4D4D4D4
D4D4D4D4D4D4D4D4D3D3D3D4D4D5D5D4D2D0D1D5DBDEDBD0
C0B1A7A7AEB8C1C8CBCBC7BEB2A79FA0A6ADB4B5B3B1B2B6
BEC6CDD2D4D5D6D5D5D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D5D5D5D5D5D4D4D3D3D2D2D3D3D4D4D4D3D3D3D3D4D4D4
D4D4D4D3D3D3D3D3D4D4D4D4D4D4D4D4D3D3D3D4D5D6D5D4
D1CECBC8C6C4C5C7CBCFD0CCC6BFBBBBBEC3C7C9C9CACBCD
D1D4D5D6D5D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D5D5D5D5D4D4D3D3D2D3D3D4D4D4D4D3D3D3D3D4D4D4
D4D4D4D4D3D3D3D3D4D4D4D4D4D3D3D4D4D5D5D3D1CFD1D6
DDE4E6E1D8CFC9C9CDD2D6D7D5D3D1D1D2D3D5D7DADCDDDE
DEDCDBD8D5D3D2D2D2D4D5D5D5D4D3D3D3D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D3D3D4D5D6D5D3D0CED0D7
E0E9ECE8DFD5CDCCCFD3D8DADBDBDBDBDADADADCDFE1E3E2
E1DEDBD8D5D3D1D1D2D4D5D5D5D4D3D3D3D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4
D3D3D3D3D2D2D3D4D5D6D6D6D5D4D4D4D4D4D5D5D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D3D3D3D4D4D5D5D3D2D0D2D6
DBE0E3E1DCD6D2D0D1D3D6D7D8D9DADAD9D8D8D9DBDDDEDD
DCDAD8D6D4D3D2D2D3D4D4D5D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D3D3
D3D2D2D1D1D1D2D4D6D7D8D7D6D5D3D3D4D5D6D5D5D4D3D3
D4D4D4D5D5D5D5D5D4D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D5D5D5D5D5D4D4D3D3D4D5D5D5D5D4D4D4D5D5D5D5
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D3D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D3
D2D1D0D0D1D2D3D4D6D7D7D7D6D4D4D4D4D5D6D5D5D4D4D3
D4D4D4D5D5D5D5D5D4D4D3D3D3D4D4D4D4D3D3D4D5D6D5D2
CECBCACCD0D4D7D8D6D4D2D1D1D1D1D1D1D1D1D0CFCECECE
CFD0D1D2D3D5D5D5D4D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D3
D2D0CFD0D1D3D5D5D5D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D3D3D3D4D5D6D5D2
CECAC9CACFD3D7D8D6D4D2D1D1D1D1D1D1D1D1D0CFCDCDCD
CECFD1D2D3D4D5D5D4D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D3D3D4D5D5D5D3
D1CECDCFD2D6D7D6D3D0CECFD1D3D5D5D4D2D2D2D3D4D4D5
D4D4D3D3D3D2D2D3D3D4D4D4D4D4D4D4D4D3D3D4D5D5D5D3
D1CFCECFD1D3D4D5D5D4D4D3D3D3D3D3D3D3D3D2D1D0D0D0
D1D2D2D3D3D4D4D4D4D4D4D4D4D4D4D4D3D3D3D3D4D4D4D4
D4D3D3D3D3D4D4D4D4D4D4D4D4D4D4D4D4D3D3D4D4D4D5D4
D4D3D3D4D5D6D6D4
D0CCCBCED2D7D9D7D1CCC9CACED3D6D6D4D1D0D0D2D4D5D5
D5D4D3D2D1D1D1D1D2D4D5D5D5D5D4D4D4D4D4D4D4D4D5D5
D5D6D5D4D3D2D1D2D3D4D5D5D5D5D5D5D5D4D4D4D4D4D4D4
D5D5D5D4D4D3D3D3D3D4D4D4D4D4D3D3D3D3D3D3D4D4D5D4
D4D3D3D3D3D3D4D4D4D3D3D3D4D4D4D4D4D3D3D3D4D5D5D4
D3D2D2D3D5D6D6D4
CFCAC8CBD0D6D8D6D0C9C6C7CCD2D6D6D4D1D0D0D2D4D5D5
D5D4D3D2D1D1D1D2D3D4D4D5D5D4D4D4D4D4D4D4D3D4D4D6
D8D9D9D8D5D2D0D0D2D3D5D6D6D6D6D5D5D5D5D5D6D7D7D7
D7D7D6D5D4D3D2D3D3D4D4D4D4D4D3D3D3D3D3D3D4D4D5D4
D4D3D3D3D3D3D4D4D4D3D3D3D4D4D4D4D4D3D3D3D4D5D5D4
D3D2D2D3D6D7D7D4
CEC7C4C6CBD1D4D3CEC9C6C7CCD1D5D6D5D4D3D2D3D4D4D4
D4D4D3D3D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D3D3D4D5
D6D8D8D8D6D4D2D2D3D3D4D5D5D5D5D5D5D5D5D5D6D6D7D7
D6D6D5D4D4D4D3D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D3D3D4D4D5D4D4
D2D1D2D4D7D9D9D4
CBC3BEBEC3CACECECCC9C8C9CCD0D4D6D6D7D6D6D5D4D4D3
D3D4D4D5D6D7D7D7D5D4D3D2D2D3D3D4D4D4D4D4D3D3D3D3
D3D3D4D5D6D6D7D6D5D4D3D2D3D3D3D3D4D4D4D4D4D5D4D4
D4D3D3D3D4D5D5D5D4D4D3D3D3D4D4D5D5D5D5D5D4D3D3D3
D4D5D5D5D5D4D4D4D4D4D4D4D4D4D3D4D4D4D4D4D4D4D4D3
D2D1D2D4D8DBDAD4
C7BCB5B5BAC1C6C9C9C8C8CACCCFD2D5D7D9D9D8D7D5D3D2
D3D4D5D7D9DADAD9D7D4D2D1D1D2D3D4D4D5D5D4D4D3D2D0
D0CFD0D3D6D9DADAD7D4D2D1D1D1D2D2D3D3D3D3D3D3D2D2
D1D1D1D2D4D5D6D6D5D4D3D3D3D4D5D5D6D6D6D5D4D2D2D2
D4D5D6D6D6D5D4D4D4D5D5D5D4D3D3D3D4D4D4D4D4D4D3D2
D1D1D2D5D9DBD9D2
BEB1A9A9AFB7BFC3C4C5C6C8CBCED1D4D6D8D8D7D5D3D2D2
D2D4D5D7D8D8D8D7D6D4D3D2D2D2D3D4D4D5D5D5D4D3D2D0
CFCECFD2D5D9DBDAD8D5D2D1D1D2D2D3D3D3D3D3D3D2D2D1
D1D1D1D2D4D5D6D6D5D4D3D3D3D4D5D5D6D6D6D5D4D3D2D3
D4D5D6D6D5D5D4D4D4D5D5D4D4D3D3D3D4D4D4D4D4D4D3D3
D2D2D3D5D8D9D5CB
B1A29A9CA4AFB8BCBDBDBEC2C8CED2D4D3D2D1D0CFCFD0D1
D2D4D5D4D3D1D1D1D2D3D4D5D5D4D4D4D4D4D4D4D4D4D4D3
D3D2D2D3D4D5D5D5D5D4D4D4D4D4D4D4D4D3D3D3D3D3D3D3
D3D3D3D3D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4
D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D5D4D4D4D4D4
D4D4D4D5D5D3CCC0
A091898D98A6B1B5B3B1B2B8C2CCD2D2CEC9C5C5C7CACDCF
D1D3D2CFCBC7C5C7CCD2D7D8D8D6D5D4D3D3D3D3D4D5D7D8
D9D9D8D5D2CFCDCED0D3D6D7D7D7D7D6D5D4D4D4D4D4D5D6
D6D7D6D5D4D2D1D2D3D5D6D6D5D4D3D2D2D1D2D3D4D5D6D5
D4D2D1D1D2D3D4D4D3D3D2D3D3D4D5D5D5D5D4D4D4D5D5D6
D6D6D5D4D1CBC1B2
907F767C8B9DAAACA8A2A1AAB8C7CFCEC6BDB8B8BDC3C9CD
D0D0CEC8C1BBB9BEC7D1D9DCDBD8D4D2D1D1D1D2D4D6D9DB
DDDEDCD6CFC9C5C7CCD2D7DADAD8D7D6D6D5D5D5D5D5D6D7
D9DAD9D7D3D0CECFD2D5D7D7D6D4D2D0CFCECFD1D4D6D8D6
D4D1CFCFD0D2D3D3D3D2D2D3D4D5D6D6D5D4D4D4D4D5D7D8
D9D8D6D3CEC5B8A5
836C62697C92A2A39A908D97A9BBC5C4BBB1ABADB5BEC6CB
CDCCC9C1B9B2B1B7C3D0DADDDBD7D2D0CFD0D1D2D3D5D7DA
DCDDDAD4CCC5C1C4CBD2D8DAD8D5D3D2D4D5D6D6D6D5D6D7
D9DADAD7D2CECCCDD0D3D5D6D4D2CFCDCCCBCCCFD2D6D7D6
D4D0CECDCFD0D2D2D2D2D2D3D5D6D6D5D3D2D1D1D3D5D7D9
DAD9D7D4CEC5B59E
7B5C4C536B8698988C7D777F93A7B4B5ADA4A0A5B0BCC5CA
CAC7C3BCB4AFAFB7C3D1DADCD9D2CDCCCDCFD1D1D1D0D1D1
D3D3D2CEC9C4C3C7CED4D8D6D1CBC8CACED4D7D8D7D5D4D4
D6D7D7D4D0CCCACACCCFD0D0CFCECCCBCACACBCDD0D3D5D4
D3D1CFCECECFD0D1D2D3D4D5D6D6D5D2CFCDCBCCCFD3D6D8
D9D8D7D6D4CDBB9E
774E383E59798E8D7C695F66798E9DA09C97979FACB9C3C6
C4C0BAB5B1AEB1B9C5D1D8D8D2CBC5C5C9CDD0D0CDC9C6C4
C3C3C3C4C5C6C9CDD2D6D5CFC6BEBABEC7D1D8DAD7D2CFCE
D0D1D2D0CDCAC8C8C8C8C9C8C8C8C8C8C8C8C9CBCCCED0D1
D1D1D1D0CFCECECFD2D4D6D7D7D6D3CFCAC6C4C5C9CFD3D6
D6D5D5D9DBD8C7A4
7543272B486B81806D56494D5F73838989898C97A6B3BCBD
BAB4AFACABADB2BAC4CDD2CFC8C0BBBCC2C8CCCBC6C0BAB6
B3B2B3B8BFC6CDD1D4D4CFC5B9AFABB2BFCED8DAD5CEC8C6
C7C9CACAC9C7C6C4C3C1C0C0C1C2C3C4C6C7C8C8C8C8C9CC
CFD2D4D2CFCBCACCD0D4D7D8D7D5D0CBC5C0BDBFC3C9CFD1
D1CFD2D9E2E3D2AA
743D1C1E3A5C72715D463739485A697175787F8B99A5ACAC
A7A19C9CA0A5ADB5BDC4C6C2BBB3AEB0B6BCC1C2BFBAB4AE
A8A5A6ADB7C2CBCFCFCCC4B8ACA2A0A9BACCD8D8D1C6BDBA
BCBFC1C3C3C3C3C1BFBCBBBABBBDBFC1C4C6C7C6C3C1C2C7
CED4D7D4CEC8C4C6CCD2D7D8D7D4D0CAC5C0BCBDC0C5C9CA
C9C8CBD6E2E7D7AD
723B19182E4C605F4E3A2C2C3745525B60666E78848D9190
8A8583858C959FA7AEB3B4B1ABA5A1A1A5AAAFB4B8B9B7B1
A8A09EA3ADB9C1C4C2BDB5ABA09A9BA7B9CCD7D5CABBB0AC
AEB3B8BBBDBEBFBEBDBBB9B8B9B9BBBFC2C5C6C3BEBABAC1
CCD6DBD7CDC2BCBEC5CDD5D8D7D5D1CECAC6C3C1C0C1C2C1
BFBEC2CEDBE0D2AA
6E3D1C17273D4C4C403228262D353E464D535B62696E706D
6965656B74808A93989C9D9B99969391909298A3AFBABEB9
ADA0999AA1AAB0B1AEA9A39C979499A7BACBD4CFBFAEA09C
9FA6ADB2B5B8B9BABBBAB9B8B7B7B8BBBFC3C3BFB8B2B3BB
C9D7DCD7C9BBB2B3BBC6CFD4D4D3D2D2D2D0CCC7C1BDB9B5
B3B2B6C1CCD1C4A0
6940221A22303B3C362E2826272B2F353C43494D4F504E4B
484749515C69747C80828385868684807B797F8FA4B7C2BE
B1A0938F92979A9995928F8D8D8F97A6B7C6CCC4B29F918D
9199A2A7ABADB0B2B5B6B7B6B4B3B4B6BABEBEB9B1AAAAB4
C3D1D7D1C2B2A7A7AFBAC4CACCCDCFD2D4D5D1C9BEB4ABA6
A3A3A8B0B9BCB193
6241291F21282F32312E2B292625252930363B3C3A373431
3032363F4A566066696A6B6D7174736D6661667790A7B5B4
A79687807F80817F7C7B7B7E8289929FAEB8BBB2A2918481
868E969B9C9EA0A3A7AAACADABAAABADB2B5B5B0A9A2A1A9
B6C1C6C1B4A69C9A9FA7AFB6BBBFC4C9CDCEC9BFB1A49993
9294989EA3A49A82
57402E2524272C303233312D28232123292F32322E292524
26292F37404A5155565555585B5E5E59524D505E72869394
8B7F736C696766656465696F767F88929BA1A19B90857D7C
80878B8C8A87878A8F94989A9B9B9CA0A4A7A8A5A09B999B
A1A6A8A59F97918D8B8D90979FA8AFB5B7B6B0A5978A807D
7F83888C8D8B816F
4C3D312B2A2C2F33373938332C252122272C2F2D28232021
252B31383E43474847454445484A4946413E3F46525D6669
66615B56524F4D4D50545B636B737A7F828382807D7B797B
7E81817C766F6B6C71787E8285888A8E929597979694908C
8782808185878680766F6C737E8B969A99948C81756B6666
6C73787A7873695B
4138322F2F3134393D403F3A322A2424282C2F2C27222023
2A31383C3F40403F3D3A3938393A3A3836343435373A3E42
464848443F3A383B414A535B62686B6A686564676C72787B
7D7B776E63595352565D646A6F74787C7F8284888B8C877C
6C5E575C69757A7362524A505F6F7B7E7971675D554F4D52
5A636A6A665E544A
3733302F2F32363B4145464139302A292B2E302D29252428
2F373C3F3F3D3A383533323334363737363533312F2E3034
3B40413C332B292E3A48535A5D5E5B56504B4C535F6C767A
78736B61564C454344494F575E656B6E6F6F71767D817C6C
553F353C4F636D66533F3337455662645E564D453F3B3B41
4B555B5B564D453D
302C2928282A303740484B4740362F2D2E3132302C29292C
32383C3C3B383532303032363D43474744413F3F4144474B
4F504B403124202A3C4F5D625D554B423D3A3D4857677274
6F675D56514C48454241444C555F6666635F5E646D736F5D
432B1F273C52605B4A382B2C36424C4F4D49443D38333337
3F484D4E4A433C36
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 288},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181P00`40o`000?l0003oooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo003ooooo8Oooo`006Oooo`030000
oooooooo02;oool300000oooo`030000oooooooo023oool300000_ooo`030000oooooooo01ooool2
00001?ooo`030000oooooooo023oool00`000?ooooooo`02oooo00<0003oooooool06oooo`D00002
oooo00<0003oooooool00_ooo`030000oooooooo01Woool5000000Coool0000000000003oooo00<0
003oooooool03oooo`006?ooo`030000oooo000002?oool00`000?ooooooo`02oooo00<0003oool0
00008_ooo`060000oooooooo0000oooo00008?ooo`030000oooo000000;oool00`000?ooo`00000P
oooo00<0003oool000000_ooo`030000oooo000001koool01`000?ooooooooooo`000?ooo`000002
oooo00<0003oool000007?ooo`050000oooooooooooo000000Coool00`000?ooo`00000@oooo000H
oooo00<0003oool000009?ooo`070000oooooooooooo0000oooo0000023oool4000000Coool0003o
ool0000Poooo00<0003oool000000_ooo`030000oooo0000023oool00`000?ooo`000002oooo00<0
003oool000007_ooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`00000Loooo00<0
003oooooool00_ooo`070000oooooooooooo0000oooo0000013oool001Soool00`000?ooo`00000T
oooo00L0003oooooooooool0003oool000008?ooo`030000oooo000000;oool00`000?ooo`00000P
oooo0P0000?oool00`000?ooo`00000Qoooo00L0003oooooooooool0003oool000007_ooo`070000
oooooooooooo0000oooo000000;oool00`000?ooo`00000Loooo00<0003oooooool00_ooo`070000
oooooooooooo0000oooo0000013oool001Soool00`000?ooo`00000Soooo00<0003oool000000_oo
o`030000oooo0000027oool200000_ooo`030000oooo0000023oool00`000?ooooooo`02oooo00<0
003oool000008?ooo`030000oooo000000;oool00`000?ooo`00000Noooo00L0003oooooooooool0
003oool000000_ooo`030000oooo000001coool01`000?ooooooooooo`000?ooo`000002oooo00<0
003oool000004?ooo`006Oooo`030000oooooooo02?oool00`000?ooooooo`02oooo00<0003ooooo
ool08Oooo`050000oooooooooooo000002;oool200000oooo`030000oooooooo023oool00`000?oo
ooooo`02oooo00<0003oooooool07?ooo`800004oooo00<0003oooooool00_ooo`030000oooooooo
01[oool200001?ooo`030000oooooooo00;oool00`000?ooooooo`0?oooo003ooooo8Oooo`00oooo
ob7oool00?oooolQoooo003ooooo8Oooo`00ooooob7oool001Gooooo00002`00007oool001Goool0
0`000?ooooooo`02oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Koool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool200000Oooo`005Oooo`030000oooooooo00;oool00`000?ooooooo`0Uoooo00<0
003oooooool09Oooo`030000oooooooo02Goool00`000?ooooooo`0Uoooo00<0003oooooool09?oo
o`030000oooooooo02Goool00`000?ooooooo`0Boooo0@00007oool000ooool00`000?ooooooo`03
oooo00<0003oooooool0ooooo`Ooool100000Oooo`003_ooo`030000oooo000000Coool00`000?oo
ooooo`3ooooo1oooo`400001oooo000>oooo00<0003oool000001?ooo`80003ooooo1oooo`800001
oooo000>oooo00<0003oool000001?ooo`030000oooooooo00;oool2;Rh^0RLW9`8Q8B427QhN0QdM
7@8O7al29BDU0S0`<08m?Cd2B4Q80Tm?C`9=CDd2ATI60S`l?08d=3@4<C4a0S<c<`8d=3@2<S8b0S0`
<08^;Rh2;Bd]0Rl_;`8b<S82=CDe13Lg=`8e=CD2=3@d0S8b<P@`<302<C4a0SLg=`91@D42CTi>0UYJ
FP9QHF42H61P0UYJFP9DE5@2DU9B0UUIF@9TI6@2L71`0WQhN09kNg/2N7Qh0W1`L09SHf<2CTi>0SPh
>08V9RH28B4Q0Rh^;P96ATH2GemO16m_K`9QHF42Cdm?0Slo?`8e=CD00c4a<C8b<S8b<P02>CTi0TI6
AP9FEUH2IFEU0Vi^KP9]KFd2IFEU0UUIF@9ADE42Cdm?0U=CD`9GEeL2F5QH0UADE09<C4`2A4A40T=3
@`99BDT2E5AD0V1PH09VIVH2I6AT0UaLG09CDe<2Cdm?0U=CD`9LG5`2I6AT0V5QH@9ADE42>CTi0R8R
8P8F5QH271`L0Rl_;`93@d<4DE5A0TM7A`8j>SX2<30`0Rh^;P8b<S82>3Ph0Shn?P93@d<2AdM714U9
B@95ADD2?Shn0SPh>08d=3@2=CDe0STi>@8n?Sh2@T920TA4A093@d<2@4100S/k>`8e=CD5oooo0@00
007oool000koool00`000?ooo`000004oooo00<0003oooooool00_ooo`8^;Rh29bLW0R4Q8@8N7Qh2
7AdM0QlO7`8U9BD2<30`0Sdm?@98B4P2Cdm?0Te=C@96ATH2?3`l0S@d=0@a<C42<c<c0S@d=08b<S82
<30`0Rh^;P8];Bd2;bl_0S8b<P8e=CD4=cLg0SDe=@8d=3@2<S8b130`<08a<C42=cLg0T51@@9>CTh2
FUYJ0V5QH@9PH602FUYJ0UADE09BDU82FEUI0VATI09`L702N7Qh0W]kN`9hN7P2L71`0V=SH`9>CTh2
>3Ph0RHV9P8Q8B42;Rh^0TI6AP9OGel4Kfm_0V5QH@9?Cdl2?clo0SDe=@03<C4a<S8b<S8b008i>CT2
ATI60UIFEP9UIFD2KVi^0Ve]K@9UIFD2FEUI0U5AD@9?Cdl2De=C0UMGE`9HF5P2E5AD0Ta<C094A4@2
@d=30TU9B@9DE5@2H61P0VIVIP9TI6@2G5aL0U=CD`9?Cdl2De=C0UaLG09TI6@2HF5Q0U5AD@8i>CT2
8R8R0QHF5P8L71`2;bl_0T=3@`AADE42AdM70SXj>P8`<302;Rh^0S8b<P8h>3P2?Shn0T=3@`97AdL4
BDU90TE5A@8n?Sh2>3Ph0S@d=08e=CD2>CTi0Shn?P92@T82A4A40T=3@`90@402>c/k0SDe=@Goool1
00000Oooo`003oooo`030000oooooooo00?oool00`000?ooooooo`02oooo0S0`<08U9BD271`L0QHF
5P8D51@25QHF0QdM7@8Z:RX2>SXj0TU9B@9BDU82DE5A0TY:BP90@402=cLg138b<P8c<c<4=3@d0S<c
<`8b<S84<C4a138b<P8c<c<2<S8b13<c<`@d=3@2=SHf0STi>@92@T82D51@0V5QH@9aLG42NWYj0WQh
N09`L704J6QX0WEeM@::RXX2WYjN0Z^[Z`:][Jd2Y:BT0YFEU@9nOWh2Hf=S0TM7A`8b<S82;2`/0SXj
>P9FEUH2M7Ad0XB4Q09oOgl2JVYZ0U1@D08j>SX3;bl_0SDe=@8o?cl2C4a<0UYJFP9VIVH2K6a/0VQX
J09MGEd2D51@0TY:BP9?Cdl2Fe]K0VQXJ09_Kfl2JFUY0UaLG09?Cdl2B4Q80T];B`9GEeL2Hf=S0VQX
J09TI6@2F5QH0T];B`94A4@2ATI60Tm?C`AGEeL2BTY:0SHf=P8R8R825aLG0QXJ6P8W9bL2=cLg0T=3
@`97AdL2A4A40Slo?`8k>c/4>3Ph0STi>@8l?3`2@d=30T];B`9BDU82EEEE0U=CD`9=CDd2ADE50Shn
?P8k>c/2>SXj0S/k>`8m?Cd2@4100T92@P94A4@2@D510SXj>PGoool100000Oooo`005Oooo`030000
oooooooo00;oool2<30`0RDU9@8L71`25QHF0Q@D508F5QH27AdM0RXZ:P8j>SX2BDU90U9BDP9ADE42
BTY:0T10@08g=cL4<S8b0S<c<`@d=3@2<c<c0S8b<P@a<C44<S8b0S<c<`8b<S84<c<c13@d=08f=SH2
>CTi0T92@P9@D502HF5Q0W5aL@9jNWX2N7Qh0W1`L0AXJ6P2MGEe0XZ:RP:NWYh2Zj^[0Zf][@:TY:@2
UIFE0WinOP9SHf<2AdM70S8b<P8/;2`2>SXj0UIFEP9dM7@2Q8B40WmoO`9ZJVX2D51@0SXj>P<_;bl2
=CDe0Slo?`9<C4`2FUYJ0VIVIP9/K6`2J6QX0UeMG@9@D502BTY:0Tm?C`9KFe/2J6QX0Vm_K`9YJFT2
G5aL0Tm?C`98B4P2Bd];0UMGE`9SHf<2J6QX0VATI09HF5P2Bd];0TA4A096ATH2Cdm?15MGE`9:BTX2
=SHf0R8R8P8G5aL26QXJ0RLW9`8g=cL2@d=30TM7A`94A4@2?clo0S/k>`@h>3P2>CTi0S`l?093@d<2
Bd];0U9BDP9EEED2De=C0Te=C@95ADD2?Shn0S/k>`8j>SX2>c/k0Sdm?@90@402@T920TA4A091@D42
>SXj1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`8k>c/2;Bd]0R4Q8@8H61P251@D0QDE
5@8L71`2:2PX0STi>@98B4P4DU9B0T];B`91@D42>3Ph0S<c<`@a<C42<S8b1S4a<@H`<302;bl_0S0`
<08a<C42=CDe0STi>@8m?Cd2@D510TE5A@9:BTX2DU9B0UmOG`9^KVh2O7al0X>3P`:1PH42NGUi0W=c
L`9gMgL2R8R80Z6QX@:i^KT2bLW90/_;b`;1`L42[jn_0YNGU`9jNWX2GemO0TU9B@93@d<2DU9B0Vi^
KP:;Rh/2VYZJ0Y:BTP9hN7P2FEUI0T51@@8h>3P00c/k>dA4A4A4A002Cdm?0UYJFP9SHf<2Jf][0Ve]
K@9WIfL2G5aL0U5AD@9>CTh2EeMG0VUYJ@9kNg/2Q8B40WmoO`9_Kfl2GEeM155AD@9IFET2HV9R0VEU
I@9NGUh2DE5A0TA4A08m?Cd2?clo0TQ8B09@D502DU9B0TU9B@8j>SX2:RXZ0QlO7`8M7Ad28b<S0R/[
:`8c<c<2>SXj0Shn?P92@T82ADE50TM7A`99BDT2BTY:0Ti>CP9DE5@2Fe]K0V5QH@9UIFD2I6AT0UmO
G`9HF5P2DE5A0TY:BP95ADD2@D510T10@094A4@2BTY:0Tm?C`9>CTh2AdM71Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`8k>c/2;Bd]0R4Q8@8H61P251@D0QDE5@8L71`2:2PX0STi>@98B4P4
DU9B0T];B`91@D42>3Ph0S<c<`@a<C42<S8b1S4a<@H`<302;bl_0S0`<08a<C42=CDe0STi>@8m?Cd2
@D510TE5A@9:BTX2DU9B0UmOG`9^KVh2O7al0X>3P`:1PH42NGUi0W=cL`9gMgL2R8R80Z6QX@:i^KT2
bLW90/_;b`;1`L42[jn_0YNGU`9jNWX2GemO0TU9B@93@d<2DU9B0Vi^KP:;Rh/2VYZJ0Y:BTP9hN7P2
FEUI0T51@@8h>3P00c/k>dA4A4A4A002Cdm?0UYJFP9SHf<2Jf][0Ve]K@9WIfL2G5aL0U5AD@9>CTh2
EeMG0VUYJ@9kNg/2Q8B40WmoO`9_Kfl2GEeM155AD@9IFET2HV9R0VEUI@9NGUh2DE5A0TA4A08m?Cd2
?clo0TQ8B09@D502DU9B0TU9B@8j>SX2:RXZ0QlO7`8M7Ad28b<S0R/[:`8c<c<2>SXj0Shn?P92@T82
ADE50TM7A`99BDT2BTY:0Ti>CP9DE5@2Fe]K0V5QH@9UIFD2I6AT0UmOG`9HF5P2DE5A0TY:BP95ADD2
@D510T10@094A4@2BTY:0Tm?C`9>CTh2AdM71Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`9>CTh2@4100S8b<P8X:2P28b<S0R4Q8@8U9BD2;Rh^0SXj>P97AdL4Cdm?0TY:BP91@D42>CTi0S<c
<`8_;bl2;Bd]0R/[:`8Y:BT4:2PX12TY:@8Z:RX2;2`/0S0`<08f=SH2?Shn0TI6AP9>CTh2EUIF0UeM
G@9RHV82IfMW0Va/K09`L704Lg=c0W5aL@9_Kfl2L71`0WQhN0:9RHT2WYjN0[>c/`;1`L42a/K60/71
`@:g]kL2YjNW0Y:BTP9mOGd2K6a/0VQXJ09eMGD2S8b<0Z>SX`:^[Zh2Xj>S0XZ:RP9^KVh2F5QH0U5A
D@03EUIFGemOGemO009XJ6P2KVi^0W5aL@9cLg<2LG5a0V][J`9TI6@2GEeM0UiNGP9YJFT2Ng]k0Xb<
S0:DU9@2Shn?0WmoO`9/K6`2GEeM15MGE`9IFET2F5QH0U5AD@96ATH2?Cdm0SXj>P8o?cl2B4Q80U5A
D@9EEED2D51@0TA4A08g=cL2:b/[0R<S8`@P82028b<S0RXZ:P8c<c<2?clo0T];B`9HF5P2IFEU0Vm_
K`9fMWH2NGUi0WQhN09fMWH2Lg=c0W5aL@9`L702KVi^0VYZJP9SHf<2FUYJ0U9BDP9?Cdl2De=C0UYJ
FP9QHF42HV9R0U]KF`Goool100000Oooo`005Oooo`030000oooooooo00;oool2CTi>0T10@08b<S82
:2PX0R<S8`8Q8B429BDU0Rh^;P8j>SX2AdM714m?C`9:BTX2@D510STi>@8c<c<2;bl_0Rd];@8[:b/2
:BTY12PX:0@Y:BT2:RXZ0R`/;08`<302=SHf0Shn?P96ATH2CTi>0UIFEP9MGEd2HV9R0VMWI`9/K6`2
L71`17=cL`9aLG42Kfm_0W1`L09hN7P2RHV90YjNWP:c/k<2`L710/K6aP;1`L42]kNg0ZNWY`:BTY82
OGem0Va/K09XJ6P2MGEe0Xb<S0:SXj<2[Zj^0Z>SX`::RXX2KVi^0UQHF09ADE400eIFEUmOGemOG`02
J6QX0Vi^KP9aLG42Lg=c0W5aL@9[Jf/2I6AT0UeMG@9NGUh2JFUY0W]kN`:<S8`2U9BD0Xn?S`9oOgl2
K6a/0UeMG@AGEeL2FEUI0UQHF09ADE42ATI60Sdm?@8j>SX2?clo0TQ8B09ADE42EEEE0U1@D094A4@2
=cLg0R/[:`8S8b<4820P0R<S8`8Z:RX2<c<c0Slo?`9;Bd/2F5QH0VEUI@9_Kfl2MWIf0WUiN@9hN7P2
MWIf0W=cL`9aLG42L71`0Vi^KP9ZJVX2Hf=S0UYJFP9BDU82Cdm?0U=CD`9JFUX2HF5Q0V9RHP9KFe/5
oooo0@00007oool001Goool00`000?ooooooo`02oooo0VMWI`9IFET2Bd];0T51@@8i>CT4=CDe0STi
>@8o?cl2ATI614];B`97AdL2@4100STi>@8c<c<2;Bd]0RLW9`8S8b<2820P11hN7P8O7al28B4Q0RHV
9P8/;2`2=CDe0T10@09<C4`2FEUI0VEUI@9aLG42NWYj0WmoO`9nOWh2MWIf0VYZJP9MGEd2EEEE0UAD
E09KFe/2IFEU0W9bLP9oOgl2RXZ:0YFEU@:OWil2YjNW0Zf][@:`/;02[Zj^0ZNWY`:MWId2U9BD0Y>C
T`:KVi/2ZZZZ0[Rh^0:m_Kd2/k>c0YjNWP:9RHT2NGUi0WEeM@03NWYjPH61PH6100B7QhL2Ph>30Win
OP9iNGT2MGEe0W=cL`9dM7@2N7Qh0X:2PP:?Shl2VYZJ0YjNWP:IVIT2S8b<0W]kN`9ZJVX2GUiN0UEE
E@9>CTh2B4Q80T51@@8l?3`2>SXj0Sdm?@95ADD2D51@0UUIF@9NGUh2G5aL0UADE098B4P2>SXj0R`/
;08P820261PH0QHF5P8K6a/29RHV0SLg=`9=CDd2JFUY0XF5Q@:LW9`2ZJVY0ZRXZ0:KVi/2Rh^;0Win
OP9kNg/2Ogmo0XB4Q0:6QXH2P8200WEeM@9[Jf/2IFEU0VQXJ09`L702MgMg0WUiN@9cLg<5oooo0@00
007oool001Goool200000oooo`9WIfL2FEUI0T];B`91@D42>CTi13De=@8i>CT2?clo0TI6APA;Bd/2
AdM70T10@08i>CT2<c<c0Rd];@8W9bL28b<S0R0P80@N7Qh27alO0R4Q8@8V9RH2;2`/0SDe=@90@402
C4a<0UUIF@9UIFD2LG5a0WYjNP9oOgl2OWin0WIfMP9ZJVX2GEeM0UEEE@9DE5@2Fe]K0VEUI@9bLW82
Ogmo0XZ:RP:EUID2WinO0ZNWY`:][Jd2/;2`0Zj^[P:WYjL2WIfM0YBDU0:CTi<2Vi^K0ZZZZP:h^;P2
_Kfm0[>c/`:NWYh2RHV90WUiN@9eMGD00gYjNX61PH61P@04QhN70X>3P`9nOWh2NGUi0WEeM@9cLg<2
M7Ad0WQhN0:2PX82Shn?0YZJVP:NWYh2VIVI0Xb<S09kNg/2JVYZ0UiNGP9EEED2CTi>0TQ8B091@D42
?3`l0SXj>P8m?Cd2ADE50U1@D09IFET2GUiN0UaLG09DE5@2B4Q80SXj>P8/;2`2820P0QPH608F5QH2
6a/K0RHV9P8g=cL2CDe=0VUYJ@:5QHD2W9bL0ZVYZ@:XZ:P2Vi^K0X^;R`9nOWh2Ng]k0WmoO`:4Q8@2
QXJ60X20P09eMGD2Jf][0VEUI@9XJ6P2L71`0WMgM`9iNGT2Lg=c1?ooo`800001oooo000Eoooo00<0
003oooooool00_ooo`9mOGd2Kfm_0V9RHP9HF5P2D51@0TY:BP96ATH2ADE50TM7A`99BDT2BTY:0TQ8
B095ADD2@4100STi>@8b<S82:b/[0R@T908N7Qh26QXJ11PH608J6QX27QhN0RDU9@8`<302?Cdm0Te=
C@9NGUh2KVi^0WinOP:=SHd2UiNG0Y^KV`:DU9@2P8200VEUI@9:BTX2>SXj0S/k>`99BDT2GEeM0Ve]
K@9dM7@2MGEe0WAdM09iNGT2Q8B40YFEU@:UYJD2/[:b0[Rh^0Bi^KT2^[Zj0[jn_P;4a<@2b<S80/K6
aP:n_[h2/K6a0ZBTY0:LW9`2Vi^K00>OWinTY:BTY:@00ZFUY@:OWil2UIFE0X^;R`:4Q8@2Ph>30XN7
Q`:=SHd2UIFE0YfMW@:SXj<4YjNW0Z6QX@:FUYH2R8R80WMgM`9VIVH2EEEE0TQ8B08n?Sh2>CTi0SXj
>P8n?Sh2AdM70U5AD@9LG5`2I6AT16YZJP9VIVH2G5aL0Te=C@8k>c/2:2PX0QTI6@8@41024A4A0Q`L
708_;bl2CDe=0WIfMP:QXJ42a/K60]WIf@;Cdm<2^[Zj0YbLW0:6QXH2PX:20X^;R`:IVIT2XJ6Q0YfM
W@:ATI42Q8B40WemO@9nOWh2QHF50Xb<S0:=SHd2R8R81Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9mOGd2Kfm_0V9RHP9HF5P2D51@0TY:BP96ATH2ADE50TM7A`99BDT2BTY:0TQ8B095ADD2
@4100STi>@8b<S82:b/[0R@T908N7Qh26QXJ11PH608J6QX27QhN0RDU9@8`<302?Cdm0Te=C@9NGUh2
KVi^0WinOP:=SHd2UiNG0Y^KV`:DU9@2P8200VEUI@9:BTX2>SXj0S/k>`99BDT2GEeM0Ve]K@9dM7@2
MGEe0WAdM09iNGT2Q8B40YFEU@:UYJD2/[:b0[Rh^0Bi^KT2^[Zj0[jn_P;4a<@2b<S80/K6aP:n_[h2
/K6a0ZBTY0:LW9`2Vi^K00>OWinTY:BTY:@00ZFUY@:OWil2UIFE0X^;R`:4Q8@2Ph>30XN7Q`:=SHd2
UIFE0YfMW@:SXj<4YjNW0Z6QX@:FUYH2R8R80WMgM`9VIVH2EEEE0TQ8B08n?Sh2>CTi0SXj>P8n?Sh2
AdM70U5AD@9LG5`2I6AT16YZJP9VIVH2G5aL0Te=C@8k>c/2:2PX0QTI6@8@41024A4A0Q`L708_;bl2
CDe=0WIfMP:QXJ42a/K60]WIf@;Cdm<2^[Zj0YbLW0:6QXH2PX:20X^;R`:IVIT2XJ6Q0YfMW@:ATI42
Q8B40WemO@9nOWh2QHF50Xb<S0:=SHd2R8R81Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`:7QhL2NWYj0Ve]K@9TI6@2G5aL0UMGE`9DE5@2De=C15ADE09CDe<2CTi>0TI6AP8n?Sh2=CDe0Rh^
;P8Y:BT292@T0R4Q8@@O7al2820P0R@T908Y:BT2<C4a0S`l?09;Bd/2GEeM0W5aL@:4Q8@2UIFE0ZFU
Y@:_[jl2/K6a0ZJVYP:<S8`2JFUY0TQ8B08d=3@2>3Ph0T];B`9SHf<2M7Ad0WIfMP9/K6`2HV9R0V5Q
H@9^KVh2QHF50YnOW`:e]KD2a<C40/c<c0;@d=04dM7A0]3@d0;>c/h2bLW90/C4a0:n_[h2^KVi0[Rh
^0:i^KT00kfm_Kno_kno_`02_Kfm0[Bd]0:WYjL2Vi^K0YBDU0:EUID2W9bL0ZJVYP:_[jl2]KFe0[Jf
]P:e]KD2/;2`0ZVYZ@:OWil2TY:B0X:2PP9`L702GUiN0U1@D096ATH2A4A40TU9B@9ADE42Fe]K0VAT
I09[Jf/2L71`0WEeM@9hN7P2NGUi0WAdM09WIfL2De=C0S/k>`8V9RH261PH0Q<C4`8I6AT2:b/[0T];
B`9jNWX2[jn_0]gMg@;dm?@2j^[Z0/S8b0:QXJ42QHF50X:2PP:CTi<2Z:RX0[Ng]`:e]KD2Z:RX0YVI
V@B?Shl2U9BD19ZJVP:DU9@5oooo0@00007oool001Goool00`000?ooooooo`02oooo0XN7Q`9jNWX2
KFe]0VATI09LG5`2EeMG0UADE09CDe<4E5AD0U=CD`9>CTh2ATI60Shn?P8e=CD2;Rh^0RTY:@8T92@2
8B4Q11lO7`8P820292@T0RTY:@8a<C42?3`l0T];B`9MGEd2LG5a0XB4Q0:EUID2YJFU0Zn_[`:a/K42
YZJV0Xb<S09YJFT2B4Q80S@d=08h>3P2Bd];0V=SH`9dM7@2MWIf0Va/K09RHV82HF5Q0Vi^KP:5QHD2
WinO0[Fe]@;4a<@2c<c<0]3@d0CAdM42d=3@0/k>cP;9bLT2a<C40[jn_P:i^KT2^;Rh0[Vi^@03_Kfm
_kno_kno00:m_Kd2];Bd0ZNWY`:KVi/2U9BD0YFEU@:LW9`2YZJV0Zn_[`:e]KD2][Jf0[Fe]@:`/;02
ZJVY0YnOW`:BTY82PX:20W1`L09NGUh2D51@0TI6AP94A4@2BDU90U5AD@9KFe/2I6AT0V][J`9`L702
MGEe0WQhN09iNGT2M7Ad0VMWI`9CDe<2>c/k0RHV9P8H61P24a<C0QTI6@8[:b/2Bd];0WYjNP:_[jl2
gMgM0_Cdm0;Zj^X2b<S80Z6QX@:5QHD2PX:20Y>CT`:XZ:P2]kNg0[Fe]@:XZ:P2VIVI18n?S`:DU9@4
VYZJ0YBDU0Goool100000Oooo`005Oooo`030000oooooooo00;oool2PX:20W=cL`9VIVH2GUiN0UYJ
FP9IFET2G5aL0V5QH@9WIfL2Jf][0VUYJ@9NGUh2CDe=0S`l?08];Bd29bLW0RHV9P8Y:BT2;Rh^0S<c
<`8h>3P2?Shn0T=3@`97AdL2C4a<0UADE09OGel2L71`0XB4Q0:GUiL2ZJVY0[Jf]PBm_Kd2/[:b0YZJ
VP9lO7`2H61P0U9BDP9GEeL2JVYZ0X61P@:>SXh2RXZ:0W]kN`9/K6`2IVIV0W5aL@:8R8P2XZ:R0[Vi
^@;8b<P2clo?1=;BdP;@d=02cLg=0/S8b0;4a<@4`/;20/?3``;6a/H2bLW900?<c<c=cLg=cLd00/[:
bP;2`/82^;Rh0Zj^[P:YZJT2ZZZZ0[6a/@:j^[X2`L711<K6aP;3`l<2_Kfm0[>c/`:VYZH2VIVI0X^;
R`9nOWh2Lg=c0V][J`9WIfL2JFUY0Vi^KP9fMWH2O7al17emO@9kNg/2OGem0X>3P`::RXX2SHf=0XN7
Q`9eMGD2G5aL0T=3@`8_;bl28b<S0R8R8P8[:b/2ATI60W9bLP:VYZH2eMGE0^g]k@;QhN42_Kfm0YBD
U09iNGT2O7al0YBDU0:b/[84a/K60[Ng]`:UYJD2UiNG0YBDU0:IVIT2WIfM0Y^KV`:ATI45oooo0@00
007oool001Goool00`000?ooooooo`02oooo0X:2PP9cLg<2IVIV0UiNGP9JFUX2FEUI0UaLG09QHF42
IfMW0V][J`9YJFT2GUiN0Te=C@8l?3`2;Bd]0RLW9`8V9RH2:BTY0Rh^;P8c<c<2>3Ph0Shn?P93@d<2
AdM70Ta<C09DE5@2GemO0W1`L0:4Q8@2UiNG0ZVYZ@:f][H4_Kfm0[:b/P:JVYX2O7al0V1PH09BDU82
EeMG0VYZJP:1PH42SXj>0XZ:RP9kNg/2K6a/0VIVIP9aLG42R8R80Z:RXP:i^KT2b<S80/o?c`CBd]82
d=3@0/g=c@;8b<P2a<C41<;2`P;3`l<2a/K60/W9b@03c<c<cLg=cLg=00;:b/X2`/;20[Rh^0:^[Zh2
ZJVY0ZZZZP:a/K42^[Zj0/71`@C6a/H2`l?30[fm_@:c/k<2YZJV0YVIV@:;Rh/2OWin0W=cL`9[Jf/2
IfMW0VUYJ@9^KVh2MWIf0WalO0AmOGd2Ng]k0WemO@:3Ph<2RXZ:0Xf=S@:7QhL2MGEe0UaLG093@d<2
;bl_0R<S8`8R8R82:b/[0TI6AP9bLW82YZJV0]GEe@;]kNd2hN7Q0[fm_@:DU9@2NGUi0WalO0:DU9@2
/[:b1<K6aP:g]kL2YJFU0YNGU`:DU9@2VIVI0YfMW@:KVi/2TI6A1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`9aLG42GUiN0U5AD@9;Bd/2C4a<0U9BDP9MGEd2KFe]0WalO0:7QhL2QXJ60WEe
M@9IFET2>c/k0RHV9P8P82029bLW0S@d=093@d<2DE5A0UeMG@9WIfL2KVi^171`L09aLG42MgMg0XB4
Q0:FUYH2Z:RX0[Ng]`;0`<02`/;20/30`0:g]kL2ZJVY0YZJVP:;Rh/2QXJ60Xf=S@:MWId2[Jf]0[Bd
]0:/[:`2Vi^K0XV9R@:0P802QXJ60YRHV0:/[:`2_Kfm0/K6aP;8b<P2a/K60/?3``;0`<02_Kfm0[^k
^`:j^[X2_;bl0[no_`;3`l<2b<S80/_;b`03c/k>clo?clo?00;=cLd2b/[:0/G5a@;1`L42_kno0/30
`0;4a<@2b<S80/c<c0;@d=02d];B0]3@d0;9bLT2_;bl0Z^[Z`:KVi/2Shn?0XZ:RP:<S8`2TI6A0YNG
U`:MWId2XJ6Q0Z>SX`:PX:02VIVI0Xj>SP:4Q8@2PX:20XZ:RP:HV9P2Y:BT0ZJVYP:JVYX2Ph>30VYZ
JP9BDU82?clo0S@d=08b<S82@D510V=SH`:?Shl2^;Rh0/g=c@;1`L42X:2P0WalO09XJ6P2LW9b0Y6A
T@:e]KD2c<c<0/g=c@:m_Kd2YjNW0YNGU`:BTY82UIFE0YRHV0:DU9@2QHF51Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`9aLG42GUiN0U5AD@9;Bd/2C4a<0U9BDP9MGEd2KFe]0WalO0:7QhL2
QXJ60WEeM@9IFET2>c/k0RHV9P8P82029bLW0S@d=093@d<2DE5A0UeMG@9WIfL2KVi^171`L09aLG42
MgMg0XB4Q0:FUYH2Z:RX0[Ng]`;0`<02`/;20/30`0:g]kL2ZJVY0YZJVP:;Rh/2QXJ60Xf=S@:MWId2
[Jf]0[Bd]0:/[:`2Vi^K0XV9R@:0P802QXJ60YRHV0:/[:`2_Kfm0/K6aP;8b<P2a/K60/?3``;0`<02
_Kfm0[^k^`:j^[X2_;bl0[no_`;3`l<2b<S80/_;b`03c/k>clo?clo?00;=cLd2b/[:0/G5a@;1`L42
_kno0/30`0;4a<@2b<S80/c<c0;@d=02d];B0]3@d0;9bLT2_;bl0Z^[Z`:KVi/2Shn?0XZ:RP:<S8`2
TI6A0YNGU`:MWId2XJ6Q0Z>SX`:PX:02VIVI0Xj>SP:4Q8@2PX:20XZ:RP:HV9P2Y:BT0ZJVYP:JVYX2
Ph>30VYZJP9BDU82?clo0S@d=08b<S82@D510V=SH`:?Shl2^;Rh0/g=c@;1`L42X:2P0WalO09XJ6P2
LW9b0Y6AT@:e]KD2c<c<0/g=c@:m_Kd2YjNW0YNGU`:BTY82UIFE0YRHV0:DU9@2QHF51Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9NGUh2AdM70SLg=`8d=3@2>SXj0TM7A`9KFe/2M7Ad0Xf=
S@:NWYh2WinO0XZ:RP9VIVH2@D510RLW9`8R8R82;Rh^0TA4A09LG5`2LG5a0X:2PP:?Shl4UiNG0Y>C
T`:>SXh2SHf=0YJFUP:VYZH2][Jf0/;2`P;5aLD2`l?30[jn_P:i^KT4]kNg0[Vi^@:n_[h2alO70]7A
d@CIfMT2c/k>0[bl_0:ZZZX2WinO0Z2PX0:[Zj/2^;Rh0/71`@;2`/82_;bl0[Bd]0:^[Zh2ZZZZ0Z^[
Z`:/[:`2[jn_0[>c/`:h^;P2_[jn0/?3``;6a/H00lS8b<[:b/[:bP02bl_;0/c<c0;>c/h2d=3@0]7A
d@CBd]82dM7A0]?Cd`;Fe]H2f][J0]_Kf`;Cdm<2`/;20Z^[Z`:FUYH2S8b<0Y6AT@:RXZ82]KFe0/K6
aP;?cll2d=3@0/_;b`;0`<02/;2`0YfMW@:=SHd2QhN70Y2@T0:RXZ82/k>c0[fm_@:h^;P2YjNW0Y6A
T@9hN7P2H61P0Ta<C08o?cl2@D510UEEE@9dM7@2U9BD0ZBTY0:JVYX2P8200VATI09IFET2J6QX0X^;
R`:`/;02b<S80/[:bP:j^[X2Y:BT0Y>CT`:>SXh2T92@0Y6AT@::RXX2N7Qh1Oooo`400001oooo000E
oooo0P0000?oool2GUiN0TM7A`8g=cL2=3@d0SXj>P97AdL2Fe]K0WAdM0:=SHd2WYjN0YnOW`::RXX2
IVIV0T51@@8W9bL28R8R0Rh^;P94A4@2G5aL0W5aL@:2PX82Shn?19NGU`:CTi<2SXj>0Xf=S@:FUYH2
YZJV0[Jf]P;2`/82aLG50/?3``:n_[h2^KVi1;Ng]`:i^KT2_[jn0/O7a`;AdM44fMWI0/k>cP:l_;`2
ZZZZ0YnOW`:PX:02Zj^[0[Rh^0;1`L42`/;20[bl_0:d];@2[Zj^0ZZZZP:[Zj/2[:b/0Zn_[`:c/k<2
^;Rh0[jn_P;3`l<2a/K600?8b<S:b/[:b/X00/_;b`;<c<`2c/k>0]3@d0;AdM44d];B0]7Ad@;Cdm<2
e]KF0][JfP;Kfm/2dm?C0/;2`P:[Zj/2UYJF0Xb<S0:ATI42XZ:R0[Fe]@;6a/H2clo?0]3@d0;;bl/2
`<300[2`/0:MWId2SHf=0XN7Q`:@T902XZ:R0[>c/`:m_Kd2^;Rh0ZNWY`:ATI42N7Qh0V1PH09<C4`2
?clo0T51@@9EEED2M7Ad0YBDU0:TY:@2VYZJ0X20P09TI6@2FEUI0VQXJ0:;Rh/2/;2`0/S8b0;:b/X2
^[Zj0ZBTY0:CTi<2SXj>0Y2@T0:ATI42RXZ:0WQhN0Coool200000Oooo`005Oooo`030000oooooooo
00;oool2DU9B0SHf=P8S8b<2820P0RTY:@8k>c/2De=C0W5aL@:=SHd2XZ:R0ZJVYP:DU9@2Lg=c0U5A
D@8j>SX2=cLg0TI6AP9MGEd2MWIf0Xb<S0:MWId2ZZZZ0[6a/@:`/;02ZJVY0Z6QX@:NWYh2Y:BT0[2`
/0:n_[h4alO70/71`@:k^k/2^KVi0/30`0;<c<`2fMWI0^KViP;`l?02m_Kf0_Shn0;clo<2i^KV0]CD
e0;2`/82][Jf0[Bd]0:i^KT2`<300/?3``:n_[h2/k>c0ZNWY`:NWYh2W9bL0YjNWP:RXZ82Z:RX0Zf]
[@:c/k<2^;Rh0[bl_0:o_kl00l71`L;2`/;2`P02aLG50/W9b@;?cll2eMGE0]WIf@;Jf]X2fMWI0]KF
eP;EeMD2fMWI0]kNgP;Ogml2e]KF0[no_`:SXj<2RXZ:0X61P@:>SXh2ZZZZ0/[:bP;Shn<2k^k^0^_[
j`;Ph>02clo?0[^k^`:VYZH2UIFE0Xf=S@:DU9@2Y:BT0[Jf]P;3`l<2a<C40[^k^`:/[:`2UiNG0Wmo
O`9XJ6P2E5AD0T];B`9BDU82I6AT0WMgM`:1PH42Ng]k0VYZJP9IFET2EEEE0VIVIP:3Ph<2XZ:R0[Ng
]`:i^KT2[Jf]0YbLW0:?Shl2SHf=192@T0:7QhL2L71`1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9BDU82=SHf0R<S8`8P8202:BTY0S/k>`9CDe<2LG5a0Xf=S@:RXZ82YZJV0YBDU09cLg<2
DE5A0SXj>P8g=cL2ATI60UeMG@9fMWH2S8b<0YfMW@:ZZZX2/K6a0[2`/0:YZJT2XJ6Q0YjNWP:TY:@2
/;2`0[jn_PC7alL2`L710[^k^`:i^KT2`<300/c<c0;IfMT2i^KV0_3`l0;fm_H2n?Sh0_?cl`;Vi^H2
e=CD0/;2`P:f][H2];Bd0[Vi^@;0`<02`l?30[jn_P:c/k<2YjNW0YjNWP:LW9`2WYjN0Z:RXP:XZ:P2
[Jf]0[>c/`:h^;P2_;bl0[no_`03`L71`/;2`/;200;5aLD2bLW90/o?c`;EeMD2fMWI0][JfP;IfMT2
e]KF0]GEe@;IfMT2g]kN0]oOg`;Fe]H2_kno0Z>SX`::RXX2PH610Xj>SP:ZZZX2b/[:0^?Sh`;^k^h2
jn_[0^3Ph0;?cll2^k^k0ZJVYP:EUID2SHf=0YBDU0:TY:@2][Jf0/?3``;4a<@2^k^k0Zb/[0:GUiL2
Ogmo0VQXJ09DE5@2Bd];0U9BDP9TI6@2MgMg0X61P@9kNg/2JVYZ0UUIF@9EEED2IVIV0X>3P`:RXZ82
]kNg0[Vi^@:][Jd2W9bL0Xn?S`:=SHd4T92@0XN7Q`9`L705oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0U9BDP8a<C426a/K0QHF5P8O7al2<C4a0TQ8B09QHF42MgMg0XV9R@:BTY82SHf=0Wmo
O`9_Kfl2I6AT0VIVIP9aLG42P8200Y2@T0:MWId2YZJV0Zj^[P:a/K42[jn_0ZZZZP:UYJD2Y:BT0ZZZ
ZP:e]KD2`L710/O7a`;6a/H2_kno1;Vi^@;2`/82d];B0^?Sh`;alO42no_k1?ooo`;hn?P2jn_[0][J
fP;9bLT2_Kfm0[Zj^P:m_Kd2`/;20/?3``:m_Kd2/[:b0ZJVYP:MWId2Vi^K0YjNWP:SXj<2Z:RX0Zf]
[@:a/K42];Bd0[Ng]`:i^KT00k^k^kbl_;bl_002_kno0/?3``;8b<P2c/k>0]?Cd`;Fe]H2emOG1=KF
eP;Jf]X2gMgM0]cLg0;?cll2];Bd0Y:BTP9fMWH2K6a/0WemO@:PX:02alO70^GUi@;_knl2j>SX0]WI
f@;6a/H2]KFe0ZNWY`:LW9`2UYJF0YRHV0:PX:02ZZZZ0[>c/`:i^KT2^[Zj0[Fe]@:[Zj/2VYZJ0XJ6
QP9bLW82I6AT0V5QH@9VIVH2KFe]0W9bLP9`L702JVYZ0VATI09UIFD2KVi^0WemO@:=SHd2V9RH0Y^K
V`:HV9P2TY:B0Y2@T0:CTi<4VIVI0Xn?S`9eMGD5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0U9BDP8a<C426a/K0QHF5P8O7al2<C4a0TQ8B09QHF42MgMg0XV9R@:BTY82SHf=0WmoO`9_Kfl2
I6AT0VIVIP9aLG42P8200Y2@T0:MWId2YZJV0Zj^[P:a/K42[jn_0ZZZZP:UYJD2Y:BT0ZZZZP:e]KD2
`L710/O7a`;6a/H2_kno1;Vi^@;2`/82d];B0^?Sh`;alO42no_k1?ooo`;hn?P2jn_[0][JfP;9bLT2
_Kfm0[Zj^P:m_Kd2`/;20/?3``:m_Kd2/[:b0ZJVYP:MWId2Vi^K0YjNWP:SXj<2Z:RX0Zf][@:a/K42
];Bd0[Ng]`:i^KT00k^k^kbl_;bl_002_kno0/?3``;8b<P2c/k>0]?Cd`;Fe]H2emOG1=KFeP;Jf]X2
gMgM0]cLg0;?cll2];Bd0Y:BTP9fMWH2K6a/0WemO@:PX:02alO70^GUi@;_knl2j>SX0]WIf@;6a/H2
]KFe0ZNWY`:LW9`2UYJF0YRHV0:PX:02ZZZZ0[>c/`:i^KT2^[Zj0[Fe]@:[Zj/2VYZJ0XJ6QP9bLW82
I6AT0V5QH@9VIVH2KFe]0W9bLP9`L702JVYZ0VATI09UIFD2KVi^0WemO@:=SHd2V9RH0Y^KV`:HV9P2
TY:B0Y2@T0:CTi<4VIVI0Xn?S`9eMGD5oooo0@00007oool001Goool00`000?ooooooo`02oooo0UaL
G08g=cL27AdM0QDE5@8L71`2:b/[0Sdm?@9:BTX2EEEE0V1PH09/K6`2NWYj0XN7Q`:CTi<2W9bL0Z6Q
X@:TY:@4YJFU0Z>SX`:RXZ82X:2P0YjNWPBMWId2WYjN0Z>SX`:[Zj/2][Jf0[no_`;5aLD2a<C40[no
_`Bj^[X2`L710/g=c@;Jf]X2i^KV0^o_k`Cdm?@2kno_0^;RhP;Bd]82`/;20[Ng]`:e]KD2^[Zj0/30
`0;3`l<2`<300[Ng]`:^[Zh2Z:RX0ZJVYP:XZ:P2Zj^[0Zj^[P:`/;02/K6a0[:b/P:c/k<2]KFe00>g
]kNh^;Rh^;P00[Zj^P:k^k/2_;bl0[no_`;3`l<2bLW90/k>cP;Bd]82e]KF0]WIf@;Jf]X2e=CD0/?3
``:SXj<2OWin0UiNGP9BDU82I6AT0XV9R@:c/k<2dM7A0]WIf@;>c/h2_;bl0Zb/[0:SXj<4XZ:R0Z6Q
X@:LW9`2UYJF0Y>CT`:EUID2W9bL0ZNWY`:a/K42];Bd0Zj^[P:SXj<2U9BD0XN7Q`9mOGd2MWIf17=c
L`9fMWH2NWYj0WmoO`:1PH42Ogmo0WUiN@9eMGD2Lg=c0WMgM`9nOWh2QhN70Y:BTP:MWId2YZJV0ZRX
Z0:MWId2PH611Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9LG5`2=cLg0QdM7@8E5AD2
71`L0R/[:`8m?Cd2BTY:0UEEE@9PH602K6a/0WYjNP:7QhL2Ti>C0YbLW0:QXJ42Y:BT1:FUY@:SXj<2
XZ:R0Z2PX0:NWYh4WIfM0YjNWP:SXj<2Zj^[0[Jf]P:o_kl2aLG50/C4a0:o_kl4^[Zj0/71`@;=cLd2
f][J0^KViP;_knl4m?Cd0^o_k`;Rh^82d];B0/;2`P:g]kL2]KFe0[Zj^P;0`<02`l?30/30`0:g]kL2
[Zj^0ZRXZ0:VYZH2Z:RX0Z^[Z`:^[Zh2/;2`0[6a/@:b/[82/k>c0[Fe]@03]kNg^;Rh^;Rh00:j^[X2
^k^k0[bl_0:o_kl2`l?30/W9b@;>c/h2d];B0]KFeP;IfMT2f][J0]CDe0;3`l<2Xj>S0WinOP9NGUh2
DU9B0VATI0:9RHT2/k>c0]7Ad@;IfMT2c/k>0[bl_0:/[:`2Xj>S1::RXP:QXJ42W9bL0YJFUP:CTi<2
UIFE0YbLW0:WYjL2/K6a0[Bd]0:^[Zh2Xj>S0YBDU0:7QhL2OGem0WIfMPAcLg<2MWIf0WYjNP9oOgl2
PH610WmoO`9iNGT2MGEe0W=cL`9gMgL2OWin0XN7Q`:BTY82WIfM0ZJVYP:XZ:P2WIfM0X61P@Goool1
00000Oooo`005Oooo`030000oooooooo00;oool2J6QX0T51@@8T92@26ATI0QhN7P8[:b/2=SHf0SPh
>08g=cL2>SXj0TQ8B09VIVH2SHf=0[>c/`;=cLd2emOG0]7Ad@;4a<@2/k>c0ZBTY0:FUYH2S8b<0XJ6
QP:5QHD2RXZ:0Y>CT`:NWYh2ZJVY0[Bd]0:m_Kd4`/;20[no_`:l_;`2^[Zj0[fm_@;3`l<2bl_;0]?C
d`;Kfm/2hN7Q0^CTi0;Ph>02eMGE0/K6aP:f][H4[Jf]0[Bd]0:m_Kd4a<C40/30`0:j^[X2][Jf0[Fe
]@:g]kL4^KVi0[Ng]`:c/k<2/;2`0Zj^[P:_[jl00k:b/[Fe]KFe]@02][Jf0[Bd]0:`/;02[Zj^0[6a
/@:i^KT2`l?30/g=c@;De=@2f=SH0]KFeP;=cLd2^;Rh0YFEU@9/K6`2BDU90S/k>`9;Bd/2L71`0YVI
V@:f][H2^k^k0Zj^[P:JVYX2SHf=0Xn?S`:KVi/2Z:RX0Z^[Z`:PX:02S8b<0WYjNP9cLg<2O7al0Y2@
T0:VYZH2]kNg0[jn_P:k^k/2];Bd0ZVYZ@:KVi/2S8b<0X20P09lO7`2Ph>30Y6AT@:OWil2XJ6Q0Y>C
T`9jNWX2HV9R0U=CD`9FEUH2IVIV0W]kN`:ATI42Y:BT0[2`/0:d];@2ZZZZ0Xj>SPGoool100000Ooo
o`005Oooo`030000oooooooo00;oool2J6QX0T51@@8T92@26ATI0QhN7P8[:b/2=SHf0SPh>08g=cL2
>SXj0TQ8B09VIVH2SHf=0[>c/`;=cLd2emOG0]7Ad@;4a<@2/k>c0ZBTY0:FUYH2S8b<0XJ6QP:5QHD2
RXZ:0Y>CT`:NWYh2ZJVY0[Bd]0:m_Kd4`/;20[no_`:l_;`2^[Zj0[fm_@;3`l<2bl_;0]?Cd`;Kfm/2
hN7Q0^CTi0;Ph>02eMGE0/K6aP:f][H4[Jf]0[Bd]0:m_Kd4a<C40/30`0:j^[X2][Jf0[Fe]@:g]kL4
^KVi0[Ng]`:c/k<2/;2`0Zj^[P:_[jl00k:b/[Fe]KFe]@02][Jf0[Bd]0:`/;02[Zj^0[6a/@:i^KT2
`l?30/g=c@;De=@2f=SH0]KFeP;=cLd2^;Rh0YFEU@9/K6`2BDU90S/k>`9;Bd/2L71`0YVIV@:f][H2
^k^k0Zj^[P:JVYX2SHf=0Xn?S`:KVi/2Z:RX0Z^[Z`:PX:02S8b<0WYjNP9cLg<2O7al0Y2@T0:VYZH2
]kNg0[jn_P:k^k/2];Bd0ZVYZ@:KVi/2S8b<0X20P09lO7`2Ph>30Y6AT@:OWil2XJ6Q0Y>CT`9jNWX2
HV9R0U=CD`9FEUH2IVIV0W]kN`:ATI42Y:BT0[2`/0:d];@2ZZZZ0Xj>SPGoool100000Oooo`005Ooo
o`030000oooooooo00;oool2LG5a0TU9B@8[:b/2820P0RHV9P8a<C42>CTi0SHf=P8^;Rh2;2`/0SXj
>P9PH602TY:B0/?3``;UiND2kno_0^GUi@;?cll2][Jf0YfMW@::RXX2NWYj0W9bLP9cLg<2Ng]k0XV9
R@:JVYX2Z:RX0[Bd]0:m_Kd2`/;20/?3``;1`L42_Kfm0[^k^`:j^[X2_;bl0[no_`;4a<@2bl_;0];B
dP;GemL2eMGE0/[:bP:j^[X2Zj^[0Z:RXP:TY:@2[jn_0[bl_0;7alL2bl_;0/W9b@;6a/H4a<C40/K6
aP;8b<P2aLG50[jn_P:d];@2ZZZZ0ZFUY@:VYZH00jb/[;>c/k>c/`02]KFe0[6a/@:ZZZX4Y:BT0Zf]
[@:k^k/2bLW90]?Cd`;Hf=P2eMGE0/_;b`:c/k<2Shn?0VATI090@402;bl_0S`l?09NGUh2QXJ60Z6Q
X@:TY:@2UIFE0X:2PP9iNGT2Ph>30YRHV0:][Jd2/k>c0Z>SX`:6QXH2JVYZ0UeMG@9VIVH2Ogmo0YfM
W@:g]kL2a/K60/_;b`;9bLT2`L710[6a/@:OWil2SXj>0XR8R0:BTY82YZJV0[Rh^0:k^k/2YJFU0X20
P09KFe/4@d=30UEEE@9_Kfl2Rh^;0Z:RXP:a/K42]kNg0Zn_[`:FUYH5oooo0@00007oool001Goool2
00000oooo`9aLG42BDU90R/[:`8P82029RHV0S4a<@8i>CT2=SHf0Rh^;P8/;2`2>SXj0V1PH0:BTY82
`l?30^GUi@;_knl2iNGU0/o?c`:f][H2WIfM0XZ:RP9jNWX2LW9b0W=cL`9kNg/2RHV90YZJVP:XZ:P2
];Bd0[fm_@;2`/82`l?30/71`@:m_Kd2^k^k0[Zj^P:l_;`2_kno0/C4a0;;bl/2d];B0]OGe`;EeMD2
b/[:0[Zj^P:[Zj/2XZ:R0ZBTY0:_[jl2_;bl0/O7a`;;bl/2bLW90/K6aPC4a<@2a/K60/S8b0;5aLD2
_[jn0[Bd]0:ZZZX2YJFU0ZJVYP03[:b//k>c/k>c00:e]KD2/K6a0ZZZZPBTY:@2[Jf]0[^k^`;9bLT2
dm?C0]SHf0;EeMD2bl_;0[>c/`:?Shl2I6AT0T10@08_;bl2?3`l0UiNGP:6QXH2XJ6Q0ZBTY0:EUID2
PX:20WUiN@:3Ph<2V9RH0Zf][@:c/k<2Xj>S0XJ6QP9ZJVX2GEeM0VIVIP9oOgl2WIfM0[Ng]`;6a/H2
bl_;0/W9b@;1`L42/K6a0YnOW`:>SXh2R8R80Y:BTP:VYZH2^;Rh0[^k^`:UYJD2P8200U]KF`A3@d<2
EEEE0Vm_K`:;Rh/2XZ:R0[6a/@:g]kL2[jn_0YJFUPCoool200000Oooo`005Oooo`030000oooooooo
00;oool2L71`0T];B`8`<302:BTY0S4a<@8o?cl2BTY:0T];B`96ATH2A4A40U1@D09^KVh2UiNG0[fm
_@;GemL2gMgM0]?Cd`:o_kl2Z:RX0Y:BTP:0P802LW9b0V][J`9]KFd2N7Qh0XN7Q`:IVIT2ZZZZ0[Rh
^0;3`l<4b<S80/C4a0:o_kl2^k^k0[Vi^@:j^[X2_;bl0/;2`P;9bLT2d=3@0]GEe@;Bd]82a/K60[Bd
]0:SXj<2Vi^K0YnOW`:][Jd2_[jn0/c<c0;Bd]82dM7A0/g=c@;<c<`2c/k>0]?Cd`;EeMD2dM7A0/C4
a0:b/[82XJ6Q0YRHV0:JVYX00jJVY[:b/[:b/P02^[Zj0[Jf]P:][Jd2Y:BT0Z:RXP:ZZZX2^KVi0/S8
b0;De=@2fm_K0]WIf@;@d=02^[Zj0YFEU@9[Jf/2ADE50S8b<P8m?Cd2Fe]K0X20P0:JVYX2WIfM0Y2@
T09oOgl2NGUi0XF5Q@:MWId2/[:b0[Rh^0:WYjL2R8R80VYZJP9KFe/2Hf=S0WemO@:LW9`2^;Rh0/S8
b0C>c/h2alO70[Vi^@:XZ:P2V9RH0Y:BTP:NWYh2/[:b0/C4a0;7alL2/K6a0Xb<S09VIVH2Bd];0TE5
A@9@D502I6AT0W]kN`:BTY82Y:BT0Zj^[P:ZZZX2Ti>C1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9`L702Bd];0S0`<08Y:BT2<C4a0Slo?`9:BTX2Bd];0TI6AP94A4@2D51@0Vi^KP:GUiL2
_Kfm0]OGe`;MgMd2dm?C0[no_`:XZ:P2TY:B0X20P09bLW82Jf][0Ve]K@9hN7P2QhN70YVIV@:ZZZX2
^;Rh0/?3``C8b<P2a<C40[no_`:k^k/2^KVi0[Zj^P:l_;`2`/;20/W9b@;@d=02eMGE0];BdP;6a/H2
];Bd0Z>SX`:KVi/2WinO0Zf][@:n_[h2c<c<0];BdP;AdM42cLg=0/c<c0;>c/h2dm?C0]GEe@;AdM42
a<C40[:b/P:QXJ42V9RH0YZJVP03YZJV/[:b/[:b00:j^[X2][Jf0Zf][@:TY:@2XZ:R0ZZZZP:i^KT2
b<S80]CDe0;Kfm/2fMWI0]3@d0:j^[X2UIFE0V][J`95ADD2<S8b0Sdm?@9KFe/2P8200YZJVP:MWId2
T92@0WmoO`9iNGT2QHF50YfMW@:b/[82^;Rh0ZNWY`:8R8P2JVYZ0U]KF`9SHf<2OGem0YbLW0:h^;P2
b<S81<k>cP;7alL2^KVi0ZRXZ0:HV9P2TY:B0YjNWP:b/[82a<C40/O7a`:a/K42S8b<0VIVIP9;Bd/2
ADE50U1@D09TI6@2Ng]k0Y:BTP:TY:@2[Zj^0ZZZZP:CTi<5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0VUYJ@99BDT2<c<c0S4a<@8o?cl2De=C0VEUI@9_Kfl2Lg=c0WMgM`9nOWh2RXZ:0YVI
V@:VYZH4[Jf]0ZRXZ0:NWYh2TY:B0XJ6QP9jNWX2LW9b0Vm_K`9cLg<2OWin0Xb<S0:MWId2[jn_0[no
_`;<c<`2d];B0]3@d0;:b/X2`L710[^k^`:i^KT2_;bl0/;2`P;9bLT2dM7A0]OGe`;Jf]X2e]KF0/O7
a`:b/[82X:2P0YJFUP:LW9`2[Zj^0/?3``;Cdm<2fMWI0]KFeP;@d=02c/k>0]?Cd`;Kfm/2hN7Q0]cL
g0;9bLT2[jn_0YNGU`:9RHT2SXj>00>OWinc/k>c/k<01<30`0:g]kL2[Jf]0ZVYZ@:_[jl2_Kfm0/_;
b`;GemL2g]kN0^3Ph0;Jf]X2alO70ZFUY@9lO7`2EUIF0T92@P9:BTX2IFEU0XJ6QP:OWil2Y:BT0YZJ
VP:=SHd2R8R80Y>CT`:VYZH2^;Rh0[bl_0:/[:`2T92@0WIfMP9XJ6P2Kfm_0XJ6QP:PX:02]kNg0/C4
a0;8b<P2a/K60/30`0:e]KD2ZJVY0YjNWP:LW9`2YZJV0[Jf]P;4a<@2alO70[Ng]`:LW9`2OGem0VEU
I@9HF5P2EUIF0U]KF`9WIfL2NGUi0Xb<S0:JVYX2Vi^K0XR8R0Goool100000Oooo`005Oooo`030000
oooooooo00;oool2JFUY0TU9B@8c<c<2<C4a0Slo?`9CDe<2IFEU0Vm_K`9cLg<2MgMg0WinOP::RXX2
VIVI0ZJVYPB][Jd2Z:RX0YjNWP:BTY82QXJ60WYjNP9bLW82Kfm_0W=cL`9nOWh2S8b<0YfMW@:_[jl2
_kno0/c<c0;Bd]82d=3@0/[:bP;1`L42^k^k0[Vi^@:l_;`2`/;20/W9b@;AdM42emOG0][JfP;Fe]H2
alO70[:b/P:PX:02UYJF0YbLW0:^[Zh2`l?30]?Cd`;IfMT2e]KF0]3@d0;>c/h2dm?C0]_Kf`;QhN42
g=cL0/W9b@:_[jl2UiNG0XV9R@:>SXh00inOWk>c/k>c/`04`<300[Ng]`:][Jd2ZJVY0Zn_[`:m_Kd2
bl_;0]OGe`;Ng]h2h>3P0][JfP;7alL2YJFU0WalO09FEUH2@T920TY:BP9UIFD2QXJ60YnOW`:TY:@2
VYZJ0Xf=S@:8R8P2Ti>C0ZJVYP:h^;P2_;bl0Zb/[0:@T902MWIf0VQXJ09_Kfl2QXJ60Z2PX0:g]kL2
a<C40/S8b0;6a/H2`<300[Fe]@:YZJT2WYjN0YbLW0:VYZH2][Jf0/C4a0;7alL2]kNg0YbLW09mOGd2
IFEU0UQHF09FEUH2Fe]K0VMWI`9iNGT2S8b<0YZJVP:KVi/2R8R81Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`9PH602ADE50SHf=P8j>SX2CTi>0VUYJ@:3Ph<2UiNG0ZBTY0B/[:`2YJFU0YRH
V0:9RHT2OGem17QhN09kNg/2OGem0WalO09iNGT2N7Qh0WYjNP9oOgl2R8R80Y>CT`:RXZ82/k>c0/K6
aP;EeMD2g=cL0]WIf@;?cll2`l?31;^k^`;1`L42bLW90];BdP;Jf]X2h>3P0^7Qh@;Jf]X2bLW90[:b
/P:MWId2Ti>C0YZJVP:^[Zh2a/K60]WIf@;Ng]h2fMWI0]7Ad@;?cll2eMGE0^7Qh@;Xj>P2i>CT0/k>
cP:^[Zh2T92@0WmoO`:5QHD00i^KVkBd];Bd]002a/K60/W9b@;1`L42^;Rh0[>c/`:g]kL2`/;20/k>
cP;IfMT2h^;R0^KViP;Ti>@2eMGE0[Ng]`:@T902K6a/0UQHF09MGEd2MGEe0Y:BTP:YZJT2[jn_0ZVY
Z@:OWil2Vi^K0Z>SX`:a/K42_Kfm0[no_`:a/K42Vi^K0XJ6QP9lO7`2PH610Y6AT@:UYJD2]KFe1;fm
_@:i^KT2];Bd0Zj^[P:YZJT4YZJV0Zf][@:g]kL2_kno0/71`@:j^[X2[:b/0YZJVP:5QHD2LG5a0UmO
G`9CDe<2DE5A0UeMG@9`L702PX:20XR8R09jNWX5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0V1PH095ADD2=SHf0SXj>P9>CTh2JFUY0X>3P`:GUiL2Y:BT1:b/[0:UYJD2V9RH0XV9R@9mOGd4
N7Qh0W]kN`9mOGd2O7al0WUiN@9hN7P2NWYj0WmoO`:8R8P2Ti>C0Z:RXP:c/k<2a/K60]GEe@;Lg=`2
fMWI0/o?c`;3`l<4^k^k0/71`@;9bLT2d];B0][JfP;Ph>02hN7Q0][JfP;9bLT2/[:b0YfMW@:CTi<2
VYZJ0Zj^[P;6a/H2fMWI0]kNgP;IfMT2dM7A0/o?c`;EeMD2hN7Q0^SXj0;Ti>@2c/k>0Zj^[P:@T902
Ogmo0XF5Q@03Vi^K];Bd];Bd00;6a/H2bLW90/71`@:h^;P2/k>c0[Ng]`;2`/82c/k>0]WIf@;Rh^82
i^KV0^CTi0;EeMD2]kNg0Y2@T09/K6`2F5QH0UeMG@9eMGD2TY:B0ZVYZ@:_[jl2ZJVY0YnOW`:KVi/2
Xj>S0[6a/@:m_Kd2_kno0[6a/@:KVi/2QXJ60WalO0:1PH42TI6A0ZFUY@:e]KD4_Kfm0[Vi^@:d];@2
[Zj^0ZVYZ@BVYZH2[Jf]0[Ng]`:o_kl2`L710[Zj^P:/[:`2VYZJ0XF5Q@9aLG42GemO0U=CD`9ADE42
GEeM0W1`L0:2PX82R8R80WYjNPGoool100000Oooo`005Oooo`030000oooooooo00;oool2EUIF0T92
@P8j>SX2ATI60V5QH@:0P802WYjN0[Jf]P;5aLD2bl_;0/G5a@:`/;02TI6A0W9bLP9KFe/2E5AD0U]K
F`9WIfL2M7Ad0WalO0:0P802Q8B40XR8R0:=SHd2Ti>C0Y^KV`:VYZH2]kNg0/[:bP;Kfm/2hn?S0]oO
g`;De=@2a/K61;jn_P;5aLD2c/k>0]WIf@;QhN42i^KV0^OWi`;Ng]h2bl_;0[:b/P:LW9`2TI6A0YVI
V@:_[jl2b<S80]_Kf`;Ph>02fMWI0]3@d0;=cLd2e=CD0^;RhP;[jn/2j>SX0]7Ad@:_[jl2Shn?0Win
OP:3Ph<00iVIVKBd];Bd]002alO70/c<c0;6a/H2_[jn0[Vi^@:l_;`2aLG50/o?c`;IfMT2h^;R0^SX
j0;YjNT2gMgM0/?3``:QXJ42PH610Vi^KP9bLW82QhN70Z6QX@:e]KD2^[Zj0[Bd]0:[Zj/2YjNW0Zb/
[0:g]kL2`<300/71`@:g]kL2YJFU0Y>CT`::RXX2SHf=0YZJVP:XZ:P2/k>c0[Ng]`:d];@2/;2`0Zb/
[0:[Zj/2[:b/0Zn_[`:b/[82]KFe0[Ng]`:h^;P4^[Zj0[Vi^@:c/k<2Xj>S0XV9R@9ZJVX2CDe=0Sdm
?@92@T82EEEE0VUYJ@9cLg<2JVYZ1Oooo`400001oooo0009oooo0`0000?oool00`000?ooooooo`03
oooo00<0003oooooool00_ooo`9FEUH2@T920SXj>P96ATH2HF5Q0X20P0:NWYh2][Jf0/G5a@;;bl/2
aLG50[2`/0:ATI42LW9b0U]KF`9DE5@2Fe]K0VMWI`9dM7@2O7al0X20P0:4Q8@2R8R80Xf=S@:CTi<2
Vi^K0ZJVYP:g]kL2b/[:0]_Kf`;Shn<2gmoO0]CDe0;6a/H4_[jn0/G5a@;>c/h2fMWI0^7Qh@;Vi^H2
inOW0]kNgP;;bl/2/[:b0YbLW0:ATI42VIVI0Zn_[`;8b<P2fm_K0^3Ph0;IfMT2d=3@0/g=c@;De=@2
h^;R0^_[j`;Xj>P2dM7A0Zn_[`:?Shl2OWin0X>3P`03VIVI];Bd];Bd00;7alL2c<c<0/K6aP:n_[h2
^KVi0[bl_0;5aLD2clo?0]WIf@;Rh^82j>SX0^WYj@;MgMd2`l?30Z6QX@:1PH42KVi^0W9bLP:7QhL2
XJ6Q0[Fe]@:j^[X2];Bd0Z^[Z`:WYjL2[:b/0[Ng]`;0`<02`L710[Ng]`:UYJD2Ti>C0XZ:RP:=SHd2
VYZJ0ZRXZ0:c/k<2]kNg0[Bd]0:`/;02[:b/0Z^[Z`:/[:`2[jn_0[:b/P:e]KD2]kNg0[Rh^0Bj^[X2
^KVi0[>c/`:SXj<2RHV90VYZJP9=CDd2?Cdm0T92@P9EEED2JFUY0W=cL`9ZJVX5oooo0@00007oool0
00Woool00`000?ooooooo`02oooo00<0003oool000001?ooo`030000oooooooo00;oool2CDe=1451
@@9DE5@2M7Ad0YJFUP:b/[82a<C40/W9b@;6a/H2^;Rh0Z2PX0:3Ph<2IfMW0UEEE@9CDe<2GUiN0Vm_
K`:0P802Rh^;0Y6AT@:FUYH2V9RH0Y^KV`:MWId2XJ6Q0ZVYZ@:h^;P2bl_;0]cLg0;UiND2h^;R0]OG
e`;:b/X4`L710/O7a`;@d=02f][J0^;RhP;Xj>P2jNWY0^7Qh@;=cLd2];Bd0YfMW@:BTY82VYZJ0Zn_
[`;8b<P2f][J0]kNgP;Fe]H2cLg=0/W9b@;@d=02g]kN0^SXj0;Vi^H2dm?C0[Fe]@:GUiL2QXJ60XV9
R@03Vi^K/;2`/;2`00;0`<02aLG50/30`0:j^[X2]kNg0[^k^`;4a<@2cLg=0]KFeP;Ng]h4i>CT0]_K
f`;5aLD2ZJVY0Xj>SP:0P802QHF50YRHV0:_[jl2_kno0/30`0:f][H2ZZZZ0ZFUY@:ZZZX2][Jf0/;2
`P;5aLD2^k^k0ZVYZ@:GUiL2SHf=0Xn?S`:JVYX2Z:RX0[:b/P:e]KD2/k>c0Zn_[`:][Jd2[jn_0[Bd
]0:j^[X2_Kfm0[bl_0:h^;P2];Bd0[>c/`:i^KT2`<300/;2`P:g]kL2VIVI0W1`L098B4P2;bl_0Rd]
;@8m?Cd2DE5A0UiNGP9JFUX5oooo0@00007oool000[oool01`000?ooooooooooo`000?ooo`000004
oooo0P0000?oool2CDe=1451@@9DE5@2M7Ad0YJFUP:b/[82a<C40/W9b@;6a/H2^;Rh0Z2PX0:3Ph<2
IfMW0UEEE@9CDe<2GUiN0Vm_K`:0P802Rh^;0Y6AT@:FUYH2V9RH0Y^KV`:MWId2XJ6Q0ZVYZ@:h^;P2
bl_;0]cLg0;UiND2h^;R0]OGe`;:b/X4`L710/O7a`;@d=02f][J0^;RhP;Xj>P2jNWY0^7Qh@;=cLd2
];Bd0YfMW@:BTY82VYZJ0Zn_[`;8b<P2f][J0]kNgP;Fe]H2cLg=0/W9b@;@d=02g]kN0^SXj0;Vi^H2
dm?C0[Fe]@:GUiL2QXJ60XV9R@03Vi^K/;2`/;2`00;0`<02aLG50/30`0:j^[X2]kNg0[^k^`;4a<@2
cLg=0]KFeP;Ng]h4i>CT0]_Kf`;5aLD2ZJVY0Xj>SP:0P802QHF50YRHV0:_[jl2_kno0/30`0:f][H2
ZZZZ0ZFUY@:ZZZX2][Jf0/;2`P;5aLD2^k^k0ZVYZ@:GUiL2SHf=0Xn?S`:JVYX2Z:RX0[:b/P:e]KD2
/k>c0Zn_[`:][Jd2[jn_0[Bd]0:j^[X2_Kfm0[bl_0:h^;P2];Bd0[>c/`:i^KT2`<300/;2`P:g]kL2
VIVI0W1`L098B4P2;bl_0Rd];@8m?Cd2DE5A0UiNGP9JFUX4oooo0P00007oool000[oool01`000?oo
ooooooooo`000?ooo`000004oooo00<0003oooooool00_ooo`96ATH2@D510TQ8B09RHV82QHF50ZNW
Y`:n_[h2`/;20[Ng]`:VYZH2TI6A0WmoO`9bLW82J6QX0VIVIP9]KFd2NWYj0X^;R`:IVIT2Xj>S0ZNW
Y`:YZJT2Z:RX0ZJVYP:UYJD2YZJV0Z^[Z`:h^;P2bLW90]WIf@;Rh^82h>3P0]OGe`;=cLd2aLG50/C4
a0;9bLT2clo?0]OGe`;Ogml2i>CT0^KViP;Ogml2cLg=0[Fe]@:OWil2U9BD0Y^KV`:^[Zh2aLG50]GE
e@;Hf=P2dM7A0/O7a`;3`l<2b/[:0]KFePCPh>02dm?C0[bl_0:VYZH2UiNG0YJFUP03WinOZj^[Zj^[
00:d];@2][Jf0[>c/`:`/;02/K6a0[Jf]P:o_kl2b<S80]7Ad@;GemL2f][J0]WIf@;AdM42_kno0ZVY
Z@:EUID2S8b<0YBDU0:WYjL2^k^k0/O7a`;2`/82/[:b0Z6QX@:IVIT2X:2P0[6a/@;2`/82bLW90[no
_`:[Zj/2UIFE0XR8R0:9RHT2U9BD0Z>SX`:`/;02]KFe0[Jf]PBe]KD2^KVi0[no_`;5aLD2a/K60/71
`@:h^;P2/;2`0Zj^[P:f][H2`L710/S8b0:n_[h2WinO0W5aL@94A4@292@T0QhN7P8Z:RX2?Shn0Ta<
C09=CDd5oooo0@00007oool000Woool00`000?ooo`000002oooo00<0003oool000001?ooo`030000
oooooooo00;oool2ATI60T51@@98B4P2HV9R0XF5Q@:WYjL2_[jn0/;2`P:g]kL2YZJV0Y6AT@9oOgl2
LW9b0VQXJ09VIVH2KFe]0WYjNP:;Rh/2VIVI0Z>SX`:WYjL2ZJVY0ZRXZ0:VYZH2YJFU0ZJVYP:[Zj/2
^;Rh0/W9b@;IfMT2h^;R0^3Ph0;GemL2cLg=0/G5a@;4a<@2bLW90/o?c`;GemL2gmoO0^CTi0;Vi^H2
gmoO0/g=c@:e]KD2WinO0YBDU0:KVi/2[Zj^0/G5a@;EeMD2f=SH0]7Ad@;7alL2`l?30/[:bP;Fe]H4
h>3P0]?Cd`:l_;`2YZJV0YNGU`:FUYH00inOWj^[Zj^[Z`02];Bd0[Jf]P:c/k<2/;2`0[6a/@:f][H2
_kno0/S8b0;AdM42emOG0][JfP;IfMT2dM7A0[no_`:YZJT2UIFE0Xb<S0:DU9@2YjNW0[^k^`;7alL2
`/;20[:b/P:QXJ42VIVI0Z2PX0:a/K42`/;20/W9b@:o_kl2Zj^[0YFEU@:8R8P2RHV90YBDU0:SXj<2
/;2`0[Fe]@:f][H4]KFe0[Vi^@:o_kl2aLG50/K6aP;1`L42^;Rh0[2`/0:^[Zh2][Jf0/71`@;8b<P2
_[jn0YnOW`9aLG42A4A40R@T908N7Qh2:RXZ0Shn?P9<C4`2CDe=1Oooo`400001oooo000:oooo00<0
003oooooool00_ooo`030000oooooooo00?oool00`000?ooooooo`02oooo0T51@@92@T82CTi>0V][
J`:@T902/K6a0/30`0:h^;P2WinO0X61P@9ZJVX2HF5Q0VEUI@9aLG42P8200Xn?S`:MWId2ZZZZ0[Fe
]@:j^[X2_;bl0[Zj^P:e]KD2/K6a0Zb/[0:[Zj/2[Zj^0[Rh^0;6a/H2dm?C1=cLg0;EeMD2c/k>0/S8
b0;7alL2bl_;0/o?c`;EeMD2f][J1=oOg`;IfMT2b<S80[>c/`:PX:02UYJF0YbLW0:][Jd2`<300/k>
cP;@d=02bLW90/30`0:k^k/2`<300/[:bP;Cdm<2emOG0]7Ad@;4a<@2][Jf0Z^[Z`>UYJD2YjNW0ZRX
Z0:WYjL4YZJV0ZVYZ@:a/K42^k^k0/C4a0;;bl/4clo?0/c<c0;3`l<2][Jf0ZNWY`:JVYX2UiNG0Z6Q
X@:c/k<2a/K60/g=c@;3`l<2[Zj^0YRHV0:=SHd2UYJF0Zb/[0;2`/82c<c<0/?3``:/[:`2Ti>C0X>3
P`:2PX82S8b<0YbLW0:ZZZX2/k>c0[Ng]`:j^[X2_[jn0/?3``;8b<P2c<c<0/[:bP;2`/82]kNg0Zb/
[0:YZJT2/K6a0[fm_@;5aLD2_;bl0YfMW@9^KVh2?clo0QdM7@8D51@27QhN0S0`<08o?cl2@d=31Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`91@D42@T920Ti>CP9[Jf/2T92@0[6a/@;0`<02
^;Rh0YnOW`:1PH42JVYZ0V5QH@9UIFD2LG5a0X20P0:?Shl2WIfM0ZZZZP:e]KD2^[Zj0[bl_0:j^[X2
]KFe0[6a/@:/[:`2Zj^[0Zj^[P:h^;P2a/K60]?Cd`CLg=`2eMGE0/k>cP;8b<P2alO70/_;b`;?cll2
eMGE0][JfPCOgml2fMWI0/S8b0:c/k<2X:2P0YJFUP:LW9`2[Jf]0/30`0;>c/h2d=3@0/W9b@;0`<02
^k^k0/30`0;:b/X2dm?C0]OGe`;AdM42a<C40[Jf]P:[Zj/3YJFU0ZNWY`:XZ:P2YjNW1:JVYP:YZJT2
/K6a0[^k^`;4a<@2bl_;1<o?c`;<c<`2`l?30[Jf]P:WYjL2VYZJ0YNGU`:QXJ42/k>c0/K6aP;=cLd2
`l?30Zj^[P:HV9P2SHf=0YJFUP:/[:`2`/;20/c<c0;3`l<2[:b/0Y>CT`:3Ph<2PX:20Xb<S0:LW9`2
ZZZZ0[>c/`:g]kL2^[Zj0[jn_P;3`l<2b<S80/c<c0;:b/X2`/;20[Ng]`:/[:`2ZJVY0[6a/@:m_Kd2
aLG50[bl_0:MWId2KVi^0Slo?`8M7Ad251@D0QhN7P8`<302?clo0T=3@`Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2@D510T92@P9>CTh2JVYZ0Xf=S@:[Zj/2]kNg0ZZZZP:<S8`2K6a/15IF
EP9WIfL2OWin0YNGU`:YZJT2][Jf0/30`0;5aLD2alO70/G5a@;1`L42^k^k0[Jf]P:b/[82/;2`0[>c
/`:j^[X2`l?30/g=c@CCdm<2clo?0/[:bP;8b<P2bLW90/g=c@;Bd]82eMGE0]KFeP;EeMD2dM7A0/S8
b0:j^[X2ZZZZ0YbLW0:FUYH2W9bL0Z^[Z`:l_;`2a/K60/O7a`:o_kl2][Jf0[6a/@:d];@2^k^k0/C4
a0;:b/X2c<c<0/[:bP;5aLD2_[jn0[Jf]P03[Zj^YZJVYZJV00:PX:04WIfM0Z6QX@:WYjL2/;2`0[Zj
^P;2`/84b<S80/G5a@:o_kl2]kNg0Zf][@:TY:@2WYjN0YnOW`:ZZZX2^k^k0/_;b`;@d=02a<C40Zj^
[P:GUiL2S8b<0YJFUP:/[:`2`/;20/g=c@;6a/H2/;2`0YRHV0:5QHD2Ogmo0XB4Q0:>SXh2VYZJ0ZJV
YP:a/K42^[Zj0/?3``;9bLT4cLg=0/W9b@:o_kl2/k>c0ZNWY`:SXj<2ZJVY0[Bd]0:j^[X2/K6a0Y>C
T`9WIfL2>SXj0QXJ6P8B4Q8271`L0Rh^;P8m?Cd2@T921Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`91@D42@T920Ti>CP9ZJVX2SHf=0Z^[Z`:g]kL2ZZZZ0Xb<S09/K6`4EUIF0VMWI`9nOWh2
UiNG0ZVYZ@:f][H2`<300/G5a@;7alL2aLG50/71`@:k^k/2][Jf0[:b/P:`/;02/k>c0[Zj^P;3`l<2
cLg=1=?Cd`;?cll2b/[:0/S8b0;9bLT2cLg=0];BdP;EeMD2e]KF0]GEe@;AdM42b<S80[Zj^P:ZZZX2
W9bL0YJFUP:LW9`2Zj^[0[bl_0;6a/H2alO70[no_`:f][H2/K6a0[Bd]0:k^k/2a<C40/[:bP;<c<`2
b/[:0/G5a@:n_[h2][Jf00>^[ZjVYZJVYZH00Z2PX0BMWId2XJ6Q0ZNWY`:`/;02^[Zj0/;2`PC8b<P2
aLG50[no_`:g]kL2[Jf]0ZBTY0:NWYh2WinO0ZZZZP:k^k/2bl_;0]3@d0;4a<@2[Zj^0YNGU`:<S8`2
UYJF0Zb/[0;2`/82cLg=0/K6aP:`/;02V9RH0XF5Q@9oOgl2Q8B40Xj>SP:JVYX2YZJV0[6a/@:j^[X2
`l?30/W9b@C=cLd2bLW90[no_`:c/k<2YjNW0Z>SX`:YZJT2];Bd0[Zj^P:a/K42Ti>C0VMWI`8j>SX2
6QXJ0Q8B4P8L71`2;Rh^0Sdm?@92@T85oooo0@00007oool001Goool00`000?ooooooo`02oooo0TM7
A`92@T82AdM70U]KF`9hN7P2Ti>C0Z6QX@:KVi/2R8R80W=cL`9WIfL2Jf][0W]kN`:@T902Xj>S0[:b
/P:k^k/2`<300/?3``;2`/82_[jn0[Zj^P:e]KD2/k>c0[Bd]0:f][H2^KVi0[jn_P;2`/82alO70/W9
b@;8b<P2aLG50/;2`P;3`l<2bLW90]7Ad@;GemL2fMWI0]?Cd`;7alL2^KVi0Z^[Z`:OWil2UYJF0Y6A
T@:BTY82Vi^K0ZVYZ@:f][H2_[jn0[fm_@:e]KD2[:b/1:JVYP:/[:`2];Bd0[fm_@;6a/H2c<c<0]3@
d0;>c/h2aLG500>h^;RZZZZZZZX00Z2PX0:LW9`2WYjN0ZFUY@:][Jd2][Jf0[jn_P;5aLD2alO70/G5
a@:o_kl2]kNg0Zn_[`:XZ:P2Xj>S0Z:RXP:UYJD2/;2`0[fm_@;9bLT2cLg=0/C4a0:c/k<2Xj>S0Y^K
V`:RXZ82/k>c0/C4a0;=cLd2b<S80[Vi^@:VYZH2Ti>C0XF5Q@9mOGd2NWYj0WmoO`:<S8`2WYjN0[6a
/@;1`L42bLW90/[:bP;7alL2`<300[Jf]P:/[:`2XZ:R0YfMW@:OWil2YJFU0ZRXZ0:MWId2PH610UYJ
FP8d=3@26a/K0QPH608V9RH2>c/k0TY:BP9=CDd5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0TM7A`92@T82AdM70U]KF`9hN7P2Ti>C0Z6QX@:KVi/2R8R80W=cL`9WIfL2Jf][0W]kN`:@T902
Xj>S0[:b/P:k^k/2`<300/?3``;2`/82_[jn0[Zj^P:e]KD2/k>c0[Bd]0:f][H2^KVi0[jn_P;2`/82
alO70/W9b@;8b<P2aLG50/;2`P;3`l<2bLW90]7Ad@;GemL2fMWI0]?Cd`;7alL2^KVi0Z^[Z`:OWil2
UYJF0Y6AT@:BTY82Vi^K0ZVYZ@:f][H2_[jn0[fm_@:e]KD2[:b/1:JVYP:/[:`2];Bd0[fm_@;6a/H2
c<c<0]3@d0;>c/h2aLG500>h^;RZZZZZZZX00Z2PX0:LW9`2WYjN0ZFUY@:][Jd2][Jf0[jn_P;5aLD2
alO70/G5a@:o_kl2]kNg0Zn_[`:XZ:P2Xj>S0Z:RXP:UYJD2/;2`0[fm_@;9bLT2cLg=0/C4a0:c/k<2
Xj>S0Y^KV`:RXZ82/k>c0/C4a0;=cLd2b<S80[Vi^@:VYZH2Ti>C0XF5Q@9mOGd2NWYj0WmoO`:<S8`2
WYjN0[6a/@;1`L42bLW90/[:bP;7alL2`<300[Jf]P:/[:`2XZ:R0YfMW@:OWil2YJFU0ZRXZ0:MWId2
PH610UYJFP8d=3@26a/K0QPH608V9RH2>c/k0TY:BP9=CDd5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0U9BDP94A4@2?Shn0TQ8B09LG5`2Lg=c0XF5Q@:=SHd4Shn?0Y2@T0:DU9@2VYZJ0Z6Q
X@:VYZH2Zj^[0Zn_[`:a/K42/[:b0Zn_[`:[Zj/4YjNW0Z^[Z`:c/k<2^k^k0/71`@C3`l<2`/;20[no
_`:k^k/2^KVi0[Ng]`:l_;`2alO70]CDe0COgml2d=3@0[Rh^0:NWYh2R8R80WinOP9mOGd2PX:20Xb<
S0:IVIT2YZJV0[6a/@:g]kL2]KFe0Zb/[0:RXZ82Vi^K0YVIV@:MWId2YJFU0[6a/@:o_kl2c<c<0]OG
e`;IfMT2dM7A00?2`/:b/[:b/[800ZJVYP:RXZ82YZJV0Zj^[P:g]kL2_kno0/G5a@C8b<P2a<C40[bl
_0:c/k<2Zj^[0ZJVYP:UYJD2YZJV0Z^[Z`:b/[82^k^k0/?3``;7alL2a<C40[fm_@:f][H2/k>c0[Ng
]`:o_kl2a/K60/_;b`;:b/X2a<C40[Vi^@:XZ:P2T92@0WMgM`9TI6@2GUiN0VYZJP:4Q8@2XZ:R0[Zj
^P;4a<@2`l?30[bl_0:c/k<2Zj^[0ZBTY0:LW9`2UiNG0YFEU@:DU9@2T92@0XB4Q09[Jf/2Bd];0Rh^
;P8O7al292@T0STi>@9@D502H61P0UiNGPGoool100000Oooo`005Oooo`800003oooo0U9BDP94A4@2
?Shn0TQ8B09LG5`2Lg=c0XF5Q@:=SHd4Shn?0Y2@T0:DU9@2VYZJ0Z6QX@:VYZH2Zj^[0Zn_[`:a/K42
/[:b0Zn_[`:[Zj/4YjNW0Z^[Z`:c/k<2^k^k0/71`@C3`l<2`/;20[no_`:k^k/2^KVi0[Ng]`:l_;`2
alO70]CDe0COgml2d=3@0[Rh^0:NWYh2R8R80WinOP9mOGd2PX:20Xb<S0:IVIT2YZJV0[6a/@:g]kL2
]KFe0Zb/[0:RXZ82Vi^K0YVIV@:MWId2YJFU0[6a/@:o_kl2c<c<0]OGe`;IfMT2dM7A00?2`/:b/[:b
/[800ZJVYP:RXZ82YZJV0Zj^[P:g]kL2_kno0/G5a@C8b<P2a<C40[bl_0:c/k<2Zj^[0ZJVYP:UYJD2
YZJV0Z^[Z`:b/[82^k^k0/?3``;7alL2a<C40[fm_@:f][H2/k>c0[Ng]`:o_kl2a/K60/_;b`;:b/X2
a<C40[Vi^@:XZ:P2T92@0WMgM`9TI6@2GUiN0VYZJP:4Q8@2XZ:R0[Zj^P;4a<@2`l?30[bl_0:c/k<2
Zj^[0ZBTY0:LW9`2UiNG0YFEU@:DU9@2T92@0XB4Q09[Jf/2Bd];0Rh^;P8O7al292@T0STi>@9@D502
H61P0UiNGPCoool200000Oooo`005Oooo`030000oooooooo00;oool2GUiN0TY:BP8k>c/2>SXj0TE5
A@9FEUH2K6a/0XB4Q0:KVi/2[jn_0[bl_0:o_kl2^KVi0Zn_[`:UYJD2WinO19jNWP:MWId2VIVI0YBD
U0:BTY82UIFE0Z2PX0:_[jl2_[jn0/O7a`;9bLT2a<C40[jn_P:g]kL2/K6a0Zj^[P:][Jd2];Bd0/C4
a0;GemL2iNGU0^CTi0;>c/h2Zj^[0XJ6QP9ZJVX2H61P0VEUI@9bLW82Ph>30YBDU0:RXZ82[Jf]0[:b
/P:^[Zh2YJFU0YZJVP:BTY82Shn?0Y:BTP:JVYX2YjNW0[Rh^0;:b/X2fMWI0]oOg`;IfMT00l_;bk^k
^k^k^`02[jn_0Z^[Z`:_[jl2]kNg0[no_`;5aLD2bLW90/_;b`;9bLT2`l?30[Zj^P:a/K42ZJVY0ZJV
YP:WYjL2Zj^[0Zn_[`:c/k<2]kNg0[^k^`:o_kl2`l?30/K6aP;9bLT4bl_;0/[:bPC7alL2b/[:0/g=
c@;;bl/2^k^k0YbLW09dM7@2D51@0T10@09<C4`2K6a/0Y:BTP:`/;02_;bl0[Rh^0:_[jl2YJFU0YnO
W`:LW9`2V9RH0Y:BTP:;Rh/2PX:20WMgM`9WIfL2DU9B0S/k>`8X:2P292@T0S8b<P9;Bd/2IVIV0WEe
M@9`L705oooo0@00007oool001Goool00`000?ooooooo`02oooo0UiNGP9:BTX2>c/k0SXj>P95ADD2
EUIF0Va/K0:4Q8@2Vi^K0Zn_[`:l_;`2_kno0[Vi^@:_[jl2YJFU0YnOW`BNWYh2WIfM0YVIV@:DU9@2
TY:B0YFEU@:PX:02[jn_0[jn_P;7alL2bLW90/C4a0:n_[h2]kNg0[6a/@:^[Zh2[Jf]0[Bd]0;4a<@2
emOG0^GUi@;Ti>@2c/k>0Z^[Z`:6QXH2JVYZ0V1PH09UIFD2LW9b0X>3P`:DU9@2XZ:R0Zf][@:b/[82
[Zj^0ZFUY@:JVYX2TY:B0Xn?S`:BTY82VYZJ0ZNWY`:h^;P2b/[:0]WIf@;Ogml2fMWI00?;bl^k^k^k
^k/00Zn_[`:[Zj/2[jn_0[Ng]`:o_kl2aLG50/W9b@;;bl/2bLW90/?3``:j^[X2/K6a0ZVYZ@:VYZH2
YjNW0Z^[Z`:_[jl2/k>c0[Ng]`:k^k/2_kno0/?3``;6a/H2bLW91<_;b`;:b/X4alO70/[:bP;=cLd2
bl_;0[^k^`:LW9`2M7Ad0U1@D090@402C4a<0Va/K0:BTY82/;2`0[bl_0:h^;P2[jn_0ZFUY@:OWil2
W9bL0YRHV0:BTY82Rh^;0X:2PP9gMgL2IfMW0U9BDP8k>c/2:2PX0R@T908b<S82Bd];0VIVIP9eMGD2
L71`1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9ZJVX2EeMG0TM7A`8o?cl2@T920Te=
C@9RHV82PX:20ZJVYP;7alL4fm_K0/c<c0:f][H2XJ6Q0YBDU0:@T902Shn?0Xf=S@:8R8P2Ph>30X61
P@:8R8P2UiNG0Zb/[0:o_kl2bl_;0/c<c0;6a/H2_Kfm0[>c/`:/[:`2Z:RX0ZVYZ@:a/K42`l?30]SH
f0;WinL2i^KV0/k>cP:UYJD2Ng]k0U]KF`9?Cdl2EEEE0V=SH`9gMgL2Rh^;0YbLW0:ZZZX2/;2`0Zf]
[@:SXj<2UiNG0Xf=S@:9RHT2SHf=0YFEU@:RXZ82/k>c0/C4a0;De=@2g=cL0]WIf@03d=3@`l?3`l?3
00:i^KT2];Bd0[Fe]@:i^KT2_[jn0/?3``;7alL2b/[:0/W9b@;3`l<2^KVi0Zn_[`:XZ:P2YjNW0ZZZ
ZP:`/;02]KFe1[Ng]`:j^[X2`/;20/c<c0;EeMD2f][J0]OGe`;?cll2a/K60/;2`P;7alL2d=3@0]CD
e0;7alL2Xj>S0W=cL`97AdL2<C4a0S/k>`9NGUh2QhN70ZNWY`:c/k<2[Zj^0Z:RXP:HV9P2UIFE0YNG
U`:FUYH2T92@0X>3P`9bLW82HF5Q0Ti>CP8l?3`2;Bd]0R<S8`8V9RH2>CTi0U=CD`9]KFd2O7al0WYj
NPGoool100000Oooo`005Oooo`030000oooooooo00;oool2JVYZ0UMGE`97AdL2?clo0T92@P9=CDd2
HV9R0X:2PP:VYZH2alO71=_Kf`;<c<`2][Jf0Z6QX@:DU9@2T92@0Xn?S`:=SHd2R8R80X>3P`:1PH42
R8R80YNGU`:/[:`2_kno0/_;b`;<c<`2a/K60[fm_@:c/k<2[:b/0ZRXZ0:YZJT2/K6a0/?3``;Hf=P2
inOW0^KViP;>c/h2YJFU0W]kN`9KFe/2Cdm?0UEEE@9SHf<2MgMg0X^;R`:LW9`2ZZZZ0[2`/0:][Jd2
Xj>S0YNGU`:=SHd2RHV90Xf=S@:EUID2XZ:R0[>c/`;4a<@2e=CD0]cLg0;IfMT00m3@d<?3`l?3``02
^KVi0[Bd]0:e]KD2^KVi0[jn_P;3`l<2alO70/[:bP;9bLT2`l?30[Vi^@:_[jl2Z:RX0ZNWY`:ZZZX2
/;2`0[Fe]@Jg]kL2^[Zj0/;2`P;<c<`2eMGE0][JfP;GemL2clo?0/K6aP;2`/82alO70]3@d0;De=@2
alO70Z>SX`9cLg<2AdM70S4a<@8k>c/2GUiN0XN7Q`:WYjL2/k>c0Zj^[P:RXZ82V9RH0YFEU@:GUiL2
UYJF0Y2@T0:3Ph<2LW9b0V5QH@9>CTh2?3`l0Rd];@8S8b<29RHV0STi>@9CDe<2KFe]0WalO09jNWX5
oooo0@00007oool001Goool00`000?ooooooo`02oooo0W9bLP9/K6`2IFEU0UmOG`9LG5`2H61P0Vm_
K`::RXX2[Jf]0/g=c@;Ogml2gMgM0/[:bP:a/K42VYZJ0Xj>SPB:RXX2RHV90XB4Q09oOgl2OGem0X>3
P`:CTi<2Z:RX0[fm_@;:b/X2cLg=0/S8b0;0`<02]kNg0[2`/0B/[:`2/k>c0/C4a0;GemL2i^KV0^GU
i@;?cll2ZZZZ0X>3P`9SHf<2DU9B0U1@D09HF5P2J6QX0WemO@:DU9@2Z:RX0[Bd]0:b/[82YjNW0YRH
V0:=SHd2RHV90Xj>SP:GUiL2Xj>S0[2`/0:l_;`2b<S80/o?c`;AdM400lk>c/W9bLW9b@02`/;20[^k
^`:e]KD2/K6a0[2`/0:e]KD2_Kfm0/C4a0;7alL2`L710[Ng]`:[Zj/2YJFU0ZJVYP:^[Zh2]kNg0[jn
_P:o_kl2_Kfm0[Zj^P:k^k/2`/;20/g=c@;GemL2fm_K0]KFeP;<c<`2`<300[Zj^P;0`<02b/[:0]3@
d0;6a/H2YJFU0WQhN09>CTh2=cLg0Slo?`9NGUh2Q8B40Z6QX@:ZZZX2Y:BT0YVIV@B@T902U9BD0YFE
U@:?Shl2Ogmo0VUYJ@9BDU82?Cdm0Rd];@8S8b<27alO0RDU9@8d=3@2BTY:0UmOG`9_Kfl2M7Ad1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`9bLW82K6a/0VEUI@9OGel2G5aL0V1PH09_Kfl2
RXZ:0Zf][@;=cLd2gmoO0]gMg@;:b/X2/K6a0YZJVP:>SXh4RXZ:0XV9R@:4Q8@2Ogmo0WemO@:3Ph<2
Ti>C0ZRXZ0:m_Kd2b/[:0/g=c@;8b<P2`<300[Ng]`:`/;04[:b/0[>c/`;4a<@2emOG0^KViP;UiND2
clo?0ZZZZP:3Ph<2Hf=S0U9BDP9@D502F5QH0VQXJ09mOGd2U9BD0ZRXZ0:d];@2/[:b0ZNWY`:HV9P2
SHf=0XV9R@:>SXh2UiNG0Z>SX`:`/;02_;bl0/S8b0;?cll2dM7A00?>c/k9bLW9bLT00/;2`P:k^k/2
]KFe0[6a/@:`/;02]KFe0[fm_@;4a<@2alO70/71`@:g]kL2Zj^[0ZFUY@:VYZH2[Zj^0[Ng]`:n_[h2
_kno0[fm_@:j^[X2^k^k0/;2`P;=cLd2emOG0]_Kf`;Fe]H2c<c<0/30`0:j^[X2`<300/[:bP;@d=02
a/K60ZFUY@9hN7P2CTi>0SLg=`8o?cl2GUiN0XB4Q0:QXJ42ZZZZ0ZBTY0:IVIT4T92@0YBDU0:EUID2
Shn?0WmoO`9YJFT2DU9B0Sdm?@8];Bd28b<S0QlO7`8U9BD2=3@d0TY:BP9OGel2Kfm_0WAdM0Goool1
00000Oooo`005Oooo`030000oooooooo00;oool2MgMg0XJ6QP:>SXh2Shn?0X^;R`:7QhL2Rh^;0Y^K
V`:`/;02aLG50/o?c`;:b/X2^;Rh0Z>SX`:BTY84RXZ:18f=S@:9RHT2Ph>30X61P@:5QHD2Ti>C0ZJV
YP:i^KT2alO70/c<c0;:b/X2aLG50[no_`:i^KT2]KFe0[Bd]0:j^[X2a/K60]KFePCQhN42d];B0[Ng
]`:IVIT2Ng]k0VEUI@9GEeL2DU9B0UUIF@9_Kfl2Rh^;0ZRXZ0:j^[X2^k^k0Zn_[`:NWYh2T92@0Xj>
SP:DU9@2WYjN0ZRXZ0:_[jl2];Bd0[Rh^0:m_Kd2`l?300?9bLW<c<c<c<`00/S8b0:n_[h2/;2`0Z:R
XP:LW9`2XJ6Q0Zf][@:k^k/2`/;20[no_`:d];@2Z:RX0Z:RXP:VYZH2/[:b0[jn_P;7alL2b/[:0/O7
a`;3`l<2`L710/C4a0;9bLT2clo?0]3@d0;;bl/2`L710[Ng]`:c/k<2]kNg0[jn_P;3`l<2^k^k0Z:R
XP:0P802GemO0Ti>CP9CDe<2JVYZ0XJ6QP:KVi/2XJ6Q0Y^KV`:BTY82S8b<0Xf=S@:BTY82UIFE0Xj>
SP9mOGd2I6AT0TY:BP8d=3@29BDU0QlO7`8N7Qh28R8R0RXZ:P8e=CD2@d=30UADE09UIFD5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0WMgM`:6QXH2SXj>0Xn?S`:;Rh/2QhN70X^;R`:KVi/2
/;2`0/G5a@;?cll2b/[:0[Rh^0:SXj<2TY:B18Z:RPB=SHd2RHV90X>3P`:1PH42QHF50Y>CT`:VYZH2
^KVi0/O7a`;<c<`2b/[:0/G5a@:o_kl2^KVi0[Fe]@:d];@2^[Zj0/K6aP;Fe]H4hN7Q0];BdP:g]kL2
VIVI0W]kN`9UIFD2EeMG0U9BDP9IFET2Kfm_0X^;R`:XZ:P2^[Zj0[^k^`:_[jl2WYjN0Y2@T0:>SXh2
U9BD0YjNWP:XZ:P2[jn_0[Bd]0:h^;P2_Kfm0/?3``03bLW9c<c<c<c<00;8b<P2_[jn0[2`/0:RXZ82
W9bL0Z6QX@:][Jd2^k^k0/;2`P:o_kl2];Bd0ZRXZ0:RXZ82YZJV0[:b/P:n_[h2alO70/[:bP;7alL2
`l?30/71`@;4a<@2bLW90/o?c`;@d=02bl_;0/71`@:g]kL2/k>c0[Ng]`:n_[h2`l?30[^k^`:RXZ82
P8200UmOG`9>CTh2De=C0VYZJP:6QXH2Vi^K0Z6QX@:KVi/2TY:B0Xb<S0:=SHd2TY:B0YFEU@:>SXh2
OGem0VATI09:BTX2=3@d0RDU9@8O7al27QhN0R8R8P8Z:RX2=CDe0T=3@`9DE5@2IFEU1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9jNWX2WYjN0[Rh^0;1`L42_Kfm0[>c/`:/[:`2[Jf]0[>c
/`:i^KT2^;Rh0Zn_[`:QXJ42TY:B0XR8R0:7QhL2Rh^;0Y:BTP:DU9@2TI6A0XZ:RP:5QHD2QhN70Y:B
TP:TY:@2][Jf0/C4a0;;bl/2cLg=0/_;b`;8b<P2a<C40/30`0:o_kl2`L710/W9b@;Cdm<2fm_K0]gM
g@;Fe]H2alO70[>c/`:JVYX2OWin0VATI09BDU82D51@0V=SH`:4Q8@2Z:RX0/30`0;2`/82]KFe0Z:R
XP:EUID2U9BD0YfMW@:XZ:P2/;2`0[6a/@:][Jd2ZZZZ0Z^[Z`:e]KD00l;2`/c<c<c<c002c<c<0[no
_`:ZZZX2U9BD0XV9R@:=SHd2WIfM0Zn_[`:k^k/2^[Zj0[6a/@:WYjL2Xj>S0ZVYZ@:f][H2a<C40/o?
c`;Bd]82dM7A0/c<c0;8b<P2a/K60/G5a@;3`l<2`<300[^k^`:d];@2[Zj^0Zb/[0:^[Zh2/[:b0[>c
/`:^[Zh2WYjN0XV9R@9dM7@2JFUY0Va/K09iNGT2RHV90YFEU@:HV9P2Ti>C0Xb<S0:8R8P2Rh^;0Y2@
T0:BTY82S8b<0WYjNP9QHF42ADE50S0`<08T92@2820P0R8R8P8T92@48b<S0RPX:08h>3P2EEEE1Ooo
o`400001oooo000Eoooo0P0000?oool2NWYj0YjNWP:h^;P2`L710[fm_@:c/k<2[:b/0Zf][@:c/k<2
^KVi0[Rh^0:_[jl2XJ6Q0Y:BTP:8R8P2QhN70X^;R`:BTY82U9BD0Y6AT@::RXX2QHF50XN7Q`:BTY82
Y:BT0[Jf]P;4a<@2bl_;0/g=c@;;bl/2b<S80/C4a0;0`<02_kno0/71`@;9bLT2dm?C0]_Kf`;MgMd2
e]KF0/O7a`:c/k<2VYZJ0WinOP9TI6@2DU9B0U1@D09SHf<2Q8B40ZRXZ0;0`<02`/;20[Fe]@:RXZ82
UIFE0YBDU0:MWId2Z:RX0[2`/0:a/K42[Jf]0ZZZZP:[Zj/2]KFe00?2`/;<c<c<c<`00/c<c0:o_kl2
ZZZZ0YBDU0:9RHT2SHf=0YfMW@:_[jl2^k^k0[Zj^P:a/K42YjNW0Z>SX`:YZJT2][Jf0/C4a0;?cll2
d];B0]7Ad@;<c<`2b<S80/K6aP;5aLD2`l?30/30`0:k^k/2];Bd0Zj^[P:/[:`2[Zj^0[:b/P:c/k<2
[Zj^0YjNWP:9RHT2M7Ad0VUYJ@9/K6`2NGUi0XV9R@:EUID2V9RH0Y>CT`:<S8`2R8R80X^;R`:@T902
TY:B0Xb<S09jNWX2HF5Q0TE5A@8`<30292@T0R0P808R8R8292@T12<S8`8X:2P2>3Ph0UEEE@Coool2
00000Oooo`005Oooo`030000oooooooo00;oool2P8200[6a/@;Fe]H2iNGU0^;RhP;EeMD2alO70[bl
_0:e]KD2[Jf]0ZBTY0:GUiL2RXZ:0X20P09lO7`2PH610X^;R`:EUID2VIVI0YFEU@:<S8`2Ph>30X:2
PP:=SHd2WinO0[Bd]0;5aLD2clo?0];BdP;AdM42clo?0/_;b`;8b<P2aLG50/K6aP;:b/X2d=3@0]KF
ePCJf]X2eMGE0/[:bP:g]kL2VIVI0WMgM`9KFe/2DE5A0V1PH0:1PH42YZJV0[no_`;2`/82]KFe0Z>S
X`:GUiL2VIVI0ZJVYP:d];@2_;bl0[Rh^0:][Jd2Xj>S0Z2PX0:ZZZX00kZj^/S8b<S8b002c<c<0[jn
_P:UYJD2S8b<0WinOP:0P802Shn?0Z:RXP:`/;02];Bd0[2`/0:/[:`2Zj^[0[6a/@:l_;`2alO70]3@
d0CEeMD2e=CD0/o?c`;8b<P2`<300[Ng]`:`/;02Zj^[0ZRXZ0BWYjL2Z:RX0ZVYZ@:XZ:P2Y:BT0Y^K
V`:@T902Q8B417emO@:2PX82RHV918j>SP::RXX2QXJ60XB4Q0:6QXH2Rh^;0Xb<S0:5QHD2Lg=c0UYJ
FP90@402;Bd]0RHV9P8Y:BT2;bl_0S8b<P8[:b/28B4Q0QhN7P8[:b/2Cdm?1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`:0P802/K6a0]KFeP;UiND2h^;R0]GEe@;7alL2_;bl0[Fe]@:][Jd2
Y:BT0YNGU`::RXX2P8200WalO0:1PH42Rh^;0YFEU@:IVIT2UIFE0Xb<S0:3Ph<2PX:20Xf=S@:OWil2
];Bd0/G5a@;?cll2d];B0]7Ad@;?cll2bl_;0/S8b0;5aLD2a/K60/[:bP;@d=02e]KF1=[JfP;EeMD2
b/[:0[Ng]`:IVIT2MgMg0U]KF`9ADE42H61P0X61P@:VYZH2_kno0/;2`P:e]KD2Xj>S0YNGU`:IVIT2
YZJV0[Bd]0:l_;`2^;Rh0Zf][@:SXj<2X:2P0ZZZZP03^[Zjb<S8b<S800;<c<`2_[jn0ZFUY@:<S8`2
OWin0X20P0:?Shl2XZ:R0[2`/0:d];@2/;2`0Zb/[0:[Zj/2/K6a0[bl_0;7alL2d=3@1=GEe@;De=@2
clo?0/S8b0;0`<02]kNg0[2`/0:[Zj/2Z:RX1:NWY`:XZ:P2ZJVY0ZRXZ0:TY:@2Vi^K0Y2@T0:4Q8@4
OGem0X:2PP:9RHT4SXj>0XZ:RP:6QXH2Q8B40XJ6QP:;Rh/2S8b<0XF5Q@9cLg<2FUYJ0T10@08];Bd2
9RHV0RTY:@8_;bl2<S8b0R/[:`8Q8B427QhN0R/[:`9?Cdl5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0XR8R0:j^[X2hN7Q0_7al@;_knl2h^;R0]7Ad@;2`/82]KFe0ZNWY`:HV9P2RHV90W]k
N`9bLW82LG5a0WUiN@:7QhL2UIFE0YZJVP:BTY82Ph>30WEeM@9bLW82Ogmo0YVIV@:e]KD2bl_;0]OG
e`;Hf=P2eMGE0]3@d0;;bl/2alO71<G5a@;8b<P2cLg=0]?Cd`;IfMT4gMgM0]WIf@;9bLT2[Zj^0Xb<
S09_Kfl2HF5Q0VYZJP:3Ph<2XJ6Q0[Fe]@:g]kL2Zj^[0YbLW0:EUID2WIfM0Zj^[P;0`<02bLW90/C4
a0:f][H2YjNW0Z2PX0:VYZH00kBd]<;2`/;2`P02aLG50[Vi^@:TY:@2SXj>0WmoO`9mOGd2QXJ60Y>C
T`:PX:02ZZZZ0[6a/@:g]kL2_;bl0/30`0;3`l<2aLG50/S8b0;=cLd2d];B0]GEe@;Cdm<2b/[:0[fm
_@:_[jl2Y:BT19nOW`:RXZ82YJFU1:NWY`:VYZH2XZ:R0Y^KV`:CTi<2RXZ:0X>3P`B1PH42Q8B40XJ6
QP:5QHD2PX:20WmoO`9nOWh2Ogmo1861P@9hN7P2IVIV0Ti>CP8f=SH2:BTY0R/[:`8i>CT2B4Q80U1@
D098B4P2>SXj0S0`<08g=cL2F5QH1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:8R8P2
^[Zj0^7Qh@;alO42kno_0^;RhP;AdM42`/;20[Fe]@:WYjL2V9RH0XV9R@9kNg/2LW9b0W5aL@9iNGT2
QhN70YFEU@:JVYX2TY:B0X>3P`9eMGD2LW9b0WmoO`:IVIT2]KFe0/_;b`;GemL2f=SH0]GEe@;@d=02
bl_;0/O7a`C5aLD2b<S80/g=c@;Cdm<2fMWI1=gMg@;IfMT2bLW90Zj^[P:<S8`2Kfm_0V5QH@9ZJVX2
Ph>30Z6QX@:e]KD2]kNg0Z^[Z`:LW9`2UIFE0YfMW@:^[Zh2`<300/W9b@;4a<@2][Jf0ZNWY`:PX:02
YZJV00>d];C2`/;2`/800/G5a@:i^KT2Y:BT0Xj>SP9oOgl2OGem0XJ6QP:CTi<2X:2P0ZZZZP:a/K42
]kNg0[bl_0;0`<02`l?30/G5a@;8b<P2cLg=0];BdP;EeMD2dm?C0/[:bP:m_Kd2[jn_0ZBTY0BOWil2
XZ:R0ZFUY@BWYjL2YZJV0Z:RXP:KVi/2Ti>C0XZ:RP:3Ph<4PH610XB4Q0:6QXH2QHF50X:2PP9oOgl2
OWin0WmoO`B1PH42N7Qh0VIVIP9>CTh2=SHf0RTY:@8[:b/2>CTi0TQ8B09@D502B4Q80SXj>P8`<302
=cLg0UQHF0Goool100000Oooo`005Oooo`030000oooooooo00;oool2TY:B0[^k^`;Jf]X2j>SX0^KV
iP;Lg=`2c/k>0/30`0:c/k<2YJFU0YFEU@:4Q8@2M7Ad0VUYJ@9WIfL2LG5a0X:2PP:ATI42UYJF0XZ:
RP9cLg<2H61P0U]KF`9^KVh2TI6A0[Ng]`;EeMD2hN7Q0]oOg`;GemL2cLg=0/K6aP;2`/82`<300/71
`@;5aLD2bl_;0];BdP;Jf]X2gmoO0^7Qh@;MgMd2dM7A0[bl_0:QXJ42R8R80WUiN@9kNg/2R8R80YVI
V@:UYJD2Xj>S0YZJVPB@T902WIfM0[>c/`;9bLT2eMGE0];BdP;4a<@2];Bd0ZVYZ@:ZZZX00k6a/KVi
^KVi^@02^[Zj0[>c/`:UYJD2UYJF0XZ:RP:4Q8@2Ph>30XJ6QP:@T902X:2P0[>c/`;5aLD4dM7A0/W9
b@:o_kl2^[Zj0[no_`;9bLT2d];B0]GEe@;;bl/2^[Zj0ZVYZ@:LW9`2UiNG0YZJVP:OWil2YJFU0ZVY
Z@:[Zj/2ZZZZ0ZJVYP:MWId2TY:B0XN7Q`9mOGd4NGUi0W]kN`AmOGd2Ng]k0WQhN09fMWH4MGEe0W9b
LP9XJ6P2EUIF0Slo?`8/;2`29RHV0S<c<`9<C4`2IfMW0WMgM`9cLg<2Hf=S15EEE@9/K6`5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0Y:BTP:k^k/2f][J0^SXj0;Vi^H2g=cL0/k>cP;0`<02
/k>c0ZFUY@:EUID2Q8B40WAdM09YJFT2IfMW0W5aL@:2PX82TI6A0YJFUP::RXX2Lg=c0V1PH09KFe/2
KVi^0Y6AT@:g]kL2eMGE0^7Qh@;Ogml2emOG0/g=c@;6a/H2`/;20/30`0;1`L42aLG50/_;b`;Bd]82
f][J0]oOg`;QhN42gMgM0]7Ad@:l_;`2XJ6Q0XR8R09iNGT2Ng]k0XR8R0:IVIT2YJFU0Z>SX`:JVYX4
T92@0YfMW@:c/k<2bLW90]GEe@;Bd]82a<C40[Bd]0:YZJT2ZZZZ00>a/K6i^KVi^KT00[Zj^P:c/k<2
YJFU0YJFUP::RXX2Q8B40X>3P`:6QXH2T92@0Z2PX0:c/k<2aLG51=7Ad@;9bLT2_kno0[Zj^P:o_kl2
bLW90];BdP;EeMD2bl_;0[Zj^P:YZJT2W9bL0YNGU`:JVYX2WinO0ZFUY@:YZJT2Zj^[0ZZZZP:VYZH2
WIfM0Y:BTP:7QhL2OGem17UiN@9kNg/4OGem0W]kN`9hN7P2MWIf17EeM@9bLW82J6QX0UIFEP8o?cl2
;2`/0RHV9P8c<c<2C4a<0VMWI`9gMgL2Lg=c0V=SH`AEEED2K6a/1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`:IVIT2];Bd0/W9b@;Bd]82dM7A0/[:bP;1`L42]kNg0Zj^[P:SXj<2UIFE0XB4
Q09bLW82IFEU0V9RHP9/K6`2OGem0Xf=S@:@T902Ogmo0VATI09;Bd/2ATI60UiNGP::RXX2^[Zj0]gM
g@;[jn/2iNGU0]SHf0;9bLT2`<300[bl_0:k^k/2_Kfm0/;2`P;:b/X2dm?C0]_Kf`COgml2fm_K0]7A
d@;2`/82/K6a0YnOW`:BTY82SHf=0Xj>SP:BTY82Ti>C0Xn?S`:8R8P2Q8B40XZ:RP:LW9`2][Jf0/k>
cPCNg]h2dm?C0/C4a0:g]kL2/K6a00>`/;2_[jn_[jl00Zn_[`:/[:`2Z:RX0Z>SX`:KVi/2T92@0XF5
Q@9nOWh2Ph>30YNGU`:d];@2dM7A0^;RhP;Ng]h2cLg=0[Vi^@:][Jd2/K6a0[no_`;>c/h2eMGE0/c<
c0:h^;P2Y:BT0YFEU@:ATI42UYJF0YjNWP:WYjL2[Jf]1;2`/0:ZZZX2WinO0Y2@T0:1PH42MGEe16m_
K`9bLW82MGEe0WIfMP9dM7@2LG5a0Ve]K@9[Jf/2J6QX0V=SH`9IFET2AdM70S@d=08V9RH2:2PX0Shn
?P9RHV82QXJ60YfMW@:NWYh2Shn?0WemO@9eMGD2P8201Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`:IVIT2];Bd0/W9b@;Bd]82dM7A0/[:bP;1`L42]kNg0Zj^[P:SXj<2UIFE0XB4Q09bLW82
IFEU0V9RHP9/K6`2OGem0Xf=S@:@T902Ogmo0VATI09;Bd/2ATI60UiNGP::RXX2^[Zj0]gMg@;[jn/2
iNGU0]SHf0;9bLT2`<300[bl_0:k^k/2_Kfm0/;2`P;:b/X2dm?C0]_Kf`COgml2fm_K0]7Ad@;2`/82
/K6a0YnOW`:BTY82SHf=0Xj>SP:BTY82Ti>C0Xn?S`:8R8P2Q8B40XZ:RP:LW9`2][Jf0/k>cPCNg]h2
dm?C0/C4a0:g]kL2/K6a00>`/;2_[jn_[jl00Zn_[`:/[:`2Z:RX0Z>SX`:KVi/2T92@0XF5Q@9nOWh2
Ph>30YNGU`:d];@2dM7A0^;RhP;Ng]h2cLg=0[Vi^@:][Jd2/K6a0[no_`;>c/h2eMGE0/c<c0:h^;P2
Y:BT0YFEU@:ATI42UYJF0YjNWP:WYjL2[Jf]1;2`/0:ZZZX2WinO0Y2@T0:1PH42MGEe16m_K`9bLW82
MGEe0WIfMP9dM7@2LG5a0Ve]K@9[Jf/2J6QX0V=SH`9IFET2AdM70S@d=08V9RH2:2PX0Shn?P9RHV82
QXJ60YfMW@:NWYh2Shn?0WemO@9eMGD2P8201Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`:GUiL2YZJV0[6a/@:f][H2]KFe0[6a/@:][Jd2ZJVY0ZJVYP:QXJ42UiNG0XR8R09hN7P2JFUY0VAT
I09[Jf/2NWYj0XR8R0:9RHT2MgMg0UYJFP91@D42?Shn0UUIF@:9RHT2_;bl0^;RhP;_knl2j>SX0]OG
e`;7alL2_Kfm0[Zj^P:i^KT2_;bl0/?3``;;bl/2eMGE0]cLg0;MgMd2fMWI0];BdP;9bLT2_kno0[Jf
]P:[Zj/2XZ:R0YVIV@:BTY82SHf=0XR8R0:1PH44OGem0XJ6QP:IVIT2/k>c0/c<c0;MgMd2h^;R0]cL
g0;AdM42a<C40[Vi^@03/;2`ZJVYZJVY00:UYJD2YjNW0Zb/[0:a/K42[jn_0Z:RXP:@T902P8200Wmo
O`:BTY82/[:b0]?Cd`;UiND2h>3P0/_;b`:c/k<2Y:BT0ZVYZ@:j^[X2c<c<0]?Cd`;:b/X2]KFe0YnO
W`:@T902SHf=0YFEU@:OWil2ZJVY0[2`/0:b/[82/K6a0Z^[Z`:OWil2Shn?0WmoO`9bLW84K6a/0Vm_
K`9aLG42L71`0Ve]K@9XJ6P2Hf=S0UmOG`9LG5`2F5QH0U1@D092@T82<c<c0R/[:`8b<S82CDe=0WEe
M@:MWId2]kNg0[Zj^P:/[:`2VIVI0X^;R`:<S8`5oooo0@00007oool001Goool200000oooo`:GUiL2
YZJV0[6a/@:f][H2]KFe0[6a/@:][Jd2ZJVY0ZJVYP:QXJ42UiNG0XR8R09hN7P2JFUY0VATI09[Jf/2
NWYj0XR8R0:9RHT2MgMg0UYJFP91@D42?Shn0UUIF@:9RHT2_;bl0^;RhP;_knl2j>SX0]OGe`;7alL2
_Kfm0[Zj^P:i^KT2_;bl0/?3``;;bl/2eMGE0]cLg0;MgMd2fMWI0];BdP;9bLT2_kno0[Jf]P:[Zj/2
XZ:R0YVIV@:BTY82SHf=0XR8R0:1PH44OGem0XJ6QP:IVIT2/k>c0/c<c0;MgMd2h^;R0]cLg0;AdM42
a<C40[Vi^@03/;2`ZJVYZJVY00:UYJD2YjNW0Zb/[0:a/K42[jn_0Z:RXP:@T902P8200WmoO`:BTY82
/[:b0]?Cd`;UiND2h>3P0/_;b`:c/k<2Y:BT0ZVYZ@:j^[X2c<c<0]?Cd`;:b/X2]KFe0YnOW`:@T902
SHf=0YFEU@:OWil2ZJVY0[2`/0:b/[82/K6a0Z^[Z`:OWil2Shn?0WmoO`9bLW84K6a/0Vm_K`9aLG42
L71`0Ve]K@9XJ6P2Hf=S0UmOG`9LG5`2F5QH0U1@D092@T82<c<c0R/[:`8b<S82CDe=0WEeM@:MWId2
]kNg0[Zj^P:/[:`2VIVI0X^;R`:<S8`4oooo0P00007oool001Goool00`000?ooooooo`02oooo0Xb<
S0:BTY82V9RH0YVIV@:GUiL4UIFE0YRHV0:LW9`2WYjN0Y^KV`:ATI42Ph>30WEeM@9]KFd2LG5a0WYj
NP:3Ph<2PX:20W9bLP9JFUX2AdM70TI6AP9QHF42Shn?0[jn_P;QhN42k>c/0^GUi@;Fe]H2b<S80/30
`0Bn_[h2`/;20/S8b0;?cll2e]KF0]WIf@;Fe]H2c/k>0/C4a0:i^KT2/K6a0Zb/[0:VYZH2X:2P0YVI
V@:BTY82S8b<0XJ6QP:0P802O7al0WemO@:5QHD2UYJF0Z^[Z`;1`L42d];B0][JfP;Kfm/2emOG0/k>
cP;1`L400k:b/ZFUYJFUY@02X:2P0ZJVYP:b/[82_kno0/?3``:g]kL2Xj>S0Xn?S`:7QhL2Ti>C0Zb/
[0;6a/H2e]KF0]?Cd`;2`/82/;2`0ZJVYP:/[:`2_;bl0/c<c0;AdM42a/K60[6a/@:JVYX2S8b<0X^;
R`:EUID2XJ6Q0Zb/[0Ba/K42[Jf]0ZJVYP:LW9`2T92@0XF5Q@9kNg/2MWIf0WAdM09cLg<2LG5a0Ve]
K@9UIFD2GEeM0UMGE`9DE5@4De=C0U1@D099BDT2@4100Sdm?@96ATH2H61P0X>3P`:VYZH2_;bl0[jn
_P:`/;02WIfM0Xf=S@:8R8P5oooo0@00007oool001Goool00`000?ooooooo`02oooo0Xb<S0:BTY82
V9RH0YVIV@:GUiL4UIFE0YRHV0:LW9`2WYjN0Y^KV`:ATI42Ph>30WEeM@9]KFd2LG5a0WYjNP:3Ph<2
PX:20W9bLP9JFUX2AdM70TI6AP9QHF42Shn?0[jn_P;QhN42k>c/0^GUi@;Fe]H2b<S80/30`0Bn_[h2
`/;20/S8b0;?cll2e]KF0]WIf@;Fe]H2c/k>0/C4a0:i^KT2/K6a0Zb/[0:VYZH2X:2P0YVIV@:BTY82
S8b<0XJ6QP:0P802O7al0WemO@:5QHD2UYJF0Z^[Z`;1`L42d];B0][JfP;Kfm/2emOG0/k>cP;1`L40
0k:b/ZFUYJFUY@02X:2P0ZJVYP:b/[82_kno0/?3``:g]kL2Xj>S0Xn?S`:7QhL2Ti>C0Zb/[0;6a/H2
e]KF0]?Cd`;2`/82/;2`0ZJVYP:/[:`2_;bl0/c<c0;AdM42a/K60[6a/@:JVYX2S8b<0X^;R`:EUID2
XJ6Q0Zb/[0Ba/K42[Jf]0ZJVYP:LW9`2T92@0XF5Q@9kNg/2MWIf0WAdM09cLg<2LG5a0Ve]K@9UIFD2
GEeM0UMGE`9DE5@4De=C0U1@D099BDT2@4100Sdm?@96ATH2H61P0X>3P`:VYZH2_;bl0[jn_P:`/;02
WIfM0Xf=S@:8R8P5oooo0@00007oool001Goool00`000?ooooooo`02oooo0WYjNP9lO7`2OWin0Wal
O09iNGT2MgMg0W]kN`:4Q8@2T92@0Y^KV`:PX:02W9bL0Y6AT@:4Q8@2Ng]k0WQhN09kNg/2OGem0W]k
N`9_Kfl2H61P0UIFEP9KFe/2Lg=c0YRHV0:n_[h2fMWI0^?Sh`;Ogml2eMGE0/_;b`C6a/H2alO70/_;
b`;?cll2e=CD0]OGe`;Fe]H2c/k>0/71`@:c/k<2YZJV0YfMW@:HV9P2UIFE0Y:BTP:@T902Shn?0Xf=
S@:;Rh/2QhN718B4Q0:8R8P2TY:B0Z6QX@:`/;02_kno0/_;b`;Cdm<2e]KF0]?Cd`;6a/H00kJf]ZJV
YZJVYP02X:2P0ZRXZ0:i^KT2bl_;0]?Cd`;;bl/2^KVi0ZBTY0BHV9P2Y:BT0[6a/@Bk^k/2]KFe0Zn_
[`:^[Zh2][Jf0/?3``;>c/h2clo?0/;2`P:/[:`2UYJF0XV9R@::RXX2UIFE0ZBTY0:^[Zh2/;2`0Zb/
[0:TY:@2WIfM0YRHV0:CTi<2Shn?0X^;R`:7QhL2PX:20WemO@9fMWH2Jf][0UmOG`9CDe<2C4a<0TY:
BP9>CTh2E5AD15QHF09FEUH2EeMG0V1PH09dM7@2SHf=0ZFUY@:b/[82/;2`0Z:RXP:@T902PH610WYj
NPGoool100000Oooo`005Oooo`030000oooooooo00;oool2NWYj0WalO09nOWh2O7al0WUiN@9gMgL2
Ng]k0XB4Q0:@T902Vi^K0Z2PX0:LW9`2TI6A0XB4Q09kNg/2N7Qh0W]kN`9mOGd2Ng]k0Vm_K`9PH602
EUIF0U]KF`9cLg<2V9RH0[jn_P;IfMT2hn?S0]oOg`;EeMD2bl_;1<K6aP;7alL2bl_;0/o?c`;De=@2
emOG0]KFeP;>c/h2`L710[>c/`:VYZH2WIfM0YRHV0:EUID2TY:B0Y2@T0:?Shl2SHf=0X^;R`:7QhL4
Q8B40XR8R0:BTY82XJ6Q0[2`/0:o_kl2bl_;0]?Cd`;Fe]H2dm?C0/K6aP03][JfYZJVYZJV00:PX:02
Z:RX0[Vi^@;;bl/2dm?C0/_;b`:i^KT2Y:BT19RHV0:TY:@2/K6a1;^k^`:e]KD2[jn_0Zj^[P:f][H2
`l?30/k>cP;?cll2`/;20Zb/[0:FUYH2RHV90XZ:RP:EUID2Y:BT0Zj^[P:`/;02[:b/0ZBTY0:MWId2
V9RH0Y>CT`:?Shl2Rh^;0XN7Q`:2PX82OGem0WIfMP9[Jf/2GemO0U=CD`9<C4`2BTY:0Ti>CP9DE5@4
F5QH0UIFEP9GEeL2H61P0WAdM0:=SHd2YJFU0[:b/P:`/;02XZ:R0Y2@T0:1PH42NWYj1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9XJ6P2IVIV0VATI09OGel2FUYJ0UUIF@9OGel2Kfm_0XB4
Q0:HV9P2Y:BT0ZJVYP:NWYh2TY:B0XJ6QP9nOWh2NGUi0WIfMP9bLW82K6a/16MWI`9`L702QHF50Z2P
X0:k^k/2clo?1=OGe`;Bd]82cLg=0/c<c0;=cLd2clo?0]?Cd`;Fe]H4fMWI0]CDe0;7alL2]KFe0Z:R
XP:BTY82QhN70X:2PP:0P802PX:20XF5Q@::RXX2Shn?0Y:BTP:ATI42Shn?18f=S@:ATI42UiNG0Z2P
X0:/[:`2^[Zj0/S8b0;Cdm<2eMGE0/_;b`03^[ZjZZZZZZZZ00:SXj<2Zj^[0[jn_P;Bd]82gMgM0][J
fP;<c<`2^[Zj0ZZZZP:PX:04W9bL0YjNWP:RXZ82Z:RX0Zn_[`:h^;P2`/;20/_;b`;@d=02c<c<0[fm
_@:XZ:P2U9BD0XV9R@::RXX2UYJF0ZBTY0B^[Zh2YZJV0YfMW@:GUiL2UIFE0YNGU`BIVIT2UYJF0Y2@
T0:8R8P2O7al0Va/K09LG5`2CTi>0TE5A@96ATH2CDe=0UUIF@9SHf<2JVYZ0Vm_K`9cLg<2Ng]k0XN7
Q`:DU9@2WYjN0Z:RXP:MWId2T92@0X61P@9dM7@2K6a/1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9XJ6P2IVIV0VATI09OGel2FUYJ0UUIF@9OGel2Kfm_0XB4Q0:HV9P2Y:BT0ZJVYP:NWYh2
TY:B0XJ6QP9nOWh2NGUi0WIfMP9bLW82K6a/16MWI`9`L702QHF50Z2PX0:k^k/2clo?1=OGe`;Bd]82
cLg=0/c<c0;=cLd2clo?0]?Cd`;Fe]H4fMWI0]CDe0;7alL2]KFe0Z:RXP:BTY82QhN70X:2PP:0P802
PX:20XF5Q@::RXX2Shn?0Y:BTP:ATI42Shn?18f=S@:ATI42UiNG0Z2PX0:/[:`2^[Zj0/S8b0;Cdm<2
eMGE0/_;b`03^[ZjZZZZZZZZ00:SXj<2Zj^[0[jn_P;Bd]82gMgM0][JfP;<c<`2^[Zj0ZZZZP:PX:04
W9bL0YjNWP:RXZ82Z:RX0Zn_[`:h^;P2`/;20/_;b`;@d=02c<c<0[fm_@:XZ:P2U9BD0XV9R@::RXX2
UYJF0ZBTY0B^[Zh2YZJV0YfMW@:GUiL2UIFE0YNGU`BIVIT2UYJF0Y2@T0:8R8P2O7al0Va/K09LG5`2
CTi>0TE5A@96ATH2CDe=0UUIF@9SHf<2JVYZ0Vm_K`9cLg<2Ng]k0XN7Q`:DU9@2WYjN0Z:RXP:MWId2
T92@0X61P@9dM7@2K6a/1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9NGUh2EeMG0U1@
D097AdL2@D510T10@098B4P2GEeM0WQhN0:CTi<2YJFU0ZZZZP:TY:@2UiNG0XV9R@9mOGd2M7Ad0Vi^
KP9YJFT2IfMW0VUYJ@9`L702Ng]k0Xb<S0:PX:02/k>c0/71`@;:b/X2c/k>1/o?c`;AdM42dm?C0]KF
eP;Jf]X4gMgM0]KFeP;6a/H2[jn_0YRHV0:4Q8@2MgMg179bLP9fMWH2OWin0XJ6QP:?Shl2UIFE0YNG
U`:FUYH2U9BD0Y:BTP:ATI42Ti>C0YNGU`:PX:02[jn_0/30`0;?cll2eMGE0/g=c@03_[jn[jn_[jn_
00:XZ:P2[Zj^0[jn_P;?cll2f][J0]cLg0;De=@2alO70[Rh^0:XZ:P2VYZJ0Y2@T0:=SHd2Ti>C0Z2P
X0:a/K42`<300/_;b`;@d=02dM7A0/W9b@:k^k/2Z:RX0YNGU`B<S8`2UIFE0Z6QX@BYZJT2Xj>S0YbL
W0BGUiL2Vi^K19jNWP:LW9`2UYJF0Xf=S@:1PH42LG5a0V1PH09ADE44BDU90U9BDP9NGUh2Jf][0WMg
M`:1PH42RHV90Y2@T0:FUYH4VIVI0YJFUP:@T902QXJ60W]kN`9`L702IfMW1Oooo`400001oooo000:
oooo0`0000;oool00`000?ooooooo`03oooo00<0003oooooool00_ooo`9NGUh2EeMG0U1@D097AdL2
@D510T10@098B4P2GEeM0WQhN0:CTi<2YJFU0ZZZZP:TY:@2UiNG0XV9R@9mOGd2M7Ad0Vi^KP9YJFT2
IfMW0VUYJ@9`L702Ng]k0Xb<S0:PX:02/k>c0/71`@;:b/X2c/k>1/o?c`;AdM42dm?C0]KFeP;Jf]X4
gMgM0]KFeP;6a/H2[jn_0YRHV0:4Q8@2MgMg179bLP9fMWH2OWin0XJ6QP:?Shl2UIFE0YNGU`:FUYH2
U9BD0Y:BTP:ATI42Ti>C0YNGU`:PX:02[jn_0/30`0;?cll2eMGE0/g=c@03_[jn[jn_[jn_00:XZ:P2
[Zj^0[jn_P;?cll2f][J0]cLg0;De=@2alO70[Rh^0:XZ:P2VYZJ0Y2@T0:=SHd2Ti>C0Z2PX0:a/K42
`<300/_;b`;@d=02dM7A0/W9b@:k^k/2Z:RX0YNGU`B<S8`2UIFE0Z6QX@BYZJT2Xj>S0YbLW0BGUiL2
Vi^K19jNWP:LW9`2UYJF0Xf=S@:1PH42LG5a0V1PH09ADE44BDU90U9BDP9NGUh2Jf][0WMgM`:1PH42
RHV90Y2@T0:FUYH4VIVI0YJFUP:@T902QXJ60W]kN`9`L702IfMW1Oooo`400001oooo000;oooo00H0
003oooooool0003oool00004oooo00<0003oooooool00_ooo`9QHF42DU9B0TA4A08h>3P2<30`0Rl_
;`8i>CT2D51@0Vi^KP:;Rh/2X:2P0ZFUY@:NWYh2TI6A0X61P@9dM7@2Jf][0VATI09PH602GemO0V9R
HP9YJFT2MGEe0XB4Q0:EUID2YJFU0[>c/`:n_[h2a/K60/c<c0;>c/h4clo?0/k>cP;AdM42f=SH0^3P
h0;UiND2gmoO0/g=c@:b/[82UYJF0WmoO`9bLW82Kfm_0W5aL@9gMgL2OGem0XB4Q0::RXX2T92@0Y>C
T`JEUID2UYJF0YNGU`:JVYX2XZ:R0Zn_[`:n_[h2c<c<0];BdP;=cLd00l71`KFe]KFe]@02[Jf]0Zj^
[P:f][H2`<300/S8b0;<c<`2bl_;0/O7a`:n_[h2/;2`0Z2PX0:DU9@2Shn?0YBDU0:RXZ82/k>c0/?3
``;=cLd2d=3@0/o?c`;8b<P2_;bl0Zf][@:MWId2Ti>C0Xn?S`:CTi<2VIVI0YnOW`:SXj<4Y:BT0Z:R
XP:PX:02WIfM0YVIV@:EUID2TY:B0Xn?S`:;Rh/2Ph>30WQhN09/K6`2H61P0UUIF@9GEeL2Fe]K0V9R
HP9]KFd2NWYj0XR8R0:DU9@2WIfM0Z2PX0:NWYh2VYZJ0YFEU@:ATI42SHf=0XJ6QP9mOGd2L71`1Ooo
o`400001oooo0009oooo10000004oooo0000oooo00001?ooo`800003oooo0V5QH@9BDU82A4A40SPh
>08`<302;bl_0STi>@9@D502KVi^0X^;R`:PX:02YJFU0YjNWP:ATI42PH610WAdM09[Jf/2I6AT0V1P
H09OGel2HV9R0VUYJ@9eMGD2Q8B40YFEU@:UYJD2/k>c0[jn_P;6a/H2c<c<0/k>cPC?cll2c/k>0]7A
d@;Hf=P2h>3P0^GUi@;Ogml2cLg=0[:b/P:FUYH2Ogmo0W9bLP9_Kfl2LG5a0WMgM`9mOGd2Q8B40XZ:
RP:@T902Ti>C1YFEU@:FUYH2UiNG0YZJVP:RXZ82[jn_0[jn_P;<c<`2d];B0/g=c@03`L71]KFe]KFe
00:][Jd2[Zj^0[Jf]P;0`<02b<S80/c<c0;;bl/2alO70[jn_P:`/;02X:2P0YBDU0:?Shl2U9BD0Z:R
XP:c/k<2`l?30/g=c@;@d=02clo?0/S8b0:l_;`2[Jf]0YfMW@:CTi<2Shn?0Y>CT`:IVIT2WinO0Z>S
X`BTY:@2XZ:R0Z2PX0:MWId2VIVI0YFEU@:BTY82Shn?0X^;R`:3Ph<2N7Qh0Va/K09PH602FEUI0UMG
E`9KFe/2HV9R0Ve]K@9jNWX2R8R80YBDU0:MWId2X:2P0YjNWP:JVYX2UIFE0Y6AT@:=SHd2QXJ60Wem
O@9`L704oooo0P00007oool000Woool00`000?ooo`000002oooo00<0003oool000001?ooo`030000
oooooooo00;oool2K6a/0UADE08o?cl2;bl_0RLW9`8V9RH2<C4a0TQ8B09VIVH2Ph>30YJFUP:JVYX2
T92@0X20P09`L702IFEU0UmOG`9LG5`2FEUI15MGE`9KFe/2I6AT0W9bLP:3Ph<2UIFE0ZJVYP:d];@2
`<300/S8b0;<c<`2bl_;0/S8b0;5aLD2alO70];BdP;Ph>04jn_[0]WIf@:l_;`2W9bL0X>3P`9gMgL2
MWIf0W]kN`:0P802Ph>30XB4Q0:5QHD2QhN70X^;R`:?Shl2U9BD0YVIV@:LW9`2X:2P0ZFUY@:/[:`2
][Jf0/30`0;9bLT2c<c<0/[:bP03`/;2^KVi^KVi00:a/K42[:b/0ZZZZP:YZJT2[:b/0[2`/0:f][H4
^[Zj0[Bd]0:[Zj/2XZ:R0YjNWP:QXJ42ZZZZ0[Jf]P;0`<02b<S80/c<c0;=cLd2b<S80[no_`:c/k<2
YjNW0Y^KV`:CTi<4SXj>0Y:BTP:LW9`2YjNW0[6a/@:d];@2[Jf]0Z2PX0:@T902Q8B40X20P0:1PH42
Q8B40XF5Q@:3Ph<2OWin0WQhN09bLW82K6a/0VQXJ09WIfL2Jf][0WIfMP:6QXH2UYJF0Z:RXP:VYZH2
Y:BT0YnOW`:KVi/2W9bL0YjNWP:MWId2UIFE0XB4Q0Goool100000Oooo`002_ooo`800002oooo00<0
003oool000001?ooo`030000oooooooo00;oool2K6a/0UADE08o?cl2;bl_0RLW9`8V9RH2<C4a0TQ8
B09VIVH2Ph>30YJFUP:JVYX2T92@0X20P09`L702IFEU0UmOG`9LG5`2FEUI15MGE`9KFe/2I6AT0W9b
LP:3Ph<2UIFE0ZJVYP:d];@2`<300/S8b0;<c<`2bl_;0/S8b0;5aLD2alO70];BdP;Ph>04jn_[0]WI
f@:l_;`2W9bL0X>3P`9gMgL2MWIf0W]kN`:0P802Ph>30XB4Q0:5QHD2QhN70X^;R`:?Shl2U9BD0YVI
V@:LW9`2X:2P0ZFUY@:/[:`2][Jf0/30`0;9bLT2c<c<0/[:bP03`/;2^KVi^KVi00:a/K42[:b/0ZZZ
ZP:YZJT2[:b/0[2`/0:f][H4^[Zj0[Bd]0:[Zj/2XZ:R0YjNWP:QXJ42ZZZZ0[Jf]P;0`<02b<S80/c<
c0;=cLd2b<S80[no_`:c/k<2YjNW0Y^KV`:CTi<4SXj>0Y:BTP:LW9`2YjNW0[6a/@:d];@2[Jf]0Z2P
X0:@T902Q8B40X20P0:1PH42Q8B40XF5Q@:3Ph<2OWin0WQhN09bLW82K6a/0VQXJ09WIfL2Jf][0WIf
MP:6QXH2UYJF0Z:RXP:VYZH2Y:BT0YnOW`:KVi/2W9bL0YjNWP:MWId2UIFE0XB4Q0Goool100000Ooo
o`002oooo`050000oooooooooooo000000Goool00`000?ooooooo`02oooo0WYjNP9JFUX2?Shn0R/[
:`8R8R828b<S0Rh^;P95ADD2HF5Q0WalO0:=SHd2S8b<0WmoO`9/K6`2G5aL15ADE09FEUH2EeMG0U=C
D`9?Cdl2CTi>0U=CD`9PH602Lg=c0XR8R0:KVi/2[:b/0[Zj^P;4a<@2b<S80/G5a@:o_kl2^[Zj0[bl
_0;:b/X2gMgM0^k^kP;bl_82hn?S0/K6aP:VYZH2SHf=0X:2PP:3Ph<2R8R80Xb<S0:;Rh/2QXJ61861
P@:4Q8@2S8b<0YBDU0:MWId2YJFU0Z^[Z`:b/[82]kNg0[fm_@;0`<04`l?30/;2`P03`<30_;bl_;bl
00:d];@2ZJVY0YbLW0:BTY82SXj>0Y:BTP:LW9`2YjNW0[6a/@Bd];@2/[:b1;2`/0:c/k<2][Jf0[^k
^`;1`L42a/K61<W9b@;3`l<2^[Zj0Zn_[`:SXj<2UIFE0XZ:RP:3Ph<2QhN70YJFUP:/[:`2_kno0/O7
a`:l_;`2Y:BT0XV9R@9eMGD2L71`0WEeM@9oOgl2RHV90Y2@T0BBTY82SXj>0XB4Q09iNGT2Kfm_0Va/
K09dM7@2Q8B40YJFUP:TY:@2ZJVY0ZRXZ0BTY:@2ZZZZ0[:b/P:e]KD2[Zj^0YRHV0Goool100000Ooo
o`005Oooo`030000oooooooo00;oool2NWYj0UYJFP8n?Sh2:b/[0R8R8P8S8b<2;Rh^0TE5A@9QHF42
O7al0Xf=S@:<S8`2Ogmo0Va/K09LG5`4E5AD0UIFEP9GEeL2De=C0Tm?C`9>CTh2De=C0V1PH09cLg<2
R8R80Y^KV`:/[:`2^[Zj0/C4a0;8b<P2aLG50[no_`:j^[X2_;bl0/[:bP;MgMd2k^k^0_;blP;Shn<2
a/K60ZJVYP:=SHd2PX:20X>3P`:8R8P2S8b<0X^;R`:6QXH4PH610XB4Q0:<S8`2U9BD0YfMW@:UYJD2
Zj^[0[:b/P:g]kL2_Kfm0/30`0C3`l<2`/;200?0`<2l_;bl_;`00[Bd]0:YZJT2W9bL0Y:BTP:>SXh2
TY:B0YbLW0:WYjL2/K6a1;Bd]0:b/[84/;2`0[>c/`:f][H2^k^k0/71`@;6a/H4bLW90/?3``:j^[X2
[jn_0Z>SX`:EUID2RXZ:0X>3P`:7QhL2UYJF0Zb/[0:o_kl2alO70[bl_0:TY:@2RHV90WEeM@9`L702
MGEe0WmoO`:9RHT2T92@19:BTP:>SXh2Q8B40WUiN@9_Kfl2K6a/0WAdM0:4Q8@2UYJF0ZBTY0:YZJT2
Z:RX1:BTY0:ZZZX2/[:b0[Fe]@:^[Zh2V9RH1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`:3Ph<2G5aL0S/k>`8U9BD271`L0QdM7@8Z:RX2@D510UeMG@9gMgL2QHF50X61P@9^KVh2EUIF0TE5
A@92@T82BDU90U=CD`9JFUX2F5QH0U5AD@9<C4`2CTi>0UYJFP9/K6`2P8200YBDU0:UYJD2/[:b0[bl
_0;0`<02_Kfm0[Jf]P:`/;02/[:b0/30`0;EeMD2j>SX0^o_k`;Ti>@2c<c<0[2`/0:JVYX4TI6A0YBD
U0:FUYH2TY:B0X^;R`:5QHD2Q8B40XR8R0:ATI42Vi^K0ZFUY@:^[Zh2];Bd0[Vi^@:l_;`2_Kfm0[^k
^`:h^;P2]kNg0[Rh^003^k^k_;bl_;bl00:f][H2YjNW0YBDU0:2PX82N7Qh0WYjNP:5QHD2Ti>C0Z:R
XP:/[:`2/k>c0[Rh^0:j^[X2^;Rh0[Jf]P:c/k<2];Bd0[Vi^@;0`<02alO70/W9b@;6a/H2_[jn0[>c
/`:VYZH2UIFE0XJ6QP9mOGd2P8200Y>CT`:`/;02b/[:0]GEe@;8b<P2Zj^[0X^;R`9cLg<2KFe]0WIf
MP:4Q8@2Ti>C0YnOW`:WYjL2ZZZZ0ZNWY`:LW9`2SHf=0WemO@9eMGD2NWYj0XR8R0:JVYX2Z:RX0Zf]
[@:[Zj/2Z:RX0ZVYZ@:d];@2`L710/S8b0;1`L42Z:RX1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`:3Ph<2G5aL0S/k>`8U9BD271`L0QdM7@8Z:RX2@D510UeMG@9gMgL2QHF50X61P@9^KVh2
EUIF0TE5A@92@T82BDU90U=CD`9JFUX2F5QH0U5AD@9<C4`2CTi>0UYJFP9/K6`2P8200YBDU0:UYJD2
/[:b0[bl_0;0`<02_Kfm0[Jf]P:`/;02/[:b0/30`0;EeMD2j>SX0^o_k`;Ti>@2c<c<0[2`/0:JVYX4
TI6A0YBDU0:FUYH2TY:B0X^;R`:5QHD2Q8B40XR8R0:ATI42Vi^K0ZFUY@:^[Zh2];Bd0[Vi^@:l_;`2
_Kfm0[^k^`:h^;P2]kNg0[Rh^003^k^k_;bl_;bl00:f][H2YjNW0YBDU0:2PX82N7Qh0WYjNP:5QHD2
Ti>C0Z:RXP:/[:`2/k>c0[Rh^0:j^[X2^;Rh0[Jf]P:c/k<2];Bd0[Vi^@;0`<02alO70/W9b@;6a/H2
_[jn0[>c/`:VYZH2UIFE0XJ6QP9mOGd2P8200Y>CT`:`/;02b/[:0]GEe@;8b<P2Zj^[0X^;R`9cLg<2
KFe]0WIfMP:4Q8@2Ti>C0YnOW`:WYjL2ZZZZ0ZNWY`:LW9`2SHf=0WemO@9eMGD2NWYj0XR8R0:JVYX2
Z:RX0Zf][@:[Zj/2Z:RX0ZVYZ@:d];@2`L710/S8b0;1`L42Z:RX1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooo`:3Ph<2EUIF0S0`<08H61P23`l?0Q8B4P8Q8B42>c/k0U]KF`9gMgL2Q8B40WYj
NP9PH602@d=30Rl_;`8`<302@4100UEEE@9TI6@2IfMW0V9RHPALG5`2I6AT0W=cL`:3Ph<2TY:B0YnO
W`:YZJT2/K6a0[Bd]0:c/k<2[Zj^0ZZZZP:/[:`2]kNg0/O7a`;Fe]H2gMgM0]SHf0;:b/X2^;Rh0ZVY
Z@:PX:02WIfM0YbLW0:KVi/2V9RH0YBDU0:CTi<2UIFE0Y^KV`:SXj<2[:b/0[>c/`:g]kL4^;Rh0[Fe
]@:a/K42[Jf]1:RXZ0:/[:`00k>c/kRh^;Rh^002]KFe0ZRXZ0:DU9@2P8200W9bLP9`L702MWIf0X61
P@:>SXh2Vi^K0ZFUY@:^[Zh4/k>c0[2`/0B/[:`2/k>c0[fm_@;5aLD2b/[:0/K6aP:m_Kd2/K6a0Z:R
XP:BTY82Q8B40WalO0:0P802UIFE0[:b/P;>c/h2fm_K0]7Ad@:g]kL2VYZJ0XB4Q09oOgl2R8R80YJF
UP:VYZH2/K6a0[Vi^@:m_Kd2^[Zj0[2`/0:QXJ42TY:B0XZ:RP:<S8`2V9RH0ZJVYP:`/;02/[:b0Zf]
[@:XZ:P2ZZZZ0[Jf]P;6a/H2d=3@0/[:bP:^[Zh5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0X>3P`9FEUH2<30`0QPH608?3`l24Q8B0R4Q8@8k>c/2Fe]K0WMgM`:4Q8@2NWYj0V1PH093@d<2
;bl_0S0`<090@402EEEE0VATI09WIfL2HV9R15aLG09TI6@2Lg=c0X>3P`:BTY82WinO0ZVYZ@:a/K42
];Bd0[>c/`:^[Zh2ZZZZ0Zb/[0:g]kL2alO70]KFeP;MgMd2f=SH0/[:bP:h^;P2ZJVY0Z2PX0:MWId2
W9bL0Y^KV`:HV9P2U9BD0Y>CT`:EUID2Vi^K0Z>SX`:/[:`2/k>c0[Ng]`Bh^;P2]KFe0[6a/@:][Jd4
Z:RX0Zb/[003/k>c^;Rh^;Rh00:e]KD2Z:RX0YBDU0:0P802LW9b0W1`L09fMWH2PH610Xj>SP:KVi/2
YJFU0Zj^[PBc/k<2/;2`1:b/[0:c/k<2_Kfm0/G5a@;:b/X2a/K60[fm_@:a/K42XZ:R0Y:BTP:4Q8@2
O7al0X20P0:EUID2/[:b0/k>cP;Kfm/2dM7A0[Ng]`:JVYX2Q8B40WmoO`:8R8P2UYJF0ZJVYP:a/K42
^KVi0[fm_@:j^[X2/;2`0Z6QX@:BTY82RXZ:0Xb<S0:HV9P2YZJV0[2`/0:b/[82[Jf]0ZRXZ0:ZZZX2
][Jf0/K6aP;@d=02b/[:0Zj^[PGoool100000Oooo`005Oooo`030000oooooooo00;oool2OGem0Te=
C@8T92@22`/;0P<30`861PH261PH0SLg=`9KFe/2Ng]k0XR8R09jNWX2FEUI0SHf=P8P820292@T0Sdm
?@9KFe/2M7Ad0WinOP9mOGd2NGUi0WMgM`9kNg/2Q8B40Xf=S@:FUYH2W9bL0Z6QX@:UYJD8Z:RX0ZZZ
ZP:_[jl2][Jf0[fm_@;2`/82a<C40/;2`P:m_Kd2]kNg0Zn_[`:VYZH2WinO19^KV`:OWil2YZJV0Zj^
[P:e]KD2_;bl0/30`0;1`L42_Kfm0[Jf]P:^[Zh2YJFU0YfMW@:HV9P2UIFE0YNGU`:NWYh00jRXZ:n_
[jn_[`02/;2`0ZRXZ0:HV9P2QhN70WQhN09aLG42L71`0W=cL`9jNWX2Ph>30Xj>SP:IVIT2XJ6Q1:>S
X`:RXZ82YJFU0Zj^[P:j^[X2aLG50/W9b@;4a<@2^KVi0Z^[Z`:LW9`2Shn?0XF5Q@:1PH42R8R80Y^K
V`:d];@2c<c<0][JfP;Fe]H2aLG50[6a/@:RXZ82WinO0ZNWY`:b/[82_;bl0/C4a0;7alL2bLW90/K6
aP:o_kl2][Jf0Zb/[0:VYZH2YjNW0Zj^[P:g]kL2^k^k0[Ng]`:^[Zh4YZJV0[>c/`;4a<@2clo?0/W9
b@:[Zj/5oooo0@00007oool001Goool200000oooo`9mOGd2CDe=0R@T908;2`/20`<30PH61P8H61P2
=cLg0U]KF`9kNg/2R8R80WYjNP9IFET2=SHf0R0P808T92@2?Cdm0U]KF`9dM7@2OWin0WemO@9iNGT2
MgMg0W]kN`:4Q8@2SHf=0YJFUP:LW9`2XJ6Q0ZFUY@RXZ:P2ZZZZ0Zn_[`:f][H2_Kfm0/;2`P;4a<@2
`/;20[fm_@:g]kL2[jn_0ZJVYP:OWil4Vi^K0YnOW`:VYZH2[Zj^0[Fe]@:l_;`2`<300/71`@:m_Kd2
][Jf0Zj^[P:UYJD2WIfM0YRHV0:EUID2UiNG0YjNWP03Z:RX[jn_[jn_00:`/;02Z:RX0YRHV0:7QhL2
N7Qh0W5aL@9`L702Lg=c0WYjNP:3Ph<2SXj>0YVIV@:QXJ44Xj>S0Z:RXP:UYJD2[Zj^0[Zj^P;5aLD2
bLW90/C4a0:i^KT2Zj^[0YbLW0:?Shl2QHF50X61P@:8R8P2Vi^K0[Bd]0;<c<`2f][J0]KFeP;5aLD2
/K6a0Z:RXP:OWil2YjNW0[:b/P:l_;`2a<C40/O7a`;9bLT2a/K60[no_`:f][H2[:b/0ZJVYP:WYjL2
[Zj^0[Ng]`:k^k/2]kNg0Zj^[PBVYZH2/k>c0/C4a0;?cll2bLW90Z^[Z`Coool200000Oooo`005Ooo
o`030000oooooooo00;oool2MgMg0TQ8B08P820220P80P00008410@261PH0STi>@9QHF42Ph>30Y6A
T@9oOgl2FUYJ0S8b<P8K6a/28R8R0T51@@9WIfL2QhN70YRHV0:KVi/2VIVI0YJFUP:GUiL2VYZJ0YbL
W0BMWId4W9bL0YjNWP:QXJ42YJFU0ZVYZ@:[Zj/2ZZZZ0ZNWY`:UYJD2YZJV0Zf][@:g]kL2_kno0/71
`@:j^[X2[Jf]0Z6QX@:KVi/2WinO0ZZZZP:h^;P2aLG50/g=c@;AdM42d=3@0/[:bP:n_[h2[jn_0YnO
W`:ATI42QhN70X61P@:0P802Q8B40Xf=S@03V9RHXJ6QXJ6Q00:VYZH2Y:BT0Y^KV`:?Shl2PX:20WIf
MP9]KFd2IVIV0VEUI@9[Jf/2M7Ad0X61P@:;Rh/2TI6A0YBDU0:FUYH2WIfM0ZVYZ@:h^;P2a<C40/W9
b@;2`/82]KFe0ZFUY@:FUYH2SHf=0XZ:RP:;Rh/2U9BD0ZBTY0:g]kL2b/[:0]KFeP;Hf=P2dM7A0/S8
b0C2`/82a/K60/g=c@;Bd]82dm?C0];BdP;AdM42c/k>0/_;b`;7alL2a<C41<;2`P;5aLD2alO70/C4
a0:l_;`2[jn_0ZFUY@:SXj<2[Jf]0[jn_P;8b<P2`/;20ZFUY@Goool100000Oooo`005Oooo`030000
oooooooo00;oool2MgMg0TQ8B08P820220P80P00008410@261PH0STi>@9QHF42Ph>30Y6AT@9oOgl2
FUYJ0S8b<P8K6a/28R8R0T51@@9WIfL2QhN70YRHV0:KVi/2VIVI0YJFUP:GUiL2VYZJ0YbLW0BMWId4
W9bL0YjNWP:QXJ42YJFU0ZVYZ@:[Zj/2ZZZZ0ZNWY`:UYJD2YZJV0Zf][@:g]kL2_kno0/71`@:j^[X2
[Jf]0Z6QX@:KVi/2WinO0ZZZZP:h^;P2aLG50/g=c@;AdM42d=3@0/[:bP:n_[h2[jn_0YnOW`:ATI42
QhN70X61P@:0P802Q8B40Xf=S@03V9RHXJ6QXJ6Q00:VYZH2Y:BT0Y^KV`:?Shl2PX:20WIfMP9]KFd2
IVIV0VEUI@9[Jf/2M7Ad0X61P@:;Rh/2TI6A0YBDU0:FUYH2WIfM0ZVYZ@:h^;P2a<C40/W9b@;2`/82
]KFe0ZFUY@:FUYH2SHf=0XZ:RP:;Rh/2U9BD0ZBTY0:g]kL2b/[:0]KFeP;Hf=P2dM7A0/S8b0C2`/82
a/K60/g=c@;Bd]82dm?C0];BdP;AdM42c/k>0/_;b`;7alL2a<C41<;2`P;5aLD2alO70/C4a0:l_;`2
[jn_0ZFUY@:SXj<2[Jf]0[jn_P;8b<P2`/;20ZFUY@Goool100000Oooo`005Oooo`030000oooooooo
00;oool2MgMg0Tm?C`8^;Rh26ATI0Q8B4P8F5QH2:2PX0TU9B@9_Kfl2T92@0YfMW@:<S8`2IVIV0Slo
?`8Y:BT2<S8b0U9BDP9jNWX2W9bL0Zn_[`:e]KD2];Bd0[:b/P:a/K42/;2`0Zf][@:YZJT2Y:BT0YnO
W`BLW9`2X:2P0ZNWY`:][Jd2[Zj^0ZRXZ0:NWYh2U9BD0Y:BTP:KVi/2ZZZZ0[Zj^P;2`/82_Kfm0[2`
/0:SXj<2WIfM0Z>SX`:b/[82`l?30]7Ad@;Hf=P2emOG0];BdP;6a/H2][Jf0Z:RXP:>SXh2OGem0W5a
L@A[Jf/2L71`0WQhN003PX:2S8b<S8b<00:DU9@2V9RH0YNGU`:BTY82R8R80WYjNP9[Jf/2GEeM0UIF
EP9HF5P2HF5Q0Vm_K`9kNg/2Ph>30XN7Q`:<S8`2UIFE0Z>SX`:d];@2`/;20/O7a`;1`L42/k>c0Z>S
X`:GUiL2TY:B0YBDU0:JVYX2Y:BT0Zn_[`:l_;`2b<S80];BdP;Hf=P2f][J1=WIf@;Kfm/2gMgM1>3P
h0;MgMd2fMWI0]GEe@CBd]82e=CD0]GEe@;Fe]H2eMGE0]?Cd`;?cll2b<S80[fm_@:_[jl2Y:BT0Z2P
X0:XZ:P2];Bd0[^k^`:f][H2W9bL1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9gMgL2
Cdm?0Rh^;P8I6AT24Q8B0QHF5P8X:2P2BDU90Vm_K`:@T902WIfM0Xb<S09VIVH2?clo0RTY:@8b<S82
DU9B0WYjNP:LW9`2[jn_0[Fe]@:d];@2/[:b0[6a/@:`/;02[Jf]0ZVYZ@:TY:@2WinO19bLW0:PX:02
YjNW0Zf][@:^[Zh2Z:RX0YjNWP:DU9@2TY:B0Y^KV`:ZZZX2^[Zj0/;2`P:m_Kd2/;2`0Z>SX`:MWId2
Xj>S0[:b/P;3`l<2dM7A0]SHf0;GemL2d];B0/K6aP:f][H2XZ:R0Xj>SP9mOGd2LG5a16][J`9`L702
N7Qh00>2PX:<S8b<S8`00YBDU0:HV9P2UiNG0Y:BTP:8R8P2NWYj0V][J`9MGEd2EUIF0UQHF09QHF42
Kfm_0W]kN`:3Ph<2QhN70Xb<S0:EUID2Xj>S0[Bd]0;2`/82alO70/71`@:c/k<2Xj>S0YNGU`:BTY82
U9BD0YZJVP:TY:@2[jn_0[bl_0;8b<P2d];B0]SHf0;Jf]X4fMWI0]_Kf`;MgMd4h>3P0]gMg@;IfMT2
eMGE1=;BdP;De=@2eMGE0]KFeP;EeMD2dm?C0/o?c`;8b<P2_Kfm0Zn_[`:TY:@2X:2P0ZRXZ0:d];@2
^k^k0[Jf]P:LW9`5oooo0@00007oool001Goool00`000?ooooooo`02oooo0WinOP9WIfL2De=C0TI6
AP90@402@T920Tm?C`9YJFT2R8R80Z:RXP:][Jd2WYjN0X20P09QHF42Cdm?0UMGE`9bLW82Ti>C0[2`
/0;0`<04a/K60/G5a@;3`l<2`L710[fm_@:h^;P2/K6a0Z^[Z`:VYZH2YJFU0ZVYZ@:_[jl2];Bd0[>c
/`:[Zj/2WIfM0Y2@T0:;Rh/2TI6A0YnOW`:^[Zh2]kNg0[Fe]@:][Jd2Y:BT0Z6QX@:XZ:P2]KFe0/;2
`P;<c<`2c/k>0/[:bP;0`<02/k>c0Z:RXP:?Shl2OGem0Va/K09PH602FEUI0UMGE`9IFET2GUiN00=V
IVI_Kfm_Kfl00WQhN0:1PH42R8R80X^;R`:6QXH2NGUi0VQXJ09GEeL2CDe=0Tm?C`9JFUX2JFUY0WIf
MP9nOWh2PH610XF5Q@:=SHd2W9bL0Zj^[P:n_[h2aLG50/;2`P:g]kL2ZZZZ0Z6QX@:OWil2Y:BT0Zf]
[@:e]KD2_Kfm0/C4a0;;bl/2dM7A0]OGe`;Kfm/2gmoO0^;RhP;Shn<4iNGU0^?Sh`;Ogml2f][J0]KF
eP;De=@2e]KF0]WIf@;MgMd2g]kN0]_Kf`;EeMD2cLg=0/C4a0:i^KT2[jn_0ZJVYP:QXJ42Xj>S0ZNW
Y`:YZJT2Y:BT0YBDU0Goool100000Oooo`005Oooo`030000oooooooo00;oool2OWin0VMWI`9CDe<2
ATI60T10@092@T82Cdm?0VUYJ@:8R8P2XZ:R0Zf][@:NWYh2P8200V5QH@9?Cdl2EeMG0W9bLP:CTi<2
/;2`0/30`0C6a/H2aLG50/?3``;1`L42_Kfm0[Rh^0:a/K42Zj^[0ZJVYP:UYJD2ZJVY0Zn_[`:d];@2
/k>c0Z^[Z`:MWId2T92@0X^;R`:ATI42WinO0Zj^[P:g]kL2]KFe0Zf][@:TY:@2XJ6Q0ZRXZ0:e]KD2
`/;20/c<c0;>c/h2b/[:0/30`0:c/k<2XZ:R0Xn?S`9mOGd2K6a/0V1PH09IFET2EeMG0UUIF@9NGUh0
0fIVIVm_Kfm_K`02N7Qh0X61P@:8R8P2Rh^;0XJ6QP9iNGT2J6QX0UMGE`9=CDd2Cdm?0UYJFP9YJFT2
MWIf0WinOP:1PH42QHF50Xf=S@:LW9`2[Zj^0[jn_P;5aLD2`/;20[Ng]`:ZZZX2XJ6Q0YnOW`:TY:@2
[Jf]0[Fe]@:m_Kd2a<C40/_;b`;AdM42emOG0]_Kf`;Ogml2h^;R0^?Sh`CUiND2hn?S0]oOg`;Jf]X2
e]KF0]CDe0;Fe]H2fMWI0]gMg@;Ng]h2fm_K0]GEe@;=cLd2a<C40[Vi^@:_[jl2YZJV0Z6QX@:SXj<2
YjNW0ZVYZ@:TY:@2U9BD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:<S8`2RXZ:0XR8
R0:3Ph<2Ogmo0WinOP:3Ph<2TY:B0ZJVYP:g]kL2_Kfm0[Fe]@:RXZ82SXj>0XB4Q0:9RHT2VYZJ0Zn_
[`;1`L42c<c<0/o?c`;@d=02clo?0/k>cP;=cLd2bl_;0/O7a`;2`/82_;bl0[Rh^0:f][H2]kNg0[Zj
^P:l_;`2^[Zj0[6a/@:TY:@2UiNG0Xn?S`:@T902UYJF0YjNWP:UYJD4YjNW0ZJVYP:WYjL2[:b/0[>c
/`:i^KT2^[Zj0[Jf]P:][Jd2XJ6Q0Y>CT`:6QXH2N7Qh0V][J`9OGel2De=C0T];B`95ADD2@d=30TA4
A003B4Q8CTi>CTi>009GEeL2I6AT0W5aL@9kNg/2OGem0W=cL`9SHf<2De=C0TY:BP9=CDd2FUYJ0V][
J`9hN7P2Ogmo0X20P0:1PH42R8R80YFEU@:XZ:P2^KVi0/?3``;4a<@2_[jn0[Jf]P:a/K42/[:b0[Rh
^0;0`<02alO70/_;b`;=cLd2clo?0]7Ad@;EeMD2f=SH0]cLg0;Ng]h6gmoO0]kNgP;Kfm/2f=SH1=GE
e@;Fe]H2f][J1=cLg0;GemL2c/k>0/C4a0:k^k/2/k>c0Zj^[P:YZJT2Y:BT0YnOW`:JVYX2UYJF0Y:B
TP:>SXh5oooo0@00007oool001Goool00`000?ooooooo`02oooo0Xb<S0::RXX2R8R80X>3P`9oOgl2
OWin0X>3P`:BTY82YZJV0[Ng]`:m_Kd2]KFe0Z:RXP:>SXh2Q8B40XV9R@:JVYX2[jn_0/71`@;<c<`2
clo?0]3@d0;?cll2c/k>0/g=c@;;bl/2alO70/;2`P:l_;`2^;Rh0[Jf]P:g]kL2^[Zj0[bl_0:j^[X2
/K6a0ZBTY0:GUiL2Shn?0Y2@T0:FUYH2WYjN0ZFUY@BWYjL2YZJV0ZNWY`:/[:`2/k>c0[Vi^@:j^[X2
][Jf0Zf][@:QXJ42Ti>C0XJ6QP9hN7P2Jf][0UmOG`9CDe<2Bd];0TE5A@93@d<2A4A400=8B4Q>CTi>
CTh00UMGE`9TI6@2LG5a0W]kN`9mOGd2Lg=c0V=SH`9CDe<2BTY:0Te=C@9JFUX2Jf][0WQhN09oOgl2
P8200X61P@:8R8P2UIFE0ZRXZ0:i^KT2`l?30/C4a0:n_[h2][Jf0[6a/@:b/[82^;Rh0/30`0;7alL2
bl_;0/g=c@;?cll2dM7A0]GEe@;Hf=P2g=cL0]kNgPKOgml2g]kN0]_Kf`;Hf=P4eMGE0]KFeP;Jf]X4
g=cL0]OGe`;>c/h2a<C40[^k^`:c/k<2[Zj^0ZVYZ@:TY:@2WinO0YZJVP:FUYH2TY:B0Xj>SPGoool1
00000Oooo`005Oooo`030000oooooooo00;oool2W9bL0Zn_[`:l_;`2`L710[jn_P:i^KT2]kNg0[^k
^`;3`l<2b/[:0/g=c@;:b/X2`l?30[fm_@:i^KT2^k^k0/71`@;9bLT2clo?0]?Cd`KDe=@6eMGE0]CD
e0;Bd]82c/k>0/_;b`;9bLT2b<S80/O7a`;5aLD2`L710[Vi^@:_[jl2Y:BT0Y^KV`:DU9@4TI6A0YBD
U0:IVIT2X:2P0ZNWY`:][Jd2[jn_0Zj^[P:ZZZX2XZ:R0YRHV0:;Rh/2OWin0W9bLP9XJ6P2HF5Q0UYJ
FP9BDU82BDU90Slo?`8g=cL2<30`0bd];@8`<302>CTi0TQ8B09IFET2JFUY0W1`L09[Jf/2GUiN0U1@
D099BDT2CTi>0UaLG09]KFd2Ng]k1861P@:2PX82QhN70Y>CT`:TY:@2]KFe0/;2`P;6a/H2aLG50/?3
``;2`/82a/K60/c<c0;Bd]82e]KF0]OGe`;Fe]H2e=CD1=?Cd`;De=@2eMGE2=KFePCEeMD2e=CD0]?C
d`;De=@2eMGE0]OGe`;Hf=P2eMGE0/o?c`;5aLD2^[Zj0[:b/PB^[Zh2[Jf]0ZVYZ@:NWYh2TI6A0XJ6
QP:3Ph<2S8b<1Oooo`400001oooo000Eoooo0P0000?oool2W9bL0Zn_[`:l_;`2`L710[jn_P:i^KT2
]kNg0[^k^`;3`l<2b/[:0/g=c@;:b/X2`l?30[fm_@:i^KT2^k^k0/71`@;9bLT2clo?0]?Cd`KDe=@6
eMGE0]CDe0;Bd]82c/k>0/_;b`;9bLT2b<S80/O7a`;5aLD2`L710[Vi^@:_[jl2Y:BT0Y^KV`:DU9@4
TI6A0YBDU0:IVIT2X:2P0ZNWY`:][Jd2[jn_0Zj^[P:ZZZX2XZ:R0YRHV0:;Rh/2OWin0W9bLP9XJ6P2
HF5Q0UYJFP9BDU82BDU90Slo?`8g=cL2<30`0bd];@8`<302>CTi0TQ8B09IFET2JFUY0W1`L09[Jf/2
GUiN0U1@D099BDT2CTi>0UaLG09]KFd2Ng]k1861P@:2PX82QhN70Y>CT`:TY:@2]KFe0/;2`P;6a/H2
aLG50/?3``;2`/82a/K60/c<c0;Bd]82e]KF0]OGe`;Fe]H2e=CD1=?Cd`;De=@2eMGE2=KFePCEeMD2
e=CD0]?Cd`;De=@2eMGE0]OGe`;Hf=P2eMGE0/o?c`;5aLD2^[Zj0[:b/PB^[Zh2[Jf]0ZVYZ@:NWYh2
TI6A0XJ6QP:3Ph<2S8b<1?ooo`800001oooo000Eoooo00<0003oooooool00_ooo`:/[:`2bl_;0^?S
h`;/k>`2jn_[0^CTi0;MgMd2f=SH1=OGe`;Hf=P2f][J0]cLg0;Ng]h4h>3P0]kNgP;Kfm/2f=SH1]KF
eP;GemL2f=SH0][JfP;Lg=`4gMgM0]_Kf`;IfMT2emOG0]CDe0;AdM42cLg=0/S8b0;2`/82^k^k0[>c
/`:ZZZX2WinO0YBDU0:<S8`2Rh^;0Y:BTP:NWYh2ZZZZ0[6a/@:`/;02YjNW0YZJVP:;Rh/2Ng]k0Va/
K09OGel2EEEE0Tm?C`9=CDd2Bd];0TM7A`90@402=cLg0Rd];@8T92@27QhN00<K6a/L71`L71`00R<S
8`8b<S82A4A40UIFEP9PH602GemO0UMGE`9=CDd2B4Q80Te=C@9IFET2J6QX0WEeM@9mOGd2PH610XF5
Q@:=SHd2VIVI0ZRXZ0:g]kL2`/;20/W9b@;<c<`2c/k>0]3@d0;De=@2fMWI0]gMg@;Ogml2g]kN0]_K
f`;GemL2dm?C0]7Ad@;@d=04clo?2<k>cP;?cll2d=3@0]7Ad@;Bd]82dm?C0]CDe0;Cdm<2d=3@0/W9
b@:o_kl2]KFe1:j^[P:`/;02/k>c0Zj^[P:QXJ42Shn?0X61P@9oOgl2T92@1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`:/[:`2bl_;0^?Sh`;/k>`2jn_[0^CTi0;MgMd2f=SH1=OGe`;Hf=P2
f][J0]cLg0;Ng]h4h>3P0]kNgP;Kfm/2f=SH1]KFeP;GemL2f=SH0][JfP;Lg=`4gMgM0]_Kf`;IfMT2
emOG0]CDe0;AdM42cLg=0/S8b0;2`/82^k^k0[>c/`:ZZZX2WinO0YBDU0:<S8`2Rh^;0Y:BTP:NWYh2
ZZZZ0[6a/@:`/;02YjNW0YZJVP:;Rh/2Ng]k0Va/K09OGel2EEEE0Tm?C`9=CDd2Bd];0TM7A`90@402
=cLg0Rd];@8T92@27QhN00<K6a/L71`L71`00R<S8`8b<S82A4A40UIFEP9PH602GemO0UMGE`9=CDd2
B4Q80Te=C@9IFET2J6QX0WEeM@9mOGd2PH610XF5Q@:=SHd2VIVI0ZRXZ0:g]kL2`/;20/W9b@;<c<`2
c/k>0]3@d0;De=@2fMWI0]gMg@;Ogml2g]kN0]_Kf`;GemL2dm?C0]7Ad@;@d=04clo?2<k>cP;?cll2
d=3@0]7Ad@;Bd]82dm?C0]CDe0;Cdm<2d=3@0/W9b@:o_kl2]KFe1:j^[P:`/;02/k>c0Zj^[P:QXJ42
Shn?0X61P@9oOgl2T92@1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`:j^[X2fMWI0_3`
l0;jn_X2nOWi0_;blP;YjNT2hn?S0]oOg`CLg=`2gmoO0^CTi0;Zj^X2kNg]0^c/k0;WinL2hN7Q0]_K
f`;GemL4e]KF0]OGe`;IfMT2fm_K0]gMg@;Ng]h4gmoO0]kNgP;Lg=`2fMWI0]KFeP;AdM42cLg=0/[:
bP;6a/H2`/;20[Zj^P:^[Zh2XJ6Q0YFEU@:ATI42UiNG0ZBTY0:_[jl2]KFe0Zn_[`:OWil2S8b<0WQh
N09VIVH2F5QH0Ta<C094A4@2@D511410@08o?cl2>CTi0S4a<@8X:2P2820P0QXJ6P<F5QH271`L0RLW
9`8f=SH2ADE50Tm?C`9ADE42CTi>0TU9B@96ATH2AdM70Te=C@9FEUH2HV9R0Vm_K`9nOWh2SHf=0YbL
W0:YZJT2]KFe0[jn_P;6a/H2c<c<0]3@d0;De=@2emOG0]_Kf`;Ng]h2h>3P0^7Qh@;Ogml2g=cL0]OG
e`;De=@2dM7A0/o?c`;>c/h4cLg=1<c<c0;=cLd2c/k>0]3@d0;AdM42d];B1=?Cd`;Bd]82c/k>0/S8
b0;0`<02^;Rh1;>c/`:f][H2^KVi0[Jf]P:YZJT2VIVI0Xb<S0:;Rh/2WIfM1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`:j^[X2fMWI0_3`l0;jn_X2nOWi0_;blP;YjNT2hn?S0]oOg`CLg=`2
gmoO0^CTi0;Zj^X2kNg]0^c/k0;WinL2hN7Q0]_Kf`;GemL4e]KF0]OGe`;IfMT2fm_K0]gMg@;Ng]h4
gmoO0]kNgP;Lg=`2fMWI0]KFeP;AdM42cLg=0/[:bP;6a/H2`/;20[Zj^P:^[Zh2XJ6Q0YFEU@:ATI42
UiNG0ZBTY0:_[jl2]KFe0Zn_[`:OWil2S8b<0WQhN09VIVH2F5QH0Ta<C094A4@2@D511410@08o?cl2
>CTi0S4a<@8X:2P2820P0QXJ6P<F5QH271`L0RLW9`8f=SH2ADE50Tm?C`9ADE42CTi>0TU9B@96ATH2
AdM70Te=C@9FEUH2HV9R0Vm_K`9nOWh2SHf=0YbLW0:YZJT2]KFe0[jn_P;6a/H2c<c<0]3@d0;De=@2
emOG0]_Kf`;Ng]h2h>3P0^7Qh@;Ogml2g=cL0]OGe`;De=@2dM7A0/o?c`;>c/h4cLg=1<c<c0;=cLd2
c/k>0]3@d0;AdM42d];B1=?Cd`;Bd]82c/k>0/S8b0;0`<02^;Rh1;>c/`:f][H2^KVi0[Jf]P:YZJT2
VIVI0Xb<S0:;Rh/2WIfM1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`;4a<@2f][J0^[Z
jP;alO42kno_0^WYj@;Shn<2g]kN0]cLg0CJf]X2gMgM0^3Ph0;Ti>@2i^KV0^GUi@;Rh^82gMgM0]WI
f@;GemL4e]KF1=OGe`;Hf=P2fMWI0][JfP;Kfm/2g=cL0]_Kf`;Jf]X2fMWI0]KFeP;Cdm<2dM7A1=3@
d0;?cll2b/[:0/30`0:c/k<2Z:RX0Z:RXP:VYZH2[jn_1;Ng]`:/[:`2V9RH0X61P@9[Jf/2FUYJ0Te=
C@93@d<2?Cdm0SXj>P@i>CT2>3Ph0S@d=08^;Rh2:2PX0R8R8P8N7Qh371`L0R0P808V9RH2;bl_0SPh
>08n?Sh2@T920TA4A093@d<2@D510Shn?P8m?Cd2?clo0TY:BP9OGel2Ng]k0YRHV0:`/;02_kno0/O7
a`;:b/X2c<c<0/o?c`;Bd]82e]KF0]SHf0;Jf]X2fm_K1=cLg0;Kfm/2f=SH0]KFeP;Cdm<2d];B1=7A
d@K@d=02clo?1=3@d0;AdM44d];B1]?Cd`;AdM42c<c<0/O7a`;1`L42_[jn0[fm_@:o_kl2`<300[fm
_@:e]KD2ZZZZ1::RXP:_[jl5oooo0@00007oool001Goool00`000?ooooooo`02oooo0/C4a0;Jf]X2
j^[Z0_7al@;_knl2jNWY0^?Sh`;Ng]h2g=cL1=[JfP;MgMd2h>3P0^CTi0;Vi^H2iNGU0^;RhP;MgMd2
fMWI0]OGe`CFe]H4emOG0]SHf0;IfMT2f][J0]_Kf`;Lg=`2fm_K0][JfP;IfMT2e]KF0]?Cd`;AdM44
d=3@0/o?c`;:b/X2`<300[>c/`:XZ:P2XZ:R0ZJVYP:_[jl4]kNg0Zb/[0:HV9P2PH610V][J`9JFUX2
CDe=0T=3@`8m?Cd2>SXj13Ti>@8h>3P2=3@d0Rh^;P8X:2P28R8R0QhN7P<L71`2820P0RHV9P8_;bl2
>3Ph0Shn?P92@T82A4A40T=3@`91@D42?Shn0Sdm?@8o?cl2BTY:0UmOG`9kNg/2V9RH0[2`/0:o_kl2
alO70/[:bP;<c<`2clo?0];BdP;Fe]H2f=SH0][JfP;Kfm/4g=cL0]_Kf`;Hf=P2e]KF0]?Cd`;Bd]84
dM7A1]3@d0;?cll4d=3@0]7Ad@CBd]86dm?C0]7Ad@;<c<`2alO70/71`@:n_[h2_Kfm0[no_`;0`<02
_Kfm0[Fe]@:ZZZX4XZ:R0Zn_[`Goool100000Oooo`005Oooo`030000oooooooo00;oool2c<c<0]KF
eP;Lg=`2g]kN0]_Kf`;Hf=P2e=CD0]?Cd`;De=@2eMGE1]KFePCGemL4e]KF3=GEe@SDe=@4eMGE1=KF
eP;EeMD2e=CD1=?Cd`;EeMD2emOG0]WIf@;GemL2d=3@0/O7a`:m_Kd4^;Rh0[bl_0:n_[h2^KVi0ZVY
Z@:BTY82NGUi0V=SH`9BDU82AdM70T10@08k>c/2=cLg0SDe=@8d=3@2<c<c0S0`<08];Bd2:RXZ0RLW
9`<V9RH29bLW0RPX:08Z:RX2:b/[0Rd];@8`<302=3@d0STi>@@m?Cd2=cLg0S0`<08^;Rh2>3Ph0UAD
E09lO7`2Y:BT0/C4a0;EeMD2f=SH0]GEe@CAdM42dm?C0]GEe@CFe]H6eMGE2=CDe0;Cdm<@e=CD1]?C
d`KDe=@2dm?C0];BdP;?cll2c<c<0/[:bPC8b<P2alO70/G5a@;2`/82_[jn0[^k^`:m_Kd2`l?31Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`;<c<`2e]KF0]cLg0;Ng]h2fm_K0]SHf0;De=@2
dm?C0]CDe0;EeMD6e]KF1=OGe`CFe]H<eMGE2=CDe0CEeMD4e]KF0]GEe@;De=@4dm?C0]GEe@;GemL2
fMWI0]OGe`;@d=02alO70[fm_@Bh^;P2_;bl0[jn_P:i^KT2ZJVY0Y:BTP9iNGT2Hf=S0U9BDP97AdL2
@4100S/k>`8g=cL2=CDe0S@d=08c<c<2<30`0Rd];@8Z:RX29bLW0bHV9P8W9bL2:2PX0RXZ:P8[:b/2
;Bd]0S0`<08d=3@2>CTi13dm?@8g=cL2<30`0Rh^;P8h>3P2E5AD0WalO0:TY:@2a<C40]GEe@;Hf=P2
eMGE1=7Ad@;Cdm<2eMGE1=KFePKEeMD8e=CD0]?Cda3De=@6dm?C1]CDe0;Cdm<2d];B0/o?c`;<c<`2
b/[:1<S8b0;7alL2aLG50/;2`P:n_[h2^k^k0[fm_@;3`l<5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo0];BdP;AdM42clo?0/g=c@;;bl/2bLW90/S8b0;:b/X2cLg=0]3@d0;Bd]82dM7A0/k>
cP;;bl/2bLW90/[:bP;<c<`2clo?0]7Ad@KCdm<4d];B0]7Ad@;@d=04clo?0]3@d0CAdM46d];B0]CD
e0;GemL2fm_K0]kNgP;Ogml2g=cL0]KFeP;?cll2b/[:0/S8b0;7alL2a<C40[Zj^P:WYjL2SXj>0WAd
M09NGUh2CTi>0T=3@`8m?Cd2>3Ph0S@d=08`<302;Rh^0Rd];@H/;2`2;Bd]0Rh^;P03;bl_<30`<30`
008_;bl2;Bd]0RXZ:P@W9bL2:RXZ0Rl_;`8e=CD2=cLg0S8b<P8/;2`2:b/[0STi>@9IFET2QHF50[:b
/P;De=@4i>CT0]gMg@;Fe]H4dm?C1=CDe0;Cdm<2d=3@1/o?c`;AdM42d];B0]CDe0;EeMD6e]KF2=OG
e`;Fe]H4eMGE1=CDe0[EeMD2e=CD0]?Cd`;AdM42clo?0/g=c@C<c<`2c/k>0]3@d0CBd]85oooo0@00
007oool001Goool200000oooo`;Bd]82dM7A0/o?c`;=cLd2bl_;0/W9b@;8b<P2b/[:0/g=c@;@d=02
d];B0]7Ad@;>c/h2bl_;0/W9b@;:b/X2c<c<0/o?c`;AdM46dm?C1=;BdP;AdM42d=3@1<o?c`;@d=04
dM7A1];BdP;De=@2emOG0]_Kf`;Ng]h2gmoO0]cLg0;Fe]H2clo?0/[:bP;8b<P2alO70/C4a0:j^[X2
YjNW0Xj>SP9dM7@2GUiN0Ti>CP93@d<2?Cdm0SPh>08d=3@2<30`0Rh^;P8];Bd6;2`/0Rd];@8^;Rh0
0bl_;c0`<30`<002;bl_0Rd];@8Z:RX49bLW0RXZ:P8_;bl2=CDe0SLg=`8b<S82;2`/0R/[:`8i>CT2
FEUI0XF5Q@:b/[82e=CD1>CTi0;MgMd2e]KF1=?Cd`CDe=@2dm?C0]3@d0K?cll2dM7A0];BdP;De=@2
eMGE1]KFePSGemL2e]KF1=GEe@CDe=@:eMGE0]CDe0;Cdm<2dM7A0/o?c`;=cLd4c<c<0/k>cP;@d=04
d];B1?ooo`800001oooo000Eoooo00<0003oooooool00_ooo`;EeMD2d=3@0/_;b`;8b<P6a/K60/W9
b@;<c<`2clo?0]7Ad@;?cll2c<c<0/S8b0;6a/H2alO70/[:bP;=cLd2d=3@2=;BdP;AdM42d=3@0/o?
c`C>c/h4clo?0]3@d0;AdM42d];B0]?Cd`;EeMD2f=SH0]_Kf`;Ng]h2h>3P0]oOg`;Lg=`2fMWI0]GE
e@;Bd]82c/k>0/O7a`:j^[X2YJFU0Xb<S09cLg<2G5aL0T];B`90@402>CTi0S<c<`8];Bd2:BTY0RHV
9P8U9BD29bLW0RXZ:P8];Bd2<30`0S4a<@03<S8b<C4a<C4a008`<302;Rh^0R/[:`8X:2P49RHV0RLW
9`8Z:RX2;Bd]0S0`<08e=CD2?clo0U=CD`9cLg<2VIVI0[no_`;Jf]X4i^KV0]kNgP;GemL4dm?C1=CD
e0;Bd]82clo?0/k>cP;=cLd2c/k>0]3@d0;Bd]82e=CD0]GEe@CFe]H:emOG1=KFeP[EeMD4e]KF0]OG
e`;Hf=P2emOG0]GEe@;Cdm<2dM7A0]3@d0;Cdm<2e]KF0]WIf@;Kfm/2fMWI1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`;EeMD2d=3@0/_;b`;8b<P6a/K60/W9b@;<c<`2clo?0]7Ad@;?cll2
c<c<0/S8b0;6a/H2alO70/[:bP;=cLd2d=3@2=;BdP;AdM42d=3@0/o?c`C>c/h4clo?0]3@d0;AdM42
d];B0]?Cd`;EeMD2f=SH0]_Kf`;Ng]h2h>3P0]oOg`;Lg=`2fMWI0]GEe@;Bd]82c/k>0/O7a`:j^[X2
YJFU0Xb<S09cLg<2G5aL0T];B`90@402>CTi0S<c<`8];Bd2:BTY0RHV9P8U9BD29bLW0RXZ:P8];Bd2
<30`0S4a<@03<S8b<C4a<C4a008`<302;Rh^0R/[:`8X:2P49RHV0RLW9`8Z:RX2;Bd]0S0`<08e=CD2
?clo0U=CD`9cLg<2VIVI0[no_`;Jf]X4i^KV0]kNgP;GemL4dm?C1=CDe0;Bd]82clo?0/k>cP;=cLd2
c/k>0]3@d0;Bd]82e=CD0]GEe@CFe]H:emOG1=KFeP[EeMD4e]KF0]OGe`;Hf=P2emOG0]GEe@;Cdm<2
dM7A0]3@d0;Cdm<2e]KF0]WIf@;Kfm/2fMWI1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`;EeMD2d];B0/o?c`;=cLd6c<c<0/k>cP;?cll2dM7A0];BdP;AdM42clo?0/g=c@C<c<`2c/k>0]3@
d0;AdM42d];B0]?Cd`KBd]84dM7A1=3@d0CAdM44d];B0]?Cd`;De=@2eMGE0]KFeP;Hf=P2fMWI1=_K
f`;Lg=`2f][J0]SHf0;Fe]H2d=3@0/S8b0:j^[X2YJFU0Xf=S@9dM7@2GEeM0T];B`8m?Cd2<c<c0R`/
;08U9BD2820P11dM7@8Q8B429bLW0Rd];@8`<302<C4a00<`<30];Bd];Bd012/[:`8];Bd2;Rh^0R`/
;08W9bL28R8R0R0P808U9BD2<c<c0TQ8B09RHV82OWin0YZJVP:d];@2b/[:0]WIf@;Ph>02gmoO0]_K
f`;Fe]H4dm?C1=CDe0;Cdm<2dM7A1]3@d0;AdM42dm?C0]CDe0cEeMD4e]KF1]GEe@SDe=@4eMGE0]KF
ePCGemL2e]KF0]CDe0CCdm<2eMGE0]OGe`;IfMT2f][J0]WIf@Goool100000Oooo`005Oooo`030000
oooooooo00;oool2eMGE0];BdP;?cll2cLg=1/c<c0;>c/h2clo?0]7Ad@;Bd]82dM7A0/o?c`;=cLd4
c<c<0/k>cP;@d=02dM7A0];BdP;Cdm<6d];B1=7Ad@C@d=04dM7A1=;BdP;Cdm<2e=CD0]GEe@;Fe]H2
f=SH0]WIf@CKfm/2g=cL0][JfP;Hf=P2e]KF0]3@d0;8b<P2^[Zj0ZFUY@:=SHd2M7Ad0UeMG@9;Bd/2
?Cdm0S<c<`8/;2`29BDU0R0P80@M7Ad28B4Q0RLW9`8];Bd2<30`0S4a<@03<30`;Bd];Bd]00@[:b/2
;Bd]0Rh^;P8/;2`29bLW0R8R8P8P82029BDU0S<c<`98B4P2HV9R0WinOP:JVYX2];Bd0/[:bP;IfMT2
h>3P0]oOg`;Kfm/2e]KF1=?Cd`CDe=@2dm?C0]7Ad@K@d=02dM7A0]?Cd`;De=@<eMGE1=KFePKEeMD8
e=CD1=GEe@;Fe]H4emOG0]KFeP;De=@4dm?C0]GEe@;GemL2fMWI0][JfP;IfMT5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo0]GEeBCDe=@<dm?C1]CDe0KCdm<4e=CD1=GEe@;De=@6dm?C0]GE
e@;GemL4f=SH0]OGe`;Bd]82bLW90[^k^`:XZ:P2TI6A0WQhN09PH602C4a<0S`l?08_;bl29BDU0QhN
7P8I6AT261PH0QXJ6P8O7al29BDU0R/[:`@_;bl00b`/;2PX:2PX:0029RHV0RTY:@8^;Rh2=3@d0S<c
<`8[:b/2820P0QXJ6P8R8R82>c/k0V5QH@::RXX2[Zj^0/C4a0;?cll2dm?C1=GEe@;Fe]H2eMGE0]CD
e0CCdm<8e=CD2=?CdbSDe=@8eMGE1=CDe0[EeMD5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0]GEeBCDe=@<dm?C1]CDe0KCdm<4e=CD1=GEe@;De=@6dm?C0]GEe@;GemL4f=SH0]OGe`;Bd]82
bLW90[^k^`:XZ:P2TI6A0WQhN09PH602C4a<0S`l?08_;bl29BDU0QhN7P8I6AT261PH0QXJ6P8O7al2
9BDU0R/[:`@_;bl00b`/;2PX:2PX:0029RHV0RTY:@8^;Rh2=3@d0S<c<`8[:b/2820P0QXJ6P8R8R82
>c/k0V5QH@::RXX2[Zj^0/C4a0;?cll2dm?C1=GEe@;Fe]H2eMGE0]CDe0CCdm<8e=CD2=?CdbSDe=@8
eMGE1=CDe0[EeMD5oooo0@00007oool001Goool00`000?ooooooo`02oooo0]?Cd`;EeMD2emOG2=WI
f@;Hf=P2emOG0]KFeP;EeMD2e]KF0]OGe`;Hf=P4fMWI0]SHf0;Fe]H4eMGE1=CDe0KEeMD8e]KF2]GE
e@;De=@2d];B0]3@d0C>c/h2clo?0];BdP;De=@4e]KF0]CDe0;>c/h2`/;20[2`/0:IVIT2Ogmo0VIV
IP9?Cdl2>c/k0R/[:`8Q8B426a/K0QXJ6P8K6a/27alO0R<S8`8W9bL2:RXZ12/[:`03:2PX9BDU9BDU
008T92@2:2PX0Rh^;P@e=CD2;Rh^0RDU9@8Q8B42;Bd]0Tm?C`9nOWh2[Zj^0]?Cd`;Shn<2h^;R0]WI
f@;@d=02cLg=0/k>cP;AdM46dm?C1=CDe0;EeMD6e]KF1=GEe@CDe=@Zdm?C0]CDe0CEeMD2e=CD0];B
dPCAdM42d];B1Oooo`400001oooo0009oooo0P0000Coool00`000?ooooooo`03oooo00<0003ooooo
ool00_ooo`;Cdm<2eMGE0]OGe`SIfMT2f=SH0]OGe`;Fe]H2eMGE0]KFeP;GemL2f=SH1=WIf@;Hf=P2
e]KF1=GEe@CDe=@6eMGE2=KFeP[EeMD2e=CD0];BdP;@d=04c/k>0/o?c`;Bd]82e=CD1=KFeP;De=@2
c/k>0/;2`P:`/;02VIVI0WmoO`9VIVH2Cdm?0S/k>`8[:b/28B4Q0Q/K6`8J6QX26a/K0QlO7`8S8b<2
9bLW0RXZ:P@[:b/00bPX:2DU9BDU9@0292@T0RPX:08^;Rh4=CDe0Rh^;P8U9BD28B4Q0Rd];@9?Cdl2
OWin0Zj^[P;Cdm<2hn?S0^;RhP;IfMT2d=3@0/g=c@;>c/h2dM7A1]?Cd`CDe=@2eMGE1]KFePCEeMD4
e=CD:]?Cd`;De=@4eMGE0]CDe0;Bd]84dM7A0];BdPGoool100000Oooo`002Oooo`030000oooo0000
00;oool00`000?ooo`000004oooo00<0003oooooool00_ooo`;Cdm<2eMGE0]OGe`;IfMT6f][J0]WI
f@;GemL2e]KF0]GEe@;Fe]H2emOG0]WIf@;Jf]X2fMWI0]SHf0;GemL<eMGE2]KFePSEeMD2e=CD0]?C
d`;Bd]82d=3@0/o?c`C>c/h2clo?0]3@d0;Cdm<2e]KF0]WIf@;GemL2c/k>0[fm_@:UYJD2RHV90Ve]
K@9BDU82>c/k0RTY:@8O7al27AdM0R8R8P8Z:RX2<30`0S4a<@8];Bd2:BTY0RDU9@8T92@00bHV9RPX
:2PX:002:BTY0RXZ:P8[:b/2;Bd]12h^;P8_;bl2=cLg0Ta<C09_Kfl2W9bL0/K6aP;Ti>@2k^k^0^SX
j0;Jf]X2c/k>0/[:bP;<c<`2d=3@0]?Cd`CDe=@2dm?C0]CDe0;EeMD2e]KF1=OGe`;Fe]H2eMGE1=CD
e1[Cdm<2e=CD2=?Cd`CBd]82dm?C0]CDe0CEeMD2dm?C0];BdPC@d=02dM7A1Oooo`400001oooo0009
oooo00<0003oool000000_ooo`030000oooo000000Coool200000oooo`;Cdm<2eMGE0]OGe`;IfMT6
f][J0]WIf@;GemL2e]KF0]GEe@;Fe]H2emOG0]WIf@;Jf]X2fMWI0]SHf0;GemL<eMGE2]KFePSEeMD2
e=CD0]?Cd`;Bd]82d=3@0/o?c`C>c/h2clo?0]3@d0;Cdm<2e]KF0]WIf@;GemL2c/k>0[fm_@:UYJD2
RHV90Ve]K@9BDU82>c/k0RTY:@8O7al27AdM0R8R8P8Z:RX2<30`0S4a<@8];Bd2:BTY0RDU9@8T92@0
0bHV9RPX:2PX:002:BTY0RXZ:P8[:b/2;Bd]12h^;P8_;bl2=cLg0Ta<C09_Kfl2W9bL0/K6aP;Ti>@2
k^k^0^SXj0;Jf]X2c/k>0/[:bP;<c<`2d=3@0]?Cd`CDe=@2dm?C0]CDe0;EeMD2e]KF1=OGe`;Fe]H2
eMGE1=CDe1[Cdm<2e=CD2=?Cd`CBd]82dm?C0]CDe0CEeMD2dm?C0];BdPC@d=02dM7A1?ooo`800001
oooo0009oooo0P0000?oool00`000?ooo`000004oooo00<0003oooooool00_ooo`;Cdm<2e=CD0]GE
e@;Fe]H6emOG1=KFePKEeMD4e]KF1=OGe`;Fe]H4eMGE1]CDe1;EeMD4e=CD1=?Cd`;Bd]84dm?C0];B
dP;@d=02c/k>0/g=c@;?cll2e]KF0]kNgP;Rh^82g=cL0/_;b`:a/K42TY:B0W9bLP9DE5@2>c/k0RPX
:08P820292@T0S4a<@90@402B4Q80TA4A08h>3P2:BTY11lO7`039RHV;Rh^;Rh^008a<C42;Bd]0RHV
9P8P82028B4Q0R/[:`8n?Sh2EUIF0W=cL`:DU9@2][Jf0]?Cd`;Vi^H2j^[Z0^CTi0;IfMT2clo?0/_;
b`;=cLd2dM7A0]CDe0;EeMD2e=CD1=?Cd`;De=@:eMGE1=CDe1CCdm<<e=CD2=?Cd`[De=@2dm?C1];B
dPGoool100000Oooo`002Oooo`030000oooooooo00;oool00`000?ooo`000004oooo00<0003ooooo
ool00_ooo`;Cdm<2e=CD0]GEe@;Fe]H6emOG1=KFePKEeMD4e]KF1=OGe`;Fe]H4eMGE1]CDe1;EeMD4
e=CD1=?Cd`;Bd]84dm?C0];BdP;@d=02c/k>0/g=c@;?cll2e]KF0]kNgP;Rh^82g=cL0/_;b`:a/K42
TY:B0W9bLP9DE5@2>c/k0RPX:08P820292@T0S4a<@90@402B4Q80TA4A08h>3P2:BTY11lO7`039RHV
;Rh^;Rh^008a<C42;Bd]0RHV9P8P82028B4Q0R/[:`8n?Sh2EUIF0W=cL`:DU9@2][Jf0]?Cd`;Vi^H2
j^[Z0^CTi0;IfMT2clo?0/_;b`;=cLd2dM7A0]CDe0;EeMD2e=CD1=?Cd`;De=@:eMGE1=CDe1CCdm<<
e=CD2=?Cd`[De=@2dm?C1];BdPGoool100000Oooo`002_ooo`800003oooo00<0003oooooool00ooo
o`030000oooooooo00;oool4e=CD1=?CdcSDe=@2dm?C1=;BdP;Cdm<2e]KF0]SHf0;GemL2e=CD0/k>
cP;:b/X2c<c<0]KFeP;Rh^82j^[Z0^SXj0;Fe]H2^KVi0YNGU`9dM7@2E5AD0SXj>P8X:2P28b<S0Rd]
;@90@402EEEE0UmOG`9GEeL2@d=30R/[:`@K6a/00bLW9c<c<c<c<`02>CTi0S0`<08Q8B425ADE0QHF
5P8Y:BT2BTY:0W5aL@:EUID2/k>c0/S8b0;GemL2gmoO0^3Ph0;MgMd2e]KF0]3@d0;>c/h2d=3@0]?C
d`CFe]H2eMGE0]CDe0KCdm=6e=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`CDe=@4
dm?C>=CDe0;Cdm<4d];B0]?Cd`;Fe]H2f=SH0]OGe`;De=@2c/k>0/[:bP;<c<`2e]KF0^;RhP;Zj^X2
j>SX0]KFeP:i^KT2UiNG0WAdM09DE5@2>SXj0RPX:08S8b<2;Bd]0T10@09EEED2GemO0UMGE`93@d<2
:b/[11/K6`039bLW<c<c<c<c008i>CT2<30`0R4Q8@8E5AD25QHF0RTY:@9:BTX2LG5a0YFEU@:c/k<2
b<S80]OGe`;Ogml2h>3P0]gMg@;Fe]H2d=3@0/k>cP;@d=02dm?C1=KFeP;EeMD2e=CD1]?CddKDe=@5
oooo0@00007oool001Goool00`000?ooooooo`02oooo0]CDe0CCdm<:d];B2]?Cd`SBd]86dm?C1=CD
e0kCdm<4e=CD1=?Cd`CBd]82e=CD0]SHf0CKfm/2e]KF0/o?c`;9bLT2b/[:0]GEe@;Shn<2kNg]0^_[
j`;GemL2][Jf0Y6AT@9/K6`2CDe=0SHf=P8X:2P29bLW0SDe=@9<C4`2Hf=S0Vi^KP9UIFD2CDe=0S4a
<@8N7Qh27AdM00<X:2Pf=SHf=SH00S/k>`8a<C427alO0Q4A4@8B4Q82:BTY0U1@D09kNg/2XJ6Q0[bl
_0;<c<`2eMGE1=WIf@;GemL2dm?C1=3@d0;AdM42eMGE1=OGe`;Fe]H2e=CD0]?Cd`CBd]84dm?C;]CD
e0;EeMD4e=CD1=?Cd`CDe=@6eMGE1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`;De=@4
dm?C2];BdP[Cdm<8d];B1]?Cd`CDe=@>dm?C1=CDe0CCdm<4d];B0]CDe0;Hf=P4fm_K0]KFeP;?cll2
bLW90/[:bP;EeMD2hn?S0^g]k@;[jn/2emOG0[Jf]P:ATI42K6a/0Te=C@8f=SH2:2PX0RLW9`8e=CD2
C4a<0V=SH`9^KVh2IFEU0Te=C@8a<C427QhN0QdM7@03:2PX=SHf=SHf008k>c/2<C4a0QlO7`8A4A42
4Q8B0RTY:@9@D502Ng]k0Z6QX@:l_;`2c<c<0]GEe@CIfMT2emOG0]?Cd`C@d=02dM7A0]GEe@CGemL2
e]KF0]CDe0;Cdm<4d];B1=?CdbkDe=@2eMGE1=CDe0CCdm<4e=CD1]GEe@Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2e=CD1=?Cd`[Bd]8:dm?C2=;BdQSCdm<6e=CD0]?Cd`CBd]82e=CD0]SH
f0CKfm/2emOG0]3@d0;:b/X2bl_;0]CDe0;QhN42jNWY0^?Sh`;;bl/2YZJV0WinOP9IFET2?clo0Rl_
;`8X:2P2;Bd]0S`l?09BDU82IVIV0W1`L09XJ6P2E5AD0S/k>`8Y:BT29BDU00<[:b/d=3@d=3@00SHf
=P8_;bl28R8R0QTI6@8K6a/2;2`/0TY:BP9[Jf/2S8b<0ZRXZ0:n_[h2clo?0]WIf@;Kfm/2fMWI0]?C
d`;?cll2c/k>0]7Ad@;EeMD4f=SH0]KFeP;De=@6d];B1=?CdcCDe=@4dm?C1=CDe0KEeMD5oooo0@00
007oool001Goool00`000?ooooooo`02oooo0]CDe0CCdm<:d];B2]?Cd`SBd]8Hdm?C1]CDe0;Cdm<4
d];B0]CDe0;Hf=P4fm_K0]OGe`;@d=02b/[:0/_;b`;De=@2hN7Q0^WYj@;Shn<2bl_;0ZJVYP9nOWh2
FEUI0Slo?`8_;bl2:2PX0Rd];@8l?3`2DU9B0VIVIP9`L702J6QX0UADE08k>c/2:BTY0RDU9@03:b/[
=3@d=3@d008f=SH2;bl_0R8R8P8I6AT26a/K0R`/;09:BTX2Jf][0Xb<S0:XZ:P2_[jn0/o?c`;IfMT2
fm_K0]WIf@;Cdm<2clo?0/k>cP;AdM42eMGE1=SHf0;Fe]H2e=CD1];BdPCCdm<de=CD1=?Cd`CDe=@6
eMGE1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`CDe=@@dm?C0]CDe0kCdm<<e=CD2=?C
d`SDe=@6dm?C0]CDe0;Fe]H4f=SH0]KFeP;AdM42cLg=0/k>cP;De=@2fm_K0]gMg@;AdM42];Bd0Xb<
S09SHf<2@D510Rd];@8W9bL2:b/[0SDe=@93@d<2De=C0V5QH@9XJ6P2I6AT0UQHF099BDT2>c/k0S<c
<`03<30`;bl_;bl_008^;Rh2:b/[0RTY:@8Z:RX2;Bd]0S8b<P8k>c/2BDU90V1PH09oOgl2Xj>S0/C4
a0;Kfm/2hn?S0]kNgP;EeMD2cLg=0/_;b`;?cll2e=CD1=SHf0;Fe]H2e=CD1=;BdPCCdm<fe=CD1=?C
d`[De=@5oooo0@00007oool001Goool200000oooo`CDe=@@dm?C0]CDe0kCdm<<e=CD2=?Cd`SDe=@6
dm?C0]CDe0;Fe]H4f=SH0]KFeP;AdM42cLg=0/k>cP;De=@2fm_K0]gMg@;AdM42];Bd0Xb<S09SHf<2
@D510Rd];@8W9bL2:b/[0SDe=@93@d<2De=C0V5QH@9XJ6P2I6AT0UQHF099BDT2>c/k0S<c<`03<30`
;bl_;bl_008^;Rh2:b/[0RTY:@8Z:RX2;Bd]0S8b<P8k>c/2BDU90V1PH09oOgl2Xj>S0/C4a0;Kfm/2
hn?S0]kNgP;EeMD2cLg=0/_;b`;?cll2e=CD1=SHf0;Fe]H2e=CD1=;BdPCCdm<fe=CD1=?Cd`[De=@4
oooo0P00007oool001Goool00`000?ooooooo`02ooooD]CDe0CBd]82dm?C0];BdP;<c<`2^KVi0YRH
V09^KVh2ADE50RPX:08N7Qh28b<S0S0`<08o?cl2Bd];0U=CD`9IFET2G5aL0UeMG@9LG5`2EeMG0Ti>
CP92@T800cDe=BXZ:RXZ:P029BDU0RPX:08a<C42?3`l0Slo?`8g=cL2:b/[0R@T908^;Rh2D51@0X61
P@:d];@2fMWI0^OWi`;Rh^82e]KF0/c<c0;:b/X2c/k>0]CDe0CHf=P2e]KF0]CDe0CBd]82dm?CA]CD
e0Goool100000Oooo`005Oooo`030000oooooooo00;ooomBe=CD1=;BdP;Cdm<2d];B0/c<c0:i^KT2
V9RH0Vi^KP95ADD2:2PX0QhN7P8S8b<2<30`0Slo?`9;Bd/2De=C0UUIF@9LG5`2GEeM0UaLG09GEeL2
CTi>0T92@P03=CDe:RXZ:RXZ008U9BD2:2PX0S4a<@8l?3`2?clo0SLg=`8[:b/292@T0Rh^;P9@D502
PH610[Bd]0;IfMT2inOW0^;RhP;Fe]H2c<c<0/[:bP;>c/h2e=CD1=SHf0;Fe]H2e=CD1=;BdP;Cdm=6
e=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_oooc[De=@4dm?C0]CDe0KEeMD2dm?C0]7A
d@;@d=02dM7A0]CDe0;GemL2fMWI0]OGe`;Bd]82b<S80[Ng]`:MWId2NWYj0U9BDP8_;bl26QXJ0QPH
608V9RH2>c/k0Te=C@AEEED2DU9B0U5AD@9FEUH2GUiN0V=SH`9PH602DU9B00<m?CdY:BTY:BT00QlO
7`8V9RH2=cLg0TQ8B09<C4`2?3`l0R0P80@:2PX2;2`/0V=SH`:OWil2cLg=1>7Qh@;GemL2c/k>0/c<
c0;@d=02e=CD1=KFeP;EeMD6dm?C@]CDe0KCdm<5oooo0@00007oool001Goool00`000?ooooooo`02
oooo>]CDe0CCdm<2e=CD1]GEe@;Cdm<2dM7A0]3@d0;AdM42e=CD0]OGe`;IfMT2emOG0];BdP;8b<P2
]kNg0YfMW@9jNWX2DU9B0Rl_;`8J6QX261PH0RHV9P8k>c/2CDe=15EEE@9BDU82DE5A0UIFEP9NGUh2
Hf=S0V1PH09BDU800cdm?BTY:BTY:@027alO0RHV9P8g=cL2B4Q80Ta<C08l?3`2820P10X:2P8/;2`2
Hf=S0YnOW`;=cLd4hN7Q0]OGe`;>c/h2c<c<0]3@d0;De=@4e]KF0]GEe@KCdm=2e=CD1]?Cd`Goool1
00000Oooo`005Oooo`030000oooooooo00;ooolje=CD1=?Cd`;De=@2eMGE0]OGe`;Fe]H2dm?C0/o?
c`;=cLd2c/k>0]CDe0;Lg=`2h>3P0]gMg@;@d=02^k^k0Z2PX0:0P802GemO0Slo?`8V9RH26a/K0R4Q
8@8d=3@2Bd];0UaLG09OGel2FEUI0Tm?C`9<C4`2De=C0V5QH@9]KFd2Kfm_0V1PH003ATI6;Bd];Bd]
008O7al29BDU0SPh>09:BTX2Cdm?0Shn?P8P82021`L70P<30`8N7Qh2D51@0XN7Q`:e]KD2c/k>1=OG
e`;De=@2dm?C1=CDe0SCdm<2e=CD1=GEeD;De=@4dm?C1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_oooc[De=@4dm?C0]CDe0;EeMD2emOG0]KFeP;Cdm<2clo?0/g=c@;>c/h2e=CD0]cLg0;Ph>02
gMgM0]3@d0:k^k/2X:2P0X20P09OGel2?clo0RHV9P8K6a/28B4Q0S@d=09;Bd/2G5aL0UmOG`9IFET2
Cdm?0Ta<C09CDe<2HF5Q0Ve]K@9_Kfl2H61P00=6ATH];Bd];Bd00QlO7`8U9BD2>3Ph0TY:BP9?Cdl2
?Shn0R0P80871`L20`<30QhN7P9@D502QhN70[Fe]@;>c/h4emOG0]CDe0;Cdm<4e=CD2=?Cd`;De=@4
eMGE@]CDe0CCdm<5oooo0@00007oool001Goool00`000?ooooooo`02oooo>=CDe0;Cdm<4d];B0]CD
e0;Fe]H4f=SH0]CDe0;>c/h2b/[:0/_;b`;De=@2h>3P0^OWi`;Rh^82c/k>0Zf][@:8R8P2I6AT0TQ8
B08d=3@2:BTY0RPX:08e=CD2BDU90UiNGP9/K6`2Jf][0V1PH09CDe<2C4a<0UADE09TI6@2Lg=c0WMg
M`9YJFT00e1@D3De=CDe=@029BDU0RLW9`8f=SH2ATI60T];B`8n?Sh2:2PX0QDE5@8@41028R8R0TE5
A@9^KVh2UIFE0[:b/P;7alL2e=CD0]cLg0;MgMd2f][J0]GEe@;@d=02c/k>0]3@d0;Cdm<4e]KF0]GE
e@;De=@6dm?C?]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;ooolhe=CD0]?Cd`CBd]82
e=CD0]KFePCHf=P2e=CD0/k>cP;:b/X2bl_;0]CDe0;Ph>02inOW0^;RhP;>c/h2[Jf]0XR8R09TI6@2
B4Q80S@d=08Y:BT2:2PX0SDe=@99BDT2GUiN0Va/K09[Jf/2H61P0U=CD`9<C4`2E5AD0VATI09cLg<2
MgMg0VUYJ@03D51@=CDe=CDe008U9BD29bLW0SHf=P96ATH2Bd];0Shn?P8X:2P25ADE0Q0@408R8R82
ADE50Vi^KP:EUID2/[:b0/O7a`;De=@2g=cL0]gMg@;Jf]X2eMGE0]3@d0;>c/h2d=3@0]?Cd`CFe]H2
eMGE0]CDe0KCdm<ne=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooocSDe=@2dm?C1=;B
dP;De=@2e]KF1=WIf@;De=@2c/k>0/S8b0;:b/X2eMGE0^?Sh`;/k>`2iNGU0/S8b0:NWYh2LG5a0T];
B`8e=CD2;Bd]0S0`<08k>c/2C4a<0V1PH09bLW82O7al0WQhN09[Jf/2Fe]K0U9BDP9HF5P2IfMW0WEe
M@9jNWX2KVi^00=GEeLn?Shn?Sh00Rh^;P8/;2`2=CDe0Slo?`94A4@2?Shn0S8b<P8W9bL29BDU0Rd]
;@8o?cl2EeMG0WAdM0:EUID2]KFe0]7Ad@;Shn<2inOW0^7Qh@;EeMD2c<c<0/[:bP;>c/h2dm?C0]OG
e`;Hf=P2e]KF0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`005Oooo`800003oooo>=CDe0;Cdm<4
d];B0]CDe0;Fe]H4fMWI0]CDe0;>c/h2b<S80/[:bP;EeMD2hn?S0^c/k0;UiND2b<S80YjNWP9aLG42
Bd];0SDe=@8];Bd2<30`0S/k>`9<C4`2H61P0W9bLP9lO7`2N7Qh0V][J`9KFe/2DU9B0UQHF09WIfL2
MGEe0WYjNP9^KVh00eMGEchn?Shn?P02;Rh^0R`/;08e=CD2?clo0TA4A08n?Sh2<S8b0RLW9`8U9BD2
;Bd]0Slo?`9GEeL2M7Ad0YFEU@:e]KD2dM7A0^?Sh`;WinL2hN7Q0]GEe@;<c<`2b/[:0/k>cP;Cdm<2
emOG0]SHf0;Fe]H2e=CD1=;BdP;Cdm<ne=CD1?ooo`800001oooo000Eoooo00<0003oooooool00_oo
ocSDe=@2dm?C1=;BdP;Cdm<2e]KF1=WIf@;EeMD2c/k>0/W9b@;:b/X2eMGE0^?Sh`;[jn/2hN7Q0/30
`0:?Shl2GUiN0SLg=`8V9RH29bLW0S<c<`96ATH2G5aL0W1`L0:2PX82Rh^;0XN7Q`9iNGT2JFUY0UiN
GP9OGel2IfMW0W5aL@9cLg<2JVYZ00=HF5Q6ATI6ATH00SPh>08d=3@2=cLg0S/k>`8n?Sh2>c/k0SLg
=`8c<c<2<C4a0S8b<P8h>3P2@d=30UYJFP9mOGd2YZJV0/g=c@;WinL2k^k^0^GUi@;Fe]H2b/[:0/O7
a`;<c<`2dm?C1=WIf@;GemL2e=CD1=;BdP;Cdm<ne=CD1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooocSDe=@2dm?C1=;BdP;Cdm<2e]KF1=WIf@;EeMD2c/k>0/W9b@;:b/X2eMGE0^?Sh`;[jn/2
hN7Q0/30`0:?Shl2GUiN0SLg=`8V9RH29bLW0S<c<`96ATH2G5aL0W1`L0:2PX82Rh^;0XN7Q`9iNGT2
JFUY0UiNGP9OGel2IfMW0W5aL@9cLg<2JVYZ00=HF5Q6ATI6ATH00SPh>08d=3@2=cLg0S/k>`8n?Sh2
>c/k0SLg=`8c<c<2<C4a0S8b<P8h>3P2@d=30UYJFP9mOGd2YZJV0/g=c@;WinL2k^k^0^GUi@;Fe]H2
b/[:0/O7a`;<c<`2dm?C1=WIf@;GemL2e=CD1=;BdP;Cdm<ne=CD1Oooo`400001oooo000Eoooo00<0
003oooooool00_ooocSDe=@8dm?C0]GEe@CGemL2eMGE0]3@d0;<c<`2cLg=0]KFeP;Ph>02i>CT0]OG
e`:c/k<2PX:20U1@D08[:b/26a/K0QlO7`8^;Rh2ADE50UiNGP9fMWH2RXZ:19NGU`:=SHd2OWin0W5a
L@9YJFT2IfMW0VIVIP9SHf<2G5aL00=CDe=:BTY:BTX00T=3@`8o?cl2?Shn0Sdm?@8j>SX2=SHf0S8b
<P8_;bl2;2`/0RXZ:P8/;2`2=3@d0TY:BP9_Kfl2WIfM0/W9b@;Vi^H2k^k^0^KViP;Fe]H2bLW90/O7
a`;<c<`2dm?C0]WIf@;Jf]X2emOG0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`005Oooo`030000
oooooooo00;ooolhe=CD2=?Cd`;EeMD4emOG0]GEe@;@d=02c<c<0/g=c@;Fe]H2h>3P0^CTi0;GemL2
/k>c0X:2PP9@D502:b/[0Q/K6`8O7al2;Rh^0TE5A@9NGUh2MWIf0XZ:RPBGUiL2SHf=0WinOP9aLG42
JFUY0VMWI`9VIVH2Hf=S0UaLG003De=CBTY:BTY:0093@d<2?clo0Shn?P8m?Cd2>SXj0SHf=P8b<S82
;bl_0R`/;08Z:RX2;2`/0S@d=09:BTX2Kfm_0YfMW@;9bLT2i^KV0^k^kP;Vi^H2e]KF0/W9b@;7alL2
c<c<0]?Cd`;IfMT2f][J0]OGe`;De=@4d];B0]?CdckDe=@5oooo0@00007oool001Goool00`000?oo
ooooo`02oooo>]CDe0KCdm<2e=CD1=GEe@;De=@2d];B0]3@d0;AdM42e]KF0]_Kf`;IfMT2b<S80ZBT
Y09fMWH2AdM70RDU9@8E5AD25aLG0RHV9P8m?Cd2F5QH0WAdM0:=SHd2WinO0ZFUY@:QXJ42UYJF0XN7
Q`9fMWH2IVIV0UQHF09>CTh3BTY:0T];B`9=CDd2C4a<0TQ8B092@T82>SXj0S4a<@8X:2P28B4Q0Q`L
708K6a/27alO0RXZ:P93@d<2JVYZ0YRHV0;4a<@2h^;R0^_[j`;Ti>@2e]KF0/_;b`;8b<P2cLg=0]?C
d`;Hf=P2fMWI0]KFeP;De=@4d];B0]?CdckDe=@5oooo0@00007oool001Goool00`000?ooooooo`02
oooo>]CDe0KCdm<2e=CD1=GEe@;De=@2d];B0]3@d0;AdM42e]KF0]_Kf`;IfMT2b<S80ZBTY09fMWH2
AdM70RDU9@8E5AD25aLG0RHV9P8m?Cd2F5QH0WAdM0:=SHd2WinO0ZFUY@:QXJ42UYJF0XN7Q`9fMWH2
IVIV0UQHF09>CTh3BTY:0T];B`9=CDd2C4a<0TQ8B092@T82>SXj0S4a<@8X:2P28B4Q0Q`L708K6a/2
7alO0RXZ:P93@d<2JVYZ0YRHV0;4a<@2h^;R0^_[j`;Ti>@2e]KF0/_;b`;8b<P2cLg=0]?Cd`;Hf=P2
fMWI0]KFeP;De=@4d];B0]?CdckDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo?]CD
e0SCdm<2e=CD1=GEe@;Fe]H2emOG0]GEe@;=cLd2^KVi0YJFUP9[Jf/2@D510R8R8P8C4a<25ADE0R<S
8`8i>CT2E5AD0W5aL@:<S8`2XJ6Q0Zj^[P:b/[82[:b/0YbLW0:4Q8@2IfMW0Ta<C08k>c/2>3Ph00=1
@D5=CDe=CDd00UMGE`9IFET2E5AD0TY:BP8m?Cd2;bl_0R4Q8@8F5QH23`l?0Ph>3P8E5AD29BDU0T51
@@9XJ6P2UIFE0[no_`;MgMd2inOW0^;RhP;Fe]H2c<c<0/[:bP;>c/h2e=CD1=SHf0;Fe]H2e=CD1=;B
dP;Cdm<ne=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooockDe=@8dm?C0]CDe0CEeMD2
e]KF0]OGe`;EeMD2cLg=0[Vi^@:FUYH2Jf][0T51@@8R8R824a<C0QDE5@8S8b<2>CTi0UADE09aLG42
S8b<0Z6QX@:^[Zh2/[:b0Zb/[0:LW9`2Q8B40VMWI`9<C4`2>c/k0SPh>003@D51CDe=CDe=009GEeL2
FEUI0UADE09:BTX2?Cdm0Rl_;`8Q8B425QHF0Pl?3`8>3Ph25ADE0RDU9@91@D42J6QX0YFEU@:o_kl2
gMgM0^OWi`;Rh^82e]KF0/c<c0;:b/X2c/k>0]CDe0CHf=P2e]KF0]CDe0CBd]82dm?C?]CDe0Goool1
00000Oooo`005Oooo`030000oooooooo00;ooolbe=CD1=?Cd`;De=@6eMGE0]?Cd`;Bd]82dM7A0];B
dP;De=@2emOG0]SHf0;IfMT2emOG0];BdP;6a/H2[jn_0X^;R`9RHV82>c/k0R0P808G5aL27alO0S4a
<@98B4P2H61P0WIfMP::RXX2WIfM0Zj^[P:i^KT2^k^k0Zj^[P:@T902JVYZ0TI6AP8_;bl2;Bd]00<l
?3a@D51@D5000V5QH@9VIVH2HF5Q0UIFEP97AdL2=SHf0RLW9`8J6QX24A4A0Pl?3`8E5AD292@T0Slo
?`9VIVH2Ti>C0[fm_@;Jf]X2iNGU0^7Qh@;Fe]H2cLg=0/_;b`;>c/h2e=CD1=SHf0;Fe]H2e=CD1=;B
dP;Cdm<ne=CD1Oooo`400001oooo000Eoooo0P0000?ooolbe=CD1=?Cd`;De=@6eMGE0]?Cd`;Bd]82
dM7A0];BdP;De=@2emOG0]SHf0;IfMT2emOG0];BdP;6a/H2[jn_0X^;R`9RHV82>c/k0R0P808G5aL2
7alO0S4a<@98B4P2H61P0WIfMP::RXX2WIfM0Zj^[P:i^KT2^k^k0Zj^[P:@T902JVYZ0TI6AP8_;bl2
;Bd]00<l?3a@D51@D5000V5QH@9VIVH2HF5Q0UIFEP97AdL2=SHf0RLW9`8J6QX24A4A0Pl?3`8E5AD2
92@T0Slo?`9VIVH2Ti>C0[fm_@;Jf]X2iNGU0^7Qh@;Fe]H2cLg=0/_;b`;>c/h2e=CD1=SHf0;Fe]H2
e=CD1=;BdP;Cdm<ne=CD1?ooo`800001oooo000Eoooo00<0003oooooool00_oooc;De=@4dm?C0]CD
e0KEeMD2dm?C0];BdP;AdM42d];B0]CDe0;GemL2f=SH0]WIf@;GemL2d];B0/G5a@:][Jd2QhN70U]K
F`8d=3@27QhN0R0P808g=cL2EUIF0W9bLP:2PX82QhN70XZ:RP:ATI42XJ6Q0[>c/`:n_[h2]kNg0Y^K
V`9aLG42B4Q80Rh^;P8];Bd00clo?eIFEUIFEP02JVYZ0W9bLP9^KVh2I6AT0UMGE`9:BTX2?Shn0S<c
<`8Y:BT28B4Q0QlO7`8W9bL2?3`l0V9RHP:@T902_;bl0]cLg0;WinL2h^;R0]KFeP;<c<`2b/[:0/k>
cP;De=@4f=SH0]KFeP;De=@4d];B0]?CdckDe=@5oooo0@00007oool001Goool00`000?ooooooo`02
oooo<]CDe0CCdm<2e=CD1]GEe@;Cdm<2d];B0]7Ad@;Bd]82e=CD0]OGe`;Hf=P2fMWI0]OGe`;Bd]82
aLG50Zf][@:7QhL2Fe]K0S@d=08N7Qh2820P0SLg=`9FEUH2LW9b0X:2PP:7QhL2RXZ:0Y6AT@:QXJ42
/k>c0[jn_P:g]kL2Vi^K0W5aL@98B4P2;Rh^0Rd];@03?cloEUIFEUIF009ZJVX2LW9b0Vi^KP9TI6@2
EeMG0TY:BP8n?Sh2<c<c0RTY:@8Q8B427alO0RLW9`8l?3`2HV9R0Y2@T0:l_;`2g=cL0^OWi`;Rh^82
e]KF0/c<c0;:b/X2c/k>0]CDe0CHf=P2e]KF0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`005Ooo
o`030000oooooooo00;ooolbe=CD1=?Cd`CDe=@4eMGE0]?Cd`CBd]82dm?C0]GEe@;Fe]H6emOG0]CD
e0;:b/X2/[:b0XR8R09GEeL2;Bd]0Q/K6`8/;2`2EUIF0XJ6QP:XZ:P2[Zj^0YnOW`::RXX2P8200XZ:
RP:QXJ42]KFe0[Rh^0:QXJ42NGUi0U9BDP8h>3P2=SHf00=7AdMMGEeMGEd00W5aL@9iNGT2N7Qh0W9b
LP9[Jf/2IFEU0UmOG`9HF5P2CDe=0Slo?`8a<C42;2`/0STi>@9LG5`2S8b<0[fm_@;Ph>02k>c/0^GU
i@;GemL2b/[:0/S8b0;=cLd2e=CD1=WIf@;Fe]H2e=CD1=;BdP;Cdm<ne=CD1Oooo`400001oooo000E
oooo00<0003oooooool00_oooc;De=@4dm?C1=CDe0CEeMD2dm?C1=;BdP;Cdm<2eMGE0]KFePKGemL2
e=CD0/[:bP:b/[82R8R80UMGE`8];Bd26a/K0R`/;09FEUH2QXJ60ZRXZ0:^[Zh2WinO0XZ:RP:0P802
RXZ:0Z6QX@:e]KD2^;Rh0Z6QX@9iNGT2DU9B0SPh>08f=SH00dM7AeeMGEeMG@02LG5a0WUiN@9hN7P2
LW9b0V][J`9UIFD2GemO0UQHF09=CDd2?clo0S4a<@8/;2`2>CTi0UaLG0:<S8`2_Kfm0^3Ph0;/k>`2
iNGU0]OGe`;:b/X2b<S80/g=c@;De=@4fMWI0]KFeP;De=@4d];B0]?CdckDe=@5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo>]CDe0;EeMD2e=CD2=?Cd`SEeMD4e]KF0/o?c`:g]kL2Rh^;0UEE
E@8X:2P26ATI0SLg=`9bLW82/K6a0]SHf0;De=@2/[:b0XR8R09]KFd2LG5a0XZ:RP:UYJD2/K6a0Z:R
XP:2PX82H61P0TU9B@96ATH00e5ADF=SHf=SH`02LW9b0W]kN`9lO7`4Ng]k17alO09jNWX2Kfm_0UYJ
FP92@T82<c<c0SPh>09IFET2Rh^;0/30`0;Vi^H2lO7a0^SXj0;GemL2b<S80/G5a@;<c<`2e=CD0]WI
f@;Jf]X2emOG0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;ooolj
e=CD0]GEe@;De=@8dm?C2=GEe@CFe]H2clo?0[Ng]`:;Rh/2EEEE0RPX:08I6AT2=cLg0W9bLP:a/K42
f=SH0]CDe0:b/[82R8R80Ve]K@9aLG42RXZ:0ZFUY@:a/K42XZ:R0X:2PP9PH602BDU90TI6AP03DE5A
Hf=SHf=S009bLW82Ng]k0WalO0AkNg/4O7al0WYjNP9_Kfl2FUYJ0T92@P8c<c<2>3Ph0UUIF@:;Rh/2
`<300^KViP;alO42j>SX0]OGe`;8b<P2aLG50/c<c0;De=@2fMWI0][JfP;GemL2e=CD1=;BdP;Cdm<n
e=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooockDe=@6dm?C0]CDe0CEeMD4e=CD0]KF
eP;GemL2dM7A0[Zj^P:<S8`2E5AD0RDU9@8H61P2?3`l0WmoO`;4a<@2kNg]0^?Sh`:e]KD2Ogmo15YJ
FP9dM7@2TY:B0ZFUY@:PX:02RXZ:0W9bLP9OGel2FEUI00=MGEeUIFEUIFD00Ve]K@9cLg<2MgMg0WYj
NP9nOWh2Ph>318R8R09nOWh2IfMW0Te=C@8i>CT2>c/k0U]KF`:>SXh2a<C40^[ZjP;emOD2j^[Z0]KF
eP;7alL2a<C40/_;b`;Cdm<4f][J0]OGe`;De=@4d];B0]?CdckDe=@5oooo0@00007oool000[oool0
0`000?ooooooo`02oooo00<0003oooooool00oooo`030000oooooooo00;ooolne=CD1]?Cd`;De=@4
eMGE1=CDe0;Fe]H2emOG0]7Ad@:j^[X2S8b<0UADE08U9BD261PH0S`l?09oOgl2a<C40^g]k@;Shn<2
]KFe0WmoO`AJFUX2M7Ad0Y:BTP:UYJD2X:2P0XZ:RP9bLW82GemO0UUIF@03GEeMIFEUIFEU009]KFd2
Lg=c0WMgM`9jNWX2OWin0X>3P`B8R8P2OWin0VMWI`9=CDd2>CTi0S/k>`9KFe/2SXj>0/C4a0;Zj^X2
mOGe0^[ZjP;Fe]H2alO70/C4a0;;bl/2dm?C1=[JfP;GemL2e=CD1=;BdP;Cdm<ne=CD1Oooo`400001
oooo0009oooo00<0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`02oooo<]CD
e0;Cdm<6e=CD0]GEe@;De=@2dm?C1=;BdP;Cdm<2eMGE0]KFePCEeMD4e]KF0/k>cP:f][H2RHV90U=C
D`8V9RH26ATI0STi>@9fMWH2]KFe0]WIf@;?cll2XZ:R0Vi^KP9:BTX2BDU90V5QH@9oOgl2UYJF0Y^K
V`:BTY82QHF50WQhN09^KVh00fQXJ6ATI6ATI002HV9R0V=SH`9VIVH2JVYZ0W1`L09eMGD4N7Qh0W5a
L@9PH602C4a<0Slo?`95ADD2IFEU0YJFUP;8b<P2jn_[0_Cdm0;YjNT2e]KF0/O7a`;4a<@2bl_;0]?C
d`CJf]X2emOG0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`002Oooo`030000oooo000000;oool0
0`000?ooo`000004oooo0P0000?ooolbe=CD0]?Cd`KDe=@2eMGE0]CDe0;Cdm<4d];B0]?Cd`;EeMD2
e]KF1=GEe@CFe]H2c/k>0[Jf]P:9RHT2De=C0RHV9P8I6AT2>CTi0WIfMP:e]KD2fMWI0/o?c`:RXZ82
KVi^0TY:BP99BDT2HF5Q0WmoO`:FUYH2Vi^K0Y:BTP:5QHD2N7Qh0Vi^KP03J6QXI6ATI6AT009RHV82
Hf=S0VIVIP9ZJVX2L71`0WEeM@AhN7P2LG5a0V1PH09<C4`2?clo0TE5A@9UIFD2UYJF0/S8b0;[jn/2
m?Cd0^WYj@;Fe]H2alO70/C4a0;;bl/2dm?C1=[JfP;GemL2e=CD1=;BdP;Cdm<ne=CD1?ooo`800001
oooo000:oooo00L0003oooooooooool0003oool000001?ooo`030000oooooooo00;ooolbe=CD1=?C
d`CDe=@4eMGE0]?Cd`;Bd]82dM7A0];BdP;EeMD2emOG1=SHf0;GemL2d];B0/O7a`:^[Zh2QHF50UAD
E08[:b/271`L0S4a<@9OGel2Shn?0Z^[Z`:RXZ82PH610UYJFPA0@402E5AD0Vi^KP:6QXH2U9BD0YVI
V@:HV9P2TI6A0X>3P`03LW9bGemOGemO009BDU82C4a<0Ti>CP9BDU82EUIF0UQHF09GEeL2EEEE0U5A
D@99BDT4@d=30U9BDP9cLg<2X:2P0/c<c0;Zj^X2lO7a0^OWi`;Fe]H2b<S80/G5a@;;bl/2dm?C0]WI
f@;Jf]X2emOG0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`002Oooo`030000oooo000000;oool0
0`000?ooo`000004oooo00<0003oooooool00_oooc;De=@4dm?C1=CDe0CEeMD2dm?C0];BdP;AdM42
d];B0]GEe@;GemL4f=SH0]OGe`;Bd]82alO70Zj^[P:5QHD2E5AD0R/[:`8L71`2<C4a0UmOG`:?Shl2
Zj^[0Z:RXP:1PH42FUYJ1410@09DE5@2KVi^0XJ6QP:DU9@2VIVI0YRHV0:ATI42Ph>300=bLW9OGemO
Gel00U9BDP9<C4`2CTi>0U9BDP9FEUH2F5QH0UMGE`9EEED2DE5A0TU9B@A3@d<2DU9B0W=cL`:PX:02
c<c<0^[ZjP;alO42inOW0]KFeP;8b<P2aLG50/_;b`;Cdm<2fMWI0][JfP;GemL2e=CD1=;BdP;Cdm<n
e=CD1Oooo`400001oooo000:oooo00<0003oooooool00_ooo`030000oooooooo00?oool00`000?oo
ooooo`02oooo<]CDe0CCdm<2e=CD0]GEe@;Fe]H2eMGE0]?Cd`;AdM42d=3@0]7Ad@;De=@2fMWI1=_K
f`;GemL2c/k>0[no_`:UYJD2PH610UUIF@8f=SH292@T0Rd];@96ATH2Hf=S0WEeM@9aLG42GUiN0TU9
B@8k>c/2?Cdm0T];B`9NGUh2M7Ad0XV9R@:KVi/2YjNW0ZJVYP:EUID00gQhN5UIFEUIF@02@41013De
=@8j>SX2?Cdm0SXj>P8d=3@2<30`0Rh^;P8a<C42>CTi0TM7A`9OGel2PX:20ZZZZP;?cll2j>SX0^g]
k@;Ti>@2e]KF0/[:bP;7alL2c<c<0]?Cd`;Hf=P2fMWI0]OGe`;De=@4d];B0]?CdckDe=@5oooo0@00
007oool001Goool00`000?ooooooo`02oooo<]CDe0CCdm<2e=CD0]GEe@;Fe]H2eMGE0]?Cd`;AdM42
d=3@0]7Ad@;De=@2fMWI1=_Kf`;GemL2c/k>0[no_`:UYJD2PH610UUIF@8f=SH292@T0Rd];@96ATH2
Hf=S0WEeM@9aLG42GUiN0TU9B@8k>c/2?Cdm0T];B`9NGUh2M7Ad0XV9R@:KVi/2YjNW0ZJVYP:EUID0
0gQhN5UIFEUIF@02@41013De=@8j>SX2?Cdm0SXj>P8d=3@2<30`0Rh^;P8a<C42>CTi0TM7A`9OGel2
PX:20ZZZZP;?cll2j>SX0^g]k@;Ti>@2e]KF0/[:bP;7alL2c<c<0]?Cd`;Hf=P2fMWI0]OGe`;De=@4
d];B0]?CdckDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo<=CDe0KCdm<2e=CD0]GE
e@CFe]H2dm?C0]3@d0;>c/h2clo?0]CDe0;Kfm/2gmoO0]kNgP;GemL2bLW90[Ng]`:NWYh2PH610V=S
H`99BDT2>3Ph0S@d=08j>SX2@d=314];B`96ATH2@T920Slo?`90@402A4A40Te=C@9NGUh2N7Qh0YFE
U@:[Zj/2/;2`0YbLW003MgMgCTi>CTi>008];Bd2820P0R8R8P8Y:BT2;Bd]0RPX:08O7al261PH0QLG
5`8Q8B42<c<c0T];B`9YJFT2SHf=0[6a/@;AdM42i^KV0^[ZjP;Shn<2e]KF0/_;b`;8b<P2cLg=0]?C
d`;Hf=P2fMWI0]OGe`;De=@4d];B0]?CdckDe=@5oooo0@00007oool001Goool00`000?ooooooo`02
oooo<=CDe0KCdm<2e=CD0]GEe@CFe]H2dm?C0]3@d0;>c/h2clo?0]CDe0;Kfm/2gmoO0]kNgP;GemL2
bLW90[Ng]`:NWYh2PH610V=SH`99BDT2>3Ph0S@d=08j>SX2@d=314];B`96ATH2@T920Slo?`90@402
A4A40Te=C@9NGUh2N7Qh0YFEU@:[Zj/2/;2`0YbLW003MgMgCTi>CTi>008];Bd2820P0R8R8P8Y:BT2
;Bd]0RPX:08O7al261PH0QLG5`8Q8B42<c<c0T];B`9YJFT2SHf=0[6a/@;AdM42i^KV0^[ZjP;Shn<2
e]KF0/_;b`;8b<P2cLg=0]?Cd`;Hf=P2fMWI0]OGe`;De=@4d];B0]?CdckDe=@5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo<=CDe0KCdm<2e=CD0]GEe@;GemL2e]KF0]CDe0;@d=02cLg=0/k>
cP;De=@2g=cL0^;RhP;QhN42emOG0/K6aP:`/;02VYZJ0XJ6QP9eMGD2IVIV0UQHF09<C4`2@T920S/k
>`8h>3P2>c/k0T51@@98B4P2Bd];0TM7A`90@402?Cdm0TI6AP9PH602Ph>30Z6QX@:[Zj/2UYJF00=/
K6`n?Shn?Sh00Q`L708A4A426ATI0RLW9`8`<302;2`/0R4Q8@8H61P25aLG0R4Q8@8e=CD2CTi>0Ve]
K@:@T902/k>c0]7Ad@;UiND2jNWY0^;RhP;EeMD2bl_;0/W9b@;=cLd2dm?C0]SHf0;IfMT2e]KF0]CD
e0CBd]82dm?C?]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;oool`e=CD1]?Cd`;De=@2
eMGE0]OGe`;Fe]H2e=CD0]3@d0;=cLd2c/k>0]CDe0;Lg=`2h^;R0^7Qh@;GemL2a/K60[2`/0:JVYX2
QXJ60WEeM@9VIVH2F5QH0Ta<C092@T82>c/k0SPh>08k>c/2@D510TQ8B09;Bd/2AdM70T10@08m?Cd2
ATI60V1PH0:3Ph<2XJ6Q0Z^[Z`:FUYH00fa/K3hn?Shn?P0271`L0Q4A4@8I6AT29bLW0S0`<08/;2`2
8B4Q0QPH608G5aL28B4Q0SDe=@9>CTh2KFe]0Y2@T0:c/k<2dM7A0^GUi@;YjNT2h^;R0]GEe@;;bl/2
bLW90/g=c@;Cdm<2f=SH0]WIf@;Fe]H2e=CD1=;BdP;Cdm<ne=CD1Oooo`400001oooo000Eoooo00<0
003oooooool00_oooc3De=@6dm?C0]CDe0;EeMD2emOG0]KFeP;De=@2d=3@1<g=c@;De=@2gMgM0^?S
h`;Rh^82emOG0/?3``:/[:`2VYZJ0Y2@T0:;Rh/2R8R80X20P09`L702Fe]K0TM7A`8m?Cd2@4100Ta<
C09IFET2GEeM0U=CD`91@D44<C4a0TQ8B09]KFd2Shn?0YbLW0:8R8P00eeMGBh^;Rh^;P023`l?0P/;
2`8K6a/2<C4a0T10@091@D42=cLg0Rd];@8Y:BT2;bl_0Shn?P9CDe<2Kfm_0Xn?S`:b/[82d=3@0^CT
i0;YjNT2h^;R0]GEe@;;bl/2bLW90/g=c@;Cdm<2f=SH0]WIf@;Fe]H2e=CD1=;BdP;Cdm<ne=CD1Ooo
o`400001oooo000Eoooo0P0000?oool`e=CD1]?Cd`;De=@2eMGE0]OGe`;Fe]H2e=CD0]3@d0C=cLd2
e=CD0]gMg@;Shn<2h^;R0]OGe`;3`l<2[:b/0YZJVP:@T902Rh^;0XR8R0:0P802L71`0U]KF`97AdL2
?Cdm0T10@09<C4`2FEUI0UeMG@9CDe<2@D51134a<@98B4P2KFe]0Xn?S`:LW9`2R8R800=MGEd^;Rh^
;Rh00Pl?3`8;2`/26a/K0S4a<@90@402@D510SLg=`8];Bd2:BTY0Rl_;`8n?Sh2De=C0Vm_K`:?Shl2
/[:b0]3@d0;Ti>@2jNWY0^;RhP;EeMD2bl_;0/W9b@;=cLd2dm?C0]SHf0;IfMT2e]KF0]CDe0CBd]82
dm?C?]CDe0Coool200000Oooo`005Oooo`030000oooooooo00;oool`e=CD1]?Cd`;De=@2eMGE1=OG
e`;De=@2d=3@0/c<c0;=cLd2dm?C0]gMg@;Ti>@2hn?S0]KFeP;2`/82[:b/0YfMW@:KVi/2XJ6Q0ZNW
Y`:UYJD2Ti>C0WQhN09NGUh2Cdm?0U5AD@9PH602L71`0WAdM09TI6@2B4Q80Rh^;P8V9RH2>CTi0UeM
G@:0P802SHf=0W]kN`03DU9B9RHV9RHV008;2`/23Ph>0RDU9@92@T82EeMG0UaLG09CDe<2BDU90T92
@P93@d<2C4a<0U]KF`9aLG42SXj>0[2`/0;?cll2hn?S0^WYj@;Rh^82eMGE0/c<c0;9bLT2cLg=0]?C
d`CHf=P2e]KF0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;oool`
e=CD1]?Cd`;De=@2eMGE1=OGe`;De=@2d=3@0/c<c0;=cLd2dm?C0]gMg@;Ti>@2hn?S0]KFeP;2`/82
[:b/0YfMW@:KVi/2XJ6Q0ZNWY`:UYJD2Ti>C0WQhN09NGUh2Cdm?0U5AD@9PH602L71`0WAdM09TI6@2
B4Q80Rh^;P8V9RH2>CTi0UeMG@:0P802SHf=0W]kN`03DU9B9RHV9RHV008;2`/23Ph>0RDU9@92@T82
EeMG0UaLG09CDe<2BDU90T92@P93@d<2C4a<0U]KF`9aLG42SXj>0[2`/0;?cll2hn?S0^WYj@;Rh^82
eMGE0/c<c0;9bLT2cLg=0]?Cd`CHf=P2e]KF0]CDe0CBd]82dm?C?]CDe0Goool100000Oooo`005Ooo
o`030000oooooooo00;oool`e=CD1]?Cd`;De=@2eMGE1=KFeP;De=@2d=3@0/g=c@;>c/h2dm?C0]cL
g0;Rh^82hN7Q0]KFeP;3`l<2/;2`0ZFUY@:WYjL2/;2`0[Zj^P:k^k/2[:b/0YBDU09kNg/2K6a/0Vi^
KP9kNg/2RHV90XZ:RP9gMgL2F5QH0SXj>P8^;Rh2?Shn0UmOG`9oOgl2SHf=0WalO003EeMG<30`<30`
008I6AT27alO0STi>@9HF5P2Kfm_0WEeM@9^KVh2IFEU0UeMG@9JFUX2GUiN0VMWI`9hN7P2TY:B0[6a
/@;>c/h2h^;R0^OWi`;QhN42eMGE0/c<c0;:b/X2c/k>0]?Cd`;GemL2f=SH0]KFeP;De=@4d];B0]?C
dckDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo<=CDe0KCdm<2e=CD0]GEe@CFe]H2
e=CD0]3@d0;=cLd2c/k>0]?Cd`;Lg=`2h^;R0^7Qh@;Fe]H2`l?30[2`/0:UYJD2YjNW0[2`/0:j^[X2
^k^k0Zb/[0:DU9@2Ng]k0Va/K09^KVh2Ng]k0XV9R@::RXX2MgMg0UQHF08j>SX2;Rh^0Shn?P9OGel2
Ogmo0Xf=S@9lO7`00eMGEc0`<30`<0026ATI0QlO7`8i>CT2F5QH0Vm_K`9eMGD2KVi^0VEUI@9MGEd2
FUYJ0UiNGP9WIfL2N7Qh0Y:BTP:a/K42c/k>0^;RhP;WinL2hN7Q0]GEe@;<c<`2b/[:0/k>cP;Cdm<2
emOG0]SHf0;Fe]H2e=CD1=;BdP;Cdm<ne=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
oc3De=@6dm?C0]CDe0;EeMD4e]KF0]CDe0;AdM44clo?0]CDe0;IfMT2g]kN0]gMg@;EeMD2b<S80[^k
^`:b/[82/K6a0[Jf]P:k^k/2_;bl0[Bd]0:VYZH2V9RH0Y2@T0:BTY82Vi^K0Z:RXP:OWil2S8b<0W5a
L@9HF5P2Cdm?0UeMG@9jNWX2UiNG0Z>SX`:DU9@00g=cLe1@D51@D002>c/k0Slo?`9DE5@2KVi^0X:2
PP:7QhL2PH610WUiN@9bLW82Kfm_0W9bLP9jNWX2R8R80YfMW@:f][H2clo?0^3Ph0;Ti>@2gmoO0]GE
e@;=cLd2c<c<0/o?c`;Cdm<4emOG0]KFeP;De=@6dm?C?]CDe0Goool100000Oooo`005Oooo`030000
oooooooo00;oool`e=CD1]?Cd`;De=@2eMGE1=KFeP;De=@2dM7A1<o?c`;De=@2fMWI0]kNgP;MgMd2
eMGE0/S8b0:k^k/2/[:b0[6a/@:f][H2^k^k0[bl_0:d];@2YZJV0YRHV0:@T902TY:B0Y^KV`:RXZ82
WinO0Xb<S09aLG42F5QH0Tm?C`9MGEd2NWYj0YNGU`:SXj<2U9BD00=cLg=@D51@D5000S/k>`8o?cl2
E5AD0Vi^KP:2PX82QhN70X61P@9iNGT2LW9b0Vm_K`9bLW82NWYj0XR8R0:MWId2][Jf0/o?c`;Ph>02
i>CT0]oOg`;EeMD2cLg=0/c<c0;?cll2dm?C1=OGe`;Fe]H2e=CD1]?CdckDe=@5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo<]CDe0CCdm<4e=CD1=GEe@;De=@6d];B0]CDe0;Fe]H4f=SH0]GE
e@;?cll2b<S80/71`@:j^[X2];Bd0[2`/0B^[Zh2/;2`0[>c/`:f][H2^;Rh0[Vi^@:g]kL2/;2`0Z6Q
X@:@T902PH610WmoO`:>SXh2YZJV0[fm_@;6a/H2^KVi00>LW9amOGemOGd00VUYJ@9WIfL2Lg=c0XB4
Q0:@T902TY:B0Xb<S0:6QXH4PX:20XR8R0:@T902W9bL0Zf][@;0`<02dM7A0]gMg@;Ph>02g=cL0]GE
e@;?cll2c/k>0]3@d0;Cdm<4e]KF0]GEe@;De=@6dm?C?]CDe0Goool100000Oooo`005Oooo`030000
oooooooo00;ooolbe=CD1=?Cd`CDe=@4eMGE0]CDe0KBd]82e=CD0]KFePCHf=P2eMGE0/o?c`;8b<P2
`L710[Zj^P:d];@2/;2`1:j^[P:`/;02/k>c0[Jf]P:h^;P2^KVi0[Ng]`:`/;02XJ6Q0Y2@T0:1PH42
Ogmo0Xj>SP:VYZH2_Kfm0/K6aP:i^KT00ibLW7emOGemO@02JFUY0VMWI`9cLg<2Q8B40Y2@T0:BTY82
S8b<0XJ6QPB2PX82R8R80Y2@T0:LW9`2[Jf]0/30`0;AdM42gMgM0^3Ph0;Lg=`2eMGE0/o?c`;>c/h2
d=3@0]?Cd`CFe]H2eMGE0]CDe0KCdm<ne=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_oo
od[De=@4dm?C0]GEe@CFe]H2d=3@0/;2`P:`/;02XJ6Q0YbLW0:TY:@2][Jf0/W9b@;Fe]H2f=SH0]7A
d@;7alL2_;bl0[>c/`:^[Zh2[:b/0[6a/@:o_kl2d];B0^;RhP;Xj>P2g=cL00?5aLF[Zj^[Zj/00YNG
U`:?Shl2T92@0YJFUP:JVYX2V9RH0YBDU0:ATI42T92@0YFEU@:MWId2YZJV0[:b/P:n_[h2b/[:0]CD
e0;Jf]X2fm_K0]WIf@;De=@2dM7A0]3@d0;AdM42e=CD1]GEe@;De=@4dm?C@=CDe0Goool100000Ooo
o`005Oooo`800003ooooB]CDe0CCdm<2eMGE1=KFeP;@d=02`/;20[2`/0:QXJ42W9bL0ZBTY0:f][H2
bLW90]KFeP;Hf=P2dM7A0/O7a`:l_;`2/k>c0Zj^[P:/[:`2/K6a0[no_`;Bd]82h^;R0^SXj0;Lg=`0
0lG5aJ^[Zj^[Z`02UiNG0Xn?S`:@T902UYJF0YZJVP:HV9P2U9BD0Y6AT@:@T902UIFE0YfMW@:VYZH2
/[:b0[jn_P;:b/X2e=CD0][JfP;Kfm/2fMWI0]CDe0;AdM42d=3@0]7Ad@;De=@6eMGE0]CDe0CCdm=0
e=CD1?ooo`800001oooo000Eoooo00<0003oooooool00_ooocSDe=@6dm?C0]CDe0;EeMD4e]KF0]CD
e0;Bd]84clo?0]CDe0;Kfm/2gmoO0][JfP;9bLT2/K6a0Y^KV`:CTi<2WYjN0[Jf]P;@d=02hn?S0^GU
i@;Lg=`2clo?0/C4a0;1`L42aLG50/_;b`;EeMD2hN7Q0^c/k0;emOD2m_Kf0^g]k@03gMgMbLW9bLW9
00:g]kL2ZZZZ0Z>SX`:PX:02WIfM0YZJVP:HV9P2VIVI0YfMW@:UYJD2[jn_0[Zj^P;4a<@2c<c<0];B
dP;EeMD4emOG0]KFeP;De=@4d];B0]?Cd`CDe=@2eMGEB=CDe0Goool100000Oooo`005Oooo`030000
oooooooo00;ooolhe=CD1]?Cd`;De=@2eMGE1=KFeP;De=@2d];B1<o?c`;De=@2fm_K0]oOg`;Jf]X2
bLW90[6a/@:KVi/2Ti>C0YjNWP:f][H2d=3@0^?Sh`;UiND2g=cL0/o?c`;4a<@2`L710/G5a@;;bl/2
eMGE0^7Qh@;/k>`2mOGe0_KfmP;]kNd00mgMgLW9bLW9b@02]kNg0ZZZZP:SXj<2X:2P0YfMW@:JVYX2
V9RH0YVIV@:MWId2YJFU0Zn_[`:j^[X2a<C40/c<c0;Bd]82eMGE1=OGe`;Fe]H2e=CD1=;BdP;Cdm<4
e=CD0]GEeDSDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo>=CDe0KCdm<2e=CD0]GE
e@CFe]H2e=CD0]7Ad@;>c/h2clo?0]CDe0;Lg=`2h>3P0]gMg@;>c/h2^KVi0ZBTY0:JVYX2X:2P0[:b
/P;6a/H2e=CD0]OGe`;Bd]82bl_;0/O7a`;9bLT2dM7A0][JfP;Rh^82inOW1>[ZjP;WinL2hN7Q00?G
emO<c<c<c<`00[no_`:b/[82YjNW0YjNWP:HV9P2UiNG0YVIV@:NWYh2YjNW0[6a/@:m_Kd2alO70/o?
c`;De=@4e]KF0]GEe@SDe=@4dm?CC]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;ooolh
e=CD1]?Cd`;De=@2eMGE1=KFeP;De=@2dM7A0/k>cP;?cll2e=CD0]cLg0;Ph>02gMgM0/k>cP:i^KT2
Y:BT0YZJVP:PX:02/[:b0/K6aP;De=@2emOG0];BdP;;bl/2alO70/W9b@;AdM42f][J0^;RhP;WinL4
j^[Z0^OWi`;QhN400mOGelc<c<c<c002_kno0[:b/P:WYjL2WYjN0YRHV0:GUiL2VIVI0YjNWP:WYjL2
/K6a0[fm_@;7alL2clo?0]CDe0CFe]H2eMGE2=CDe0CCdm=>e=CD1Oooo`400001oooo000Eoooo00<0
003oooooool00_oooc[De=@4dm?C0]CDe0KEeMD2e=CD0];BdP;@d=02dM7A0]CDe0;IfMT2g=cL0][J
fP;Bd]82a/K60[Vi^@:_[jl4Zj^[0Zn_[`:d];@2^;Rh0[bl_0;0`<02a/K60/g=c@;De=@2fMWI0]_K
f`;Hf=P2d];B0/_;b`;4a<@2`<3000>n_[jk^k^k^k/00[Fe]@:[Zj/2X:2P0YJFUP:@T902TY:B0YVI
V@:SXj<2[jn_0[^k^`;6a/H2clo?0]KFePCHf=P2e]KF0]?Cd`;Bd]82dm?CE]CDe0Goool100000Ooo
o`005Oooo`030000oooooooo00;ooolje=CD1=?Cd`;De=@6eMGE0]CDe0;Bd]82d=3@0]7Ad@;De=@2
fMWI0]cLg0;Jf]X2d];B0/K6aP:i^KT2[jn_1:^[Z`:_[jl2];Bd0[Rh^0:l_;`2`<300/K6aP;=cLd2
e=CD0]WIf@;Kfm/2f=SH0];BdP;;bl/2a<C40/30`003_[jn^k^k^k^k00:e]KD2Zj^[0Z2PX0:FUYH2
T92@0Y:BTP:IVIT2Xj>S0Zn_[`:k^k/2a/K60/o?c`;Fe]H4f=SH0]KFeP;Cdm<2d];B0]?CdeKDe=@5
oooo0@00007oool001Goool00`000?ooooooo`02oooo1=CDe0KCdm<ne=CD1]?Cd`;De=@8eMGE0]?C
d`;?cll2a/K60[Ng]`:VYZH2UiNG0Y6AT@:FUYH2Xj>S0[>c/`;2`/82cLg=0]7Ad@;Bd]82cLg=0/?3
``:f][H2Z:RX0YnOW`:MWId00j2PX:FUYJFUY@02YZJV0Z2PX0:FUYH2SHf=0XV9R@:>SXh2VIVI0ZNW
Y`:e]KD2`/;20/c<c0;Cdm<2f=SH0]WIf@;Hf=P2eMGE0];BdP;AdM42d];B0]CDe0KEeMD2e=CD1=?C
ddSDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo1=CDe0KCdm<ne=CD1]?Cd`;De=@8
eMGE0]?Cd`;?cll2a/K60[Ng]`:VYZH2UiNG0Y6AT@:FUYH2Xj>S0[>c/`;2`/82cLg=0]7Ad@;Bd]82
cLg=0/?3``:f][H2Z:RX0YnOW`:MWId00j2PX:FUYJFUY@02YZJV0Z2PX0:FUYH2SHf=0XV9R@:>SXh2
VIVI0ZNWY`:e]KD2`/;20/c<c0;Cdm<2f=SH0]WIf@;Hf=P2eMGE0];BdP;AdM42d];B0]CDe0KEeMD2
e=CD1=?CddSDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo@]CDe0KCdm<2e=CD1=GE
e@;De=@2d];B0]3@d0;AdM42e]KF0]cLg0;Ogml2emOG0/;2`P:VYZH2S8b<0WinOP:2PX82UIFE0Zb/
[0;0`<02bl_;0/k>cP;;bl/2`L710[>c/`:RXZ82TY:B18N7Q`03SXj>UiNGUiNG00:MWId2VYZJ0Y>C
T`:<S8`2RXZ:0Y6AT@:NWYh2[Jf]0[bl_0;7alL2clo?0]GEe@;Hf=P2fMWI0]OGe`;De=@4dM7A0];B
dP;De=@6eMGE0]CDe0CCdm=8e=CD1Oooo`400001oooo000Eoooo0P0000?ooom2e=CD1]?Cd`;De=@4
eMGE0]CDe0;Bd]82d=3@0]7Ad@;Fe]H2g=cL0]oOg`;GemL2`/;20ZJVYP:<S8`2OWin0X:2PP:EUID2
[:b/0/30`0;;bl/2c/k>0/_;b`;1`L42/k>c0Z:RXP:BTY84QhN700>>SXjGUiNGUiL00YfMW@:JVYX2
Ti>C0Xb<S0::RXX2TI6A0YjNWP:][Jd2_;bl0/O7a`;?cll2eMGE0]SHf0;IfMT2emOG0]CDe0CAdM42
d];B0]CDe0KEeMD2e=CD1=?CddSDe=@4oooo0P00007oool001Goool00`000?ooooooo`02oooo@=CD
e0SCdm<2eMGE1=KFeP;De=@2dM7A0/k>cP;@d=02e]KF0]kNgP;Shn<2g]kN0/_;b`:`/;02UYJF0XR8
R0::RXX2VYZJ0Zj^[P:o_kl2bLW90/c<c0;9bLT2`<300[>c/`:SXj<2Ti>C18Z:RP03TI6AVi^KVi^K
00BSXj<2WYjN0YZJVP:IVIT2X:2P0Z^[Z`:h^;P2a<C40/c<c0;Bd]82e]KF1=OGe`;Fe]H2e=CD1];B
dP;De=@4eMGE1=CDe0CCdm=8e=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_oood3De=@8
dm?C0]GEe@CFe]H2e=CD0]7Ad@;>c/h2d=3@0]KFeP;Ng]h2hn?S0]kNgP;;bl/2/;2`0YJFUP:8R8P2
RXZ:0YZJVP:^[Zh2_kno0/W9b@;<c<`2bLW90/30`0:c/k<2Xj>S0Y>CT`B:RXX00i6ATI^KVi^KV`04
Xj>S0YjNWP:JVYX2VIVI0Z2PX0:[Zj/2^;Rh0/C4a0;<c<`2d];B0]KFePCGemL2e]KF0]CDe0KBd]82
e=CD1=GEe@CDe=@4dm?CB=CDe0Goool100000Oooo`005Oooo`030000oooooooo00;oool@e=CD2]?C
d`[De=@4dm?C6=CDe0KCdm<4e=CD1=GEe@;De=@2d];B0]3@d0;AdM42eMGE0]_Kf`;Ng]h2fm_K0]3@
d0;0`<02/K6a1:NWY`:^[Zh2^;Rh0/71`@;8b<P4bl_;0/O7a`:n_[h2/[:b0ZNWY`:OWil2X:2P00>V
YZJ][Jf][Jd00[Bd]0:e]KD2/k>c0[6a/@:b/[82][Jf0[jn_P;6a/H2cLg=0];BdP;De=@2eMGE0]KF
ePCEeMD2e=CD1]?CdeKDe=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo4=CDe0[Cdm<:
e=CD1=?CdaSDe=@6dm?C1=CDe0CEeMD2e=CD0];BdP;@d=02dM7A0]GEe@;Kfm/2g]kN0]_Kf`;@d=02
`<300[6a/@BWYjL2[Zj^0[Rh^0;1`L42b<S81<_;b`;7alL2_[jn0[:b/P:WYjL2WinO0Z2PX003YZJV
[Jf][Jf]00:d];@2]KFe0[>c/`:a/K42/[:b0[Jf]P:n_[h2a/K60/g=c@;Bd]82e=CD0]GEe@;Fe]H4
eMGE0]CDe0KCdm=Fe=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`;De=@:eMGE1=CD
e0CCdm<4d];B1=?Cd`KDe=@8dm?C3=CDe0[Cdm<@e=CD1]?Cd`;De=@2eMGE0]KFeP;EeMD2e=CD0]7A
d@;>c/h2bl_;0/S8b0;6a/H2a<C40/G5a@;7alL2bl_;0/o?c`;@d=02c<c<0/K6aP:o_kl4^k^k00>n
_[k3`l?3`l<00/O7a`C9bLT2b/[:0/_;b`;=cLd2dM7A0]CDe0;EeMD2e]KF0]GEe@;De=@6dm?CG]CD
e0Goool100000Oooo`005Oooo`030000oooooooo00;oool2e=CD2]GEe@CDe=@4dm?C1=;BdPCCdm<6
e=CD2=?Cd`cDe=@:dm?C4=CDe0KCdm<2e=CD0]GEe@;Fe]H2eMGE0]CDe0;AdM42c/k>0/_;b`;8b<P2
a/K60/C4a0;5aLD2alO70/_;b`;?cll2d=3@0/c<c0;6a/H2_kno1;^k^`03_[jn`l?3`l?300;7alL4
bLW90/[:bP;;bl/2cLg=0]7Ad@;De=@2eMGE0]KFeP;EeMD2e=CD1]?CdekDe=@5oooo0@00007oool0
01Goool00`000?ooooooo`02oooo1=CDe0SEeMD4e=CD1=?Cd`;Bd]84dm?C2=CDe0SCdm<>e=CD2=?C
d`[De=@4dm?C1=CDe0CEeMD2dm?C0]7Ad@;?cll2dM7A0]KFeP;MgMd2i>CT0^KViP;QhN42f=SH0/o?
c`C9bLT2cLg=0];BdP;Fe]H2emOG0]GEe@;Cdm<4dM7A00?Bd];Cdm?Cdm<00]GEe@;GemL2f][J0]cL
g0;MgMd4g]kN0]cLg0;Kfm/2f=SH0]GEe@;Cdm<6d];B0]CDe0KEeMD2e=CD1]?CddkDe=@5oooo0@00
007oool000?oool500000_ooo`030000oooooooo00;oool00`000?ooooooo`03oooo00<0003ooooo
ool00_ooo`CDe=@8eMGE1=CDe0CCdm<2d];B1=?Cd`SDe=@8dm?C3]CDe0SCdm<:e=CD1=?Cd`CDe=@4
eMGE0]?Cd`;AdM42clo?0]7Ad@;Fe]H2gMgM0^CTi0;Vi^H2hN7Q0]SHf0;?cll4bLW90/g=c@;Bd]82
e]KF0]OGe`;EeMD2dm?C1=7Ad@03d];Bdm?Cdm?C00;EeMD2emOG0][JfP;Lg=`2gMgM1=kNgP;Lg=`2
fm_K0]SHf0;EeMD2dm?C1];BdP;De=@6eMGE0]CDe0KCdm=>e=CD1Oooo`400001oooo0005oooo00L0
003oooooooooool0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`02ooooB]CD
e0CCdm<2e=CD0]GEe@;Fe]H2eMGE0]?Cd`;@d=02c/k>0]3@d0;GemL2h>3P0^WYj@;/k>`2j>SX0]oO
g`;EeMD2cLg=0/c<c0;?cll2dm?C0]SHf0;Jf]X8fm_K1M[JfP;Lg=`2gmoO0^7Qh@;Shn<2h^;R0^7Q
h@;Ng]h2fm_K0]SHf0;EeMD2dm?C1=7Ad@;Bd]82e=CD1]GEe@;De=@6dm?CC]CDe0Goool100000Ooo
o`001Oooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004oooo0P0000?ooom:
e=CD1=?Cd`;De=@2eMGE0]KFeP;EeMD2dm?C0]3@d0;>c/h2d=3@0]OGe`;Ph>02jNWY0^c/k0;Xj>P2
gmoO0]GEe@;=cLd2c<c<0/o?c`;Cdm<2f=SH0][JfPSKfm/5f][J0]cLg0;Ogml2hN7Q0^?Sh`;Rh^82
hN7Q0]kNgP;Kfm/2f=SH0]GEe@;Cdm<4dM7A0];BdP;De=@6eMGE0]CDe0KCdm=>e=CD1?ooo`800001
oooo0005oooo00L0003oooooooooool0003oool000000_ooo`030000oooo000000Coool00`000?oo
ooooo`02oooo2=?Cd`CBd]82dm?C0]CDe0;EeMD6e]KF0]GEe@[De=@4eMGE8=CDe0KCdm<4e=CD1=GE
e@;Cdm<2d];B0]3@d0;Bd]82e]KF0]_Kf`;Ph>02hn?S0^7Qh@;Lg=`2e]KF0];BdP;@d=02dM7A0]?C
d`;Fe]H2emOG0]SHf0;IfMT4f][J00?IfMWHf=SHf=P00]SHf0;IfMT2fm_K0]gMg@;Ng]h2gMgM0]cL
g0;Jf]X2f=SH0]KFeP;De=@2dm?C1=;BdP;Cdm<4e=CD0]GEeECDe=@4dm?C1Oooo`400001oooo0005
oooo00L0003oooooooooool0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`02
oooo2=?Cd`CBd]82dm?C0]CDe0;EeMD6e]KF0]GEe@[De=@4eMGE8=CDe0KCdm<4e=CD1=GEe@;Cdm<2
d];B0]3@d0;Bd]82e]KF0]_Kf`;Ph>02hn?S0^7Qh@;Lg=`2e]KF0];BdP;@d=02dM7A0]?Cd`;Fe]H2
emOG0]SHf0;IfMT4f][J00?IfMWHf=SHf=P00]SHf0;IfMT2fm_K0]gMg@;Ng]h2gMgM0]cLg0;Jf]X2
f=SH0]KFeP;De=@2dm?C1=;BdP;Cdm<4e=CD0]GEeECDe=@4dm?C1Oooo`400001oooo0004oooo0P00
00Coool00`000?ooooooo`02oooo00<0003oooooool00oooo`030000oooooooo00;oool2dm?C1=;B
dPKAdM42d];B0]CDe0;Fe]H2emOG0]SHf0;GemL2e]KF0]GEe@CCdm<2e=CD0]GEe@;Fe]H4eMGE0]CD
e0CCdm<6e=CD2]GEe@CDe=@6dm?C7=CDe0[EeMD4e=CD1=?Cd`;De=@7eMGE1]CDe0SEeMD/e=CD0]?C
dd3De=@2dm?C1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`;Cdm<4d];B1]7Ad@;Bd]82
e=CD0]KFeP;GemL2f=SH0]OGe`;Fe]H2eMGE1=?Cd`;De=@2eMGE0]KFePCEeMD2e=CD1=?Cd`KDe=@:
eMGE1=CDe0KCdm<Le=CD2]GEe@CDe=@4dm?C0]CDe0OEeMD6e=CD2=GEeBcDe=@2dm?C@=CDe0;Cdm<5
oooo0@00007oool001Goool00`000?ooooooo`02oooo0];BdP;AdM44d=3@0]7Ad@;Bd]82dm?C0]CD
e0;Fe]H6emOG0]KFePSDe=@2eMGE0]KFePCEeMD4e=CD0]?Cd`KDe=@:eMGE1=CDe0KCdm<8e=CD1=?C
d`;De=@2eMGE0]KFeP;EeMD2d];B0/k>cP;;bl/2b/[:0/c<c0;@d=02e=CD0]OGe`;Hf=P2e]KF0]CD
e0;Bd]8?dM7A0]3@d0;?cll6c/k>0/o?c`;@d=02dM7A0];BdP;Cdm<6eMGE1=CDe0KCdm=De=CD0]?C
d`Goool100000Oooo`005Oooo`030000oooooooo00;oool2d];B0]7Ad@C@d=02dM7A0];BdP;Cdm<2
e=CD0]KFePKGemL2e]KF2=CDe0;EeMD2e]KF1=GEe@CDe=@2dm?C1]CDe0[EeMD4e=CD1]?Cd`SDe=@4
dm?C0]CDe0;EeMD2e]KF0]GEe@;Bd]82c/k>0/_;b`;:b/X2c<c<0]3@d0;De=@2emOG0]SHf0;Fe]H2
e=CD0];BdPoAdM42d=3@0/o?c`K>c/h2clo?0]3@d0;AdM42d];B0]?Cd`KEeMD4e=CD1]?CdeCDe=@2
dm?C1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`;Bd]82d=3@0/o?c`;@d=02dM7A0]?C
d`KEeMDne=CD1]?Cd`;De=@2eMGE0]KFeP;EeMD2d];B0/k>cP;:b/X2bLW90/[:bP;?cll2dm?C0]OG
e`;Hf=P2e]KF0]CDe0;Bd]8?dM7A0]3@d0;?cll6cLg=0/k>cP;?cll2dM7A0];BdP;Cdm<2e=CD1=GE
e@CDe=@6dm?CB=CDe0CCdm<2e=CD1]GEe@;Cdm<5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0];BdP;@d=02clo?0]3@d0;AdM42dm?C1]GEeCkDe=@6dm?C0]CDe0;EeMD2e]KF0]GEe@;Bd]82
c/k>0/[:bP;9bLT2b/[:0/o?c`;Cdm<2emOG0]SHf0;Fe]H2e=CD0];BdPoAdM42d=3@0/o?c`K=cLd2
c/k>0/o?c`;AdM42d];B0]?Cd`;De=@4eMGE1=CDe0KCdm=8e=CD1=?Cd`;De=@6eMGE0]?Cd`Goool1
00000Oooo`005Oooo`030000oooooooo00;oool2dM7A0/k>cP;=cLd2clo?0];BdP;Fe]H2emOG0]KF
eP;Cdm<2d=3@0/k>cP;?cll2dM7A0]?Cd`CEeMD2e=CD1];BdP;Cdm<4e=CD0]GEe@CDe=@6dm?C1=;B
dPCCdm<@e=CD1=?Cd`;De=@6eMGE0]?Cd`;AdM42clo?0/k>cP;?cll2dM7A0]?Cd`;De=@4eMGE1=CD
e0oCdm<2d];B0]7Ad@K@d=02dM7A1=;BdPCCdm<Fe=CD2=?Cd`[De=@8dm?C6=CDe0CCdm<6e=CD0]GE
e@CDe=@4dm?C0]CDe0;EeMD4e]KF0]CDe0Goool100000Oooo`005Oooo`800003oooo0]7Ad@;>c/h2
cLg=0/o?c`;Bd]82e]KF0]OGe`;Fe]H2dm?C0]3@d0;>c/h2clo?0]7Ad@;Cdm<4eMGE0]CDe0KBd]82
dm?C1=CDe0;EeMD4e=CD1]?Cd`CBd]84dm?C4=CDe0CCdm<2e=CD1]GEe@;Cdm<2dM7A0/o?c`;>c/h2
clo?0]7Ad@;Cdm<2e=CD1=GEe@CDe=@?dm?C0];BdP;AdM46d=3@0]7Ad@CBd]84dm?C5]CDe0SCdm<:
e=CD2=?CdaSDe=@4dm?C1]CDe0;EeMD4e=CD1=?Cd`;De=@2eMGE1=KFeP;De=@4oooo0P00007oool0
01Goool00`000?ooooooo`02oooo0]3@d0;<c<`2bl_;0/k>cP;Bd]82emOG0]WIf@;GemL2dM7A0/c<
c0;9bLT2b/[:0/k>cP;Cdm<4e]KF0]CDe0;AdM44d=3@0];BdP;De=@6eMGE0]CDe0;Cdm<2d];B2=7A
d@;Bd]82e=CD2=GEeA3De=@6eMGE0]KFeP;EeMD2e=CD0]?Cd`;Bd]82dM7A0];BdP;Cdm<2e=CD3MGE
e@kDe=@6eMGE1=CDe0SCdm<:e=CD3=?Cd`CDe=@2eMGE1=CDe0[Cdm<6e=CD1]?Cd`[De=@6dm?C0]CD
e0CEeMD2e=CD0]?Cd`CBd]82dm?C0]GEe@CFe]H2e=CD1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`;@d=02c<c<0/_;b`;>c/h2d];B0]OGe`;IfMT2emOG0]7Ad@;<c<`2bLW90/[:bP;>c/h2
dm?C1=KFeP;De=@2dM7A1=3@d0;Bd]82e=CD1]GEe@;De=@2dm?C0];BdPSAdM42d];B0]CDe0SEeMD@
e=CD1]GEe@;Fe]H2eMGE0]CDe0;Cdm<2d];B0]7Ad@;Bd]82dm?C0]CDe0gEeMD>e=CD1]GEe@CDe=@8
dm?C2]CDe0cCdm<4e=CD0]GEe@CDe=@:dm?C1]CDe0KCdm<:e=CD1]?Cd`;De=@4eMGE0]CDe0;Cdm<4
d];B0]?Cd`;EeMD4e]KF0]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;oool2clo?0/[:
bP;8b<P2bl_;0]3@d0;Fe]H2f=SH0]KFeP;@d=02bLW90/K6aP;7alL2c<c<0];BdPCFe]H2e=CD0]7A
d@C@d=02d];B0]CDe0KEeMD2e=CD0]?Cd`;Bd]86dM7A0];BdP;Cdm<4e=CD1=GEe@kDe=@2dm?C1=CD
e0;Fe]H2f=SH1=WIf@;Hf=P2eMGE0];BdPC@d=02d];B0]?Cd`;EeMD8e]KF2MGEe@;Fe]H:emOG0]KF
eP;EeMD2e=CD0]?Cd`;Bd]84dm?C2]CDe0cCdm<4e=CD0]GEe@CDe=@:dm?C1]CDe0KCdm<:e=CD1]?C
d`;De=@4eMGE0]CDe0;Cdm<4d];B0]?Cd`;Fe]H4emOG0]CDe0Goool100000Oooo`005Oooo`030000
oooooooo00;oool2clo?0/[:bP;8b<P2bl_;0]3@d0;Fe]H2f=SH0]KFeP;@d=02bLW90/K6aP;7alL2
c<c<0];BdPCFe]H2e=CD0]7Ad@C@d=02d];B0]CDe0KEeMD2e=CD0]?Cd`;Bd]86dM7A0];BdP;Cdm<4
e=CD1=GEe@kDe=@2dm?C1=CDe0;Fe]H2f=SH1=WIf@;Hf=P2eMGE0];BdPC@d=02d];B0]?Cd`;EeMD8
e]KF2MGEe@;Fe]H:emOG0]KFeP;EeMD2e=CD0]?Cd`;Bd]84dm?C2]CDe0cCdm<4e=CD0]GEe@CDe=@:
dm?C1]CDe0KCdm<:e=CD1]?Cd`;De=@4eMGE0]CDe0;Cdm<4d];B0]?Cd`;Fe]H4emOG0]CDe0Goool1
00000Oooo`005Oooo`030000oooooooo00;oool2c/k>0/O7a`;4a<@2a/K60/_;b`;AdM42e=CD0]?C
d`;>c/h2bLW90/K6aP;7alL2c<c<0]7Ad@;EeMD2e]KF0]GEe@;De=@2dm?C0];BdP;Cdm<:e=CD1]?C
dakDe=@4dm?C0]CDe0;EeMD2e]KF1]SHf0;Fe]H2e=CD1=;BdPCCdm<2e=CD4MGEe@CFe]H4emOG1=KF
eP;EeMD6e=CD1=?Cdd;De=@4dm?C1=CDe0;EeMD4e=CD0];BdP;AdM42d];B0]CDe0;GemL4fMWI0]CD
e0Goool100000Oooo`005Oooo`030000oooooooo00;oool2c/k>0/O7a`;4a<@2a/K60/_;b`;AdM42
e=CD0]?Cd`;>c/h2bLW90/K6aP;7alL2c<c<0]7Ad@;EeMD2e]KF0]GEe@;De=@2dm?C0];BdP;Cdm<:
e=CD1]?CdakDe=@4dm?C0]CDe0;EeMD2e]KF1]SHf0;Fe]H2e=CD1=;BdPCCdm<2e=CD4MGEe@CFe]H4
emOG1=KFeP;EeMD6e=CD1=?Cdd;De=@4dm?C1=CDe0;EeMD4e=CD0];BdP;AdM42d];B0]CDe0;GemL4
fMWI0]CDe0Goool100000Oooo`005Oooo`030000oooooooo00;oool2bl_;0/?3``Bn_[h2`l?30/[:
bPC>c/h2c<c<0/W9b@;8b<P2bLW90/c<c0;@d=02e=CD1=KFeP;GemL4e]KF0]GEe@CDe=@4dm?C1=CD
e0;EeMD2e]KF1]OGe`;EeMD2e=CD0]?Cd`CBd]84dm?C2]CDe0cCdm<2e=CD0]GEe@CFe]H2emOG0]KF
eP;EeMD2e=CD0]?Cd`;Bd]88dm?C2MCDe0;EeMD6e=CD1]?Cd`;De=@6eMGE1=CDe0KCdm<4e=CD2]GE
e@;De=@6dm?C0]CDe0SEeMDBe=CD0]?Cda3De=@2dm?C0];BdP;AdM42d];B0]CDe0;Hf=P2fm_K0][J
fP;De=@5oooo0@00007oool001Goool00`000?ooooooo`02oooo0/_;b`;3`l<4_[jn0/?3``;:b/X4
c/k>0/c<c0;9bLT2b<S80/W9b@;<c<`2d=3@0]CDe0CFe]H2emOG1=KFeP;EeMD4e=CD1=?Cd`CDe=@2
eMGE0]KFePKGemL2eMGE0]CDe0;Cdm<4d];B1=?Cd`[De=@<dm?C0]CDe0;EeMD4e]KF0]OGe`;Fe]H2
eMGE0]CDe0;Cdm<2d];B2=?Cd`WDe=@2eMGE1]CDe0KCdm<2e=CD1]GEe@CDe=@6dm?C1=CDe0[EeMD2
e=CD1]?Cd`;De=@8eMGE4]CDe0;Cdm<@e=CD0]?Cd`;Bd]82dM7A0];BdP;De=@2f=SH0]_Kf`;Jf]X2
e=CD1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`;7alL2_;bl1;Fe]@:j^[X2`L710/K6
aPC9bLT4b<S80/[:bP;<c<`2clo?0];BdP;EeMD2emOG1=WIf@;Hf=P2emOG0]GEe@;Cdm<2d];B0]?C
d`;De=@2eMGE0]OGe`;IfMT4f][J0]WIf@;GemL2e=CD0];BdPCAdM42d];B0]?Cd`CDe=@4eMGE1=CD
e0;Cdm<2d];B1=3@d0;?cll2d=3@0]?Cd`;Fe]H2fMWI1=[JfP;GemL2e=CD0];BdPKAdM44d];B2m?C
d`CBd]86dM7A0];BdP;De=@2eMGE1=KFeP;EeMD2e=CD1]?Cd`;De=@4eMGE1]KFeP;EeMD2e=CD1];B
dP;De=@2eMGE1]KFeP;EeMD6e=CD1]GEe@;De=@6dm?C3=CDe0;Cdm<2d];B1=7Ad@;Bd]82eMGE0]WI
f@;Kfm/2fMWI0];BdPGoool100000Oooo`005Oooo`800003oooo0/O7a`:l_;`4]KFe0[Zj^P;1`L42
a/K61<W9b@C8b<P2b/[:0/c<c0;?cll2d];B0]GEe@;GemL4fMWI0]SHf0;GemL2eMGE0]?Cd`;Bd]82
dm?C0]CDe0;EeMD2emOG0]WIf@CJf]X2fMWI0]OGe`;De=@2d];B1=7Ad@;Bd]82dm?C1=CDe0CEeMD4
e=CD0]?Cd`;Bd]84d=3@0/o?c`;@d=02dm?C0]KFeP;IfMT4f][J0]OGe`;De=@2d];B1]7Ad@CBd]8;
dm?C1=;BdPKAdM42d];B0]CDe0;EeMD4e]KF0]GEe@;De=@6dm?C0]CDe0CEeMD6e]KF0]GEe@;De=@6
d];B0]CDe0;EeMD6e]KF0]GEe@KDe=@6eMGE0]CDe0KCdm<<e=CD0]?Cd`;Bd]84dM7A0];BdP;EeMD2
fMWI0]_Kf`;IfMT2d];B1?ooo`800001oooo000Eoooo00<0003oooooool00_ooo`:n_[h2/K6a1:VY
Z@:_[jl2]kNg0[no_`;3`l<2a<C40/G5a@;6a/H2b<S80/_;b`;>c/h2dM7A0]CDe0;Fe]H4f=SH0]OG
e`;EeMD2dm?C1];BdP;De=@2eMGE0]OGe`KHf=P2emOG0]KFeP;De=@2dm?C1];BdP;Cdm<4e=CD1]GE
e@;De=@2dm?C0];BdP;@d=02clo?0/k>cP;?cll2d];B0]GEe@;IfMT2fm_K0][JfP;Hf=P2eMGE0];B
dPCAdM44d];B2m?Cd`CBd]88dM7A0];BdP;De=@2eMGE1=KFeP;EeMD2e=CD1]?Cd`;De=@4eMGE1]KF
eP;EeMD2e=CD0]?Cd`;Bd]82dm?C0]CDe0;EeMD4e]KF1=GEe@KDe=@4eMGE1=CDe0KCdm<<e=CD1=?C
d`CBd]82dm?C0]GEe@;Hf=P2fMWI0]GEe@;;bl/5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0[jn_P:a/K44ZJVY0Zn_[`:g]kL2_kno0/?3``;4a<@2aLG50/K6aP;8b<P2bl_;0/k>cP;AdM42
e=CD0]KFePCHf=P2emOG0]GEe@;Cdm<6d];B0]CDe0;EeMD2emOG1]SHf0;GemL2e]KF0]CDe0;Cdm<6
d];B0]?Cd`CDe=@6eMGE0]CDe0;Cdm<2d];B0]3@d0;?cll2c/k>0/o?c`;Bd]82eMGE0]WIf@;Kfm/2
f][J0]SHf0;EeMD2d];B1=7Ad@CBd]8;dm?C1=;BdPSAdM42d];B0]CDe0;EeMD4e]KF0]GEe@;De=@6
dm?C0]CDe0CEeMD6e]KF0]GEe@;De=@2dm?C0];BdP;Cdm<2e=CD0]GEe@CFe]H4eMGE1]CDe0CEeMD4
e=CD1]?Cd`cDe=@4dm?C1=;BdP;Cdm<2eMGE0]SHf0;IfMT2eMGE0/_;b`Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2/K6a0Z:RXP:JVYX2W9bL0ZBTY0:_[jl2^;Rh0[bl_0Bm_Kd2_[jn0/;2
`P;8b<P2c/k>0];BdP;De=@2dm?C0];BdP;AdM42d=3@1<o?c`;@d=02dM7A0];BdP;De=@2eMGE0]CD
e0;Cdm<6dM7A0];BdP;Cdm<2e=CD1=GEeACDe=@4dm?C1=;BdP;Cdm<2e=CD2=GEe@oDe=@Fdm?CC=CD
e0;EeMD@e=CD1=GEe@;Cdm<2c<c<0/30`0Goool100000Oooo`005Oooo`030000oooooooo00;oool2
/K6a0Z:RXP:JVYX2W9bL0ZBTY0:_[jl2^;Rh0[bl_0Bm_Kd2_[jn0/;2`P;8b<P2c/k>0];BdP;De=@2
dm?C0];BdP;AdM42d=3@1<o?c`;@d=02dM7A0];BdP;De=@2eMGE0]CDe0;Cdm<6dM7A0];BdP;Cdm<2
e=CD1=GEeACDe=@4dm?C1=;BdP;Cdm<2e=CD2=GEe@oDe=@Fdm?CC=CDe0;EeMD@e=CD1=GEe@;Cdm<2
c<c<0/30`0Goool100000Oooo`005Oooo`030000oooooooo00;oool2X:2P0Y6AT@:9RHT2SHf=0YRH
V0:VYZH2/K6a0[Fe]@:c/k<2/K6a0[:b/P:h^;P2`/;20/c<c0CBd]82c/k>0/W9b@C5aLD2alO70/[:
bP;=cLd2clo?0]7Ad@;Cdm<2d];B0/o?c`;;bl/2alO70/G5a@;7alL2c<c<0];BdP;GemL4f=SH0]KF
eP;EeMD2e=CD2=?Cd`;De=@2eMGE0]OGe`;Hf=P4fMWI0]SHf0;EeMD2d];B0/o?c`;=cLd2c/k>0]3@
d0;Cdm<2e]KF2=OGe`;Fe]H00mGEeMCDe=CDe008e=CD0]GEe@CFe]H2emOG0]KFeP;EeMD2e=CD0];B
dP;AdM42d];B0]?Cd`;EeMD4e]KF0]GEe@;De=@2dm?C1=;BdP;AdM42d];B0]?Cd`;De=@2eMGE0]KF
eP;EeMD2e=CD0];BdPCAdM42d];B0]?Cd`CDe=@4dm?C0];BdPCCdm<2e=CD2=GEe@KDe=@4eMGE1]KF
eP;EeMD2e=CD0]7Ad@;;bl/2`L710[:b/PGoool100000Oooo`005Oooo`030000oooooooo00;oool2
X:2P0Y6AT@:9RHT2SHf=0YRHV0:VYZH2/K6a0[Fe]@:c/k<2/K6a0[:b/P:h^;P2`/;20/c<c0CBd]82
c/k>0/W9b@C5aLD2alO70/[:bP;=cLd2clo?0]7Ad@;Cdm<2d];B0/o?c`;;bl/2alO70/G5a@;7alL2
c<c<0];BdP;GemL4f=SH0]KFeP;EeMD2e=CD2=?Cd`;De=@2eMGE0]OGe`;Hf=P4fMWI0]SHf0;EeMD2
d];B0/o?c`;=cLd2c/k>0]3@d0;Cdm<2e]KF2=OGe`;Fe]H00mGEeMCDe=CDe008e=CD0]GEe@CFe]H2
emOG0]KFeP;EeMD2e=CD0];BdP;AdM42d];B0]?Cd`;EeMD4e]KF0]GEe@;De=@2dm?C1=;BdP;AdM42
d];B0]?Cd`;De=@2eMGE0]KFeP;EeMD2e=CD0];BdPCAdM42d];B0]?Cd`CDe=@4dm?C0];BdPCCdm<2
e=CD2=GEe@KDe=@4eMGE1]KFeP;EeMD2e=CD0]7Ad@;;bl/2`L710[:b/PGoool100000Oooo`005Ooo
o`030000oooooooo00;oool2T92@0WmoO`9fMWH2O7al0X^;R`:MWId2ZZZZ0Zb/[0:XZ:P2XZ:R0Z6Q
X@:ZZZX2^;Rh0/O7a`;?cll2c/k>0/K6aP:m_Kd4^;Rh0[fm_@;3`l<2bLW90/g=c@C@d=02c/k>0/S8
b0;1`L42^k^k0[Vi^@:n_[h2alO70]7Ad@;IfMT2g=cL0]_Kf`;Hf=P2e=CD0];BdPKAdM42d];B0]CD
e0;Fe]H2fMWI0]_Kf`;MgMd2g]kN0]cLg0;Fe]H2clo?0/W9b@;5aLD2alO70/c<c0;Bd]82emOG1=[J
fP;Hf=P2emOG0mKFeP[EeMD2e]KF0]OGe`;IfMT2f][J0]WIf@;GemL2dm?C0]3@d0;>c/h2clo?0];B
dP;EeMD4emOG0]KFeP;De=@2d];B0]3@d0;?cll2c/k>0/o?c`;AdM42e=CD0]KFeP;Hf=P2e]KF0]CD
e0;AdM44clo?0]3@d0;Bd]86dm?C1=;BdP;Cdm<2e=CD0]GEe@CFe]H2eMGE2=CDe0;EeMD2emOG0]SH
f0;IfMT2f=SH0]KFeP;Cdm<2c/k>0/G5a@:h^;P2YJFU1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`:@T902Ogmo0WIfMP9lO7`2Rh^;0YfMW@:ZZZX2[:b/0ZRXZ0:RXZ82XJ6Q0ZZZZP:h^;P2
alO70/o?c`;>c/h2a/K60[fm_@Bh^;P2_Kfm0/?3``;9bLT2cLg=1=3@d0;>c/h2b<S80/71`@:k^k/2
^KVi0[jn_P;7alL2dM7A0]WIf@;Lg=`2fm_K0]SHf0;De=@2d];B1]7Ad@;Bd]82e=CD0]KFeP;IfMT2
fm_K0]gMg@;Ng]h2g=cL0]KFeP;?cll2bLW90/G5a@;7alL2c<c<0];BdP;GemL4f][J0]SHf0;GemL3
e]KF2]GEe@;Fe]H2emOG0]WIf@;Jf]X2fMWI0]OGe`;Cdm<2d=3@0/k>cP;?cll2d];B0]GEe@CGemL2
e]KF0]CDe0;Bd]82d=3@0/o?c`;>c/h2clo?0]7Ad@;De=@2e]KF0]SHf0;Fe]H2e=CD0]7Ad@C?cll2
d=3@0];BdPKCdm<4d];B0]?Cd`;De=@2eMGE1=KFeP;EeMD8e=CD0]GEe@;GemL2f=SH0]WIf@;Hf=P2
e]KF0]?Cd`;>c/h2aLG50[Rh^0:UYJD5oooo0@00007oool001Goool00`000?ooooooo`02oooo0X>3
P`9/K6`2HV9R0VUYJ@9lO7`2TY:B0Z:RXP:SXj<2VYZJ0Y2@T0:=SHd2UiNG0ZVYZ@:k^k/2aLG50/C4
a0:k^k/2/K6a0Z^[Z`:][Jd2]KFe0[jn_P;6a/H2bl_;0/g=c@;<c<`2bLW90/71`@:i^KT2/[:b0[6a
/@:g]kL2`l?30]3@d0;Jf]X2gMgM0]_Kf`;GemL2d];B0]3@d0;?cll2d=3@0]7Ad@;Bd]82dm?C0]GE
e@;GemL2f][J0]cLg0;MgMd2f][J0]CDe0;<c<`2aLG50/71`@;4a<@2bl_;0];BdP;Hf=P2f][J0]SH
f0;EeMD2dm?C0];BdP03e=CDeMGEeMGE00KFe]H2eMGE0]KFeP;GemL2fMWI1=[JfP;GemL2d];B0/k>
cP;<c<`2cLg=0]3@d0;Cdm<2eMGE0]KFeP;De=@2d];B0/o?c`;=cLd2c<c<0/_;b`;<c<`2clo?0];B
dP;Fe]H2emOG0]KFeP;De=@2d=3@0/k>cP;=cLd2clo?0]3@d0[Bd]82dm?C0]GEe@CFe]H2eMGE0]?C
d`;Bd]84dM7A0]?Cd`;EeMD2emOG0]WIf@;Jf]X2fMWI0]OGe`;De=@2c/k>0/G5a@:e]KD2WYjN1Ooo
o`400001oooo000Eoooo0P0000?oool2Ph>30Va/K09RHV82JFUY0WalO0:BTY82XZ:R0Z>SX`:JVYX2
T92@0Xf=S@:GUiL2ZJVY0[^k^`;5aLD2a<C40[^k^`:a/K42Zj^[0Zf][@:e]KD2_[jn0/K6aP;;bl/2
cLg=0/c<c0;9bLT2`L710[Vi^@:b/[82/K6a0[Ng]`;3`l<2d=3@0][JfP;MgMd2fm_K0]OGe`;Bd]82
d=3@0/o?c`;@d=02dM7A0];BdP;Cdm<2eMGE0]OGe`;Jf]X2g=cL0]gMg@;Jf]X2e=CD0/c<c0;5aLD2
`L710/C4a0;;bl/2d];B0]SHf0;Jf]X2f=SH0]GEe@;Cdm<2d];B00?De=CEeMGEeMD01]KFeP;EeMD2
e]KF0]OGe`;IfMT4f][J0]OGe`;Bd]82c/k>0/c<c0;=cLd2d=3@0]?Cd`;EeMD2e]KF0]CDe0;Bd]82
clo?0/g=c@;<c<`2bl_;0/c<c0;?cll2d];B0]KFeP;GemL2e]KF0]CDe0;@d=02c/k>0/g=c@;?cll2
d=3@2];BdP;Cdm<2eMGE1=KFeP;EeMD2dm?C0];BdPCAdM42dm?C0]GEe@;GemL2fMWI0][JfP;IfMT2
emOG0]CDe0;>c/h2aLG50[Fe]@:NWYh4oooo0P00007oool001Goool00`000?ooooooo`02oooo0W]k
N`9LG5`2C4a<0U=CD`9[Jf/2QXJ619RHV0:<S8`2OGem0WMgM`9oOgl2Ti>C0ZNWY`:d];@2]KFe0Zf]
[@:TY:@2X:2P0ZFUY@:`/;02_;bl0/G5a@C:b/X2alO70/?3``:l_;`2];Bd1:n_[`:g]kL2`l?30]7A
d@;Jf]X2g=cL0]WIf@;Bd]82cLg=0/c<c0;=cLd2clo?1]7Ad@;@d=04dM7A1=?Cd`;Bd]82c/k>0/W9
b@;4a<@2`l?30/O7a`;>c/h2e=CD0]SHf0;Fe]H2dM7A0/_;b`;8b<P2b/[:00?>c/kDe=CDe=@00]OG
e`;Hf=P2emOG0]GEe@CDe=@2e]KF1=OGe`;De=@2d=3@0/c<c0C:b/X2c<c<0/o?c`C@d=02clo?0/k>
cP;<c<`2bl_;1<[:bP;;bl/2cLg=0]3@d0;Cdm<2eMGE0]CDe0;Cdm<2dM7A0/o?c`C>c/h2clo?0]3@
d0;AdM42d];B0]?Cd`;De=@2eMGE1=KFeP;EeMD2d];B0/o?c`;=cLd2bl_;0/c<c0;?cll2dm?C0]KF
eP;Hf=P2fMWI0]SHf0;GemL2e]KF0]CDe0;=cLd2^k^k0YjNWPGoool100000Oooo`005Oooo`030000
oooooooo00;oool2Ng]k0UaLG09<C4`2De=C0V][J`:6QXH4V9RH0Xb<S09mOGd2MgMg0WmoO`:CTi<2
YjNW0[Bd]0:e]KD2[Jf]0ZBTY0:PX:02YJFU0[2`/0:l_;`2aLG51<[:bP;7alL2`l?30[bl_0:d];@4
[jn_0[Ng]`;3`l<2dM7A0][JfP;Lg=`2fMWI0];BdP;=cLd2c<c<0/g=c@;?cll6dM7A0]3@d0CAdM44
dm?C0];BdP;>c/h2bLW90/C4a0;3`l<2alO70/k>cP;De=@2f=SH0]KFeP;AdM42bl_;0/S8b0;:b/X0
0lk>c]CDe=CDe002emOG0]SHf0;GemL2eMGE1=CDe0;Fe]H4emOG0]CDe0;@d=02c<c<1<[:bP;<c<`2
clo?1=3@d0;?cll2c/k>0/c<c0;;bl/4b/[:0/_;b`;=cLd2d=3@0]?Cd`;EeMD2e=CD0]?Cd`;AdM42
clo?1<k>cP;?cll2d=3@0]7Ad@;Bd]82dm?C0]CDe0;EeMD4e]KF0]GEe@;Bd]82clo?0/g=c@;;bl/2
c<c<0/o?c`;Cdm<2e]KF0]SHf0;IfMT2f=SH0]OGe`;Fe]H2e=CD0/g=c@:k^k/2WYjN1Oooo`400001
oooo000Eoooo00<0003oooooool00_ooo`9gMgL2CTi>0SPh>08n?Sh2FEUI0WUiN@:>SXh2SHf=0Wal
O09YJFT2GemO0VIVIP9iNGT2SXj>0YfMW@:PX:02W9bL19NGU`:OWil2[:b/0[Vi^@;3`l<2a/K60/C4
a0;0`<02^[Zj0[Fe]@:a/K42[Zj^0[6a/@:i^KT2aLG50]7Ad@CHf=P2d];B0/_;b`C5aLD2bLW90/g=
c@C@d=02cLg=0/W9b@;6a/H2a<C41/?3``;4a<@2aLG50/K6aP;9bLT2cLg=0];BdP;Fe]H2eMGE0/o?
c`;6a/H2_[jn0[Zj^P:n_[h00lO7am7AdM7Ad@02f=SH0][JfP;GemL2d];B0/o?c`;>c/h2d=3@0]7A
d@;Bd]82d=3@0/g=c@;:b/X8b<S80/W9b@k8b<P2bLW90/_;b`;<c<`2c/k>0]3@d0SAdM42d=3@0/o?
c`C>c/h2clo?0];BdP;De=@2e]KF1=OGe`;Fe]H2dm?C0/o?c`;:b/X2a/K60/C4a0;5aLD2bLW90/o?
c`;Cdm<4e]KF1=GEe@;IfMT2fm_K0]SHf0;7alL2Y:BT1Oooo`400001oooo000Eoooo00<0003ooooo
ool00_ooo`9gMgL2CTi>0SPh>08n?Sh2FEUI0WUiN@:>SXh2SHf=0WalO09YJFT2GemO0VIVIP9iNGT2
SXj>0YfMW@:PX:02W9bL19NGU`:OWil2[:b/0[Vi^@;3`l<2a/K60/C4a0;0`<02^[Zj0[Fe]@:a/K42
[Zj^0[6a/@:i^KT2aLG50]7Ad@CHf=P2d];B0/_;b`C5aLD2bLW90/g=c@C@d=02cLg=0/W9b@;6a/H2
a<C41/?3``;4a<@2aLG50/K6aP;9bLT2cLg=0];BdP;Fe]H2eMGE0/o?c`;6a/H2_[jn0[Zj^P:n_[h0
0lO7am7AdM7Ad@02f=SH0][JfP;GemL2d];B0/o?c`;>c/h2d=3@0]7Ad@;Bd]82d=3@0/g=c@;:b/X8
b<S80/W9b@k8b<P2bLW90/_;b`;<c<`2c/k>0]3@d0SAdM42d=3@0/o?c`C>c/h2clo?0];BdP;De=@2
e]KF1=OGe`;Fe]H2dm?C0/o?c`;:b/X2a/K60/C4a0;5aLD2bLW90/o?c`;Cdm<4e]KF1=GEe@;IfMT2
fm_K0]SHf0;7alL2Y:BT1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`9eMGD2@d=30RLW
9`8[:b/2B4Q80V][J`:1PH42P8200Ve]K@9FEUH2BDU90Te=C@9OGel2Lg=c0X>3P`J9RHT2S8b<0YNG
U`:VYZH2/k>c0[bl_0:m_Kd2^[Zj0[Bd]0:_[jl2[:b/0Z^[Z`:][Jd2/[:b0[Zj^P;4a<@2cLg=0];B
dP;?cll2b<S80/30`0:k^k/2_;bl0/;2`P;8b<P2c<c<0/_;b`;6a/H2`<300[Zj^P:f][H2/k>c0[:b
/P:c/k<2^;Rh0[no_`;6a/H2cLg=0]7Ad@CDe=@2clo?0/G5a@:i^KT2[jn_0Z^[Z`:b/[800kno_lk>
c/k>cP02f=SH0][JfP;EeMD2c/k>0/S8b0;6a/H2alO70/W9b@C:b/X2bLW90/O7a`;6a/H2a<C40/?3
``;1`L44`<300/71`@;2`/82`l?30/C4a0;6a/H2alO72<S8b0;9bLT2c<c<0/o?c`;Bd]82e=CD0];B
dP;?cll2bl_;0/[:bP;<c<`2d=3@0]CDe0;GemL2f=SH0]OGe`;EeMD2d=3@0/_;b`;5aLD2`<300[fm
_@:o_kl2`l?30/W9b@;?cll4dM7A0/o?c`;Bd]82fMWI0^;RhP;Shn<2d];B0ZZZZPGoool100000Ooo
o`005Oooo`030000oooooooo00;oool2MGEe0T=3@`8W9bL2:b/[0TQ8B09[Jf/2PH610X20P09]KFd2
EUIF0TU9B@9=CDd2GemO0W=cL`:3Ph<6RHV90Xb<S0:GUiL2YZJV0[>c/`:l_;`2_Kfm0[Zj^P:d];@2
[jn_0Zb/[0:[Zj/2[Jf]0[:b/P:j^[X2a<C40/g=c@;Bd]82clo?0/S8b0;0`<02^k^k0[bl_0;2`/82
b<S80/c<c0;;bl/2a/K60/30`0:j^[X2][Jf0[>c/`:b/[82/k>c0[Rh^0:o_kl2a/K60/g=c@;AdM44
e=CD0/o?c`;5aLD2^KVi0Zn_[`:[Zj/2/[:b00>o_ko>c/k>c/h00]SHf0;Jf]X2eMGE0/k>cP;8b<P2
a/K60/O7a`;9bLT4b/[:0/W9b@;7alL2a/K60/C4a0;3`l<2`L711<30`0;1`L42`/;20/?3``;4a<@2
a/K60/O7a`S8b<P2bLW90/c<c0;?cll2d];B0]CDe0;Bd]82clo?0/_;b`;:b/X2c<c<0]3@d0;De=@2
emOG0]SHf0;GemL2eMGE0]3@d0;;bl/2aLG50/30`0:m_Kd2_kno0/?3``;9bLT2clo?1=7Ad@;?cll2
d];B0]WIf@;Rh^82hn?S0];BdP:ZZZX5oooo0@00007oool001Goool00`000?ooooooo`02oooo0WAd
M08m?Cd271`L0QhN7P8j>SX2G5aL0W9bLP9aLG42GEeM0TI6AP8g=cL2>CTi0TQ8B09JFUX2JFUY0W5a
L@9eMGD2N7Qh0WmoO`:;Rh/2VIVI0ZFUY@B/[:`2YjNW0Z6QX@BLW9`2X:2P0ZFUY@:][Jd2]KFe0[fm
_@;4a<@2a/K60/;2`P:k^k/2/k>c0Zj^[P:`/;02][Jf0[bl_0;1`L42`/;20[no_`:j^[X2];Bd0Zj^
[P:XZ:P2YJFU0ZJVYP:][Jd2]kNg0/;2`P;;bl/4clo?0/c<c0;4a<@2^;Rh0Zb/[0:RXZ82X:2P0ZVY
Z@03^[Zjc<c<c<c<00CHf=P2dM7A0/K6aP:m_Kd2^[Zj0[bl_0:o_kl2`L712<?3``;1`L42_kno0[bl
_0:k^k/2^[Zj0[^k^`:m_Kd2_kno0/71`@;4a<@2a/K60/O7a`;6a/H2`l?30/71`@;2`/82alO70/k>
cP;De=@2emOG0]CDe0;>c/h2b<S80/C4a0;6a/H2c<c<0];BdP;GemL2f=SH0]OGe`;De=@2d=3@0/[:
bP;5aLD2`<300[bl_0:m_Kd2`<300/G5a@;9bLT2b/[:0/W9b@;8b<P2bl_;0]KFeP;Rh^82inOW0]OG
e`:][Jd5oooo0@00007oool000?oool5000000Coool0000000000003oooo00<0003oooooool00ooo
o`030000oooooooo00;oool2M7Ad0Sdm?@8L71`27QhN0SXj>P9LG5`2LW9b0W5aL@9MGEd2ATI60SLg
=`8i>CT2B4Q80UYJFP9YJFT2LG5a0WEeM@9hN7P2Ogmo0X^;R`:IVIT2YJFU1:b/[0:WYjL2XJ6Q19bL
W0:PX:02YJFU0Zf][@:e]KD2_Kfm0/C4a0;6a/H2`/;20[^k^`:c/k<2[Zj^0[2`/0:f][H2_;bl0/71
`@;2`/82_kno0[Zj^P:d];@2[Zj^0ZRXZ0:UYJD2YZJV0Zf][@:g]kL2`/;20/_;b`C?cll2c<c<0/C4
a0:h^;P2[:b/0Z:RXP:PX:02ZJVY00>j^[[<c<c<c<`01=SHf0;AdM42a/K60[fm_@:j^[X2_;bl0[no
_`;1`L48`l?30/71`@:o_kl2_;bl0[^k^`:j^[X2^k^k0[fm_@:o_kl2`L710/C4a0;6a/H2alO70/K6
aP;3`l<2`L710/;2`P;7alL2c/k>0]CDe0;GemL2e=CD0/k>cP;8b<P2a<C40/K6aP;<c<`2d];B0]OG
e`;Hf=P2emOG0]CDe0;@d=02b/[:0/G5a@;0`<02_;bl0[fm_@;0`<02aLG50/W9b@;:b/X2bLW90/S8
b0;;bl/2e]KF0^;RhP;WinL2emOG0Zf][@Goool100000Oooo`001Oooo`050000oooooooooooo0000
00Coool00`000?ooo`000004oooo00<0003oooooool00_ooo`9bLW82>c/k0QTI6@8H61P2;Rh^0Ta<
C09PH602GemO0Ti>CP8j>SX4;2`/0SLg=`95ADD2DU9B0U]KF`9PH602IVIV0Vi^KP9hN7P2Q8B40Xf=
S@:ATI42T92@0XZ:RP:5QHD2Ph>30XF5Q@:<S8`2UIFE0YnOW`:WYjL2[Zj^0[>c/`:d];@2/K6a0Z^[
Z`:UYJD4XJ6Q0ZFUY@:ZZZX2[jn_0[Bd]0:h^;P2^KVi0[Ng]`:a/K42Z:RX0Z2PX0:NWYh2Xj>S0Zf]
[@:i^KT2`L710/C4a0;2`/82_Kfm0[Fe]@:[Zj/2X:2P0YZJVP:KVi/2YjNW00>i^KW<c<c<c<`00]OG
e`;EeMD2b/[:0[^k^`:`/;02[:b/0Zj^[P:c/k<2^;Rh0[^k^`:m_Kd2_[jn0[no_`:n_[h2_Kfm0[^k
^`:i^KT2^;Rh1;Vi^@:k^k/2_kno0/;2`P;5aLD2a/K60/?3``:n_[h4^[Zj0/71`@;<c<`2e]KF0]_K
f`;GemL2cLg=0/;2`P:l_;`2_[jn0/G5a@;=cLd2eMGE0]SHf0;GemL2eMGE0]7Ad@;>c/h2b/[:0/K6
aP;3`l<2`L710/30`0;1`L42`/;20/71`@:o_kl2_[jn0/;2`P;>c/h2fm_K0^3Ph0;Bd]82ZZZZ1Ooo
o`400001oooo0005oooo00<0003oooooool00_ooo`070000oooooooooooo0000oooo000000Coool2
00000oooo`9bLW82>c/k0QTI6@8H61P2;Rh^0Ta<C09PH602GemO0Ti>CP8j>SX4;2`/0SLg=`95ADD2
DU9B0U]KF`9PH602IVIV0Vi^KP9hN7P2Q8B40Xf=S@:ATI42T92@0XZ:RP:5QHD2Ph>30XF5Q@:<S8`2
UIFE0YnOW`:WYjL2[Zj^0[>c/`:d];@2/K6a0Z^[Z`:UYJD4XJ6Q0ZFUY@:ZZZX2[jn_0[Bd]0:h^;P2
^KVi0[Ng]`:a/K42Z:RX0Z2PX0:NWYh2Xj>S0Zf][@:i^KT2`L710/C4a0;2`/82_Kfm0[Fe]@:[Zj/2
X:2P0YZJVP:KVi/2YjNW00>i^KW<c<c<c<`00]OGe`;EeMD2b/[:0[^k^`:`/;02[:b/0Zj^[P:c/k<2
^;Rh0[^k^`:m_Kd2_[jn0[no_`:n_[h2_Kfm0[^k^`:i^KT2^;Rh1;Vi^@:k^k/2_kno0/;2`P;5aLD2
a/K60/?3``:n_[h4^[Zj0/71`@;<c<`2e]KF0]_Kf`;GemL2cLg=0/;2`P:l_;`2_[jn0/G5a@;=cLd2
eMGE0]SHf0;GemL2eMGE0]7Ad@;>c/h2b/[:0/K6aP;3`l<2`L710/30`0;1`L42`/;20/71`@:o_kl2
_[jn0/;2`P;>c/h2fm_K0^3Ph0;Bd]82ZZZZ1?ooo`800001oooo0005oooo00<0003oooooool00_oo
o`070000oooooooooooo0000oooo000000Coool00`000?ooooooo`02oooo0Vi^KP8m?Cd271`L0QLG
5`8W9bL2?Cdm14a<C090@402<S8b0RPX:08V9RH2;Bd]0SDe=@8n?Sh2ATI60Te=C@9CDe<2Fe]K0V9R
HP9YJFT2KVi^0W1`L09]KFd2JFUY16EUI@9[Jf/2M7Ad0X20P0::RXX2Ti>C0YRHV0:LW9`2WIfM0Y^K
V`:IVIT2UYJF0Y>CT`:ATI42T92@0Y:BTP:HV9P2Xj>S0Zn_[`:j^[X2_[jn0[Vi^@:][Jd2X:2P0YVI
V@:JVYX2XJ6Q0ZZZZP:`/;02/K6a0Zj^[P:YZJT2Xj>S0YbLW0:GUiL2U9BD0YVIV@:WYjL00kZj^/_;
bl_;b`02e=CD0/o?c`:o_kl2[Zj^0Z2PX0:LW9`2WinO0ZJVYP:][Jd2/[:b0[Fe]@:h^;P2^KVi0[Zj
^P:k^k/2^[Zj0[Vi^@:h^;P4]kNg0[Rh^0:k^k/2_kno1<?3``:o_kl2^;Rh0[:b/P:c/k<2^k^k0/W9
b@;GemL2g=cL0]OGe`;9bLT2^k^k0[:b/P:c/k<2^k^k0/K6aP;?cll4e=CD0]?Cd`KBd]82d=3@0/c<
c0;7alL2`L710[fm_@:i^KT2]KFe0[>c/`:b/[82][Jf0/71`@;<c<`2dM7A0/C4a0:PX:05oooo0@00
007oool000Goool01`000?ooooooooooo`000?ooo`000002oooo00<0003oool000001?ooo`030000
oooooooo00;oool2KVi^0Sdm?@8L71`25aLG0RLW9`8m?Cd4C4a<0T10@08b<S82:2PX0RHV9P8];Bd2
=CDe0Shn?P96ATH2CDe=0U=CD`9KFe/2HV9R0VUYJ@9^KVh2L71`0Ve]K@9YJFT4IFEU0V][J`9dM7@2
P8200XZ:RP:CTi<2V9RH0YbLW0:MWId2Vi^K0YVIV@:FUYH2Ti>C0Y6AT@:@T902TY:B0YRHV0:SXj<2
[jn_0[Zj^P:n_[h2^KVi0Zf][@:PX:02VIVI0YZJVP:QXJ42ZZZZ0[2`/0:a/K42[Zj^0ZVYZ@:SXj<2
W9bL0YNGU`:DU9@2VIVI0ZNWY`03^[Zjbl_;bl_;00;De=@2clo?0[no_`:^[Zh2X:2P0YbLW0:OWil2
YZJV0Zf][@:b/[82]KFe0[Rh^0:i^KT2^[Zj0[^k^`:j^[X2^KVi0[Rh^0Bg]kL2^;Rh0[^k^`:o_kl4
`l?30[no_`:h^;P2/[:b0[>c/`:k^k/2bLW90]OGe`;Lg=`2emOG0/W9b@:k^k/2/[:b0[>c/`:k^k/2
a/K60/o?c`CDe=@2dm?C1];BdP;@d=02c<c<0/O7a`;1`L42_Kfm0[Vi^@:e]KD2/k>c0[:b/P:f][H2
`L710/c<c0;AdM42a<C40Z2PX0Goool100000Oooo`001?ooo`800004oooo00<0003oooooool00_oo
o`030000oooooooo00?oool00`000?ooooooo`02oooo0VUYJ@90@4028R8R0QXJ6P8R8R82<30`0S/k
>`8l?3`2=SHf0Rh^;P8X:2P29RHV0RLW9`8[:b/2;bl_0SDe=@8l?3`2@d=30TU9B@9=CDd2Cdm?0U1@
D09>CTh2Bd];0TQ8B097AdL2BDU90U5AD@9LG5`2JFUY0WAdM09lO7`2P8200X:2PP:3Ph<2QHF518J6
QP:4Q8@2P8200W]kN`9iNGT2Ogmo0Xn?S`:TY:@2]kNg0/;2`P:n_[h2/K6a0Z2PX0:CTi<2Shn?0Y:B
TP:GUiL2VYZJ0YVIV@:EUID2TY:B0Xn?S`B=SHd2Shn?0YNGU`:VYZH00kNg]lK6a/K6aP02c<c<0/C4
a0:b/[82WinO0Y6AT@:=SHd2TI6A0YVIV@:RXZ82YjNW0Z^[Z`:][Jd2/;2`0[:b/P:e]KD2][Jf0[Ng
]`:f][H2];Bd0[>c/`:d];@2][Jf0[Zj^PBn_[h2^KVi0[6a/@BZZZX2];Bd0/?3``;AdM42emOG0]7A
d@;2`/82/[:b1:NWY`:_[jl2^[Zj0/C4a0;:b/X2c<c<0/g=c@;?cll2d];B0]CDe0;EeMD2dM7A0/W9
b@:n_[h2];Bd0Z^[Z`:VYZH4Xj>S0ZRXZ0:`/;02^KVi0[bl_0:a/K42Ti>C1Oooo`400001oooo000E
oooo00<0003oooooool00_ooo`9YJFT2@4100R8R8P8J6QX28R8R0S0`<08k>c/2?3`l0SHf=P8^;Rh2
:2PX0RHV9P8W9bL2:b/[0Rl_;`8e=CD2?3`l0T=3@`99BDT2CDe=0Tm?C`9@D502CTi>0T];B`98B4P2
AdM70TU9B@9ADE42G5aL0VUYJ@9dM7@2O7al0X20P0:2PX82Ph>30XF5Q@B6QXH2Q8B40X20P09kNg/2
NGUi0WmoO`:?Shl2Y:BT0[Ng]`;2`/82_[jn0[6a/@:PX:02Ti>C0Xn?S`:BTY82UiNG0YZJVP:IVIT2
UIFE0Y:BTP:?Shl4SHf=0Xn?S`:GUiL2YZJV00>g]kO6a/K6a/H00/c<c0;4a<@2/[:b0YnOW`:ATI42
SHf=0Y6AT@:IVIT2XZ:R0ZNWY`:[Zj/2[Jf]0[2`/0:b/[82]KFe0[Jf]P:g]kL2][Jf0[Bd]0:c/k<2
];Bd0[Jf]P:j^[X4_[jn0[Vi^@:a/K44ZZZZ0[Bd]0;3`l<2dM7A0]OGe`;AdM42`/;20[:b/PBWYjL2
[jn_0[Zj^P;4a<@2b/[:0/c<c0;=cLd2clo?0];BdP;De=@2eMGE0]7Ad@;9bLT2_[jn0[Bd]0:[Zj/2
YZJV1:>SX`:XZ:P2/;2`0[Vi^@:l_;`2/K6a0Y>CT`Goool100000Oooo`005Oooo`030000oooooooo
00;oool2HV9R0T51@@8Y:BT27alO0R4Q8@8X:2P2;bl_0S8b<P8a<C42;Rh^0R/[:`8Y:BT29RHV12DU
9@8Y:BT2<30`0SHf=P8k>c/2?3`l0SXj>P8g=cL2=3@d0S4a<@8`<302<S8b0SHf=P8o?cl2BTY:0UIF
EP9PH602IVIV0VUYJ@9ZJVX2Jf][0Ve]K@9aLG42M7Ad0W=cL`9]KFd2IVIV0V5QH@9VIVH2MgMg0Y2@
T0:WYjL2]KFe0[Bd]0:WYjL2UYJF0XN7Q`:0P802Ogmo0X20P0:1PH42Ogmo0WalO0AkNg/2OWin0X:2
PP:9RHT2TY:B0YnOW`03[Zj^^;Rh^;Rh00:k^k/2/[:b0Z:RXP:ATI42Q8B40X61P@:6QXH2SXj>0YJF
UP:KVi/2W9bL0YjNWP:PX:02Xj>S0ZNWY`:ZZZX2[:b/0Zf][@:[Zj/2ZZZZ0Z^[Z`:][Jd2/[:b1;Fe
]@:`/;02ZJVY0Z:RXP:QXJ42ZJVY0[Jf]P;1`L42a/K60/71`@:d];@2YZJV0YbLW0:JVYX2WinO0ZNW
Y`:_[jl2][Jf0[^k^`:o_kl2a<C40/W9b@;=cLd2c/k>0/W9b@:o_kl2/K6a0ZBTY0:IVIT2Ti>C0Y:B
TP:DU9@2V9RH0YjNWP:SXj<2Y:BT0YZJVP:2PX85oooo0@00007oool001Goool00`000?ooooooo`02
oooo0V9RHP91@D42:BTY0QlO7`8Q8B42:2PX0Rl_;`8b<S82<C4a0Rh^;P8[:b/2:BTY0RHV9P@U9BD2
:BTY0S0`<08f=SH2>c/k0S`l?08j>SX2=cLg0S@d=08a<C42<30`0S8b<P8f=SH2?clo0TY:BP9FEUH2
H61P0VIVIP9YJFT2JVYZ0V][J`9]KFd2LG5a0WAdM09cLg<2KFe]0VIVIP9QHF42IVIV0WMgM`:@T902
YjNW0[Fe]@:d];@2YjNW0YJFUP:7QhL2P8200WmoO`:0P802PH610WmoO`9lO7`4Ng]k0WinOP:2PX82
RHV90Y:BTP:OWil00jj^[[Rh^;Rh^002^k^k0[:b/P:RXZ82TI6A0XB4Q0:1PH42QXJ60Xj>SP:FUYH2
Vi^K0YbLW0:NWYh2X:2P0Z>SX`:WYjL2ZZZZ0Zb/[0:][Jd2Zj^[0ZZZZP:[Zj/2[Jf]0[:b/PBe]KD2
/;2`0ZVYZ@:RXZ82XJ6Q0ZVYZ@:f][H2`L710/K6aP;1`L42];Bd0ZJVYP:LW9`2VYZJ0YnOW`:WYjL2
[jn_0[Jf]P:k^k/2_kno0/C4a0;9bLT2cLg=0/k>cP;9bLT2_kno0[6a/@:TY:@2VIVI0Y>CT`:BTY82
U9BD0YRHV0:NWYh2Xj>S0ZBTY0:JVYX2PX:21Oooo`400001oooo000Eoooo00<0003oooooool00_oo
o`9GEeL2@4100Rh^;P8U9BD292@T0RLW9`8/;2`2<30`0S8b<P8c<c<2<C4a0Rd];@8X:2P28b<S0R4Q
8@8S8b<2:BTY0Rl_;`@b<S82;Rh^0RTY:@8U9BD292@T0RHV9P8Y:BT2;bl_0SLg=`90@402BTY:0U5A
D@9EEED2EUIF15EEE@9HF5P2Fe]K15iNGP9IFET2DU9B0Te=C@9@D502GUiN0W9bLP:6QXH2Ti>C0YBD
U0:;Rh/2Ogmo0W=cL`9/K6`2JFUY0VMWI`9VIVH2IFEU0VATI09UIFD2JFUY0Vm_K`9fMWH2Ogmo0XR8
R0:BTY800i^KVj6QXJ6QX@02XJ6Q0Y^KV`:@T902QHF50WemO@9lO7`2P8200XN7Q`:;Rh/2S8b<0XZ:
RPB7QhL2RXZ:0Xn?S`:DU9@2V9RH0YZJVPBKVi/2W9bL0Z2PX0:TY:@2YjNW0ZRXZ0:UYJD2X:2P0Y^K
V`:IVIT2Vi^K0Z6QX@:VYZH2Z:RX0ZFUY@:OWil2UiNG0Y6AT@:=SHd2Rh^;0Xf=S@:@T902UiNG0YnO
W`:XZ:P2[jn_0[Fe]@:g]kL2][Jf0[2`/0:UYJD2UiNG0XZ:RP:0P802OGem0WmoO`:3Ph<2R8R80Xb<
S0:=SHd2Rh^;0X61P@9_Kfl5oooo0@00007oool001Goool00`000?ooooooo`02oooo0UMGE`90@402
;Rh^0RDU9@8T92@29bLW0R`/;08`<302<S8b0S<c<`8a<C42;Bd]0RPX:08S8b<28B4Q0R<S8`8Y:BT2
;bl_138b<P8^;Rh2:BTY0RDU9@8T92@29RHV0RTY:@8_;bl2=cLg0T10@09:BTX2DE5A0UEEE@9FEUH4
EEEE0UQHF09KFe/4GUiN0UUIF@9BDU82CDe=0U1@D09NGUh2LW9b0XJ6QP:CTi<2U9BD0X^;R`9oOgl2
Lg=c0Va/K09YJFT2IfMW0VIVIP9UIFD2I6AT0VEUI@9YJFT2Kfm_0WIfMP9oOgl2R8R80Y:BTP03Vi^K
XJ6QXJ6Q00:QXJ42Vi^K0Y2@T0:5QHD2OGem0WalO0:0P802QhN70X^;R`:<S8`2RXZ:18N7Q`::RXX2
Shn?0YBDU0:HV9P2VYZJ19^KV`:LW9`2X:2P0ZBTY0:WYjL2Z:RX0ZFUY@:PX:02Vi^K0YVIV@:KVi/2
XJ6Q0ZJVYP:XZ:P2YJFU0YnOW`:GUiL2TI6A0Xf=S@:;Rh/2SHf=0Y2@T0:GUiL2WinO0ZRXZ0:_[jl2
]KFe0[Ng]`:f][H2/;2`0ZFUY@:GUiL2RXZ:0X20P09mOGd2Ogmo0X>3P`:8R8P2S8b<0Xf=S@:;Rh/2
PH610Vm_K`Goool100000Oooo`005Oooo`030000oooooooo00;oool2C4a<0Sdm?@8a<C42:b/[0RXZ
:P8/;2`2;bl_0S<c<`8g=cL2>CTi0SPh>08c<c<2;2`/0RDU9@8Q8B428R8R0RLW9`8/;2`2;bl_0Rd]
;@8X:2P28b<S0R0P808Q8B429BDU0R/[:`8a<C42>3Ph0Shn?P93@d<2AdM70TQ8B097AdL2ADE50TA4
A095ADD2B4Q80TY:BP99BDT2ATI60T51@@8n?Sh2?clo0TI6AP9BDU82GEeM0VIVIP9YJFT2IVIV0V5Q
H@9KFe/2EUIF0U9BDP9?Cdl4CDe=0U1@D09DE5@2Fe]K0V=SH`9[Jf/2Lg=c0WYjNP9oOgl00h:2PX>3
Ph>3P`02PX:20X20P09mOGd2Ng]k0WUiN@9kNg/2OWin1861P@9lO7`2MWIf0Vm_K`9[Jf/2K6a/0W5a
L@9hN7P2OWin0X:2PP:5QHD2R8R80XZ:RP:>SXh2TY:B0YFEU@BGUiL2UYJF0YBDU0:@T902S8b<0XN7
Q`:2PX82P8200X61P@:5QHD2QhN70XJ6QP:0P802MWIf0Vm_K`9/K6`2Lg=c0WinOP:;Rh/2UYJF0YZJ
VP:IVIT2U9BD0Xb<S0:1PH42MGEe0V][J`AVIVH2K6a/0W=cL`9hN7P2NWYj0WQhN09cLg<2JFUY0U]K
F`Goool100000Oooo`005Oooo`800003oooo0Ta<C08m?Cd2<C4a0R/[:`8Z:RX2;2`/0Rl_;`8c<c<2
=cLg0STi>@8h>3P2<c<c0R`/;08U9BD28B4Q0R8R8P8W9bL2;2`/0Rl_;`8];Bd2:2PX0R<S8`8P8202
8B4Q0RDU9@8[:b/2<C4a0SPh>08n?Sh2@d=30TM7A`98B4P2AdM70TE5A@94A4@2ADE50TQ8B09:BTX2
BDU90TI6AP91@D42?Shn0Slo?`96ATH2DU9B0UeMG@9VIVH2JFUY0VIVIP9QHF42Fe]K0UIFEP9BDU82
Cdm?14e=C@9@D502E5AD0U]KF`9SHf<2Jf][0W=cL`9jNWX2Ogmo00>2PX:3Ph>3Ph<00X:2PP:0P802
OGem0W]kN`9iNGT2Ng]k0WinOPB1PH42O7al0WIfMP9_Kfl2Jf][0Va/K09aLG42N7Qh0WinOP:2PX82
QHF50XR8R0::RXX2SXj>0Y:BTP:EUID4UiNG0YJFUP:DU9@2T92@0Xb<S0:7QhL2PX:20X20P0:1PH42
QHF50XN7Q`:6QXH2P8200WIfMP9_Kfl2K6a/0W=cL`9nOWh2Rh^;0YJFUP:JVYX2VIVI0YBDU0:<S8`2
PH610WEeM@9[Jf/4IVIV0Va/K09cLg<2N7Qh0WYjNP9hN7P2Lg=c0VUYJ@9KFe/4oooo0P00007oool0
01Goool00`000?ooooooo`02oooo0T51@@8h>3P2<S8b12l_;`8a<C42=3@d0STi>@8m?Cd2@4100Slo
?`8j>SX2<S8b0RXZ:P@T92@2:2PX0R`/;08_;bl2;2`/0RLW9`8R8R82820P0R<S8`8Z:RX2<C4a0SPh
>08l?3`2?clo1410@08o?cl2?Cdm0SXj>P8i>CT2>3Ph0STi>@@j>SX2>3Ph0SHf=P@d=3@2=CDe0SLg
=`8j>SX2?Shn0T92@P96ATH4B4Q80TA4A08o?cl2>SXj0SPh>08k>c/2@D510TY:BP9CDe<2Fe]K0V9R
HP9XJ6P2Jf][0VYZJP03J6QXIFEUIFEU009TI6@2IfMW0Va/K09bLW82N7Qh0W]kN`9mOGd2Ng]k0WMg
M`9^KVh2Hf=S0UUIF@9CDe<2DU9B0UIFEP9MGEd2I6AT0VYZJP9_Kfl2M7Ad0WQhN09lO7`2Ogmo0X:2
PP:4Q8@2R8R80X^;R`:<S8`2QhN70WalO09/K6`2GUiN0UMGE`9LG5`2JFUY0WEeM@9jNWX2Lg=c0V9R
HP9BDU82BTY:0U1@D09OGel2Kfm_0W]kN`9nOWh2NGUi0W5aL@9WIfL2GEeM0UEEE@9?Cdl2CDe=0U9B
DP9JFUX2Hf=S16YZJP9VIVH2GUiN0UADE09:BTX5oooo0@00007oool001Goool00`000?ooooooo`02
oooo0T51@@8h>3P2<S8b12l_;`8a<C42=3@d0STi>@8m?Cd2@4100Slo?`8j>SX2<S8b0RXZ:P@T92@2
:2PX0R`/;08_;bl2;2`/0RLW9`8R8R82820P0R<S8`8Z:RX2<C4a0SPh>08l?3`2?clo1410@08o?cl2
?Cdm0SXj>P8i>CT2>3Ph0STi>@@j>SX2>3Ph0SHf=P@d=3@2=CDe0SLg=`8j>SX2?Shn0T92@P96ATH4
B4Q80TA4A08o?cl2>SXj0SPh>08k>c/2@D510TY:BP9CDe<2Fe]K0V9RHP9XJ6P2Jf][0VYZJP03J6QX
IFEUIFEU009TI6@2IfMW0Va/K09bLW82N7Qh0W]kN`9mOGd2Ng]k0WMgM`9^KVh2Hf=S0UUIF@9CDe<2
DU9B0UIFEP9MGEd2I6AT0VYZJP9_Kfl2M7Ad0WQhN09lO7`2Ogmo0X:2PP:4Q8@2R8R80X^;R`:<S8`2
QhN70WalO09/K6`2GUiN0UMGE`9LG5`2JFUY0WEeM@9jNWX2Lg=c0V9RHP9BDU82BTY:0U1@D09OGel2
Kfm_0W]kN`9nOWh2NGUi0W5aL@9WIfL2GEeM0UEEE@9?Cdl2CDe=0U9BDP9JFUX2Hf=S16YZJP9VIVH2
GUiN0UADE09:BTX5oooo0@00007oool001Goool00`000?ooooooo`02oooo0SLg=`8c<c<2<30`12l_
;`8b<S82=SHf0S/k>`91@D42ADE50TI6AP91@D42>CTi0S0`<08Z:RX2:BTY0R/[:`8^;Rh2<30`0Rd]
;@8Y:BT29BDU0R@T908X:2P2;bl_0SLg=`8l?3`4?clo0Sdm?@8j>SX2>3Ph0SDe=@8c<c<2<S8b0S<c
<`8d=3@2=SHf13Lg=`8f=SH2=CDe0S<c<`8a<C42;bl_0Rh^;P8`<302=3@d0S/k>`90@402@D510S`l
?08c<c<2:b/[0RTY:@8^;Rh2>SXj0TQ8B09CDe<2FUYJ0UeMG@9NGUh2Fe]K0UIFEP03D51@Bd];Bd];
009<C4`2De=C0UmOG`9/K6`2MWIf0WYjNP9hN7P2Lg=c0V][J`9QHF42EUIF0Ta<C095ADD2@d=30TA4
A099BDT2Cdm?0UMGE`9NGUh2IFEU0V][J`9^KVh4Kfm_0W5aL@9fMWH2OGem0X61P@9lO7`2K6a/0UEE
E@8o?cl2=CDe0S`l?09?Cdl2Hf=S0Ve]K@9VIVH2De=C0Slo?`8c<c<2=cLg0TE5A@9FEUH2HV9R0VAT
I09NGUh2EUIF0Te=C@95ADD2?clo13/k>`91@D42Bd];0UEEE@AKFe/2EUIF0Te=C@95ADD2?Cdm1Ooo
o`400001oooo000Eoooo00<0003oooooool00_ooo`8g=cL2<c<c0S0`<0@_;bl2<S8b0SHf=P8k>c/2
@D510TE5A@96ATH2@D510STi>@8`<302:RXZ0RTY:@8[:b/2;Rh^0S0`<08];Bd2:BTY0RDU9@8T92@2
:2PX0Rl_;`8g=cL2?3`l13lo?`8m?Cd2>SXj0SPh>08e=CD2<c<c0S8b<P8c<c<2=3@d0SHf=P@g=cL2
=SHf0SDe=@8c<c<2<C4a0Rl_;`8^;Rh2<30`0S@d=08k>c/2@4100T51@@8l?3`2<c<c0R/[:`8Y:BT2
;Rh^0SXj>P98B4P2De=C0UYJFP9MGEd2GUiN0U]KF`9FEUH00e1@D4];Bd];B`02C4a<0U=CD`9OGel2
K6a/0WIfMP9jNWX2N7Qh0W=cL`9[Jf/2HF5Q0UIFEP9<C4`2ADE50T=3@`94A4@2BDU90Tm?C`9GEeL2
GUiN0VEUI@9[Jf/2KVi^16m_K`9aLG42MWIf0WemO@:1PH42O7al0Va/K09EEED2?clo0SDe=@8l?3`2
Cdm?0V=SH`9]KFd2IVIV0U=CD`8o?cl2<c<c0SLg=`95ADD2EUIF0V9RHP9TI6@2GUiN0UIFEP9=CDd2
ADE50Slo?`@k>c/2@D510T];B`9EEED4Fe]K0UIFEP9=CDd2ADE50Sdm?@Goool100000Oooo`005Ooo
o`030000oooooooo00;oool2<30`0R`/;08Y:BT4:2PX0RXZ:P8`<302=cLg0T10@098B4P2Bd];0TM7
A`90@402=SHf0Rl_;`8];Bd2;Rh^0S4a<@8b<S82<30`0R`/;0@Y:BT2;2`/0S8b<P8h>3P4?3`l0S/k
>`8h>3P2=CDe0S8b<P@`<302<S8b0SHf=P8m?Cd2@d=314M7A`94A4@2@D5113lo?`91@D42A4A40TM7
A`9;Bd/2Cdm?0U1@D09;Bd/2@4100S4a<@8T92@2820P0RXZ:P8l?3`2Cdm?0UeMG@9RHV82GEeM0UEE
E@9;Bd/2@T9200<m?Cdj>SXj>SX00Sdm?@98B4P2EeMG0VMWI`9bLW82M7Ad0Vm_K`9WIfL2GEeM0UIF
EP9ADE42C4a<0TQ8B095ADD2@T920T51@@94A4@2C4a<0UEEE@9OGel4IVIV0V=SH`9OGel2GUiN0VAT
I09]KFd2Lg=c0Vm_K`9MGEd2@d=30R/[:`8O7al29bLW0S`l?09BDU82H61P0U]KF`9:BTX2>3Ph0R/[
:`8/;2`2=SHf0T92@P9<C4`2Cdm?0Te=C@99BDT2A4A40Sdm?@8h>3P4<c<c0SLg=`8o?cl2B4Q80Te=
C@9>CTh2BTY:0T=3@`8l?3`2=SHf1Oooo`400001oooo000Eoooo00<0003oooooool00_ooo`8`<302
;2`/0RTY:@@X:2P2:RXZ0S0`<08g=cL2@4100TQ8B09;Bd/2AdM70T10@08f=SH2;bl_0Rd];@8^;Rh2
<C4a0S8b<P8`<302;2`/12TY:@8/;2`2<S8b0SPh>0@l?3`2>c/k0SPh>08e=CD2<S8b130`<08b<S82
=SHf0Sdm?@93@d<4AdM70TA4A091@D44?clo0T51@@94A4@2AdM70T];B`9?Cdl2D51@0T];B`90@402
<C4a0R@T908P8202:RXZ0S`l?09?Cdl2GEeM0V9RHP9MGEd2EEEE0T];B`92@T800cdm?CXj>SXj>P02
?Cdm0TQ8B09GEeL2IfMW0W9bLP9dM7@2Kfm_0VMWI`9MGEd2EUIF0U5AD@9<C4`2B4Q80TE5A@92@T82
@D510TA4A09<C4`2EEEE0UmOG`AVIVH2Hf=S0UmOG`9NGUh2I6AT0Ve]K@9cLg<2Kfm_0UeMG@93@d<2
:b/[0QlO7`8W9bL2?3`l0U9BDP9PH602Fe]K0TY:BP8h>3P2:b/[0R`/;08f=SH2@T920Ta<C09?Cdl2
CDe=0TU9B@94A4@2?Cdm0SPh>0@c<c<2=cLg0Slo?`98B4P2CDe=0Ti>CP9:BTX2@d=30S`l?08f=SH5
oooo0@00007oool001Goool00`000?ooooooo`3ooooo1oooo`400001oooo000Eoooo00<0003ooooo
ool0ooooo`Ooool100000Oooo`005Oooo`030000oooooooo00;oool00`000?ooooooo`0Uoooo00<0
003oooooool09Oooo`030000oooooooo02Goool00`000?ooooooo`0Uoooo00<0003oooooool09?oo
o`030000oooooooo02Goool00`000?ooooooo`0Boooo0@00007oool001Goool200000oooo`030000
oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003ooooo
ool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo00<0003oooooool01oooo`030000
oooooooo00Ooool00`000?ooooooo`06oooo00<0003oooooool01oooo`030000oooooooo00Ooool0
0`000?ooooooo`07oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07
oooo00<0003oooooool01oooo`030000oooooooo00Ooool00`000?ooooooo`07oooo0P00007oool0
01Gooooo00002`00007oool00?oooolQoooo003ooooo8Oooo`00ooooob7oool00?oooolQoooo0000
\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-13.248, -11.0235, 0.501294, 
  0.501294}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output"]
}, Closed]],

Cell[BoxData[
    RowBox[{"\[FilledCircle]", " ", 
      StyleBox["Correlation",
        FontVariations->{"Underline"->True}]}]], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(corRelation[Flatten[a], Flatten[b]]\)], "Input"],

Cell[BoxData[
    \(0.750271162109481082`\)], "Output"]
}, Open  ]],

Cell[BoxData[
    RowBox[{"\[FilledCircle]", " ", 
      StyleBox["Ratio",
        FontVariations->{"Underline"->True}]}]], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[\((32\ 32)\)/\((128\ 128)\)]\)], "Input"],

Cell[BoxData[
    \(0.0625`\)], "Output"]
}, Open  ]]
},
FrontEndVersion->"Microsoft Windows 3.0",
ScreenRectangle->{{0, 800}, {0, 544}},
WindowSize->{763, 407},
WindowMargins->{{Automatic, 2}, {36, Automatic}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{
Cell[1709, 49, 204, 4, 90, "Input"],
Cell[1916, 55, 63, 1, 30, "Input"],
Cell[1982, 58, 512, 13, 210, "Input"],
Cell[2497, 73, 190, 4, 30, "Input"],
Cell[2690, 79, 310, 10, 30, "Text",
  Evaluatable->False],
Cell[3003, 91, 3291, 68, 922, "Input"],
Cell[6297, 161, 65, 1, 30, "Input"],
Cell[6365, 164, 413, 13, 152, "Input"],
Cell[6781, 179, 209, 6, 54, "Input"],
Cell[6993, 187, 360, 10, 110, "Input"],
Cell[7356, 199, 262, 8, 82, "Input"],
Cell[7621, 209, 58, 1, 30, "Input"],
Cell[7682, 212, 309, 5, 50, "Input"],

Cell[CellGroupData[{
Cell[8016, 221, 66, 1, 30, "Input"],
Cell[8085, 224, 24307, 575, 296, 3833, 318, "GraphicsData", 
"PostScript", "Graphics"],
Cell[32395, 801, 137, 3, 29, "Output"]
}, Closed]],
Cell[32547, 807, 68, 1, 27, "Input"],
Cell[32618, 810, 54, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[32697, 815, 72, 1, 30, "Input"],
Cell[32772, 818, 31257, 661, 296, 3833, 318, "GraphicsData", 
"PostScript", "Graphics"],
Cell[64032, 1481, 137, 3, 29, "Output"]
}, Closed]],
Cell[64184, 1487, 138, 3, 27, "Input"],
Cell[64325, 1492, 44, 1, 30, "Input"],
Cell[64372, 1495, 242, 6, 66, "Input",
  PageBreakWithin->Automatic],

Cell[CellGroupData[{
Cell[64639, 1505, 60, 1, 30, "Input"],
Cell[64702, 1508, 1806, 32, 318, "Output"]
}, Closed]],

Cell[CellGroupData[{
Cell[66545, 1545, 47, 1, 27, "Input"],
Cell[66595, 1548, 5663, 96, 340, "Output"]
}, Closed]],

Cell[CellGroupData[{
Cell[72295, 1649, 83, 1, 27, "Input"],
Cell[72381, 1652, 27693, 617, 296, 3833, 318, "GraphicsData", 
"PostScript", "Graphics"],
Cell[100077, 2271, 137, 3, 29, "Output"]
}, Closed]],

Cell[CellGroupData[{
Cell[100251, 2279, 99, 2, 27, "Input"],
Cell[100353, 2283, 22882, 558, 296, 3833, 318, "GraphicsData", 
"PostScript", "Graphics"],
Cell[123238, 2843, 137, 3, 29, "Output"]
}, Closed]],
Cell[123390, 2849, 70, 1, 27, "Input"],
Cell[123463, 2852, 66, 1, 30, "Input"],
Cell[123532, 2855, 64, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[123621, 2860, 67, 1, 30, "Input"],
Cell[123691, 2863, 31079, 659, 296, 3833, 318, "GraphicsData", 
"PostScript", "Graphics"],
Cell[154773, 3524, 137, 3, 29, "Output"]
}, Closed]],
Cell[154925, 3530, 138, 3, 27, "Input"],
Cell[155066, 3535, 44, 1, 30, "Input"],
Cell[155113, 3538, 246, 6, 84, "Input",
  PageBreakWithin->Automatic],
Cell[155362, 3546, 77, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[155464, 3551, 102, 2, 30, "Input"],
Cell[155569, 3555, 31257, 661, 296, 3833, 318, "GraphicsData", 
"PostScript", "Graphics"],
Cell[186829, 4218, 137, 3, 29, "Output"]
}, Closed]],

Cell[CellGroupData[{
Cell[187003, 4226, 76, 1, 27, "Input"],
Cell[187082, 4229, 55, 1, 29, "Output"]
}, Open  ]],
Cell[187152, 4233, 65, 1, 30, "Input"],
Cell[187220, 4236, 100, 2, 30, "Input"],

Cell[CellGroupData[{
Cell[187345, 4242, 66, 1, 30, "Input"],
Cell[187414, 4245, 84656, 1781, 296, 38821, 1210, "GraphicsData", 
"PostScript", "Graphics"],
Cell[272073, 6028, 137, 3, 29, "Output"]
}, Closed]],
Cell[272225, 6034, 276, 7, 27, "Input"],
Cell[272504, 6043, 69, 1, 30, "Input"],
Cell[272576, 6046, 64, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[272665, 6051, 67, 1, 30, "Input"],
Cell[272735, 6054, 199113, 3194, 296, 38821, 1210, "GraphicsData", 
"PostScript", "Graphics"],
Cell[471851, 9250, 137, 3, 29, "Output"]
}, Closed]],
Cell[472003, 9256, 138, 3, 27, "Input"],
Cell[472144, 9261, 46, 1, 30, "Input"],
Cell[472193, 9264, 245, 6, 66, "Input",
  PageBreakWithin->Automatic],
Cell[472441, 9272, 77, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[472543, 9277, 102, 2, 30, "Input"],
Cell[472648, 9281, 189470, 3075, 296, 38821, 1210, "GraphicsData", 
"PostScript", "Graphics"],
Cell[662121, 12358, 137, 3, 29, "Output"]
}, Closed]],
Cell[662273, 12364, 138, 3, 27, "Input"],

Cell[CellGroupData[{
Cell[662436, 12371, 68, 1, 30, "Input"],
Cell[662507, 12374, 55, 1, 29, "Output"]
}, Open  ]],
Cell[662577, 12378, 132, 3, 30, "Input"],

Cell[CellGroupData[{
Cell[662734, 12385, 63, 1, 30, "Input"],
Cell[662800, 12388, 41, 1, 29, "Output"]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

